/*-----------------------------------------------------------------------------*
 * AdapterOpticalDiscNotifications.cpp                                         *
 *-----------------------------------------------------------------------------*
 * SW-COMPONENT: VD_DeviceManager                                              *
 * PROJECT     : G3G                                                           *
 * COPYRIGHT   : (c) 2013-2020 Robert Bosch GmbH, Hildesheim                   *
 *                                                                             *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * doxygen style header                                                        *
 *-----------------------------------------------------------------------------*/
/*!
 * \file AdapterOpticalDiscNotifications.cpp
 *
 * \brief contains functions to derive the cddda media state.
 * Use by client handler of media manager
 *
 * \version 12.06.2015, Rajeev Narayanan Sambhu (RBEI/ECS1), version 1.0
 *          Initial version
 *
 * \copyright Copyright (c) Robert Bosch Car Multimedia GmbH  2010-2016
 */

/*-----------------------------------------------------------------------------*
 * Includes                                                                    *
 *-----------------------------------------------------------------------------*/


#include <stdio.h>
#include "Config.h"


#define INCLUDE_VD_DVM_BASICS
#define INCLUDE_VD_DVM_AILAHL
#define INCLUDE_VD_DVM_FIDIAG

#include "Common.h"
#include "Enums.h"
#include "Device.h"
#include "StateTable.h"
#include "AdapterOpticalDiscNotifications.h"
#include "utils.h"
#include "Diagnosis.h"
#include "interface/DeviceManagerInterface.h"


/*-----------------------------------------------------------------------------*
 * ETG Tracing                                                                 *
 *-----------------------------------------------------------------------------*/
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_dvm.h"

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_VD_DEVICEMANAGER_CLIENTHANDLER_VD_MEDIAMANAGER
#include "trcGenProj/Header/AdapterOpticalDiscNotifications.cpp.trc.h"
#endif
#include "ETGTrace.h"
#endif //VARIANT_S_FTR_ENABLE_UNITTEST

using namespace statetbl;



#ifdef VARIANT_S_FTR_ENABLE_UNITTEST
static IDeviceManagerInterfaceGeneral   *g_pIDeviceManagerInterfaceGeneral   = NULL;
#define DVMGRIF_GENERAL                  g_pIDeviceManagerInterfaceGeneral
#else
#define DVMGRIF_GENERAL          DeviceManagerInterface::GetInstance()
#endif


CDDA_IFOutput_t AdapterOpticalDiscNotifications::m_CDDA_IFOutput_t[] =
{
    /*------------------------------------------------------------------------------------------------------------------------------*/
    /*No.  RESULT (17 entries):*/
    /*  tOpticalDiscSlotTypes                CDDATypes           CDDAInsertState         CDDAMediaState                               */
    {INITIALISING,                  MMGR_INIT,          MMGR_INSERT_ANY,       MMGR_MEDIA_NOT_READY},
    {INITIALISED,                   MMGR_INIT,          MMGR_INSERT_ANY,       MMGR_MEDIA_READY},
    {LASTMODE_EMTPY,                MMGR_NO_MEDIA,      MMGR_INSERT_ANY,       MMGR_MEDIA_READY},
    {LASTMODE_INSERTED_CDDA,        MMGR_AUDIO,         MMGR_INSERT_BEFOREON,  MMGR_MEDIA_READY},
    {LASTMODE_INSERTED_CDROM,       MMGR_DATA,          MMGR_INSERT_BEFOREON,  MMGR_MEDIA_READY},
    {LASTMODE_INSERTED_CDERROR,     MMGR_INCORRECT,     MMGR_INSERT_BEFOREON,  MMGR_MEDIA_NOT_READY},
    {INSERTING,                     MMGR_INSERTION,     MMGR_INSERT_AFTERON,   MMGR_MEDIA_NOT_READY},
    {INSERTED_CDAUDIO,              MMGR_AUDIO,         MMGR_INSERT_AFTERON,   MMGR_MEDIA_READY},
    {INSERTED_CDROM,                MMGR_DATA,          MMGR_INSERT_AFTERON,   MMGR_MEDIA_READY},
    {INSERTED_CDERROR,              MMGR_INCORRECT,     MMGR_INSERT_AFTERON,   MMGR_MEDIA_NOT_READY},
    {INSERTED_AUTOMATIC_CDAUDIO,    MMGR_AUDIO,         MMGR_INSERT_AUTOMATIC, MMGR_MEDIA_READY},
    {INSERTED_AUTOMATIC_CDROM,      MMGR_DATA,          MMGR_INSERT_AUTOMATIC, MMGR_MEDIA_READY},
    {INSERTED_AUTOMATIC_CDERROR,    MMGR_INCORRECT,     MMGR_INSERT_AUTOMATIC, MMGR_MEDIA_NOT_READY},
    {EJECTING,                      MMGR_EJECTING,      MMGR_INSERT_AUTOMATIC, MMGR_MEDIA_READY},
    {EJECTING,                      MMGR_EJECTING,      MMGR_INSERT_AFTERON, MMGR_MEDIA_NOT_READY},
    {EJECTED_READY_TO_REMOVE,       MMGR_MEDIA_IN_SLOT, MMGR_INSERT_AUTOMATIC, MMGR_MEDIA_READY},
    {EJECTED_EMPTY,                 MMGR_NO_MEDIA,      MMGR_INSERT_AUTOMATIC, MMGR_MEDIA_READY}

};

AdapterOpticalDiscNotifications::AdapterOpticalDiscNotifications()
{
    ETG_TRACE_USR4(("Begin  : AdapterOpticalDiscNotifications()"));
    m_u8NumberofCDDAStates = 17;
    strcpy(m_cCDDASerialNumber,"");
    ETG_TRACE_USR4(("End  : AdapterOpticalDiscNotifications()"));
}
/*------------------------------------------------------------------------------------------------------------------------------------------------------*
 * tVoid vCalculateOpticalDiscState(IN mplay_fi_tcl_e8_CdType e8CDType,IN mplay_fi_tcl_e8_MediaState e8MediaState,IN mplay_fi_tcl_e8_InsertState e8InsertState) *                                                                  *
 *------------------------------------------------------------------------------------------------------------------------------------------------------*/

tDMOpticalDiscSlotState AdapterOpticalDiscNotifications::vCalculateOpticalDiscState(IN tU8 u8CDType,IN tU8 u8InsertState,IN tU8 u8MediaState)
{


    ETG_TRACE_USR4(("Begin  : vCalculateOpticalDiscState"));

    CDDAInsertState  e8CDDAInsertState = (CDDAInsertState)u8InsertState;
    tDMOpticalDiscSlotState  eOpticalDiscSlotState = UNDEFINED_STATE;


    ETG_TRACE_USR3(( "vCalculateOpticalDiscState: Start: u8CDType= %d, u8MediaState= %d u8InsertState= %d", ETG_ENUM(CDDATypes, u8CDType), ETG_ENUM(CDDAMediaState, u8MediaState),ETG_ENUM(CDDAInsertState, u8InsertState) ));

    for (int i = 0; i<m_u8NumberofCDDAStates;i++)
    {
        if(MMGR_INSERT_ANY == m_CDDA_IFOutput_t[i].CDDAinfo.e8CDDAInsertState )
        {

            //if the CDDA InsertState is MMGR_INSERT_AUTOMATIC, then there is a chance that, the CD slot is EJECTED_EMPTY
            if(MMGR_INSERT_AUTOMATIC != e8CDDAInsertState)
            {
            	ETG_TRACE_USR3(( "vCalculateOpticalDiscState:Setting the Insert state to MMGR_INSERT_ANY"));

                e8CDDAInsertState = m_CDDA_IFOutput_t[i].CDDAinfo.e8CDDAInsertState;
            }
        }
        else
        {
            //setting the state back to the input insert state
            e8CDDAInsertState =(CDDAInsertState) u8InsertState;
        }


        if((m_CDDA_IFOutput_t[i].CDDAinfo.e8CDDATypes == u8CDType) &&
                (m_CDDA_IFOutput_t[i].CDDAinfo.e8CDDAMediaState == u8MediaState) &&
                (m_CDDA_IFOutput_t[i].CDDAinfo.e8CDDAInsertState == e8CDDAInsertState)
                )
        {
            eOpticalDiscSlotState = m_CDDA_IFOutput_t[i].result4Client;
            ETG_TRACE_USR3(( "vCalculateOpticalDiscState: eOpticalDiscSlotState = %d",ETG_ENUM(tDMOpticalDiscSlotState, eOpticalDiscSlotState)));


            break;
        }
        else
        {
            ETG_TRACE_USR3(( "vCalculateOpticalDiscState: Unhandled cd slot state"));


        }


    }//for (int i = 0; i<m_u8NumberofCDDAStates;i++)


    ETG_TRACE_USR4(("End  : vCalculateOpticalDiscState"));
    return  eOpticalDiscSlotState;

}
/*---------------------------------------------------------------------*
 * tVoid vSendResult(const tDMOpticalDiscSlotState  eOpticalDiscSlotState)       *
 *---------------------------------------------------------------------*/
tVoid AdapterOpticalDiscNotifications::vSendResult(IN const tDMOpticalDiscSlotState eOpticalDiscSlotState)
{
    ETG_TRACE_USR4(("Begin  : vSendResult"));
    DEVICE_CONNECTSTATUS_E eCDDAConnectionStatus = USB_DEV_UNDEFINED;

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
        ETG_TRACE_USR4(("Sending the CDDA Slot state"));
        StateTable::GetInstance()->vSetOpticalDiscSlotStateNotified(enOptDiscSlotStateNotified, eOpticalDiscSlotState);
#endif

    switch( eOpticalDiscSlotState)
    {
        case INSERTED_CDAUDIO:
        case LASTMODE_INSERTED_CDDA:
        case INSERTED_AUTOMATIC_CDAUDIO:
            eCDDAConnectionStatus = USB_DEV_INTERNAL_APPLY_CONNECT;
            break;

        case EJECTING:
        case EJECTED_READY_TO_REMOVE:
        case EJECTED_EMPTY:
            eCDDAConnectionStatus = USB_DEV_INTERNAL_APPLY_REMOVED_BY_USR;
            break;
    }

    CDevice l_CDevice;
    vFillCDDADevice(l_CDevice,eCDDAConnectionStatus);
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
    StateTable::GetInstance()->vsetCDDANotified(enCDDANotified,eCDDAConnectionStatus, &l_CDevice);
#endif


    ETG_TRACE_USR4(("End  : vSendResult"));


}
/*----------------------------------------------------*
 * ~AdapterOpticalDiscNotifications()                 *
 *----------------------------------------------------*/

AdapterOpticalDiscNotifications::~AdapterOpticalDiscNotifications()
{

}

/*-----------------------------------------------------------------------------------------------------------------------*
 * tVoid vFillCDDADevice(CDevice &f_CDevice, DEVICE_CONNECTSTATUS_E eCDDAConnectionStatus,std::string strCDDAMountPoint) *
 *-----------------------------------------------------------------------------------------------------------------------*/

tVoid AdapterOpticalDiscNotifications::vFillCDDADevice(CDevice &f_CDevice, DEVICE_CONNECTSTATUS_E eCDDAConnectionStatus)
{
    std::string strCDDAMountPoint = "";
    std::string strSRx            = "";
    std::string strSGx            = "";

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
    if(TRUE == bFindScsiCDDrive(OUT strSRx, OUT strSGx))
    {
        strCDDAMountPoint  = strSRx;
        strCDDAMountPoint += ",";
        strCDDAMountPoint += strSGx;
        ETG_TRACE_USR4(("vFillCDDADevice: strCDDAMountPoint: %s ",strCDDAMountPoint.c_str()));
    }
    else
    {

        ETG_TRACE_FATAL(("vFillCDDADevice: bFindScsiCDDrive(...) == FALSE!!"));
    }
#endif
   
    vFillCDDADevice(f_CDevice, eCDDAConnectionStatus, strCDDAMountPoint);
}


tVoid AdapterOpticalDiscNotifications::vUpdateOpticalDiscSerialID(char f_cCDDASerialNumber[1024])
{
    ETG_TRACE_USR4(("Begin  : vUpdateOpticalDiscSerialID"));
    strcpy(m_cCDDASerialNumber,f_cCDDASerialNumber);
    ETG_TRACE_USR4(("End  : vUpdateOpticalDiscSerialID"));

}
tVoid AdapterOpticalDiscNotifications::vFillCDDADevice(OUT CDevice &f_CDevice, IN DEVICE_CONNECTSTATUS_E eCDDAConnectionStatus, IN std::string & strCDDAMountPoint)
{
    ETG_TRACE_USR4(("Begin  : vFillCDDADevice"));

    f_CDevice.m_eConnectStatus = eCDDAConnectionStatus;
    f_CDevice.m_eDeviceType = CGlobalEnumerations::DTY_CDDA;
    //dummy data
    f_CDevice.m_iProductID = 0xbeaf;
    f_CDevice.m_iVendorID = 0xdead;
    f_CDevice.m_cMountPoint = strCDDAMountPoint;
    f_CDevice.m_cDeviceName = "CDDA";
    f_CDevice.m_cFSType = "ISO9660";
    f_CDevice.m_eFSType = CGlobalEnumerations::FSTY_ISO9660;

#ifdef USE_RANDOMID_CDDA
    //map to CDevice-elements
    srand((unsigned int)time(NULL));
    char cSerial[256]; //@todo intermediate solution - has to be replaced by scsi commands
    snprintf(cSerial,sizeof(cSerial),"TESTID:%d",rand()); //@todo intermediate solution
    f_CDevice.m_cSerialID = cSerial;
#else
    f_CDevice.m_cSerialID = m_cCDDASerialNumber;
#endif
    f_CDevice.m_cShortSerial = "";
    f_CDevice.m_cManufacturer = "NISSAN";

    ETG_TRACE_USR4(("End  : vFillCDDADevice"));
}


//if dependency with DVMGRIF_GENERAL has been changed then shift this simulation part to the trace
//philosophy don't have test code in production code or at least have less test code in production code.
tVoid AdapterOpticalDiscNotifications::vSimulateCDDAConnection(IN DEVICE_CONNECTSTATUS_E eCDDAConnectionStatus)
{
    ETG_TRACE_FATAL(("Begin: vSimulateCDDAConnection"));
    ETG_TRACE_FATAL(("vSimulateCDDAConnection eCDDAConnectionStatus %d:",ETG_ENUM(DEVICE_CONNECTSTATUS_TYPE,eCDDAConnectionStatus)));

    CDevice l_CDevice;
    vFillCDDADevice(l_CDevice,eCDDAConnectionStatus);
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
    StateTable::GetInstance()->vsetCDDANotified(enCDDANotified,eCDDAConnectionStatus, &l_CDevice);
#endif

    ETG_TRACE_FATAL(("End  : vSimulateCDDAConnection"));
}


/*------------------------------------------------------------------------------------------------*
 * tVoid vSimulateCDDAConnection(std::string f_cConnectiontype)                                   *
 *------------------------------------------------------------------------------------------------*/

tVoid AdapterOpticalDiscNotifications::vSimulateCDDAConnection(IN DEVICE_CONNECTSTATUS_E eCDDAConnectionStatus,IN std::string &strCDDAMountPoint)
{
    ETG_TRACE_FATAL(("Begin  : vSimulateCDDAConnection"));
    ETG_TRACE_FATAL(("vSimulateCDDAConnection eCDDAConnectionStatus %d:",ETG_ENUM(DEVICE_CONNECTSTATUS_TYPE,eCDDAConnectionStatus)));
    ETG_TRACE_FATAL(("vSimulateCDDAConnection strCDDAMountPoint     %s:",strCDDAMountPoint.c_str()));

    CDevice l_CDevice;
    vFillCDDADevice(OUT l_CDevice,IN eCDDAConnectionStatus,IN strCDDAMountPoint);
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
    StateTable::GetInstance()->vsetCDDANotified(enCDDANotified,eCDDAConnectionStatus, &l_CDevice);
#endif
    
    ETG_TRACE_FATAL(("End  : vSimulateCDDAConnection"));
}

/*---------------------------------------------------------------------*
 * tVoid vSendCDInfo(IN trOpticalDiskCDInfo OpticalDiskCDInfo)       *
 *---------------------------------------------------------------------*/
tVoid AdapterOpticalDiscNotifications::vSendCDInfo(IN trOpticalDiskCDInfo f_OpticalDiskCDInfo)
{
    ETG_TRACE_USR4(("Begin  : vSendCDInfo"));

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
        ETG_TRACE_USR4(("Sending the Optical drive CD Info"));
        StateTable::GetInstance()->vSetOpticalDiscCDInfoNotified(enOptDiscCDInfoNotified, f_OpticalDiskCDInfo);
#endif
    ETG_TRACE_USR4(("End  : vSendCDInfo"));


}

/*------------------------------------------------------------------------------------------------------------------------------------*
 * tBool bCalculateOpticalDiscMalfunctionState(IN tenOpticalDeviceState f_OpticalDeviceState,IN tU8 u8MediaState)                              *
 *------------------------------------------------------------------------------------------------------------------------------------*/

tBool AdapterOpticalDiscNotifications::bCalculateOpticalDiscMalfunctionState(IN tenOpticalDeviceState f_OpticalDeviceState,IN tU8 u8MediaState)
{
    ETG_TRACE_USR4(("Begin  : bCalculateOpticalDiscMalfunctionState"));
    tBool f_bMalfunctionStatus = FALSE;
    if(DEVICE_NOT_READY == f_OpticalDeviceState && MMGR_MEDIA_NOT_READY == u8MediaState)
    {
        f_bMalfunctionStatus = TRUE;
    }
    ETG_TRACE_USR4(("End  : bCalculateOpticalDiscMalfunctionState"));
    return f_bMalfunctionStatus;


}
/*--------------------------------------------------------------------------*
*tVoid vSendMalfunctionStatus(tBool f_bOpticalDiscMalfunctionstatus)        *
*---------------------------------------------------------------------------*/
tVoid AdapterOpticalDiscNotifications::vSendMalfunctionStatus(IN tBool f_bOpticalDiscMalfunctionstatus)
{
    ETG_TRACE_USR4(("Begin  : vSendMalfunctionStatus"));
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
    ETG_TRACE_USR4(("Sending the Optical drive Malfunction Status"));
    StateTable::GetInstance()->SetMalfunctionOpticalDisc(IN f_bOpticalDiscMalfunctionstatus);
#endif
    ETG_TRACE_USR4(("End    : vSendMalfunctionStatus f_bOpticalDiscMalfunctionstatus: %d",ETG_ENUM(BOOL,f_bOpticalDiscMalfunctionstatus)));
}


////////////////////////////////////////////////////////////////////////////////
// <EOF>


