/* -------------------------------------------------------------------------- */
/**
 *   @defgroup meengine mecodec_rip.hpp
 *   @ingroup  MEEngine
 *   @author   Vignesh Kumar, 2017
 *
 *   cd ripper codec.
 */
/* -------------------------------------------------------------------------- */

#if !defined(ME_CODEC_RIP_HPP_INCLUDED)
#define ME_CODEC_RIP_HPP_INCLUDED

/* -------------------------------------------------------------------------- */
#include <iostream>
#include <string>
#include "1_common/megstcodec.hpp"

/* -------------------------------------------------------------------------- */

namespace me {

/* -------------------------------------------------------------------------- */

struct codec_rip_t : public gstcodec_t,
                     public single_t<codec_rip_t> {

   /* ----------------------------------------------------------------------- */

   friend class single_t<codec_rip_t>;

   /* ----------------------------------------------------------------------- */

   virtual strings_t onexts() const {
      trace("codec_rip:onexts()");
      static strings_t fts;
      if(fts.empty()) {
         fts.append(string_t("rip"));
      }
      return fts;
   }

   virtual int_t onisaudio(string_t const &ext0) const {
      if(0 == onexts().find_f(ext0)) return -1;
      return 1; //
   }
   virtual int_t onisvideo(string_t const &ext0) const {
      if(0 == onexts().find_f(ext0)) return -1;
      return 0;
   }

   /* ----------------------------------------------------------------------- */

   codec_rip_t() : loop(0), line(0), src(0), enc(0),
           conv(0), sink(0) {
      pump = false;
   }

   /* ----------------------------------------------------------------------- */

   virtual void_t oninit() {

      trace("me:codec_rip:oninit");

      /* -------------------------------------------------------------------- */
       /*gst_init() call is moved to mediaengineserver::init  to avoid the multiple calls to platform  
         Fix for NCG3D-118451*/
      // setup pipeline;

      // mainloop;
      loop = g_main_loop_new(0, false); asrt(NULL != loop);

      // pipeline;
      line = gst_pipeline_new("line"); asrt(NULL != line);

      trace("me:codec_rip:oninit:pipeline");

      /* -------------------------------------------------------------------- */
      // elements;

      src = gst_element_factory_make("cdparanoiasrc", "src");   asrt(NULL != src);

      conv = gst_element_factory_make("audioconvert", "conv");  asrt(NULL != conv);

      sink  = gst_element_factory_make("filesink",    "sink" ); asrt(NULL != sink);

      /* -------------------------------------------------------------------- */
      // configure;

      g_object_set((GObject*)src, "mode", "normal", NULL);

      /* -------------------------------------------------------------------- */
      // link;

      gst_bin_add_many(GST_BIN(line), src, conv, NULL);

      /* -------------------------------------------------------------------- */
      // callbacks;

      GstBus *bus = gst_pipeline_get_bus(GST_PIPELINE(line));
      gst_bus_add_watch(bus, cb_bus_call, this);

      #if defined (ENABLE_GSTREAMER_1_0)
         // gst_bus_set_sync_handler(bus, cb_bus_sync, this) is modified to
         // gst_bus_set_sync_handler(bus, cb_bus_sync, this, NULL)
         gst_bus_set_sync_handler(bus, cb_bus_sync, this, NULL);
      #else
         gst_bus_set_sync_handler(bus, cb_bus_sync, this);
      #endif

      gst_object_unref(bus);

      /* -------------------------------------------------------------------- */

      trace("me:codec_rip:oninit:finished");
   }

   /* ----------------------------------------------------------------------- */

   virtual void_t mainloop() {

      /* -------------------------------------------------------------------- */

      trace("me:codec_rip:mainloop");

      if(NULL != loop)
      {  
         g_main_loop_run(loop); // main routine;
      }

      trace("me:codec_rip:mainloop:finished");

      /* -------------------------------------------------------------------- */

      GstBus *bus = gst_pipeline_get_bus(GST_PIPELINE(line));

      #if defined (ENABLE_GSTREAMER_1_0)
         // gst_bus_set_sync_handler(bus, 0, 0) is modified to
         // gst_bus_set_sync_handler(bus, 0, 0, NULL)
         gst_bus_set_sync_handler(bus, 0, 0, NULL);
      #else
         gst_bus_set_sync_handler(bus, 0, 0);
      #endif

      gst_object_unref(bus);

      /* -------------------------------------------------------------------- */

      GstStateChangeReturn res = gst_element_set_state(line, GST_STATE_NULL);

      trace("me:codec_rip:mainloop:GST_STATE_NULL:res:", res);

      /* -------------------------------------------------------------------- */
      // handle async statechange;

      int_t seconds = 0;
      while(GST_STATE_CHANGE_ASYNC == res) {

         GstState st0, st1;
         st1 = st0 = GST_STATE_NULL;
         GstClockTime const timeout = 1000000000ull; // nanoseconds (1 second);

         trace("me:codec_rip:state:async:wait:", (int_t)timeout);

         res = gst_element_get_state(line, &st0, &st1, timeout);

         trace("me:codec_rip:state:async:", st0, " -> ", st1, " res:", res);

         if(GST_STATE_CHANGE_ASYNC == res) {

            trace("me:codec_rip:state:still:async:seconds:", ++seconds);
         }
      }

      /* -------------------------------------------------------------------- */
      // cleanup;

      if(enc)
      {
         gst_element_unlink_many((GstElement*)src, conv, enc, sink, NULL);
      }

      if(line) {
         gst_object_unref(line);
         line = NULL;
      }
      if(loop) {
         g_main_loop_unref(loop);
         loop = NULL;
      }
   }

   /* ----------------------------------------------------------------------- */

   virtual void_t onfini() {

      trace("me:codec_rip:onfini");

      if(loop) {

         trace("me:codec_rip:onfini:mainloop:quit");

         g_main_loop_quit(loop);
      }

      trace("me:codec_rip:onfini:gst:deinit");
   }

   /* ----------------------------------------------------------------------- */

   virtual int_t onctrl(state_t const &state) {

      playstate_e const ps = state.playstate;

      trace("me:codec_rip:onctrl:", estr2(ps));

      /* -------------------------------------------------------------------- */

      trace("me:codec_rip:onctrl:log:pat:", state.logconfig().str());

      int_t const res = gstcodec_t::configlog(state.logconfig);

      trace("codec_rip:onctrl:configlog:res:", res);

      /* -------------------------------------------------------------------- */

      switch(ps) {
         case PLAYSTATE_STOP:
            gst_element_set_state(line, GST_STATE_NULL);
            break;
         case PLAYSTATE_PAUSE:
            gst_element_set_state(line, GST_STATE_PAUSED);
            break;
         case PLAYSTATE_PLAY: {

            /* -------------------------------------------------------------------- */

            GstState st0 = GST_STATE_NULL;
            gst_element_get_state(line, &st0, 0, GST_CLOCK_TIME_NONE);
            if(GST_STATE_PLAYING == st0) {
               gst_element_set_state(line, GST_STATE_NULL);
            }

            /* -------------------------------------------------------------------- */

            // see meprotocol.hpp l.480+;
            string_t ext         = estr1(out.ripconf().encodeformat);
            string_t encodername = estr4(out.ripconf().encodeformat);
            //fetch_encoder(out.ripconf().encodeformat, ext);

            enc = gst_element_factory_make(encodername.at(), "encoder"); asrt(NULL != enc);

            g_object_set(enc, "bitrate", (int)out.ripconf().encodequality(), NULL);

            /* -------------------------------------------------------------------- */
            // url -> input device;

            string_t url = in.url; // e.g: url=/dev/sr0.rip
            string_t devid;
            auto const p = url.find_f('.');
            if(p) {
               devid = url.upto(p - 1);
            } else {
               trace("codec_rip:onctrl:url:", url, "err:missing extension");
               break;
            }

            trace("me:codec_rip:onctrl:play:url:", url, " input:devid:", devid);

            g_object_set(src, "device", devid.at(), NULL);

            /* -------------------------------------------------------------------- */
            // set track number;

            trace("me:codec_rip:onctrl:track:num:", out.ripconf().tracknumber());
            g_object_set(src, "track", (unsigned int)out.ripconf().tracknumber(), NULL);

            /* -------------------------------------------------------------------- */
            // output sink location;

            #if 0
               string_t outfile(out.dev, "/Track_", out.ripconf().tracknumber(), '.', ext);
               //outfile = outfile + "/Track_" + std::to_string(out.rip.tracknumber) + "." + ext.at();

               trace("me:codec_rip:onctrl:output:dev:", outfile);

               g_object_set(sink, "location", outfile.at(), NULL);
            #endif

            std::string outfile = (char*)out.dev().at();
            outfile = outfile + "/Track_" + std::to_string(out.ripconf().tracknumber()) + "." + ext.at();
            trace("me:codec_rip:onctrl:output:dev:", outfile.c_str());

            g_object_set(sink, "location", outfile.c_str(), NULL);

            /* -------------------------------------------------------------- */

            gst_bin_add_many(GST_BIN(line), enc, sink, NULL);
            gst_element_link_many(src, conv, enc, sink, NULL);

            gst_element_set_state(line, GST_STATE_PLAYING);

            /* -------------------------------------------------------------- */

            break;
         }
         default:break;
      }
      trace("me:codec_rip:onctrl:pos:", in.pos().str());

      GstState st0, st1;
      st1 = st0 = GST_STATE_NULL;
      GstClockTime const timeout = 1000000000ull; // nanoseconds (1 second);
      gst_element_get_state(line, &st0, &st1, timeout);

      trace("me:codec_rip:onctrl:st0:", st0, " st:", st1);

      return 0;
   }

   /* ----------------------------------------------------------------------- */

   static gboolean cb_bus_call(GstBus *bus, GstMessage *msg, gpointer that) {

      trace("me:codec_rip:cb_bus_call");

      asrt(NULL != that);
      codec_rip_t &codec = *(codec_rip_t*)that;

      GstMessageType const type = GST_MESSAGE_TYPE(msg);

      trace("me:codec_rip:cb_bus_call:type:", gst_message_type_get_name(type));

      switch(type) {
          case GST_MESSAGE_EOS: {

              trace("me:codec_rip:cb_bus_call:EOS");

              codec.out.reason = REASON_END_OF_STREAM;

              codec.forward(codec.out); // signal owner;
              break;
          }
         case GST_MESSAGE_STATE_CHANGED: {

            GstState state0, state1;
            gst_message_parse_state_changed(msg, &state0, &state1, 0);

            trace("me:codec_rip:cb_bus_call:", GST_OBJECT_NAME(msg->src),
                                    " state:", gst_element_state_get_name(state0),
                                       " -> ", gst_element_state_get_name(state1));
            break;
         }
         case GST_MESSAGE_STREAM_STATUS: {

            GstStreamStatusType status;
            gst_message_parse_stream_status(msg, &status, 0);

            trace("me:codec_rip:cb_bus_call:", GST_OBJECT_NAME(msg->src),
                                   " status:", status,
                                          ":", streamstatus(status));
            break;
         }
         case GST_MESSAGE_INFO: {

            GError *info = 0;
            gchar  *txt  = 0;
            gst_message_parse_info(msg, &info, &txt);
            gchar  *str  = gst_error_get_message(info->domain, info->code);

            trace("me:codec_rip:cb_bus_call:INFO:element:name:", GST_OBJECT_NAME(msg->src));
            trace("me:codec_rip:cb_bus_call:msg0:", info->message);
            trace("me:codec_rip:cb_bus_call:msg1:", (str ? str : "none"));
            trace("me:codec_rip:cb_bus_call:msg2:", (txt ? txt : "none"));

            g_error_free(info);
            g_free(str);
            g_free(txt);
            break;
         }
         case GST_MESSAGE_WARNING: {

            GError *warntxt = 0;
            gchar  *txt  = 0;
            gst_message_parse_warning(msg, &warntxt, &txt);
            gchar  *str  = gst_error_get_message(warntxt->domain, warntxt->code);

            trace("me:codec_rip:cb_bus_call:WARN:element:name:", GST_OBJECT_NAME(msg->src));
            trace("me:codec_rip:cb_bus_call:msg0:", warntxt->message);
            trace("me:codec_rip:cb_bus_call:msg1:", (str ? str : "none"));
            trace("me:codec_rip:cb_bus_call:msg2:", (txt ? txt : "none"));

            g_error_free(warntxt);
            g_free(str);
            g_free(txt);
            break;
         }
         case GST_MESSAGE_ERROR: {

            GError *error = 0;
            gchar  *txt   = 0;
            gst_message_parse_error(msg, &error, &txt);
            gchar  *str   = gst_error_get_message(error->domain, error->code);

            trace("me:codec_rip:cb_bus_call:ERR:element:name:", GST_OBJECT_NAME(msg->src), " domain:", gstdomain(error->domain), " code:", error->code);
            trace("me:codec_rip:cb_bus_call:msg0:", error->message);
            trace("me:codec_rip:cb_bus_call:msg1:", (str ? str : "none"));
            trace("me:codec_rip:cb_bus_call:msg2:", (txt ? txt : "none"));

            if(GST_RESOURCE_ERROR == error->domain) {
               codec.out.reason = REASON_DEVICE_ERROR;
               codec.forward(codec.out); // signal owner;
            }

            if(GST_STREAM_ERROR == error->domain) {

                if(GST_STREAM_ERROR_FAILED == error->code) {

                    trace("me:codec_rip:cb_bus_call:REASON_PIPELINE_ERROR");
                    codec.out.reason = REASON_PIPELINE_ERROR;

                } else if(GST_STREAM_ERROR_TYPE_NOT_FOUND == error->code) {
                    trace("me:codec_rip:cb_bus_call:REASON_FORMAT_ERROR");
                    codec.out.reason = REASON_FORMAT_ERROR;
                }
                codec.forward(codec.out);
            }
            g_error_free(error);
            g_free(str);
            g_free(txt);
            break;
         }
         default: break;
      }
      return true;
   }

   /* ----------------------------------------------------------------------- */

   GMainLoop  *loop;

   // pipeline;
   GstElement *line, *src, *enc, *conv, *sink;

   /* ----------------------------------------------------------------------- */
};

/* -------------------------------------------------------------------------- */

#ifdef __cplusplus
   extern "C" {
#endif

codec_t *instance() {
   return &codec_rip_t::inst();
}

#ifdef __cplusplus
   }
#endif

/* -------------------------------------------------------------------------- */

} // me;

/* -------------------------------------------------------------------------- */

#include "1_common/mecodecif.hpp"

/* -------------------------------------------------------------------------- */

 #endif // ME_CODEC_RIP_HPP_INCLUDED

/* -------------------------------------------------------------------------- */
