/* -------------------------------------------------------------------------- */
/**
 *   @defgroup meengine mecodec_pcm.hpp
 *   @ingroup  MEngine
 *   @author   Stephan Pieper, 2013
 *
 *   Pcm streaming codec.
 */
/* -------------------------------------------------------------------------- */

#if !defined(ME_CODEC_PCM_HPP_INCLUDED)
#define ME_CODEC_PCM_HPP_INCLUDED

/* -------------------------------------------------------------------------- */

#include "1_common/megstcodec.hpp"

/* -------------------------------------------------------------------------- */

namespace me {

/* -------------------------------------------------------------------------- */

struct codec_pcm_t : public gstcodec_t,
                     public single_t<codec_pcm_t> {

   /* ----------------------------------------------------------------------- */

   friend class single_t<codec_pcm_t>;

   /* ----------------------------------------------------------------------- */

   virtual strings_t onexts() const {
      trace("codec_pcm:onexts()");
      static strings_t fts;
      if(fts.empty()) {
         fts.append(string_t("pcm"));
      }
      return fts;
   }

   virtual int_t onisaudio(string_t const &ext0) const {
      if(0 == onexts().find_f(ext0)) return -1;
      return 1; // file extension *.pcm always supports audio;
   }
   virtual int_t onisvideo(string_t const &ext0) const {
      if(0 == onexts().find_f(ext0)) return -1;
      return 0; // file extension *.pcm never supports video;
   }

   /* ----------------------------------------------------------------------- */

   codec_pcm_t() : loop(0), line(0), src(0), caps(0), filter(0),
      queue(0), conv(0), vol(0), sink(0){

      pump = false;
#ifndef ENABLE_GSTREAMER_1_0
      csource = 0;
      controller = 0;
#endif
   }

   /* ----------------------------------------------------------------------- */

   virtual void_t oninit() {

      trace("me:codec_pcm:oninit");

      /* -------------------------------------------------------------------- */
       /*gst_init() call is moved to mediaengineserver::init  to avoid the multiple calls to platform  
         Fix for NCG3D-118451*/
      // setup pipeline;

      // mainloop;
      loop = g_main_loop_new(0, false); asrt(NULL != loop);

      // pipeline;
      line = gst_pipeline_new("line"); asrt(NULL != line);

      trace("me:codec_pcm:oninit:pipeline");

      // elements;
      src = gst_element_factory_make("alsasrc", "src"); asrt(NULL != src);
      if(0 == src) {
         trace("me:codec_bt:oninit:new:alsasrc:failed");
      }

      /* -------------------------------------------------------------------- */
      // caps filter;

      filter = gst_element_factory_make("capsfilter", "filter");
      if(0 == filter) {
         trace("me:codec_bt:oninit:new:capsfilter:failed");
      }

      /* -------------------------------------------------------------------- */

      queue = gst_element_factory_make("queue", "queue"); asrt(NULL != queue);
      if(0 == queue) {
         trace("me:codec_bt:oninit:new:queue:failed");
      }

      conv = gst_element_factory_make("audioconvert", "conv"); asrt(NULL != conv);
      if(0 == conv) {
         trace("me:codec_bt:oninit:new:audioconvert:failed");
      }

      vol = gst_element_factory_make("volume", "vol"  ); asrt(NULL != vol);
      if(0 == vol) {
         trace("me:codec_bt:oninit:new:volume:failed");
      }
      else
      {
#ifndef ENABLE_GSTREAMER_1_0
          controller = gst_object_control_properties ((GObject*)vol, "volume", NULL);

          if(NULL != controller)
          {
               csource = (GstControlSource*)gst_interpolation_control_source_new();
               if(NULL != csource)
               {
                   gst_interpolation_control_source_set_interpolation_mode((GstInterpolationControlSource*)csource, GST_INTERPOLATE_LINEAR);
                   gst_controller_set_control_source (controller, "volume", csource);
               }
          }
#endif
      }
      sink = gst_element_factory_make("alsasink", "sink" ); asrt(NULL != sink);
      if(0 == sink) {
         trace("me:codec_bt:oninit:new:alsasink:failed");
      }

      /* -------------------------------------------------------------------- */
      // configure;

      g_object_set((GObject*)sink, "sync" , false, NULL);

      /* -------------------------------------------------------------------- */
      // link;

      gst_bin_add_many(GST_BIN(line), src, filter, queue, conv, vol, sink, NULL);
      gst_element_link_many(src, filter, queue, conv, vol, sink, NULL);

      /* -------------------------------------------------------------------- */
      // callbacks;

      GstBus *bus = gst_pipeline_get_bus(GST_PIPELINE(line));
      gst_bus_add_watch(bus, cb_bus_call, this);
#if defined (ENABLE_GSTREAMER_1_0)
      gst_bus_set_sync_handler(bus, cb_bus_sync, this ,NULL);//gst_bus_set_sync_handler(bus, cb_bus_sync, this) is modified to gst_bus_set_sync_handler(bus, cb_bus_sync, this, NULL)
#else
      gst_bus_set_sync_handler(bus, cb_bus_sync, this);
#endif
      gst_object_unref(bus);

      /* -------------------------------------------------------------------- */

      trace("me:codec_pcm:oninit:finished");
   }

   /* ----------------------------------------------------------------------- */

   virtual void_t mainloop() {

      /* -------------------------------------------------------------------- */

      trace("me:codec_pcm:mainloop");

      if(NULL != loop)
      {  
         g_main_loop_run(loop); // main routine;
      }

      trace("me:codec_pcm:mainloop:finished");

      /* -------------------------------------------------------------------- */

      GstBus *bus = gst_pipeline_get_bus(GST_PIPELINE(line));
#if defined (ENABLE_GSTREAMER_1_0)
      gst_bus_set_sync_handler(bus, 0, 0, NULL);//gst_bus_set_sync_handler(bus, 0, 0) is modified tp gst_bus_set_sync_handler(bus, 0, 0, NULL)
#else
      gst_bus_set_sync_handler(bus, 0, 0);
#endif
      gst_object_unref(bus);

      /* -------------------------------------------------------------------- */

      GstStateChangeReturn res = gst_element_set_state(line, GST_STATE_NULL);

      trace("me:codec_pcm:mainloop:GST_STATE_NULL:res:", res);

      /* -------------------------------------------------------------------- */
      // handle async statechange;

      int_t seconds = 0;
      while(GST_STATE_CHANGE_ASYNC == res) {

         GstState st0, st1;
         st1 = st0 = GST_STATE_NULL;
         GstClockTime const timeout = 1000000000ull; // nanoseconds (1 second);

         trace("me:codec_pcm:state:async:wait:", (int_t)timeout);

         res = gst_element_get_state(line, &st0, &st1, timeout);

         trace("me:codec_pcm:state:async:", st0, " -> ", st1, " res:", res);

         if(GST_STATE_CHANGE_ASYNC == res) {

            trace("me:codec_pcm:state:still:async:seconds:", ++seconds);
         }
      }

      /* -------------------------------------------------------------------- */
      // cleanup;

      gst_element_unlink_many((GstElement*)src, filter, queue, conv, vol, sink, NULL);

      if(caps) {
         gst_object_unref(caps);
         caps = NULL;
      }
      if(line) {
         gst_object_unref(line);
         line = NULL;
      }
      if(loop) {
         g_main_loop_unref(loop);
         loop = NULL;
      }
   }

   /* ----------------------------------------------------------------------- */

   virtual void_t onfini() {

      trace("me:codec_pcm:onfini");

      if(loop) {

         trace("me:codec_pcm:onfini:mainloop:quit");

         g_main_loop_quit(loop);
      }

      trace("me:codec_pcm:onfini:gst:deinit");
      //gst_deinit();
   }

   /* ----------------------------------------------------------------------- */

   virtual int_t onctrl(state_t const &state) {

      playstate_e const ps = state.playstate;

      trace("me:codec_pcm:onctrl:", estr2(ps));

      /* -------------------------------------------------------------------- */

      int_t const res = gstcodec_t::configlog(state.logconfig);

      trace("me:codec_pcm:onctrl:configlog:res:", res);

      /* -------------------------------------------------------------------- */

      switch(ps) {
         case PLAYSTATE_STOP:
            gst_element_set_state(line, GST_STATE_NULL);
            break;
         case PLAYSTATE_PAUSE:
            gst_element_set_state(line, GST_STATE_PAUSED);
            break;
         case PLAYSTATE_PLAY: {

            /* -------------------------------------------------------------------- */

            GstState st0 = GST_STATE_NULL;
            gst_element_get_state(line, &st0, 0, GST_CLOCK_TIME_NONE);
            if(GST_STATE_PLAYING == st0) {
               gst_element_set_state(line, GST_STATE_NULL);
            }

            /* -------------------------------------------------------------------- */
            // vol;

            flt_t volume = out.vol;
            volume /= 100;
            trace("me:codec_pcm:onctrl:vol:", volume);

            g_object_set(vol, "volume", volume, NULL); // 0.0 - 1.0;

            /* -------------------------------------------------------------- */
            // url -> input device;

            string_t url = in.url; // e.g.: url=1.pcm -> alsa: hw:1,0;
            string_t devid = url.upto(url.find_f('.') - 1);

            trace("me:codec_pcm:onctrl:play:url:", url, " input:devid:", devid);

            if(devid.isuint()) {

               string_t dev("hw:");
               dev << devid << ",0";

               trace("me:codec_pcm:onctrl:input:dev:", dev);

               g_object_set(src, "device", dev.at(), NULL);

            } else {

               string_t dev = devid;

               trace("me:codec_pcm:onctrl:input:dev:", dev);

               g_object_set(src, "device", dev.at(), NULL);
            }

            /* -------------------------------------------------------------- */
            // output device dev;

            trace("me:codec_pcm:onctrl:output:dev:", out.dev());

            g_object_set(sink, "device", out.dev().at(), NULL); // e.g. "hw:1,0";

            /* -------------------------------------------------------------- */
            // (re)create caps;

            // channels;
            if(-1 == out.fmt().channels) {

               out.fmt().channels = 2;
            }
            trace("me:codec_pcm:oninit:caps:channels:", out.fmt().channels());
            //gst_caps_set_simple(caps, "channels", G_TYPE_INT, (char*)out.fmt.channels, NULL);

            if(-1 == out.fmt().samplerate || 0 == out.fmt().samplerate) {

               out.fmt().samplerate = 44100;
            }
            trace("me:codec_pcm:oninit:caps:samplerate:", out.fmt().samplerate());
            //gst_caps_set_simple(caps, "rate", G_TYPE_INT, (char*)out.fmt.samplerate, NULL);

            if(-1 == out.fmt().samplewidth) {

               out.fmt().samplewidth = 16;
            }
            trace("me:codec_pcm:oninit:caps:samplewidth:", out.fmt().samplewidth());
            //gst_caps_set_simple(caps, "width", G_TYPE_INT, (char*)out.fmt.samplewidth, NULL);

            if(-1 == out.fmt().sampledepth) {

               out.fmt().sampledepth = 16;
            }
            trace("me:codec_pcm:oninit:caps:sampledepth:", out.fmt().sampledepth());
            //gst_caps_set_simple(caps, "depth", G_TYPE_INT, (char*)out.fmt.sampledepth, NULL);

            if(caps) {
               gst_object_unref(caps);
               caps = NULL;
            }

            caps = gst_caps_new_simple("audio/x-raw-int", "channels", G_TYPE_INT, (int)out.fmt().channels,
                                                          "rate",     G_TYPE_INT, (int)out.fmt().samplerate,
                                                          "width",    G_TYPE_INT, (int)out.fmt().samplewidth,
                                                          "depth",    G_TYPE_INT, (int)out.fmt().sampledepth,
                                                          NULL);

            // gnome local: #define G_BYTE_ORDER ...
            // #define G_LITTLE_ENDIAN 1234
            // #define G_BIG_ENDIAN    4321
            if(0 == out.fmt().endianess) {
               trace("me:codec_pcm:oninit:caps:endianness:little:", G_LITTLE_ENDIAN);
               gst_caps_set_simple(caps, "endianness", G_TYPE_INT, G_LITTLE_ENDIAN, NULL);
            }
            if(1 == out.fmt().endianess) {
               trace("me:codec_pcm:oninit:caps:endianness:big:", G_BIG_ENDIAN);
               gst_caps_set_simple(caps, "endianness", G_TYPE_INT, G_BIG_ENDIAN, NULL);
            }
            if(0 == out.fmt().signedness || 1 == out.fmt().signedness) {
               trace("me:codec_pcm:oninit:caps:signedness:", out.fmt().signedness());
               gst_caps_set_simple(caps, "signed", G_TYPE_BOOLEAN, out.fmt().signedness(), NULL);
            }

            /* ----------------------------------------------------------- */
            // attach caps to filter;

            g_object_set(G_OBJECT(filter), "caps", caps, NULL);

            /* -------------------------------------------------------------- */

            gst_element_set_state(line, GST_STATE_PLAYING);

            trace("me:codec_pcm:onctrl:format:RAMP : URL :", url, "   pos:", in.pos().str());
#ifndef ENABLE_GSTREAMER_1_0
            GValue vol_min = { 0, };
            GValue vol_max = { 0, };
            g_value_init (&vol_min, G_TYPE_DOUBLE);
            g_value_init (&vol_max, G_TYPE_DOUBLE);

            g_value_set_double (&vol_min, 0.0);
            g_value_set_double (&vol_max, 1.0);
            if(NULL != csource)
            {
                gst_interpolation_control_source_set ((GstInterpolationControlSource*)csource,0 * GST_MSECOND, &vol_min);
                gst_interpolation_control_source_set ((GstInterpolationControlSource*)csource,in.sourceswitchramp() * GST_MSECOND, &vol_max);
            }
#endif
            /* -------------------------------------------------------------- */

            break;
         }
         default:break;
      }
      trace("me:codec_pcm:onctrl:pos:", in.pos().str());

      GstState st0, st1;
      st1 = st0 = GST_STATE_NULL;
      GstClockTime const timeout = 1000000000ull; // nanoseconds (1 second);
      gst_element_get_state(line, &st0, &st1, timeout);

      trace("me:codec_pcm:onctrl:st0:", st0, " st:", st1);

      return 0;
   }

   /* ----------------------------------------------------------------------- */

   static gboolean cb_bus_call(GstBus *bus, GstMessage *msg, gpointer that) {

      trace("me:codec_pcm:cb_bus_call");

      asrt(NULL != that);
      codec_pcm_t &codec = *(codec_pcm_t*)that;

      GstMessageType const type = GST_MESSAGE_TYPE(msg);

      trace("me:codec_pcm:cb_bus_call:type:", gst_message_type_get_name(type));

      switch(type) {
         case GST_MESSAGE_STATE_CHANGED: {

            GstState state0, state1;
            gst_message_parse_state_changed(msg, &state0, &state1, 0);

            trace("me:codec_pcm:cb_bus_call:", GST_OBJECT_NAME(msg->src),
                                          " state:", gst_element_state_get_name(state0),
                                             " -> ", gst_element_state_get_name(state1));
            break;
         }
         case GST_MESSAGE_STREAM_STATUS: {

            GstStreamStatusType status;
            gst_message_parse_stream_status(msg, &status, 0);

            trace("me:codec_pcm:cb_bus_call:", GST_OBJECT_NAME(msg->src), " status:", status, ":", streamstatus(status));
            break;
         }
         case GST_MESSAGE_INFO: {

            GError *info = 0;
            gchar  *txt  = 0;
            gst_message_parse_info(msg, &info, &txt);
            gchar  *str  = gst_error_get_message(info->domain, info->code);

            trace("me:codec_pcm:cb_bus_call:INFO:element:name:", GST_OBJECT_NAME(msg->src));
            trace("me:codec_pcm:cb_bus_call:msg0:", info->message);
            trace("me:codec_pcm:cb_bus_call:msg1:", (str ? str : "none"));
            trace("me:codec_pcm:cb_bus_call:msg2:", (txt ? txt : "none"));

            g_error_free(info);
            g_free(str);
            g_free(txt);
            break;
         }
         case GST_MESSAGE_WARNING: {

            GError *warntxt = 0;
            gchar  *txt  = 0;
            gst_message_parse_warning(msg, &warntxt, &txt);
            gchar  *str  = gst_error_get_message(warntxt->domain, warntxt->code);

            trace("me:codec_pcm:cb_bus_call:WARN:element:name:", GST_OBJECT_NAME(msg->src));
            trace("me:codec_pcm:cb_bus_call:msg0:", warntxt->message);
            trace("me:codec_pcm:cb_bus_call:msg1:", (str ? str : "none"));
            trace("me:codec_pcm:cb_bus_call:msg2:", (txt ? txt : "none"));

            g_error_free(warntxt);
            g_free(str);
            g_free(txt);
            break;
         }
         case GST_MESSAGE_ERROR: {

            GError *error = 0;
            gchar  *txt   = 0;
            gst_message_parse_error(msg, &error, &txt);
            gchar  *str   = gst_error_get_message(error->domain, error->code);

            trace("me:codec_pcm:cb_bus_call:ERR:element:name:", GST_OBJECT_NAME(msg->src), " domain:", gstdomain(error->domain), " code:", error->code);
            trace("me:codec_pcm:cb_bus_call:msg0:", error->message);
            trace("me:codec_pcm:cb_bus_call:msg1:", (str ? str : "none"));
            trace("me:codec_pcm:cb_bus_call:msg2:", (txt ? txt : "none"));

            if(GST_RESOURCE_ERROR == error->domain &&
               GST_RESOURCE_ERROR_OPEN_WRITE == error->code) {

               codec.out.reason = REASON_DEVICE_ERROR;
               codec.forward(codec.out); // signal owner;
            }
            g_error_free(error);
            g_free(str);
            g_free(txt);
            break;
         }
         default: break;
      }
      return true;
   }

   /* ----------------------------------------------------------------------- */

   GMainLoop  *loop;

   GstCaps *caps;
   GstElement *filter;

   // pipeline;
   GstElement *line, *src, *queue, *conv, *vol, *sink;
#ifndef ENABLE_GSTREAMER_1_0
   GstController* controller;
   GstControlSource* csource;
#endif
   /* ----------------------------------------------------------------------- */
};

/* -------------------------------------------------------------------------- */

#ifdef __cplusplus
   extern "C" {
#endif

codec_t *instance() {
   return &codec_pcm_t::inst();
}

#ifdef __cplusplus
   }
#endif

/* -------------------------------------------------------------------------- */

} // me;

/* -------------------------------------------------------------------------- */

#include "1_common/mecodecif.hpp"

/* -------------------------------------------------------------------------- */

 #endif // ME_CODEC_PCM_HPP_INCLUDED

/* -------------------------------------------------------------------------- */
