/* -------------------------------------------------------------------------- */
/**
 *   @defgroup meengine mecodec_av.hpp
 *   @ingroup  MEngine
 *   @author   Stephan Pieper, 2013
 *
 *   AudioVideo codec (GStreamer decodebin2).
 */
/* -------------------------------------------------------------------------- */

#if !defined(ME_CODEC_AV_HPP_INCLUDED)
#define ME_CODEC_AV_HPP_INCLUDED

/* -------------------------------------------------------------------------- */

#include "1_common/megstcodec.hpp"

/* -------------------------------------------------------------------------- */

namespace me {
/* -------------------------------------------------------------------------- */

#define ME_GSTCODEC_PLAYTIME_TIMEOUT 200 // ms;

#define ME_CODEC_AV_HPP_FAST_TIMEOUT  1000
#define ME_CODEC_AV_HPP_FAST_SEEKSTEP 1000ll

/* -------------------------------------------------------------------------- */

//#define ME_CODEC_AV_HPP_ENABLE_SPECTRUM

/* -------------------------------------------------------------------------- */

//#define ME_CODEC_AV_HPP_ENABLE_SLOWMOTION

/* -------------------------------------------------------------------------- */

struct codec_av_t : public gstcodec_t,
                    public single_t<codec_av_t> {

   /* ----------------------------------------------------------------------- */

   friend class single_t<codec_av_t>;

   /* ----------------------------------------------------------------------- */

   virtual strings_t onexts() const {
      trace("codec_av:onexts()");
      static strings_t fts;
      if(fts.empty()) {
         // audio;
         fts.append(string_t("lpcm")); // 1982; Linear PCM; Compact Disc (CD-DA);
         fts.append(string_t( "iff")); // 1985; Interchange File Format; Electronic Arts;
         fts.append(string_t("aiff")); // 1988; Apple wav pendant; audio interchange file format; LPCM container;
         fts.append(string_t( "aif")); // "
         fts.append(string_t( "wav")); // 1991; MS Riff-Wave; PCM container;
         fts.append(string_t( "mp3")); // 1993;
         fts.append(string_t( "m4a")); // 1993; mp4 audio only;
         fts.append(string_t( "m4b")); // 1993; audiobooks, podcast files (can bookmark);
         fts.append(string_t( "m4p")); // 1993; DRM protected aac;
         fts.append(string_t( "m4r")); // 1993; iPhone ringtones;
         fts.append(string_t( "aac")); // 1997;
         fts.append(string_t( "wma")); // 1999;
         fts.append(string_t("flac")); // 2000; lossless audio;
         fts.append(string_t( "sbc")); // 2001; ADP (audio distributon profile) bluetooth audio low complexity subband codec;
         fts.append(string_t( "ogg")); // 2002;
         fts.append(string_t(  "aa")); // 2004; audible; proprietary DRM container for mp3;
         // video
       //fts.append(string_t( "120")); // 1984;
       //fts.append(string_t( "261")); // 1988;
         fts.append(string_t( "mov")); // 1990; Apple Quicktime;
         fts.append(string_t( "dat")); // 1991; Philips VCD (Video Compact Disc);  MPEG-1; Audio Layer II;
         fts.append(string_t( "avi")); // 1992; Microsoft Audio Video Interleave container; based on RIFF;
         fts.append(string_t("mpeg")); // 1993 (MPEG-1); 1995 (MPEG-2);
         fts.append(string_t( "mpg")); // "
         fts.append(string_t( "263")); // 1996;
         fts.append(string_t("divx")); // 1998; Microsoft mpeg hack of Jerome Rota;
         fts.append(string_t( "mp4")); // 1999; based on Quicktime;
         fts.append(string_t( "m4v")); // 1999; mp4 video only;
         fts.append(string_t( "asf")); // 2000; Microsoft Advanced Streaming/Systems Format; container for e.g. WMA/WMV;
         fts.append(string_t( "264")); // 2001;
         fts.append(string_t( "flv")); // 2003; Flash video container;
         fts.append(string_t( "vp6")); // 2003; Flash8 video;
         fts.append(string_t( "wmv")); // 2003; Microsoft video;
         fts.append(string_t( "3g2")); // 2009; 3G UMTS multimedia;
         fts.append(string_t( "3gp")); // 2009; 3G CDMA2000 multimedia;
       //fts.append(string_t( "265")); // 2013;
         fts.append(string_t( "mkv")); //2017; MKV Video format
      }
      return fts;
   }

   virtual int_t onisaudio(string_t const &ext0) const {
      strings_t const exts = onexts();
      if(0 == exts.find_f(ext0)) return -1;
      return exts.id(exts.find_f(ext0)) < 16 ? 1 : 0;
   }
   virtual int_t onisvideo(string_t const &ext0) const {
      strings_t const exts = onexts();
      if(0 == exts.find_f(ext0)) return -1;
      return 15 < exts.id(exts.find_f(ext0)) ? 1 : 0;
   }

   /* ----------------------------------------------------------------------- */

   codec_av_t() : loop(0), line(0), src(0), bin(0), aconv(0), avol(0), aspect(0), asink(0),
      vconv(0), vsink0(0),
      vtee(0), vqueue0(0), vqueue1(0), vsink1(0),
      pt_tid(0), ffwd_tid(0), frev_tid(0), slow(0), v4lcsc(0), vqueue(0), playbackspeed_check(0),is_audio_stream(0){

      pump = false;
      nullstate = false;
      stopState = false;
      ramptime = -1;
#ifndef ENABLE_GSTREAMER_1_0
      csource = 0;
      controller = 0;
#endif
   }

   /* ----------------------------------------------------------------------- */

   virtual void_t oninit() {

      /* -------------------------------------------------------------------- */

       /*gst_init() call is moved to mediaengineserver::init  to avoid the multiple calls to platform
         Fix for NCG3D-118451*/
      nullstate = false ;
      stopState = false;
      semlock.init(string_t("<", "codec_av", ".lock>"));
      /* -------------------------------------------------------------------- */
      // setup pipeline;

      // mainloop;
      loop = g_main_loop_new(0, false); asrt(NULL!=loop);

      // pipeline;
      line = gst_pipeline_new("line"); asrt(NULL!=line);

      trace("codec_av:oninit:pipeline:common");

      // common;
      src = gst_element_factory_make("filesrc", "src"); asrt(NULL!=src);
      if(0 == src) {
         trace("codec_bt:oninit:new:filesrc:failed");
      }

      #if defined(ENABLE_GSTREAMER_1_0)
         // decodebin2 is replaced with decodebin.
         bin = gst_element_factory_make("decodebin", "bin"); asrt(NULL!=bin);
      #else
         bin = gst_element_factory_make("decodebin2", "bin"); asrt(NULL!=bin);
      #endif

      gst_bin_add_many(GST_BIN(line), src, bin, (GstElement*)NULL);
      gst_element_link_many(src, bin, NULL);

      /* -------------------------------------------------------------------- */
      // audio;

      trace("codec_av:oninit:pipeline:audio");

      aconv = gst_element_factory_make("audioconvert", "aconv" ); asrt(NULL!=aconv);
      if(0 == aconv) {
         trace("codec_bt:oninit:new:audioconvert:failed");
      }

      avol = gst_element_factory_make("volume", "avol"); asrt(NULL!=avol);
      if(0 == avol) {
         trace("codec_bt:oninit:new:volume:failed");
      }
      else
      {
#ifndef ENABLE_GSTREAMER_1_0
          //To apply the ramp on source change.
          controller = gst_object_control_properties ((GObject*)avol, "volume", NULL);

          if(NULL != controller)
          {
               csource = (GstControlSource*)gst_interpolation_control_source_new();
               if(NULL != csource)
               {
                   gst_interpolation_control_source_set_interpolation_mode((GstInterpolationControlSource*)csource, GST_INTERPOLATE_LINEAR);
                   gst_controller_set_control_source (controller, "volume", csource);
               }
          }
#endif
      }

      #if defined(ME_CODEC_AV_HPP_ENABLE_SPECTRUM)
         aspect = gst_element_factory_make ("spectrum", "aspect"); asrt(NULL!=aspect);
         if(0 == aspect) {
            trace("codec_bt:oninit:new:spectrum:failed");
         }
      #endif

      asink = gst_element_factory_make("alsasink", "asink"); asrt(NULL!=asink);
      if(0 == asink) {
         trace("codec_bt:oninit:new:alsasink:failed");
      }

      /* -------------------------------------------------------------------- */
      // config;

      #if defined(ME_CODEC_AV_HPP_ENABLE_SPECTRUM)
         g_object_set((GObject*)aspect, "bands",                    20);

         //g_object_set((GObject*)aspect, "bands",                    20, // default: 128
         //                               "threshold",               -60,
         //                               "post-messages",          true,
         //                               "interval",          100000000, // nanoseconds
         //                               "message-magnitude",      true,
         //                               "message-phase",          true,
         //                               "multi-channel",         false, 0);
      #endif

      g_object_set((GObject*)asink, "sync" , false, NULL);

      #if defined(TARGET_BUILD)
         //g_object_set((GObject*)asink, "alignment-threshold", 120000000, 0);
      #endif

      /* -------------------------------------------------------------------- */
      // callbacks;

      GstBus *bus = gst_pipeline_get_bus(GST_PIPELINE(line));
      gst_bus_add_watch(bus, cb_bus_call, this);

      #if defined(ENABLE_GSTREAMER_1_0)
         gst_bus_set_sync_handler(bus, cb_bus_sync, this, NULL);
         // gst_bus_set_sync_handler(bus, cb_bus_sync, this )
         // is modified to gst_bus_set_sync_handler(bus, cb_bus_sync, this ,NULL)
      #else
         gst_bus_set_sync_handler(bus, cb_bus_sync, this);

      #endif
      gst_object_unref(bus);

      // hotlinking;
      g_signal_connect(bin, "pad-added", (GCallback)cb_pad_added, this);
      g_signal_connect (bin, "no-more-pads", (GCallback)cb_no_more_pad_added, this);

      /* -------------------------------------------------------------------- */
      // dbus;

      //dconnect();

      /* -------------------------------------------------------------------- */
      is_audio_stream = 0;

      trace("codec_av:oninit:exit");
   }

   /* ----------------------------------------------------------------------- */

   void_t init_video() {

      trace("codec_av:init_video:display:", out.display().str());

      /* -------------------------------------------------------------------- */
      // video;

      if(0 == vsink0) {

         trace("pe:codec_av:init_video");

         if(out.display().num() < 1 || 2 < out.display().num()) {

            trace("pe:codec_av:init_video:numscreens:not supported");
            asrt(0);
         }
        if((1 == in.videopropertycontrol) && (0 == v4lcsc))
        {
            v4lcsc = gst_element_factory_make("v4l_csc", "v4lcsc");

             if(v4lcsc) {

                 trace("codec_av:init_video:creating v4lcsc");
                 /* set all the video properties of the current video object*/
                 g_object_set(G_OBJECT(v4lcsc),        "brightness", (gint)out.vprops().brightness(),       NULL);
                 g_object_set(G_OBJECT(v4lcsc),               "hue", (gint)out.vprops().hue(),              NULL);
                 g_object_set(G_OBJECT(v4lcsc),        "saturation", (gint)out.vprops().saturation(),       NULL);
                 g_object_set(G_OBJECT(v4lcsc),          "contrast", (gint)out.vprops().contrast(),         NULL);
                 g_object_set(G_OBJECT(v4lcsc), "brightness-offset", (gint)out.vprops().brightnessoffset(), NULL);
                 g_object_set(G_OBJECT(v4lcsc), "saturation-offset", (gint)out.vprops().saturationoffset(), NULL);
                 g_object_set(G_OBJECT(v4lcsc),        "hue-offset", (gint)out.vprops().hueoffset(),        NULL);

                 gchar  videosource[] = "/sys/class/video4linux/video";
                 for(int count = 0; count <= 4 ; count++)
                 {
                    gchar devicepath[80];
                    snprintf(devicepath, sizeof(devicepath), "%s%d/name",videosource,count);
                    FILE* fp = fopen(devicepath, "r");
                    if(fp != 0)
                    {
                        string_t videodevicename;
                        fgets(videodevicename.at(0), 30, fp);
                        videodevicename = videodevicename.at(0);
                        trace("video device name: ", videodevicename);
                        fclose(fp);
                        int_t res = 0;
                        res = videodevicename.contains(video_memory_mapping_device);
                        if (0 == res)
                        {
                            gchar devicename [30];
                            snprintf(devicename, sizeof(devicename), "%s%d","/dev/video",count);
                            trace("pe:codec_av:init_video:devicename:",devicename);
                            g_object_set(G_OBJECT(v4lcsc), "devicename", devicename, NULL);
                            break;
                        }
                    }
                    else{
                        break;
                    }
                 }

                 if(0 == vqueue) {
                     vqueue = gst_element_factory_make("queue",  "video queue");
                     asrt(NULL!=vqueue);
                 }
                 gst_bin_add_many(GST_BIN(line), v4lcsc, vqueue, (GstElement*)NULL);
             }
         }

         vsink0 = gst_element_factory_make("gst_apx_sink", "vsink0");

         if(vsink0) { // target (project configuration);

            screen_t const& screen0 = out.display().screens()[0];

            trace("pe:codec_av:init_video:apx:vsink0:", screen0.str(), " numscreens:", out.display().screens().size());

            g_object_set(G_OBJECT(vsink0), "display-width",  screen0.rsl().width(), NULL);
            g_object_set(G_OBJECT(vsink0), "display-height", screen0.rsl().height(), NULL);
            g_object_set(G_OBJECT(vsink0), "layer-id",       screen0.layer(), NULL);
            g_object_set(G_OBJECT(vsink0), "surface-id",     screen0.srf(), NULL);

            gst_bin_add_many(GST_BIN(line), vsink0, (GstElement*)NULL);

            if(2 == out.display().screens().size()) {

               vtee    = gst_element_factory_make("tee",      "tee");
               vqueue0 = gst_element_factory_make("queue", "queue0");
               vqueue1 = gst_element_factory_make("queue", "queue1");

               vsink1 = gst_element_factory_make("gst_apx_sink", "vsink1");
               screen_t const& screen1 = *(screen_t*)&out.display().screens()[1];
               trace("codec_av:init_video:apx:vsink1:", screen1.str());

               g_object_set(G_OBJECT(vsink1), "display-width",  screen1.rsl().width(), NULL);
               g_object_set(G_OBJECT(vsink1), "display-height", screen1.rsl().height(), NULL);
               g_object_set(G_OBJECT(vsink1), "layer-id",       screen1.layer(), NULL);
               g_object_set(G_OBJECT(vsink1), "surface-id",     screen1.srf(), NULL);

               g_object_set(G_OBJECT(vqueue0), "max-size-buffers", 4, NULL);
               g_object_set(G_OBJECT(vqueue1), "max-size-buffers", 4, NULL);

               gst_bin_add_many(GST_BIN(line), vtee, vqueue0, vqueue1, vsink1, (GstElement*)NULL);

               if(v4lcsc) {
                  gst_element_link_many(v4lcsc, vqueue, vtee, NULL);
               }
               gst_element_link_many(vqueue0, vsink0, NULL);
               gst_element_link_many(vqueue1, vsink1, NULL);

               GstElementClass *klass = GST_ELEMENT_GET_CLASS(vtee);
               GstPadTemplate *tee_src_pad_template  = gst_element_class_get_pad_template(klass, "src%d");

               if(!tee_src_pad_template ) {

                  trace("codec_av:init_video:error:unable to get pad template");

               } else {

                  trace("codec_av:init_video:linking pads.");

                  GstPad *tpad0 = gst_element_request_pad(vtee, tee_src_pad_template, NULL, NULL);
                  trace("codec_av:init_video:request tee pad for vsink0:", gst_pad_get_name(tpad0));
                  GstPad *qpad0 = gst_element_get_static_pad(vqueue0, "sink");

                  GstPad *tpad1 = gst_element_request_pad(vtee, tee_src_pad_template, NULL, NULL);
                  trace("codec_av:init_video:request tee pad for vsink1:", gst_pad_get_name(tpad1));
                  GstPad *qpad1 = gst_element_get_static_pad(vqueue1, "sink");

                  if(gst_pad_link(tpad0, qpad0) != GST_PAD_LINK_OK) {

                     trace("codec_av:init_video:error:tee for v4lqueue could not be linked");
                  }

                  if(gst_pad_link(tpad1, qpad1) != GST_PAD_LINK_OK) {
                     trace("codec_av:init_video:error:tee for v4lqueue could not be linked");
                  }
                  gst_object_unref(GST_OBJECT(qpad0));
                  gst_object_unref(GST_OBJECT(qpad1));
                  gst_object_unref(GST_OBJECT(tpad0));
                  gst_object_unref(GST_OBJECT(tpad1));
               }
            }
         } else { // Ubuntu simulation;

            vconv = gst_element_factory_make("ffmpegcolorspace", "vconv"); asrt(NULL!=vconv);

            vsink0 = gst_element_factory_make("autovideosink", "vsink0");
            if(vsink0) {
               trace("codec_av:init_video:vsink:autovideosink");
            } else {
               vsink0 = gst_element_factory_make("xvimagesink", "vsink0");
               if(vsink0) {
                  trace("codec_av:init_video:vsink:xvimagesink");
               } else {
                  vsink0 = gst_element_factory_make("ximagesink", "vsink0");
                  if(vsink0) {
                     trace("codec_av:init_video:vsink:ximagesink");
                  }
               }
            }
            gst_bin_add_many(GST_BIN(line), vconv, vsink0, (GstElement*)NULL);
         }
         asrt(NULL!=vsink0);
      }
   }

   /* ----------------------------------------------------------------------- */

   virtual void_t mainloop() {

      /* -------------------------------------------------------------------- */

      trace("codec_av:mainloop");
      if(NULL != loop)
      {
         g_main_loop_run(loop); // main routine;
      }
      trace("codec_av:mainloop:finished");

      /* -------------------------------------------------------------------- */

      GstBus *bus = gst_pipeline_get_bus(GST_PIPELINE(line));

      #if defined(ENABLE_GSTREAMER_1_0)
         // gst_bus_set_sync_handler(bus, 0, 0 )
         // is modified to gst_bus_set_sync_handler(bus, 0, 0 , NULL)
         gst_bus_set_sync_handler(bus, 0, 0 , NULL);
      #else
         gst_bus_set_sync_handler(bus, 0, 0);
      #endif

      gst_object_unref(bus);

      /* -------------------------------------------------------------------- */
      semlock.enter();
      int_t res = setstate(line, GST_STATE_NULL);

      /* -------------------------------------------------------------------- */
      // handle async statechange;

      if(GST_STATE_CHANGE_ASYNC == res) {

         GstState st1;

         res = getstate(line, st1, 10);

         trace("codec_av:mainloop:getstate:res", res, -1 == res ? " timeout" : "");
      }
      nullstate = true;
      semlock.exit();
      /* -------------------------------------------------------------------- */

      stoptimers();

      /* -------------------------------------------------------------------- */
      // cleanup audio;

      // delete pipeline;
      gst_element_unlink_many((GstElement*)src, bin, NULL);

      gst_element_unlink_many(aconv, avol,
      #if defined(ME_CODEC_AV_HPP_ENABLE_SPECTRUM)
         aspect,
      #endif
      asink, NULL);

      /* -------------------------------------------------------------------- */
      // cleanup video;
      if(v4lcsc)
      {
          trace("codec_av:Cleaning v4lcsc element");
          gst_element_unlink_many(v4lcsc, vqueue, vsink0, NULL);
      }

      if(vconv) {
         gst_element_unlink_many(vconv, vsink0, NULL);
      }

      if(vsink1) {
         gst_element_unlink_many(vtee, vqueue0, vqueue1, vsink1, NULL);
      }

      /* -------------------------------------------------------------------- */

      if(line) {
         gst_object_unref(line);
         line = NULL;
      }
      if(loop) {
         g_main_loop_unref(loop);
         loop = NULL;
      }

      /* -------------------------------------------------------------------- */

      vsink0 = NULL;
      vsink1 = NULL;
      v4lcsc = NULL;
      vqueue = NULL;
      semlock.fini();
      trace("codec_av:mainloop:exit");

      /* -------------------------------------------------------------------- */
   }

   /* ----------------------------------------------------------------------- */

   virtual void_t onfini() {

      /* -------------------------------------------------------------------- */

      trace("codec_av:onfini");

      /* -------------------------------------------------------------------- */
      // dbus;

      //disolate();

      /* -------------------------------------------------------------------- */

      if(loop) {

         trace("codec_av:onfini:mainloop:quit");

         g_main_loop_quit(loop);
      }

      trace("codec_av:onfini:gst:deinit");
      //gst_deinit();
   }

   /* ----------------------------------------------------------------------- */

   virtual int_t onctrl(state_t const &state) {

      playstate_e const ps = state.playstate;

      trace("codec_av:onctrl:", estr2(ps));

      /* -------------------------------------------------------------------- */

      int_t res = gstcodec_t::configlog(state.logconfig);

      trace("codec_av:onctrl:configlog:res:", res);

      /* -------------------------------------------------------------------- */

      switch(ps) {
         case PLAYSTATE_STOP: {

            mute(avol, true);
            if(!stopState)
            {
            stopState = true;
            stoptimers();

            semlock.enter();
            setstate(line, GST_STATE_NULL);

            nullstate = true;
            semlock.exit();
            }
            break;
         }
         case PLAYSTATE_PAUSE: {

            /* -------------------------------------------------------------- */

            mute(avol, true);

            stoptimers();

            /* -------------------------------------------------------------- */
            // url;

            gchar *loc;
            g_object_get(G_OBJECT(src), "location", &loc, NULL);

            string_t const url0 = (byte_t*)loc;
            g_free(loc);
            loc = NULL;

            string_t const url1 = out.url;
            fileExt = url1.ext();
            trace("codec_av:onctrl:PAUSE:videosupport:",in.videosupport() , "extension :",fileExt);

            if(url0 != url1 || url0.empty()) {

               setstate(line, GST_STATE_READY);
               trace("codec_av:onctrl:set:url:", url0, " -> ", url1);

               g_object_set(G_OBJECT(src), "location", url1.at(), NULL);

            } else {

               trace("codec_av:onctrl:set:url:", url0, " identical");

               setstate(line, GST_STATE_PAUSED);
            }

            /* -------------------------------------------------------------- */
            // dev;

            trace("codec_av:onctrl:dev:", out.dev());

#ifdef TARGET_BUILD
            g_object_set(asink, "device", out.dev().at(), NULL); // e.g. "hw:1,0";
#else
            g_object_set(asink, "device", "default", NULL); // e.g. "hw:1,0";
#endif

            /* -------------------------------------------------------------- */

            break;
         }
         case PLAYSTATE_PLAY: {

            /* -------------------------------------------------------------- */
            // dev;

            trace("codec_av:onctrl:dev:", out.dev());
#ifdef TARGET_BUILD
            g_object_set(asink, "device", out.dev().at(), NULL); // e.g. "hw:1,0";
#else
            g_object_set(asink, "device", "default", NULL); // e.g. "hw:1,0";
#endif

            /* -------------------------------------------------------------- */
            // vol;

            flt_t volume = out.vol;
            volume /= 100;
            trace("codec_av:onctrl:vol:", volume);

            g_object_set(avol, "volume", volume, NULL); // 0.0 - 1.0;

            /* -------------------------------------------------------------- */
            // url;

            gchar *loc = NULL;
            g_object_get(G_OBJECT(src), "location", &loc, NULL);

            string_t const url0 = (byte_t*)loc;
            g_free(loc);
            string_t const url1 = out.url;
            fileExt = url1.ext();
            trace("codec_av:onctrl:PLAY:videosupport:",in.videosupport() , "extension :",fileExt);
            nullstate = false;
            stopState = false;
            if(url0.empty()){
                trace("codec_av:onctrl:set:new url:", url1);
                g_object_set(G_OBJECT(src), "location", url1.at(), NULL);
            }
            else if(url0 != url1) {
               setstate(line, GST_STATE_READY);
               trace("codec_av:onctrl:set:url:", url0, " -> ", url1);
               g_object_set(G_OBJECT(src), "location", url1.at(), NULL);

            } else {

               trace("codec_av:onctrl:set:url:", url0, " identical");
            }

            /* -------------------------------------------------------------- */

            res = setstate(line, GST_STATE_PLAYING);
            trace("codec_av:onctrl:set:url0:", url0, " url1: ",url1 , "  pos :" ,in.pos().ms());
#ifndef ENABLE_GSTREAMER_1_0
            if(url0.empty() || ((url0 == url1) && (0 < in.pos().ms())))
            {
                GValue vol_min = { 0, };
                GValue vol_max = { 0, };
                g_value_init (&vol_min, G_TYPE_DOUBLE);
                g_value_init (&vol_max, G_TYPE_DOUBLE);

                g_value_set_double (&vol_min, 0.0);
                g_value_set_double (&vol_max, 1.0);

                trace("codec_av:onctrl:RAMP applied on Volume");
                if(NULL != csource)
                {
                    gst_interpolation_control_source_set ((GstInterpolationControlSource*)csource,0 * GST_MSECOND, &vol_min);
                    gst_interpolation_control_source_set ((GstInterpolationControlSource*)csource,in.sourceswitchramp() * GST_MSECOND, &vol_max);
                }
            }
#endif
            if(-1 == in.pos().ms) {
                mute(avol, false);
            }

            if(0 == pt_tid) {
                trace("codec_av:onctrl:start:pt_tid");
                pt_tid = g_timeout_add(ME_GSTCODEC_PLAYTIME_TIMEOUT , (GSourceFunc)cb_playtime, this); // start playtime
            }
            /* -------------------------------------------------------------- */
            break;
         }

         case PLAYSTATE_SET: {

             if(v4lcsc) {
                 /* set all the video properties of the current video object*/
                 g_object_set(G_OBJECT(v4lcsc), "brightness",  (gint)out.vprops().brightness(), NULL);
                 g_object_set(G_OBJECT(v4lcsc), "hue", (gint)out.vprops().hue(), NULL);
                 g_object_set(G_OBJECT(v4lcsc), "saturation", (gint)out.vprops().saturation(), NULL);
                 g_object_set(G_OBJECT(v4lcsc), "contrast", (gint)out.vprops().contrast(), NULL);
                 g_object_set(G_OBJECT(v4lcsc), "brightness-offset", (gint)out.vprops().brightnessoffset(), NULL);
                 g_object_set(G_OBJECT(v4lcsc), "saturation-offset",  (gint)out.vprops().saturationoffset(), NULL);
                 g_object_set(G_OBJECT(v4lcsc), "hue-offset", (gint)out.vprops().hueoffset(), NULL);
             }
         }
         default:break;
      }

      return 0;
   }

   /* ----------------------------------------------------------------------- */

   void_t stoptimers() {

      if(pt_tid) {
         trace("codec_av:stoptimers:stop:pt_tid");
         g_source_remove(pt_tid); // stop playtime;
         pt_tid = 0;
      }

      if (1 == playbackspeed_check ) {
         if(1 == slow) {
           if((vsink0 || vsink1) && asink)
           {
                g_object_set((GObject*)asink, "sync" , false, NULL);
                normalmotion(line, vsink0, vsink1);
           }
           else if (asink)
           {
               normalmotion(line, asink, 0);
           }
           slow = 0;
           playbackspeed_check = 0;
         }
      }
      else if(ffwd_tid) {
            trace("codec_av:stoptimers:stop:ffwd_tid");
            g_source_remove(ffwd_tid);
            ffwd_tid = 0;
         }

      if (1 == playbackspeed_check) {
         if(-1 == slow) {
           if((vsink0 || vsink1) && asink)
           {
               g_object_set((GObject*)asink, "sync" , false, NULL);
                normalmotion(line, vsink0, vsink1);
           }
           else if (asink)
           {

               normalmotion(line, asink, 0);
           }
           slow = 0;
            playbackspeed_check = 0;
         }
      }
      else if(frev_tid) {
            trace("codec_av:stoptimers:stop:frev_tid");
            g_source_remove(frev_tid);
            frev_tid = 0;
         }

   }

  /* ----------------------------------------------------------------------- */

   static void cb_pad_added(GstElement *element, GstPad *pad, gpointer that) {

      gchar* element_name = gst_element_get_name(element);

      trace("codec_av:cb_pad_added:", string_t(element_name));
      g_free(element_name);

      asrt(NULL!=that);
      codec_av_t &c = *(codec_av_t*)that;

      #if defined(ENABLE_GSTREAMER_1_0)
         GstCaps *caps  = gst_pad_query_caps(pad,NULL); // gst_pad_get_caps(pad) is replaced with gst_pad_query_caps
      #else
         GstCaps *caps  = gst_pad_get_caps(pad);
      #endif

      GstStructure *strct = gst_caps_get_structure(caps, 0);

      string_t name0(gst_structure_get_name(strct));
      trace("codec_av:cb_pad_added:name:", name0);

      if(name0.find_f(string_t("audio"))) {

         gst_bin_add_many(GST_BIN(c.line), c.aconv, c.avol,
         #if defined(PE_CODEC_AV_HPP_ENABLE_SPECTRUM)
             c.aspect,
         #endif

         c.asink, (GstElement*)NULL);

         #if defined(ENABLE_GSTREAMER_1_0)
            GstPad *apad = gst_element_get_static_pad(c.aconv, "sink"); asrt(NULL!=apad);
            // gst_element_get_pad is replaced with gst_element_get_static_pad
         #else
            GstPad *apad = gst_element_get_pad(c.aconv, "sink"); asrt(NULL!=apad);
         #endif

         GstPadLinkReturn const res = gst_pad_link(pad, apad);

         gst_element_link_many(c.aconv, c.avol,
         #if defined(PE_CODEC_AV_HPP_ENABLE_SPECTRUM)
            c.aspect,
         #endif
         c.asink, NULL);

         trace("codec_av:cb_pad_added:audio:pad_link:res:", res);

         gst_object_unref(apad);
         c.is_audio_stream = 1;

      } else if((name0.find_f(string_t("video"))) && (c.onisvideo(c.fileExt)) && (c.in.videosupport)) {

         c.init_video();

         if(c.vconv) { // Ubuntu simulation;

            trace("codec_av:cb_pad_added:video:vconv");

            #if defined(ENABLE_GSTREAMER_1_0)
               GstPad *vpad = gst_element_get_static_pad(c.vconv, "sink"); asrt(NULL!=vpad);
               // gst_element_get_pad is replaced with gst_element_get_static_pad
            #else
               GstPad *vpad = gst_element_get_pad(c.vconv, "sink"); asrt(NULL!=vpad);
            #endif

            GstPadLinkReturn const res = gst_pad_link(pad, vpad);
            gst_object_unref(vpad);

            gst_element_link_many(c.vconv, c.vsink0, NULL);

            trace("codec_av:cb_pad_added:video:pad_link:vconv:res:", res);

         } else { // target (project configuration);

             if(0 == c.vsink1) { // single screen;

               trace("codec_av:cb_pad_added:video:single screen");

               GstPad *vpad0 = NULL;

               #if defined(ENABLE_GSTREAMER_1_0)
                    if(c.v4lcsc) {
                     trace("codec_av:cb_pad_added:pad linked with v4l_csc");
                     vpad0 = gst_element_get_static_pad(c.v4lcsc, "sink"); asrt(NULL!=vpad0);
                  } else {
                     vpad0 = gst_element_get_static_pad(c.vsink0, "sink"); asrt(NULL!=vpad0);
                  }
               #else
                 if(c.v4lcsc) {
                     trace("codec_av:cb_pad_added:pad linked with v4l_csc");
                     vpad0 = gst_element_get_pad(c.v4lcsc, "sink"); asrt(NULL!=vpad0);
                  } else {
                     vpad0 = gst_element_get_pad(c.vsink0, "sink"); asrt(NULL!=vpad0);
                  }
               #endif
               GstPadLinkReturn const res = gst_pad_link(pad, vpad0);
               gst_object_unref(vpad0);

               if(c.v4lcsc) {
                  gst_element_link_many(c.v4lcsc, c.vqueue, c.vsink0, NULL);
                  // Make sure the element are sync with elements in pipeline to make sure to receive events
                  gst_element_sync_state_with_parent (c.v4lcsc);
                  gst_element_sync_state_with_parent (c.vqueue);
                  trace("codec_av:init_video:linking v4lcsc");
               }
               trace("codec_av:cb_pad_added:video:pad_link:vsink:res:", res);
               // Make sure the element are sync with elements in pipeline to make sure to receive events
               gst_element_sync_state_with_parent (c.vsink0);
            } else { // double screen;

               trace("codec_av:cb_pad_added:video:double screen");

               GstPad *vpad1 = NULL;

               #if defined(ENABLE_GSTREAMER_1_0)
                   if(c.v4lcsc) {
                     vpad1 = gst_element_get_static_pad(c.v4lcsc, "sink"); asrt(NULL!=vpad1);
                  } else {
                     vpad1 = gst_element_get_static_pad(c.vtee, "sink"); asrt(NULL!=vpad1);
                  }
               #else
                  if(c.v4lcsc) {
                     vpad1 = gst_element_get_pad(c.v4lcsc, "sink"); asrt(NULL!=vpad1);
                  } else {
                     vpad1 = gst_element_get_pad(c.vtee, "sink"); asrt(NULL!=vpad1);
                  }
               #endif
               GstPadLinkReturn const res = gst_pad_link(pad, vpad1);
               gst_object_unref(vpad1);

               trace("codec_av:cb_pad_added:video:pad_link:tsink:res:", res);
            }
         }
      }
   }

   /* ----------------------------------------------------------------------- */

   static gboolean cb_bus_call(GstBus* /*bus*/, GstMessage *msg, gpointer that) {

       asrt(NULL!=that);
       codec_av_t &codec = *(codec_av_t*)that;

       GstMessageType const type = GST_MESSAGE_TYPE(msg);

       //trace("codec_av:cb_bus_call:type:", gst_message_type_get_name(type));

       switch(type) {
          case GST_MESSAGE_EOS: {

             trace("codec_av:cb_bus_call:EOS");

             codec.stoptimers();

             codec.out.reason = REASON_END_OF_STREAM;

             codec.forward(codec.out); // signal owner;
             break;
          }
          case GST_MESSAGE_DURATION: {

             gint64 dur0 = -1;
             GstFormat format = GST_FORMAT_UNDEFINED;
             gst_message_parse_duration(msg, &format, &dur0);
#if defined(TARGET_BUILD)
             trace("codec_av:cb_bus_call:", GST_OBJECT_NAME(msg->src),
                                   " format:", gst_format_get_name(format),
                                " duration: ", dur0);
#endif
             if((gint64)GST_CLOCK_TIME_NONE != dur0 && GST_FORMAT_TIME == format) {
                codec.out.dur().ms    = dur0 / 1000000;
                codec.out.dur().bytes = codec.duration_bytes(codec.line);
             }
             break;
          }
          case GST_MESSAGE_STATE_CHANGED: {

             GstState state0, state1;
             gst_message_parse_state_changed(msg, &state0, &state1, 0);

             trace("codec_av:cb_bus_call:", GST_OBJECT_NAME(msg->src),
                                    " state:", gst_element_state_get_name(state0),
                                       " -> ", gst_element_state_get_name(state1));
             break;
          }
          case GST_MESSAGE_STREAM_STATUS: {

             GstStreamStatusType status;
             gst_message_parse_stream_status(msg, &status, 0);

             trace("codec_av:cb_bus_call:", GST_OBJECT_NAME(msg->src), " status:", status, ":", streamstatus(status));
             break;
          }
          case GST_MESSAGE_TAG: {

             trace("codec_av:cb_bus_call:", GST_OBJECT_NAME(msg->src), " received:taglist");

             //#define ME_CODEC_AV_PARSE_TAGS
             #if defined (ME_CODEC_AV_PARSE_TAGS)

                GstTagList *tags = 0;

                gst_message_parse_tag(msg, &tags);

                gst_tag_list_foreach(tags, cb_handle_tag, 0);
                gst_tag_list_free(tags);
             #endif

             break;
          }
          case GST_MESSAGE_INFO: {

             GError *info0 = 0;
             gchar  *txt   = 0;
             gst_message_parse_info(msg, &info0, &txt);
             gchar  *msg1  = gst_error_get_message(info0->domain, info0->code);

             trace("codec_av:cb_bus_call:INFO:element:name:", GST_OBJECT_NAME(msg->src));
             trace("codec_av:cb_bus_call:msg0:", info0->message);
             trace("codec_av:cb_bus_call:msg1:", (msg1 ? msg1 : "none"));
             trace("codec_av:cb_bus_call:msg2:", (txt  ? txt  : "none"));

             g_error_free(info0);
             g_free(msg1);
             g_free(txt);
             break;
          }
          case GST_MESSAGE_WARNING: {

             GError *warntxt = 0;
             gchar  *txt  = 0;
             gst_message_parse_warning(msg, &warntxt, &txt);
             gchar  *msg1 = gst_error_get_message(warntxt->domain, warntxt->code);

             trace("codec_av:cb_bus_call:WARN:element:name:", GST_OBJECT_NAME(msg->src));
             trace("codec_av:cb_bus_call:msg0:", warntxt->message);
             trace("codec_av:cb_bus_call:msg1:", (msg1 ? msg1 : "none"));
             trace("codec_av:cb_bus_call:msg2:", (txt  ?  txt : "none"));
             trace("codec_av:cb_bus_call:msg3:", warntxt->domain);
             trace("codec_av:cb_bus_call:msg4:", warntxt->code);

             g_error_free(warntxt);
             g_free(msg1);
             g_free(txt);
             break;
          }
          case GST_MESSAGE_ERROR: {

             GError *error = 0;
             gchar  *txt   = 0;
             gst_message_parse_error(msg, &error, &txt);
             gchar  *msg1  = gst_error_get_message(error->domain, error->code);

             string_t ename = GST_OBJECT_NAME(msg->src);
             int_t    ecode = error->code;

             trace("codec_av:cb_bus_call:ERR:element:name:", ename.at(),
                                                    " domain:", gstdomain(error->domain),
                                                      " code:", ecode);

             trace("codec_av:cb_bus_call:msg0:", error->message);
             trace("codec_av:cb_bus_call:msg1:", (msg1 ? msg1 : "none"));
             trace("codec_av:cb_bus_call:msg2:", (txt  ? txt  : "none"));

             if(GST_RESOURCE_ERROR            == error->domain &&
               (GST_RESOURCE_ERROR_BUSY       == error->code ||
                GST_RESOURCE_ERROR_READ       == error->code ||
                GST_RESOURCE_ERROR_OPEN_WRITE == error->code ||
                GST_RESOURCE_ERROR_NOT_FOUND == error->code  )) {

                if(!codec.stopState)
                {
                codec.stoptimers();

                trace("codec_av:cb_bus_call:REASON_URL_ERROR");

                codec.out.reason = REASON_URL_ERROR;
                codec.forward(codec.out);
                }
                else
                {
                   trace("codec_av:GST_RESOURCE_ERROR STOP request is already under process, So ignoring the resource error   ");
                }
             }
             if(GST_STREAM_ERROR == error->domain) {

              if(!codec.stopState)
              {
                codec.stoptimers();

                if((GST_STREAM_ERROR_TYPE_NOT_FOUND == error->code) || (GST_STREAM_ERROR_WRONG_TYPE == error->code) || (GST_STREAM_ERROR_FAILED == error->code)){

                   trace("codec_av:cb_bus_call:REASON_FORMAT_ERROR");

                   codec.out.reason = REASON_FORMAT_ERROR;
                }
                codec.forward(codec.out);
              }
              else
              {
                 trace("codec_av:GST_STREAM_ERROR STOP  request is already under process, So ignoring the resource error   ");
              }
             }
             g_error_free(error);
             g_free(msg1);
             g_free(txt);
             break;
          }
          case GST_MESSAGE_ELEMENT: {

             GstStructure const *s    = gst_message_get_structure(msg);
             string_t     const  name = gst_structure_get_name(s);

             if(string_t("spectrum") == name) {

               GstClockTime endtime;
               if(!gst_structure_get_clock_time(s, "endtime", &endtime)) {
                  endtime = GST_CLOCK_TIME_NONE;
               }

               trace("codec_av:cb_bus_call:", name, " endtime:", (int_t)endtime);

               GValue const *mags = gst_structure_get_value(s, "magnitude");

               string_t spectrum;
               for(int_t i = 0; i < gst_value_list_get_size(mags); ++i) {

                  flt_t const mag = g_value_get_float(gst_value_list_get_value(mags, i));

                  spectrum << i << ":" << (int_t)mag << " ";

                  //  freq = (gdouble) ((AUDIOFREQ / 2) * i + AUDIOFREQ / 4) / spect_bands;
               }
               trace("codec_av:cb_bus_call:spectrum:", spectrum);
             }
             break;
          }
          default: break;
       }
       return true;
   }

   /* ----------------------------------------------------------------------- */

   int_t streamseek() {

       GstState st = GST_STATE_NULL;
       int_t res = -1;

       //gst_element_get_state(line, &st, 0, ME_GSTCODEC_GETSTATE_TIMEOUT);
       gst_element_get_state(line, &st, 0, 0);

       // seek works only in PLAYING or PAUSED;
       if(GST_STATE_PLAYING != st && GST_STATE_PAUSED != st) {
           trace("codec_av:streamseek:delayed:state:", gststate(st));
           thr_t::wait(30);
           return -1;
       }

       // shortcut for in state;
       pos_t &pos = in.pos();

       if(-1 != in.pos().ms()) {
           trace("codec_av:streamseek:in:pos:", in.pos().ms(), " out:pos:", out.pos().ms());
           if((0 == in.pos().ms) && (-1 == out.pos().ms))
           {
                trace("codec_av:streamseek:skip seek for new track");
                mute(avol, false);
                pos.clear();
                return -1;
           }
       }

       // check if client requests current position and prevent seek in this case;
       if(in.pos().ms() == out.pos().ms()) {
           trace("codec_av:streamseek:already there");
           return -1;
       }

      // get duration from pipeline (ms & bytes);
      dur_t &dur = out.dur();
      if(-1 == dur.ms()) {
         semlock.enter();
         if(!nullstate){
             dur.ms() = duration_ms(line);
         }
         else {
             trace("codec_av:streamseek:pipeline is in Null state");
             semlock.exit();
             return -1;
         }
         semlock.exit();
      }
      if(-1 == dur.bytes()) {
         semlock.enter();
         if(!nullstate){
             dur.bytes() = duration_bytes(line);
         }
         else {
             trace("codec_av:streamseek:pipeline is in Null state");
             semlock.exit();
             return -1;
         }
         semlock.exit();
      }

       if(pos.empty()) {
           //trace("codec_av:streamseek:pos:empty:nop state:", st);
           return -1;
       }

       gchar *loc = NULL;
       g_object_get(G_OBJECT(src), "location", &loc, NULL);

       string_t const url0 = (byte_t*)loc;
       g_free(loc);
       string_t const url1 = out.url;

       if(((0 != pos.ms()) || (url0 == url1)) && (is_stream_seekable(line))) {

           int_t next_keyframe_interval_reached = false;

           while(false == next_keyframe_interval_reached) {

               trace("codec_av:streamseek:tgtpos:", pos.str(), " dur:", dur.str());

         if(-1 != pos.pct()) {
            if(0 < dur.ms() && -1 == pos.ms()) {
               // derive in.pos.ms from out.dur.ms and in.pos.pct;
               if(100 == pos.pct()) {
                  pos.ms() = dur.ms() - 1500; // 1500ms before eos in case of 100%;
               } else {  // derive in.pos.ms from in.pos.pct;
                  pos.ms() = m1::rdt(dur.ms(), pos.pct(), 100ll);
               }
            }
            if(0 < dur.bytes() && -1 == pos.bytes()) {
               // derive in.pos.bytes from out.dur.bytes and in.pos.pct;
               if(100 == pos.pct()) {
                  pos.bytes() = dur.bytes() - 2048; // 2kByte before eof in case of 100%;
               } else {
                  pos.bytes() = m1::rdt(dur.bytes(), pos.pct(), 100ll);
               }
            }
         }
         res = -1;
         if(-1 != pos.ms()) {
             semlock.enter();
             if(!nullstate){
                 res = gst_element_seek(line, 1.0, GST_FORMAT_TIME, GST_SEEK_FLAG_FLUSH,
                    GST_SEEK_TYPE_SET, pos.ms() * 1000000,
                    GST_SEEK_TYPE_NONE, GST_CLOCK_TIME_NONE) ? 0 : -1;
             }
             else {
                 trace("codec_av:streamseek:pipeline is in Null state");
                 semlock.exit();
                 return -1;
             }
             semlock.exit();
         } else if(-1 != pos.bytes()) {
             semlock.enter();
             if(!nullstate){
                res = gst_element_seek(line, 1.0, GST_FORMAT_BYTES, GST_SEEK_FLAG_FLUSH,
                   GST_SEEK_TYPE_SET, pos.bytes(),
                   GST_SEEK_TYPE_NONE, GST_CLOCK_TIME_NONE) ? 0 : -1;
             }
             else {
                 trace("codec_av:streamseek:pipeline is in Null state");
                 semlock.exit();
                 return -1;
             }
             semlock.exit();
         } else if(-1 != pos.pct() && 0 < dur.ms()) {
            int_t const tgt = m1::rdt(dur.ms(), pos.pct(), 100ll);
            trace("codec_av:streamseek:tgt:", tgt, "ms");
            semlock.enter();
            if(!nullstate){
                res = gst_element_seek(line, 1.0, GST_FORMAT_TIME, GST_SEEK_FLAG_FLUSH,
                   GST_SEEK_TYPE_SET, tgt * 1000000,
                   GST_SEEK_TYPE_NONE, GST_CLOCK_TIME_NONE) ? 0 : -1;
            }
            else {
                trace("codec_av:streamseek:pipeline is in Null state");
                semlock.exit();
                return -1;
            }
            semlock.exit();
         } else {
            trace("codec_av:streamseek:insufficient arguments");
            res = -1;
         }

               // poll stream position; only returned when pipeline is back to PLAYING after seek;
               int_t retry = 0;
               int_t const retry_max = 100;

               int_t pos_ms = -1;
               while(-1 == pos_ms && retry ++ < retry_max) {
                   pos_ms = position_ms(line);
                   thr_t::wait(5);
                   trace("codec_av:streamseek:pos_ms:", pos_ms);

                   if(pos_ms == -1) {
                       gst_element_get_state(line, &st, 0, ME_GSTCODEC_GETSTATE_TIMEOUT);
                       if((GST_STATE_PLAYING != st) && (GST_STATE_PAUSED != st)) {
                           trace("codec_av:streamseek:error:state:", gst_element_state_get_name(st));
                           return -1;
                       }
                   }
               }
               trace("codec_av:streamseek:curpos:", pos_ms, " tgtpos:", pos.str(), " diff:", pos.ms() - pos_ms);

               if((out.speed() < 0 || out.speed() > 1)) {           // doing trick play,
                   if((pos_ms - out.pos().ms()) * out.speed() > 0) { // new position must at right direction,
                       next_keyframe_interval_reached = true;
                   }
               } else { //doing seek, any new position is acceptable,
                   next_keyframe_interval_reached = true;
               }

               if(!next_keyframe_interval_reached) {
                   // retry seek with increased distance;
                   if(0 < out.speed()) {
                       pos.ms() += 1000;
                       if(dur.ms() < pos.ms()) {
                           trace("codec_av:streamseek:forward:exit:pos:", pos.ms());
                           return res;
                       }
                   } else {
                       pos.ms() -= 1000;
                       if(pos.ms() < 0) {
                           trace("codec_av:streamseek:backward:exit:pos:", pos.ms());
                           return res;
                       }
                   }
               }
               trace("pe:codec_av:streamseek:res:", res, " next_reached:", next_keyframe_interval_reached,
                       " curpos:", pos_ms, " tgtpos:", pos.ms());
           }
       }
       mute(avol, false);

       if(0 == res) {
           trace("codec_av:streamseek:res:0:complete");
           pos.clear(); // indicate seek complete;
       }
       return res;
   }

   /* ----------------------------------------------------------------------- */
   // timer callbacks;

   static gboolean cb_playtime(gpointer *that) {

      asrt(NULL!=that);
      codec_av_t &codec = *(codec_av_t*)that;
      gboolean res = 1;

      if(NULL != codec.line) {

          /* -------------------------------------------------------------------- */
          // adjust stream position to possibly set in.pos;
          // out.dur will have been set as well after following call;
          codec.streamseek();

          /* -------------------------------------------------------------------- */
          // handle speed change (ffwd, frev);

         if(0 != codec.in.speed()) {

            codec.out.speed() = codec.in.speed();
            codec.in.speed() = ME_SPEED_NONE;

            if(1 < codec.out.speed() && 0 == codec.ffwd_tid) { // ffwd;

               if(codec.frev_tid) {
                  trace("codec_av:cb_playtime:stop:frev_tid");
                  g_source_remove(codec.frev_tid);
                  codec.frev_tid = 0;
                  trace("codec_av:cb_playtime:frev:ramptime:-1");
                  codec.ramptime = -1;
               }

               trace("codec_av:cb_playtime:start:ffwd_tid:ramp:", codec.in.ramp().str());
               if(0 < codec.in.ramp().steps().size()) {
                  trace("codec_av:cb_playtime:ffwd:ramptime:0");
                  codec.ramptime = 0;
               }

               if(codec.in.speedstate) {
                  codec.playbackspeed_check = 1;
                  if((codec.vsink0 || codec.vsink1) && (codec.asink)) {
                      g_object_set(codec.asink, "sync" , true, NULL);
                     if(0 == codec.slowmotion(codec.line, codec.out.speed, codec.vsink0, 0)) {
                        codec.slow = 1;
                        trace("me:codec_av:cb_playtime:slow:1");
                     }
                  } else if (codec.asink) {

                     if(0 == codec.slowmotion(codec.line, codec.out.speed, codec.asink, 0)) {
                        codec.slow = 1;
                        trace("me:codec_av:cb_playtime:slow:1");
                     }
                  }
               } else {
                  codec.playbackspeed_check = 0;
                  cb_ffwd((gpointer*)&codec);
                  codec.ffwd_tid = g_timeout_add(ME_CODEC_AV_HPP_FAST_TIMEOUT, (GSourceFunc)cb_ffwd, &codec);
               }
            } else if(0 > codec.out.speed && 0 == codec.frev_tid) { // frev;

               if(codec.ffwd_tid) {
                  trace("codec_av:cb_playtime:stop:ffwd_tid");
                  g_source_remove(codec.ffwd_tid);
                  codec.ffwd_tid = 0;
                  trace("codec_av:cb_playtime:ffwd:ramptime:-1");
                  codec.ramptime = -1;
               }

               trace("codec_av:cb_playtime:start:frev_tid:ramp:", codec.in.ramp().str());
               if(0 < codec.in.ramp().steps().size()) {
                  trace("codec_av:cb_playtime:frev:ramptime:0");
                  codec.ramptime = 0;
               }

               if(codec.in.speedstate) {
                  codec.playbackspeed_check = 1;
                  if((codec.vsink0 || codec.vsink1) && (codec.asink)) {
                      g_object_set(codec.asink, "sync" , true, NULL);
                     if(0 == codec.slowmotion(codec.line, (codec.out.speed), codec.vsink0, 0)) {
                        codec.slow = -1;
                        trace("me:codec_av:cb_playtime:slow:-1");
                     }
                  } else if (codec.asink) {

                     if(0 == codec.slowmotion(codec.line, (codec.out.speed), codec.asink, 0)) {
                        codec.slow = -1;
                        trace("me:codec_av:cb_playtime:slow:-1");
                     }
                  }
               } else {
                  codec.playbackspeed_check = 0;
                  cb_frev((gpointer*)&codec);
                  codec.frev_tid = g_timeout_add(ME_CODEC_AV_HPP_FAST_TIMEOUT, (GSourceFunc)cb_frev, &codec);
               }
            } else if(1 == codec.out.speed()) {

               codec.playbackspeed_check = 0;
               if(codec.in.speedstate) {
                  if(1 == codec.slow) {
                     if((codec.vsink0 || codec.vsink1) && (codec.asink)) {
                         g_object_set(codec.asink, "sync" , true, NULL);
                        codec.normalmotion(codec.line, codec.vsink0, codec.vsink1);
                     } else if (codec.asink) {

                        codec.normalmotion(codec.line, codec.asink, 0);
                     }
                     codec.slow = 0;
                  }
               } else if(codec.ffwd_tid) {
                  trace("codec_av:cb_playtime:stop:ffwd_tid");
                  g_source_remove(codec.ffwd_tid);
                  codec.ffwd_tid = 0;
               }

               if(codec.in.speedstate) {
                  if(-1 == codec.slow) {
                     if((codec.vsink0 || codec.vsink1) && (codec.asink)) {
                         g_object_set(codec.asink, "sync" , true, NULL);
                        codec.normalmotion(codec.line, codec.vsink0, codec.vsink1);
                     } else if (codec.asink) {

                        codec.normalmotion(codec.line, codec.asink, 0);
                     }
                     codec.slow = 0;
                  }
               } else if(codec.frev_tid) {
                  trace("codec_av:cb_playtime:stop:frev_tid");
                  g_source_remove(codec.frev_tid);
                  codec.frev_tid = 0;
               }
            } else {
               trace("codec_av:cb_playtime:unhandled");
            }
         }

         /* -------------------------------------------------------------------- */

         dur_t &dur = codec.out.dur();
         pos_t &pos = codec.out.pos();

         // set current out.pos;
         pos.bytes() = codec.position_bytes(codec.line);
         pos.ms()    = codec.position_ms(codec.line);

         if(-1 != dur.ms()) {
            pos.pct() = m1::rdt(pos.ms(), 100ll, dur.ms());
         } else if(-1 != dur.bytes()) {
            pos.pct() = m1::rdt(pos.bytes(), 100ll, dur.bytes());
         }
         trace("codec_av:cb_playtime:", codec.out.str());

         if(-1 != dur.ms() || -1 != dur.bytes()) {

            static string_t url0;
            static int_t ms  = 0;
            static int_t cnt = 0;

            // detect changed url;
            string_t const url1 = codec.out.url();

            if(url0 != url1) {

               trace("codec_av:cb_playtime:url0:", url0, " url1:", url1);

               url0 = url1;
               cnt = 0;
            }

            // detect blocked pipeline;
            if(ms == pos.ms()) {
               trace("codec_av:cb_playtime:", cnt, " same playtime updates");
               ++cnt;
            } else {
               ms = pos.ms();
               cnt = 0;
            }
            static int_t const maxsame = ME_GSTCODEC_PLAYTIME_TIMEOUT / 10;
            if(maxsame == cnt) {
               trace("codec_av:cb_playtime:error:max same playtime updates");
               codec.out.reason() = REASON_CODEC_ERROR;
               codec.forward(codec.out);
               cnt = 0;
            } else {
               codec.forward(codec.out);
            }
         }
      }
      else {
          res = 0;
      }
      return res;
   }

   /* ----------------------------------------------------------------------- */
   // choose current speed in ramp array from ramptime;

   static int_t curspeed(int_t const &fullspeed, ramp_t const &ramp, int_t const &ramp_time) {

      trace("codec_av:curspeed:fullspeed:", fullspeed,       " ramp:", ramp.str(),
                             " ramp_time:", ramp_time, " steps:size:", ramp.steps().size());

      //int_t cur_speed = 0; // sim4hi: commented out as variable is not used

      if(0 <= ramp_time) { // ramp active;

         int_t window = 0;
         for(int_t i = 0; i < ramp().steps().size(); ++i) {
            window += ramp.steps()[i].dur().ms();
            trace("codec_av:curspeed:window:", window);
            if(ramp_time <= window) {
               int_t const partialspeed = ramp.steps()[i].lvl() * fullspeed / 100; // partial speed;
               trace("codec_av:curspeed:partialspeed:", partialspeed);
               return partialspeed;
            }
         }
      }
      trace("codec_av:curspeed:fullspeed:", fullspeed);
      return fullspeed;
   }

   /* ----------------------------------------------------------------------- */

   static gboolean cb_ffwd(gpointer *that) {

      asrt(NULL!=that);
      codec_av_t &codec = *(codec_av_t*)that;

      int_t const pos_ms = codec.position_ms(codec.line);
      int_t const dur_ms = codec.duration_ms(codec.line);

      if(-1 == pos_ms || -1 == dur_ms) {
         trace("codec_av:cb_ffwd:pos:", pos_ms, "ms dur:", dur_ms, "ms");
         return 0;
      }

      int_t const cur_speed = curspeed(codec.out.speed(), codec.in.ramp(), codec.ramptime);

      if(cur_speed == codec.out.speed()) {
         codec.ramptime = -1; // ramp over;
      }

      int_t const ofs_ms = ME_CODEC_AV_HPP_FAST_SEEKSTEP * (cur_speed - 1); // 9000 @ 10x; // one seek step (ms);

      if(0 <= codec.ramptime) {
         codec.ramptime += ME_CODEC_AV_HPP_FAST_SEEKSTEP;// * cur_speed;
      }

      trace("codec_av:cb_ffwd:pos:", pos_ms, "ms dur:", dur_ms, "ms cur_speed:", cur_speed, " ofs:", ofs_ms, "ms ramptime:", codec.ramptime);

      int_t tgt_ms = (pos_ms + ofs_ms < dur_ms) ? pos_ms + ofs_ms : dur_ms;

      trace("codec_av:cb_ffwd:tgt:", tgt_ms, "ms");

      pos_t &pos = codec.in.pos();
      dur_t &dur = codec.out.dur();

      if(tgt_ms < dur_ms) {

         pos.ms = tgt_ms;
         codec.mute(codec.avol, true);
         codec.streamseek();
         codec.mute(codec.avol, false);

      }
      if( (tgt_ms >= dur_ms && pos.empty()) || (pos.ms() >= dur_ms) ) {  // eos
         trace("codec_av:cb_ffwd:eos");
         pos.clear();
         dur.clear();
         codec.out.reason = REASON_END_OF_STREAM;
         codec.forward(codec.out); // signal owner;
      }
      return 1;
   }

   /* ----------------------------------------------------------------------- */

   static gboolean cb_frev(gpointer *that) {

      asrt(NULL!=that);
      codec_av_t &codec = *(codec_av_t*)that;

      int_t const pos_ms = codec.position_ms(codec.line);
      int_t const dur_ms = codec.duration_ms(codec.line);

      if(-1 == pos_ms || -1 == dur_ms) {
         trace("codec_av:cb_frev:pos:", pos_ms, "ms dur:", dur_ms, "ms");
         return 0;
      }

      int_t const cur_speed = -curspeed(-codec.out.speed(), codec.in.ramp(), codec.ramptime);

      if(cur_speed == codec.out.speed()) {
         codec.ramptime = -1; // ramp over;
      }

      int_t const ofs_ms = -ME_CODEC_AV_HPP_FAST_SEEKSTEP * (cur_speed - 1) ; // 11000 @ 10x; // one seek step (ms);

      if(0 <= codec.ramptime) {
         codec.ramptime += ME_CODEC_AV_HPP_FAST_SEEKSTEP; // * -cur_speed;
      }

      trace("codec_av:cb_frev:pos:", pos_ms, "ms dur:", dur_ms, "ms cur_speed:", cur_speed, " ofs:", ofs_ms, "ms ramptime:", codec.ramptime);

      int_t tgt_ms = (0 < pos_ms - ofs_ms) ? pos_ms - ofs_ms : 0;

      trace("codec_av:cb_frev:tgt:", tgt_ms, "ms");

      pos_t &pos = codec.in.pos();
      dur_t &dur = codec.out.dur();

      if(tgt_ms) {

         pos.ms = tgt_ms;
         codec.mute(codec.avol, true);
         codec.streamseek();
         codec.mute(codec.avol, false);

      }

      if(!tgt_ms && pos.empty() || !pos.ms()){ // eos
         trace("codec_av:cb_frev:eos");
         pos.clear();
         dur.clear();
         codec.out.reason() = REASON_END_OF_STREAM;
         codec.forward(codec.out); // signal owner;
      }
      return 1;
   }

   static void cb_no_more_pad_added(GstElement *element, GstPad *pad, gpointer that) {
       asrt(NULL!=that);
       codec_av_t &c = *(codec_av_t*)that;

       trace("codec_av:no_more_pad_added callback");

       if(c.is_audio_stream)
       {
           gst_element_set_state(c.aconv, GST_STATE_PLAYING);
           gst_element_set_state(c.avol, GST_STATE_PLAYING);
           gst_element_set_state(c.asink, GST_STATE_PLAYING);
           c.is_audio_stream = 0;
       }
   }

   /* ----------------------------------------------------------------------- */

   GMainLoop  *loop;

   // common;
   GstElement *line, *src, *bin;

   // audio;
   GstElement *aconv, *avol, *aspect, *asink;

   // video;
   GstElement *vconv, *vsink0, *v4lcsc;

   // rse (optional);
   GstElement *vtee, *vqueue0, *vqueue1, *vsink1, *vqueue;

   // timers;
   guint pt_tid, ffwd_tid, frev_tid;

   int_t slow;

   int_t ramptime;
   int_t  playbackspeed_check, is_audio_stream;

   bool nullstate,stopState;
   lock_t<smp_t> semlock;
   string_t video_memory_mapping_device = "m2m";
   string_t fileExt;

#ifndef ENABLE_GSTREAMER_1_0
   GstController* controller;
   GstControlSource* csource;
#endif
   /* ----------------------------------------------------------------------- */
};

/* -------------------------------------------------------------------------- */

#ifdef __cplusplus
   extern "C" {
#endif

codec_t *instance() {
   return &codec_av_t::inst(); //lint !e64
}

#ifdef __cplusplus
   }
#endif

/* -------------------------------------------------------------------------- */

} // me;

/* -------------------------------------------------------------------------- */

#include "1_common/mecodecif.hpp"

/* -------------------------------------------------------------------------- */

 #endif // ME_CODEC_AV_HPP_INCLUDED

/* -------------------------------------------------------------------------- */
