/*
 * MediaEngine.h
 *
 *  Created on: Mar 25, 2014
 *      Author: thm3hi
 */

#ifndef MEDIAENGINE_H_
#define MEDIAENGINE_H_

#include "1_common/metrace.hpp"

#include "ThreadFactory.h"
#include "Dispatcher.h"
#include "generated/MediaEngineSM.h"

#include <map>

class MediaEngineServer : public MediaEngineSM , public TFThread//, 
                          //public single_t<MediaEngineServer>
{
public:

    //friend struct single_t<MediaEngineServer>;

    MediaEngineServer();
    ~MediaEngineServer();

    /**
     * Inits a media engine instance
     *
     * @return != 0: error, = 0: OK
     */
    int Init();
    int InitSM();
    void Create();

    void Do(int functionID, void *ptr);

    int StopEventProcessed();
    int DoneEventProcessed();
    int Configure(const tConfigString _1);
    int AnswerPing();
    int AnswerSetConfig();

    /**
     * creates the gstreamer pipeline
     *
     * param[in] PEStateString - requested state
     * @return =1: setup successfull, !=1: setup failed
     */
    int Setup(const tPEStateString PEStateString);

    /**
     * called if setup was successfull
     *
     * - calls SendEvent("PLAY");
     *
     * param[in] PEStateString - requested state
     * @return =0: no error
     */
    int SetupOk(const tPEStateString PEStateString);

    /**
     * called if an requested action failed
     *
     * - calls SMF::GetTargetState(OUT targetState)
     * - the target state changes the the action in the PEStateString state before sending
     * - sets the correct reason
     * - calls SendAnswer(PEStateString);
     *
     * param[in] PEStateString - requested state
     * @return =0: no error
     */
    int ActionFailed(const tPEStateString PEStateString);

    /**
     * Sets pipeline to play
     *
     * - calls SMF::GetTargetState(OUT targetState)
     * - the target state changes the the action in the PEStateString state before sending
     * - sets OK
     * - calls SendAnswer(PEStateString);
     *
     * param[in] PEStateString - requested state
     * @return =1: no error, =0: not successfull
     */
    int Play(const tPEStateString PEStateString);

    /**
     * discards pipeline
     *
     * - calls SMF::GetTargetState(OUT targetState)
     * - the target state changes the the action in the PEStateString state before sending
     * - sets "Pipeline Error" or similar
     * - calls SendAnswer(PEStateString);
     *
     * param[in] PEStateString - requested state
     * @return =0: no error
     */
    int FatalError(const tPEStateString PEStateString);

    /**
     * called if an requested action was successfull
     *
     * - calls SMF::GetTargetState(OUT targetState)
     * - the target state changes the the action in the PEStateString state before sending
     * - calls SendAnswer(PEStateString);
     *
     * param[in] PEStateString - requested state
     * @return =0: no error
     */
    int ActionOk(const tPEStateString PEStateString);

    /**
     * discards buffers and sends event "STOP" to goto target state stop
     *
     * - calls SendEvent("STOP");
     *
     * param[in] PEStateString - requested state
     * @return =0: no error
     */
    int PlayToStop(const tPEStateString PEStateString);

    /**
     * discards the pipeline and sends an OK to caller
     *
     * - calls SMF::GetTargetState(OUT targetState)
     * - the target state changes the the action in the PEStateString state before sending
     * - sets OK
     * - calls SendAnswer(PEStateString);
     *
     * param[in] PEStateString - requested state
     * @return =0: no error
     */
    int Stop(const tPEStateString PEStateString);

    /**
     * stored new values of the state struct to internal members (if needed)
     *
     * - calls SMF::GetTargetState(OUT targetState)
     * - the target state changes the the action in the me::state_t state before sending
     * - sets OK
     * - calls SendAnswer(state);
     *
     * param[in] PEStateString - requested state
     * @return =0: no error
     */
    int ReStop(const tPEStateString PEStateString);

    /**
     * Pauses pipeline
     *
     * param[in] PEStateString - requested state
     * @return =0: no error
     */
    int Pause(const tPEStateString PEStateString);

    /**
     * checks parameter and decides to go back to play or to pause.
     *
     * param[in] PEStateString - requested state
     * @return =1: got to pause, =0: return to play
     */
    int RePlay(const tPEStateString PEStateString);

    /**
     * clears the buffer and sends a play to re-start playing with new parameters
     *
     * - calls SendEvent("PLAY")
     *
     * param[in] PEStateString - requested state
     * @return =1: got to pause, =0: return to play
     */
    int PlayToPause(const tPEStateString PEStateString);

    /**
     * stored new values of the state struct to internal members (if needed) and may prepare a seek to internally
     *
     * - calls SMF::GetTargetState(OUT targetState)
     * - the target state changes the the action in the PEStateString state before sending
     * - sets OK
     * - calls SendAnswer(PEStateString);
     *
     * param[in] PEStateString - requested state
     * @return =0: no error
     */
    int RePause(const tPEStateString PEStateString);

    /**
     * formats the playback status to be sent to USB Control
     *
     * - calls P1.hpp communication framework to forward status to Media Player
     *
     * param[in] state - requested state
     * @return =0: ok
     */
    int ForwardPlaybackStatus(const tPEStateString PEStateString);

    /**
     * appends (reads) given URL to buffer (1MB)
     *
     * - only available in state PAUSE or PLAY
     * - given speed is taken into account
     * - sets "EOF" as reason
     * - calls SendAnswer(state);
     *
     * param[in] PEStateString - requested state
     * @return =1: no error
     */
    int Buffer(const tPEStateString PEStateString);

    /**
     * signal failure of MediaObject buffering to PlayerManager
     *
     * - do not change current PEState
     * - expect error handling strategy from PlayerManager (e.g. SKIP)
     *
     * param[in] PEStateString - requested state
     * @return =0: ok
     */
    int BufferFailed(const tPEStateString PEStateString);

    /**
     * stops buffering in codec
     *
     * param[in] PEStateString - requested state
     * @return =0: ok
     */
    int SetFlush(const tPEStateString PEStateString);

    int switchObserver(const tStateMachineName stateMachineName);

    int SeekTo(const tPEStateString PEStateString);

    int SetOutputDevice(const tAudioOutputDevice outputDevice);

    /**
     * set video properties in the av codec for video objects
     *
     * param[in] PEStateString - requested state
     * @return =0: ok
     */
    int SetVideoProperties(const tPEStateString PEStateString);

    /* interface functions for one codec */
    typedef int_t (tFnExts)(OUT byte_t *extensions, IN int_t extensionsSize);
    typedef int_t (tFnConfig)(IN byte_t *dllName, IN byte_t *serverName, IN me::state_call_t callBack, IN void_t *context, IN int_t local);
    typedef int_t (tFnInit)();
    typedef int_t (tFnFini)();
    typedef int_t (tFnCtrl)(IN me::state_t *state);
    
    /* codec storage */
    typedef struct {
        char soName[256];
        void *soHandle;
        int closed;

        string_t str() const {
           string_t s;
           s <<    "name:" << soName
             << " handle:" << (void_t*)soHandle;
           return s;
        }
        
        /* functions */
        tFnExts *exts;
        tFnConfig *config;
        tFnInit *init;
        tFnFini *fini;
        tFnCtrl *ctrl;
    } tCodecStorage;
    
    tCodecStorage *mCodec;    
    tCodecStorage mNULLCodec;
	 
    me::state_t mPEState;    
    me::state_t mNotificationState;

private:

    /* map holds all known codecs */
    typedef std::map<string, tCodecStorage> tCodecsMap;
    typedef std::map<string, tCodecStorage>::iterator tCodecsMapIt;
    tCodecsMap mCodecs;

    tStateMachineName mObservingStateMachineName;

    int mComponentId;
    int mResendPlaybackStatus;    
    int mResendNowPlayingStatus;
    int mSetVideoProperty;

    tAudioOutputDevice mOutputDevice;

    /* callback for updates from codec */
    static void_t Callback(IN me::state_t const& state, void_t *context);

    void SaveContext();
    void LoadContext();
    int GetTargetPlaybackState(tPEPlaybackState& playbackState);
};

#endif /* MEDIAENGINE_H_ */
