////////////////////////////////////////////////////////////////////////////////
/**
 * @defgroup MediaEngineClient MediaEngineClient
 * @author Matthias Thömel
 *
 * @brief Public interface for the MediaEngineClient to start and stop the process
 * @{
 */
////////////////////////////////////////////////////////////////////////////////

#ifndef _MEDIAENGINE_H_
#define _MEDIAENGINE_H_

////////////////////////////////////////////////////////////////////////////////

#include <iostream>

#include "1_common/meclientif.h"

#include "ILocalSPM.h"
#include "ThreadFactory.h"
#include "RequestResponseSM.h"

////////////////////////////////////////////////////////////////////////////////

/**
 * definition of class MediaEngineClient
 * encapsulate customer specific actions of the Mediaplayer
 */
class MediaEngineClient : public ILocalSPM, public TFThread, public me::clientif_t
{
public:

   //SPM part

   /**
    * Constructor of component.
    * This function is used by LocalSPM to store the MediaEngineClient componentID in mComponentID.
    *
    * @param[in] componentID component ID assigned by SPM
    * @return void
    */
   MediaEngineClient(tComponentID componentID);

   /**
    * Destructor of component.
    *
    * @return void
    */
   virtual ~MediaEngineClient();

   /**
    * This function is used by LocalSPM to create the MediaEngineClient.
    * @attention: running in SPM thread context
    * Inform LocalSPM that Create is ready -> CreateDone(0)
    *
    * @return void
    */
   void Create();

   /**
    * This function is used by LocalSPM to trigger the MediaEngineClient initialization.
    * @attention: running in SPM thread context
    * Inform LocalSPM that Init is ready -> InitDone(0)
    *
    * @return < 0: error, = 0: OK
    */
   tResult Init(tInitReason reason);

   /**
    * This function is used by LocalSPM to start the MediaEngineClient.
    * @attention: running in SPM thread context
    * From now on all other mediaplayer components are available.
    * Start the MediaEngineClient process.
    *
    * @return < 0: error, = 0: OK
    */
   tResult Run();

   /**
    * This function starts a user function in a thread from the thread factory's thread pool.
    * After leaving while loop used thread is stopped and released for other users
    * This holds the thread implementation of the start and supervision threads
    *
    * @return void
    */
   void Do(int functionID, void *ptr);

   /**
    * This function is used by LocalSPM to stop the MediaEngineClient.
    * @attention: running in SPM thread context
    * Set state machine to final state -> SendMessage(STOP_SM).
    * LocalSPM will be informed after STOP_SM event is processed
    *
    * @return != 0: error, = 0: OK
    */
   tResult Stop();

   /**
    * This function is used by LocalSPM to cleanup the MediaEngineClient.
    * @attention: running in SPM thread context
    * Set state machine to final state -> SendMessage(DONE).
    * LocalSPM will be informed after DONE event is processed.
    *
    * @return != 0: error, = 0: OK
    */
   tResult Done();

   /**
   * Returns answer if the component is a state machine
   *
   * @return true or false
   */
   tBoolean IsComponentSM() {return true;}; //component is a state machine

   /* ----------------------------------------------------------------------- */

   virtual void_t init();
   virtual void_t fini();

   /* ----------------------------------------------------------------------- */

   virtual int_t action(me::state_t &state);

   /* ----------------------------------------------------------------------- */

   virtual void_t answer_state(me::state_t const& state);
   virtual void_t answer_codecs(me::codecs_t &codecs);

   /* ----------------------------------------------------------------------- */

private:

   int mShutdown;
   int mMediaEngineProcessId;
   volatile int mShutdownSupervision;
   int mStartupByLocalPSM;
   int mStartupNormal;

   /* local class to handle the ping sending */
   class rrPing: public RequestResponseSM
   {
      int HandleInitRequest() {
         // TODO: use config value here
         return 0;
      }

      int HandleSuccessRequest(const char *allParameters) {

         if(allParameters && strlen_r(allParameters)) {
            me::trace("client:SupervisionThread:ping ok");
            success = 1;
         } else {
               me::trace("client:SupervisionThread:ping timeout");
            success = 0;
         }
         return 0;
      }
   public:
      int success;
   };
   rrPing mRrPing;

    void StartThread();
    void SupervisionThread();
    void ShutdownSupervision();
    void CleanUp(int const cleanSem=0);
    tResult ConfigureME();
    tResult StartME();
    void ShutDownME(int count);
    int_t StartProcess(byte_t *argv[]);
};

////////////////////////////////////////////////////////////////////////////////

#endif  //_MEDIAENGINE_H_

////////////////////////////////////////////////////////////////////////////////
