/* -------------------------------------------------------------------------- */

#include "MediaEngineClient.h"

#include "DataProvider.h"

#if defined(TARGET_BUILD)
   #include "iap2_commands.h"
#endif

#include "LocalSPM.h"
#include "Dispatcher.h"

#include <sys/types.h>
#include <sys/wait.h>
#include <fcntl.h>  // for O_* constants;

/* -------------------------------------------------------------------------- */

/*lint -save -e578 -e774 -e1401 */

#define START_THREAD 1
#define SUPERVISION_THREAD 2

#define TOLERATE_MISSING_PROCESS 0 // for test: if no media engine process is availaable: do not fatal assert, simply retry

#define MEDIA_ENGINE_BIN "/mediaengineserver_out.out"
#define MEDIA_ENGINE_START_SEM "mestart"

/* -------------------------------------------------------------------------- */

MediaEngineClient::MediaEngineClient(const tComponentID componentID) :
   ILocalSPM(componentID) {

   #if defined(TARGET_BUILD)
      et_vTraceOpen();
   #endif

   traces("client");

   trace("client:after startup");

   /* we are not in shutdown phase */
   mShutdown = 0;

   /* not started by LocalSPM right now */
   mStartupByLocalPSM = 0;
}

MediaEngineClient::~MediaEngineClient() {

   traces("client");

   #if defined(TARGET_BUILD)
      et_vTraceClose();
   #endif
}

/* -------------------------------------------------------------------------- */

void MediaEngineClient::Create() {

   traces("client");

   CreateDone(0);
}

/* -------------------------------------------------------------------------- */

tResult MediaEngineClient::Init(tInitReason reason)
{
   traces("client");
   trace ("client:reason:", (int)reason);

   return InitDone(0);
}

/* -------------------------------------------------------------------------- */

tResult MediaEngineClient::Run()
{
   traces("client");

   /* we are not in shutdown phase */
   mShutdown = 0;

   /* the LocalSPM starts us */
   mStartupByLocalPSM = 1;

   mMediaEngineProcessId = -1;
   /* start the start-thread */
   LocalSPM::GetThreadFactory().Do(this, START_THREAD, NULL);

   return 0;
}

/* -------------------------------------------------------------------------- */

void MediaEngineClient::StartThread()
{
   traces("client");

   tResult ret = MP_NO_ERROR;

   mStartupNormal = 1;
   // cleanup shared resources before startup
#ifdef VARIANT_S_FTR_ENABLE_UNITTEST
   CleanUp(1);  //RTC Bug 284655
#endif

   /* synchronize startup 1 */
   sem_t* handle = sem_open(MEDIA_ENGINE_START_SEM, O_CREAT | O_EXCL, S_IRWXU | S_IRWXG | S_IRWXO, (unsigned int)0);
   MP_FATAL_ASSERT((sem_t*)0 != handle);

   /* endless (until shutdown) loop for restarting the MediaEngine process */
   while(!mShutdown) {

      mMediaEngineProcessId = -1;
      ret = StartME();
      if(MP_ERR_MEDIA_ENGINE_STARTUP_ERROR == ret) {
         /*restart ME */
         ShutDownME(4);
         continue;
      }
      /* disable  SyncMediaEngineStartup for LSIM setup */
      #ifdef DISABLE_ME_SYNCSTARTUP
         trace("client:StartThread: waiting for ME server...");
         sleep(1);
      #else
      /* synchronize startup 3 */
      if(false == LocalSPM::GetDataProvider().SyncMediaEngineStartup()) {
         sleep(1);
      } else {
         ret = sem_wait(handle);
         MP_FATAL_ASSERT(-1 != ret);
      }
      #endif
      trace("client:StartThread:...received sync.");

      ret = ConfigureME();

      if(MP_ERR_MEDIA_ENGINE_STARTUP_ERROR == ret) {
         /*restart ME */
         ShutDownME(4);
         continue;
      }

      if(true == LocalSPM::GetDataProvider().SupervisionThreadOn()) {

         /* spawn the supervision thread */
         trace("client:StartThread:starting supervision");
         mShutdownSupervision = 1;
         LocalSPM::GetThreadFactory().Do(this, SUPERVISION_THREAD, NULL);

         /* the supervision thread sets the mShutdownSupervision value to 0 at startup */
         /* wait until the supervision thread is started */
         int retries = 40;
         while(mShutdownSupervision == 1 && retries) {
            usleep(100*1000L);
            retries--;
         }
      }

      /* if started the very first time by the Local SPM */
      if(mStartupByLocalPSM) {

         /* signal the run result to LocalSPM, it will continue startup process now */
         RunDone(ret);

         /* this is only done once, all atomated restarts do not send a RUN_READY message at all */
         mStartupByLocalPSM = 0;
      } else {

         tGeneralString param;
         LocalSPM::GetDeviceDispatcher().ParameterSTREAM_ERROR(OUT param, sizeof(param), REASON_RESTART_ERROR);
         Dispatcher().GetInstance().SendMessage("DeviceDispatcherSM::STREAM_ERROR", param);
      }

      /* wait for end of media engine process */
      trace("client:StartThread:wait for pid:", mMediaEngineProcessId, " bin:", MEDIA_ENGINE_BIN);

      if(1 < mMediaEngineProcessId) {
         if(true == LocalSPM::GetDataProvider().MediaEngineUseFork()) {

            int status = 0;
            ret = waitpid(mMediaEngineProcessId, OUT &status, IN 0);
            trace("client:StartThread:waitpid:status:", status, -1 == ret ? " failed": " succeeded");

         } else {

            // mediaengine_out.out started by rootdaemon (root)
            while(1) {
               if(-1 == kill(mMediaEngineProcessId, 0) && ESRCH == errno) {
                  trace("client:StartThread:process gone");
                  break;
               }
               usleep(500*1000L);
            }
         }
      } else {
         trace("client:StartThread:kill process by name");
         mStartupNormal = 0;
         CleanUp();
      }

      if(true == LocalSPM::GetDataProvider().SupervisionThreadOn()) {
         /* shutdown the supervision thread */
         ShutdownSupervision();
      }
      ShutDownME(0);

   } // while(!mShutdown)

    /* signal end of this thread */
    mShutdown++;

   #ifdef DISABLE_ME_SYNCSTARTUP
      sleep(1);
   #else
      /* synchronize shutdown 2 */
      if(false == LocalSPM::GetDataProvider().SyncMediaEngineStartup() || 0 == mStartupNormal) {
         sleep(1);
      } else {
         ret = sem_wait(handle);
         MP_FATAL_ASSERT(-1 != ret);
      }
   #endif

   ret = sem_close(handle);
   MP_FATAL_ASSERT(-1 != ret);
   handle = (sem_t*)0;
   ret = sem_unlink(MEDIA_ENGINE_START_SEM);
   //MP_FATAL_ASSERT(-1 != res);
   if(ret) {
      trace("client::StartThread:shutdown:error:unlink:failed.");
   }
   trace("client::StartThread:shutdown:...received sync.");

   /* send the DONE_READY signal from here because the IpcProvider cannot manage calls anymore from outsite */
   DoneDone(0);
}

/* -------------------------------------------------------------------------- */

void MediaEngineClient::ShutdownSupervision() {

   traces("client");

   /* shutdown the supervision thread */
   mShutdownSupervision = 1;
   mRrPing.SendEvent(mRrPing.SUCCESS_REQUEST, "end");
}

/* -------------------------------------------------------------------------- */

void MediaEngineClient::SupervisionThread()
{
   traces("client");

   mShutdownSupervision = 0;

   /* endless loop until the MediaEngine does not answer */
   while(!mShutdownSupervision) {

      mRrPing.success = 0;
      int res;
      res = mRrPing.DoEventAnswer("MediaEngineSM::PING", "", NULL, LocalSPM::GetDataProvider().MediaEnginePingTimeoutMs());
      if(MP_NO_ERROR != res) {
         trace("client:DoEventAnswer failed:res:", res, " err:", errorString(res));
         return;
      }

      /* should this thread end? */
      if(mShutdownSupervision) {
         trace("client:SupervisionThread:end1");
         return;
      }

      /* ping failed? */
      if(!Dispatcher::GetInstance().IsAllTransientBlocked() && !mRrPing.success) {
         trace("client:SupervisionThread:Server did not answer to ping");
         break;

      } else { // ping was successfull

         /* sleep a while and ping again later */
         int count = LocalSPM::GetDataProvider().MediaEngineTimeBetweenPingMs() / 250;
         while(!mShutdownSupervision && count) {
            usleep(250*1000L);
            count--;
         }
      }
   } // endless loop

   /* if not in shutdown mode: kill the media engine */
   if(!mShutdownSupervision) {

      /* is a valid pid given? */
      if(mMediaEngineProcessId > 1) {
         /* loop ended, that means MediaEngine is not alive anymore: kill it */
         trace("client:SupervisionThread:send kill to MediaEngine-pid:", mMediaEngineProcessId);
         mStartupNormal = 0;
         CleanUp();
      } else {
         trace("client:SupervisionThread:No valid MediaEngine-pid:", mMediaEngineProcessId);
      }
   }

   /* supervision thread ends here, a new one will be started on restarting the MediaEngine process*/
   trace("client:SupervisionThread:end2");
}

/* -------------------------------------------------------------------------- */

void MediaEngineClient::Do(int functionID, void *ptr)
{
   traces("client");
   trace ("client:Do:fid:", functionID, " ptr:", (void_t*)ptr);

   /* two tasks are running as thraeds: */
   switch(functionID) {
      case START_THREAD: // fork thread to execute MediaEngine process

         // set the threads name
         ThreadFactory::SetName("MEStart");

         StartThread();

         break;
      case SUPERVISION_THREAD: // supervision thread which pings the MediaEngine

         // set the threads name
         ThreadFactory::SetName("MESupervision");

         SupervisionThread();

         break;
      default:
         break;
   }
}

/* -------------------------------------------------------------------------- */

tResult MediaEngineClient::Stop()
{
   traces("client");

   tResult ret = MP_NO_ERROR;

   /* signal that we want to shutdown now */
   mShutdown = 1;

   /* send message to media engine to stop now */
   Dispatcher().GetInstance().SendMessage("MediaEngineSM::STOP_SM", "");

   if(1 == LocalSPM::GetDataProvider().CDRippingSupport())
   {
       Dispatcher().GetInstance().SendMessage("MediaEngineRipperSM::STOP_SM", "");
   }

   /* shutdown the supervision */
   ShutdownSupervision();

   // The STOP_DONE is sent by the MediaEngine
   //StopDone(0);

   return ret;
}

/* -------------------------------------------------------------------------- */

tResult MediaEngineClient::Done()
{
   traces("client");
   tResult ret = MP_NO_ERROR;

   /* send message to media engine to be done now */
   Dispatcher().GetInstance().SendMessage("MediaEngineSM::DONE", "");

   if(1 == LocalSPM::GetDataProvider().CDRippingSupport())
   {
       Dispatcher().GetInstance().SendMessage("MediaEngineRipperSM::DONE", "");
   }

   // The DONE_READY is sent by the StartThread
   // DoneDone(0);

   return ret;
}

/* -------------------------------------------------------------------------- */

void MediaEngineClient::CleanUp(int const cleanSem)
{
   traces("client");
   int retStatus = -1;

   if(cleanSem)
   {
       retStatus = remove("/dev/shm/sem.mestart");
       trace ("client:CleanUp: remove() for sem.mestart. retStatus:",strerror(errno));
   }
   else
   {
       if(0 < mMediaEngineProcessId)
       {
           trace ("client:CleanUp: pid of mediaengine process:", mMediaEngineProcessId);
           retStatus = kill(mMediaEngineProcessId,SIGTERM);
           trace ("client:CleanUp:kill() for me_sm_srv. retStatus:",retStatus,"statusStr:",strerror(errno));
           mMediaEngineProcessId = -1;
       }

       //Remove only matching files from /dev/shm directory
       retStatus = RemoveDirectory("/dev/shm",false,"sem.smp:");
       trace ("client:CleanUp:RemoveDirectory() for 'sem.smp:'. retStatus:", retStatus);

       retStatus = RemoveDirectory("/dev/shm",false,"shm:");
       trace ("client:CleanUp:RemoveDirectory() for 'shm:'. retStatus:", retStatus);
   }
}

tResult MediaEngineClient::ConfigureME()
{
   traces("client");

   tResult ret = MP_NO_ERROR;
   /* local class to handle the config sending */
   class rrConfig: public RequestResponseSM
   {
      int HandleInitRequest() {
         return 0;
      }

      int HandleSuccessRequest(const char *allParameters) {
         if(allParameters && strlen_r(allParameters)) {

            /* get the pid of the media engine process */
            mediaEnginePID = atoi(allParameters);
            trace("client:rrConfig:HandleSuccessRequest:sending config ok; PID=", mediaEnginePID);
            success = 1;
         } else {
            trace("client:rrConfig:HandleSuccessRequest:timeout sending config");
            success = 0;
         }
         return 0;
      }
   public:
      int success;
      int mediaEnginePID;
   };

   mMediaEngineProcessId = -1;
   /* get the media engine config values from the data provider */
   state_t st;
   LocalSPM::GetDataProvider().GetDefaultPEState(OUT st);

   /* get the media engine config values to transfer to the new process */
   tConfigString configString;
   SMF::Marshal((char *) configString, sizeof(configString), DOUBLE_MARSHAL_SEPARATOR,
           "i\
      llllllll\
      llllllll\
      llllllll\
      llllllll\
      l\
      llllllll\
      llllllll\
      llllllll\
      llllllll\
      l\
      tt\
      ll\
      lllllltlll",
      ILocalSPM::GetComponentId(),
      //      st.ramp().num(),
      st.ramp().steps()[ 0].lvl(), st.ramp().steps()[ 0].dur().ms(), st.ramp().steps()[ 1].lvl(), st.ramp().steps()[ 1].dur().ms(),
      st.ramp().steps()[ 2].lvl(), st.ramp().steps()[ 2].dur().ms(), st.ramp().steps()[ 3].lvl(), st.ramp().steps()[ 3].dur().ms(),
      st.ramp().steps()[ 4].lvl(), st.ramp().steps()[ 4].dur().ms(), st.ramp().steps()[ 5].lvl(), st.ramp().steps()[ 5].dur().ms(),
      st.ramp().steps()[ 6].lvl(), st.ramp().steps()[ 6].dur().ms(), st.ramp().steps()[ 7].lvl(), st.ramp().steps()[ 7].dur().ms(),
      st.ramp().steps()[ 8].lvl(), st.ramp().steps()[ 8].dur().ms(), st.ramp().steps()[ 9].lvl(), st.ramp().steps()[ 9].dur().ms(),
      st.ramp().steps()[10].lvl(), st.ramp().steps()[10].dur().ms(), st.ramp().steps()[11].lvl(), st.ramp().steps()[11].dur().ms(),
      st.ramp().steps()[12].lvl(), st.ramp().steps()[12].dur().ms(), st.ramp().steps()[13].lvl(), st.ramp().steps()[13].dur().ms(),
      st.ramp().steps()[14].lvl(), st.ramp().steps()[14].dur().ms(), st.ramp().steps()[15].lvl(), st.ramp().steps()[15].dur().ms(),
      st.display().num(),
      st.display().screens()[0].rsl().width(), st.display().screens()[0].rsl().height(), st.display().screens()[0].layer(), st.display().screens()[0].srf(),
      st.display().screens()[1].rsl().width(), st.display().screens()[1].rsl().height(), st.display().screens()[1].layer(), st.display().screens()[1].srf(),
      st.display().screens()[2].rsl().width(), st.display().screens()[2].rsl().height(), st.display().screens()[2].layer(), st.display().screens()[2].srf(),
      st.display().screens()[3].rsl().width(), st.display().screens()[3].rsl().height(), st.display().screens()[3].layer(), st.display().screens()[3].srf(),
      st.display().screens()[4].rsl().width(), st.display().screens()[4].rsl().height(), st.display().screens()[4].layer(), st.display().screens()[4].srf(),
      st.display().screens()[5].rsl().width(), st.display().screens()[5].rsl().height(), st.display().screens()[5].layer(), st.display().screens()[5].srf(),
      st.display().screens()[6].rsl().width(), st.display().screens()[6].rsl().height(), st.display().screens()[6].layer(), st.display().screens()[6].srf(),
      st.display().screens()[7].rsl().width(), st.display().screens()[7].rsl().height(), st.display().screens()[7].layer(), st.display().screens()[7].srf(),
      st.local(),
      LocalSPM::GetDataProvider().MediaEngineProcessPriority().c_str(),
      LocalSPM::GetDataProvider().MediaEngineProcessNiceness().c_str(),
      (tU64)LocalSPM::GetDataProvider().MediaEngineUseFork(),
      (tU64)LocalSPM::GetDataProvider().CDRippingSupport(),
      (tU64)LocalSPM::GetDataProvider().V4lCaptureWidth(),
      (tU64)LocalSPM::GetDataProvider().V4lCaptureHeight(),
      (tU64)LocalSPM::GetDataProvider().V4lFPSNumerator(),
      (tU64)LocalSPM::GetDataProvider().V4lFPSDenominator(),
      (tU64)LocalSPM::GetDataProvider().V4lMotion(),
      (tU64)LocalSPM::GetDataProvider().V4lPixFormat(),
      LocalSPM::GetDataProvider().V4lInputName().c_str(),
      (tU64)LocalSPM::GetDataProvider().RampvalueonSourceChange(),
      (tU64)LocalSPM::GetDataProvider().VideoIndexingOn(),
      (tU64)LocalSPM::GetDataProvider().SetVideoProperty()
   );

   trace("client:ConfigureME:ramp:", st.ramp().str());

   for(int_t i = 0; i < st.display().screens().size(); ++i) {

      trace("client:ConfigureME:screen:", i, ":", st.display().screens()[i].str());
   }

   /* retry-loop for sending config values */
   trace("client:ConfigureME:sending config values");
   rrConfig rrConfig;
   int retries = 0;
   rrConfig.success = 0;
   rrConfig.EnableTimerClass(TIMER_CLASS_ALL);
   retries = LocalSPM::GetDataProvider().MediaEngineConfigureRetries();
   while(retries) {

      trace("client:ConfigureME:retries:", retries);
      /* the media engine will return its PID with the configure answer */
      /* send the config data and wait for the answer (time defined in local class) */
      ret = rrConfig.DoEventAnswer("MediaEngineSM::SET_CONFIG", configString, NULL, LocalSPM::GetDataProvider().MediaEngineConfigureTimeoutMs());
      if(MP_NO_ERROR != ret) {
         trace("client:ConfigureME::DoEventAnswer failed:ret:", ret, " err:", ret, errorString(ret));
         break;
      }
      if(rrConfig.success) {
         break;
      }
      retries--;
   }

   /* if successful, get the pid of the media engine */
   if(rrConfig.success) {
      mMediaEngineProcessId = rrConfig.mediaEnginePID;
   } else {
      mMediaEngineProcessId = -1;
   }

   /* media engine did not answer to config call at all */
   if (!retries || ret) {
      trace("client:ConfigureME::did not get the config values");
      ret = MP_ERR_MEDIA_ENGINE_STARTUP_ERROR;
   }

   trace("client:ConfigureME:bin:", MEDIA_ENGINE_BIN, " pid:", mMediaEngineProcessId);
   return ret;
}

/* -------------------------------------------------------------------------- */

tResult MediaEngineClient::StartME()
{
   traces("client");

   tResult ret = MP_NO_ERROR;

   /* clean up the earlier resources and kill if any older active ME process to prevent multiple ME process run */
   CleanUp();

   string mepath;
   char *pvar = getenv("MEDIAENGINE_PATH");
   if(pvar) {
      mepath = pvar; // env;
   } else {
      #if defined(MEDIAENGINE_PATH)
         mepath = string(M1_EXPAND_AND_QUOTE(MEDIAENGINE_PATH)); // xml;
      #else
         mepath = "."; // fallback;
      #endif
   }

   mepath += MEDIA_ENGINE_BIN;

   /* start the media engine */
   /* fork this process and replace the child process by the MediaEngineProcess */
   trace("client:StartME:path:", mepath.c_str());
   char *commandStr[] = {(char *)mepath.c_str(),NULL};
   ret = StartProcess(commandStr);
   if(0 == ret)
   {
      trace("client:StartME: StartProcess() success");
   }
   else
   {
      trace("client:StartME: StartProcess() failed");
      ret = MP_ERR_MEDIA_ENGINE_STARTUP_ERROR;
   }
   return ret;
}

/* -------------------------------------------------------------------------- */

void MediaEngineClient::ShutDownME(int count)
{
   traces("client");

   /* wait a little before restarting the new media engine */
   while(!mShutdown && count) {
      usleep(250*1000L);
      count--;
   }

   if(!mShutdown) {
      mStartupNormal = 0;
      CleanUp();
      trace("client:ShutDownME:restarting:bin:", MEDIA_ENGINE_BIN, "...");
   }
}

int_t MediaEngineClient::StartProcess(byte_t *argv[])
{
    traces("client");

    if(0 == argv) {
       return -1;
    }

    pid_t pid;

    trace("Environment variable values can be set from rbcm-ccamediaplayer.service file");
    switch(pid = vfork())
    {
       case  0: /* child */;
          execv(argv[0], argv);
          _exit(-1);
       case -1: // parent:error;
          return -1;
          default: // parent:good;
          {
              trace("client:StartProcess:pid:", pid);
              return 0;
          }
    }
    return 0;
}

/* -------------------------------------------------------------------------- */

void_t MediaEngineClient::init() {
   traces("client");
}
void_t MediaEngineClient::fini() {
   traces("client");
}

/* -------------------------------------------------------------------------- */

int_t MediaEngineClient::action(me::state_t &state) {
   traces("client");
   return 0;
}

 /* -------------------------------------------------------------------------- */

void_t MediaEngineClient::answer_state(me::state_t const& state) {
   traces("client");
}
void_t MediaEngineClient::answer_codecs(me::codecs_t &codecs) {
   traces("client");
}

/* -------------------------------------------------------------------------- */
