/* -------------------------------------------------------------------------- */
/**
 *   @defgroup meprotocol meprotocol.hpp
 *   @ingroup  MEngine
 *   @author   Stephan Pieper, 2013
 *
 *   Type definitions.
 */
 /* ------------------------------------------------------------------------- */

#if !defined(ME_PROTOCOL_HPP_INCLUDED)
#define ME_PROTOCOL_HPP_INCLUDED

/* -------------------------------------------------------------------------- */

#if !defined(M1_ENV)
   #define M1_ENV M1_ENV_LINUX
#endif

//#define M1_FEAT_SMP_DEBUG
//#define M1_FEAT_THR_DEBUG
//#define M1_FEAT_SHM_DEBUG
//#define M1_FEAT_RING_DEBUG
//#define M1_FEAT_FS_DEBUG
//#define M1_FEAT_COM_DEBUG
//#define M1_FEAT_PROC_DEBUG

/* -------------------------------------------------------------------------- */

#define M1_FEAT_PRINTF

#include "m1.hpp"
using namespace m1;

/* -------------------------------------------------------------------------- */

namespace me {

   /* ----------------------------------------------------------------------- */

   #define XENUM3(X) \
      X(PLAYSTATE_NOP, = 0, "nop")\
      X(PLAYSTATE_STOP,   , "stop")\
      X(PLAYSTATE_PAUSE,  , "pause")\
      X(PLAYSTATE_PLAY,   , "play")\
      X(PLAYSTATE_SHUT,   , "shut")\
      X(PLAYSTATE_SET,    , "set")
      XENUM3_IMPL(playstate_e)
   #undef XENUM3

   /* ----------------------------------------------------------------------- */

   #define XENUM4(X) \
      X(REASON_BUFFER_AVAILABLE,   =    15, "buffer_available", " buffer_available once the data available for the fitst time")\
      X(REASON_BUFFER_UNAVAILABLE, =    14, "buffer_unavailable", "buffer unavailable when no data received from the bt")\
      X(REASON_STREAM_INPROGRESS,  =    13, "stream_inprogress",  "send respond on stream delay")\
      X(REASON_ACK_LOGCONFIG,      =    12, "ack_logconfig",      "acknowledge logconfig applied")\
      X(REASON_NONE,               =    11, "none",               "default")\
      X(REASON_ACK,                =    10, "ack",                "acknowledge")\
      X(REASON_CODECS,             =     9, "codecs",             "scan url for codecs")\
      X(REASON_IDLE,               =     8, "idle",               "out: low  cpu load")\
      X(REASON_BUSY,               =     7, "busy",               "out: high cpu load")\
      X(REASON_EMPTY,              =     6, "empty",              "out: audible playback of buffered files ended (finished)")\
      X(REASON_END_OF_STREAM,      =     5, "eos",                "out: end of stream (playback of current file ended; send next_handle)")\
      X(REASON_NEW_TRACK,          =     4, "new",                "out: request next url for buffering")\
      X(REASON_END_OF_FILE,        =     3, "eof",                "out: end of file (internal use)")\
      X(REASON_FLUSH,              =     2, "flush",              "in:  flush buffered files when current file is finished")\
      X(REASON_BUFFER,             =     1, "buffer",             "in:  buffer given url")\
      X(REASON_OK,                 =     0, "ok",                 "okay")\
      X(REASON_ERROR,              =    -1, "error",              "unspecified failure")\
      X(REASON_ACTION_ERROR,       =    -2, "error_action",       "statechange failure (e.g. stop -> pause)")\
      X(REASON_URL_ERROR,          =    -3, "error_url",          "url not available (e.g. file does not exist)")\
      X(REASON_FORMAT_ERROR,       =    -4, "error_format",       "incorrect file format (not playable)")\
      X(REASON_DRM_ERROR,          =    -5, "error_drm",          "drm protected file (not released)")\
      X(REASON_DEVICE_ERROR,       =    -6, "error_device",       "device unavailable (write error)")\
      X(REASON_READ_ERROR,         =    -7, "error_read",         "read error (e.g. damaged block)")\
      X(REASON_FATAL_READ_ERROR,   =    -8, "error_fatal_read",   "gst streaming error (former STREAM_ERROR)")\
      X(REASON_UNDERVOLTAGE_ERROR, =    -9, "error_undervoltage", "device in undervoltage (read error)")\
      X(REASON_ARGUMENT_ERROR,     =   -10, "error_argument",     "wrong argument")\
      X(REASON_CODEC_ERROR,        =   -11, "error_codec",        "codec failure")\
      X(REASON_PIPELINE_ERROR,     =   -12, "error_pipeline",     "pipeline failure")\
      X(REASON_FORK_ERROR,         =   -13, "error_fork",         "server process fork failure")\
      X(REASON_RESTART_ERROR,      =   -14, "error_restart",      "mediaengine restart after kill or reset")
      XENUM4_IMPL(reason_e)
   #undef XENUM4

   #define reason_e_format "l"
   #define reason_e_init REASON_OK

   /* ----------------------------------------------------------------------- */

   #define XENUM3(X) \
      X(TERMCMD_NONE,            = -1, "none")\
      X(TERMCMD_HELP,                , "help")\
      X(TERMCMD_GST_LOG_OFF,         , "gst_log_off")\
      X(TERMCMD_GST_LOG_ON,          , "gst_log_on")\
      X(TERMCMD_GST_LOG_LS,          , "gst_log_ls")\
      X(TERMCMD_GST_LOG_CLR,         , "gst_log_clr")\
      X(TERMCMD_GST_LOG_LVL,         , "gst_log_lvl")\
      X(TERMCMD_GST_LOG_RM,          , "gst_log_rm")\
      X(TERMCMD_GST_LOG_ADD,         , "gst_log_add")\
      X(TERMCMD_GST_LOG_FILE_TOGGLE, , "gst_log_file_toggle")\
      X(TERMCMD_GST_LOG_FILE_URL,    , "gst_log_file_url")\
      X(TERMCMD_GST_LOG_FILE_LIMIT,  , "gst_log_file_limit")\
      X(TERMCMD_LAST,                , "last")
      XENUM3_IMPL(termcmd_e)
   #undef XENUM3

   /* ----------------------------------------------------------------------- */

   // redefinition of GstDebugInfo
   // https://www.freedesktop.org/software/gstreamer-sdk/data/docs/latest/gstreamer-0.10/gstreamer-GstInfo.html#GstDebugLevel
   #define XENUM2(X) \
      X(GSTLEVEL_NONE,    "none")\
      X(GSTLEVEL_ERROR,   "error")\
      X(GSTLEVEL_WARNING, "warning")\
      X(GSTLEVEL_INFO,    "info")\
      X(GSTLEVEL_DEBUG,   "debug")\
      X(GSTLEVEL_LOG,     "log")\
      X(GSTLEVEL_FIXME,   "fixme")\
      X(GSTLEVEL_TRACE,   "trace")\
      X(GSTLEVEL_MEMDUMP, "memdump")\
      X(GSTLEVEL_COUNT,   "count")
      XENUM2_IMPL(gstlevel_e)
   #undef XENUM2

   /* ----------------------------------------------------------------------- */

   #define XENUM3(X) \
      X(ME_SPEED_NONE,   =   0, "frev")\
      X(ME_SPEED_NORMAL, =   1, "normal")\
      X(ME_SPEED_FFWD10, =  10, "ffwd10")\
      X(ME_SPEED_FFWD20, =  20, "ffwd20")\
      X(ME_SPEED_FFWD3200, = 3200, "ffwd3200")\
      X(ME_SPEED_FREV10, = -10, "frev10")\
      X(ME_SPEED_FREV20, = -20, "frev20")\
      X(ME_SPEED_FREV3200, = -3200, "frev3200")\
      X(ME_SPEED_LAST,        , "last")
      XENUM3_IMPL(speed_e)
   #undef XENUM3

   #define speed_e_format "l"
   #define speed_e_init ME_SPEED_NORMAL

   /* ----------------------------------------------------------------------- */

   struct pos_t : nif_t<pos_t> {
      static string_t name() {
         return "pos_t";
      }
      pos_t() : bytes(-1), pct(-1), ms(-1) {
      }
      int_t empty() const {
         return (-1 == bytes() && -1 == pct() && -1 == ms()) ? 1 : 0;
      }
      void_t clear() {
         bytes() = pct() = ms() = -1;
      }
      string_t str() const {
         return string_t(bytes(), "b:", pct(), "%:", ms(), "m");
      }
      if_t<int_t> bytes;
      if_t<int_t>   pct;
      if_t<int_t>    ms;
      nif_t<>      done;
   };

   /* ----------------------------------------------------------------------- */

   struct dur_t : nif_t<dur_t> {
      static string_t name() {
         return "dur_t";
      }
      dur_t() : bytes(-1), ms(-1) {
      }
      int_t empty() const {
         return (-1 == bytes() && -1 == ms()) ? 1 : 0;
      }
      void_t clear() {
         bytes() = -1;
         ms()    = -1;
      }
      string_t str() const {
         return string_t(bytes(), "b:", ms(), "m");
      }
      if_t<int_t> bytes;
      if_t<int_t>    ms;
      nif_t<>      done;
   };

   /* ----------------------------------------------------------------------- */
   // logpattern_t;

   struct logpattern_t : nif_t<logpattern_t> {
      static string_t name() {
         return "logpattern_t";
      }
      logpattern_t() : lvl(GSTLEVEL_NONE) {
      }
      int_t match(string_t const &pat0) const {
         return 0 == pat0.find_f(pat) ? 0 : 1;
      }
      string_t str() const {
         return string_t("lvl:", estr2((gstlevel_e)lvl()),
                        " pat:", pat());
      }
      if_t<int_t>    lvl;
      if_t<string_t> pat;
      nif_t<>       done;
   };

   /* ----------------------------------------------------------------------- */
   // logfile_t;

   #define XENUM3(X) \
      X(TOGGLE_NONE, = -1, "none")\
      X(TOGGLE_OFF,  =  0, "off")\
      X(TOGGLE_ON,   =  1, "on")\
      X(TOGGLE_LAST,     , "last")
      XENUM3_IMPL(toggle_e)
   #undef XENUM3

   struct logfile_t : nif_t<logfile_t> {
      static string_t name() {
         return "logfile_t";
      }
      logfile_t() : tog(TOGGLE_NONE), limit(-1) {
      }
      string_t str() const {
         return string_t("tog:", tog(),
                      " limit:", limit(),
                        " url:", url());
      }
      int_t empty() const {
         return (TOGGLE_NONE == tog   &&
                          -1 == limit &&
                          url().empty());
      }
      if_t<toggle_e>  tog;
      if_t<int_t>   limit;
      if_t<string_t>  url;
      nif_t<>        done;
   };

   /* ----------------------------------------------------------------------- */
   // logconfig_t;

   struct logconfig_t : nif_t<logconfig_t> {
      static string_t name() {
         return "logconfig_t";
      }
      logconfig_t() : tog(TOGGLE_NONE), lvl(-1) {
      }
      int_t match(string_t const pat) {
         int_t respats = 0;
         for(auto r = pats().range(); r.x; ++r) {
            if(r.x->pat().find_f(pat)) respats = 1;
         }
         if(0 == respats) return 0;
         int_t resdels = 0;
         for(auto r = dels().range(); r.x; ++r) {
            if(r.x->pat().find_f(pat)) resdels = 1;
         }
         if(1 == resdels) return 0;
         return 1;
      }
      int_t empty() const {
         return (TOGGLE_NONE == tog &&
                 0 == lvl &&
                 pats().empty() &&
                 dels().empty() &&
                 file().empty());
      }
      int_t filter() const {
         return (pats().empty() &&
                 dels().empty());
      }
      string_t str() const {
         return string_t("tog:", tog(),
                        " lvl:", lvl(),
                       " pats:", pats().size(),
                       " dels:", dels().size(),
                       " file:", file().str());
      }
      if_t<toggle_e>       tog;
      if_t<int_t>          lvl;
      if_t<vector_t<logpattern_t>> mutable pats;
      if_t<vector_t<logpattern_t>> mutable dels;
      if_t<logfile_t>     file;
      nif_t<>             done;
   };

   /* ----------------------------------------------------------------------- */
   // step_t;

   struct step_t : nif_t<step_t> {
      static string_t name() {
         return "step_t";
      }
      step_t() : lvl(-1) {
      }
      int_t empty() const {
         return dur().empty() && (-1 == lvl());
      }
      string_t str() const {
         return string_t("dur:", dur().str(), " lvl:", lvl(), '%');
      }
      if_t<dur_t>  dur;
      if_t<int_t>  lvl; // percentage of speed
      nif_t<>     done;
   };

   /* ----------------------------------------------------------------------- */
   // ramp_t;

   struct ramp_t : nif_t<ramp_t> {
      static string_t name() {
         return "ramp_t";
      }
      ramp_t() {
      }
      string_t str() const {
         string_t s;
         for(int_t i = 0; i < steps().size(); ++i) {
            if(false == steps()[i].empty()) {
               s.cat(i, ':', steps()[i].str(), ' ');
            }
         }
         return s;
      }
      if_t<vector_t<step_t>> mutable steps;
      nif_t<>                         done;
   };

   /* ----------------------------------------------------------------------- */
   // resolution;

   struct rsl_t : nif_t<rsl_t> { // resolution;
      static string_t name() {
         return "rsl_t";
      }
      rsl_t() : width(-1), height(-1) {
      }
      string_t str() const {
         return string_t("width:", width(),
                       " height:", height());
      }
      if_t<int_t>  width;
      if_t<int_t> height;
      nif_t<>       done;
   };

   /* ----------------------------------------------------------------------- */
   // screen;

   struct screen_t : nif_t<screen_t> {
      static string_t name() {
         return "screen_t";
      }
      screen_t() : layer(-1), srf(-1) {
      }
      string_t str() const {
         return string_t("rsl:", rsl().str(),
                      " layer:", layer(),
                        " srf:", srf());
      }
      if_t<rsl_t>   rsl;
      if_t<int_t> layer;
      if_t<int_t>   srf;
      nif_t<>      done;
   };

   /* ----------------------------------------------------------------------- */
   // display;

   struct display_t : nif_t<display_t> {
      static string_t name() {
         return "display_t";
      }
      display_t() : num(-1) {
      }
      string_t str() const {
         string_t s;
         for(int_t i = 0; i < screens().size(); ++i) {
            s.cat(i, ':', screens()[i].str());
            if(i < screens().size() - 1) s.cat(' ');
         }
         return s;
      }
      if_t<vector_t<screen_t>> mutable screens;
      if_t<int_t>                          num; // valid screens;
      nif_t<>                             done;
   };

   /* ----------------------------------------------------------------------- */
   // sample format;

   #define XENUM3(X) \
      X(ENDIANESS_NONE, = -1, "none")\
      X(ENDIANESS_LE,   =  0, "LE")\
      X(ENDIANESS_BE,   =  1, "BE")\
      X(ENDIANESS_LAST,     , "last")
      XENUM3_IMPL(endianess_e)
   #undef XENUM3

   #define XENUM3(X) \
      X(SIGNEDNESS_NONE,     = -1, "none")\
      X(SIGNEDNESS_UNSIGNED, =  0, "unsigned")\
      X(SIGNEDNESS_SIGNED,   =  1, "signed")\
      X(SIGNEDNESS_LAST,         , "last")
      XENUM3_IMPL(signedness_e)
   #undef XENUM3

   // bluetooth format;
   #define XENUM3(X) \
      X(BTF_NONE, = -1, "none")\
      X(BTF_SBC,  =  0, "sbc")\
      X(BTF_MP3,  =  1, "mp3")\
      X(BTF_AAC,  =  2, "aac")\
      X(BTF_LAST,     , "last")
      XENUM3_IMPL(btf_e)
   #undef XENUM3

   typedef int_t channels_i;

   typedef int_t samplerate_i;
   #define samplerate_i_format "l"
   #define samplerate_i_init 0
   #define samplerate_i_none ((samplerate_i)-1)

   typedef int_t samplewidth_i;
   typedef int_t sampledepth_i;

   struct fmt_t : nif_t<fmt_t> {
      static string_t name() {
         return "fmt_t";
      }
      fmt_t() : channels(-1), samplerate(-1), samplewidth(-1), sampledepth(-1),
         endianess(ENDIANESS_NONE), signedness(SIGNEDNESS_NONE), btf(BTF_NONE) {
      }
      string_t str() const {
         return string_t("chan:", channels(),
                        " rate:", samplerate(),
                       " width:", samplewidth(),
                       " depth:", sampledepth(),
                      " endian:", endianess(),
                        " sign:", signedness());
      }
      if_t<channels_i>       channels;
      if_t<samplerate_i>   samplerate;
      if_t<samplewidth_i> samplewidth;
      if_t<sampledepth_i> sampledepth;
      if_t<endianess_e>     endianess;
      if_t<signedness_e>   signedness;
      if_t<btf_e>                 btf;
      nif_t<>                    done;
   };

   /* ----------------------------------------------------------------------- */

   struct vprops_t : nif_t<vprops_t> {
      static string_t name() {
         return "vprops_t";
      }
      vprops_t() : on(0),
          brightness(100), hue(0), saturation(100),
          contrast(100), brightnessoffset(0),
          saturationoffset(0), hueoffset(0) {
      }
      string_t str() const {
         return string_t("on:", on(),
                " brightness:", brightness(),
                       " hue:", hue(),
                " saturation:", saturation(),
                  " contrast:", contrast(),
          " brightnessoffset:", brightnessoffset(),
          " saturationoffset:", saturationoffset(),
                 " hueoffset:", hueoffset());
      }
      if_t<int_t>               on;
      if_t<int_t>       brightness; // tU8;
      if_t<int_t>              hue; // tS16;
      if_t<int_t>       saturation; // tU8
      if_t<int_t>         contrast; // tU8;
      if_t<int_t> brightnessoffset; // tS16;
      if_t<int_t> saturationoffset; // tS16;
      if_t<int_t>        hueoffset; // tU8;
      nif_t<>                 done;
   };
#define brightness_i_none 0
#define hue_i_none 0
#define saturation_i_none 0
#define contrast_i_none 0
#define brightnessoffset_i_none 0
#define saturationoffset_i_none 0
#define hueoffset_i_none 0
   /* ----------------------------------------------------------------------- */

   #if defined(TARGET_BUILD)
      #define MP3_DEC "mfw_mp3encoder"
   #else
      #define MP3_DEC "lame"
   #endif

   #define XENUM4(X) \
      X(ME_EF_MP3,       = 0x00,   "mp3", MP3_DEC)\
      X(ME_EF_WAV,       = 0x01,   "wav", "?")\
      X(ME_EF_AAC,       = 0x02,   "aac", "?")\
      X(ME_EF_M4A,       = 0x03,   "m4a", "?")\
      X(ME_EF_WMA,       = 0x04,   "wma", "?")\
      X(ME_EF_OGG,       = 0x05,   "ogg", "?")\
      X(ME_EF_FLAC,      = 0x06,  "flac", "?")\
      X(ME_EF_AIF,       = 0x07,   "aif", "?")\
      X(ME_EF_AIFF,      = 0x08,  "aiff", "?")\
      X(ME_EF_UNDEFINED, = 0x99, "undef", "?")
      XENUM4_IMPL(encfmt_e)
   #undef XENUM4

   #define XENUM3(X) \
      X(ME_EQ_LOW,       =  32, "EQ_LOW")\
      X(ME_EQ_FM,        =  96, "EQ_FM")\
      X(ME_EQ_CD,        = 128, "EQ_CD")\
      X(ME_EQ_HIGH,      = 320, "EQ_HIGH")\
      X(ME_EQ_UNDEFINED, = 999, "EQ_UNDEFINED")
      XENUM3_IMPL(encqlt_e)
   #undef XENUM3

   struct ripconf_t : nif_t<ripconf_t> {
      static string_t name() {
         return "ripconf_t";
      }
      ripconf_t() : tracknumber(1), encodeformat(ME_EF_MP3),
         encodequality(ME_EQ_FM) {
      }
      string_t str() const {
         return string_t("tracknumer:", tracknumber(),
                      " encodeformat:", estr2(encodeformat),
                     " encodequality:", estr2(encodequality));
      }
      if_t<int_t>    tracknumber;
      if_t<encfmt_e> encodeformat;
      if_t<encqlt_e> encodequality;
      nif_t<>        done;
   };

   /* ----------------------------------------------------------------------- */

   // process priority & niceness boundaries and their mapped values in top;
   // chrt -r -p 99 pid   ->   top:  RT
   // chrt -r -p 98 pid   ->   top: -99
   // ...
   // chrt -r -p  1 pid   ->   top:  -2
   //
   // renice -20 -p pid   ->   top: -20
   // ...
   // renice  19 -p pid   ->   top:  19

   struct conf_t : nif_t<conf_t> {
      static string_t name() {
         return "conf_t";
      }
      conf_t() : prio("30"), // displays as -31 in top;
                 nice("-4"), // displays as  -4 in top;
                 fork(1) {
      }
      string_t str() const {
         return string_t("prio:", prio(),
                        " nice:", nice(),
                        " fork:", fork());
      }
      if_t<string_t> prio; // process prioriry;
      if_t<string_t> nice; // process niceness;
      if_t<int_t>    fork; // 1 = startup via fork;
                           // 0 = startup via system call;
      nif_t<>        done;
   };

   /* ----------------------------------------------------------------------- */

   struct v4lprop_t : nif_t<v4lprop_t> { // dvd input v4l capture properies ;
      static string_t name() {
         return "v4lprop_t";
      }
      v4lprop_t() : capturewidth(-1), captureheight(-1),fpsnumerator(-1),fpsdenominator(-1),motion(-1),pixformat(10),v4linput("") {
      }
      string_t str() const {
          return string_t("capturewidth:", capturewidth(),
                  " captureheight:", captureheight(),
                  " fpsnumerator:", fpsnumerator(),
                  " fpsdenominator:", fpsdenominator(),
                  " motion:", motion(),
                  " pixformat:", pixformat(),
                  " v4linput:",v4linput());
      }
      if_t<int_t>  capturewidth;
      if_t<int_t>  captureheight;
      if_t<int_t>  fpsnumerator;
      if_t<int_t>  fpsdenominator;
      if_t<int_t>  motion;
      if_t<int_t>  pixformat;
      if_t<string_t>  v4linput;
      nif_t<>       done;
   };

   /* ----------------------------------------------------------------------- */
#define XENUM3(X) \
        X(ME_SPEEDSTATE_OFF, = 0ll, "off")\
        X(ME_SPEEDSTATE_ON,       , "on")\
        X(ME_SPEEDSTATE_INVALID,  , "invalid")
   XENUM3_IMPL(speedstate_e)
#undef XENUM3

#define speedstate_e_format "l"
#define speedstate_e_init ME_SPEEDSTATE_INVALID

   /* ----------------------------------------------------------------------- */

   struct state_t : nif_t<state_t> {
      static string_t name() {
         return "state_t";
      }
      enum : int_t { cid = 1 };
      state_t() : playstate(PLAYSTATE_NOP), reason(REASON_OK), handle(-1),
         speed(ME_SPEED_NONE), vol(-1), local(1), speedstate(ME_SPEEDSTATE_OFF),videosupport(-1),videopropertycontrol(-1) {
      }
      //TODO: check correct copy & assign with custom ctors;
      //      calling base ctor/assign explicitly required ?
      //state_t(playstate_e const &ps0) : state_t() { // ctor delegation;
      //   playstate = ps0;                           // not supported by Eclipse build;
      //}
      state_t(playstate_e const &ps0) : playstate(PLAYSTATE_NOP),
         reason(REASON_OK), handle(-1), speed(ME_SPEED_NONE), vol(-1), local(1), sourceswitchramp(-1),videopropertycontrol(-1) { // ctor delegation;
         playstate = ps0;
      }
      string_t str(int_t const& var0 = 0) const {
         switch(var0) {
            default:
            case 0: {
               return string_t("state:", estr2(playstate),
                             " handle:", handle(),
                                " pos:", pos().str(),
                                " dur:", dur().str(),
                              " speed:", speed(),
                                " vol:", vol(),
                                " url:", string_t(url()).name(),
                                " dev:", dev(),
                                " log:", logconfig().str(),
                         " speedstate:", estr2(speedstate),
                             " reason:", estr2(reason),
                             "videosupport", videosupport(),
                    "sourceswitchramp:", sourceswitchramp(),
                    "videopropertycontrol:", videopropertycontrol());
            }
            case 1: {
               return string_t("server:", server(),
                                " proc:", string_t(proc()).name(),
                               " codec:", codec(),
                               " local:", local());
            }
            case 2: {
               return string_t("ramp:", ramp().str(),
                           " display:", display().str(),
                               " fmt:", fmt().str());
            }
            case 3: {
               return string_t("vprops:", vprops().str());
            }
            case 4: {
               return string_t("ripconf:", ripconf().str());
            }
         }
      }
      if_t<playstate_e>   playstate;
      if_t<reason_e>         reason;
      if_t<string_t>            url;
      if_t<int_t>            handle;
      if_t<pos_t>               pos;
      if_t<dur_t>               dur;
      if_t<string_t>            dev; // ALSA output device (e.g. hw:1,0)
      //if_t<speed_e>           speed;
      if_t<int_t>             speed;
      if_t<int_t>               vol;
      if_t<string_t>           proc;
      if_t<string_t>         server;
      if_t<string_t>          codec;
      if_t<logconfig_t>   logconfig;
      if_t<ramp_t>             ramp;
      if_t<display_t>       display;
      if_t<fmt_t>               fmt;
      if_t<int_t>             local;
      if_t<vprops_t>         vprops;
      if_t<speedstate_e> speedstate;
      if_t<ripconf_t>       ripconf;
      if_t<v4lprop_t>       v4lprop;
      if_t<int_t>  sourceswitchramp;
      if_t<int_t>      videosupport;
      if_t<int_t> videopropertycontrol;
      nif_t<>                  done;
   };

   /* ----------------------------------------------------------------------- */

   struct reg_t : nif_t<reg_t> {
      static string_t name() {
         return "reg_t";
      }
      enum : int_t { cid = 3 };
      reg_t() : nid(-1), up(1) {
      }
      reg_t(int_t const& nid0, int_t const& up0) : nid(nid0), up(up0) {
      }
      string_t str() const {
         return string_t(up ? "register" : "release", " nid:", nid());
      }
      if_t<int_t> nid;
      if_t<int_t>  up; // nonzero = register; 0 = release;
      nif_t<>    done;
   };

   /* ----------------------------------------------------------------------- */

   struct codecs_t : nif_t<codecs_t> {
      static string_t name() {
         return "codecs_t";
      }
      enum : int_t { cid = 4 };
      if_t<string_t>   ext;
      if_t<strings_t> dlls;
      nif_t<>         done;
   };

   /* ----------------------------------------------------------------------- */

   struct header_t : nif_t<header_t> {
      static string_t name() {
         return "header_t";
      }
      enum : int_t { cid = 5 };
      header_t() : reason(REASON_OK), speed(0) {
      }
      string_t str() const {
         return string_t("reason:", estr2(reason),
                           " url:", url,
                        " handle:", handle(),
                           " pos:", pos().str(),
                           " dur:", dur().str(),
                        "  speed:", speed());
      }
      if_t<reason_e> reason;
      if_t<string_t>    url;
      if_t<int_t>    handle;
      if_t<pos_t>       pos;
      if_t<dur_t>       dur;
      if_t<int_t>     speed;
      nif_t<>          done;
   };

   /* ----------------------------------------------------------------------- */

   struct ctrl_t : nif_t<ctrl_t> {
      static string_t name() {
         return "ctrl_t";
      }
      enum : int_t { cid = 6 };
      ctrl_t() : playstate(PLAYSTATE_NOP), sync(0) {
      }
      ctrl_t(playstate_e const& playstate0, int_t sync0) :
         playstate(playstate0), sync(sync0) {
      }
      string_t str() const {
         return string_t("playstate:", estr2(playstate),
                             " sync:", sync());
      }
      if_t<playstate_e> playstate;
      if_t<int_t>            sync;
      nif_t<>                done;
   };

   /* ----------------------------------------------------------------------- */

   struct buffer_t : nif_t<buffer_t> {
      static string_t name() {
         return "buffer_t";
      }
      enum : int_t { cid = 7 };
      string_t str() const {
         return string_t("size:", data().size(),
                      " header:", header().str());
      }
      if_t<header_t>              header;
      if_t<vector_t<byte_t, false>> data;
      nif_t<>                       done;
   };

   typedef void_t (*state_call_t) (state_t const &, void_t *context);
   typedef void_t (*reg_call_t) (reg_t const &);

   /* ----------------------------------------------------------------------- */

} // me;

/* -------------------------------------------------------------------------- */

#endif // ME_PROTOCOL_HPP_INCLUDED

/* -------------------------------------------------------------------------- */

