/*
 * ClientProtocol.cpp
 *
 *  Created on: Jul 22, 2014
 *      Author: Matthias Thömel
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <syslog.h>
#include <string.h>
#include <errno.h>
#include <stdlib.h>

#include "GMPClientProtocol.h"
#include "Utils.h"

int GMPClientProtocol::Connect(const char *commandPipeName)
{
    /*
     * the connect to the rootd is done on every send command because a fifo is only usable by the thread which
     * opens it. therefore a open by Create (done by another thread) is not working
     */
    return 0;
}

int GMPClientProtocol::Command(const int commandNo, const char *arguments, const int timeoutMs)
{
    tSendCommand command;
    int err = -1;
    int returnFd = 0;

    /* lock the command processing */
    //not needed mCommandLock.lock();

    /* create the return pipe */
    char returnPipeName[512];
    strncpy(returnPipeName, "/tmp/GMP.cmd.result.XXXXXX", sizeof(returnPipeName));
    returnPipeName[sizeof(returnPipeName)-1] = 0;
    returnFd = mkstemp(returnPipeName);
	if(returnFd < 0)
	{
		syslog(LOG_ERR, "%s: mkstemp failed: error=%s", returnPipeName, strerror(errno));
		err = openReturnPipeFailed;
		goto error;
	}
	else
	{
		close(returnFd);
		unlink(returnPipeName);
		returnFd = 0 ;		
	}
    err = mkfifo(returnPipeName, S_IRWXU | S_IRWXG);
    if (err < 0) {
        syslog(LOG_ERR, "%s: mkfifo(%s) failed: error=%s", ROOTD_CLIENT, returnPipeName, strerror(errno));
        err = mkfifoReturnPipeFailed;
        goto error;
    }

    /* open the pipe */
    returnFd = open(returnPipeName, O_RDWR);
    if (returnFd < 0) {
        syslog(LOG_ERR, "%s: open(%s, ...) failed: error=%s", ROOTD_CLIENT, returnPipeName, strerror(errno));
        err = openReturnPipeFailed;
        goto error;
    }

    /* build up the command and argument message */
    command.commandNo = commandNo;
    strncpy(command.returnPipeName, returnPipeName, sizeof(command.returnPipeName)-1);
    command.returnPipeName[sizeof(command.returnPipeName)-1] = 0;
    strncpy(command.arguments, arguments, sizeof(command.arguments));
    command.arguments[sizeof(command.arguments)-1] = 0;
    command.messageLength = sizeof(tSendCommand) - sizeof(command.arguments) + strlen_r(command.arguments) +1;

    /* open the command pipe to the root daemon */
    int FdCommandPipe;
    FdCommandPipe = open(GMPCommands::PipeName(), O_WRONLY);
    if (FdCommandPipe < 0) {
        syslog(LOG_ERR, "%s: could not open %s", ROOTD_CLIENT, GMPCommands::PipeName());
        err = openCommandPipeFailed;
        goto error_2;
    }
    // syslog(LOG_INFO, "%s: opened: %s, fd=%d", ROOTD_CLIENT, GMPCommands::PipeName(), FdCommandPipe);

    syslog(LOG_INFO, "%s: sending commandNo=%d, args=%s",
            ROOTD_CLIENT, command.commandNo, command.arguments);

    /* send the command */
    int sentBytes;
    sentBytes = write(FdCommandPipe, &command, command.messageLength);
    if (sentBytes != command.messageLength) {
        syslog(LOG_ERR, "%s: send to rootd failed: error=%s, message=%d, args=%s", ROOTD_CLIENT,
                strerror(errno), command.commandNo, command.arguments);
        err = writeCommandPipeFailed;
        goto error_2;
    }

    /* close the named pipe to the root daemon */
    close(FdCommandPipe);

    /* wait for the answer */
    int err2;

    fd_set rfds;
    struct timeval tv;

    /* wait loop */
    while(1) {

        FD_ZERO(&rfds);
        FD_SET(returnFd, &rfds);

        /* wait for data */
        if (timeoutMs) {
            tv.tv_sec = timeoutMs/1000;
            tv.tv_usec = (timeoutMs - (tv.tv_sec*1000)) * 1000000L;
        } else {
        tv.tv_sec = 1; /* wait for one second */
        tv.tv_usec = 0;
        }
        err2 = select(returnFd+1, &rfds, NULL, NULL, &tv);

        /* wait successful? */
        if (err2 == -1) { // select failed?
            err = readReturnPipeSelectFailed;
            break; // error on select
        } else if (err2) { // data available ?
            err2 = read(returnFd, &err, sizeof(err));
            if (err2 < 0) {
                syslog(LOG_ERR, "%s: read(%d, %s) failed: error=%s", ROOTD_CLIENT, returnFd, returnPipeName, strerror(errno));
                err = readReturnPipeFiled;
            }
            break;
        } else { // timeout?
            if (mStop) { // stop flag set, end this function
                err = readReturnPipeStopRequested;
                break;
            }
            if (timeoutMs) {
                err = readReturnPipeSelectTimeout;
                break;
            }
        }

        /* restart waiting for data */
    }

error_2:
    /* close the return code pipe and delete it */
    close(returnFd);
    unlink(returnPipeName);

    syslog(LOG_INFO, "%s: got return value: %d", ROOTD_CLIENT, err);

error:
    //not needed mCommandLock.unlock();
    return err;
}

int GMPClientProtocol::Disconnect()
{
    /*
     * A close is done after every write to the command fifo, please see also the Create() method
     */
    return 0;
}
