/**
 * @defgroup PictureManager PictureManager
 * @ingroup Mediaplayer
 * @author Kalyankumar Subbiah
 *
 * Public interface for  PictureManager to control slideShow
 * @{
 */

#ifndef _PICTUREMANAGER_H_
#define _PICTUREMANAGER_H_

#include "PictureManagerSM.h"
#include "ILocalSPM.h"
#include "ThreadFactory.h"
#include "Lock.h"

class PictureManager : public PictureManagerSM, public ILocalSPM , public TFThread
{

public:

//SPM part

    /**
     * Constructor of component.
     * This function is used by LocalSPM to store the PictureManager componentID in mComponentID.
     *
     * @param[in] componentID component ID assigned by SPM
     * @return void
     */
    PictureManager(const tComponentID componentID);

    /**
     * Destructor of component.
     *
     * @return void
     */
    virtual ~PictureManager();

    /**
     * This function is used by LocalSPM to create the PictureManager.
     * @attention: running in SPM thread context
     * Create the PictureManager state machine (including create of message queue).
     * Inform LocalSPM that Create is ready -> CreateDone(0)
     *
     * @return void
     */
    void Create();

    /**
     * This function is used by LocalSPM to trigger the PictureManager initialization.
     * @attention: running in SPM thread context
     * Init the PictureManager state machine.
     * Register PictureManagerSM with dispatcher.
     * Inform LocalSPM that Init is ready -> InitDone(0)
     *
     * @return != 0: error, = 0: OK
     */
    tResult Init(tInitReason reason);

    /**
     * Initialize own member variables
     *
     * @return != 0: error, = 0: OK
     */
    tResult InitSM();

    /**
     * This function is used by LocalSPM to start the PictureManager.
     * @attention: running in SPM thread context
     * From now on all other mediaplayer components are available.
     * Start the PictureManager thread and the state machine.
     *
     * @return != 0: error, = 0: OK
     */
    tResult Run();

    /**
     * This function starts a user function in a thread from the thread factory's thread pool.
     * After leaving while loop used thread is stopped and released for other users.
     * This thread function calls statemanchine message dispatch function
     *
     * @return void
     */
    void Do(int functionID, void *ptr);

   /**
     * This function is used by LocalSPM to stop the PictureManager.
     * @attention: running in SPM thread context
     * Set state machine to final state -> SendMessage(STOP_SM).
     * LocalSPM will be informed after STOP_SM event is processed
     *
     * @return != 0: error, = 0: OK
     */
    tResult Stop();

    /**
     * Inform LocalSPM that Stop is ready -> StopDone(0)
     *
     * @return != 0: error, = 0: OK
     */
    tResult StopEventProcessed();

    /**
     * This function is used by LocalSPM to cleanup the PictureManager.
     * @attention: running in SPM thread context
     * Set state machine to final state -> SendMessage(DONE).
     * LocalSPM will be informed after DONE event is processed.
     *
     * @return != 0: error, = 0: OK
     */
    tResult Done();

    /**
     * Inform LocalSPM that Done is ready -> DoneDone(0)
     *
     * @return != 0: error, = 0: OK
     */
    tResult DoneEventProcessed();

    /**
     * Returns the current state the state machine is in (for debugging of shutdown problems)
     * @param[inout] stateName buffer for storing the current state name
     *
     * @return pointer to stateName
     */
    char *GetSMStateName(OUT tGeneralString stateName, IN size_t size);

    /**
    * Returns answer if the component is a state machine
    *
    * @return true or false
    */
    tBoolean IsComponentSM() { return true; }; //component is a state machine

    /**
     * Function to start slideshow called by Interface
     * If given listId and tag are valid starts the slide show
     * Otherwise sends error response
     *
     * @param[in] listID - id of the list which should be used for slideshow
     * @param[in] objectID - corresponds to the item from which  the slideshow should be started
     *
     *  @return != 0: error, = 0: OK
     **/
     tResult StartSlideshow(const tListID listID, const tObjectID objectID);


    /**
     * Function to Handle SlideShow Actions
     *
     * @param[in] action - action to be done[PLAY/PAUSE/NEXT/PREVIOUS/STOP]
     * @param[in] skipCount - Number of items to be skipped if Action = [PREVIOUS/NEXT]
     *
     *  @return != 0: error, = 0: OK
     */
     tResult RequestSlideshowAction(const tPlaybackAction playbackAction,const tNextPrevSkipCount nextPrevSkipCount = 1);

    /**
     * Function to update NowShowing Property
     *
     *  @return != 0: error, = 0: OK
     */

     tResult UpdateNowShowing();

    /**
     * Function which handles PREVIOUS event
     *
     * @param[in] nextPrevSkipCount - Number of items to be skipped
     * @return != 0: error, = 0: OK
     */
     tResult PreviousAction(const tNextPrevSkipCount nextPrevSkipCount);


    /**
     * Function which handles NEXT event
     *
     * @param[in] skipCount - Number of items to be skipped
     * @return != 0: error, = 0: OK
     */
     tResult NextAction(const tNextPrevSkipCount nextPrevSkipCount) ;

    /**
     * Function to update SlideShowState Property
     *
     *  @return != 0: error, = 0: OK
     */
     tResult UpdateSlideShowState();

    /**
     * Function which handles TIMEOUT event
     * OnTimeOut NextAction(skipCount) is called with skipCount = 1
     * @return != 0: error, = 0: OK
     */
     tResult TimeOutAction();

    /**
     * Function which starts the slideshow, called on entry of Playing state
     * It starts the slideshow timer and updates slideshowstate
     *
     * @return != 0: error, = 0: OK
     */
     tResult NewSlideShow();

    /**
     * Function to start SlideShow timer
     *
     *  @return != 0: error, = 0: OK
     */
     tResult StartTimer();

    /**
     * Function to stop slideShow timer
     *
     *  @return != 0: error, = 0: OK
     */
     tResult StopTimer();

    /**
     * SlideShow timer callback
     *
     * @param[in] timerID timer ID
     * @param[in] instance pointer to own (pictureManager) instance
     * @param[in] userData pointer to possible user data
     * @return true or false
     */
     static bool SlideShowTimerCallBack(timer_t timerID , void* instance ,const void *userData);


    /**
     * Function to set SlideShow timer interval
     *
     * @param[in] slideshowTime - in seconds
     * @return != 0: error, = 0: OK
     */
     tResult SetSlideshowTime(const tSlideshowTime slideshowTime);


    /**
     * Function returns the SlideShow timer interval
     *
     * @param[out] slideshowTime - in seconds
     * @return != 0: error, = 0: OK
     */
     tResult GetSlideshowTime(tSlideshowTime &slideshowTime);

    /**
     * Function returns the current Nowshowing information
     *
     * @param[out] nowShowing
     * @return != 0: error, = 0: OK
     */
     tResult GetNowShowing(tNowShowing &nowShowing);

    /**
     * Function returns current slideshow state
     * There is a mapping between internal states and external states
     *
     * @param[out] playbackState external (HMI) playback state
     * @return != 0: error, = 0: OK
     */
     tResult GetSlideshowState(tHMIPlaybackState &playbackState);


private:


     /**
      * Function to set current playback state
      *
      * param[in] playbackState - current (HMI) playback state
      *
      * @return != 0: error, = 0: OK
      */

     tResult SetSlideshowState(const tHMIPlaybackState playbackState);

     /**
      * Function to set current active listID
      *
      * param[in] listID - active listID
      *
      * @return != 0: error, = 0: OK
      */
     tResult SetNowshowingListID(const tListID listID);

     /**
      * Function to set current currently show images position in the list
      *
      * param[in] position - currently show images position in the list
      *
      * @return != 0: error, = 0: OK
      */
     tResult SetNowshowingPosition(const tPosition position);

     /**
      * Function to set current show Object
      *
      * param[in] Object - current show Object
      *
      * @return != 0: error, = 0: OK
      */
     tResult SetNowshowingObject(const tMediaObject Object);

     /**
      * Function to set name of nextfile
      *
      * param[in] nextFileName - name of nextfile
      *
      * @return != 0: error, = 0: OK
      */
     tResult SetNowshowingNextFileName(const tFilename nextFileName);

     /**
      * Function to set nowshowing image state
      *
      * param[in] state -  nowshowing image state
      *
      * @return != 0: error, = 0: OK
      */
     tResult SetNowshowingState(const  tNowPlayingState state);

    /**
     * Function to restart slideshow timer
     *
     * @return != 0: error, = 0: OK
     */
     tResult RestartTimer();


    /**
     * Function which perform the given action (PLAY/NEXT/PRE)
     *
     * @ param [in] playbackAction - (PLAY/NEXT/PRE), position - next position
     * @return != 0: error, = 0: OK
     */
     tResult ProcessAction(const tPlaybackAction playbackAction, const tPosition position, const tListSize listSize = 0);

    /**
     * Function to initialize PictureManager members 
     *
     * @return != 0: error, = 0: OK
     */
     tResult InitMembers();

    tHMIPlaybackState m_HMIPlaybackState;   /**< last playback state send to HMI */
    tNowShowing m_nowShowing;               /**< last Nowshowing object send to HMI */
    Timer m_SlideShowTimer;                 /**< SlideShow timer */
    timer_t m_SlideShowTimerID;             /**< SlideShow timer index */
    Lock m_Mutex;                           /**< lock access to member variables */

};

#endif //_PICTUREMANAGER_H_

/** @} */




