/* ETG definitions */
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_mp.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_SPM
#ifdef TARGET_BUILD
#include "trcGenProj/Header/LocalSPM.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_SPM
#endif
#endif
#include "FunctionTracer.h"
#include "VarTrace.h"

/* others */
#include "LocalSPM.h"
#include "Dispatcher.h"


#define STRINGIZE(x) #x
#define STRINGIZE_VALUE_OF(x) STRINGIZE(x)

/**
 * Function used by LocalSPM to launch the PlayerEngine
 * This function will launch the PlayerEngine process through the DBUS method invocation.
 *
 * @return none
 */
static void LaunchPlayEngineThroughDBusInvocation()
{
    //fix for GMMY17-14978: As Playerengine reserves the object path, dbus activation using a introspect dbus method is done much earlier in the start up of Mediaplayer.
    system("dbus-send --session --dest=DBus.PlayerEngine  /PlayerEngine org.freedesktop.DBus.Introspectable.Introspect");//
    sleep(2);
}

LocalSPM::LocalSPM():
#ifndef VARIANT_S_FTR_ENABLE_ROOTDAEMON_SERVER
                      mRootDaemon       ( COMPONENT_ID_ROOT_DAEMON ),
#endif
                      mMediaEngineClient( COMPONENT_ID_MEDIA_ENGINE ),
                      mDBManager        ( COMPONENT_ID_DB_MANAGER ),
                      mDataProvider     ( COMPONENT_ID_DATA_PROVIDER ),
                      mIPCProvider      ( COMPONENT_ID_IPC_PROVIDER ),
                      mIndexer          ( COMPONENT_ID_INDEXER ),
                      mAlbumArtIndexer  ( COMPONENT_ID_ALBUMART_INDEXER ),
                      mUSBControl       ( COMPONENT_ID_USB_CONTROL ),
                      mCDDAControl      ( COMPONENT_ID_CDDA_CONTROL ),
                      mDVDControl       ( COMPONENT_ID_DVD_CONTROL ),
                      mCSControl        ( COMPONENT_ID_CS_CONTROL ),
                      mIPODControl      ( COMPONENT_ID_IPOD_CONTROL ),
                      mIPODControlHelper ( COMPONENT_ID_IPOD_CONTROL_HELPER ),
                      mMTPControl       ( COMPONENT_ID_MTP_CONTROL ),
                      mBTControl        ( COMPONENT_ID_BT_CONTROL ),
                      mListControl      ( COMPONENT_ID_LIST_CONTROL ),
                      mPlayerManager    ( COMPONENT_ID_PLAYER_MANAGER ),
                      mDeviceDispatcher ( COMPONENT_ID_DEVICE_DISPATCHER ),
                      mPictureManager   ( COMPONENT_ID_PICTURE_MANAGER ),
                      mCDRipperControl   ( COMPONENT_ID_CD_RIPPER )
{
    InitErrorStrings();
    mSPMState = SPM_STATE_OFF;
    mStopStatistics = false;

    /* sets timer class for SPM */
    mTimer.SetTimerClass(TIMER_CLASS_SPM);
    mMemUsageMB = 0;

    //CID 17759 (#1 of 1): Uninitialized pointer field (UNINIT_CTOR)
    mConfiguration = 0;
    mOutputWrapper = 0;
    mCustomControl = 0;
    mTimerID = 0;
}

LocalSPM::~LocalSPM()
{
    /* work around for undefined destruction sequence between static and dynamic objects,
     * by this call the Dispatcher is not beeing destroyed before the local spm is destroyed
     * must be solved by a clean solution */
    Dispatcher::GetInstance().ForceNoDestruction();

    // GetThreadFactory().KillAllThreads();
}


int LocalSPM::InitAll(void)
{
    ENTRY;

    // trace current mediaplayer version
    ETG_TRACE_FATAL(("GMP version: %50s. Build date: %50s", STRINGIZE_VALUE_OF(VERSION_NUM_OF_GENMP), STRINGIZE_VALUE_OF(VERSION_DATE_OF_GENMP)));

    mTimer.StartSMTimer(OUT mTimerID, GetDataProvider().SPMStartupTimeout(), 0L, "LocalSPMStartSM::TIMEOUT");

    unsigned int iter;
    int ret = 0;
    tInitReason initReason;

    // if the state is UNDERVOLTAGE then the Init function must be called with reason IR_UNDERVOLTAGE
    if (SPM_STATE_UNDERVOLTAGE == mSPMState)
        initReason = IR_UNDERVOLTAGE;
    else
        initReason =IR_BOOT;
    
    //allow RequestResponse SMs
    Dispatcher::GetInstance().UnblockAllTransient();

    for(iter=0; iter < components.size(); iter++) 
    {
        ret = components[iter]->Init(IN initReason);
        if (ret) break;
    }
    return ret; 
}

int LocalSPM::RunAll() 
{ 
    ENTRY;

    /* restart startup supervision timer */
    mTimer.CancelTimer(IN mTimerID);
    mTimer.StartSMTimer(OUT mTimerID, GetDataProvider().SPMStartupTimeout(), 0L, "LocalSPMStartSM::TIMEOUT");

    unsigned int iter;
    int ret = 0;
    for(iter=0; iter < components.size(); iter++) 
    {
        ret = components[iter]->Run();
        if (ret) break;
    }

    /* start the statistics thread */
    mStopStatistics = false;
    //LocalSPM::GetThreadFactory().Do(this, 1, NULL); //disabled due to bad implementation

    return ret; 
}

int LocalSPM::StopAll(void)
{
    ENTRY;

    /* cancel the statistics thread */
    mStopStatistics = true;

    /* start shutdown supervision timer */
    mTimer.StartSMTimer(OUT mTimerID, GetDataProvider().SPMShutdownTimeout(), 0L, "LocalSPMStopSM::TIMEOUT_STOP");

    unsigned int iter;
    int ret = 0;

    //release/block all RequestResponse SMs
    Dispatcher::GetInstance().BlockAllTransient();

    //call Stop for all SMs
    for(iter=components.size(); iter > 0; iter--)
    {
        ret = components[iter-1]->Stop();
        if (ret) break;
    }

    return ret; 
}

int LocalSPM::StopAllTimeout(void)
{
    ENTRY;

    StateChangeTimeout();
    DoneAllSM();

    return 0;
}

int LocalSPM::DoneAllSM(void)
{
    ENTRY;

    /* restart shutdown supervision timer */
    mTimer.CancelTimer(IN mTimerID);
    mTimer.StartSMTimer(OUT mTimerID, GetDataProvider().SPMShutdownTimeout(), 0L, "LocalSPMStopSM::TIMEOUT_DONE_SM");

    unsigned int iter;
    int ret = 0;

    for(iter=components.size(); iter > 0; iter--)
    {
        if (components[iter-1]->IsComponentSM())
        {
            ret = components[iter-1]->Done();
            if (ret) break;
        }
    }

    return ret;
}

int LocalSPM::DoneAllSMTimeout(void)
{
    ENTRY;

    StateChangeTimeout();
    DoneAll();

    return 0;
}

int LocalSPM::DoneAll(void)
{
    ENTRY;

    /* restart shutdown supervision timer */
    mTimer.CancelTimer(IN mTimerID);

    unsigned int iter;
    int ret = 0;

    for(iter=components.size(); iter > 0; iter--)
    {
        if (false == components[iter-1]->IsComponentSM())
        {
            ret = components[iter-1]->Done();
            if (ret) break;
        }
    }

    //Calling Done() to all non-statemachine components is processed synchronously
    //timeout during that time calling Done causes transition to final State, events DONE_READY ignored.
    //This causes an FATAL_ASSERT later on due to IsAllInDone() returns false.
    //To avoid the ASSERT we start the time AFTER Done() calls. Apart from that it gives all SM components +30 secs to reach DONE state.
    //During that 30 seconds SPM will generate EM backtrace in case of a hung scenario.
    int doneAllToMS = 3 * GetDataProvider().SPMShutdownTimeout(); // 30 seconds
    VARTRACE(doneAllToMS);
    mTimer.StartSMTimer(OUT mTimerID, doneAllToMS, 0L, "LocalSPMStopSM::TIMEOUT_DONE");
    return ret;
}

int LocalSPM::DoneAllTimeout(void)
{
    ENTRY;

    StateChangeTimeout();

    return 0;
}

bool LocalSPM::StatisticsTimer(timer_t timerID , void *pObject, const void *userData)
{
    (void) pObject;
    (void) timerID;
    LocalSPM *_this = (LocalSPM *)userData;
    _this->PrintStatistics();
    return 0;
}

void LocalSPM::Do(int functionID, void *ptr)
{
    while(!mStopStatistics) {
        PrintStatistics();
        sleep(2);
    }
}

void LocalSPM::PrintStatistics()
{
    /* ask for statics data and print them */

    FILE *fpStat = fopen("/tmp/GMPStat.log", "w");
    if (!fpStat) return;

    /* header */
    time_t t = time(NULL);
    struct tm tm = *localtime(&t);
    fprintf(fpStat, "GMP Statistics:\n");;
    fprintf(fpStat, "time: %d-%02d-%02d %02d:%02d:%02d\n", tm.tm_year + 1900, tm.tm_mon + 1, tm.tm_mday, tm.tm_hour, tm.tm_min, tm.tm_sec);

    /* entries from the components */
    unsigned int iter;
    int ret = 0;
    for(iter=components.size(); iter > 0; iter--)
    {
        tStatistics stat;

        ret = components[iter-1]->Statistics(OUT stat);
        if (ret) break;

        if (!strlen_r(stat)) continue;

        fprintf(fpStat, "%s: %s", components[iter-1]->GetComponentName(), stat);
        fprintf(fpStat, "\n");
    }

    /* end of log */
    fclose(fpStat);

    size_t memory_usage_MB = getCurrentRSS()/1024/1024;
    size_t mallo_trim_limit = (size_t)GetDataProvider().CriticalMemoryUsageMB(); //MB

    if(mMemUsageMB != memory_usage_MB) {
        if(memory_usage_MB >= mallo_trim_limit) {
            ETG_TRACE_ERR(("Mediaplayer high memory usage! (%dMB)",memory_usage_MB));

            if(mMemUsageMB < mallo_trim_limit) {
                ETG_TRACE_ERRMEM(("Mediaplayer high memory usage! (%dMB)",memory_usage_MB));
                malloc_trim(0);
            }
        }
        mMemUsageMB = memory_usage_MB;
        VARTRACE(mMemUsageMB);
    }

    return;
}

int LocalSPM::StateChangeTimeout(void)
{
    ENTRY
    ETG_TRACE_FATAL(("Mediaplayer state change timed out!! "));

    /* cancel the startup supervision timer */
    mTimer.CancelTimer(IN mTimerID);

    LogComponentStates();

    return 0;
}

int LocalSPM::CheckComponentsState(tComponentState state, tComponentGroup group /*= COMPONENT_GROUP_ALL*/)
{
    unsigned int iter;

    for(iter=0; iter < components.size(); iter++)
    {
        if((COMPONENT_GROUP_SM == group) && (false == components[iter]->IsComponentSM()))
            continue;

        if((COMPONENT_GROUP_NON_SM == group) && (components[iter]->IsComponentSM()))
            continue;

        if(state != components[iter]->GetState())
            return 0;
    }

    /*all the components are in the requested state*/
    return 1 ;
}

void LocalSPM::LogComponentStates(void)
{
    ENTRY_INTERNAL

    unsigned int iter;

    for(iter=0; iter < components.size(); iter++)
    {
        tGeneralString smStateName;
        ETG_TRACE_ERRMEM(("Component (%32s) State is -> %32s[%32s]",components[iter]->GetComponentName(),components[iter]->GetStateName(), components[iter]->GetSMStateName(smStateName, sizeof(smStateName))));
        ETG_TRACE_ERR(("Component (%32s) State is -> %32s[%32s]",components[iter]->GetComponentName(),components[iter]->GetStateName(), components[iter]->GetSMStateName(smStateName, sizeof(smStateName))));
    }

#ifdef TARGET_BUILD
    /* send myself a SIGUSR2 to trace out all callstacks of all threads */
    //Do not trigger callstack generation due to time critical operation here (INIT->NORMAL)
    //see GMMY17-3593
    //raise(34);
#endif
}

int LocalSPM::SetState(tComponentID componentID, tComponentState componentState)
{
    ENTRY_INTERNAL

    tResult result = -1; //component is not registered with the LocalSPM
    unsigned int iter;
    for(iter=0; iter < components.size(); iter++)
    {
        if(componentID == components[iter]->GetComponentId())
        {
            /*Set the state for component with ID:componentID */
            components[iter]->SetState(componentState);
            result = 0;
        }
    }

    /*Inform the SPM statemachine with the state set response*/
    if((componentState == COMPONENT_STATE_INIT ) || (componentState == COMPONENT_STATE_RUNNING ))
    {
        LocalSPMStartSM::SendEvent(LocalSPMStartSM::SET_STATE_DONE,NULL);
    }
    else if((componentState == COMPONENT_STATE_STOP ) || (componentState == COMPONENT_STATE_DONE ))
    {
        LocalSPMStopSM::SendEvent(LocalSPMStopSM::SET_STATE_DONE,NULL);
    }
    else
    {
        if(MP_ERR_SM_NO_CURRENT_STATE != LocalSPMStartSM::GetState())
        {
            LocalSPMStartSM::SendEvent(LocalSPMStartSM::SET_STATE_DONE,NULL);
        }
        if(MP_ERR_SM_NO_CURRENT_STATE != LocalSPMStopSM::GetState())
        {
            LocalSPMStopSM::SendEvent(LocalSPMStopSM::SET_STATE_DONE,NULL);
        }
    }

    return result;
}

int LocalSPM::SetInitState(tComponentID componentID)
{
    return SetState(IN componentID, IN COMPONENT_STATE_INIT);
}

int LocalSPM::SetRunState(tComponentID componentID)
{
    return SetState(IN componentID, IN COMPONENT_STATE_RUNNING);
}

int LocalSPM::SetStopState(tComponentID componentID)
{
    return SetState(IN componentID, IN COMPONENT_STATE_STOP);
}

int LocalSPM::SetDoneState(tComponentID componentID)
{
    return SetState(IN componentID, IN COMPONENT_STATE_DONE);
}

int LocalSPM::IsAllInit()
{ 
    ENTRY_INTERNAL

    return CheckComponentsState(COMPONENT_STATE_INIT);
}

int LocalSPM::IsAllInStop()
{
    ENTRY_INTERNAL

    return CheckComponentsState(COMPONENT_STATE_STOP);
}

int LocalSPM::IsAllSMInDone()
{
    ENTRY_INTERNAL

    return CheckComponentsState(COMPONENT_STATE_DONE, COMPONENT_GROUP_SM);
}

int LocalSPM::IsAllInDone()
{ 
    ENTRY_INTERNAL

    return CheckComponentsState(COMPONENT_STATE_DONE, COMPONENT_GROUP_ALL);
}

int LocalSPM::IsAllRun()
{
    ENTRY_INTERNAL

    return CheckComponentsState(COMPONENT_STATE_RUNNING);
}

/************************************************************
 *
 *	Implementation for Public APIs provided by LocalSPM
 *
 ************************************************************/
LocalSPM& LocalSPM::GetInstance(void)
{
    static LocalSPM spm;
    return spm;
}   

tResult LocalSPM::UnregisterAll()
{
    //components.clear();
    if( 0 < components.capacity() )
    {
        vector<ILocalSPM *>().swap(components); //used instead of clear to guarantee a memory reallocation
    }
    return 0;
}

tResult LocalSPM::Register(ILocalSPM *component)
{
    components.push_back(component);
    return 0;
}

tResult LocalSPM::Register(Configuration *component)
{
    ENTRY

    mConfiguration = component;
    components.push_back(mConfiguration);
    return 0;
}

tResult LocalSPM::Register(OutputWrapper *outputWrapper)
{
    ENTRY

    mOutputWrapper = outputWrapper;
    components.push_back(mOutputWrapper);
    return 0;
}

tResult LocalSPM::Register(CustomControl *customControl)
{
    ENTRY

    mCustomControl = customControl;
    components.push_back(mCustomControl);
    return 0;
}

tResult LocalSPM::CreateMediaPlayer(void)
{
    ENTRY

#ifndef VARIANT_S_FTR_ENABLE_ROOTDAEMON_SERVER
    Register(&mRootDaemon);       // COMPONENT_ID_ROOT_DAEMON = 19
#endif
    Register(&mDBManager);        // COMPONENT_ID_DB_MANAGER = 9
    Register(&mDataProvider);     // COMPONENT_ID_DATA_PROVIDER = 10
    Register(&mIPCProvider);      // COMPONENT_ID_IPC_PROVIDER = 11   Attention: used also in ME case for dbus message loop

    Register(&mUSBControl);       // COMPONENT_ID_USB_CONTROL = 1
    Register(&mIPODControlHelper);// COMPONENT_ID_IPOD_CONTROL_HELPER = 22
    Register(&mIPODControl);      // COMPONENT_ID_IPOD_CONTROL = 2
    if (LocalSPM::GetDataProvider().IsDeviceTypeSupported(DTY_MTP)) {
        Register(&mMTPControl);       // COMPONENT_ID_MTP_CONTROL = 3
    }
    if (LocalSPM::GetDataProvider().IsDeviceTypeSupported(DTY_BLUETOOTH)) {
        Register(&mBTControl);        // COMPONENT_ID_BT_CONTROL = 16
    }
    if (LocalSPM::GetDataProvider().IsDeviceTypeSupported(DTY_CDDA)) {
        Register(&mCDDAControl);      // COMPONENT_ID_CDDA_CONTROL = 20
    }
    if (GetDataProvider().ContentSharingEnabled()) {
        Register(&mCSControl);        // COMPONENT_ID_CS_CONTROL = 21
    }
    if ((LocalSPM::GetDataProvider().IsDeviceTypeSupported(DTY_DVD_DRIVE)) && (GetDataProvider().UseDVDControl())) {
           Register(&mDVDControl);      // COMPONENT_ID_DVD_CONTROL = 24
       }
    Register(&mDeviceDispatcher); // COMPONENT_ID_DEVICE_DISPATCHER = 7

    Register(&mIndexer);          // COMPONENT_ID_INDEXER = 0
    if (GetDataProvider().CoverArtFlow() || GetDataProvider().AlbumArtThumbnail()) {
        Register(&mAlbumArtIndexer);  // COMPONENT_ID_ALBUMART_INDEXER = 23
    }
    Register(&mListControl);      // COMPONENT_ID_LIST_CONTROL = 4
    Register(&mPlayerManager);    // COMPONENT_ID_PLAYER_MANAGER = 6
    if (GetDataProvider().UsePictureManager()) {
        Register(&mPictureManager);   // COMPONENT_ID_PICTURE_MANAGER = 17
    }

    if(GetDataProvider().UseMediaEngine()) {
        Register(&mMediaEngineClient); // COMPONENT_ID_MEDIA_ENGINE = 18
    }
    else
    {
#ifndef TARGET_BUILD
        LaunchPlayEngineThroughDBusInvocation();
#endif
    }

    if(GetDataProvider().CDRippingSupport()) {
            Register(&mCDRipperControl);      // COMPONENT_ID_CD_RIPPER = 25
       }
    /*Create thread factory*/
    mThreadFactory.Create();
    mThreadFactoryLowPrio.Create();
    /*Navigate Create to all the registered ILocalSPM based internal MP components*/
    unsigned int iter;
    for(iter=0; iter < components.size(); iter++) 
    {
        components[iter]->Create();
    }

    /*Create SPM internal State machines -> Start and Stop*/
    if(MP_ERR_SM_NO_CURRENT_STATE == LocalSPMStartSM::GetState())
    {
        LocalSPMStartSM::Create();
    }
    if(MP_ERR_SM_NO_CURRENT_STATE == LocalSPMStopSM::GetState())
    {
        LocalSPMStopSM::Create();
    }

    return 0;
}

tResult LocalSPM::StateChangeNormal(void)
{
    ENTRY

    // if in state OFF then switch to NORMAL
    if (SPM_STATE_OFF == mSPMState)
    {
        /*Call the Configuration load method */
        GetConfiguration().LoadSettings();

        /*register START State machine with dispatcher*/
        Dispatcher::GetInstance().Register(LocalSPMStartSM::GetInstance());

        /*initialize LocalSPM Start State machine*/
        LocalSPMStartSM::Reset(); //LocalSPMStartSM::Init();

        /* enable SPM timers */
        mTimer.EnableTimerClass(TIMER_CLASS_SPM);

        /* send all components to state normal by using the state machine */
        while(LocalSPMStartSM::STATE_MACHINE_FINISHED != LocalSPMStartSM::StateMachine_Main()) {}
        ETG_TRACE_USR1(("End of LocalSPM Start State machine "));

        Dispatcher::GetInstance().DeRegister(LocalSPMStartSM::GetInstance());

        mTimer.CancelTimer(mTimerID);

        /*trace all the configuration values set for GMP into ETG*/
        GetDataProvider().TraceMediaplayerConfigurations();

        /* enable all timers */
        mTimer.EnableTimerClass(TIMER_CLASS_ALL);

        // set new state to Normal
        mSPMState = SPM_STATE_NORMAL;

        return (IsAllRun())?MP_NO_ERROR:-1;
    }

    // all other states
    return MP_NO_ERROR;
}

tResult LocalSPM::StateChangeOff(void)
{
    ENTRY

    // if in state NORMAL then switch to OFF
    if (SPM_STATE_NORMAL == mSPMState)
    {
        // set new state to OFF
        mSPMState = SPM_STATE_OFF;

        /*register STOP State machine with dispatcher*/
        Dispatcher::GetInstance().Register(LocalSPMStopSM::GetInstance());

        /* Initialize LocalSPM Stop State machine */
        LocalSPMStopSM::Reset();    //LocalSPMStopSM::Init();

        /* enable SPM timers only (disables all other timers) */
        mTimer.EnableTimerClass(TIMER_CLASS_SPM);

        /* send all components to state off */
        while(LocalSPMStopSM::STATE_MACHINE_FINISHED != LocalSPMStopSM::StateMachine_Main()) {}
        ETG_TRACE_USR1(("End of LocalSPM Stop State machine "));

        /* disables all timers */
        mTimer.EnableTimerClass(TIMER_CLASS_NONE);

        //Dispatcher::GetInstance().DeRegister(LocalSPMStopSM::GetInstance());
        Dispatcher::GetInstance().DeRegisterAll();

        /* cancel the shutdown timer */
        mTimer.CancelTimer(mTimerID);

        return (IsAllInDone())?MP_NO_ERROR:-1;
    }
    // if in state UNDERVOLTAGE then switch to OFF. A state change of all components is not necessary.
    else if (SPM_STATE_UNDERVOLTAGE == mSPMState)
    {
        // set new state to OFF
        mSPMState = SPM_STATE_OFF;
        return MP_NO_ERROR;
    }

    // all other states
    return MP_NO_ERROR;
}

tResult LocalSPM::StateChangeUndervoltage(const tUndervoltage undervoltage)
{
    ENTRY
    VARTRACE(undervoltage)

    // if an under voltage was detected and current state is NORMAL, then switch all components in OFF state
    if (undervoltage && SPM_STATE_NORMAL == mSPMState)
    {
        /*register STOP State machine with dispatcher*/
        Dispatcher::GetInstance().Register(LocalSPMStopSM::GetInstance());

        /* Initialize LocalSPM Stop State machine */
        LocalSPMStopSM::Reset();    //LocalSPMStopSM::Init();

        /* send all components to state off */
        while(LocalSPMStopSM::STATE_MACHINE_FINISHED != LocalSPMStopSM::StateMachine_Main()) {}
        ETG_TRACE_USR1(("End of LocalSPM Stop State machine "));

        Dispatcher::GetInstance().DeRegister(LocalSPMStopSM::GetInstance());

        mTimer.CancelTimer(mTimerID);

        // set new state to UNDERVOLTAGE
        mSPMState = SPM_STATE_UNDERVOLTAGE;

        return (IsAllInDone())?MP_NO_ERROR:-1;
    }
    // if under voltage is finished and and current state is UNDERVOLTAGE, then switch all components in NORMAL state 
    else if (FALSE == undervoltage && SPM_STATE_UNDERVOLTAGE == mSPMState)
    {

        /*register START State machine with dispatcher*/
        Dispatcher::GetInstance().Register(LocalSPMStartSM::GetInstance());

        /*initialize LocalSPM Start State machine*/
        LocalSPMStartSM::Reset(); //LocalSPMStartSM::Init();

        /* send all components to state normal by using the state machine */
        while(LocalSPMStartSM::STATE_MACHINE_FINISHED != LocalSPMStartSM::StateMachine_Main()) {}
        ETG_TRACE_USR1(("End of LocalSPM Start State machine "));

        Dispatcher::GetInstance().DeRegister(LocalSPMStartSM::GetInstance());

        mTimer.CancelTimer(mTimerID);

        // set new state to Normal
        mSPMState = SPM_STATE_NORMAL;

        return (IsAllRun())?MP_NO_ERROR:-1;
    }

    // all other states
    return MP_NO_ERROR;
}

int LocalSPM::isAllRunExternal()
{
    return IsAllRun();
}

ThreadFactory &LocalSPM::GetThreadFactory(void)
{
    return GetInstance().mThreadFactory;
}

ThreadFactoryLowPrio &LocalSPM::GetThreadFactoryLowprio(void)
{
    return GetInstance().mThreadFactoryLowPrio;
}
DataProvider &LocalSPM::GetDataProvider(void)
{
    return GetInstance().mDataProvider;
}

IPCProvider &LocalSPM::GetIPCProvider(void)
{
    return GetInstance().mIPCProvider;
}

Configuration& LocalSPM::GetConfiguration(void)
{
    return *GetInstance().mConfiguration;
}

OutputWrapper& LocalSPM::GetOutputWrapper(void)
{
    return *GetInstance().mOutputWrapper;
}

CustomControl& LocalSPM::GetCustomControl(void)
{
    return *GetInstance().mCustomControl;
}

MediaEngineClient &LocalSPM::GetMediaEngineClient(void)
{
    return GetInstance().mMediaEngineClient;
}

#ifndef VARIANT_S_FTR_ENABLE_ROOTDAEMON_SERVER
RootDaemon &LocalSPM::GetRootDaemon(void)
{
    return GetInstance().mRootDaemon;
}
#endif

DBManager &LocalSPM::GetDBManager(void)
{
    return GetInstance().mDBManager;
}

USBControl &LocalSPM::GetUSBControl(void)
{
    return GetInstance().mUSBControl;
}

CDDAControl &LocalSPM::GetCDDAControl(void)
{
    return GetInstance().mCDDAControl;
}

CSControl &LocalSPM::GetCSControl(void)
{
    return GetInstance().mCSControl;
}

iPodControl &LocalSPM::GetIPODControl(void)
{
    return GetInstance().mIPODControl;
}

iPodControlHelper &LocalSPM::GetIPODControlHelper(void)
{
    return GetInstance().mIPODControlHelper;
}

MTPControl &LocalSPM::GetMTPControl(void)
{
    return GetInstance().mMTPControl;
}

DeviceDispatcher &LocalSPM::GetDeviceDispatcher(void)
{
    return GetInstance().mDeviceDispatcher;
}

Indexer &LocalSPM::GetIndexer(void)
{
    return GetInstance().mIndexer;
}

AlbumArtIndexer &LocalSPM::GetAlbumArtIndexer(void)
{
    return GetInstance().mAlbumArtIndexer;
}

ListControl &LocalSPM::GetListControl(void)
{
    return GetInstance().mListControl;
}

PlayerManager &LocalSPM::GetPlayerManager(void)
{
    return GetInstance().mPlayerManager;
}

PictureManager &LocalSPM::GetPictureViewer(void)
{
    return GetInstance().mPictureManager;
}
BTControl &LocalSPM::GetBTControl(void)
{
    return GetInstance().mBTControl;
}

tSPMState LocalSPM::GetSPMState(void) const
{
    return mSPMState;
}

DVDControl &LocalSPM::GetDVDControl(void)
{
    return GetInstance().mDVDControl;
}
CDRipperControl &LocalSPM::GetCDRipperControl(void)
{
    return GetInstance().mCDRipperControl;
}
