/**
 * @defgroup LocalSPM LocalSPM
 * @ingroup MediaPlayer
 * @author Dinesh D
 *
 * Public interface to Start and Shutdown the Media-Player.
 * Internally it manages the start and stop of all the internal components of
 * the media player (like PlayerManager , Indexer , Data Base manager , etc...)
 * and also provides public interface to all other global components like Dispatcher ,
 * DataProvider , ThreadFactory etc...
 * @{
 */

#ifndef LOCALSPM_H_
#define LOCALSPM_H_

/*LocalSPM generated statemachine's headers*/
#include "LocalSPM_StartSM.h"
#include "LocalSPM_StopSM.h"

/*Global component's headers*/
#include "ThreadFactory.h"
#include "DBManager.h"
#include "DataProvider.h"
#include "IPCProvider.h"

/*Functional component's headers*/
#ifndef VARIANT_S_FTR_ENABLE_ROOTDAEMON_SERVER
#include "GMPRootDaemon.h"
#endif
#include "2_client/gmp/MediaEngineClient.h"
#include "Indexer.h"
#include "AlbumArtIndexer.h"
#include "USBControl.h"
#include "CDDAControl.h"
#include "CSControl.h"
#include "ListControl.h"
#include "CustomControl.h"
#include "iPodControl.h"
#include "iPodControlHelper.h"
#include "MTPControl.h"
#include "BTControl.h"
#include "PlayerManager.h"
#include "DeviceDispatcher.h"
#include "Configuration.h"
#include "OutputWrapper.h"
#include "PictureManager.h"
#include "DVDControl.h"
#include "CDRipperControl.h"

/**
 * Class LocalSPM implements the component LocalSPM which manages Start and Shutdown of
 * all other internal media player components and provides interface to common global components
 *
 */
class LocalSPM : public LocalSPMStartSM, public LocalSPMStopSM, public TFThread {

public:
    /**
     * Creates singleton for Local SPM
     *
     * @return pointer to global static instance of LocalSPM object
     */
    static LocalSPM &GetInstance(void);

    /**
     * Interface to unregister all components from LocalSPM
     *
     * @return int = 0: ok, !=0: failed
     */
    tResult UnregisterAll();

    /**
     * SPM Interface for local components to register themselves with the LocalSPM to
     * handle their Start and Stop
     *
     * @param[in] component component object which implements the ILocalSPM interface
     *
     * @return int = 0: ok, !=0: failed
     */
    tResult Register(ILocalSPM *component);

    /**
     * Interface to register custom configuration component with Generic MediaPlayer
     *
     * @param[in] configuration pointer to custom configuration object which derives from the Configuration
     *
     * @return int = 0: ok, !=0: failed
     */
    tResult Register(Configuration *configuration);

    /**
     * Interface to register custom output wrapper component with Generic MediaPlayer
     *
     * @param[in] outputWrapper pointer to custom output wrapper object which derives from the OutWrapper class
     *
     * @return int = 0: ok, !=0: failed
     */
    tResult Register(OutputWrapper *outputWrapper);

    /**
     * Interface to register custom control component with Generic MediaPlayer
     *
     * @param[in] customControl pointer to custom control object which derives from the CustomControl class
     *
     * @return int = 0: ok, !=0: failed
     */
    tResult Register(CustomControl *customControl);

    /**
     * interface to create Mediaplayer components
     *
     * @return int = 0: ok, !=0: failed
     */
    tResult CreateMediaPlayer(void);

    /**
     * interface to request Mediaplayer transition to NORMAL state
     *
     * @return int = 0: ok, !=0: failed
     */
    tResult StateChangeNormal(void);
    //tResult StartUp(void);

    /**
     * interface to request Mediaplayer transition to OFF state
     *
     * @return int = 0: ok, !=0: failed
     */
    tResult StateChangeOff(void);

    /**
     * interface to request Mediaplayer transition to UNDERVOLTAGE state or back to NORMAL
     * Roadmap 13024
     *
     * if (undervoltage == TRUE)
     * - Send all components to OFF (StopAll and DoneAll)
     * else
     * - Send all components to NORMAL (InitAll(IR_UNDERVOLTAGE) and RunAll)
     *
     * @param[in] undervoltage undervoltage flag
     *
     * @return int = 0: ok, !=0: failed
     */
    tResult StateChangeUndervoltage(const tUndervoltage undervoltage); //Roadmap 13024: 100%

    /**
     * interface to get the Thread Factory instance
     *
     * return pointer to global thread factory object
     */
    static ThreadFactory &GetThreadFactory(void);


     /**
     * interface to get the Thread Factory instance for low prio
     *
     * return pointer to global thread factory object
     */
    static ThreadFactoryLowPrio &GetThreadFactoryLowprio(void);

    /**
     * interface to get the DataProvider instance
     *
     * return pointer to global data provider object
     */
    static DataProvider &GetDataProvider(void);

    /**
     * interface to get the IPCProvider instance
     *
     * return pointer to global IPC provider object
     */
    static IPCProvider &GetIPCProvider(void);

    /**
     * interface to get the Configuration instance
     *
     * return pointer to custom Configuration object
     */
    static Configuration& GetConfiguration(void);

    /**
     * interface to get the OutputWrapper instance
     *
     * return pointer to custom OutputWrapper object
     */
    static OutputWrapper& GetOutputWrapper(void);

    /**
     * interface to get the CustomControl instance
     *
     * return pointer to custom custom control object
     */
    static CustomControl& GetCustomControl(void);

    /**
     * interface to get the Database manager instance
     *
     * return pointer to global database manager object
     */
    static MediaEngineClient &GetMediaEngineClient(void);
#ifndef VARIANT_S_FTR_ENABLE_ROOTDAEMON_SERVER
    static RootDaemon &GetRootDaemon(void);
#endif
    static DBManager &GetDBManager(void);

    static Indexer &GetIndexer(void);
    static AlbumArtIndexer &GetAlbumArtIndexer(void);
    static ListControl &GetListControl(void);
    static PlayerManager &GetPlayerManager(void);
    static PictureManager &GetPictureViewer(void);

    static USBControl &GetUSBControl(void);
    static CDDAControl &GetCDDAControl(void);
    static CSControl &GetCSControl(void);
    static iPodControl &GetIPODControl(void);
    static iPodControlHelper &GetIPODControlHelper(void);
    static MTPControl &GetMTPControl(void);
    static BTControl &GetBTControl(void);
    static DeviceDispatcher &GetDeviceDispatcher(void);
    static DVDControl &GetDVDControl(void);
    static CDRipperControl &GetCDRipperControl(void);
    /**
     * returns the current state of the Local SPM
     * @param[out]
     * @return SPMState current state
     */
    tSPMState GetSPMState(void) const;

    void Do(int functionID, void *ptr);
    void PrintStatistics();
    int isAllRunExternal();

private:

    vector <ILocalSPM *> components;    /**< list of all local components which implements ILocalSPM interface */

    ThreadFactory    mThreadFactory;
    ThreadFactoryLowPrio mThreadFactoryLowPrio;
#ifndef VARIANT_S_FTR_ENABLE_ROOTDAEMON_SERVER
    RootDaemon       mRootDaemon;
#endif
    MediaEngineClient mMediaEngineClient;
    DBManager        mDBManager;
    DataProvider     mDataProvider;
    IPCProvider      mIPCProvider;

    Configuration    *mConfiguration;
    OutputWrapper    *mOutputWrapper;
    CustomControl    *mCustomControl;

    Indexer          mIndexer;
    AlbumArtIndexer  mAlbumArtIndexer;
    USBControl       mUSBControl;
    CDDAControl      mCDDAControl;
    DVDControl       mDVDControl;
    CSControl        mCSControl;
    iPodControl      mIPODControl;
    iPodControlHelper   mIPODControlHelper;
    MTPControl       mMTPControl;
    BTControl        mBTControl;
    ListControl      mListControl;
    PlayerManager    mPlayerManager;
    DeviceDispatcher mDeviceDispatcher;
    PictureManager   mPictureManager;
    CDRipperControl  mCDRipperControl;

    pthread_cond_t   mWaitCondition ;     /**< Condition used to wait for Start and Stop of Mediaplayer */
    pthread_mutex_t  mWaitMutex ;        /**< Mutex lock used to wait for Start and Stop of Mediaplayer */

    Timer            mTimer;
    timer_t          mTimerID;
    size_t           mMemUsageMB;

    bool             mStopStatistics;

    tSPMState        mSPMState;

    enum
    {
        LOCALSPM_START_SM_THREAD,
        LOCALSPM_STOP_SM_THREAD
    };

    /**
     * private Constructor
     *
     */
    LocalSPM();
    LocalSPM(LocalSPM const&);        // Don't Implement
    void operator=(LocalSPM const&);  // Don't implement
    /**
     * Destructor
     *
     */
    virtual ~LocalSPM();

    /**
     * Implements StartStateMachine action InitAll to init all the registered -
     * ILocaSPM based components
     * In case of leaving UNDERVOLTAGE state initReason is IR_UNDERVOLTAGE instaed of IR_BOOT //Roadmap 13024
     *
     *@return int = 0: ok, != 0: failed
     */
    int InitAll(void); //Roadmap 13024: 100%

    /**
     * Implements StartStateMachine action RunAll to run all the registered -
     * ILocaSPM based components
     *
     * @return int = 0: ok, != 0: failed
     */
    int RunAll();

    /**
     * Implements StartStateMachine action StopAll to stop all the registered -
     * ILocaSPM based components
     *
     * @return int = 0: ok, != 0: failed
     */
    int StopAll(void);

    int StopAllTimeout(void);

    /**
     * Implements StartStateMachine action DoneAll to DeInit all the registered -
     * ILocaSPM based SM components
     *
     * @return int = 0: ok, != 0: failed
     */
    int DoneAllSM(void);

    int DoneAllSMTimeout(void);

    /**
     * Implements StartStateMachine action DoneAll to DeInit all the registered -
     * ILocaSPM based non SM components
     *
     * @return int = 0: ok, != 0: failed
     */
    int DoneAll(void);

    int DoneAllTimeout(void);

    /**
     * Function used by Statemachine to check whether all the registered components are
     * already initialised successfully
     *
     * @return int = 0: ok, != 0: failed
     */
    int IsAllInit() ;

    /**
     * Function used by Statemachine to check whether all the registered components have
     * reached the running state successfully
     *
     * @return int = 0: ok, != 0: failed
     */
    int IsAllRun() ;

    /**
     * Function used by Statemachine to check whether all the registered components have
     * been stopped
     *
     * @return int = 0: ok, != 0: failed
     */
    int IsAllInStop();

    /**
     * Function used by Statemachine to check whether all the registered SM components
     * (like PlayerManager) have been deinitialised
     *
     * @return int = 0: ok, != 0: failed
     */
    int IsAllSMInDone();

    /**
     * Function used by Statemachine to check whether all the registered non SM components
     * (like DBManager) have been deinitialised
     *
     * @return int = 0: ok, != 0: failed
     */
    int IsAllInDone();

    int CheckComponentsState(tComponentState state, tComponentGroup group = COMPONENT_GROUP_ALL);

    int SetState(tComponentID componentID,tComponentState componentState);
    int SetRunState(tComponentID componentID);
    int SetInitState(tComponentID componentID);
    int SetStopState(tComponentID componentID);
    int SetDoneState(tComponentID componentID);

    int StateChangeTimeout(void);
    void LogComponentStates(void);

    static bool StatisticsTimer(timer_t timerID , void *pObject, const void *userData);
};

#endif //LOCALSPM_H_

/** @} */
