#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_mp.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_INTERFACE
#ifdef TARGET_BUILD
#include "trcGenProj/Header/MediaPlayerInterface.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_INTERFACE
#endif
#endif

#include <tstring.h>
#include <string.h>
#include "MediaPlayerInterface.h"
#include "LocalSPM.h"
#include "TimeTrace.h"
#include <stdlib.h>
#include "VarTrace.h"
#include "FunctionTracer.h"

#define LOCK Locker locker(&mInterfaceLock);


MediaPlayerInterface* MediaPlayerInterface::pMediaPlayer = NULL;

MediaPlayerInterface& MediaPlayerInterface::GetInstance(void)
{
    ENTRY_INTERNAL

    if (pMediaPlayer == NULL)
    {
        pMediaPlayer = new MediaPlayerInterface;
    }
    return *pMediaPlayer;
}

tResult MediaPlayerInterface::UnregisterAll()
{
    ENTRY

    return LocalSPM::GetInstance().UnregisterAll();
}

tResult MediaPlayerInterface::Register(Configuration *configuration)
{
    ENTRY

    return LocalSPM::GetInstance().Register(configuration);
}

tResult MediaPlayerInterface::Register(OutputWrapper *outputWrapper)
{
    ENTRY

    return LocalSPM::GetInstance().Register(outputWrapper);
}

tResult MediaPlayerInterface::Register(CustomControl *customControl)
{
    ENTRY

    return LocalSPM::GetInstance().Register(customControl);
}

tResult MediaPlayerInterface::CreateMediaPlayer(void)
{
    ENTRY
    LOCK
    TimeTrace ticks (__PRETTY_FUNCTION__);
    tResult retVal = LocalSPM::GetInstance().CreateMediaPlayer();
    isCreated = TRUE;
    mAllocateState = ALS_DEALLOCATED;
    mAudioOutputDevice[0] = '\0';
    ticks.elapsed();
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::StateChangeNormal(void)
{
    ENTRY
    LOCK
    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (FALSE != isCreated)
    {
        retVal = LocalSPM::GetInstance().StateChangeNormal();
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was created", __PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::StateChangeOff(void)
{
    ENTRY
    LOCK
    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (FALSE != isCreated)
    {
        retVal = LocalSPM::GetInstance().StateChangeOff();
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(retVal)
    return retVal;
}


tResult MediaPlayerInterface::SendAllocate(const tAudioOutputDevice audioOutputDevice)
{
    ENTRY
    VARTRACE(audioOutputDevice)
    //LOCK
    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    // get current spm state
    tSPMState SPMState = LocalSPM::GetInstance().GetSPMState();
    if (SPM_STATE_NORMAL == SPMState)
    {
        retVal = LocalSPM::GetPlayerManager().SendAllocate(IN audioOutputDevice);
    }
    // if DeviceManager signals under voltage and the MediaPlayer is not yet in undervoltage state
    else if (SPM_STATE_UNDERVOLTAGE == SPMState)
    {
        strncpy_r(mAudioOutputDevice, audioOutputDevice, sizeof mAudioOutputDevice);
        mAllocateState = ALS_ALLOCATED;
        retVal = MP_NO_ERROR;
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::SendDeAllocate()
{
    ENTRY
    //LOCK
    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    // get current spm state
    tSPMState SPMState = LocalSPM::GetInstance().GetSPMState();
    if (SPM_STATE_NORMAL == SPMState)
    {
        retVal = LocalSPM::GetPlayerManager().SendDeAllocate();
    }
    else if (SPM_STATE_UNDERVOLTAGE == SPMState)
    {
        mAllocateState = ALS_DEALLOCATED;
        retVal = MP_NO_ERROR;
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::SendSourceActivity(const tSourceActivity sourceActivity)
{
    ENTRY
    VARTRACE(sourceActivity)
    //LOCK
    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    // get current spm state
    tSPMState SPMState = LocalSPM::GetInstance().GetSPMState();
    if (SPM_STATE_NORMAL == SPMState)
    {
        retVal = LocalSPM::GetPlayerManager().SendSourceActivity(IN sourceActivity);
    }
    else if (SPM_STATE_UNDERVOLTAGE == SPMState)
    {
        if (SA_ON == sourceActivity)
            mAllocateState = ALS_ACTIVE;
        else
            mAllocateState = ALS_ALLOCATED;
        retVal =  MP_NO_ERROR;
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::ClearLastSource()
{
    ENTRY

    tResult retVal= -1;
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        /* Reset listID */
        LocalSPM::GetPlayerManager().SetListID(LIST_ID_NONE);

        /* Reset current media object */
        tMediaObject mediaObject;
        InitMediaObject(OUT mediaObject);
        LocalSPM::GetPlayerManager().SetMediaObject(IN mediaObject);

        retVal = MP_NO_ERROR;
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }

    return retVal;
}

tResult MediaPlayerInterface::SendMuteStatus(const tMuteState muteState)
{
    ENTRY
    VARTRACE(muteState)

    tResult retVal= -1;
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetDeviceDispatcher().SendMuteStatus(IN muteState);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::CreateMediaPlayerIndexedList(
        tListID &listID,
        tListSize &listSize,
        const tListType listType,
        const tFilterTag1 tag1,
        const tFilterTag2 tag2,
        const tFilterTag3 tag3,
        const tFilterTag4 tag4,
        const tDeviceID deviceID,
        tUserID userID)
{
    ENTRY
    VARTRACE(listType)
    VARTRACE(tag1)
    VARTRACE(tag2)
    VARTRACE(tag3)
    VARTRACE(tag4)
    VARTRACE(deviceID)
    VARTRACE(userID)
    LOCK
    memset(&listID,0, sizeof listID);
    memset(&listSize,0, sizeof listSize);

    tResult retVal= -1;
    tStreaming streaming = false;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        if((LocalSPM::GetDataProvider().PersonalizationSupported()) && (LTY_USER_FAVORITES_SONG != listType))
        {
            userID = 0; // For lists other than User favorites list, set UserID to 0
        }
        retVal = LocalSPM::GetListControl().CreateMediaPlayerIndexedList(OUT listID, OUT listSize, IN listType, IN streaming, IN tag1, IN tag2, IN tag3, IN tag4, IN deviceID, userID);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(listID)
    VARTRACE(listSize)
    VARTRACE(retVal)
    return retVal;
}


tResult MediaPlayerInterface::CreateMediaPlayerPlaylistList(
        tListID &listID,
        tListSize &listSize,
        const tFilename fileName)
{
    ENTRY
    VARTRACE(fileName)
    LOCK
    memset(&listID,0, sizeof listID);
    memset(&listSize,0, sizeof listSize);

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetListControl().CreateMediaPlayerPlaylistList(OUT listID, OUT listSize, IN fileName);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(listID)
    VARTRACE(listSize)
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::CreateMediaPlayerFileList(
        tListID &listId,
        tListSize &listSize,
        tNumberOfMediaObjects &numberOfMediaObjects,
        tNumberOfPlayLists &numberOfPlayLists,
        tNumberOfFolders &numberOfFolders,
        const tPath path,
        const tDeviceID deviceID,
        tFileTypeSelection filetypeSelection)
{
    ENTRY
    VARTRACE(path)
    VARTRACE(deviceID)
    VARTRACE(filetypeSelection)
    LOCK
    memset(&listId, 0, sizeof listId);
    memset(&listSize, 0, sizeof listSize);
    memset(&numberOfMediaObjects, 0, sizeof numberOfMediaObjects);
    memset(&numberOfPlayLists, 0, sizeof numberOfPlayLists);
    memset(&numberOfFolders, 0, sizeof numberOfFolders);

    /* Sanity check for path */
    tPath localPath;
    if(NULL == FastUTF8::StartsWith((const FastUTF8::tString)path, (const FastUTF8::tString)"/")) //lint !e1773
    {
        ETG_TRACE_ERR(("%s is called with a path without leading /",__PRETTY_FUNCTION__));
        strncpy_r(OUT localPath, IN "/", IN sizeof(localPath));
        strncat_r(OUT localPath, IN path, IN sizeof(localPath));
    }
    else
    {
        strncpy_r(OUT localPath, IN path, IN sizeof(localPath));
    }

    /* Check if fileTypeSelection fits to active mediaContext */
    tMediaContext mediaContext;
    LocalSPM::GetDBManager().GetMediaContext(OUT mediaContext);
    if(((MC_AUDIO == mediaContext)
            &&
            (FTS_AUDIO != filetypeSelection)
            &&
            (FTS_PLAYLIST != filetypeSelection)
            &&
            (FTS_AUDIO_PLAYLIST != filetypeSelection))
            ||
            ((MC_VIDEO == mediaContext)
                    &&
                    (FTS_VIDEO != filetypeSelection)))
    {
        ETG_TRACE_USR3(("fileTpeSelection (%u) does not fit to active mediaContext (%u)", filetypeSelection, mediaContext));

    }

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetListControl().CreateMediaPlayerFileList(
                OUT listId,
                OUT listSize,
                OUT numberOfMediaObjects,
                OUT numberOfPlayLists,
                OUT numberOfFolders,
                IN LTY_FILELIST,
                IN localPath,
                IN deviceID,
                IN false, //streaming
                IN filetypeSelection);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(listId)
    VARTRACE(listSize)
    VARTRACE(numberOfMediaObjects)
    VARTRACE(numberOfPlayLists)
    VARTRACE(numberOfFolders)
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::CreateMediaPlayerCDList(
        tListID &listId,
        tListSize &listSize,
        tNumberOfMediaObjects &numberOfMediaObjects,
        tNumberOfPlayLists &numberOfPlayLists,
        tNumberOfFolders &numberOfFolders,
        const tPath path,
        const tDeviceID deviceID,
        tFileTypeSelection filetypeSelection)
{
    ENTRY
    VARTRACE(path)
    VARTRACE(deviceID)
    (void)filetypeSelection;
    LOCK
    memset(&listId, 0, sizeof listId);
    memset(&listSize, 0, sizeof listSize);
    memset(&numberOfMediaObjects, 0, sizeof numberOfMediaObjects);
    memset(&numberOfPlayLists, 0, sizeof numberOfPlayLists);
    memset(&numberOfFolders, 0, sizeof numberOfFolders);

    /* Sanity check for path */
    tPath localPath;
    if(NULL == FastUTF8::StartsWith((const FastUTF8::tString)path, (const FastUTF8::tString)"/")) //lint !e1773
    {
        ETG_TRACE_ERR(("%s is called with a path without leading /",__PRETTY_FUNCTION__));
        strncpy_r(OUT localPath, IN "/", IN sizeof(localPath));
        strncat_r(OUT localPath, IN path, IN sizeof(localPath));
    }
    else
    {
        strncpy_r(OUT localPath, IN path, IN sizeof(localPath));
    }

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetListControl().CreateMediaPlayerFileList(
                OUT listId,
                OUT listSize,
                OUT numberOfMediaObjects,
                OUT numberOfPlayLists,
                OUT numberOfFolders,
                IN LTY_CD,
                IN localPath,
                IN deviceID);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(listId)
    VARTRACE(listSize)
    VARTRACE(numberOfMediaObjects)
    VARTRACE(numberOfPlayLists)
    VARTRACE(numberOfFolders)
    VARTRACE(retVal)
    return retVal;
}


tResult MediaPlayerInterface::CreateMediaPlayerFileListExt(
        tListID &listId,
        tListSize &listSize,
        tNumberOfMediaObjects &numberOfMediaObjects,
        tNumberOfPlayLists &numberOfPlayLists,
        tNumberOfFolders &numberOfFolders,
        tPosition &position,
        const tPath filename,
        const tDeviceID deviceID,
        tFileTypeSelection filetypeSelection)
{
    ENTRY
    VARTRACE(filename)
    VARTRACE(deviceID)
    VARTRACE(filetypeSelection)
    LOCK
    memset(&listId, 0, sizeof listId);
    memset(&listSize, 0, sizeof listSize);
    memset(&numberOfMediaObjects, 0, sizeof numberOfMediaObjects);
    memset(&numberOfPlayLists, 0, sizeof numberOfPlayLists);
    memset(&numberOfFolders, 0, sizeof numberOfFolders);
    memset(&position, 0, sizeof position);

    /* Sanity check for path */
    tPath localFilename;
    if(NULL == FastUTF8::StartsWith((const FastUTF8::tString)filename, (const FastUTF8::tString)"/")) //lint !e1773
    {
        ETG_TRACE_ERR(("%s is called with a path without leading /",__PRETTY_FUNCTION__));
        strncpy_r(OUT localFilename, IN "/", IN sizeof(localFilename));
        strncat_r(OUT localFilename, IN filename, IN sizeof(localFilename));
    }
    else
    {
        strncpy_r(OUT localFilename, IN filename, IN sizeof(localFilename));
    }

    /* Check if fileTypeSelection fits to active mediaContext */
    tMediaContext mediaContext;
    LocalSPM::GetDBManager().GetMediaContext(OUT mediaContext);
    if(((MC_AUDIO == mediaContext)
            &&
            (FTS_AUDIO != filetypeSelection)
            &&
            (FTS_PLAYLIST != filetypeSelection)
            &&
            (FTS_AUDIO_PLAYLIST != filetypeSelection))
            ||
            ((MC_VIDEO == mediaContext)
                    &&
                    (FTS_VIDEO != filetypeSelection)))
    {
        ETG_TRACE_ERR(("fileTpeSelection (%u) does not fit to active mediaContext (%u)", filetypeSelection, mediaContext));
    }

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetListControl().CreateMediaPlayerFileList(
                OUT listId,
                OUT listSize,
                OUT numberOfMediaObjects,
                OUT numberOfPlayLists,
                OUT numberOfFolders,
                OUT position,
                IN localFilename,
                IN deviceID,
                IN false, //streaming
                IN filetypeSelection);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(listId)
    VARTRACE(listSize)
    VARTRACE(numberOfMediaObjects)
    VARTRACE(numberOfPlayLists)
    VARTRACE(numberOfFolders)
    VARTRACE(position)
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::CreateQuickPlayList(
        tListInfo &listInfo,
        const tListType listType,
        const tFilterTag1 tag1,
        const tFilterTag2 tag2,
        const tFilterTag3 tag3,
        const tFilterTag4 tag4,
        const tPlayContinuation playContinuation,
        const tDeviceID deviceID)
{
    ENTRY
    VARTRACE(listType)
    VARTRACE(tag1)
    VARTRACE(tag2)
    VARTRACE(tag3)
    VARTRACE(tag4)
    VARTRACE(playContinuation)
    VARTRACE(deviceID)
    LOCK
    memset(&listInfo, 0, sizeof listInfo);

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        tObjectID filterTags[4];
        filterTags[0] = tag1.tag;
        filterTags[1] = tag2.tag;
        filterTags[2] = tag3.tag;
        filterTags[3] = tag4.tag;
        retVal = LocalSPM::GetCustomControl().CreateQuickPlayList(OUT listInfo, IN listType, IN filterTags, IN playContinuation, IN deviceID);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(listInfo)
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::RequestMediaPlayerIndexedListSlice(
        vector<tMediaObject> &mediaObjectVector,
        const tListID listID,
        const tIndex startIndex,
        const tIndex sliceSize)
{
    ENTRY
    VARTRACE(listID)
    VARTRACE(startIndex)
    VARTRACE(sliceSize)
    LOCK
    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetListControl().RequestMediaPlayerIndexedListSlice(OUT mediaObjectVector, IN listID, IN startIndex, IN sliceSize);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }

    if(MP_NO_ERROR == retVal)
    {
        tListInfo listInfo;
        tBoolean withListSize = false;
        InitListInfo(listInfo);
        retVal = LocalSPM::GetListControl().GetListInfo(OUT listInfo, IN listID, IN withListSize);

        /* If CodesetConversion supported and  device is a MassStorageDevice,then
           invoke ConvertCodeSetFromSysLang2UTF8 */
        if (LocalSPM::GetDataProvider().CodeSetConversionSupported())
        {
            tDeviceInfo deviceInfo;
            InitDeviceInfo(deviceInfo);
            retVal = LocalSPM::GetDBManager().GetDeviceInfo(OUT deviceInfo, IN listInfo.deviceID);

            if(IsMassStorageDevice(deviceInfo.deviceType))
            {
                ConvertCodeSetFromSysLang2UTF8(INOUT mediaObjectVector);
            }
        }

#if USE_IMAGE_OBJECT
        if (LTY_IMAGE == listInfo.listType)
        {
            retVal = LocalSPM::GetListControl().ReplaceFilenameWithAbsolutePath(INOUT mediaObjectVector);
        }
        else
        {
#endif
            tDeviceInfo deviceInfo;
            tAlbumArtIndexingComplete completeFlag = false;
            InitDeviceInfo(deviceInfo);
            retVal = LocalSPM::GetDBManager().GetDeviceInfo(OUT deviceInfo, IN listInfo.deviceID);
            //RTC-Bug:340990
            //Provide CoverArtFlow,Thumbnail Info in Lists if Needed.Thus unnecessary Focus Change in AlbumArtIndexer could be avoided!
            if((LocalSPM::GetDataProvider().IsAlbumArtIndexingSupported(IN deviceInfo.deviceType))&&
               (LocalSPM::GetDataProvider().CoverArtFlow() &&
                LocalSPM::GetDataProvider().IsCoverArtInfoNeededInList(IN listInfo.listType))&&
                (LocalSPM::GetDataProvider().AlbumArtThumbnail() &&
                LocalSPM::GetDataProvider().IsThumbnailInfoNeededInList(IN listInfo.listType)))
            {
                retVal = LocalSPM::GetListControl().AddCoverArtAndThumbnail(INOUT mediaObjectVector);
                if (retVal) //cover art or thumbnail should be available, but not indexed yet
                {
                    retVal = LocalSPM::GetDBManager().GetAlbumArtIndexingComplete(OUT completeFlag, IN listInfo.deviceID);
                    if(MP_NO_ERROR == retVal && false == completeFlag)//cover art should be available, but not indexed yet
                    {
                        /* Send CHANGE_FOCUS message to AlbumArtIndexerSM to change focus of album art indexing */
                        retVal = LocalSPM::GetAlbumArtIndexer().SendChangeFocus(IN listInfo.deviceID, IN listID, IN startIndex, IN sliceSize);
                    }
                }
            }
            else
            {
                if((LocalSPM::GetDataProvider().IsAlbumArtIndexingSupported(IN deviceInfo.deviceType))&&
                    LocalSPM::GetDataProvider().CoverArtFlow() &&
                    LocalSPM::GetDataProvider().IsCoverArtInfoNeededInList(IN listInfo.listType))
                {
                    retVal = LocalSPM::GetListControl().AddCoverArt(INOUT mediaObjectVector);
                    if (retVal) //cover art not indexed yet
                    {
                        retVal = LocalSPM::GetDBManager().GetAlbumArtIndexingComplete(OUT completeFlag, IN listInfo.deviceID);
                        if(MP_NO_ERROR == retVal && false == completeFlag)//cover art should be available, but not indexed yet
                        {
                            /* Send CHANGE_FOCUS message to AlbumArtIndexerSM to change focus of album art indexing */
                            retVal = LocalSPM::GetAlbumArtIndexer().SendChangeFocus(IN listInfo.deviceID, IN listID, IN startIndex, IN sliceSize);
                        }
                    }
                }
                else if((LocalSPM::GetDataProvider().IsAlbumArtIndexingSupported(IN deviceInfo.deviceType))&&
                        LocalSPM::GetDataProvider().AlbumArtThumbnail() &&
                        LocalSPM::GetDataProvider().IsThumbnailInfoNeededInList(IN listInfo.listType))
                {
                    retVal = LocalSPM::GetListControl().AddThumbnail(INOUT mediaObjectVector);
                    if (retVal) //thumbnail not indexed yet
                    {
                        retVal = LocalSPM::GetDBManager().GetAlbumArtIndexingComplete(OUT completeFlag, IN listInfo.deviceID);
                        if(MP_NO_ERROR == retVal && false == completeFlag)//cover art should be available, but not indexed yet
                        {
                            /* Send CHANGE_FOCUS message to AlbumArtIndexerSM to change focus of album art indexing */
                            retVal = LocalSPM::GetAlbumArtIndexer().SendChangeFocus(IN listInfo.deviceID, IN listID, IN startIndex, IN sliceSize);
                        }
                    }
                }
            }
#if USE_IMAGE_OBJECT
        }
#endif
    }
    ticks.elapsed();
    VARTRACE(mediaObjectVector)
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::RequestMediaPlayerFileListSlice(
        vector<tFiles> &mediaFileVector,
        const tListID listID,
        const tIndex startIndex,
        const tIndex sliceSize)
{
    ENTRY
    VARTRACE(listID)
    VARTRACE(startIndex)
    VARTRACE(sliceSize)
    LOCK
    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetListControl().RequestMediaPlayerFileListSlice(OUT mediaFileVector, IN listID, IN startIndex, IN sliceSize);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    if(MP_NO_ERROR == retVal)
    {
        tListInfo listInfo;
        tBoolean withListSize = false;
        InitListInfo(listInfo);
        retVal = LocalSPM::GetListControl().GetListInfo(OUT listInfo, IN listID, IN withListSize);

        tDeviceInfo deviceInfo;
        InitDeviceInfo(deviceInfo);
        retVal = LocalSPM::GetDBManager().GetDeviceInfo(OUT deviceInfo, IN listInfo.deviceID);
        if (LocalSPM::GetDataProvider().IsAlbumArtIndexingSupported(IN deviceInfo.deviceType) && LocalSPM::GetDataProvider().CoverArtFlow())
        {
            retVal = LocalSPM::GetListControl().AddCoverArt(INOUT mediaFileVector,IN listInfo.deviceID);
            if (retVal)
            {
                tAlbumArtIndexingComplete completeFlag = false;
                retVal = LocalSPM::GetDBManager().GetAlbumArtIndexingComplete(OUT completeFlag, IN listInfo.deviceID);
                if(MP_NO_ERROR == retVal && false == completeFlag)//cover art should be available, but not indexed yet
                {
                    /* Send CHANGE_FOCUS message to AlbumArtIndexerSM to change focus of album art indexing */
                    retVal = LocalSPM::GetAlbumArtIndexer().SendChangeFocus(IN listInfo.deviceID, IN listID, IN startIndex, IN sliceSize);
                }
            }
        }
    }
    ticks.elapsed();
    VARTRACE(mediaFileVector)
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::ReleaseList(
        const tListID listID)
{
    ENTRY
    VARTRACE(listID)
    LOCK
    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetListControl().ReleaseList(IN listID);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::RequestListInformation(
        tListType &listType,
        tListID &parentListID,
        tDeviceID &deviceID,
        tPosition &position,
        tListSize &listSize,
        tPath &path,
        const tListID listID)
{
    ENTRY
    VARTRACE(listID)
    LOCK
    memset(&listType, 0, sizeof listType);
    memset(&parentListID, 0, sizeof parentListID);
    memset(&deviceID, 0, sizeof deviceID);
    memset(&position, 0, sizeof position);
    memset(&listSize, 0, sizeof listSize);
    memset(&path, 0, sizeof(tPath));

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        // get repeat mode and set path to "/" if repeat mode is ALL
        tRepeatMode repeatMode;
        retVal = LocalSPM::GetListControl().GetRepeatMode(OUT repeatMode);
        retVal = LocalSPM::GetListControl().RequestListInformation(OUT listType, OUT parentListID, OUT deviceID, OUT position, OUT listSize, IN listID);
        if (RPT_ALL == repeatMode)
        {
            strncpy_r(path, "/", sizeof path);
        }
        else
        {
            // get path of file lists
            LocalSPM::GetListControl().GetPath(OUT path, IN listID);
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(listID)
    VARTRACE(listType)
    VARTRACE(parentListID)
    VARTRACE(deviceID)
    VARTRACE(position)
    VARTRACE(listSize)
    VARTRACE(retVal)
    VARTRACE(path)
    return retVal;
}


tResult MediaPlayerInterface::Quicksearch(
        tPosition &outposition,
        tSearchResult &quicksearchresult,
        const tListID listID,
        tPosition startposition,
        tSearchString searchstring
)
{
    ENTRY
    VARTRACE(listID)
    VARTRACE(searchstring)
    LOCK
    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetListControl().Quicksearch(OUT outposition, OUT quicksearchresult, IN listID, IN startposition, IN searchstring);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(outposition)
    VARTRACE(quicksearchresult)
    VARTRACE(retVal)
    return retVal;
}
tResult MediaPlayerInterface::SearchKeyboardMediaPlayerList(
        vector<tSearchKeyboard> &searchKeyboardList,
        const tListID listID)
{
    ENTRY
    VARTRACE(listID)
    LOCK

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetListControl().SearchKeyboardMediaPlayerList(OUT searchKeyboardList, IN listID);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(searchKeyboardList)
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::SendPlaybackMode(
        const tPlaybackMode playbackMode)
{
    ENTRY
    VARTRACE(playbackMode)
    LOCK

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetPlayerManager().SendPlaybackMode(IN playbackMode);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(retVal)
    return retVal;
}


tResult MediaPlayerInterface::GetPlaybackMode(
        tPlaybackMode &playbackMode)
{
    ENTRY
    //LOCK

    memset(&playbackMode, 0, sizeof playbackMode);

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
        {
            retVal = LocalSPM::GetListControl().GetPlaybackMode(OUT playbackMode);
        }
        else
        {   // set default value
            playbackMode = PBM_NORMAL;
            retVal = MP_NO_ERROR;
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(playbackMode)
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::SendRepeatMode(
        const tRepeatMode repeatMode)
{
    ENTRY
    VARTRACE(repeatMode)
    LOCK

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetPlayerManager().SendRepeatMode(IN repeatMode);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::GetRepeatMode(
        tRepeatMode &repeatMode)
{
    ENTRY
    //LOCK

    memset(&repeatMode, 0, sizeof repeatMode);

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
        {
            retVal = LocalSPM::GetListControl().GetRepeatMode(OUT repeatMode);
        }
        else
        {
            //set default value
            repeatMode = RPT_NONE;
            retVal = MP_NO_ERROR;
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(repeatMode)
    VARTRACE(retVal)
    return retVal;
}


tResult MediaPlayerInterface::GetMediaObject(
        tMediaObject &mediaObject,
        const tMediaObjectID mediaObjectID,
        const tCategoryType categoryType)
{
    ENTRY
    VARTRACE(mediaObjectID)
    VARTRACE(categoryType)
    LOCK

    memset(&mediaObject, 0, sizeof mediaObject);

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetDBManager().GetMediaObject(OUT mediaObject, IN mediaObjectID, IN categoryType,IN MY_MEDIA,IN true /*isObjectWithSameCTYNeeded*/);
        if(MP_NO_ERROR == retVal && LocalSPM::GetDataProvider().CodeSetConversionSupported())
        {
            // If the device is a mass storage device,then invoke  ConvertCodeSetFromSysLang2UTF8
            if (IsMassStorageDevice(mediaObject.deviceType))
            {
                vector<tMediaObject> nowplayingMediaObject;
                nowplayingMediaObject.push_back(mediaObject);
                ConvertCodeSetFromSysLang2UTF8(INOUT nowplayingMediaObject);
                mediaObject = nowplayingMediaObject.at(0);
            }
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(mediaObject)
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::GetMediaObjectAlbumArtInfo(
        const tAlbumArt albumArtString,
        const tSize sizeX,
        const tSize sizeY,
        const tMimeType mimeType,
        const tUserContext userContext)
{
    ENTRY
    VARTRACE(albumArtString)
    VARTRACE(sizeX)
    VARTRACE(sizeY)
    VARTRACE(mimeType)
    VARTRACE(userContext)
    LOCK

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        //NCG3D-245881: Albumart not present for video files
        tFileFormat fileFormat = FFT_UNKNOWN;
        tFileType fileType = FT_UNKNOWN;

        tResult fileFormatError  = LocalSPM::GetDBManager().GetFileFormat(OUT fileFormat, OUT fileType, IN albumArtString);

        if((MP_NO_ERROR == fileFormatError) && (FT_VIDEO == fileType))
        {
            retVal = MP_ERROR;
            ETG_TRACE_ERR(("Album art request should not be received for video file.Aborting the request"));
        }
        else
        {
            retVal = LocalSPM::GetDataProvider().GetMediaObjectAlbumArtInfo(IN albumArtString, IN sizeX, IN sizeY, IN mimeType, IN userContext);
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(retVal)
    return retVal;
}


tResult MediaPlayerInterface::GetMediaObjectAlbumArt(
        const tAlbumArt albumArtString,
        const tSize sizeX,
        const tSize sizeY,
        const tMimeType mimeType,
        const tUserContext userContext)
{
    ENTRY
    VARTRACE(albumArtString)
    VARTRACE(sizeX)
    VARTRACE(sizeY)
    VARTRACE(mimeType)
    VARTRACE(userContext)
    LOCK

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        //NCG3D-245881: Albumart not present for video files
        tFileFormat fileFormat = FFT_UNKNOWN;
        tFileType fileType = FT_UNKNOWN;

        tResult fileFormatError = LocalSPM::GetDBManager().GetFileFormat(OUT fileFormat, OUT fileType, IN albumArtString);

        if((MP_NO_ERROR == fileFormatError) && (FT_VIDEO == fileType))
        {
            retVal = MP_ERROR;
            ETG_TRACE_ERR(("Album art request should not be received for video file.Aborting the request"));
        }
        else
        {
            retVal = LocalSPM::GetDataProvider().GetMediaObjectAlbumArt(IN albumArtString, IN sizeX, IN sizeY, IN mimeType, IN userContext);
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::GetMediaplayerDeviceConnections(
        tNumberOfDevices &numberOfDevices,
        vector<tDeviceInfo> &deviceInfo,
        tBoolean isForDipo)
{
    ENTRY
    //LOCK

    memset(&numberOfDevices, 0, sizeof numberOfDevices);
    memset(&deviceInfo, 0, sizeof deviceInfo);

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetDBManager().GetMediaplayerDeviceConnections(OUT numberOfDevices, OUT deviceInfo, IN true/*allDevices*/);

        // set all device with connection state CS_UNDERVOLTAGE as connected
        vector<tDeviceInfo>::iterator it;
        for(it = deviceInfo.begin(); it < deviceInfo.end(); it++)
        {
            if (CS_UNDERVOLTAGE == it->connectionState)
            {
                it->connected = TRUE;
            }
            //Bug NCG3D-109483
            if ( ( (it->deviceType == DTY_USB) || (it->deviceType == DTY_MTP) ) && LocalSPM::GetDataProvider().CodeSetConversionSupported() ) {
                //Check if usb name is valid UTF-8, if not convert to valid UTF-8
                ConvertDevicenameCodeSetFromSysLang2UTF8((unsigned char *)it->deviceName, sizeof(it->deviceName));
                //ETG_TRACE_USR3(("MediaPlayerInterface::ConvertDevicenameCodeSetFromSysLang2UTF8 after conversion %s",it->deviceName ));
            }
            /*USBPortNumberInfo CAFI06-122 .. isForDipo Handle for PSARCCB-7593*/
            if((LocalSPM::GetDataProvider().AddPortNumberToDeviceName()) && (!isForDipo) && (DCT_USB == it->connectionType))
            {
                tUSBPortNumber retPortNumber;
                strncpy_r(retPortNumber,PORTNUM_DEFAULT,sizeof(retPortNumber));
                tDeviceName tempDevicename = {0};
                retVal = LocalSPM::GetDBManager().GetDevicePortNumberInfo(IN it->deviceID, OUT retPortNumber);
                sprintf(tempDevicename,"%s:%s",retPortNumber,it->deviceName);
                strncpy_r(it->deviceName, tempDevicename,sizeof(it->deviceName));
                ETG_TRACE_USR3(("Interface::GetMediaplayerDeviceConnections:DeviceName: %s",it->deviceName));
            }

            /*DeviceSysPathInfo CMG3G-12660 */
            if((LocalSPM::GetDataProvider().AddDeviceSysPath()) && (DCT_USB == it->connectionType))
            {
                tPath retDeviceSyspath;
                retDeviceSyspath[0] = '\0';
                retVal = LocalSPM::GetDBManager().GetSysPathInfoFromMap(IN it->deviceID, OUT retDeviceSyspath);
                strncpy_r(it->deviceSyspath, retDeviceSyspath,sizeof(it->deviceSyspath));
                ETG_TRACE_USR3(("Interface::GetMediaplayerDeviceConnections:deviceSyspath: %s",it->deviceSyspath));
            }
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(numberOfDevices);
    VARTRACE(deviceInfo);
    VARTRACE(retVal);
    return retVal;
}

tResult MediaPlayerInterface::GetDeviceID(OUT tDeviceID &deviceID,IN const tMountPoint mountPoint)
{
    ENTRY

    tResult retVal = -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetDBManager().GetDevice(OUT deviceID,IN mountPoint);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(deviceID);
    VARTRACE(retVal);
    return retVal;
}

tResult MediaPlayerInterface::GetDeviceInfo(
        tDeviceInfo &deviceInfo,
        const tDeviceID deviceID)
{
    ENTRY
    //LOCK

    memset(&deviceInfo, 0, sizeof deviceInfo);

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetDBManager().GetDeviceInfo(OUT deviceInfo, IN deviceID);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(deviceInfo);
    VARTRACE(retVal);
    return retVal;
}

tResult MediaPlayerInterface::GetConnectedDeviceCount(
        tNumberOfDevices &numberOfDevices)
{
    ENTRY
    //LOCK

    memset(&numberOfDevices, 0, sizeof numberOfDevices);

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        tDeviceCount devCount = 0;
        retVal = LocalSPM::GetDBManager().GetConnectedDeviceCount(OUT devCount);
        numberOfDevices = (tNumberOfDevices)devCount;
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(numberOfDevices)
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::GetImageFileCount(
        tNumberOfMediaObjects &numberOfImageFiles,
        const tDeviceID deviceID)
{
    ENTRY
    //LOCK

    numberOfImageFiles = 0;

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetDBManager().GetNumberofImages(OUT numberOfImageFiles, IN deviceID);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(numberOfImageFiles);
    VARTRACE(retVal);
    return retVal;
}

tResult MediaPlayerInterface::GetNumberOfImagesByPath(tNumberOfMediaObjects &numberOfImages, const tDeviceID deviceID, const tPath path)
{
    ENTRY;
    //LOCK

    numberOfImages = 0;

    tResult retVal = -1;
    TimeTrace ticks(__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState()) {
        retVal = LocalSPM::GetDBManager().GetNumberofImagesByPath(OUT numberOfImages, IN deviceID, IN path);
    }
    else {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state", __PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(numberOfImages);
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::GetIndexingState(
        vector<tIndexingStateResult> &indexingState)
{
    ENTRY
    //LOCK

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
        {
            retVal = LocalSPM::GetDBManager().GetIndexingState(OUT indexingState);
        }
        else
        {
            //set default value
            indexingState.clear();
            retVal = MP_NO_ERROR;
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(indexingState)
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::GetBTProfileInfo(tBTProfileList &profileList)
{
    ENTRY
    //LOCK

    tResult retVal= -1;
    unsigned int i= 0;
    TimeTrace ticks (__PRETTY_FUNCTION__);

    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
        {
            retVal = LocalSPM::GetIPODControl().GetBTProfileInfo(OUT profileList);
        }
        else
        {
            // set default value
            profileList.clear();
            retVal = MP_NO_ERROR;
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();

    for(i=0; i < profileList.size(); i++) {
        VARTRACE(profileList.at(i).deviceID);
        VARTRACE(profileList.at(i).BTProfile);
    }
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::GetDiPONowPlaying(tDeviceID &deviceID, tDiPONowPlaying &nowPlaying)
{
    ENTRY
    //LOCK

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);

    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
        {
            retVal = LocalSPM::GetIPODControl().GetDiPONowPlaying(OUT deviceID, OUT nowPlaying);
        }
        else
        {
            // set default value
            deviceID = DEVICE_ID_NOT_SET;
            InitDiPONowPlaying(nowPlaying);
            retVal = MP_NO_ERROR;
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();

    VARTRACE(deviceID);
    VARTRACE(nowPlaying);
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::GetDiPOPlaybackStatus(tDeviceID &deviceID, tHMIPlaybackState &playbackState)
{
    ENTRY
    //LOCK

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);

    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
        {
            retVal = LocalSPM::GetIPODControl().GetDiPOPlaybackStatus(OUT deviceID, OUT playbackState);
        }
        else
        {
            // set default value
            deviceID = DEVICE_ID_NOT_SET;
            playbackState = HMI_PBS_UNDEFINED;
            retVal = MP_NO_ERROR;
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();

    VARTRACE(deviceID);
    VARTRACE(playbackState);
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::GetDiPOPlaybackShuffleMode(tDeviceID &deviceID, tPlaybackMode &playbackMode)
{
    ENTRY
    //LOCK

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);

    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
        {
            retVal = LocalSPM::GetIPODControl().GetDiPOPlaybackShuffleMode(OUT deviceID, OUT playbackMode);
        }
        else
        {
            // set default value
            deviceID = DEVICE_ID_NOT_SET;
            playbackMode = tPlaybackMode_init;
            retVal = MP_NO_ERROR;
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();

    VARTRACE(deviceID);
    VARTRACE(playbackMode);
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::GetDiPOPlaybackRepeatMode(tDeviceID &deviceID, tRepeatMode &repeatMode)
{
    ENTRY
    //LOCK

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);

    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
        {
            retVal = LocalSPM::GetIPODControl().GetDiPOPlaybackRepeatMode(OUT deviceID, OUT repeatMode);
        }
        else
        {
            // set default value
            deviceID = DEVICE_ID_NOT_SET;
            repeatMode = tRepeatMode_init;
            retVal = MP_NO_ERROR;
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();

    VARTRACE(deviceID);
    VARTRACE(repeatMode);
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::GetDiPOPlaytime(tDeviceID &deviceID, tPlaytime &elapsedPlaytime, tPlaytime &totalPlaytime)
{
    ENTRY
    //LOCK

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);

    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
        {
            retVal = LocalSPM::GetIPODControl().GetDiPOPlaytime(OUT deviceID, OUT elapsedPlaytime, OUT totalPlaytime);
        }
        else
        {
            //set default value
            deviceID = DEVICE_ID_NOT_SET;
            elapsedPlaytime = tPlaytime_init;
            totalPlaytime = tPlaytime_init;
            retVal = MP_NO_ERROR;
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(deviceID);
    VARTRACE(elapsedPlaytime)
    VARTRACE(totalPlaytime)
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::GetDiPOCallState(tDeviceID &deviceID, tDiPOCallState &callStateList)
{
    ENTRY
    //LOCK

    tResult retVal= -1;
    unsigned int i= 0;
    TimeTrace ticks (__PRETTY_FUNCTION__);

    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
        {
            retVal = LocalSPM::GetIPODControl().GetDiPOCallState(OUT deviceID, OUT callStateList);
        }
        else
        {
            //set default value
            deviceID = DEVICE_ID_NOT_SET;
            callStateList.clear();
            retVal = MP_NO_ERROR;
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(deviceID);
    for(i=0; i < callStateList.size(); i++) {
        VARTRACE(callStateList.at(i));
    }
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::GetDiPOCallDuration(tDeviceID &deviceID, unsigned int &callDuraion)
{
    ENTRY
    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);

    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
        {
            retVal = LocalSPM::GetIPODControl().GetDiPOCallDuration(OUT deviceID, OUT callDuraion);
        }
        else
        {
            //set default value
            deviceID = DEVICE_ID_NOT_SET;
            callDuraion = 0;
            retVal = MP_NO_ERROR;
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();

    return retVal;
}

tResult MediaPlayerInterface::GetDiPOCommunications(tDeviceID &deviceID, tDiPOCommunications &communications)
{
    ENTRY
    //LOCK

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);

    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
        {
            retVal = LocalSPM::GetIPODControl().GetDiPOCommunications(OUT deviceID, OUT communications);
        }
        else
        {
            // set default value
            deviceID = DEVICE_ID_NOT_SET;
            InitDiPOCommunications(communications);
            retVal = MP_NO_ERROR;
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();

    VARTRACE(deviceID);
    VARTRACE(communications);
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::GetDiPOLocationInfo(tDeviceID &deviceID, tBool &startStopLocationInfo, tDiPOLocationInfoType &locationInfoType)
{
    ENTRY
    //LOCK

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);

    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
        {
            retVal = LocalSPM::GetIPODControl().GetDiPOLocationInfo(OUT deviceID, OUT startStopLocationInfo,OUT locationInfoType);
        }
        else
        {
            // set default value
            deviceID = DEVICE_ID_NOT_SET;
            startStopLocationInfo = false;
            InitDiPOLocationInfoType(locationInfoType);
            //InitDiPOCommunications(communications);
            retVal = MP_NO_ERROR;
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();

    return retVal;
}

tResult MediaPlayerInterface::GetDiPOGPRMCDataStatusValues(tDeviceID &deviceID, tMountPoint &USBSerialNumber, tDiPOGPRMCDataStatusValues &GPRMCDataStatusValues)
{
    ENTRY
    //LOCK

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);

    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
        {
            retVal = LocalSPM::GetIPODControl().GetDiPOGPRMCDataStatusValues(OUT deviceID, OUT USBSerialNumber, OUT GPRMCDataStatusValues);
        }
        else
        {
            // set default value
            deviceID = DEVICE_ID_NOT_SET;
            // sim4hi 180629: set USBSerialNumber to 0 instead of '0'==0x30==48 as fix for Coverity CID-183946
            memset(USBSerialNumber,0,sizeof(USBSerialNumber));
            InitDiPOGPRMCDataStatusValues(GPRMCDataStatusValues);
            retVal = MP_NO_ERROR;
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();

    return retVal;
}

tResult MediaPlayerInterface::GetIndexingState(tIndexingState &indexingState, const tDeviceID deviceID)
{
    ENTRY
    //LOCK

    VARTRACE(deviceID)
    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);

    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
        {
            retVal = LocalSPM::GetDBManager().GetIndexingState(OUT indexingState, IN deviceID);
        }
        else
        {
            // set default value
            indexingState = IDS_NOT_STARTED;
            retVal = MP_NO_ERROR;
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(indexingState)
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::ActiveMediaDeviceGet(
        tDeviceInfo &deviceInfo,
        tBoolean isForDipo)
{
    ENTRY
    //LOCK

    memset(&deviceInfo, 0, sizeof deviceInfo);

    // set default value
    InitDeviceInfo(deviceInfo);

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
        {
            retVal = LocalSPM::GetCustomControl().ActiveMediaDeviceGet(OUT deviceInfo);

            /*USBPortNumberInfo CAFI06-122 .. isForDipo Handle for PSARCCB-7593*/
            if((MP_NO_ERROR == retVal) && LocalSPM::GetDataProvider().AddPortNumberToDeviceName() && !isForDipo)
            {
                tUSBPortNumber retPortNumber;
                strncpy_r(retPortNumber,PORTNUM_DEFAULT,sizeof(retPortNumber));
                tDeviceName tempDevicename = {0};
                retVal = LocalSPM::GetDBManager().GetDevicePortNumberInfo(IN deviceInfo.deviceID, OUT retPortNumber);
                snprintf (tempDevicename, sizeof(tempDevicename), "%s:%s",retPortNumber,deviceInfo.deviceName); //Hotspot Fix RTC178460
                strncpy_r(deviceInfo.deviceName, tempDevicename,sizeof(tempDevicename));
                ETG_TRACE_USR3(("Interface::ActiveMediaDeviceGet:DeviceName: %s",deviceInfo.deviceName));
            }
        }
        else
        {
            retVal = MP_NO_ERROR;
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(deviceInfo)
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::ActiveMediaDeviceSet(
        const tDeviceID deviceID,
        const tBoolean activeSource,
        tMediaContext mediaContext)
{
    ENTRY
    VARTRACE(deviceID)
    VARTRACE(activeSource)
    VARTRACE(mediaContext)
    LOCK

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetCustomControl().SendActiveMediaDeviceSet(IN deviceID, IN activeSource, IN mediaContext);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(retVal)
    return retVal;
}
tResult MediaPlayerInterface::SendPlaybackSpeed(
        const tPEPlaybackDirection playbackDirection,
        const tPlaybackSpeed playbackSpeed)
{
    ENTRY
    VARTRACE(playbackDirection);
    VARTRACE(playbackSpeed);
    LOCK
    tResult retVal= -1;
    speedstate_e isPlaybackSpeed = ME_SPEEDSTATE_ON;
    tPlaybackAction playbackAction;

    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        tNextPrevSkipCount steps = 0;
        if(0 == steps)
        {
            steps = 1;
        }
        if(PBK_FORWARD == playbackDirection && PBK_SPEED_1X != playbackSpeed)
        {
            playbackAction = PBA_FFWD_START;
        }
        else if(PBK_FORWARD == playbackDirection && PBK_SPEED_1X == playbackSpeed)
        {
            playbackAction = PBA_FFWD_STOP;
        }
        else if(PBK_BACKWARD == playbackDirection && PBK_SPEED_1X != playbackSpeed)
        {
            playbackAction = PBA_FREV_START;
        }
        else if(PBK_BACKWARD == playbackDirection && PBK_SPEED_1X == playbackSpeed)
        {
            playbackAction = PBA_FREV_STOP;
        }
        retVal = LocalSPM::GetPlayerManager().SendPlaybackAction(IN playbackAction, IN steps,playbackSpeed,isPlaybackSpeed);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(retVal)
    return retVal;
}
tResult MediaPlayerInterface::SendPlaybackAction(
        const tPlaybackAction playbackAction,
        const tNextPrevSkipCount nextPrevSkipCount)
{
    ENTRY
    VARTRACE(playbackAction)
    VARTRACE(nextPrevSkipCount)
    LOCK

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        tNextPrevSkipCount steps = nextPrevSkipCount;
        if(0 == steps)
        {
            steps = 1;
        }
        retVal = LocalSPM::GetPlayerManager().SendPlaybackAction(IN playbackAction, IN steps);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::SendSeekTo(
        const tPlaytime playpointPosition,
        const tPlaypointFormat playpointFormat)
{
    ENTRY
    VARTRACE(playpointPosition)
    VARTRACE(playpointFormat)
    LOCK

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetPlayerManager().SendSeekTo(IN playpointPosition, IN playpointFormat);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::PlayItemFromList(
        tListSize &listSize,
        const tListID listID,
        const tIndex index,
        const tPlaytime offset)
{
    ENTRY
    VARTRACE(listID)
    VARTRACE(index)
    VARTRACE(offset)
    LOCK

    memset(&listSize, 0, sizeof listSize);

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);

    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetCustomControl().PlayItemFromList(OUT listSize, IN listID, IN index, IN offset);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(listSize)
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::PlayItemFromListByTag(
        tListSize &listSize,
        const tListID listID,
        const tObjectID objectID)
{
    ENTRY
    VARTRACE(listID)
    VARTRACE(objectID)
    LOCK

    memset(&listSize, 0, sizeof listSize);

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);

    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetCustomControl().PlayItemFromListByTag(OUT listSize, IN listID, IN objectID);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(listSize)
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::PlayMediaPlayerObject(tListID &listID,
        tListSize &listSize,
        const tObjectID objectID,
        const tPlayContinuation playContinuation)
{
    ENTRY
    VARTRACE(objectID)
    VARTRACE(playContinuation)
    LOCK

    memset(&listSize, 0, sizeof listSize);
    memset(&listID, 0, sizeof listID);

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetCustomControl().PlayMediaPlayerObject(OUT listID, OUT listSize, IN objectID, IN playContinuation);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(listID);
    VARTRACE(listSize)
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::GetNowPlaying(
        tNowPlaying &nowPlaying)
{
    ENTRY
    //LOCK

    memset(&nowPlaying, 0, sizeof nowPlaying);

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
        {
            retVal = LocalSPM::GetPlayerManager().GetNowPlaying(OUT nowPlaying);

            if(MP_NO_ERROR == retVal && LocalSPM::GetDataProvider().CodeSetConversionSupported())
            {
                // If the device is a mass storage device or MTP Device,then invoke  ConvertCodeSetFromSysLang2UTF8
                if ((IsMassStorageDevice(nowPlaying.object.deviceType)) || (DTY_MTP == nowPlaying.object.deviceType))
                {
                    vector<tMediaObject> nowplayingMediaObject;
                    nowplayingMediaObject.push_back(nowPlaying.object);
                    ConvertCodeSetFromSysLang2UTF8(INOUT nowplayingMediaObject);
                    nowPlaying.object = nowplayingMediaObject.at(0);
                }
            }
        }
        else
        {
            //set default value
            nowPlaying.listID = 0;
            nowPlaying.state = NP_NO_DEVICE;
            nowPlaying.position = 0;
            nowPlaying.objectID = 0;
            InitMediaObject(nowPlaying.object);
            retVal = MP_NO_ERROR;
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(nowPlaying)
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::GetCurrentFolderPath(tPath &currentFolderPath, tListID& listID)
{
    ENTRY
    //LOCK

    listID = 0;
    currentFolderPath[0] = '\0';

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
        {
            tNowPlaying nowPlaying;
            retVal = LocalSPM::GetPlayerManager().GetNowPlaying(OUT nowPlaying);
            if (MP_NO_ERROR == retVal)
            {
                tListInfo listInfo;
                retVal = LocalSPM::GetListControl().GetListInfo(listInfo, nowPlaying.listID, FALSE);
                if (MP_NO_ERROR == retVal)
                {
                    retVal = MP_NO_ERROR;
                    // if folder list and device is a mass storage device
                    if ((IsFileList(listInfo.listType)) &&
                            ((IsMassStorageDevice(nowPlaying.object.deviceType)) || (DTY_MTP == nowPlaying.object.deviceType)))
                    {
                        listID = nowPlaying.listID;
                        strncpy_r(currentFolderPath, nowPlaying.object.path, sizeof currentFolderPath);
                    }
                }
            }
        }
        else
        {
            //use default value
            retVal = MP_NO_ERROR;
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(listID)
    VARTRACE(currentFolderPath)
    return retVal;
}

tResult MediaPlayerInterface::GetPlaytime(
        tPlaytime &elapsedPlaytime,
        tPlaytime &totalPlaytime,
        tObjectID &objectID)
{
    ENTRY
    //LOCK

    memset(&elapsedPlaytime, 0, sizeof elapsedPlaytime);
    memset(&totalPlaytime, 0, sizeof totalPlaytime);
    objectID = OBJECT_ID_NONE;

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
        {
            //Note:objectID is reliable only for Mass Storage devices & MTP Devices
            retVal = LocalSPM::GetPlayerManager().GetPlaytime(OUT elapsedPlaytime, OUT totalPlaytime, OUT objectID);
        }
        else
        {
            //set default value
            elapsedPlaytime = 0;
            totalPlaytime = 0;
            retVal = MP_NO_ERROR;
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(elapsedPlaytime)
    VARTRACE(totalPlaytime)
    VARTRACE(objectID)
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::GetPlaybackState(
        tHMIPlaybackState &playbackState)
{
    ENTRY
    //LOCK

    memset(&playbackState, 0, sizeof playbackState);

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
        {
            retVal = LocalSPM::GetPlayerManager().GetPlaybackState(OUT playbackState);
        }
        else
        {
            //set default value
            playbackState = HMI_PBS_STOPPED;
            retVal = MP_NO_ERROR;
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(playbackState)
    VARTRACE(retVal)
    return retVal;
}



tResult MediaPlayerInterface::ActivateFavorite(
        tFavoriteID &retFavID,
        const tFavoriteID favID,
        const tLogicalAVChannel audioChn,
        const tLogicalAVChannel videoChn)
{
    ENTRY
    VARTRACE(favID)
    VARTRACE(audioChn)
    VARTRACE(videoChn)
    LOCK

    memset(&retFavID, 0, sizeof retFavID);

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetCustomControl().GetFavoritesManager().ActivateFavorite(OUT retFavID, IN favID, IN audioChn, IN videoChn);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(retFavID)
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::StoreFavorite(
        tFavoriteID &retFavID,
        const tObjectID objectID)
{
    ENTRY
    VARTRACE(objectID)
    LOCK

    memset(&retFavID, 0, sizeof retFavID);

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetCustomControl().GetFavoritesManager().StoreFavorite(OUT retFavID, IN objectID);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(retFavID)
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::DeleteFavorite(
        tFavoriteID &retFavID,
        const tFavoriteID favID)
{
    ENTRY
    VARTRACE(favID)
    LOCK

    memset(&retFavID, 0, sizeof retFavID);

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetCustomControl().GetFavoritesManager().DeleteFavorite(OUT retFavID, IN favID);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(retFavID)
    VARTRACE(retVal)
    return retVal;
}


tResult MediaPlayerInterface::GetFavoriteInfo(
        tFavoriteInfo &favInfo,
        const tFavoriteID favID)
{
    ENTRY
    VARTRACE(favID)
    //LOCK

    memset(&favInfo, 0, sizeof favInfo);

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetCustomControl().GetFavoritesManager().GetFavoriteInfo(OUT favInfo, IN favID);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(favInfo)
    VARTRACE(retVal)
    return retVal;
}


tResult MediaPlayerInterface::GetAllFavoriteInfo(vector<tFavoriteInfo> &favInfo)
{
    ENTRY
    //LOCK

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetCustomControl().GetFavoritesManager().GetAllFavoriteInfo(OUT favInfo);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(favInfo)
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::ResetFavorites(void)
{
    ENTRY
    LOCK

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetCustomControl().GetFavoritesManager().ResetFavorites();
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::ClearMediaPlayerData(void)
{
    ENTRY
    LOCK

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        /*Before recreating DB move all connected devices to CS_DISCONNECTED.*/
        tNumberOfDevices numberOfDevices = 0;
        vector<tDeviceInfo> deviceInfos;
        tDeviceInfo deviceInfoDVD;
        InitDeviceInfo(deviceInfoDVD);
        retVal = LocalSPM::GetDBManager().GetMediaplayerDeviceConnections(OUT numberOfDevices, OUT deviceInfos, IN true);//Getting list of all the devices from db
        VARTRACE(numberOfDevices);
        if (retVal)
        {
            ETG_TRACE_ERR(("GetMediaplayerDeviceConnections returned error: %d", retVal));
        }
        else
        {
            for( unsigned int i = 0; i<numberOfDevices; i++ )
            {
                /* Copy the device Info Of DTY_DVD_DRIVE as the device is internally connected*/
                if(deviceInfos[i].deviceType == DTY_DVD_DRIVE)
                {
                    deviceInfoDVD = deviceInfos[i];
                    VARTRACE(deviceInfoDVD);
                }
                /*Connection state is changed only for devices which are connected*/
                if(deviceInfos[i].connectionState == CS_CONNECTED)
                {
                    deviceInfos[i].connectionState = CS_DISCONNECTED;
                    retVal = LocalSPM::GetDBManager().SetConnectionState(deviceInfos[i].deviceID, deviceInfos[i].connectionState);
                    if(MP_NO_ERROR != retVal) //CID 22350
                    {
                        ETG_TRACE_ERR(("Error while SetConnectionState"));
                    }
                }
            }
        }
        /*Call UpdateMediaPlayerDeviceConnections to make sure that he mediaplayer deivce connections updated properly before SPM_STATE_OFF */
        retVal = LocalSPM::GetOutputWrapper().UpdateMediaPlayerDeviceConnections(); //CID 22350
        if(MP_NO_ERROR != retVal)
        {
            ETG_TRACE_ERR(("Error while UpdateMediaPlayerDeviceConnections"));
        }

        if(1 == LocalSPM::GetDataProvider().CDRippingSupport())
        {
            if(1 == LocalSPM::GetDVDControl().IsInState(DVDControlSM::Ripping))
            {
                LocalSPM::GetCDRipperControl().StopRippingByUser();
            }

            ETG_TRACE_USR3(("Delete all ripped contents"));
            tDeleteMediaByTagsParam deleteMediaByTagsParam;
            memset(&deleteMediaByTagsParam,0,sizeof(tDeleteMediaByTagsParam));
            deleteMediaByTagsParam.listType = LTY_SONG;
            deleteMediaByTagsParam.tag1.tag     = 0;
            deleteMediaByTagsParam.tag2.tag     = 0;
            deleteMediaByTagsParam.tag3.tag     = 0;
            deleteMediaByTagsParam.tag4.tag    = 0;
            deleteMediaByTagsParam.deviceID = MUSICBOX_DEVICE_ID;
            retVal = LocalSPM::GetCustomControl().DeleteMediaContent(deleteMediaByTagsParam, 1);
            if(MP_NO_ERROR != retVal)
            {
                ETG_TRACE_ERR(("Error while deleting media content"));
            }
        }

        retVal = LocalSPM::GetDataProvider().ClearMediaPlayerData();//CID 22350
        if(MP_NO_ERROR != retVal)
        {
            ETG_TRACE_ERR(("Error while ClearMediaPlayerData"));
        }
        else
        {
            // Fix for RTC 726828: After clearing DB content, send updated device connections list to clients.
            // On receiving the same, device list maintained by clients are to be cleared.
            retVal = LocalSPM::GetOutputWrapper().UpdateMediaPlayerDeviceConnections();
            if(MP_NO_ERROR != retVal)
            {
                ETG_TRACE_ERR(("Error while UpdateMediaPlayerDeviceConnections"));
            }
        }

        /*Connection state of DTY_DVD_DRIVE is changed to attached after factory reset(NCG3D-87674)*/
        if((deviceInfoDVD.deviceType == DTY_DVD_DRIVE) && (deviceInfoDVD.deviceID != DEVICE_ID_NOT_SET))
        {
            deviceInfoDVD.connectionState = CS_ATTACHED;
            deviceInfoDVD.deviceID = DEVICE_ID_NOT_SET ;
            VARTRACE(deviceInfoDVD);

            deviceInfos.clear();
            deviceInfos.push_back(deviceInfoDVD);
            retVal = LocalSPM::GetDBManager().DeviceChanged(INOUT deviceInfos);
            if (retVal != MP_NO_ERROR)
            {
                ETG_TRACE_ERR(("Error on notifying DVD Drive connection after factory reset"));
            }

        }

        // Get onDeviceManagerNotifyConnection CCA property for device connection update
        ETG_TRACE_USR4(("Requesting DeviceManagerNotifyConnection"));
        LocalSPM::GetOutputWrapper().RequestDeviceManagerNotifyConnection();

    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(retVal)
    return retVal;
}


tResult MediaPlayerInterface::SetLanguage(
        const tLanguageType language)
{
    ENTRY
    VARTRACE(language)
    LOCK

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetCustomControl().SetLanguage(IN language);
        retVal = LocalSPM::GetIPODControl().SetCurrentLanguage(IN language);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::GetLanguage(tLanguageType &language)
{
    ENTRY
    //LOCK

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetDBManager().GetLanguage(OUT language);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(retVal)
    VARTRACE(language)
    return retVal;
}

tResult MediaPlayerInterface::SetOutsideTemperature(const tOutsideTemperature outsideTemperature)
{
    ENTRY
    VARTRACE(outsideTemperature)
    LOCK

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        LocalSPM::GetDataProvider().OutsideTemperature = (int) outsideTemperature;
        retVal = LocalSPM::GetIPODControl().OutsideTemperatureChanged();
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::SetDiPOStopReason(const tDiPOStopReason diPOStopReason)
{
    ENTRY
    VARTRACE(diPOStopReason)
    LOCK

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        LocalSPM::GetDataProvider().DiPOStopReason = (int) diPOStopReason;
        retVal = MP_NO_ERROR;
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(retVal)
    return retVal;
}

#if 0
tResult MediaPlayerInterface::DeviceChanged(const tUndervoltage undervoltage, const vector<tDeviceProperty> deviceProperties)
{
    ENTRY
    VARTRACE(deviceProperties)
    VARTRACE(undervoltage)
    LOCK

    tResult retVal= -1;

    TimeTrace ticks (__PRETTY_FUNCTION__);
    // get current spm state
    tSPMState SPMState = LocalSPM::GetInstance().GetSPMState();
    if (SPM_STATE_NORMAL == SPMState)
    {
        // if DeviceManager signals under voltage
        if (undervoltage)
        {
            // read current playermanager state and store it in member variables
            retVal = LocalSPM::GetPlayerManager().GetAllocateState(OUT mAllocateState, OUT mAudioOutputDevice);
            if (retVal) return retVal;

            // change to under voltage state
            retVal = LocalSPM::GetInstance().StateChangeUndervoltage(true);
            if (retVal) return retVal;
        }
        else
        {
            // inform MediaPlayer only if we are not in undervoltage
            retVal = LocalSPM::GetDBManager().DeviceChanged(deviceProperties);
        }

    }
    // if DeviceManager signals end of under voltage and the MediaPlayer is  in undervoltage state
    else if (SPM_STATE_UNDERVOLTAGE == SPMState && FALSE == undervoltage)
    {
        // change to normal voltage state
        retVal = LocalSPM::GetInstance().StateChangeUndervoltage(false);
        if (retVal) return retVal;

        // restore playermanager state
        retVal = LocalSPM::GetPlayerManager().RestoreAllocateState(IN mAllocateState, IN mAudioOutputDevice);
        if (retVal) return retVal;

        // we are back in Normal and can inform the MediaPlayer about changed devices
        retVal = LocalSPM::GetDBManager().DeviceChanged(deviceProperties);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(retVal)
    return retVal;
}
#endif

tResult MediaPlayerInterface::DeviceChanged(const tUndervoltage undervoltage, const vector<tDeviceInfo> deviceInfos)
{
    ENTRY
    VARTRACE(deviceInfos);
    VARTRACE(undervoltage);
    LOCK

    tResult retVal = -1;

    TimeTrace ticks(__PRETTY_FUNCTION__);
    // get current spm state
    tSPMState SPMState = LocalSPM::GetInstance().GetSPMState();
    tBool isMultiPartition = false;

    if((LocalSPM::GetDataProvider().MultiPartitionSupported())) // Check if Multipartition feature is supported
    {
        if(!deviceInfos.empty())  //Check if device present in deviceInfos vector is a part of Multipartition device
        {
            vector<tDeviceInfo>::const_iterator it;
            for(it = deviceInfos.begin(); it != deviceInfos.end(); it++)
            {
                ETG_TRACE_COMP(("DeviceChanged: deviceName : %s",it->deviceName));
                ETG_TRACE_COMP(("DeviceChanged: AccessoryName2 : %s",it->accessoryName2));
                if((it->deviceType==DTY_USB) ||(it->deviceType==DTY_SD ))
                {
                    if(it->connectionState == CS_UNDEFINED) return retVal; //If device connected is undefined, continue loop to next device

                    /*Multipartition device check*/
                    tNumberOfPartitions nPartitionCount = 0;

                    if(0 != strncmp(it->accessoryName2,"",strlen_r(it->accessoryName2)))
                    {
                        nPartitionCount = 1;
                        //searching from the next element in the list
                        vector<tDeviceInfo>::const_iterator it2;
                        for (it2 = it+1; it2 != deviceInfos.end(); it2++)
                        {
                            ETG_TRACE_COMP(("DeviceChanged: it2->accessoryName2:%s ",it2->accessoryName2));
                            if(((it2->deviceType==DTY_USB) ||(it2->deviceType==DTY_SD )) && (!strncmp(it->accessoryName2,it2->accessoryName2,sizeof(tAccessoryName2))))
                            {
                                nPartitionCount++;
                                ETG_TRACE_COMP(("DeviceChanged: nPartitionCount : %d",nPartitionCount));
                            }
                        }
                    }
                    if(nPartitionCount > 1)
                    {
                        isMultiPartition = true;
                        ETG_TRACE_USR4(("Multi Partition Device : %d partitions",nPartitionCount));
                        break;
                    }
                }
            }
        }
    }

    if (SPM_STATE_NORMAL == SPMState)
    {
        // if DeviceManager signals under voltage
        if (undervoltage)
        {
            // read current playermanager state and store it in member variables
            retVal = LocalSPM::GetPlayerManager().GetAllocateState(OUT mAllocateState, OUT mAudioOutputDevice);
            if (retVal) return retVal;

            // change to under voltage state
            retVal = LocalSPM::GetInstance().StateChangeUndervoltage(IN true);
            if (retVal) return retVal;

        }
        else
        {
            //inform MediaPlayer only if we are not in undervoltage
            if(isMultiPartition)
            {
                retVal = LocalSPM::GetDBManager().MultiPartitionDeviceChanged(IN deviceInfos);
            }
            else
            {
                retVal = LocalSPM::GetDBManager().DeviceChanged(IN deviceInfos);
            }
        }
    }
    // if DeviceManager signals end of under voltage and the MediaPlayer is  in undervoltage state
    else if (SPM_STATE_UNDERVOLTAGE == SPMState && FALSE == undervoltage)
    {
        // change to normal voltage state
        retVal = LocalSPM::GetInstance().StateChangeUndervoltage(IN false);
        if (retVal) return retVal;

        // restore playermanager state
        retVal = LocalSPM::GetPlayerManager().RestoreAllocateState(IN mAllocateState, IN mAudioOutputDevice);
        if (retVal) return retVal;

        //PSARCCB-6616 :Check Any BT Device is in CS_UNDERVOLTAGE.If so, update its ConnectionState to CS_ATTACHED
        tDeviceID btDeviceID;
        retVal = LocalSPM::GetDBManager().GetDevice(OUT btDeviceID, IN DTY_BLUETOOTH,IN false /*allDevices*/,IN false /*virtualDevice*/,IN CS_UNDERVOLTAGE);
        if(DEVICE_ID_NOT_SET != btDeviceID)
        {
            tDeviceInfo deviceInfo;
            vector<tDeviceInfo> btDeviceInfos;
            retVal = LocalSPM::GetDBManager().GetDeviceInfo(OUT deviceInfo, IN btDeviceID);
            if(MP_NO_ERROR == retVal)
            {
                deviceInfo.connectionState = CS_ATTACHED;
                btDeviceInfos.push_back(deviceInfo);
                retVal = LocalSPM::GetDBManager().DeviceChanged(IN btDeviceInfos);
            }
        }

        // we are back in Normal and can inform the MediaPlayer about changed devices
        if(isMultiPartition)
        {
            retVal = LocalSPM::GetDBManager().MultiPartitionDeviceChanged(IN deviceInfos);
        }
        else
        {
            retVal = LocalSPM::GetDBManager().DeviceChanged(IN deviceInfos);
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was in Normal state", __PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(retVal);
    return retVal;
}

MediaPlayerInterface::MediaPlayerInterface()
{
    ENTRY
    mOpticalDiscSlotState = DM_OPTDISC_UNDEFINED_STATE;
    isCreated = FALSE;
}

MediaPlayerInterface::~MediaPlayerInterface()
{
    ENTRY
}

tResult MediaPlayerInterface::TransferTags(const tDeviceID deviceTag,
        const vector<tTagTransfer> transferTags,
        tTagTransferStatus &transferStatus,
        vector<tUntransferredTag> &untransferredTags)
{
    ENTRY
    VARTRACE(deviceTag)
    VARTRACE(transferTags)
    LOCK

    tResult retVal = -1;
    TimeTrace ticks(__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState()) {
        tDeviceInfo deviceInfo;
        LocalSPM::GetDBManager().GetDeviceInfo(OUT deviceInfo, IN deviceTag);
        retVal = LocalSPM::GetIPODControl().SendTransferTags(IN deviceInfo.mountPoint,
                IN transferTags, OUT transferStatus, OUT untransferredTags);
    } else {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(transferStatus)
    VARTRACE(untransferredTags)
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::SendAppControlConnect(const tMountPoint mountPoint,
        const tProtocolName protocol, const tBundleSeedID bundleSeedID, const tAppName appName, const tAppLaunchOption launchOption)
{
    ENTRY
    VARTRACE(mountPoint)
    VARTRACE(protocol)
    VARTRACE(bundleSeedID)
    VARTRACE(appName)
    VARTRACE(launchOption)
    LOCK

    tResult retVal = -1;
    TimeTrace ticks(__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState()) {

        if(LocalSPM::GetDataProvider().iPodControlIAP2RequestAppLaunchEnabled())
        {
            retVal = LocalSPM::GetIPODControl().SendAppControlConnect(IN mountPoint,
                    IN protocol, IN bundleSeedID, IN appName, IN launchOption);
        }
        else
        {
            ETG_TRACE_USR3(("RequestAppLaunch is not enabled"));
        }

    } else {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::SendAppControlCommand(const tMountPoint mountPoint,
        const tAppName appName, const tSessionID sessionID, const tSize sizeOfBuffer,
        const tCommandBufferPtr commandBuffer, const tUserContext userContext)
{
    ENTRY
    VARTRACE(mountPoint)
    VARTRACE(appName)
    VARTRACE(sessionID)
    VARTRACE(sizeOfBuffer)
    LOCK

    tResult retVal = -1;
    TimeTrace ticks(__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState()) {
        retVal = LocalSPM::GetIPODControl().SendAppControlCommand(IN mountPoint,
                IN appName, IN sessionID, IN sizeOfBuffer, IN commandBuffer, IN userContext);
    } else {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::SendAppControlClose(const tMountPoint mountPoint,
        const tAppName appName, const tSessionID sessionID)
{
    ENTRY
    VARTRACE(mountPoint)
    VARTRACE(appName)
    VARTRACE(sessionID)
    LOCK

    tResult retVal = -1;
    TimeTrace ticks(__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState()) {
        retVal = LocalSPM::GetIPODControl().SendAppControlClose(IN mountPoint, IN appName, IN sessionID);
    } else {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::StartAudio(const tMountPoint mountPoint)
{
    ENTRY
    VARTRACE(mountPoint)
    LOCK

    tResult retVal = -1;
    TimeTrace ticks(__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState()) {
        retVal = LocalSPM::GetIPODControl().SendStartAudio(IN mountPoint);
    } else {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::StopAudio(const tMountPoint mountPoint)
{
    ENTRY
    VARTRACE(mountPoint)
    LOCK

    tResult retVal = -1;
    TimeTrace ticks(__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState()) {
        retVal = LocalSPM::GetIPODControl().SendStopAudio(IN mountPoint);
    } else {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::Umount(tErrorCode &errorCode, const tDeviceID deviceID)
{
    ENTRY;
    VARTRACE(deviceID)
    LOCK

    tResult retVal = -1;
    errorCode = MP_NO_ERROR;
    TimeTrace ticks(__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState()) {
        retVal = LocalSPM::GetDeviceDispatcher().SendUmount(OUT errorCode, IN deviceID);
    } else {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::Reindexing(const tDeviceID deviceID)
{
    ENTRY;
    VARTRACE(deviceID)
    LOCK

    tResult retVal = -1;
    TimeTrace ticks(__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState()) {
        //Roadmap 13005 SupportOfInternalFlash
        tDeviceCount deviceCount = 1;
        retVal = LocalSPM::GetIndexer().SendReindexing(IN deviceCount, IN deviceID);
    } else {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::DiagnosisPlaySong(tDeviceType deviceType, tFilename filename)
{
    ENTRY;
    VARTRACE(deviceType)
    VARTRACE(filename)
    LOCK

    tResult retVal = -1;
    TimeTrace ticks(__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState()) {
        retVal = LocalSPM::GetCustomControl().DiagnosisPlaySong(IN deviceType, IN filename);
    } else {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::CreateInternalFileNamePlaylist(tPath &playlistPath, const tDeviceID deviceTag, const tPlaylistName playlistName)
{
    ENTRY;
    VARTRACE(deviceTag)
    VARTRACE(playlistName)
    LOCK

    tResult retVal = -1;
    TimeTrace ticks(__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState()) {
        // can be activated as soon as function is implemented
        retVal = LocalSPM::GetListControl().CreateInternalFileNamePlaylist(OUT playlistPath, IN deviceTag, IN playlistName);
    } else {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(playlistPath);
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::AddFileNamesToPlaylist(const tPath playlistPath, const tDeviceID deviceTag, const vector<string> filenamePathVector, const vector<tPosition> positionVector)
{
    ENTRY;
    VARTRACE(playlistPath)
    LOCK

    tResult retVal = -1;
    TimeTrace ticks(__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState()) {
        tPlayListAddOperation PlayListAddOperation;

        strncpy_r(PlayListAddOperation.PlayListPath, playlistPath,sizeof(PlayListAddOperation.PlayListPath));
        PlayListAddOperation.deviceID = deviceTag;
        PlayListAddOperation.filenamePaths = filenamePathVector;
        PlayListAddOperation.ListOfPositions = positionVector;
        retVal = LocalSPM::GetCustomControl().AddFileNamesToPlayList(IN PlayListAddOperation);
    } else {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::DeleteEntriesFromPlaylist(const tPath playlistPath, const tDeviceID deviceTag, const vector<tPosition> positionVector)
{
    ENTRY;
    VARTRACE(playlistPath)

    LOCK

    tResult retVal = -1;
    TimeTrace ticks(__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState()) {
        tPlayListDeleteOperation PlayListDeleteOperation;
        strncpy_r(PlayListDeleteOperation.PlayListPath,playlistPath,sizeof(PlayListDeleteOperation.PlayListPath));
        PlayListDeleteOperation.ListOfPositions = positionVector;
        // sim4hi 180629: initialized deviceID field as fix for Coverity CID-148384
        PlayListDeleteOperation.deviceID = deviceTag;
        // can be activated as soon as function is implemented
        retVal = LocalSPM::GetCustomControl().DeleteEntriesFromPlaylist(IN PlayListDeleteOperation);
    } else {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::RenameInternalPlaylist(const tDeviceID deviceTag, const tPath playlistPath,  const tPlaylistName playlistName)
{
    ENTRY;
    VARTRACE(playlistPath)
    VARTRACE(playlistName)
    LOCK

    tResult retVal = -1;
    TimeTrace ticks(__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState()) {
        // can be activated as soon as function is implemented
        retVal = LocalSPM::GetListControl().RenameInternalPlaylist(IN deviceTag, IN playlistPath, IN playlistName);
    } else {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::DeleteInternalPlaylist(const tDeviceID deviceTag, const tPath playlistPath)
{
    ENTRY;
    VARTRACE(playlistPath)
    LOCK

    tResult retVal = -1;
    TimeTrace ticks(__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState()) {
        // can be activated as soon as function is implemented
        retVal = LocalSPM::GetListControl().DeleteInternalPlaylist(IN deviceTag, IN playlistPath);
    } else {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::CreateMediaPlayerInternalPlaylistList(tListID& listID, tListSize& listSize, const tDeviceID deviceTag)
{
    ENTRY;
    VARTRACE(deviceTag)
    LOCK

    tResult retVal = -1;
    TimeTrace ticks(__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState()) {
        retVal = LocalSPM::GetListControl().CreateMediaPlayerInternalPlaylistList(OUT listID, OUT listSize, IN deviceTag);
    } else {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::GetListPositionOfObject(tPosition &position, const tListID listID, const tObjectID objectID)
{
    ENTRY;
    VARTRACE(listID)
    VARTRACE(objectID)
    LOCK

    tResult retVal = -1;
    TimeTrace ticks(__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState()) {
        retVal = LocalSPM::GetListControl().GetPositionInList(OUT position, IN listID, IN objectID);
    }
    else {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state", __PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::StartSlideshow(const tListID listID, const tObjectID objectID)
{
    ENTRY;
    VARTRACE(listID)
    VARTRACE(objectID)
    LOCK

    tResult retVal = -1;
    TimeTrace ticks(__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState()) {
        retVal = LocalSPM::GetPictureViewer().StartSlideshow(OUT listID, IN objectID);
    } else {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::SendSlideshowAction(const tPlaybackAction playbackAction, const tNextPrevSkipCount nextPrevSkipCount)
{
    ENTRY;
    VARTRACE(playbackAction)
    VARTRACE(nextPrevSkipCount)
    LOCK

    tResult retVal = -1;
    TimeTrace ticks(__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState()) {
        retVal = LocalSPM::GetPictureViewer().RequestSlideshowAction(IN playbackAction, IN nextPrevSkipCount);
    } else {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::GetSlideshowTime(tSlideshowTime &slideshowTime)
{
    ENTRY;
    //LOCK

    tResult retVal = -1;
    TimeTrace ticks(__PRETTY_FUNCTION__);
    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
        {
            retVal = LocalSPM::GetPictureViewer().GetSlideshowTime(OUT slideshowTime);
        }
        else
        {
            // use default values
            slideshowTime = 0;
            retVal = MP_NO_ERROR;
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Init state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(slideshowTime)
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::SetSlideshowTime(const tSlideshowTime slideshowTime)
{
    ENTRY;
    VARTRACE(slideshowTime)
    LOCK

    tResult retVal = -1;
    TimeTrace ticks(__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState()) {
        retVal = LocalSPM::GetPictureViewer().SetSlideshowTime(IN slideshowTime);
    } else {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();

    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::GetNowShowing(tNowShowing &nowShowing)
{
    ENTRY;
    //LOCK

    tResult retVal = -1;
    TimeTrace ticks(__PRETTY_FUNCTION__);
    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
        {
            retVal = LocalSPM::GetPictureViewer().GetNowShowing(OUT nowShowing);
        }
        else
        {
            // use default values
            nowShowing.listID = 0;
            nowShowing.state = NP_NO_DEVICE;
            nowShowing.position = 0;
            InitMediaObject(nowShowing.object);
            nowShowing.nextFile[0] = '\0';
            retVal = MP_NO_ERROR;
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Init state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(nowShowing)
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::GetSlideshowState(tHMIPlaybackState &playbackState)
{
    ENTRY;
    //LOCK

    tResult retVal = -1;
    TimeTrace ticks(__PRETTY_FUNCTION__);
    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState()) {
            retVal = LocalSPM::GetPictureViewer().GetSlideshowState(OUT playbackState);
        }
        else
        {
            // use default values
            playbackState = HMI_PBS_STOPPED;
            retVal = MP_NO_ERROR;
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(playbackState)
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::DiPOActiveDevice(const tDeviceID deviceID, const tDiPOActive diPOActive, const tUserContext userContext)
{
    ENTRY;
    VARTRACE(deviceID)
    VARTRACE(diPOActive)
    VARTRACE(userContext)
    LOCK

    tResult retVal = -1;
    TimeTrace ticks(__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState()) {
        retVal = LocalSPM::GetIPODControl().SendDiPOActiveDevice(IN deviceID, IN diPOActive, IN userContext);
    } else {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();

    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::DiPORoleSwitchRequest(const tDeviceID deviceID, const tDiPORoleStatus diPORoleStatus, const tUserContext userContext,const tAppInfo appInfo)
{
    ENTRY;
    VARTRACE(deviceID)
    VARTRACE(diPORoleStatus)
    VARTRACE(userContext)
    LOCK

    tResult retVal = -1;
    TimeTrace ticks(__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState()) {
        retVal = LocalSPM::GetIPODControl().SendDiPORoleSwitchRequest(IN deviceID, IN diPORoleStatus, IN userContext,IN appInfo);
    } else {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();

    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::DiPORoleSwitchRequest(const tDeviceID deviceID, const tDiPORoleStatus diPORoleStatus, const tUserContext userContext)
{
    ENTRY;
    tResult retVal = -1;

    tAppInfo appinfo;

    retVal  = DiPORoleSwitchRequest(deviceID,diPORoleStatus, userContext,appinfo);

    return retVal;
}

tResult MediaPlayerInterface::DiPORoleSwitchRequiredResult(const tDeviceID deviceID, const tDiPOSwitchReqResponse diPOSwitchReqResponse)
{
    ENTRY
    tResult retVal = -1;

    tAppInfo appinfo;

    retVal = DiPORoleSwitchRequiredResult(deviceID,diPOSwitchReqResponse,appinfo);

    return retVal;
}


tResult MediaPlayerInterface::DiPORoleSwitchRequiredResult(const tDeviceID deviceID, const tDiPOSwitchReqResponse diPOSwitchReqResponse,const tAppInfo appInfo)
{
    ENTRY

    VARTRACE(deviceID)
    VARTRACE(diPOSwitchReqResponse)
    LOCK

    tResult retVal = -1;
    TimeTrace ticks(__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState()) {
        retVal = LocalSPM::GetIPODControl().OnDiPORoleSwitchRequiredResult(IN deviceID, IN diPOSwitchReqResponse,appInfo);
    } else {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();

    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::DiPOTransferGPSData(tDeviceID deviceID, const tMountPoint USBSerialNumber, const tDiPOGPGGAData diPOGPGGAData,const tDiPOGPRMCData diPOGPRMCData, const tDiPOGPGSVData diPOGPGSVData, const tDiPOGPHDTData diPOGPHDTData, const tUserContext userContext)
{
    ENTRY;
    VARTRACE(deviceID)
    VARTRACE(USBSerialNumber)
    VARTRACE(diPOGPGGAData)
    VARTRACE(diPOGPRMCData)
    VARTRACE(diPOGPGSVData)
    VARTRACE(diPOGPHDTData)
    VARTRACE(userContext)
    LOCK

    tResult retVal = -1;
    TimeTrace ticks(__PRETTY_FUNCTION__);

    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState()) {
        if( 0 == deviceID )
        {
            // Fix for NCG3D-196834: Metadata not updated once device is unplugged during active CPW session
            // Prioritize iAP Wifi over iAP USB session when device is connected over both USB and Wifi.
            deviceID = GetAppleDeviceIDUsingUSBSerialNumber(USBSerialNumber);
        }

        if(DEVICE_ID_NOT_SET != deviceID)
        {
            retVal = LocalSPM::GetIPODControl().SendDiPOTransferGPSData(IN deviceID, IN diPOGPGGAData, IN diPOGPRMCData, IN diPOGPGSVData, IN diPOGPHDTData, IN userContext);
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();

    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::DiPOTransferDRData(tDeviceID deviceID, const tMountPoint USBSerialNumber, const tDiPOPASCDData diPOPASCDData, const tDiPOPAGCDData diPOPAGCDData, const tDiPOPAACDData diPOPAACDData, const tUserContext userContext)
{
    ENTRY;
    VARTRACE(deviceID)
    VARTRACE(USBSerialNumber)
    VARTRACE(diPOPASCDData)
    VARTRACE(diPOPAGCDData)
    VARTRACE(diPOPAACDData)
    VARTRACE(userContext)
    LOCK

    tResult retVal = -1;
    TimeTrace ticks(__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        if( 0 == deviceID )
        {
            // Fix for NCG3D-196834: Metadata not updated once device is unplugged during active CPW session
            // Prioritize iAP Wifi over iAP USB session when device is connected over both USB and Wifi.
            deviceID = GetAppleDeviceIDUsingUSBSerialNumber(USBSerialNumber);
        }

        if(DEVICE_ID_NOT_SET != deviceID)
        {
            retVal = LocalSPM::GetIPODControl().SendDiPOTransferDRData(IN deviceID, IN diPOPASCDData, IN diPOPAGCDData, IN diPOPAACDData, IN userContext);
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();

    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::DiPORequestAudioDevice(const tDeviceID deviceID)
{
    ENTRY;
    VARTRACE(deviceID)
    LOCK

    tResult retVal = -1;
    TimeTrace ticks(__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetPlayerManager().SendGetAudioDevice(IN deviceID);
    } else {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();

    VARTRACE(retVal)
    return retVal;

}

tResult MediaPlayerInterface::DiPOReleaseAudioDevice(const tDeviceID deviceID)
{
    ENTRY;
    VARTRACE(deviceID)
    LOCK

    tResult retVal = -1;
    TimeTrace ticks(__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState()) {
        // do nothing
    } else {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();

    VARTRACE(retVal)
    return retVal;

}

tResult MediaPlayerInterface::DiPORequestPlaybackAction(
        const tDeviceID deviceID,
        const tPlaybackAction playbackAction,
        const tNextPrevSkipCount nextPrevSkipCount)
{
    ENTRY
    VARTRACE(deviceID);
    VARTRACE(playbackAction);
    VARTRACE(nextPrevSkipCount);
    LOCK

    tResult retVal = -1;
    TimeTrace ticks(__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState()) {
        tNextPrevSkipCount steps = nextPrevSkipCount;
        if(0 == steps) {
            steps = 1;
        }
        retVal = LocalSPM::GetIPODControl().SendPlaybackAction(IN deviceID, IN playbackAction, IN steps);
    } else {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::DiPOSendInitiateCall(const tMountPoint USBSerialNumber, const tDiPOInitiateCall initiateCall)
{
    ENTRY;
    VARTRACE(USBSerialNumber);
    VARTRACE(initiateCall);
    LOCK

    tResult retVal = -1;
    TimeTrace ticks(__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState()) {

        tDiPOCallControlsConfiguration callControlsConfig;
        callControlsConfig = LocalSPM::GetDataProvider().GetDiPOCallControlsConfiguration();
        if(callControlsConfig.bits.isInitiateCallAvailable)
        {
            tDeviceID deviceID = GetAppleDeviceIDUsingUSBSerialNumber(USBSerialNumber);

            if(DEVICE_ID_NOT_SET != deviceID)
            {
                retVal = LocalSPM::GetIPODControl().SendDiPOInitiateCall(IN deviceID, initiateCall);
            }
            else
            {
                ETG_TRACE_ERR(("DeviceID not Found for USBSerialNumber: %s", USBSerialNumber));
            }
        }
        else
        {
            ETG_TRACE_USR3(("DiPOInitiateCall: feature not enabled, CallControl config = %d",callControlsConfig.value));
        }
    } else {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::DiPOSendAcceptCall(const tMountPoint USBSerialNumber, const tDiPOAcceptCall acceptCall)
{
    ENTRY;
    VARTRACE(USBSerialNumber);
    LOCK

    tResult retVal = -1;
    TimeTrace ticks(__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState()) {

        tDiPOCallControlsConfiguration callControlsConfig;
        callControlsConfig = LocalSPM::GetDataProvider().GetDiPOCallControlsConfiguration();
        if(callControlsConfig.bits.isAcceptCallAvailable)
        {
            tDeviceID deviceID = GetAppleDeviceIDUsingUSBSerialNumber(USBSerialNumber);

            if(DEVICE_ID_NOT_SET != deviceID)
            {
                retVal = LocalSPM::GetIPODControl().SendDiPOAcceptCall(IN deviceID,acceptCall);
            }
            else
            {
                ETG_TRACE_ERR(("DeviceID not Found for USBSerialNumber: %s", USBSerialNumber));
            }
        }
        else
        {
            ETG_TRACE_USR3(("DiPOAcceptCall: feature not enabled, CallControl config = %d",callControlsConfig.value));
        }
    } else {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::DiPOSendEndCall(const tMountPoint USBSerialNumber, const tDiPOEndCall endCall)
{
    ENTRY;
    VARTRACE(USBSerialNumber);
    LOCK

    tResult retVal = -1;
    TimeTrace ticks(__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState()) {

        tDiPOCallControlsConfiguration callControlsConfig;
        callControlsConfig = LocalSPM::GetDataProvider().GetDiPOCallControlsConfiguration();
        if(callControlsConfig.bits.isEndCallAvailable)
        {
            tDeviceID deviceID = GetAppleDeviceIDUsingUSBSerialNumber(USBSerialNumber);

            if(DEVICE_ID_NOT_SET != deviceID)
            {
                retVal = LocalSPM::GetIPODControl().SendDiPOEndCall(IN deviceID, endCall);
            }
            else
            {
                ETG_TRACE_ERR(("DeviceID not Found for USBSerialNumber: %s", USBSerialNumber));
            }
        }
        else
        {
            ETG_TRACE_USR3(("DiPOEndCall: feature not enabled, CallControl config = %d",callControlsConfig.value));
        }

    } else {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::DiPOSendSwapCalls(const tMountPoint USBSerialNumber)
{
    ENTRY;
    VARTRACE(USBSerialNumber);
    LOCK

    tResult retVal = -1;
    TimeTrace ticks(__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState()) {

        tDeviceID deviceID = GetAppleDeviceIDUsingUSBSerialNumber(USBSerialNumber);

        if(DEVICE_ID_NOT_SET != deviceID)
        {
            retVal = LocalSPM::GetIPODControl().SendSwapCalls(IN deviceID);
        }
        else
        {
            ETG_TRACE_ERR(("DeviceID not Found for USBSerialNumber: %s", USBSerialNumber));
        }
    } else {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::SendSwapCalls(const tDeviceID deviceID)
{
    ENTRY;
    VARTRACE(deviceID);
    LOCK

    tResult retVal = -1;
    TimeTrace ticks(__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState()) {
        retVal = LocalSPM::GetIPODControl().SendSwapCalls(IN deviceID);
    } else {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::StopMediaPlaybackResult(const tDeviceID deviceID)
{
    ENTRY;
    VARTRACE(deviceID)
    tResult ret = MP_NO_ERROR;
    /* Get device type and mount point from DataProvider */
    //tDeviceInfo deviceInfo;
    //ret = LocalSPM::GetDBManager().GetDeviceInfo(OUT deviceInfo, IN deviceID);

    // The following if-then-else statement is commented out to eliminate Lint error 774
    // cause by value of ret always = MP_NO_ERROR
    /*
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while getting device info from DataProvider (ErrorCode:%s)", errorString(ret)));
    }
    else
    {
     */
    /* Send STOP_STREAMING_ANSWER message to specific DeviceControlSM */
    char msgToSendString[64];
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    strncpy_r(OUT msgToSendString, IN "STOP_STREAMING_ANSWER", IN sizeof(msgToSendString));

    ret = LocalSPM::GetIPODControl().ParameterSTOP_STREAMING_ANSWER(OUT parameterString,
            IN size,
            IN deviceID);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
        parameterString[0] = '\0';
    }

    ret = LocalSPM::GetDeviceDispatcher().RouteMessage(IN /*deviceInfo.deviceType*/ DTY_IPHONE, IN msgToSendString, IN parameterString);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while sending message via SMF (ErrorCode:%s)", errorString(ret)));
    }
    //}   The if-then-else statement is commented out

    VARTRACE(ret)
    return ret;
}

tResult MediaPlayerInterface::RequestAVActivationResult(const tSuccess success)
{
    ENTRY
    VARTRACE(success)
    LOCK

    tResult retVal= -1;
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        tDeviceID deviceID = DEVICE_ID_NOT_SET; //Not used yet
        retVal = LocalSPM::GetPlayerManager().SendGetAudioDeviceAnswer(IN deviceID, IN success);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    VARTRACE(retVal)
    return retVal;
}


tBoolean MediaPlayerInterface::MyMediaSupported()
{
    ENTRY
    return (tBoolean) LocalSPM::GetDataProvider().MyMediaVisible();
}

tResult MediaPlayerInterface::SetVehicleBTAddress(const tBTMACAddress address)
{
    ENTRY
    VARTRACE(address)
    tResult retVal= -1;
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetDBManager().StoreBTMACAddress(IN address);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    return retVal;
}

tResult MediaPlayerInterface::ConvertCodeSetFromSysLang2UTF8(vector<tMediaObject> &mediaObjectVector)
{
    ENTRY
    tResult retval = MP_NO_ERROR;

    string fromCodeset = "";


    //Finds the most commonly used character encoding for current system language
    retval = FindCharEncodingForSysLanguage(OUT fromCodeset);

    for(std::vector<tMediaObject>::iterator iter = mediaObjectVector.begin(); iter != mediaObjectVector.end(); iter++)
    {
        //Iterate through the MediaObject vector and Find is there any mediaObject with non-zero metadata convertFlag
        if((*iter).metadataConvertFlag)
        {
            //Reconstruct the metadata to how they were given by taglib from metadataHeader
            retval = ReconstructLatin1Metadata(INOUT *iter);

            /*Covert the Non-UTF8 Strings to UTF8 considering they are the current system Language character Encoding.
            Reason:Atleast the non-UTF8 strings in the current system Language character Encoding be displayed properly in HMI*/
            retval = Convert2UTF8FromGivenCodeSet(INOUT *iter,IN fromCodeset);
        }
    }
    return retval;
}

void MediaPlayerInterface::ConvertDevicenameCodeSetFromSysLang2UTF8(FastUTF8::tString str, const size_t size)
{
    ENTRY
    tResult retval = MP_NO_ERROR;

    /* sanities */
    if (!str) return;
    if (*str == 0) return;

    /* check if str is non utf-8 */
    if (g_utf8_validate((gchar *)str , -1, NULL) != true) {
        ETG_TRACE_USR2(("Device name is not valid UTF-8"));
        string Codeset = "";
        //Finds the most commonly used character encoding for current system language
        retval = FindCharEncodingForSysLanguage(OUT Codeset);
        const gchar* fromCodeset = (const gchar*)Codeset.c_str();
        ETG_TRACE_USR2(("Not valid UTF-8, convert from code set: %s", fromCodeset));

        gchar *newUTF8String;

        /* convert it as language adjusted and store it as UTF-8 */
        newUTF8String = g_convert((gchar *)str, -1, "UTF-8", fromCodeset, NULL, NULL, NULL);
        //ETG_TRACE_USR2(("g_convert conversion  %d", errno));
        if (newUTF8String) {
            ETG_TRACE_USR2(("conversion success: from: %s", str));
            ETG_TRACE_USR2(("conversion success: to  : %s", newUTF8String));
            strncpy_r(OUT (char *)str, IN newUTF8String, IN size);
            g_free(newUTF8String);
        }
    }
    else {
        ETG_TRACE_USR2(("Valid UTF-8. No conversion done"));
    }
}
tResult MediaPlayerInterface::FindCharEncodingForSysLanguage(string &codeset)
{
    tResult ret = MP_NO_ERROR;

#if CODESET_CONVERSION_TRACES_NEEDED
    FILE* codesetConvLogFile = NULL;
    codesetConvLogFile = fopen(IN LocalSPM::GetDataProvider().CodeSetConvLogFilePath().c_str(), "a");
#endif

    /* Language and its assumed coding is taken from the attached doc CMG3G-6919 */
    tLanguageType language;
    LocalSPM::GetDBManager().GetLanguage(OUT language);

    switch (language)
    {
        case LNG_NA_ENGLISH:
        case LNG_UK_ENGLISH :
        case LNG_GERMAN:
        case LNG_ITALIAN :
        case LNG_SWEDISH :
        case LNG_SPANISH :
        case LNG_NA_SPANISH:
        case LNG_DUTCH :
        case LNG_PORTUGUESE :
        case LNG_NORWEGIAN :
        case LNG_DANISH:
        case LNG_BRAZILIAN_PORTUGUESE:
        case LNG_NA_FRENCH:/*LNG_NA_FRENCH is considered as Canadian French*/
            codeset = (const gchar *)"ISO-8859-1";
            break;
        case LNG_FRENCH  :
        case LNG_FINNISH :
            codeset = (const gchar *)"ISO-8859-15";
            break;
        case LNG_GREEK:
            codeset = (const gchar *)"ISO-8859-7";
            break;
        case LNG_JAPANESE:
            codeset = (const gchar *)"CP932";
            break;
        case LNG_ARABIC:
            codeset = (const gchar *)"ISO-8859-6";
            break;
        case LNG_STANDARD_CHINESE: /* LNG_CANTONESE and LNG_STANDARD_CHINESE is considered as Simplified Chinese */
        case LNG_CANTONESE:
            codeset = (const gchar *)"GB18030";
            break;
        case LNG_TURKISH:
            codeset = (const gchar *)"ISO-8859-9";
            break;
        case LNG_KOREAN:
            codeset = (const gchar *)"CP949";
            break;
        case LNG_TRADITIONAL_CHINESE:
            codeset = (const gchar *)"CP950";
            break;
        case LNG_HUNGARIAN:
        case LNG_CZECH :
        case LNG_SLOVAK :
        case LNG_ROMANIAN:
        case LNG_SLOVENIAN:
        case LNG_CROATIAN:
        case LNG_POLISH:
            codeset = (const gchar *)"ISO-8859-2";
            break;
        case LNG_RUSSIAN:
        case LNG_BULGARIAN:
        case LNG_UKRAINIAN:
            codeset = (const gchar *)"ISO-8859-5" ;
            break;
        case LNG_THAI:
            codeset = (const gchar *)"ISO-8859-11" ;
            break;
        default:
            codeset = (const gchar *)"ISO-8859-1" ;
            break;
    }
#if CODESET_CONVERSION_TRACES_NEEDED
    if(codesetConvLogFile)
    {
        fprintf(codesetConvLogFile, "Language:%d Codeset:%s", language, codeset.c_str());
        fclose(codesetConvLogFile);
    }
#endif

    return ret;
}

tResult MediaPlayerInterface::ReconstructLatin1Metadata(tMediaObject &mediaObject)
{
    ENTRY;

    tResult ret = MP_NO_ERROR;

#if CODESET_CONVERSION_TRACES_NEEDED
    FILE* codesetConvLogFile = NULL;
    codesetConvLogFile = fopen(IN LocalSPM::GetDataProvider().CodeSetConvLogFilePath().c_str(), "a");

    if(codesetConvLogFile)
    {
        fprintf(codesetConvLogFile,"Inside:ReconstructLatin1Metadata\n");
        fprintf(codesetConvLogFile,"mediaObject.metadataConvertFlag:%d\n", mediaObject.metadataConvertFlag);
        HexDumpMediaObjectMetadataFields(mediaObject,codesetConvLogFile);
    }
#endif
    ETG_TRACE_USR4(("Before Reconstuction :mediaObject.metadataConvertFlag:%d", mediaObject.metadataConvertFlag));
    VARTRACE(mediaObject.MetadataField1);
    VARTRACE(mediaObject.MetadataField2);
    VARTRACE(mediaObject.MetadataField3);
    VARTRACE(mediaObject.MetadataField4);
    VARTRACE(mediaObject.title);

    /*
    1.  Check the bitfield of metadataConvertFlag.
    bit0- MetadataField1
    bit1- MetadataField2
    bit2- MetadataField3
    bit3- MetadataField4
    bit4- Title

    2.  If set,then that metadata is reconstructed to latin1 string
    i.e the metadata given by tagInfo by reading the metadata header.
     */

    if(mediaObject.metadataConvertFlag & BITFLAG_METADATAFIELD1)
    {
        ret = ReconstructTheGivenMetadata(mediaObject.MetadataField1);
        if(false == ret){mediaObject.metadataConvertFlag &= 0XFE;}
    }

    if(mediaObject.metadataConvertFlag & BITFLAG_METADATAFIELD2)
    {
        ret = ReconstructTheGivenMetadata(mediaObject.MetadataField2);
        if(false == ret){mediaObject.metadataConvertFlag &= 0XFD;}
    }

    if(mediaObject.metadataConvertFlag & BITFLAG_METADATAFIELD3)
    {
        ret = ReconstructTheGivenMetadata(mediaObject.MetadataField3);
        if(false == ret){mediaObject.metadataConvertFlag &= 0XFB;}
    }

    if(mediaObject.metadataConvertFlag & BITFLAG_METADATAFIELD4)
    {
        ret = ReconstructTheGivenMetadata(mediaObject.MetadataField4);
        if(false ==ret){mediaObject.metadataConvertFlag &= 0XF7;}
    }

    if(mediaObject.metadataConvertFlag & BITFLAG_TITLE)
    {
        ret = ReconstructTheGivenMetadata(mediaObject.title);
        if(false == ret){mediaObject.metadataConvertFlag &= 0XEF;}
    }
#if CODESET_CONVERSION_TRACES_NEEDED
    if(codesetConvLogFile)
    {
        fprintf(codesetConvLogFile,"Final MetadataFields\n");
        fprintf(codesetConvLogFile,"mediaObject.metadataConvertFlag:%d\n", mediaObject.metadataConvertFlag);
        HexDumpMediaObjectMetadataFields(mediaObject,codesetConvLogFile);
        fprintf(codesetConvLogFile,"Leaving:ReconstructLatin1Metadata\n");

        fclose(codesetConvLogFile);
    }
#endif

    ETG_TRACE_USR4(("After Reconstruction:mediaObject.metadataConvertFlag:%d", mediaObject.metadataConvertFlag));
    VARTRACE(mediaObject.MetadataField1);
    VARTRACE(mediaObject.MetadataField2);
    VARTRACE(mediaObject.MetadataField3);
    VARTRACE(mediaObject.MetadataField4);
    VARTRACE(mediaObject.title);
    return ret;
}

tResult MediaPlayerInterface::Convert2UTF8FromGivenCodeSet(tMediaObject &mediaObject,string &codeset)
{
    ENTRY;

    tResult retval = MP_NO_ERROR;

    tBoolean lang2UTF8ConvertedMetadata1 = false;
    tBoolean lang2UTF8ConvertedMetadata2 = false;
    tBoolean lang2UTF8ConvertedMetadata3 = false;
    tBoolean lang2UTF8ConvertedMetadata4 = false;
    tBoolean lang2UTF8ConvertedTitle = false;

#if CODESET_CONVERSION_TRACES_NEEDED
    FILE* codesetConvLogFile = NULL;
    codesetConvLogFile = fopen(IN LocalSPM::GetDataProvider().CodeSetConvLogFilePath().c_str(), "a");
#endif

    const gchar* fromCodeset = (const gchar*)codeset.c_str();

    if(mediaObject.metadataConvertFlag & BITFLAG_METADATAFIELD1)
    {
        lang2UTF8ConvertedMetadata1 = ConvertTheGivenMetadata2UTF8(mediaObject.MetadataField1,fromCodeset);
    }

    if(mediaObject.metadataConvertFlag & BITFLAG_METADATAFIELD2)
    {
        lang2UTF8ConvertedMetadata2 = ConvertTheGivenMetadata2UTF8(mediaObject.MetadataField2,fromCodeset);
    }

    if(mediaObject.metadataConvertFlag & BITFLAG_METADATAFIELD3)
    {
        lang2UTF8ConvertedMetadata3 = ConvertTheGivenMetadata2UTF8(mediaObject.MetadataField3,fromCodeset);
    }

    if(mediaObject.metadataConvertFlag & BITFLAG_METADATAFIELD4)
    {
        lang2UTF8ConvertedMetadata4 = ConvertTheGivenMetadata2UTF8(mediaObject.MetadataField4,fromCodeset);
    }

    if(mediaObject.metadataConvertFlag & BITFLAG_TITLE)
    {
        lang2UTF8ConvertedTitle = ConvertTheGivenMetadata2UTF8(mediaObject.title,fromCodeset);
    }
#if CODESET_CONVERSION_TRACES_NEEDED
    if(codesetConvLogFile)
    {
        fprintf(codesetConvLogFile,"Inside:Convert2UTF8FromGivenCodeSet\n");

        fprintf(codesetConvLogFile,"From CodeSet :%s",fromCodeset);
        fprintf(codesetConvLogFile,"mediaObject.metadataConvertFlag:%d\n", mediaObject.metadataConvertFlag);

        fprintf(codesetConvLogFile,"Is g_convert of MetadataField1 success? :%d",lang2UTF8ConvertedMetadata1);
        fprintf(codesetConvLogFile,"Is g_convert of MetadataField2 success? :%d",lang2UTF8ConvertedMetadata2);
        fprintf(codesetConvLogFile,"Is g_convert of MetadataField3 success? :%d",lang2UTF8ConvertedMetadata3);
        fprintf(codesetConvLogFile,"Is g_convert of MetadataField4 success? :%d",lang2UTF8ConvertedMetadata4);
        fprintf(codesetConvLogFile,"Is g_convert of Title success? :%d",lang2UTF8ConvertedTitle);

        fprintf(codesetConvLogFile,"MetadataFields To be send to HMI");
        HexDumpMediaObjectMetadataFields(mediaObject,codesetConvLogFile);

        fprintf(codesetConvLogFile,"Leaving:Convert2UTF8FromGivenCodeSet\n");
        fclose(codesetConvLogFile);

    }
#else
    (void)lang2UTF8ConvertedMetadata1;
    (void)lang2UTF8ConvertedMetadata2;
    (void)lang2UTF8ConvertedMetadata3;
    (void)lang2UTF8ConvertedMetadata4;
    (void)lang2UTF8ConvertedTitle;
#endif
    //ETG_TRACING
    VARTRACE(mediaObject.MetadataField1);
    VARTRACE(mediaObject.MetadataField2);
    VARTRACE(mediaObject.MetadataField3);
    VARTRACE(mediaObject.MetadataField4);
    VARTRACE(mediaObject.title);

    return retval;
}
tBoolean MediaPlayerInterface::ReconstructTheGivenMetadata(tMetadata &MetadataField)
{
    tBoolean ret = false;
    gchar* res = NULL;

    res = g_convert((gchar*)MetadataField,strlen_r((const char*)MetadataField),"ISO-8859-1","UTF-8",NULL,NULL,NULL);
    if(res)
    {
        strncpy_r(MetadataField,(const char*)res,sizeof(MetadataField));
        g_free(res);
        ret = true;
    }
    return ret;
}

tBoolean MediaPlayerInterface::ConvertTheGivenMetadata2UTF8(tMetadata &MetadataField,const gchar* &fromCodeset)
{
    tBoolean ret = false;

    /*
        Now, g_convert is invoked to convert metadata from the most commonly used codeset of the current system Language
        to UTF-8 codeset
     */
    gchar* lang2UTF8ConvertedMetadata = g_convert((gchar *)MetadataField,strlen_r((const char*)MetadataField), "UTF-8", fromCodeset, NULL, NULL, NULL);

    if(lang2UTF8ConvertedMetadata)
    {
        strncpy_r(MetadataField,(const char*)lang2UTF8ConvertedMetadata,sizeof(MetadataField));
        g_free(lang2UTF8ConvertedMetadata);
        ret = true;
    }
    else
    {
        string latinCodesetString = "ISO-8859-1";

        if(latinCodesetString == string((const char*)fromCodeset))
        {
            ETG_TRACE_FATAL(("Conversion from Latin1 codeset to UTF-8 failed.Non-UTF8 string is given to HMI!"));
        }
        else
        {
            ETG_TRACE_ERR(("Conversion from Syslanguage codeset to UTF-8 failed.Converting from Latin1 to UTF-8"));

            const gchar* latinCodeset = (const gchar*)latinCodesetString.c_str();
            ret = ConvertTheGivenMetadata2UTF8(MetadataField,latinCodeset);
        }
    }

    return ret;
}
void MediaPlayerInterface::HexDumpMediaObjectMetadataFields(tMediaObject &mediaObject,FILE* codesetConvLogFile)
{

    if(codesetConvLogFile)
    {
        hexDump(codesetConvLogFile,const_cast<char*>("mediaObject.MetadataField1"), (void *)mediaObject.MetadataField1,strlen_r(mediaObject.MetadataField1));
        hexDump(codesetConvLogFile,const_cast<char*>("mediaObject.MetadataField2"), (void *)mediaObject.MetadataField2,strlen_r(mediaObject.MetadataField2));
        hexDump(codesetConvLogFile,const_cast<char*>("mediaObject.MetadataField3"), (void *)mediaObject.MetadataField3,strlen_r(mediaObject.MetadataField3));
        hexDump(codesetConvLogFile,const_cast<char*>("mediaObject.MetadataField4"), (void *)mediaObject.MetadataField4,strlen_r(mediaObject.MetadataField4));
        hexDump(codesetConvLogFile,const_cast<char*>("mediaObject.title"), (void *)mediaObject.title,strlen_r(mediaObject.title));
    }
}

tResult MediaPlayerInterface::OpticalDiscStatusChanged(const tDMOpticalDiscSlotState opticalDiscSlotState) //Roadmap 15002_CDAudio_DetectionAndDiagnosis
{
    ENTRY
    tResult retval = MP_NO_ERROR;
    if(mOpticalDiscSlotState != opticalDiscSlotState)
    {
        mOpticalDiscSlotState = opticalDiscSlotState; //store this value for the getter GetOpticalDiscStatus to be accessable for the service
        LocalSPM::GetOutputWrapper().UpdateOpticalDiscSlotState();

        /*NCG3D-2815:When Auto reinsertion of a disc occurs,disc shouldnt become the Active mediaSource in HMI.
         * To acheive this,FormerConnectionState is set to CS_AUTO_REINSERTED
         */
        if(DM_OPTDISC_INSERTED_AUTOMATIC_CDAUDIO == mOpticalDiscSlotState)
        {
            retval = LocalSPM::GetDBManager().UpdateFormerConnectionState(IN DTY_CDDA,IN CS_AUTO_REINSERTED);
        }
        else if(DM_OPTDISC_INSERTED_AUTOMATIC_CDROM == mOpticalDiscSlotState)
        {
            retval = LocalSPM::GetDBManager().UpdateFormerConnectionState(IN DTY_CDROM,IN CS_AUTO_REINSERTED);
        }
        if(DM_OPTDISC_EJECTING == mOpticalDiscSlotState)
        {
            LocalSPM::GetCDDAControl().cddaEjected();
        }
    }

    VARTRACE(retval);
    return retval;
}

tResult MediaPlayerInterface::GetOpticalDiscSlotStatus(OUT tDMOpticalDiscSlotState &opticalDiscSlotState) //Roadmap 15002_CDAudio_DetectionAndDiagnosis
{
    ENTRY;
    tResult retval = MP_NO_ERROR;

    opticalDiscSlotState = mOpticalDiscSlotState;

    ETG_TRACE_USR4(("GetOpticalDiscStatus: opticalDiscEjectCmd: %d retval:0x%x",ETG_CENUM(tDMOpticalDiscSlotState,opticalDiscSlotState),retval));
    return retval;
}

tResult MediaPlayerInterface::SendEjectOpticalDisc(tDMOpticalDiscEjectCmd enOpticalDiscEjectCmd) //Roadmap 15002_CDAudio_DetectionAndDiagnosis
{
    ENTRY
    tResult retval = MP_NO_ERROR;
    LocalSPM::GetOutputWrapper().SendEjectOpticalDisc(enOpticalDiscEjectCmd);

    ETG_TRACE_USR4(("SendEjectOpticalDisc: opticalDiscEjectCmd: %d retval:0x%x",ETG_CENUM(tDMOpticalDiscEjectCmd,enOpticalDiscEjectCmd),retval));
    return retval;
}

tResult MediaPlayerInterface::SendOpticalDiscEjectMethodResult(tDMOpticalDiscEjectMethodResult entOpticalDiscEjectMethodResult)
{
    ENTRY
    tResult retval = MP_NO_ERROR;
    LocalSPM::GetOutputWrapper().SendOpticalDiscEjectMethodResult(entOpticalDiscEjectMethodResult);

    ETG_TRACE_USR4(("SendOpticalDiscEjectMethodResult: entOpticalDiscEjectMethodResult: %d retval:0x%x",ETG_CENUM(tDMOpticalDiscEjectMethodResult,entOpticalDiscEjectMethodResult),retval));
    return retval;
}

bool MediaPlayerInterface::IsDVDDrive()
{
    ENTRY
    tResult retval = MP_NO_ERROR;

    retval = LocalSPM::GetDataProvider().UseDVDControl();

    ETG_TRACE_USR4(("IsDVDDrive: retval : %d",retval));
    return retval;
}

tResult MediaPlayerInterface::SendScanMode(const tScanMode scanMode)
{
    ENTRY
    VARTRACE(scanMode)
    LOCK

    tResult retVal = MP_NO_ERROR;
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        if(LocalSPM::GetDataProvider().TrackScanningSupported())
        {
            retVal = LocalSPM::GetPlayerManager().SendScanMode(IN scanMode);
        }
        else
        {
            ETG_TRACE_USR3(("MediaPlayer Doesnt support track Scanning"));
            retVal = -1;
        }
    }

    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
        retVal = -1;
    }
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::GetScanMode(tScanMode &scanMode)
{
    ENTRY
    //LOCK

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
        {
            retVal = LocalSPM::GetPlayerManager().GetScanMode(OUT scanMode);
        }
        else
        {   // set default value
            scanMode = false;
            retVal = MP_NO_ERROR;
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(scanMode)
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::CreateMediaPlayerIndexedImageFolderList(
        tListID &listID,
        tListSize &listSize,
        const tListType listType,
        const tDeviceID deviceID)
{
    ENTRY
    VARTRACE(listType)
    VARTRACE(deviceID)
    tFilterTag1 tag1;
    tFilterTag2 tag2;
    tFilterTag3 tag3;
    tFilterTag4 tag4;
    LOCK
    memset(&listID,0, sizeof listID);
    memset(&listSize,0, sizeof listSize);
    memset(&tag1,0, sizeof tag1);
    memset(&tag2,0, sizeof tag2);
    memset(&tag3,0, sizeof tag3);
    memset(&tag4,0, sizeof tag4);
    tResult retVal= -1;
    tStreaming streaming = false;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetListControl().CreateMediaPlayerIndexedList(OUT listID, OUT listSize, IN listType, IN streaming, IN tag1, IN tag2, IN tag3, IN tag4, IN deviceID);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(listID)
    VARTRACE(listSize)
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::CreateMediaPlayerIndexedImageFolderItemList(
        tListID &listId,
        tListSize &listSize,
        const tPath path,
        const tDeviceID deviceID)
{
    ENTRY
    VARTRACE(path)
    VARTRACE(deviceID)
    tNumberOfMediaObjects numberOfMediaObjects;
    tNumberOfPlayLists numberOfPlayLists;
    tNumberOfFolders numberOfFolders;
    tFileTypeSelection filetypeSelection = FTS_IMAGE;
    LOCK
    memset(&listId, 0, sizeof listId);
    memset(&listSize, 0, sizeof listSize);
    memset(&numberOfMediaObjects, 0, sizeof numberOfMediaObjects);
    memset(&numberOfPlayLists, 0, sizeof numberOfPlayLists);
    memset(&numberOfFolders, 0, sizeof numberOfFolders);
    tPath localPath;
    if(NULL == FastUTF8::StartsWith((const FastUTF8::tString)path, (const FastUTF8::tString)PATH_DELIMITER)) //lint !e1773
    {
        ETG_TRACE_ERR(("%s is called with a path without leading /",__PRETTY_FUNCTION__));
        strncpy_r(OUT localPath, IN PATH_DELIMITER, IN sizeof(localPath));
        strncat_r(OUT localPath, IN path, IN sizeof(localPath));
    }
    else
    {
        strncpy_r(OUT localPath, IN path, IN sizeof(localPath));
    }
    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetListControl().CreateMediaPlayerFileList(
                OUT listId,
                OUT listSize,
                OUT numberOfMediaObjects,
                OUT numberOfPlayLists,
                OUT numberOfFolders,
                IN LTY_IMAGE_FOLDER_ITEM,
                IN localPath,
                IN deviceID,
                IN false, //streaming
                IN filetypeSelection);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(listId)
    VARTRACE(listSize)
    VARTRACE(retVal)
    return retVal;
}

#if USE_IMAGE_OBJECT
tResult MediaPlayerInterface::RequestMediaPlayerIndexedImageFolderListSlice(
        vector<tImageObject> &ImageObjectVector,
        const tListID listID,
        const tIndex startIndex,
        const tIndex sliceSize)
{
    ENTRY
    VARTRACE(listID)
    VARTRACE(startIndex)
    VARTRACE(sliceSize)
    LOCK
    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        vector<tMediaObject> mediaObjectVector;
        tImageObject imageObject;
        memset(&imageObject, 0, sizeof imageObject);
        tNumberOfMediaObjects numberOfImages = 0;
        tListInfo listInfo;
        InitListInfo(listInfo);
        retVal = LocalSPM::GetListControl().RequestMediaPlayerIndexedListSlice(OUT mediaObjectVector, IN listID, IN startIndex, IN sliceSize);
        if (MP_NO_ERROR == retVal)
        {
            retVal = LocalSPM::GetListControl().GetListInfo(listInfo, listID, FALSE);
            if (MP_NO_ERROR == retVal)
            {
                for(unsigned int i=0;i<(mediaObjectVector.size());i++)
                {
                    VARTRACE(mediaObjectVector[i]);
                    if (LTY_IMAGE_FOLDER == listInfo.listType)
                    {
                        imageObject.type = FT_FOLDER;
                        LocalSPM::GetDBManager().GetNumberofImagesByPath(OUT numberOfImages, IN mediaObjectVector[i].deviceID, IN mediaObjectVector[i].path);
                        imageObject.numberofImages = numberOfImages;
                    }
                    else
                    {
                        imageObject.type = FT_IMAGE;
                        imageObject.numberofImages = 0;
                    }
                    if(strncmp(mediaObjectVector[i].path,PATH_DELIMITER,strlen_r(mediaObjectVector[i].path)))
                    {
                        tUInt len=strlen_r(mediaObjectVector[i].path);
                        if(len)
                        {
                            tPath folderTitle;
                            strncpy_r(OUT folderTitle, IN  mediaObjectVector[i].path, IN sizeof(folderTitle));
                            folderTitle[len-1]='\0';
                            string path(folderTitle);
                            tS32 found = path.find_last_of(PATH_DELIMITER);

                            if ( INVALID_PATH != found)
                            {
                                VARTRACE(found);
                                strncpy_r(OUT imageObject.folderTitle, IN ((path.substr(found+1)).c_str()), IN sizeof(imageObject.folderTitle));
                            }
                            else
                            {
                                ETG_TRACE_ERR(("Path without / is invalid : %s",mediaObjectVector[i].path));
                            }
                        }
                        else
                        {
                            ETG_TRACE_ERR(("%s is not a valid path",mediaObjectVector[i].path));
                        }
                    }
                    else
                    {
                        strncpy_r(OUT imageObject.folderTitle, IN mediaObjectVector[i].path, IN sizeof(imageObject.folderTitle));
                    }
#if IMAGE_ATTRIBUTES_IN_METADATA
                    strncpy_r(OUT imageObject.imageSize, IN  mediaObjectVector[i].MetadataField1, IN sizeof(imageObject.imageSize));
                    strncpy_r(OUT imageObject.dateTime, IN  mediaObjectVector[i].MetadataField2, IN sizeof(imageObject.dateTime));
                    strncpy_r(OUT imageObject.dimension, IN  mediaObjectVector[i].MetadataField3, IN sizeof(imageObject.dimension));
                    strncpy_r(OUT imageObject.resolution, IN  mediaObjectVector[i].MetadataField4, IN sizeof(imageObject.resolution));
#else
                    imageObject.fileSize = mediaObjectVector[i].fileSize;
                    strncpy_r(OUT imageObject.dateTime, IN  mediaObjectVector[i].dateTime, IN sizeof(imageObject.dateTime));
                    strncpy_r(OUT imageObject.dimension, IN  mediaObjectVector[i].MetadataField2, IN sizeof(imageObject.dimension));
                    strncpy_r(OUT imageObject.resolution, IN  mediaObjectVector[i].MetadataField1, IN sizeof(imageObject.resolution));
#endif
                    imageObject.fileFormat = mediaObjectVector[i].fileFormat;
                    imageObject.objectID = mediaObjectVector[i].objectID;
                    strncpy_r(OUT imageObject.imageTitle, IN  mediaObjectVector[i].title, IN sizeof(imageObject.imageTitle));
                    strncpy_r(OUT imageObject.path, IN  mediaObjectVector[i].path, IN sizeof(imageObject.path));
                    strncpy_r(OUT imageObject.imageFile, IN  mediaObjectVector[i].fileName, IN sizeof(imageObject.imageFile));
                    imageObject.deviceID = mediaObjectVector[i].deviceID;
                    strncpy_r(OUT imageObject.mountPoint, IN  mediaObjectVector[i].mountPoint, IN sizeof(imageObject.mountPoint));
                    ImageObjectVector.push_back(imageObject);
                }
            }
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(ImageObjectVector)
    VARTRACE(retVal)
    return retVal;
}
#endif

tResult MediaPlayerInterface::DeviceManagerNotifyCDInfo(IN trOpticalDiskCDInfo OpticalDiskCDInfo) //RNAIVI- CD Info
{
    ENTRY
    tResult retval = MP_NO_ERROR;
    strncpy_r(mOpticalDriveVersion,OpticalDiskCDInfo.OpticalDriveVersion, sizeof mOpticalDriveVersion);
    mOpticalDeviceState = OpticalDiskCDInfo.OpticalDeviceState;
    LocalSPM::GetOutputWrapper().UpdateCDInfo();
    ETG_TRACE_USR4(("DeviceManagerNotifyCDInfo: OpticalDriveVersion: %s",mOpticalDriveVersion));
    ETG_TRACE_USR4(("DeviceManagerNotifyCDInfo: OpticalDeviceState: %d",mOpticalDeviceState));
    return retval;
}

tResult MediaPlayerInterface::GetOpticalDiscCDInfo(OUT trOpticalDiskCDInfo &OpticalDiskCDInfo) //RNAIVI- CD Info
{
    ENTRY;
    tResult retval = MP_NO_ERROR;

    strncpy_r(OpticalDiskCDInfo.OpticalDriveVersion, mOpticalDriveVersion, sizeof OpticalDiskCDInfo.OpticalDriveVersion);
    OpticalDiskCDInfo.OpticalDeviceState = mOpticalDeviceState;

    ETG_TRACE_USR4(("GetOpticalDiscCDInfo: OpticalDriveVersion: %s",OpticalDiskCDInfo.OpticalDriveVersion));
    ETG_TRACE_USR4(("GetOpticalDiscCDInfo: OpticalDeviceState: %d",OpticalDiskCDInfo.OpticalDeviceState));
    return retval;
}

tResult MediaPlayerInterface::UpdateVRSessionStatus(IN tVRSessionStatus vrstatus,IN tSiriStatus siristatus,IN tVRSupportStatus vrsupport)
{
    ENTRY
    VARTRACE(vrstatus)
    VARTRACE(vrsupport)
    VARTRACE(siristatus)

    tResult retval = MP_NO_ERROR;

    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    retval = LocalSPM::GetBTControl().ParameterHFP_VOICE_RECOGNITION_SESSION_STATUS(OUT parameterString,IN size,IN vrstatus,IN siristatus,IN vrsupport);

    if( MP_NO_ERROR != retval )
    {
        ETG_TRACE_ERR(("Error while preparing parameter string"));
    }
    else
    {
        retval = LocalSPM::GetDeviceDispatcher().RouteMessage(IN DTY_BLUETOOTH, IN "HFP_VOICE_RECOGNITION_SESSION_STATUS", IN parameterString);

        if( MP_NO_ERROR != retval )
        {
            ETG_TRACE_ERR(("Error while sending internal event via SMF"));
        }
    }

    return retval;
}
tResult MediaPlayerInterface::CreateMediaPlayerListSearchString( OUT tListID   &listID,
        OUT tListSize &listSize,
        OUT tDeviceID &deviceID,
        IN const tListID listIDParent,
        IN const tListID listIDReplaceable,
        IN const tSearchString searchString) //Roadmap 16014 'Full text search'
        {
    ENTRY

    //In parameter
    VARTRACE(listIDParent)
    VARTRACE(listIDReplaceable)
    VARTRACE(searchString)

    LOCK
    memset(&listID,  0, sizeof listID);
    memset(&listSize,0, sizeof listSize);
    memset(&deviceID,0, sizeof deviceID);

    tResult retVal= -1;

    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetListControl().CreateMediaPlayerIndexedListWithSearchString(OUT listID, OUT listSize, OUT deviceID, IN listIDParent, IN listIDReplaceable, IN searchString);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(listID)
    VARTRACE(listSize)
    VARTRACE(deviceID)
    VARTRACE(retVal)
    return retVal;
        }

tResult MediaPlayerInterface::SendVideoProperty(
        IN const tVideoProperty videoProperty,
        IN const tPropertyValue propertyValue)
{
    ENTRY
    VARTRACE(videoProperty)
    VARTRACE(propertyValue)
    LOCK

    tResult retVal= -1;
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetPlayerManager().SendVideoProperty(IN videoProperty,IN propertyValue);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::GetVideoBrightness(OUT tVideoBrightness &videoBrightness)
{
    ENTRY
    //LOCK

    memset(&videoBrightness, 0, sizeof videoBrightness);

    tResult retVal= -1;
    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
        {
            retVal = LocalSPM::GetDBManager().GetVideoBrightness(OUT videoBrightness);
        }
        else
        {
            //set default value
            videoBrightness = LocalSPM::GetDataProvider().DefaultVideoBrightness();
            retVal = MP_NO_ERROR;
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
    }
    VARTRACE(videoBrightness)
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::GetVideoHue(OUT tVideoHue &videoHue)
{
    ENTRY
    //LOCK

    memset(&videoHue, 0, sizeof videoHue);

    tResult retVal= -1;
    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
        {
            retVal = LocalSPM::GetDBManager().GetVideoHue(OUT videoHue);
        }
        else
        {
            //set default value
            videoHue = LocalSPM::GetDataProvider().DefaultVideoHue();
            retVal = MP_NO_ERROR;
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
    }
    VARTRACE(videoHue)
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::GetVideoSaturation(OUT tVideoSaturation &videoSaturation)
{
    ENTRY
    //LOCK

    memset(&videoSaturation, 0, sizeof videoSaturation);

    tResult retVal= -1;
    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
        {
            retVal = LocalSPM::GetDBManager().GetVideoSaturation(OUT videoSaturation);
        }
        else
        {
            //set default value
            videoSaturation = LocalSPM::GetDataProvider().DefaultVideoSaturation();
            retVal = MP_NO_ERROR;
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
    }
    VARTRACE(videoSaturation)
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::GetVideoContrast(OUT tVideoContrast &videoContrast)
{
    ENTRY
    //LOCK

    memset(&videoContrast, 0, sizeof videoContrast);

    tResult retVal= -1;
    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
        {
            retVal = LocalSPM::GetDBManager().GetVideoContrast(OUT videoContrast);
        }
        else
        {
            //set default value
            videoContrast = LocalSPM::GetDataProvider().DefaultVideoContrast();
            retVal = MP_NO_ERROR;
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
    }
    VARTRACE(videoContrast)
    VARTRACE(retVal)
    return retVal;
}
tResult MediaPlayerInterface::GetPlaybackSpeed(OUT tPlaybackSpeed &playbackSpeed)
{
    ENTRY
       //LOCK

       memset(&playbackSpeed, 0, sizeof playbackSpeed);

       tResult retVal= -1;
       if (FALSE != isCreated)
       {
           if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
           {
               LocalSPM::GetPlayerManager().GetPlaybackSpeed(OUT playbackSpeed);
               retVal = MP_NO_ERROR;
           }
           else
           {
               //set default value
               playbackSpeed = (tPlaybackSpeed)LocalSPM::GetDataProvider().DefaultPlaybackspeed();
               retVal = MP_NO_ERROR;
           }
       }
       else
       {
           ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
       }
       VARTRACE(playbackSpeed)
       VARTRACE(retVal)
       return retVal;
}
tResult MediaPlayerInterface::GetVideoBrightnessOffset(OUT tVideoBrightnessOffset &videoBrightnessOffset)
{
    ENTRY
    //LOCK

    memset(&videoBrightnessOffset, 0, sizeof videoBrightnessOffset);

    tResult retVal= -1;
    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
        {
            retVal = LocalSPM::GetDBManager().GetVideoBrightnessOffset(OUT videoBrightnessOffset);
        }
        else
        {
            //set default value
            videoBrightnessOffset = LocalSPM::GetDataProvider().DefaultVideoBrightnessOffset();
            retVal = MP_NO_ERROR;
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
    }
    VARTRACE(videoBrightnessOffset)
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::GetVideoHueOffset(OUT tVideoHueOffset &videoHueOffset)
{
    ENTRY
    //LOCK

    memset(&videoHueOffset, 0, sizeof videoHueOffset);

    tResult retVal= -1;
    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
        {
            retVal = LocalSPM::GetDBManager().GetVideoHueOffset(OUT videoHueOffset);
        }
        else
        {
            //set default value
            videoHueOffset = LocalSPM::GetDataProvider().DefaultVideoHueOffset();
            retVal = MP_NO_ERROR;
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
    }
    VARTRACE(videoHueOffset)
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::GetVideoSaturationOffset(OUT tVideoSaturationOffset &videoSaturationOffset)
{
    ENTRY
    //LOCK

    memset(&videoSaturationOffset, 0, sizeof videoSaturationOffset);

    tResult retVal= -1;
    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
        {
            retVal = LocalSPM::GetDBManager().GetVideoSaturationOffset(OUT videoSaturationOffset);
        }
        else
        {
            //set default value
            videoSaturationOffset = LocalSPM::GetDataProvider().DefaultVideoSaturationOffset();
            retVal = MP_NO_ERROR;
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
    }
    VARTRACE(videoSaturationOffset)
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::ClearLastModePause(const tDeviceID deviceID)
{
    ENTRY
    VARTRACE(deviceID)

    tResult retVal = MEDIAPLAYER_ERROR_INVALID_STATE;

    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetPlayerManager().ClearLastModePause(IN deviceID);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }

    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::GetWiFiCredentialsResult(const tDeviceID deviceId, tWiFiAPCredentials wifiCredentials, bool bError)
{
    ENTRY

    tResult retVal = MEDIAPLAYER_ERROR_INVALID_STATE;

    if(SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        if(false == bError)
        {
            LocalSPM::GetIPODControl().AccessoryWiFiCredentials(deviceId, wifiCredentials);
            retVal = MEDIAPLAYER_SUCCESS;
        }
        else
        {
            tDeviceInfo deviceInfo;
            InitDeviceInfo(deviceInfo);
            retVal = LocalSPM::GetDBManager().GetDeviceInfo(OUT deviceInfo, IN deviceId);
            if(MP_NO_ERROR != retVal)
            {
                ETG_TRACE_ERR((" Error while retrieving MediaplayerDeviceConections (ErrorCode: %s)", errorString(retVal)));
            }
            else
            {
                ETG_TRACE_USR4((" Disconnect IAP2BT device in DB and Disconnect SPP "));
                LocalSPM::GetIPODControl().SendDBChangeConnectionState(IN deviceId, IN CS_DISCONNECTED);
                retVal = LocalSPM::GetOutputWrapper().SendDisconnectSPPService(IN deviceInfo.appleDeviceMACAddress);
            }
        }
    }
    else
    {
        ETG_TRACE_FATAL((" %s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }



    VARTRACE(retVal);
    return retVal;
}

tResult MediaPlayerInterface::DiPOStartIAP2OverCarplayWiFi(const tGeneralString btMacAddress)
{
    ENTRY

    VARTRACE(btMacAddress);

    tResult retVal = MEDIAPLAYER_ERROR_INVALID_STATE;
    if(SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        if(LocalSPM::GetDataProvider().RequestIAP2SPPConnection() &&
                (LocalSPM::GetDataProvider().iPodControlSupportIAP2BT() || LocalSPM::GetDataProvider().IAPBTConnectionOnlyForSmartApp()))
        {

            tNumberOfDevices numberOfDevices;
            vector<tDeviceInfo> deviceInfos;

            retVal = LocalSPM::GetDBManager().GetMediaplayerDeviceConnections(OUT numberOfDevices, OUT deviceInfos,true);
            if (retVal)
            {
                ETG_TRACE_ERR(("GetMediaplayerDeviceConnections returned error: %d", retVal));
            }
            else
            {
                VARTRACE(numberOfDevices);
                VARTRACE(deviceInfos);

                // Fix for NCG3D-196687: Metadata not updated once device is unplugged during active CPW session
                // Do not rely on iAP BT session availability to start iAP Wifi session for a known CPW device.
                bool bFound = false;
                for( unsigned int i = 0; i < numberOfDevices; i++ )
                {
                    // If a known CPW device is found, initiate iAP Wifi
                    if( ( DTY_IPHONE == deviceInfos[i].deviceType )
                        && ( 0 == strncmp(deviceInfos[i].appleDeviceMACAddress, btMacAddress, strlen_r(deviceInfos[i].appleDeviceMACAddress)) )
                        && ( DCT_WIFI == deviceInfos[i].connectionType ) )
                    {
                        ETG_TRACE_USR4((" Starting iAP Wifi for known device: %s ", deviceInfos[i].deviceName));

                        // Retain existing info and update connectionState alone to CS_ATTACHED
                        deviceInfos[i].connectionState = CS_ATTACHED;
                        LocalSPM::GetDBManager().DeviceChanged(deviceInfos);
                        bFound = true;
                        break;
                    }
                }

                // If no known CPW is found, check for an iAP BT device in attached state
                // and add a new entry for iAP Wifi device in DB.
                if( false == bFound)
                {
                    for( unsigned int i = 0; i < numberOfDevices; i++ )
                    {
                        // If it is a new CPW device, add a new entry in DB
                        if( ( DTY_IPHONE == deviceInfos[i].deviceType )
                                && ( 0 == strncmp(deviceInfos[i].appleDeviceMACAddress, btMacAddress, strlen_r(deviceInfos[i].appleDeviceMACAddress)) )
                                && ( DCT_BLUETOOTH == deviceInfos[i].connectionType )
                                && ( CS_ATTACHED == deviceInfos[i].connectionState )
                                && ( deviceInfos[i].diPOCaps == DIPO_CAP_CARPLAY_WIFI_FEASIBLE ) )
                        {
                            ETG_TRACE_USR4((" Starting iAP Wifi for newly connected device: %s ", deviceInfos[i].deviceName));

                            vector<tDeviceInfo> wifiDevicesInfo;
                            tDeviceInfo wifiDevice;
                            tMountPoint mountPoint;
                            memset(mountPoint, '\0',sizeof(mountPoint));
                            size_t len = strlen_r(deviceInfos[i].mountPoint);
                            char appendString[6] = "_wifi";
                            strncpy_r(wifiDevice.mountPoint, deviceInfos[i].mountPoint, sizeof(wifiDevice.mountPoint));
                            strncpy_r(&wifiDevice.mountPoint[len],appendString,sizeof(appendString));
                            VARTRACE(wifiDevice.mountPoint);

                            len = strlen_r(deviceInfos[i].serialNumber);
                            strncpy_r(wifiDevice.serialNumber, deviceInfos[i].serialNumber, sizeof(wifiDevice.serialNumber));
                            strncpy_r(&wifiDevice.serialNumber[len],appendString,sizeof(appendString));
                            VARTRACE(wifiDevice.serialNumber);

                            strncpy_r(wifiDevice.UUID, deviceInfos[i].UUID, sizeof(wifiDevice.UUID));
                            strncpy_r(wifiDevice.deviceVersion, deviceInfos[i].deviceVersion, sizeof(wifiDevice.deviceVersion));
                            strncpy_r(wifiDevice.deviceName, deviceInfos[i].deviceName, sizeof(wifiDevice.deviceName));
                            strncpy_r(wifiDevice.accessoryName, deviceInfos[i].accessoryName, sizeof(wifiDevice.accessoryName));
                            strncpy_r(wifiDevice.appleDeviceMACAddress, deviceInfos[i].appleDeviceMACAddress, sizeof(wifiDevice.appleDeviceMACAddress));

                            wifiDevice.deviceType = deviceInfos[i].deviceType;
                            wifiDevice.connectionState = CS_ATTACHED;
                            wifiDevice.fileSystemType  = deviceInfos[i].fileSystemType;
                            wifiDevice.partitionNumber = deviceInfos[i].partitionNumber;
                            wifiDevice.totalSize       = deviceInfos[i].totalSize;
                            wifiDevice.freeSize        = deviceInfos[i].freeSize;
                            wifiDevice.productID       = deviceInfos[i].productID;
                            wifiDevice.connectionType  = DCT_WIFI;
                            wifiDevice.deviceState     = DS_INITIALIZING;

                            wifiDevicesInfo.push_back(wifiDevice);

                            LocalSPM::GetDBManager().DeviceChanged(wifiDevicesInfo);
                            break;
                        }
                    }

                }
            }

        }


    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }

    VARTRACE(retVal);
    return retVal;
}

tResult MediaPlayerInterface::GetDiPOPower(tDeviceID &deviceID, tDiPOPower &power)
{
    ENTRY
    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);

    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
        {
            retVal = LocalSPM::GetIPODControl().GetDiPOPower(OUT deviceID, OUT power);
        }
        else
        {
            // set default value
            deviceID = DEVICE_ID_NOT_SET;
            InitDiPOPower(power);
            retVal = MP_NO_ERROR;
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();

    VARTRACE(deviceID);
    VARTRACE(power);
    VARTRACE(retVal)
    return retVal;
}
tResult MediaPlayerInterface::GetDiskError(OUT tBool& isError, OUT tU8& diskError)
{
    ENTRY
    tResult retVal= -1;
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetDVDControl().GetDiskError(isError, diskError);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::GetDiskMechanicalInfo(OUT tU8& mechanicalInfo)
{
    ENTRY
    tResult retVal= -1;
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetDVDControl().GetDiskMechanicalInfo(mechanicalInfo);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::GetDVDDriveInfo(OUT tU8& discType, OUT tU8& driveMode)
{
    ENTRY
    tResult retVal= -1;
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetDVDControl().GetDVDDriveInfo(discType, driveMode);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::GetSkipInfo(OUT tBool& skip)
{
    ENTRY
    tResult retVal= -1;
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetDVDControl().GetSkipInfo(skip);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    VARTRACE(retVal)
    return retVal;
}
tResult MediaPlayerInterface::GetDVDMenuPlaybackOngoing(OUT tBool& dvdMenu)
{
    ENTRY
    tResult retVal= -1;
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetDVDControl().GetMenuPlaybackOngoing(dvdMenu);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    VARTRACE(dvdMenu)
    return retVal;
}
tResult MediaPlayerInterface::GetDirectSelectInfo(OUT tBool& directSelect)
{
    ENTRY
    tResult retVal= -1;
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetDVDControl().GetDirectSelectInfo(directSelect);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    VARTRACE(retVal)
    return retVal;
}
tResult MediaPlayerInterface::GetKeyCommandResponse(OUT tU8& response)
{
    ENTRY
    tResult retVal= -1;
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetDVDControl().GetKeyCommandResponse(response);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    VARTRACE(retVal)
    return retVal;
}
tResult MediaPlayerInterface::GetAngleInfo(OUT tU8& angleTrack, tU8& totalAngle)
{
    ENTRY
    tResult retVal= -1;
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetDVDControl().GetAngleInfo(angleTrack, totalAngle);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::GetSubtitleInfo(OUT tBool& state, OUT tU8& current, tU8& total, OUT tU16& SubLang)
{
    ENTRY
    tResult retVal= -1;
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetDVDControl().GetSubtitleInfo(state, current, total, SubLang);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::GetAudioChannelInfo(OUT tU8& totalChannels, OUT tBool& subWoofer, tU8& Assignment)
{
    ENTRY
    tResult retVal= -1;
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetDVDControl().GetAudioChannelInfo(totalChannels, subWoofer, Assignment);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::GetDisplayMode(OUT tU8& displayMode)
{
    ENTRY
    tResult retVal= -1;
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetDVDControl().GetDisplayMode(displayMode);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::SetDisplayMode(IN tU8 displayMode)
{
    ENTRY
    tResult retVal= -1;
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetDVDControl().SetDisplayMode(displayMode);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::GetPlayStatusSetting(OUT tBool& playStatus)
{
    ENTRY
    tResult retVal= -1;
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetDVDControl().GetPlayStatusSetting(playStatus);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::SetPlayStatusSetting(IN tBool playStatus)
{
    ENTRY
    tResult retVal= -1;
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetDVDControl().SetPlayStatusSetting(playStatus);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::GetDRCSetting(OUT tU8& drc)
{
    ENTRY
    tResult retVal= -1;
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetDVDControl().GetDRCSetting(drc);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::SetDRCSetting(IN tU8 drc)
{
    ENTRY
    tResult retVal= -1;
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetDVDControl().SetDRCSetting(drc);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::GetCurrentAudioInfo(OUT tU8& soundformat, OUT tU8& VcdAudio,
        OUT tU8& DvdAudio,OUT tU8& totalOutput , OUT tU16& AudioLang)
{
    ENTRY
    tResult retVal= -1;
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetDVDControl().GetCurrentAudioInfo(soundformat, VcdAudio, DvdAudio, totalOutput , AudioLang);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::SetAngleMarkSetting(IN tBool AngleMark)
{
    ENTRY
    tResult retVal= -1;
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetDVDControl().SetAngleMarkSetting(AngleMark);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::GetAngleMarkSetting(OUT tBool& AngleMark)
{
    ENTRY
    tResult retVal= -1;
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetDVDControl().GetAngleMarkSetting(AngleMark);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::SetLanguageSetting(IN tU16 Audio, IN tU16 SubTitel, IN tU16 Menu)
{
    ENTRY
    tResult retVal= -1;
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetDVDControl().SetLanguageSetting(Audio, SubTitel, Menu);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::GetLanguageSetting(OUT tU16& Audio, OUT tU16& SubTitel, OUT tU16& Menu)
{
    ENTRY
    tResult retVal= -1;
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetDVDControl().GetLanguageSetting(Audio, SubTitel, Menu);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    VARTRACE(retVal)
    return retVal;
}
tResult MediaPlayerInterface::RequestDiscOperation(tU8 Operation)
{
    ENTRY
    //LOCK

    tResult retVal= -1;
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetDVDControl().RequestDiscOperation(Operation);
    }
    else
    {

        retVal = MP_NO_ERROR;
    }
    VARTRACE(Operation)
    VARTRACE(retVal)
    return retVal;
}
tResult MediaPlayerInterface::RequestVCDAudioChannel(tU8 SelectionType, tU8 AudioMode)
{
    ENTRY
    //LOCK

    tResult retVal= -1;
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetDVDControl().RequestVCDAudioChannel(SelectionType, AudioMode);
    }
    else
    {

        retVal = MP_NO_ERROR;
    }
    VARTRACE(SelectionType)
    VARTRACE(AudioMode)
    VARTRACE(retVal)
    return retVal;
}
tResult MediaPlayerInterface::RequestDVDAudioChannel(tU8 SelectionType, tU8 AudioMode)
{
    ENTRY
    //LOCK

    tResult retVal= -1;
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetDVDControl().RequestDVDAudioChannel(SelectionType, AudioMode);
    }
    else
    {

        retVal = MP_NO_ERROR;
    }
    VARTRACE(SelectionType)
    VARTRACE(AudioMode)
    VARTRACE(retVal)
    return retVal;
}
tResult MediaPlayerInterface::SendNavigationKey(tU8 Action)
{
    ENTRY
    //LOCK

    tResult retVal= -1;
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetDVDControl().SendNavigationKey(Action);
    }
    else
    {
        retVal = MP_NO_ERROR;
    }
    VARTRACE(Action)
    VARTRACE(retVal)
    return retVal;
}
tResult MediaPlayerInterface::SendTouchCommand(tU8 Action, tU16 Xcod, tU16 Ycod, tU16 maxXcod, tU16 maxYcod)
{
    ENTRY

    tResult retVal= -1;
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetDVDControl().SendTouchCommand(Action, Xcod, Ycod, maxXcod, maxYcod);
    }
    else
    {

        retVal = MP_NO_ERROR;
    }
    VARTRACE(Action)
    VARTRACE(Xcod)
    VARTRACE(Ycod)
    VARTRACE(maxXcod)
    VARTRACE(maxYcod)
    VARTRACE(retVal)
    return retVal;
}
tResult MediaPlayerInterface::RequestDirectSearch(tU32 titel, tU32 chapter)
{
    ENTRY
    //LOCK

    tResult retVal= -1;
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetDVDControl().RequestDirectSearch(titel, chapter);
    }
    else
    {

        retVal = MP_NO_ERROR;
    }

    // VARTRACE(vv/*titel*/)
    //VARTRACE(chapter)
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::RequestAngleChange(tU8 SelectionType, tU8 Angle)
{
    ENTRY

    tResult retVal= -1;
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetDVDControl().RequestAngleChange(SelectionType, Angle);
    }
    else
    {

        retVal = MP_NO_ERROR;
    }
    VARTRACE(SelectionType)
    VARTRACE(Angle)
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::SetMenuPlayBackControl(tU8 MenuControl)
{
    ENTRY
    //LOCK

    tResult retVal= -1;
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetDVDControl().SetMenuPlayBackControl(MenuControl);
    }
    else
    {

        retVal = MP_NO_ERROR;
    }
    VARTRACE(MenuControl)
    VARTRACE(retVal)
    return retVal;
}
tResult MediaPlayerInterface::SelectDirectNumber(tU32 number)
{
    ENTRY
    //LOCK

    tResult retVal= -1;
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetDVDControl().SelectDirectNumber(number);
    }
    else
    {

        retVal = MP_NO_ERROR;
    }
    //VARTRACE(number)
    VARTRACE(retVal)
    return retVal;
}
tResult MediaPlayerInterface::RequestTitleSearch(tU32 number)
{
    ENTRY
    //LOCK

    tResult retVal= -1;
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetDVDControl().RequestTitleSearch(number);
    }
    else
    {

        retVal = MP_NO_ERROR;
    }
    //VARTRACE(number)
    VARTRACE(retVal)
    return retVal;
}
tResult MediaPlayerInterface::RequestSubtitleChange(tBool state, tU8  selectionType , tU8 Subtitel)
{
    ENTRY
    //LOCK

    tResult retVal= -1;
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetDVDControl().RequestSubtitleChange(state, selectionType, Subtitel);
    }
    else
    {

        retVal = MP_NO_ERROR;
    }
    VARTRACE(state)
    VARTRACE(selectionType)
    VARTRACE(Subtitel)
    VARTRACE(retVal)
    return retVal;
}
tResult MediaPlayerInterface::RequestCMSkip(tU8 SkipValue)
{
    ENTRY
    //LOCK

    tResult retVal= -1;
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetDVDControl().RequestCMSkip(SkipValue);
    }
    else
    {

        retVal = MP_NO_ERROR;
    }
    VARTRACE(SkipValue)
    VARTRACE(retVal)
    return retVal;
}
//>--Roadmap 16003 : 'CD Ripping With Gracenote'
tResult MediaPlayerInterface::ReorderInternalPlaylist(IN const tPath &playlistPath, IN const tDeviceID deviceID, IN const tPosition oldPosition,IN const tPosition newPosition)
{
    ENTRY;
    LOCK
    VARTRACE(playlistPath)
    VARTRACE(deviceID)
    VARTRACE(oldPosition)
    VARTRACE(newPosition)
    tResult retVal = -1;
    TimeTrace ticks(__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState()) {
        retVal = LocalSPM::GetListControl().ReorderInternalPlaylist(OUT playlistPath, IN deviceID, IN oldPosition, IN newPosition);
    } else {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    return retVal;
}

tResult MediaPlayerInterface::DeleteMediaContent(IN const tListType listType, const tFilterTag1 tag1, const tFilterTag2 tag2, const tFilterTag3 tag3, const tFilterTag4 tag4, const tDeviceID deviceID)
{
    ENTRY;
    LOCK
    VARTRACE(listType)
    VARTRACE(tag1)
    VARTRACE(tag2)
    VARTRACE(tag3)
    VARTRACE(tag4)
    VARTRACE(deviceID)
    tResult retVal = -1;

    TimeTrace ticks(__PRETTY_FUNCTION__);

    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState()){

        tDeleteMediaByTagsParam deleteMediaByTagsParam;
        memset(&deleteMediaByTagsParam,0,sizeof(tDeleteMediaByTagsParam));
        deleteMediaByTagsParam.listType = listType;
        deleteMediaByTagsParam.tag1     = tag1;
        deleteMediaByTagsParam.tag2     = tag2;
        deleteMediaByTagsParam.tag3     = tag3;
        deleteMediaByTagsParam.tag4     = tag4;
        deleteMediaByTagsParam.deviceID = deviceID;

        retVal = LocalSPM::GetCustomControl().DeleteMediaContent(deleteMediaByTagsParam);

    }
    else {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }

    ticks.elapsed();
    return retVal;

}

tResult MediaPlayerInterface::DeleteMediaContent(IN const vector<tObjectID> mediaObjectIDVector, const tDeviceID deviceID)
{
    ENTRY;
    LOCK

    VARTRACE(deviceID)
    tResult retVal = -1;

    TimeTrace ticks(__PRETTY_FUNCTION__);

    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState()){

        tDeleteMediaByObjectIDParam deleteMediaByObjectIDParam;
        deleteMediaByObjectIDParam.objectIDs = mediaObjectIDVector;
        deleteMediaByObjectIDParam.deviceID = deviceID;

        retVal = LocalSPM::GetCustomControl().DeleteMediaContent(deleteMediaByObjectIDParam);

    }
    else {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }

    ticks.elapsed();
    return retVal;

}


tResult MediaPlayerInterface::DeleteMediaContent(IN const vector<string> mediaObjectUrlVector, const tDeviceID deviceID)
{
    ENTRY;
    LOCK

    VARTRACE(deviceID)
    tResult retVal = -1;

    TimeTrace ticks(__PRETTY_FUNCTION__);

    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState()){

        tDeleteMediaByUrlParam deleteMediaByUrlParam;
        deleteMediaByUrlParam.objectUrl = mediaObjectUrlVector;
        deleteMediaByUrlParam.deviceID = deviceID;

        retVal = LocalSPM::GetCustomControl().DeleteMediaContent(deleteMediaByUrlParam);

    }
    else {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }

    ticks.elapsed();
    return retVal;

}

tResult MediaPlayerInterface::EditMetaData(IN tEditMetaDataByID editMetaDataByID)
{
    ENTRY;
    LOCK
    tResult retVal = -1;
    TimeTrace ticks(__PRETTY_FUNCTION__);

    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState()){

        retVal = LocalSPM::GetCustomControl().EditMetaData(editMetaDataByID);

    }
    else {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }

    ticks.elapsed();
    return retVal;


}

tResult MediaPlayerInterface::EditMetaData(IN tEditMetaDataByUrl editMetaDataByUrl)
{
    ENTRY;
    LOCK
    tResult retVal = -1;
    TimeTrace ticks(__PRETTY_FUNCTION__);

    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState()){

        retVal = LocalSPM::GetCustomControl().EditMetaData(editMetaDataByUrl);

    }
    else {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }

    ticks.elapsed();
    return retVal;
}
//<--Roadmap 16003 : 'CD Ripping With Gracenote'

tResult MediaPlayerInterface::SetAutoRipping(tBoolean AutoRippingMode)
{
    ENTRY
    tResult retVal= MP_NO_ERROR;

    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
        {
            retVal = LocalSPM::GetCDRipperControl().SetAutoRippingMode(AutoRippingMode);
        }
        else
        {

        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
    }

    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::SetGracenoteLookupMode(tBoolean GNlookupMode)
{
    ENTRY
    tResult retVal= MP_NO_ERROR;
    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
        {
            tGracenoteLookupMode lookupMode;
            lookupMode = (tGracenoteLookupMode)GNlookupMode;
            retVal = LocalSPM::GetCDRipperControl().SetGracenoteDBLookupMode(lookupMode);
        }
        else
        {
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
    }
    VARTRACE(retVal)
    return retVal;
}
tResult MediaPlayerInterface::GetAlbumStringsforGracenote(OUT vector<string> &albumStringVector)
{
    ENTRY
    tResult retVal= MP_NO_ERROR;

    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
        {
            retVal = LocalSPM::GetCDRipperControl().GetAlbumStringsforGracenote(albumStringVector);
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
    }

    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::ExtractGracenoteMetadata(IN string albumTitle, tBool multipleAlbumSelection)
{
    ENTRY
    tResult retVal= MP_NO_ERROR;

    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
        {
            retVal = LocalSPM::GetCDRipperControl().ExtractGracenoteMetadata(albumTitle,multipleAlbumSelection);
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
    }

    VARTRACE(retVal)
    return retVal;
}
tResult MediaPlayerInterface::GetGNMetadataExtractionStatus(tGNmetadataExtractionResult& GNmetadataExtractionResult)
{
    ENTRY
    tResult retVal= MP_NO_ERROR;
    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
        {
            retVal = LocalSPM::GetCDRipperControl().GetGNMetadataExtractionStatus(GNmetadataExtractionResult);
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
    }
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::GetGracenoteVersion(tGracenoteVersion &GracenoteVersion)
{
    ENTRY
    tResult retVal= MP_NO_ERROR;
    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
        {
            retVal = LocalSPM::GetCDRipperControl().GetGracenoteVersion(GracenoteVersion);
        }
        else
        {
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
    }
    VARTRACE(retVal)
    return retVal;
}
tResult MediaPlayerInterface::StopRipping(void)
{
    ENTRY
    tResult retVal= MP_NO_ERROR;

    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
        {
            retVal = LocalSPM::GetCDRipperControl().StopRippingByUser();
        }
        else
        {

        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
    }

    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::StartRipping(void)
{
    ENTRY
    tResult retVal= MP_NO_ERROR;

    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
        {
            retVal = LocalSPM::GetDVDControl().StartRippingByUser();
        }
        else
        {

        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
    }

    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::GetRippingStatus(tUInt &RippingState, tUInt &TotalTracks, tUInt &CurrentRippingIndex, tUInt &Percentage)
{
    ENTRY
    tResult retVal= MP_NO_ERROR;

    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
        {
            retVal = LocalSPM::GetCDRipperControl().RippingStatus(RippingState, TotalTracks, CurrentRippingIndex, Percentage);
        }
        else
        {

        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
    }

    VARTRACE(retVal)
    return retVal;
}


tResult MediaPlayerInterface::GetAutoRippingStatus(tBoolean &AutoRippingMode)
{
    ENTRY
    tResult retVal= MP_NO_ERROR;

    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
        {
            retVal = LocalSPM::GetCDRipperControl().GetAutoRippingStatus(AutoRippingMode);
        }
        else
        {

        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
    }

    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::GetRippingError(tResult &RippingError)
{
    ENTRY
    tResult retVal= MP_NO_ERROR;
    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
        {
            retVal = LocalSPM::GetCDRipperControl().GetRippingError(RippingError);
        }
        else
        {
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
    }
    VARTRACE(retVal)
    return retVal;
}
tResult MediaPlayerInterface::GetGracenoteLookupMode(tGracenoteLookupMode& GracenoteLookupMode)
{
    ENTRY
    tResult retVal= MP_NO_ERROR;
    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
        {
            GracenoteLookupMode =  LocalSPM::GetCDRipperControl().GetGracenoteDBLookupMode();
        }
        else
        {
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
    }
    VARTRACE(GracenoteLookupMode);
    return retVal;
}
tResult MediaPlayerInterface::DatabaseStatistics(void)
{
    ENTRY

    tResult retVal= -1;
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetDBManager().DatabaseStatistics();
    }
    else
    {

        retVal = MP_NO_ERROR;
    }
    return retVal;
}

tResult MediaPlayerInterface::GetRippingEncodingQuality(OUT tEncodingQuality &rippingEncodingQuality)
{
    ENTRY
    tResult retVal= -1;
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetCDRipperControl().GetRippingEncodingQuality(rippingEncodingQuality);
    }
    else
    {

        retVal = MP_NO_ERROR;
    }
    return retVal;
}

tResult MediaPlayerInterface::SetRippingEncodingQuality(IN tEncodingQuality rippingEncodingQuality)
{
    ENTRY
    tResult retVal= -1;
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetCDRipperControl().SetRippingEncodingQuality(rippingEncodingQuality);
    }
    else
    {

        retVal = MP_NO_ERROR;
    }
    return retVal;
}
tResult MediaPlayerInterface::getStreamingInfo(OUT tU8 &device, OUT tU8 &tag, OUT tS16 &info)
{
    ENTRY

    tResult retVal= MP_NO_ERROR;
    tResponseMsg l_info = REASON_OK;
    tDeviceType type;
    tDeviceID deviceId;

    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
        {
            retVal = LocalSPM::GetPlayerManager().getStreamingInfo(l_info, type, deviceId);
            if(retVal != MP_NO_ERROR)


            {
                ETG_TRACE_ERR(("Error: no response msg available for active device"));
            }
            info = (tS16)l_info;
            tag = (tU8)deviceId;
            device = (tU8)type;
        }
        else
        {

        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
    }
    VARTRACE(device)
    VARTRACE(tag)
    VARTRACE(info)
    return retVal;
}
//>--Roadmap CMG3G-10221 : 'Scene Recorder '
tResult MediaPlayerInterface::SetFilePermissions(IN std::vector<tFilePermission> pfilePermissions, IN tDeviceID const pDeviceID)
{
    ENTRY
    LOCK
    VARTRACE(pDeviceID)
    tResult retVal = -1;

    TimeTrace ticks(__PRETTY_FUNCTION__);

    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState()){

        tSetFilePermission setMediaPermissionByObjectID;
        setMediaPermissionByObjectID.objectIDs = pfilePermissions;
        setMediaPermissionByObjectID.deviceID = pDeviceID;
        retVal = LocalSPM::GetCustomControl().SetFilePermissions(setMediaPermissionByObjectID);

    }
    else {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }

    ticks.elapsed();
    return retVal;
}
tResult MediaPlayerInterface::MediaplayerFolderInfo(IN tDeviceID deviceID, IN tPath Path,
        IN tFileTypeSelection fileTypeSelection, OUT tFolderInfo& folderinfo)
{
    ENTRY
    LOCK
    VARTRACE(deviceID)
    tResult retVal = -1;

    TimeTrace ticks(__PRETTY_FUNCTION__);

    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState()){
        retVal = LocalSPM::GetListControl().MediaplayerFolderInfo(IN deviceID, IN Path,
                                                                  IN fileTypeSelection,OUT folderinfo);
    }
    else {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }

    ticks.elapsed();
    return retVal;
}
//<--Roadmap CMG3G-10221 : 'Scene Recorder '

tResult MediaPlayerInterface::GetRGUpdate(tDeviceID &deviceID, tDiPORouteGuidanceUpdate &RGUpdate)
{
    ENTRY
    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    deviceID = DEVICE_ID_NOT_SET;
    InitDiPORouteGuidanceUpdate(RGUpdate);
    retVal = MP_NO_ERROR;

    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
        {
#ifdef IPODCONTROL_IAP2_PF_R26
#ifdef TARGET_BUILD
            retVal = LocalSPM::GetIPODControl().GetRGUpdate(OUT deviceID, OUT RGUpdate);
#endif
#endif
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();

    VARTRACE(deviceID);
    VARTRACE(RGUpdate);
    VARTRACE(retVal)
    return retVal;
}


tResult MediaPlayerInterface::GetRGManeuverUpdate(tDeviceID &deviceID, tDiPORouteGuidanceManeuverUpdate &RGManeuverUpdate)
{
    ENTRY
    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    deviceID = DEVICE_ID_NOT_SET;
    InitDiPORouteGuidanceManeuverUpdate(RGManeuverUpdate);
    retVal = MP_NO_ERROR;

    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
        {
#ifdef IPODCONTROL_IAP2_PF_R26
#ifdef TARGET_BUILD
            retVal = LocalSPM::GetIPODControl().GetRGManeuverUpdate(OUT deviceID, OUT RGManeuverUpdate);
#endif
#endif
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();

    VARTRACE(deviceID);
    VARTRACE(RGManeuverUpdate);
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::PullRGUpdate(tDeviceID &deviceID, tDiPORouteGuidanceUpdate &RGUpdate)
{
    ENTRY
    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    deviceID = DEVICE_ID_NOT_SET;
    InitDiPORouteGuidanceUpdate(RGUpdate);
    retVal = MP_NO_ERROR;

    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
        {
#ifdef IPODCONTROL_IAP2_PF_R26
#ifdef TARGET_BUILD
            retVal = LocalSPM::GetIPODControl().PullRGUpdate(OUT deviceID, OUT RGUpdate);
#endif
#endif
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();

    VARTRACE(deviceID);
    VARTRACE(RGUpdate);
    VARTRACE(retVal)
    return retVal;
}


tResult MediaPlayerInterface::PullRGManeuverUpdate(tDeviceID &deviceID, tDiPORouteGuidanceManeuverUpdate &RGManeuverUpdate)
{
    ENTRY
    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    deviceID = DEVICE_ID_NOT_SET;
    InitDiPORouteGuidanceManeuverUpdate(RGManeuverUpdate);
    retVal = MP_NO_ERROR;

    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
        {
#ifdef IPODCONTROL_IAP2_PF_R26
#ifdef TARGET_BUILD
            retVal = LocalSPM::GetIPODControl().PullRGManeuverUpdate(OUT deviceID, OUT RGManeuverUpdate);
#endif
#endif
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();

    VARTRACE(deviceID);
    VARTRACE(RGManeuverUpdate);
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::DiPOStartRouteGuidance(const tMountPoint USBSerialNumber, tRouteGuidanceDisplayComponentIDList RouteGuidanceDisplayComponentIDs, tBoolean SourceNameAvail, tBoolean SourceSupportsRouteGuidance)
{
    ENTRY
    VARTRACE(USBSerialNumber);

    tResult retVal = MEDIAPLAYER_ERROR_INVALID_STATE;
    if(SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        tDeviceID deviceID = GetAppleDeviceIDUsingUSBSerialNumber(USBSerialNumber);
        if(DEVICE_ID_NOT_SET != deviceID)
        {
            retVal = LocalSPM::GetIPODControl().SendDiPOStartRouteGuidance(IN deviceID, IN RouteGuidanceDisplayComponentIDs, IN SourceNameAvail, IN SourceSupportsRouteGuidance);
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }

    VARTRACE(retVal);
    return retVal;
}

tResult MediaPlayerInterface::DiPOStopRouteGuidance(const tMountPoint USBSerialNumber, tRouteGuidanceDisplayComponentIDList RouteGuidanceDisplayComponentIDs)
{
    ENTRY
    VARTRACE(USBSerialNumber);

    tResult retVal = MEDIAPLAYER_ERROR_INVALID_STATE;
    if(SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        tDeviceID deviceID = GetAppleDeviceIDUsingUSBSerialNumber(USBSerialNumber);
        if(DEVICE_ID_NOT_SET != deviceID)
        {
            retVal = LocalSPM::GetIPODControl().SendDiPOStopRouteGuidance(IN deviceID, IN RouteGuidanceDisplayComponentIDs);
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }

    VARTRACE(retVal);
    return retVal;
}

tResult MediaPlayerInterface::DiPOSetRGDisplayComponentInfo(const tDiPORGDisplayComponentList RGDisplayComponentList)
{
    ENTRY

    tResult retVal = MEDIAPLAYER_ERROR_INVALID_STATE;
    vector<tDiPORGDisplayComponentInfo>::const_iterator it;

    if(SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        for(it = RGDisplayComponentList.begin(); it != RGDisplayComponentList.end(); it++)
        {
            retVal = LocalSPM::GetDBManager().UpdateRouteGuidanceDisplayComponentInfo(*it);
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }

    VARTRACE(RGDisplayComponentList);

    VARTRACE(retVal);
    return retVal;
}

tResult MediaPlayerInterface::DiPOGetUSBSerialNumber(const tDeviceID deviceID, tMountPoint USBSerialNumber)
{
    ENTRY
    VARTRACE(deviceID);

    tResult retVal = MEDIAPLAYER_ERROR_INVALID_STATE;
    if(SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
            tNumberOfDevices numberOfDevices;
            vector<tDeviceInfo> deviceInfos;
            retVal = LocalSPM::GetDBManager().GetMediaplayerDeviceConnections(OUT numberOfDevices, OUT deviceInfos,true);
            if (retVal)
            {
                ETG_TRACE_ERR(("GetMediaplayerDeviceConnections returned error: %d", retVal));
            }
            else
            {
                VARTRACE(numberOfDevices);
                VARTRACE(deviceInfos);
                for(unsigned int i = 0; i<numberOfDevices; i++ )
                {
                    if((deviceInfos[i].deviceID == deviceID))
                    {
                        if(deviceInfos[i].diPOCaps == DIPO_CAP_CARPLAY_WIFI){
                            strncpy_r(USBSerialNumber, deviceInfos[i].appleDeviceUSBSerialNumber, sizeof(tMountPoint));
                            VARTRACE(deviceInfos[i].appleDeviceUSBSerialNumber);
                            break;
                        }
                        if(deviceInfos[i].diPOCaps == DIPO_CAP_CARPLAY){
                            strncpy_r(USBSerialNumber, deviceInfos[i].mountPoint, sizeof(tMountPoint));
                            VARTRACE(deviceInfos[i].mountPoint);
                            break;
                        }
                    }
                }

            }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }

    VARTRACE(retVal);
    return retVal;
}


//<--Roadmap 16003 : 'CD Ripping With Gracenote'
tResult MediaPlayerInterface::AddFileNamesToPlaylistByTag(IN const tListType listType, IN const tFilterTag1 tag1, IN const tFilterTag2 tag2, IN const tFilterTag3 tag3, IN const tFilterTag4 tag4, IN const tPath playlistPath, IN const tDeviceID deviceID)
{
    ENTRY
    VARTRACE(deviceID);
    VARTRACE(playlistPath);
    VARTRACE(listType);
    VARTRACE(tag1);
    VARTRACE(tag2);
    VARTRACE(tag3);
    tResult retVal = MEDIAPLAYER_ERROR_INVALID_STATE;
    if(SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        tPlayListAddOperationByTags PlayListAddOperationByTags;
        memset(&PlayListAddOperationByTags,0,sizeof(PlayListAddOperationByTags));

        PlayListAddOperationByTags.deviceID = deviceID;
        PlayListAddOperationByTags.tag1     = tag1;
        PlayListAddOperationByTags.tag2     = tag2;
        PlayListAddOperationByTags.tag3     = tag3;
        PlayListAddOperationByTags.tag4     = tag4;
        PlayListAddOperationByTags.listType = listType;
        strncpy_r(PlayListAddOperationByTags.playlistPath, playlistPath,sizeof(PlayListAddOperationByTags.playlistPath));
        retVal = LocalSPM::GetCustomControl().AddFileNamesToPlayListByTag(IN PlayListAddOperationByTags);

    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    VARTRACE(retVal);
    return retVal;
}
//>--Roadmap 16003 : 'CD Ripping With Gracenote'

tResult MediaPlayerInterface::VideoManagerSourceUpdate(tVideoSource videoSource , tVideoSourceState status )
{
    ENTRY
    VARTRACE(videoSource);
    VARTRACE(status);
    tResult retVal = MEDIAPLAYER_ERROR_INVALID_STATE;
    if(SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
         if(videoSource == VIDEO_SOURCE_DVD)
         {
          retVal = LocalSPM::GetDVDControl().DVDVideoSourceUpdate(status);
         }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    VARTRACE(retVal);
    return retVal;
}
//<--Roadmap 17001 : 'Personalization'
tResult MediaPlayerInterface::StorePersonalizedFavorite(OUT tFavoriteID &retFavID, IN tObjectID objectID, IN tUserID userID)
{
    ENTRY
    VARTRACE(objectID);
    VARTRACE(userID);
    memset(&retFavID, 0, sizeof retFavID);
    tResult retVal = MEDIAPLAYER_ERROR_INVALID_STATE;
    if(SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        /*Check UserID with current User and update the DB about the favorite*/
        retVal = LocalSPM::GetCustomControl().StorePersonalizedFavorite(OUT retFavID, IN objectID, IN userID);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    VARTRACE(retVal);
    return retVal;
}

tResult MediaPlayerInterface::DeletePersonalizedFavorite(OUT tObjectID objectID, IN tUserID userID)
{
    ENTRY
    VARTRACE(objectID);
    VARTRACE(userID);

    tResult retVal = MEDIAPLAYER_ERROR_INVALID_STATE;
    if(SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        /*Check UserID with current User and update the DB about the favorite*/
        retVal = LocalSPM::GetCustomControl().DeletePersonalizedFavorite(IN objectID, IN userID);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    VARTRACE(retVal);
    return retVal;
}
//<--Roadmap 17001 : 'Personalization'

tResult MediaPlayerInterface::AddOobPairedDeviceResult(const tDeviceID deviceID,const tBoolean result)
{
    ENTRY

    tResult retVal = -1;
    TimeTrace ticks(__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState()) {

        retVal = LocalSPM::GetIPODControl().AddOobPairedDeviceResult(deviceID,result);

    } else {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(retVal)
    return retVal;
}
tResult MediaPlayerInterface::RequestDVDTemperature(const tUserContext userContext)
{
    ENTRY
    VARTRACE(userContext)
    tResult retVal= -1;
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetDVDControl().RequestDVDTemperature(userContext);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::GetDeviceIDForFolderStructureCleared(OUT tDeviceID &deviceID)
{
    ENTRY

    deviceID = DEVICE_ID_NOT_SET;

       tResult retVal= -1;
       if (FALSE != isCreated)
       {
           if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
           {
               deviceID = LocalSPM::GetCustomControl().GetDeviceIDForFolderStructureCleared();

               //clear the device ID.

               LocalSPM::GetCustomControl().SetDeviceIDForFolderStructureCleared(DEVICE_ID_NOT_SET);
               retVal = MP_NO_ERROR;
           }
           else
           {
               //set default value
               deviceID = DEVICE_ID_NOT_SET;
               retVal = MP_NO_ERROR;
           }
       }
       else
       {
           ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
       }
       VARTRACE(deviceID);
       VARTRACE(retVal);
       return retVal;
}
tResult MediaPlayerInterface::GetValuesForReadyToPlay(tDeviceSerialNumber &deviceAddress , tBool &PlayingStatus)
{
    ENTRY

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
        {
            retVal = LocalSPM::GetCustomControl().GetDeviceAddressForReadyToPlay(OUT deviceAddress);
            PlayingStatus = LocalSPM::GetCustomControl().GetPlayingStatus();
        }
        else
        {
            //set default value
            PlayingStatus = 0;
            strncpy(deviceAddress,"",sizeof(deviceAddress));
            retVal = MP_NO_ERROR;
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(deviceAddress);
    VARTRACE(PlayingStatus);
    VARTRACE(retVal)
    return retVal;

}

tDeviceID MediaPlayerInterface::GetAppleDeviceIDUsingUSBSerialNumber(const tMountPoint USBSerialNumber)
{
    ENTRY;

    tResult retVal = MP_NO_ERROR;

    tDeviceID deviceID = DEVICE_ID_NOT_SET;

    tNumberOfDevices numberOfDevices = 0;
    vector<tDeviceInfo> deviceInfos;

    retVal = LocalSPM::GetDBManager().GetMediaplayerDeviceConnections(OUT numberOfDevices, OUT deviceInfos,true);
    if(MP_NO_ERROR != retVal)
    {
        ETG_TRACE_ERR(("GetMediaplayerDeviceConnections returned error: %d", retVal));
    }
    else
    {
        VARTRACE(numberOfDevices);
        VARTRACE(deviceInfos);

        // Fix for NCG3D-196834: Metadata not updated once device is unplugged during active CPW session
        // Prioritize iAP Wifi over iAP USB session when device is connected over both USB and Wifi.
        // First check for a CPW device with the serial number
        for( unsigned int i = 0; i< numberOfDevices; ++i )
        {
            if( (0 == strncasecmp(deviceInfos[i].appleDeviceUSBSerialNumber, USBSerialNumber, strlen_r(deviceInfos[i].appleDeviceUSBSerialNumber)))
                    && (deviceInfos[i].diPOCaps == DIPO_CAP_CARPLAY_WIFI))
            {
                deviceID = deviceInfos[i].deviceID;
                ETG_TRACE_USR4((" CPW DeviceID Found: %d", deviceID));
                break;
            }
        }

        // If CPW device is not found check for a CP USB device
        if( DEVICE_ID_NOT_SET == deviceID )
        {
            for( unsigned int i = 0; i< numberOfDevices; ++i )
            {
                if( (0 == strncasecmp(deviceInfos[i].mountPoint, USBSerialNumber, strlen_r(deviceInfos[i].mountPoint)) )
                    && (deviceInfos[i].diPOCaps == DIPO_CAP_CARPLAY))
                {
                    deviceID = deviceInfos[i].deviceID;
                    ETG_TRACE_USR4((" CP USB DeviceID Found: %d", deviceID));
                    break;
                }
            }
        }

        if( DEVICE_ID_NOT_SET == deviceID )
        {
            ETG_TRACE_ERR(("DeviceID not found for serial number : %s ", USBSerialNumber));
        }
    }

    return deviceID;
}

tResult MediaPlayerInterface::SendAppleHIDCommand(const tPlaybackHIDCommand playbackHIDCommand,const tBTButtonEvent keyEvent)
{
    ENTRY
    VARTRACE(playbackHIDCommand)
    VARTRACE(keyEvent)
    LOCK

    tResult retVal= -1;
    TimeTrace ticks (__PRETTY_FUNCTION__);
    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retVal = LocalSPM::GetIPODControl().SendAppleHIDCommand(IN playbackHIDCommand, IN keyEvent);
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }
    ticks.elapsed();
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::GetDiPODeviceTime(tDeviceID &deviceID, tDiPODeviceTime &deviceTime)
{
    ENTRY
    tResult retVal = MP_ERROR;

    if (FALSE != isCreated)
    {
        if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
        {
            retVal = LocalSPM::GetIPODControl().GetDiPODeviceTime(OUT deviceID, OUT deviceTime);
        }
        else
        {
            ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer was created",__PRETTY_FUNCTION__));
    }

    VARTRACE(deviceID);
    VARTRACE(deviceTime);
    VARTRACE(retVal)
    return retVal;
}

tResult MediaPlayerInterface::SendMediaEnginePlay()
{
    ENTRY
    tResult retval = -1;

    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retval = LocalSPM::GetDeviceDispatcher().RouteMessage(IN DTY_IPHONE, IN "START_AUDIO_DEVICE", IN NULL);
        if( MP_NO_ERROR != retval)
        {
            ETG_TRACE_ERR(("Error while sending internal event via SMF"));
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }

    VARTRACE(retval);
    return retval;
}

tResult MediaPlayerInterface::SendMediaEngineStop()
{
    ENTRY
    tResult retval = -1;

    if (SPM_STATE_NORMAL == LocalSPM::GetInstance().GetSPMState())
    {
        retval = LocalSPM::GetDeviceDispatcher().RouteMessage(IN DTY_IPHONE, IN "STOP_AUDIO_DEVICE", IN NULL);
        if( MP_NO_ERROR != retval)
        {
            ETG_TRACE_ERR(("Error while sending internal event via SMF"));
        }
    }
    else
    {
        ETG_TRACE_FATAL(("%s is called before MediaPlayer is in Normal state",__PRETTY_FUNCTION__));
    }

    VARTRACE(retval);
    return retval;
}
