#include <iostream>
using namespace std;

#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <pthread.h>
#include <errno.h>
#include <sys/prctl.h>
#include <setjmp.h>
#include <fcntl.h>
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_mp.h"
#include <unistd.h>
#include <sys/syscall.h>
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_FRAMEWORK
#ifdef TARGET_BUILD
#include "trcGenProj/Header/ThreadFactory.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_FRAMEWORK
#endif
#endif

#include "ThreadFactory.h"
#include "FunctionTracer.h"
#include "TraceDefinitions.h"
#include "Framework_Errorcodes.h"
#include "Utils.h"
#include "Timer.h"
#define DEFAULT_STACKSIZE 0x100000 //1MB
const char* ThreadFactoryLowPrio::mLowprioSlicepath = "/sys/fs/cgroup/cpu/rbcm.slice/rbcm-medialowprio.slice/rbcm-medialowprio-task.slice/tasks";

ThreadFactory::ThreadFactory(unsigned int iNumOfThreadsUsed)
{
    MP_FATAL_ASSERT(iNumOfThreadsUsed != 0);
    MP_FATAL_ASSERT(iNumOfThreadsUsed <= mMaxNumOfThreads);

    mNoOfThreads = (int)iNumOfThreadsUsed;
    ETG_TRACE_USR4(("ThreadFactory: Number of Threads started: mNoOfThreads:%d",mNoOfThreads));

    mInitialized = false;
    mThreadContext.reserve(mNoOfThreads);
    memset(&mLock, 0, sizeof(mFreeWorkerThreads));
    memset(&mFreeWorkerThreads, 0, sizeof(mFreeWorkerThreads));
}

int ThreadFactory::Do(TFThread *thread, int param , void *userContext, void *cleanUpHandle)
{
    ENTRY
    if(!mInitialized)
    {
        ETG_TRACE_ERR(("Thread registration rejected as ThreadFactory is not initialized yet ! "));
        return MP_ERR_TF_NOT_INIT;
    }

    int threadIndex;

    /* wait for a free thread */
    int value;
    sem_getvalue(&mFreeWorkerThreads, &value);
    ETG_TRACE_USR3(("mFreeWorkerThreads=%d", value));

    int res;
    while(1) {
        res = sem_wait(&mFreeWorkerThreads);
        if (!res) break;
        res = errno;
        if (res != EINTR) break;
    }
    MP_FATAL_ASSERT(res == 0);

    // lock this section
    MP_FATAL_ASSERT(pthread_mutex_lock(&mLock) == 0);

    // search loop
    while(1) {

        // find an available worker thread
        for(threadIndex=0; threadIndex<mNoOfThreads; threadIndex++) {

            // is this thread free?
            if (!mThreadContext[threadIndex].busy) {
                mThreadContext[threadIndex].busy = 1; // now this thread is booked

                // unlock this section
                MP_FATAL_ASSERT(pthread_mutex_unlock(&mLock) == 0);

                // set the user values
                mThreadContext[threadIndex].userContext = userContext;
                mThreadContext[threadIndex].threadFunction = thread;
                mThreadContext[threadIndex].functionID = param;
                mThreadContext[threadIndex].cleanUpHandle = cleanUpHandle;

                // unlock the mutex before trigger the thread
                pthread_mutex_unlock(&mThreadContext[threadIndex].mtx);

                return threadIndex;
            }
        }

        // unlock this section
        MP_FATAL_ASSERT(pthread_mutex_unlock(&mLock) == 0);

        // no thread found: this cannot be happen
        MP_FATAL_ASSERT(threadIndex != mNoOfThreads);
    }

    return -1;
}

void *ThreadFactory::startFunction(void *context)
{
    ENTRY_INTERNAL
    tThreadContext *ctx = (tThreadContext *)context;
    int ret;

    MP_FATAL_ASSERT(ctx != NULL);

    // init the threads name
    ResetName(ctx);

    // set the thread property to enable the cancellation later on
    pthread_setcancelstate(PTHREAD_CANCEL_ENABLE ,NULL);
    pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS ,NULL);

    // endless control loop
    while(!ctx->stopThread) {

        // lock the mutex to wait for next task to do
        ret = pthread_mutex_lock(&ctx->mtx);
        if (ret || ctx->stopThread) break; // error in locking or stop requested

        // test
        // printf("thread %d: running", ctx->count);

        // do the user function
        ctx->threadFunction->Do(ctx->functionID, ctx->userContext);

        // reset the thread name
        ResetName(ctx);

        // mark thread is not busy anymore, can be reused
        ctx->busy = 0;
        ctx->cleanUpHandle = NULL;

        // one worker got free, count this
        sem_post(ctx->freeWorkerThreads);
    }

    // end of thread
    ETG_TRACE_USR1(("thread %d: stopping", ctx->count));
    return NULL;
}

void ThreadFactory::Create(void)
{
    ENTRY_INTERNAL

    int i;

    // setup the lock mutex
    MP_FATAL_ASSERT(pthread_mutex_init(&mLock, NULL) == 0);

    /* set the number of threads that are free initial */
    memset(&mFreeWorkerThreads, 0, sizeof(mFreeWorkerThreads));
    sem_init(&mFreeWorkerThreads, 0, (unsigned int)mNoOfThreads);

    // create all threads but dont start them
    for (i=0; i<mNoOfThreads; i++) {

        CreateThread(i);
    }
    mInitialized = true;
}

int ThreadFactory::CreateThread(const int threadIndex)
{
    ENTRY_INTERNAL
    int res;

    // init the factory thread context
    memset(&mThreadContext[threadIndex], 0, sizeof(ThreadFactory::tThreadContext));
    mThreadContext[threadIndex].count = threadIndex+1;  // set the count of this thread for debugging only
    mThreadContext[threadIndex].freeWorkerThreads = &mFreeWorkerThreads;

    // setup the wait mutex
    MP_FATAL_ASSERT(pthread_mutex_init(&mThreadContext[threadIndex].mtx, NULL) == 0);

    // lock the mutex to stop the thread after starting
    MP_FATAL_ASSERT(pthread_mutex_lock(&mThreadContext[threadIndex].mtx) == 0);

    // set the default thread attributes
    MP_FATAL_ASSERT(pthread_attr_init(&mThreadContext[threadIndex].mThreadAttr) == 0);

    // set the size of the stack. Default stack size is too small.
    pthread_attr_setstacksize(&mThreadContext[threadIndex].mThreadAttr, DEFAULT_STACKSIZE);

    // create the thread
    res = pthread_create(&mThreadContext[threadIndex].mThreadHandle, &mThreadContext[threadIndex].mThreadAttr, &ThreadFactory::startFunction, &mThreadContext[threadIndex]);
    if (res) {
        ETG_TRACE_FATAL(("pthread_create: err=%d", res));
        MP_FATAL_ASSERT(res == 0);
    }

    return 0;
}

bool ThreadFactory::JoinTimeoutCallBack(timer_t timerID , void* instance ,const void *userData)
{
    ENTRY_INTERNAL

    jmp_buf *jumpBuffer = (jmp_buf *)userData;
    //ThreadFactory *self = (ThreadFactory *)instance;

    /* jump after the pthread_join position */
    longjmp (*jumpBuffer, 1);

    return 0;
}

int ThreadFactory::CancelThread(const int threadIndex)
{
    ENTRY_INTERNAL

    if(!mInitialized)
    {
        ETG_TRACE_ERR(("CancelThread rejected as ThreadFactory is not initialized yet ! "));
        return MP_ERR_TF_NOT_INIT;
    }

    // send the interrupting signal to thread
    pthread_cancel(mThreadContext[threadIndex].mThreadHandle);

#if 0 //Fix for GMMY17-8324
    // create a supervision timer to jump over the join after a certain time (3000 ms)
    Timer joinTimeout;
    timer_t joinTimeoutID;
    jmp_buf jumpBuffer;
    joinTimeout.StartTimer(OUT joinTimeoutID, IN 3000L, 0L, IN this, IN &JoinTimeoutCallBack, IN (void *)&jumpBuffer);

    /* set the jump back point for long jump out of join timeout handler */
    switch(setjmp(jumpBuffer))
    {
        case 0: // normal processing
            // is thread ended?
            pthread_join(mThreadContext[threadIndex].mThreadHandle, NULL);
            break;
        case 1: // abort by timer callback
            ETG_TRACE_ERR(("CancelThread: pthread_join() skipped"));
            break;
        default:
            break;
    }

    // end the join timeout timer
    joinTimeout.CancelTimer(IN joinTimeoutID);
#else
    pthread_join(mThreadContext[threadIndex].mThreadHandle, NULL);
#endif
    // unlock the mutex to enable the destroy
    pthread_mutex_unlock(&mThreadContext[threadIndex].mtx);

    // destroy the mutex
    pthread_mutex_destroy(&mThreadContext[threadIndex].mtx);

    return 0;
}

int ThreadFactory::SendSignal(const int threadIndex, const int sig)
{
    ENTRY_INTERNAL

    if(!mInitialized)
    {
        ETG_TRACE_ERR(("SendSignal rejected as ThreadFactory is not initialized yet ! "));
        return MP_ERR_TF_NOT_INIT;
    }

    //tgkill(-1, mThreadContext[threadIndex].mThreadHandle, sig);
    pthread_kill(mThreadContext[threadIndex].mThreadHandle, sig);

    return 0;
}

int ThreadFactory::KillThread(const int threadIndex)
{
    ENTRY_INTERNAL

    if(!mInitialized)
    {
        ETG_TRACE_ERR(("KillThread rejected as ThreadFactory is not initialized yet ! "));
        return MP_ERR_TF_NOT_INIT;
    }

    if (mThreadContext[threadIndex].busy) {

        CancelThread(threadIndex);

        CreateThread(threadIndex);

        // signal that one worker got free
        sem_post(mThreadContext[threadIndex].freeWorkerThreads);
    }

    return 0;
}

int ThreadFactory::CleanUpThreads(void *cleanUpHandle)
{
    ENTRY

    if(!mInitialized)
    {
        ETG_TRACE_ERR(("CleanUpThreads rejected as ThreadFactory is not initialized yet ! "));
        return MP_ERR_TF_NOT_INIT;
    }

    if(cleanUpHandle != NULL) {
        for (int i=0; i<mNoOfThreads; i++) {
            if (mThreadContext[i].busy && cleanUpHandle == mThreadContext[i].cleanUpHandle) {
#if 0 //Fix for GMMY17-8324
                ETG_TRACE_ERR(("killing thread %d", i));

                CancelThread(i);

                CreateThread(i);

                // signal that one worker got free
                sem_post(mThreadContext[i].freeWorkerThreads);
#else
                ETG_TRACE_ERR(("Mediaplayer thread ID%d still running (pthread_cancel disabled)", i));
                //ETG_TRACE_ERRMEM(("Mediaplayer thread ID%d still running (pthread_cancel disabled)", i));
#endif
            }
        }
    }
    return 0;
}

int ThreadFactory::KillAllThreads()
{
    ENTRY_INTERNAL
    int i;

    if(!mInitialized)
    {
        ETG_TRACE_ERR(("KillAllThreads rejected as ThreadFactory is not initialized yet ! "));
        return MP_ERR_TF_NOT_INIT;
    }

    mInitialized = false;

    // wakeup all threads to stop them
    for (i=0; i<mNoOfThreads; i++) {

        // flag to stop the thread
        mThreadContext[i].stopThread = 1;

        // unlock the mutex to let the thread run
        MP_FATAL_ASSERT(pthread_mutex_unlock(&mThreadContext[i].mtx) == 0);
    }

    // now join all threads
    for (i=0; i<mNoOfThreads; i++) {
        CancelThread(i);
    }
    return 0;
}

void ThreadFactory::SetName(const char *name)
{
    prctl(PR_SET_NAME, name, 0, 0, 0, 0);
}

void ThreadFactory::ResetName(tThreadContext *ctx)
{
    // reset the thread name
    char name[64];
    snprintf(name, sizeof(name)-1, "MP_Thread_%02d", ctx->count);
    SetName(name);
}

ThreadFactory::tThreadID ThreadFactory::GetThreadId()
{
    return static_cast<ThreadFactory::tThreadID>(pthread_self());
}

int ThreadFactory::GetThreadsRunning()
{
    int value;
    sem_getvalue(&mFreeWorkerThreads, &value);
    return mNoOfThreads - value;
}

int ThreadFactory::SetPriority(int prio)
{
    int policy;
    struct sched_param param;

    pthread_getschedparam(pthread_self(), &policy, &param);

    param.__sched_priority = prio;
    policy = SCHED_FIFO;

    pthread_setschedparam(pthread_self(), policy, &param);

    return 0;
}

/*
 *   Below Function are defined for Low priority thread class .
 *
 */

void ThreadFactoryLowPrio::Create(const int prio)
{
    ENTRY

    int i;

    // setup the lock mutex
    MP_FATAL_ASSERT(pthread_mutex_init(&mLock, NULL) == 0);

    /* set the number of threads that are free initial */
    memset(&mFreeWorkerThreads, 0, sizeof(mFreeWorkerThreads));
    sem_init(&mFreeWorkerThreads, 0, (unsigned int)mNoOfThreads);

    // create all threads but dont start them
    for (i=0; i<mNoOfThreads; i++) {

        CreateThread(i,prio);
    }
    mInitialized = true;
}

int ThreadFactoryLowPrio::CreateThread(const int threadIndex,const int prio)
{
    ENTRY
    int res;
    struct sched_param param;

    // init the factory thread context
    memset(&mThreadContext[threadIndex], 0, sizeof(ThreadFactory::tThreadContext));
    mThreadContext[threadIndex].count = threadIndex+1;  // set the count of this thread for debugging only
    mThreadContext[threadIndex].freeWorkerThreads = &mFreeWorkerThreads;

    // setup the wait mutex
    MP_FATAL_ASSERT(pthread_mutex_init(&mThreadContext[threadIndex].mtx, NULL) == 0);

    // lock the mutex to stop the thread after starting
    MP_FATAL_ASSERT(pthread_mutex_lock(&mThreadContext[threadIndex].mtx) == 0);

    // set the default thread attributes
    MP_FATAL_ASSERT(pthread_attr_init(&mThreadContext[threadIndex].mThreadAttr) == 0);

    // create the thread
    res = pthread_create(&mThreadContext[threadIndex].mThreadHandle, &mThreadContext[threadIndex].mThreadAttr, &ThreadFactoryLowPrio::startFunction, &mThreadContext[threadIndex]);
    if (res) {
        ETG_TRACE_FATAL(("pthread_create: err=%d", res));
        MP_FATAL_ASSERT(res == 0);
    }
    else
    {
        SetPriority(prio,mThreadContext[threadIndex].mThreadHandle);
    }

    return 0;
}
int ThreadFactoryLowPrio::SetPriority(int prio,pthread_t tid)
{
    ENTRY
    int policy;
    struct sched_param param;

    pthread_getschedparam(tid, &policy, &param);

    param.__sched_priority = prio;
    policy = SCHED_FIFO;

    pthread_setschedparam(tid, policy, &param);

    return 0;
}

void *ThreadFactoryLowPrio::startFunction(void *context)
{
    ENTRY
    tThreadContext *ctx = (tThreadContext *)context;
    int ret;

    MP_FATAL_ASSERT(ctx != NULL);

    // init the threads name
    ResetName(ctx);

    // set the thread property to enable the cancellation later on
    pthread_setcancelstate(PTHREAD_CANCEL_ENABLE ,NULL);
    pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS ,NULL);
    StoreThreadId(syscall(SYS_gettid));
    // endless control loop
    while(!ctx->stopThread) {

        // lock the mutex to wait for next task to do
        ret = pthread_mutex_lock(&ctx->mtx);
        if (ret || ctx->stopThread) break; // error in locking or stop requested

        // test
        // printf("thread %d: running", ctx->count);

        // do the user function
        ctx->threadFunction->Do(ctx->functionID, ctx->userContext);

        // reset the thread name
        ResetName(ctx);

        // mark thread is not busy anymore, can be reused
        ctx->busy = 0;
        ctx->cleanUpHandle = NULL;

        // one worker got free, count this
        sem_post(ctx->freeWorkerThreads);
    }

    // end of thread
    ETG_TRACE_USR1(("ThreadFactoryLowPrio thread %d: stopping", ctx->count));
    return NULL;
}

void ThreadFactoryLowPrio::StoreThreadId(const pid_t & thList)
{
    ENTRY
    int status = -1;
    int fileDes = 0;
    string threadId = to_string(thList);
    string file = mLowprioSlicepath;

    fileDes = open(file.c_str(), O_WRONLY | O_APPEND);
    status = errno;
    if(fileDes >= 0)
    {
        size_t write_r = write(fileDes, threadId.c_str(), threadId.size());
        if(write_r != threadId.size())
        {
            status = errno;
            ETG_TRACE_ERR(("storeThreadId()  :  %d -- %s", write_r, threadId.c_str()));
        }
        close(fileDes);
    }
    else
    {
        ETG_TRACE_ERR(("storeThreadId():  open: %d %s", status, strerror(status)));
        ETG_TRACE_ERR(("ERROR :: Open unsucessfull %s ", file.c_str()));
    }
}
