/*
 * ErrorCatcher.cpp
 *
 *  Created on: Mar 30, 2015
 *      Author: thm3hi
 */

#include <string.h>
#include "LongRunnerCatcher.h"

#define ERROR_CATCH_SIGNAL 36

void ErrorCatcherTimeoutCallback(int signo, siginfo_t *info, void *context)
{
    (void)context;

    if((info->si_code == SI_TIMER) && (signo == ERROR_CATCH_SIGNAL))
    {
        jmp_buf *env = (jmp_buf *)(info->si_value.sival_ptr);
        longjmp (*env, -1);
    }

    return ;
}

void LongRunnerCatcherCreateTimer(timer_t *timerId, const long timeout, const void *context)
{
    struct sigevent sev;
    struct itimerspec its;

    /* create the timer fireing a signal to own thread */
    memset(&sev, 0, sizeof(sev));
    sev.sigev_notify = SIGEV_SIGNAL;
    sev.sigev_signo = ERROR_CATCH_SIGNAL;
    sev.sigev_value.sival_ptr = (void *)context;
    *timerId = 0;
    if (timer_create(CLOCK_MONOTONIC, &sev, timerId) == -1)
    {
        return;
    }

    /* set the timeout value */
    memset(&its, 0, sizeof(its)); // set struct to zero
    its.it_value.tv_sec = timeout/1000;
    its.it_value.tv_nsec = (timeout%1000)*1000000L;

    /* run only once (give endless time to prevent process crash on timer_delete with fired timer) */
    its.it_interval.tv_sec = 1000000L;
    its.it_interval.tv_nsec = 0;

    /* start the timer */
    if (timer_settime(*timerId, 0, &its, NULL) == -1)
    {
        return;
    }

    return;
}

void LongRunnerCatcherInit()
{
    struct sigaction SigAction;

    sigemptyset(&SigAction.sa_mask);

    SigAction.sa_sigaction = ErrorCatcherTimeoutCallback;
    SigAction.sa_flags = SA_SIGINFO;

    sigaction(ERROR_CATCH_SIGNAL, &SigAction, NULL);
}

void LongRunnerDestroyTimer(const timer_t timerId)
{
    /* stop the timer */
    struct itimerspec its;
    memset(&its, 0, sizeof(its)); // set struct to zero
    timer_settime(timerId, 0, &its, NULL);

    // delete the timer
    timer_delete(timerId);
}

