/**
 * @defgroup Timer Timer
 * @ingroup MediaPlayer
 * @author Dinesh D
 *
 * Public interface to create high resolution timer functionality
 * @{
 */

#ifndef _TIMER_H_
#define _TIMER_H_

#include <signal.h>
#include <time.h>
#include <vector>

#include "Lock.h"

#define SIGACTION_THREAD 1 /* use a separate thread to receive the sigaction usr1 signal */

#if SIGACTION_THREAD
#include <sys/types.h>
#endif

/* timer class bit definitions */
#define TIMER_CLASS_NONE    (0)         /* no tomer should be handled */
#define TIMER_CLASS_SPM     (1 << 0)    /* class for SPM timers */
#define TIMER_CLASS_OTHERS  (1 << 1)    /* class for others */
#define TIMER_CLASS_ALL     (0xff)      /* all timer classes enabled */

/** 
 * Class which provide high resolution timer features
 */
class Timer {

public:

    /**
     * Constructor
     *
     * @return void
     */
    Timer();

    /**
     * Destructor
     *
     * @return void
     */
    virtual ~Timer();

    /**
     * signature of timer user callback function on timer expiry . Timer user implements this
     * as static function to get the callback on each timer intervals
     *
     * @param[in] timerID   timer user index which differentiates the multiple timer instances
     *                      (supplied during timer start)
     * @param[in] pObject   pointer to object using the timer feature.
     * @param[in] userData  pointer to userData supplied during timer start
     *
     * @return bool FALSE to stop the timer , TRUE to repeat the timer
     */
    typedef bool (*tCallBackFn)(timer_t timerID , void *pObject, const void *userData);

    /**
     * Creates a high-resolution timer and starts it immediately
     *
     * @param[in] timerIndex            timer user index to differentiate multiple timers from same user
     * @param[in] timeMilliseconds      first timeout in milliseconds
     * @param[in] intervalMilliseconds  timer interval in milliseconds
     * @param[in] pObject               object context for timer callback call (pointer to class)
     * @param[in] pCallBackFn           function inside the object that will be called on timer expiring
     * @param[in] userData              additional pointer to user supplied data which will be send back via user call
     *                                  back function when the timer expires
     *
     * @return bool = true: ok, = false: failed
     */
    bool StartTimer(timer_t &timerIndex, long timeMilliseconds, long intervalMilliseconds, void *pObject,tCallBackFn pCallBackFn, const void *userData);

    /**
     * * Creates a hr timer for the use of Statemachines , it dispatches message back to SM on the expiry
     *
     * @param[in] timerIndex            timer user index to differentiate multiple timers from same user
     * @param[in] timeMilliseconds      timer interval in milli seconds
     * @param[in] intervalMilliseconds  timer interval in milliseconds
     * @param[in] message               message that will be sent after timer expired
     *
     * @return bool = true: ok, = false: failed
     */

    bool StartSMTimer(timer_t &timerIndex , long timeMilliseconds, long intervalMilliseconds, const char *message);

    /**
     * Stops the timer
     *
     * @param[in] timerIndex       user Timer Index for timer which needs to be cancelled.
     *
     * @return void
     */
    void CancelTimer (timer_t timerIndex);

    /**
     * Sets timer class of
     *
     * @param[in] timerIndex       my own timer class number
     *
     * @return void
     */
    void SetTimerClass(const int timerClass);

    /**
     * Enables only one timer class and disables all other classes
     *
     * @param[in] timerClass       timer class which should be enabled from now on
     *
     * @return void
     */
    void EnableTimerClass(const int timerClass);
    int mTimerClass;

private:

    /**
     * Stops all running timers
     *
     * @return void
     */
    void CancelAllTimers();

    typedef struct
    {
        timer_t timerId;
        bool    isSMtimer;
        Timer   *self;
        void    *userObject;
        const void 	*userData;
        tCallBackFn pCallBackFn;
        int used;
        unsigned long magic;
    }TimerContext;

    enum {TIMER_MAGIC = 0x63253956};

    std::vector <Timer::TimerContext *> m_contexts;

    struct sigaction mSigAction;

#if SIGACTION_THREAD
    static void *SigactionThread(void *context);
#endif

    static void DefaultCallback(int signo, siginfo_t *info, void *context);
    bool StartTimerInternal(timer_t &timerIndex ,long timeMilliseconds, long intervallMilliseconds, bool isSMtimer,void *pObject,tCallBackFn pCallBackFn, const void *userData);
    Lock m_Lock;
};

#endif //_TIMER_H_

/** @} */
