/**
 * @defgroup ThreadFactory ThreadFactory
 * @author Matthias Thoemel
 *
 * Public interface for the Thread Factory
 * @{
 */

#ifndef _THREADFACTORY_H_
#define _THREADFACTORY_H_

#include <pthread.h>
#include <semaphore.h>
#include <vector>

#define NUMOFSTARTEDTHREADSCONFIGURABLE
#define LOW_PRIO_TASK 19
class TFThread
{
public:
    virtual void Do(int param, void *userPtr) = 0;
};

class ThreadFactory
{

public:

    /**
     * Constructor
     *
     */
    ThreadFactory(unsigned int iNumOfThreadsUsed = mMaxNumOfThreads);

    /**
     * Starts a user function in a thread from the thread factory's thread pool
     *
     * @param[in] thread pointer to the user class with a Do method in it
     * @param[in] param additional parameter to select between functions that should be run
     *                  as threads in the users class
     * @param[in] userContext additional pointer to user supplied data which will be routed to
     *                        the user function as first parameter
     * @return index of triggered thread in thread context
     */
    int Do(TFThread *thread, int param , void *userContext, void *cleanUpHandle = NULL);

    /**
     * This function is used by LocalSPM to create the ThreadFactory.
     * Inform LocalSPM that Create is ready -> CreateDone(0)
     *
     * @return void
     */
    void Create(void);

    /**
     * Interface to create a thread.
     *
     * @param[in] threadIndex index in thread context used to cancel related thread
     *
     * @return < 0: error, = 0: OK
     */
    int CreateThread(const int threadIndex);

    /**
     * Interface to cancel a thread created by the threadfactory.
     *
     * @param[in] threadIndex index in thread context used to cancel related thread
     *
     * @return < 0: error, = 0: OK
     */
    int CancelThread(const int threadIndex);

    /**
     * Interface to kill a thread created by the threadfactory.
     *
     * @param[in] threadIndex index in thread context used to kill related thread
     *
     * @return < 0: error, = 0: OK
     */
    int KillThread(const int threadIndex);

    /**
     * Interface to kill all the threads matching cleanUpHandle.
     *
     * @return < 0: error, = 0: OK
     */
    int CleanUpThreads(void *cleanUpHandle = NULL);

    /**
     * Interface to kill all the threads created by the threadfactory.
     * Should be last point while shutting down the module and should be called from the main thread.
     *
     * @return < 0: error, = 0: OK
     */
    int KillAllThreads();

    /**
     * Interface to set the name of thie current thread. This method hould be called in the threads entry method
     *
     * @param[in] name thread name to set
     *
     * @return none
     */
    static void SetName(const char *name);

    int SendSignal(const int threadIndex, const int sig);
    typedef pthread_t tThreadID;
    static tThreadID GetThreadId();
    static int SetPriority(int prio);
    int GetThreadsRunning();
    int GetMaxThreads() {return mNoOfThreads;};

protected:

    typedef struct {
        void *userContext;
        TFThread *threadFunction;
        int functionID;
        int count;
        int returnValue;
        pthread_mutex_t mtx;
        int stopThread;
        int busy;
        sem_t *freeWorkerThreads;
        pthread_t mThreadHandle;
        pthread_attr_t mThreadAttr;
        void *cleanUpHandle;
    } tThreadContext;

    static void *startFunction(void *context);
    static void ResetName(tThreadContext *ctx);
    static bool JoinTimeoutCallBack(timer_t timerID , void* instance ,const void *userData);

    enum {mMaxNumOfThreads = 30};
    std::vector<tThreadContext>    mThreadContext;
    int               mNoOfThreads;


    pthread_mutex_t mLock;
    sem_t mFreeWorkerThreads;
    // weg pthread_mutex_t mOneWorkerFree;
    bool            mInitialized;



};
/* Thread pool calss to create low priority threads.
 */
class ThreadFactoryLowPrio : public ThreadFactory
{
private :

    int SetPriority(int prio, pthread_t tid);
    static void *startFunction(void *context);

    enum {mMaxNumOfLowPrioThreads = 10};

    // The thread id update with low prio cgroup
    static void StoreThreadId(const pid_t & thList);

public:
    /* Low Prio thread to Slice path
     */
    static const char* mLowprioSlicepath;

    /* Constructor to initialize the number of low priority threads.
     */
    ThreadFactoryLowPrio(unsigned int iNumOfThreadsUsed = mMaxNumOfLowPrioThreads):ThreadFactory(iNumOfThreadsUsed){}
    /**
     * This function is used by LocalSPM to create the ThreadFactory.
     * Inform LocalSPM that Create is ready -> CreateDone(0)
     *
     * @return void
     */
    void Create(const int prio=LOW_PRIO_TASK);

    /**
     * Interface to create a thread.
     *
     * @param[in] threadIndex index in thread context used to cancel related thread
     *
     * @return < 0: error, = 0: OK
     */
    int CreateThread(const int threadIndex, const int prio);
};

#endif //_THREADFACTORY_H_
/** @} */
