/**
 * @defgroup Framework StateMachineFramework
 * @author Matthias Thömel
 *
 * This is the interface of the remote state machine manager tied to the Dispatcher
 *
 * @{
 */

#ifndef _REMOTESTATEMACHINEMANAGER_H_
#define _REMOTESTATEMACHINEMANAGER_H_

#include <stdio.h>
#include <stdlib.h>
#include <dbus/dbus.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <pthread.h>
#include <sys/prctl.h>

#include <Framework_Errorcodes.h>

class RemoteStateMachineManager // : public TFThread
{
public:
    /**
     * Create the general DBus connection and starts the receiver thread
     *
     * @return = MP_NO_ERROR: no error
     * = MP_ERR_DISP_OPEN_DBUS: could not open DBus interface
     * = MP_ERR_DISP_SPAWN_RECV_THREAD: could not spawn Dbus receiver thread
     */
    int Init();

    /**
     * Start the Dbus Main loop (for MediaEngine(Test) only)
     *
     * @return = MP_NO_ERROR: no error
     * = MP_ERR_DISP_SPAWN_RECV_THREAD: could not start the thread
     */
    int StartDbusMainLoop();

    /**
     * Stops the receiver thread and closes the DBus connection
     *
     * @return = MP_NO_ERROR: no error
     */
    int DeInit();

    /**
     * Function transfers a SMF message and a desired answer (optional) to another
     * state machine in another process by using Dbus signal broadcast to all
     * listening processes.
     *
     * @param[in] message pointer to a SMF message text
     * @param[in] answer pointer to a possible desired answer (=NULL: no answer wanted)
     * @return = MP_NO_ERROR: no error
     */
    int SendMessage(const char *message, const char *answer);

    /**
     * Function used by Dbus to call back: it receives all messages the DBus filter
     * is set to and selects if a messages must be consumed here.
     * it calls the Dispatcher SendMessageAnswerLocal to route the incoming messages
     * to the local Dispatcher.
     *
     * @param[in] connection handle of the current DBus connectionj
     * @param[in] pDBusMessage received DBus message
     * @param[in] user_data not used
     * @return != 0: error, = 0: OK
     */
    static DBusHandlerResult DbusSignalReceived (DBusConnection *connection, DBusMessage *pDBusMessage, void *user_data);

    /**
     * This thread runs the Dbus send and receive loop, called by Do function from Thread Factory
     * From here the call back DbusSignalReceived is called.
     * @return void
     */
    static void* DbusDispatcherThread(void *_pConnection);

    /**
     * This function starts a user function in a thread from the thread factory's thread pool.
     * After leaving while loop used thread is stopped and released for other users
     * @param[in] _pConnection Dbus connection handle
     *
     * @return void
     */
    void Do(int functionID, void *ptr);

    static void DbusUnregister (DBusConnection *connection, void *user_data);

    int mStopThread;
    DBusConnection  *mConnection;

private:

    int AppendMessage(DBusMessageIter *args, const char *message);
    static int AssembleMessage(char *message, size_t messageSize, DBusMessageIter *args);

    static void SetName(const char *name){
        prctl(PR_SET_NAME, name, 0, 0, 0, 0);
    }

    tGeneralString mObjectPath;

    // TDOD thm3hi remothe this on using ThreadFactory
    pthread_attr_t m_attr;
    pthread_t m_threadId;
};

#endif /* _REMOTESTATEMACHINEMANAGER_H_ */
