/**
 * @defgroup    Ipc Ipc
 * @ingroup     MediaPlayer
 * @author      Dinesh D
 *
 * Provide IPC interface to communicate between the components which are in a
 * different processes (e.g. PlayerEngine which can be communicated via dbus IPC ).*
 * This is the base class for both client and service interface classes for IPC
 *
 * @{
 */
#ifndef IPCBASE_H_
#define IPCBASE_H_

#include "IpcTypes.h"
#include <dbus/dbus.h>
#include "BaseTypes.h"
#include "Lock.h"
#include <string.h>
#include <string>
using namespace std;

/**
 * Well-known bus types. See dbus_bus_get().
 * the bus types are taken from DBusBusType(which is from file dbus-shared.h).
 * The additional bus type GMPDBUS_BUS_CUSTOMSESSION added to connect to dbus session bus with a different address
 */
typedef enum
{
  GMPDBUS_BUS_SESSION,    /**< The login session bus */
  GMPDBUS_BUS_SYSTEM,     /**< The systemwide bus */
  GMPDBUS_BUS_STARTER,     /**< The bus that started us, if any */
  GMPDBUS_BUS_CUSTOMSESSION //Connects to Custom dbus session 'unix:path=/tmp/shared/iddbus/lxcdbus'
} GMPDBusBusType;
/**
 * IPC Interface class
 */

class IpcBase
{

public:

    /**
     * Constructor
     *
     * @return void
     */
    IpcBase();

    /**
     * Destructor
     *
     * @return void
     */
    virtual ~IpcBase();

    /**
     * This is a function which starts the dbus dispatcher
     */
    int DispatchMessages(void);

    /**
     * Common signal callback function which client can override to .
     *
     * @param[in] name  signal name received from the dbus
     * @param[in] params    params signal parameters combined as expected by the
     *                      MediaPlayer state machine framework
     *                      e.g : "Pause \xff 34,567,test_string \xff "
     */
    virtual tBoolean SignalReceived(const char* name , const char* params);

    /**
     * Asynchronous method answer callback function which client can override to .
     *
     * @param[in] serial reply serial to identify the method call
     * @param[in] params answer parameters
     *
     */
    virtual tBoolean MethodAnswered(dbus_uint32_t serial , const char* params);

    /**
     * Method triggered from the remote client
     *
     * @param[in] serial reply serial to identify the method call
     * @param[in] name method name
     * @param[in] params method parameters
     *
     */
    virtual tBoolean MethodRequested(dbus_uint32_t serial , const char* name , const char* params);

    /**
     * Error reply callback .
     *
     * @param[in] serial    serial no of the method for which this error has occured
     * @param[in] error     error string
     */
    virtual tBoolean Error(dbus_uint32_t serial , const char* errorName,const char* errorMessage);

    /**
     * Adds a match with the filter string to receive message from the dbus
     *
     * @param[in] busType type of bus Session or System to which filter has to be appled
     * @param[in] filter  string filter specifying the interaface name
     *
     * @return = 0: Ok , < 0 : failure
     */
    int SetFilter(GMPDBusBusType busType , const char* filter);

    /**
     * removes the match with the filter string to stop receiving message from the dbus which matches the filter
     *
     * @param[in] busType type of bus Session or System to which match has to be removed
     * @param[in] filter  string filter
     *
     * @return = 0: Ok , < 0 : failure
     */
    int RemoveFilter(GMPDBusBusType busType , const char* filter);

    /**
     * Registers a new dbus service with dbus-daemon on its session bus
     *
     * @param[in] service name of service
     * @param[in] path  serive path string(unused)
     * @param[in] interface serive interface string(unused)
     *
     * @return = 0: Ok , < 0 : failure
     */
    int RegisterService(const char* service ,const char* path ,const char* interface);
    int RegisterObjectPath(GMPDBusBusType busType ,const char* path);

    /*Internal function to parse the message received from the dbus*/
    DBusHandlerResult ParseMessage(DBusMessage *pDBusMessage);

    void ParseMessageWithByteArray(DBusMessage *pDBusMessage,string& params);

    /**
     * Appends the String with Length of the Byte Array(of type DBUS_TYPE_ARRAY) and its pointer
     */
    void AppendByteArray(DBusMessageIter&,string&);

    /**
     *Register Callback Function
     */
    void RegisterCallbackFunction(GMPDBusBusType busType);
    /**
     *DeRegister Callback Function
     */
    void DeRegisterCallbackFunction(GMPDBusBusType busType);
    void ParseMessage_iter(DBusMessageIter *iter, string& params);


protected:
    DBusConnection *mSessionBusConnection ;
    DBusConnection *mSystemBusConnection ;
    //mCustomSessionBusConnection is a DBusConnection which need to connect to separate session bus in which "com.bosch.MediaAgentService" Service is listening
    //details about iddbus can be found here https://inside-docupedia.bosch.com/confluence/display/gen3generic/CM+Containers
    DBusConnection *mCustomSessionBusConnection;
    Lock m_Mutex;
};

#endif /*IPCBASE_H_*/

/** @} */

