#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_mp.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_TEST
#ifdef TARGET_BUILD
#include "trcGenProj/Header/USBControlTest.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_TEST
#endif
#endif

#include "ConfigTest.h"
#include "ConfigTestData.h"
#include "USBControl.h"
#include "FunctionTracer.h"
#include "PlayerEngineIpcIF.h"
#include "LocalSPM.h"
#include "CppUnitDefinitions.h"
#include "Dispatcher.h"
#include "RequestResponseSM.h"
#include "TimeTrace.h"

#include "VideoTagInfo.h"
#define USE_VIDEO_METADATA_READ_PROCESS 1

#include "USBControlTest.h"

#include "taginfo.h"


using namespace TagInfo;

class USBControlTestSM: public RequestResponseSM
{
    int HandleInitRequest()
    {
        mResult = MP_NO_ERROR;
        return 0;
    }

    int HandleSuccessRequest(const char *allParameters)
    {
        /* the request/expect state machine has a general SUCCESS_REQUEST message which is not
        // specialized to a service state machine be called, so all kind of parameters must
        // be processed by hand in this method. */
        ETG_TRACE_USR1(("USBControlTestSM::HandleSuccessRequest allParameters:%s", allParameters));
        return 0;
    }

    int HandleActionError(const me::reason_e reason)
    {
        //ETG_TRACE_ERR(("USBControlTestSM::HandleActionError reason:%u", reason));
        (void)reason;
        mResult = MP_ERR_ACTION_ERROR;
        return 0;
    }

public:
    tResult mResult;
};


void USBControlTest::setUp()
{
    ticks.begin();
}

void USBControlTest::tearDown()
{
    ticks.elapsed();
}


/**********************************************************************
 * Test cases
 *********************************************************************/

void USBControlTest::InitIPC()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    tAudioOutputDevice audioOutputDevice;
    strncpy_r(OUT audioOutputDevice, IN USB_ALSA_DEVICENAME, IN sizeof(audioOutputDevice));

    /* Call StartAllocateAudioOutput */
    ret = LocalSPM::GetDeviceDispatcher().StartAllocateAudioOutput(IN audioOutputDevice);

    sleep(2); //Wait a little bit until IPC is initialized

    // test result
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
}

void USBControlTest::AllocateAudioInput()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    USBControlTestSM objMyRRSM;

    /* Send ALLOCATE_AUDIO_INPUT message to USBControlSM */
    char messageString[64];
    strncpy_r(OUT messageString, IN "USBControlSM::ALLOCATE_AUDIO_INPUT", IN sizeof(messageString));

    ret = objMyRRSM.DoEventAnswer(IN messageString, (char *)NULL, NULL, USB_CTRL_SM_ANSWER_TIMEOUT_MS);

    // test result
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(MP_NO_ERROR == objMyRRSM.mResult);
}

void USBControlTest::SwitchObserver()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    USBControlTestSM objMyRRSM;

    /* Send SWITCH_OBSERVER message to USBControlSM */
    char messageString[64];
    strncpy_r(OUT messageString, IN "USBControlSM::SWITCH_OBSERVER", IN sizeof(messageString));

    ret = objMyRRSM.DoEventAnswer(IN messageString, (char *)NULL, NULL, USB_CTRL_SM_ANSWER_TIMEOUT_MS);

    // test result
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(MP_NO_ERROR == objMyRRSM.mResult);
}

void USBControlTest::StartStreaming()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    USBControlTestSM objMyRRSM;

    /* Send START_STREAMING message to USBControlSM */
    char messageString[64];
    strncpy_r(OUT messageString, IN "USBControlSM::START_STREAMING", IN sizeof(messageString));

    ret = objMyRRSM.DoEventAnswer(IN messageString, (char *)NULL, NULL, USB_CTRL_SM_ANSWER_TIMEOUT_MS);

    // test result
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(MP_NO_ERROR == objMyRRSM.mResult);
}

void USBControlTest::PlayAudio()
{
    ENTRY_TEST

    tURL URL;
#ifndef TARGET_BUILD
    char *pwd = get_current_dir_name();
    strncpy_r(OUT URL, IN pwd, IN sizeof(URL));
    strncat_r(OUT URL, IN "/Customer/Simulation/CustomControl/test/music/Garbage/Garbage-Garbage-11-Fix_Me_Now.mp3", IN sizeof(URL));
    free(pwd);
#else
    strncpy_r(OUT URL, IN "/opt/bosch/test/data/GMP/test/music/Garbage/Garbage-Garbage-11-Fix_Me_Now.mp3", IN sizeof(URL));
#endif

    tObjectID objectID = 4711;
    tPlaytime position = 75000; //1:15min

    Play(IN URL, IN objectID, IN position);
}

void USBControlTest::PlayWithUmlauts()
{
    ENTRY_TEST

    tURL URL;
#ifndef TARGET_BUILD
    char *pwd = get_current_dir_name();
    strncpy_r(OUT URL, IN pwd, IN sizeof(URL));
    strncat_r(OUT URL, IN "/Customer/Simulation/CustomControl/test/music/Grönemeyer_-_Anders/Herbert_Grönemeyer_-_Bleibt_Alles_Anders.mp3", IN sizeof(URL));
    free(pwd);
#else
    strncpy_r(OUT URL, IN "/opt/bosch/test/data/GMP/test/music/Grönemeyer_-_Anders/Herbert_Grönemeyer_-_Bleibt_Alles_Anders.mp3", IN sizeof(URL));
#endif

    tObjectID objectID = 4712;
    tPlaytime position = 75000; //1:15min

    Play(IN URL, IN objectID, IN position);
}

void USBControlTest::PlayVideo()
{
    ENTRY_TEST

    tURL URL;
#ifndef TARGET_BUILD
    char *pwd = get_current_dir_name();
    strncpy_r(OUT URL, IN pwd, IN sizeof(URL));
    strncat_r(OUT URL, IN "/Customer/Simulation/CustomControl/test/video/01_SIF_MediaClip_01.mp4", IN sizeof(URL));
    free(pwd);
#else
    strncpy_r(OUT URL, IN "/opt/bosch/test/data/GMP/test/video/01_SIF_MediaClip_01.mp4", IN sizeof(URL));
#endif

    tObjectID objectID = 9911;
    tPlaytime position = 0;

    Play(IN URL, IN objectID, IN position);
}

void USBControlTest::Play(const tURL URL, const tObjectID objectID, const tPlaytime position)
{
    ENTRY_INTERNAL

    tResult ret = MP_NO_ERROR;

    tDeviceType deviceType = DTY_USB;
    tDeviceID deviceID = 2;
    tMountPoint mountPoint = {0};
    tUUID uuid = {0};
    tStreaming streaming = false;

    USBControlTestSM objMyRRSM;

    /* Send PLAY message to USBControlSM */
    char messageString[256];
    strncpy_r(OUT messageString, IN "USBControlSM::PLAY", IN sizeof(messageString));
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    ret = LocalSPM::GetUSBControl().ParameterPLAY(OUT parameterString, IN size,
            IN deviceType, IN deviceID, IN URL, IN mountPoint, IN uuid, IN (tPEHandle)objectID, IN position, IN streaming);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    ret = objMyRRSM.DoEventAnswer(IN messageString, IN parameterString, NULL, USB_CTRL_SM_ANSWER_TIMEOUT_MS);

    // test result
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(MP_NO_ERROR == objMyRRSM.mResult);

    sleep(5); //play at least 5s
}

void USBControlTest::Buffer()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    tDeviceType deviceType = DTY_USB;
    tDeviceID deviceID = 2;
    tURL URL;
    tMountPoint mountPoint = {0};
    tObjectID objectID = 4713;

#ifndef TARGET_BUILD
    char *pwd = get_current_dir_name();
    strncpy_r(OUT URL, IN pwd, IN sizeof(URL));
    strncat_r(OUT URL, IN "/Customer/Simulation/CustomControl/test/music/Garbage/Garbage-Garbage-12-Milk.mp3", IN sizeof(URL));
    free(pwd);
#else
    strncpy_r(OUT URL, IN "/opt/bosch/test/data/GMP/test/music/Garbage/Garbage-Garbage-12-Milk.mp3", IN sizeof(URL));
#endif

    USBControlTestSM objMyRRSM;

    /* Send BUFFER message to USBControlSM */
    char messageString[256];
    strncpy_r(OUT messageString, IN "USBControlSM::BUFFER", IN sizeof(messageString));
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    ret = LocalSPM::GetUSBControl().ParameterBUFFER(OUT parameterString, IN size,
            IN deviceType, IN deviceID, IN URL, IN mountPoint, IN (tPEHandle)objectID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    ret = objMyRRSM.DoEventAnswer(IN messageString, IN parameterString, NULL, USB_CTRL_SM_ANSWER_TIMEOUT_MS);

    // test result
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(MP_NO_ERROR == objMyRRSM.mResult);
}

void USBControlTest::Pause()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    tDeviceType deviceType = DTY_USB;
    tDeviceID deviceID = 2;
    tURL URL;
    tMountPoint mountPoint = {0};

#ifndef TARGET_BUILD
    char *pwd = get_current_dir_name();
    strncpy_r(OUT URL, IN pwd, IN sizeof(URL));
    strncat_r(OUT URL, IN "/Customer/Simulation/CustomControl/test/music/Garbage/Garbage-Garbage-11-Fix_Me_Now.mp3", IN sizeof(URL));
    free(pwd);
#else
    strncpy_r(OUT URL, IN "/opt/bosch/test/data/GMP/test/music/Garbage/Garbage-Garbage-11-Fix_Me_Now.mp3", IN sizeof(URL));
#endif

    USBControlTestSM objMyRRSM;

    /* Send PAUSE message to USBControlSM */
    char messageString[256];
    strncpy_r(OUT messageString, IN "USBControlSM::PAUSE", IN sizeof(messageString));
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    ret = LocalSPM::GetUSBControl().ParameterPAUSE(OUT parameterString, IN size,
            IN deviceType, IN deviceID, IN URL, IN mountPoint);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    ret = objMyRRSM.DoEventAnswer(IN messageString, IN parameterString, NULL, USB_CTRL_SM_ANSWER_TIMEOUT_MS);

    // test result
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(MP_NO_ERROR == objMyRRSM.mResult);

    sleep(2); //pause at least 2s
}

void USBControlTest::Stop()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    tDeviceType deviceType = DTY_USB;
    tDeviceID deviceID = 2;
    tURL URL = {0};
    tMountPoint mountPoint = {0};

    USBControlTestSM objMyRRSM;

    /* Send STOP message to USBControlSM */
    char messageString[256];
    strncpy_r(OUT messageString, IN "USBControlSM::STOP", IN sizeof(messageString));
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    ret = LocalSPM::GetUSBControl().ParameterSTOP(OUT parameterString, IN size,
            IN deviceType, IN deviceID, IN URL, IN mountPoint);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    ret = objMyRRSM.DoEventAnswer(IN messageString, IN parameterString, NULL, USB_CTRL_SM_ANSWER_TIMEOUT_MS);

    // test result
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(MP_NO_ERROR == objMyRRSM.mResult);
}

void USBControlTest::Resume()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    tDeviceType deviceType = DTY_USB;
    tDeviceID deviceID = 2;
    tURL URL = {0};
    tMountPoint mountPoint = {0};

    USBControlTestSM objMyRRSM;

    /* Send RESUME message to USBControlSM */
    char messageString[256];
    strncpy_r(OUT messageString, IN "USBControlSM::RESUME", IN sizeof(messageString));
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    ret = LocalSPM::GetUSBControl().ParameterRESUME(OUT parameterString, IN size,
            IN deviceType, IN deviceID, IN URL, IN mountPoint);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    ret = objMyRRSM.DoEventAnswer(IN messageString, IN parameterString, NULL, USB_CTRL_SM_ANSWER_TIMEOUT_MS);

    // test result
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(MP_NO_ERROR == objMyRRSM.mResult);

    sleep(5); //play at least 5s
}

void USBControlTest::SeekTo()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    tDeviceType deviceType = DTY_USB;
    tDeviceID deviceID = 2;
    tURL URL = {0};
    tMountPoint mountPoint = {0};
    tPlaytime position = 105000; //1:45min

    USBControlTestSM objMyRRSM;

    /* Send SEEK_TO message to USBControlSM */
    char messageString[256];
    strncpy_r(OUT messageString, IN "USBControlSM::SEEK_TO", IN sizeof(messageString));
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    ret = LocalSPM::GetUSBControl().ParameterSEEK_TO(OUT parameterString, IN size,
            IN deviceType, IN deviceID, IN URL, IN mountPoint, IN position);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    ret = objMyRRSM.DoEventAnswer(IN messageString, IN parameterString, NULL, USB_CTRL_SM_ANSWER_TIMEOUT_MS);

    // test result
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(MP_NO_ERROR == objMyRRSM.mResult);

    sleep(3); // to hear something
}

void USBControlTest::FfwdStartStop()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    tDeviceType deviceType = DTY_USB;
    tDeviceID deviceID = 2;
    tURL URL = {0};
    tMountPoint mountPoint = {0};
    tCueingRate rate = 10; //after every second jump 10s

    USBControlTestSM objMyRRSM;

    /* Send FFWD message to USBControlSM */
    char messageString[256];
    strncpy_r(OUT messageString, IN "USBControlSM::FFWD", IN sizeof(messageString));
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);
    speedstate_e IsPlaybackSpeed = ME_SPEEDSTATE_OFF;
    ret = LocalSPM::GetUSBControl().ParameterFFWD(OUT parameterString, IN size,
            IN deviceType, IN deviceID, IN URL, IN mountPoint, IN rate, IN IsPlaybackSpeed);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    ret = objMyRRSM.DoEventAnswer(IN messageString, IN parameterString, NULL, USB_CTRL_SM_ANSWER_TIMEOUT_MS);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(MP_NO_ERROR == objMyRRSM.mResult);

    sleep(3); //Do Ffwd for 3s

    /* Send FFWD_STOP message to USBControlSM */
    strncpy_r(OUT messageString, IN "USBControlSM::FFWD_STOP", IN sizeof(messageString));

    ret = LocalSPM::GetUSBControl().ParameterFFWD_STOP(OUT parameterString, IN size,
            IN deviceType, IN deviceID, IN URL, IN mountPoint,IN IsPlaybackSpeed);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    ret = objMyRRSM.DoEventAnswer(IN messageString, IN parameterString, NULL, USB_CTRL_SM_ANSWER_TIMEOUT_MS);

    // test result
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(MP_NO_ERROR == objMyRRSM.mResult);
}

void USBControlTest::FrevStartStop()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    tDeviceType deviceType = DTY_USB;
    tDeviceID deviceID = 2;
    tURL URL = {0};
    tMountPoint mountPoint = {0};
    tCueingRate rate = 10; //after every second jump 10s

    USBControlTestSM objMyRRSM;

    /* Send FREV message to USBControlSM */
    char messageString[256];
    strncpy_r(OUT messageString, IN "USBControlSM::FREV", IN sizeof(messageString));
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);
    speedstate_e IsPlaybackSpeed = ME_SPEEDSTATE_OFF;

    ret = LocalSPM::GetUSBControl().ParameterFREV(OUT parameterString, IN size,
            IN deviceType, IN deviceID, IN URL, IN mountPoint, IN rate, IN IsPlaybackSpeed);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    ret = objMyRRSM.DoEventAnswer(IN messageString, IN parameterString, NULL, USB_CTRL_SM_ANSWER_TIMEOUT_MS);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(MP_NO_ERROR == objMyRRSM.mResult);

    sleep(3); //Do Frev for 3s

    /* Send FREV_STOP message to USBControlSM */
    strncpy_r(OUT messageString, IN "USBControlSM::FREV_STOP", IN sizeof(messageString));

    ret = LocalSPM::GetUSBControl().ParameterFREV_STOP(OUT parameterString, IN size,
            IN deviceType, IN deviceID, IN URL, IN mountPoint, IN IsPlaybackSpeed);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    ret = objMyRRSM.DoEventAnswer(IN messageString, IN parameterString, NULL, USB_CTRL_SM_ANSWER_TIMEOUT_MS);

    // test result
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(MP_NO_ERROR == objMyRRSM.mResult);
}

void USBControlTest::InitDeviceConnection()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    tMountPoint mountPoint = {0};
    tDeviceID deviceID = 0;

    tDeviceInfo deviceInfo;
    ret = LocalSPM::GetDBManager().GetDevice(OUT deviceID, IN DTY_USB, IN "USB");
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    ret = LocalSPM::GetDBManager().GetDeviceInfo(OUT deviceInfo, deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == 0);

    /* Send INIT_DEVICE_CONNECTION message to USBControlSM */
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    tDeviceInfoString deviceInfoString;
    DataProvider::MarshalDeviceInfo(deviceInfoString, deviceInfo);

    ret = LocalSPM::GetUSBControl().ParameterINIT_DEVICE_CONNECTION(OUT parameterString,
        IN size, IN deviceInfoString);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    ret = LocalSPM::GetUSBControl().SendEventByName("INIT_DEVICE_CONNECTION", IN parameterString);

    sleep(1);

    // test result
    CPPUNIT_ASSERT(ret == 0);
}

void USBControlTest::RemoveDeviceConnection()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    tMountPoint mountPoint = {0};
    tDeviceID deviceID = 2;

    /* Send REMOVE_DEVICE_CONNECTION message to USBControlSM */
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    ret = LocalSPM::GetUSBControl().ParameterREMOVE_DEVICE_CONNECTION(OUT parameterString,
        IN size, IN mountPoint, IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    ret = LocalSPM::GetUSBControl().SendEventByName("REMOVE_DEVICE_CONNECTION", IN parameterString);

    sleep(1);

    // test result
    CPPUNIT_ASSERT(ret == 0);
}

void USBControlTest::GetFingerprint()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    tMountPoint mountPoint;
    tDeviceID deviceID = 2;
    tFingerprint fingerprintShortcut = FINGERPRINT_SHORTCUT;
    tMemorySize usedMemory = 31249984;
    tFingerprint lastFingerprint = {0};
    sprintf(lastFingerprint, "%s%u", fingerprintShortcut, usedMemory);

#ifndef TARGET_BUILD
    char *pwd = get_current_dir_name();
    strncpy_r(OUT mountPoint, IN pwd, IN sizeof(mountPoint));
    strncat_r(OUT mountPoint, IN "/Customer/Simulation/CustomControl/test", IN sizeof(mountPoint));
    free(pwd);
#else
    strncpy_r(OUT mountPoint, IN "/opt/bosch/test/data/GMP/test", IN sizeof(mountPoint));
#endif

    USBControlTestSM objMyRRSM;

    /* Send GET_FINGERPRINT message to USBControlSM */
    char messageString[256];
    strncpy_r(OUT messageString, IN "USBControlSM::GET_FINGERPRINT", IN sizeof(messageString));
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    ret = LocalSPM::GetUSBControl().ParameterGET_FINGERPRINT(OUT parameterString, IN size, IN mountPoint, IN deviceID, IN lastFingerprint);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    ret = objMyRRSM.DoEventAnswer(IN messageString, IN parameterString, NULL, USB_CTRL_SM_ANSWER_TIMEOUT_MS);

    // test result
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(MP_NO_ERROR == objMyRRSM.mResult);
}

void USBControlTest::GetMetadata()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    tMountPoint mountPoint;
    tReadPosition readPosition = {0};
    tDeviceID deviceID = 2;

#ifndef TARGET_BUILD
    char *pwd = get_current_dir_name();
    strncpy_r(OUT mountPoint, IN pwd, IN sizeof(mountPoint));
    strncat_r(OUT mountPoint, IN "/Customer/Simulation/CustomControl/test", IN sizeof(mountPoint));
    free(pwd);
#else
    strncpy_r(OUT mountPoint, IN "/opt/bosch/test/data/GMP/test", IN sizeof(mountPoint));
#endif

    USBControlTestSM objMyRRSM;

    /* Send GET_METADATA message to USBControlSM */
    char messageString[256];
    strncpy_r(OUT messageString, IN "USBControlSM::GET_METADATA", IN sizeof(messageString));
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    ret = LocalSPM::GetUSBControl().ParameterGET_METADATA(OUT parameterString, IN size, IN mountPoint, IN readPosition, IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    ret = objMyRRSM.DoEventAnswer(IN messageString, IN parameterString, NULL, USB_CTRL_SM_ANSWER_TIMEOUT_MS);

    // test result
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(MP_NO_ERROR == objMyRRSM.mResult);
}


/* use a request / response state machine for verifying the received AlbumArt data object */
class AlbumArtRR : public RequestResponseSM
{
    int HandleInitRequest()
    {
        mAlbumArtObjectPointer = NULL;
        result = MP_NO_ERROR;
        return 0;
    }

    int HandleLoopBackMessage(const char *allParameters)
    {
        (void)allParameters;
        HandleSuccessRequest(NULL);
        return 0;
    }

    int HandleSuccessRequest(const char *allParameters)
    {
        if (!allParameters || strlen_r(allParameters) == 0)
        {
            /* set an error */
            result = MP_ERR_DP_NO_ALBUM_ART_AVAIL;
        }
        /* read out the pointer to the album art object */
        UnMarshal(IN allParameters, "p", OUT &mAlbumArtObjectPointer);

        /* no album art object? */
        if (!mAlbumArtObjectPointer)
        {
            /* set an error */
            result = MP_ERR_DP_NO_ALBUM_ART_AVAIL;
        }
        return 0;
    }
    public:
    tAlbumArtObjectPtr mAlbumArtObjectPointer;
    tResult result;
};


#define USE_REPLACE_TESTCONTENT
#ifndef TARGET_BUILD
#define ALBUM_ART_TEST_URL_1    (char*)"/Customer/Simulation/CustomControl/stick_2/Supertramp/Supertramp_-_Singles_-_00_-_Its_Raining_Again.mp3"
#define ALBUM_ART_TEST_URL_2    (char*)"/Customer/Simulation/CustomControl/stick_2/Unheilig/Grosse_Freiheit/Unheilig_-_Grosse_Freiheit_-_16_-_Neuland.mp3"
#define ALBUM_ART_TEST_URL_3    (char*)"/Customer/Simulation/CustomControl/test/music/Garbage/Garbage-Garbage-06-A_Stroke_of_Luck.mp3"
#define ALBUM_ART_TEST_URL_4    (char*)"/Customer/Simulation/CustomControl/test/Garbage-Garbage-03-Only_Happy_When_It_Rains.MP3"
#define ALBUM_ART_TEST_URL_5    (char*)"/Customer/Simulation/CustomControl/test/music/Grönemeyer_-_Anders/Herbert_Grönemeyer_-_Nach_Mir.mp3"
#define ALBUM_ART_TEST_URL_6    (char*)"/Customer/Simulation/CustomControl/stick_3/01_WAV_Speedway.wav"
#define ALBUM_ART_TEST_URL_7    (char*)"/Customer/Simulation/CustomControl/stick_3/has-tags.m4a"
#define ALBUM_ART_TEST_URL_8    (char*)"/Customer/Simulation/CustomControl/stick_3/WMA_with_02_Pictures_embedded.wma"
#else
#define ALBUM_ART_TEST_URL_1    (char*)"/opt/bosch/test/data/GMP/stick_2/Supertramp/Supertramp_-_Singles_-_00_-_Its_Raining_Again.mp3"
#define ALBUM_ART_TEST_URL_2    (char*)"/opt/bosch/test/data/GMP/stick_2/Unheilig/Grosse_Freiheit/Unheilig_-_Grosse_Freiheit_-_16_-_Neuland.mp3"
#define ALBUM_ART_TEST_URL_3    (char*)"/opt/bosch/test/data/GMP/test/music/Garbage/Garbage-Garbage-06-A_Stroke_of_Luck.mp3"
#define ALBUM_ART_TEST_URL_4    (char*)"/opt/bosch/test/data/GMP/test/Garbage-Garbage-03-Only_Happy_When_It_Rains.MP3"
#define ALBUM_ART_TEST_URL_5    (char*)"/opt/bosch/test/data/GMP/test/music/Grönemeyer_-_Anders/Herbert_Grönemeyer_-_Nach_Mir.mp3"
#define ALBUM_ART_TEST_URL_6    (char*)"/opt/bosch/test/data/GMP/stick_3/01_WAV_Speedway.wav"
#define ALBUM_ART_TEST_URL_7    (char*)"/opt/bosch/test/data/GMP/stick_3/has-tags.m4a"
#define ALBUM_ART_TEST_URL_8    (char*)"/opt/bosch/test/data/GMP/stick_3/WMA_with_02_Pictures_embedded.wma"
#endif

void USBControlTest::GetAlbumArt()
{
    ENTRY_TEST

#ifdef USE_REPLACE_TESTCONTENT
    GetAlbumArt(ALBUM_ART_TEST_URL_1, MMT_PNG, 12089, true,  (char*)"/tmp/Art-MP3-Its_Raining_Again.png");//album art available
    GetAlbumArt(ALBUM_ART_TEST_URL_2, MMT_PNG, 12089, true,  (char*)"/tmp/Art-MP3-Neuland.png");          //album art available
    GetAlbumArt(ALBUM_ART_TEST_URL_3, MMT_JPG, 11993, true,  (char*)"/tmp/Art-MP3-A_Stroke_of_Luck.jpg"); //album art available
    GetAlbumArt(ALBUM_ART_TEST_URL_4, MMT_PNG,     0, false, (char*)"/tmp/Art-none");                     //album art NOT available
    GetAlbumArt(ALBUM_ART_TEST_URL_5, MMT_JPG, 68296, true,  (char*)"/tmp/Art-MP3-Nach_Mir.jpg");         //album art is available as file not as embedded inside mp3 file
#else
    GetAlbumArt(ALBUM_ART_TEST_URL_1, MMT_JPG, 20774, true,  (char*)"/tmp/Art-MP3-Its_Raining_Again.jpg");//album art available
    GetAlbumArt(ALBUM_ART_TEST_URL_2, MMT_JPG, 39644, true,  (char*)"/tmp/Art-MP3-Neuland.jpg");          //album art available
    GetAlbumArt(ALBUM_ART_TEST_URL_3, MMT_JPG, 11993, true,  (char*)"/tmp/Art-MP3-A_Stroke_of_Luck.jpg"); //album art available
    GetAlbumArt(ALBUM_ART_TEST_URL_4, MMT_PNG,     0, false, (char*)"/tmp/Art-none");                     //album art NOT available
    GetAlbumArt(ALBUM_ART_TEST_URL_5, MMT_JPG, 68296, true,  (char*)"/tmp/Art-MP3-Nach_Mir.jpg");         //album art is available as file not as embedded inside mp3 file
#endif

#ifndef TARGET_BUILD
    GetAlbumArt(ALBUM_ART_TEST_URL_7, MMT_PNG,     79, true,  (char*)"/tmp/Art-m4a.png");          //m4a

    //enable when we are using taglib version 1.8/TagInfo which has direct APIs for these formats or MediaInfo supports these formats
#if 0
    GetAlbumArt(ALBUM_ART_TEST_URL_6, MMT_JPG,   7749, true,  (char*)"/tmp/Art-WAV_Speedway.png"); //Wave
    GetAlbumArt(ALBUM_ART_TEST_URL_8, MMT_JPG,   9309, true,  (char*)"/tmp/Art-wma.jpg");          //wma
#endif
#endif
}

void USBControlTest::GetAlbumArt(tAlbumArt albumArt, tMimeType expectedMimeType, tImageSize expectedImageSize, tBool artAvailable, char* imageFileName)
{
    ENTRY_INTERNAL

    tResult ret = MP_NO_ERROR;
    AlbumArtRR objMyRRSM;
    tAlbumArt albumArtPath;

    ETG_TRACE_USR3(("Get album art test for url : %s",albumArt));

#ifndef TARGET_BUILD
    char *pwd = get_current_dir_name();
    strncpy_r(OUT (char *)albumArtPath, IN (const char *)pwd, IN sizeof(tAlbumArt));
    strncat_r(OUT (char *)albumArtPath, IN albumArt, IN sizeof(tAlbumArt));
    free(pwd);
#else
    strncpy_r(OUT (char *)albumArtPath, IN albumArt, IN sizeof(tAlbumArt));
#endif

    /* Frame GET_ALBUM_ART message to USBControlSM */
    char messageString[256];
    strncpy_r(OUT messageString, IN "USBControlSM::GET_ALBUM_ART", IN sizeof(messageString));

    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    ret = LocalSPM::GetUSBControl().ParameterGET_ALBUM_ART(OUT parameterString, IN size,IN albumArtPath);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    ret = objMyRRSM.DoEventAnswer(IN messageString, IN parameterString, NULL, USB_CTRL_SM_ANSWER_TIMEOUT_MS);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    if(artAvailable)        // expecting that file contains a valid album art , then verify against albumart attributes like size and type
    {
        CPPUNIT_ASSERT(MP_NO_ERROR == objMyRRSM.result);
        ETG_TRACE_USR3(("Albumart found with size:mime->(%d,%d) for url->%s\n",objMyRRSM.mAlbumArtObjectPointer->imageSize,objMyRRSM.mAlbumArtObjectPointer->mimeType,albumArt));
        ETG_TRACE_USR3(("Albumart expected   size:mime->(%d,%d) for url->%s\n",expectedImageSize,expectedMimeType,albumArt));

        CPPUNIT_ASSERT(objMyRRSM.mAlbumArtObjectPointer->mimeType == expectedMimeType);
        CPPUNIT_ASSERT(objMyRRSM.mAlbumArtObjectPointer->imageSize == expectedImageSize);

#ifndef TARGET_BUILD
        FILE *imageFile;
        imageFile = fopen(imageFileName,"wb");
        fwrite(objMyRRSM.mAlbumArtObjectPointer->imageData,1,objMyRRSM.mAlbumArtObjectPointer->imageSize, imageFile);
        fclose(imageFile);
#else
        (void)imageFileName;
#endif
        free(objMyRRSM.mAlbumArtObjectPointer->imageData);
        delete objMyRRSM.mAlbumArtObjectPointer;
    }
    else        // expecting that given file does not contain any valid album art .
    {
        CPPUNIT_ASSERT(MP_NO_ERROR != objMyRRSM.result);
        ETG_TRACE_USR3(("Expected result: %d, Album art not available for file: %s", objMyRRSM.result, albumArt));
    }
}

void USBControlTest::Umount()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    tDeviceID deviceID = 0;
    tMountPoint mountPoint = "/media/DUMMY_DUMMY";

    USBControlTestSM objMyRRSM;

    /* Send UMOUNT message to USBControlSM */
    char messageString[256];
    strncpy_r(OUT messageString, IN "USBControlSM::UMOUNT", IN sizeof(messageString));
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    ret = LocalSPM::GetUSBControl().ParameterUMOUNT(OUT parameterString, IN size,
            IN deviceID, IN mountPoint);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    ret = objMyRRSM.DoEventAnswer(IN messageString, IN parameterString, NULL, USB_CTRL_SM_ANSWER_TIMEOUT_MS);

    // test result
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(MP_NO_ERROR == objMyRRSM.mResult);
}

void USBControlTest::ExtractVideoTags()
{
    ENTRY

    tResult res;
    tDeviceID deviceID;
    tListSize listSize;
    tListID listID;
    tNumberOfMediaObjects numberOfMediaObjects;
    tNumberOfPlayLists numberOfPlayLists;
    tNumberOfFolders numberOfFolders;
    tPath path;

    /* set device id */
    res = LocalSPM::GetDBManager().GetDevice(OUT deviceID, IN DTY_USB, IN "USB");
    CPPUNIT_ASSERT(res == 0);

    /* create a file list from a specific directory */
    strncpy_r(OUT path, IN "/video", IN sizeof(path));
    res = LocalSPM::GetListControl().CreateMediaPlayerFileList(
                OUT listID,
                OUT listSize,
                OUT numberOfMediaObjects,
                OUT numberOfPlayLists,
                OUT numberOfFolders,
                IN LTY_FILELIST,
                IN path,
                IN deviceID);
    CPPUNIT_ASSERT(res == 0);

    /* get the list slice */
    vector<tMediaObject> mediaObjectVector;
    res= LocalSPM::GetListControl().RequestMediaPlayerIndexedListSlice(OUT mediaObjectVector, IN listID, 0, listSize);
    CPPUNIT_ASSERT(res == 0);

    for(unsigned int iter = 0 ; iter < mediaObjectVector.size() ; iter++)
    {
        tURL URL;
        strncpy_r(OUT URL, IN mediaObjectVector[iter].mountPoint, IN sizeof(URL));
        strncat_r(OUT URL, IN mediaObjectVector[iter].fileName, IN sizeof(URL));

        ETG_TRACE_USR1(("USBControlTest::ExtractVideoMetaData for file -> :%s",URL));
        TimeTrace ticks("ExtractVideoTags");

#if USE_VIDEO_METADATA_READ_PROCESS
        string target((char *)URL);
        VideoTagInfo * info = new VideoTagInfo(URL);
#else
        string target((char *)URL);
        Info * info = Info::create_tag_info(target);
#endif

        if(info )
        {
            if(info->read())
            {
                ETG_TRACE_USR1(("USBControlTest::ExtractVideoMetaData => Title              :%s",info->get_title().toCString(TRUE)));
                ETG_TRACE_USR1(("USBControlTest::ExtractVideoMetaData => Video Codec        :%s",info->get_videocodec().toCString(TRUE)));
                //ETG_TRACE_USR1(("USBControlTest::ExtractVideoMetaData => Audio Codec        :%s",info->get_audiocodec().toCString(TRUE)));
                //ETG_TRACE_USR1(("USBControlTest::ExtractVideoMetaData => Container format   :%s",info->get_containerformat().toCString(TRUE)));
                ETG_TRACE_USR1(("USBControlTest::ExtractVideoMetaData => Duration(nanosec)  :%lld",info->get_duration()));
                //ETG_TRACE_USR1(("USBControlTest::ExtractVideoMetaData => episode            :%u", info->get_episode()));
                ETG_TRACE_USR1(("USBControlTest::ExtractVideoMetaData => bitrate            :%u", info->get_bitrate()));
                ETG_TRACE_USR1(("USBControlTest::ExtractVideoMetaData => tracknumber        :%u", info->get_tracknumber()));
                //ETG_TRACE_USR1(("USBControlTest::ExtractVideoMetaData => composer           :%s", info->get_composer().toCString(TRUE)));
                //ETG_TRACE_USR1(("USBControlTest::ExtractVideoMetaData => artist             :%s", info->get_artist().toCString(TRUE)));
                //ETG_TRACE_USR1(("USBControlTest::ExtractVideoMetaData => album              :%s", info->get_album().toCString(TRUE)));
                //ETG_TRACE_USR1(("USBControlTest::ExtractVideoMetaData => genre              :%s", info->get_genre().toCString(TRUE)));
                //ETG_TRACE_USR1(("USBControlTest::ExtractVideoMetaData => language code      :%s", info->get_languagecode().toCString(TRUE)));
                //ETG_TRACE_USR1(("USBControlTest::ExtractVideoMetaData => copyrigths         :%s", info->get_copyrights().toCString(TRUE)));
                //ETG_TRACE_USR1(("USBControlTest::ExtractVideoMetaData => encoder            :%s", info->get_encoder().toCString(TRUE)));
                ETG_TRACE_USR1(("USBControlTest::ExtractVideoMetaData => width              :%u", info->get_width()));
                ETG_TRACE_USR1(("USBControlTest::ExtractVideoMetaData => height             :%u", info->get_height()));
                //ETG_TRACE_USR1(("USBControlTest::ExtractVideoMetaData => mpegversion        :%u", info->get_mpegversion()));
                ETG_TRACE_USR1(("USBControlTest::ExtractVideoMetaData => profile            :%s", info->get_profile().toCString(TRUE)));
                ETG_TRACE_USR1(("USBControlTest::ExtractVideoMetaData => level              :%f", info->get_level()));
                ETG_TRACE_USR1(("USBControlTest::ExtractVideoMetaData => framerate          :%u", info->get_framerate()));
            }
            delete info;
        }
        else
        {
            ETG_TRACE_ERR(("USBControl::ExtractVideoMetaData -> Retrieving metadata using libtaginfo failed for URL (%s)",URL));
        }

        ticks.elapsed();
    }

}

void USBControlTest::ExtractAudioTags_ID3V1()
{
    ENTRY

    tResult res;
    tDeviceID deviceID;
    tListSize listSize;
    tListID listID;
    tNumberOfMediaObjects numberOfMediaObjects;
    tNumberOfPlayLists numberOfPlayLists;
    tNumberOfFolders numberOfFolders;
    tPath path;

    /* set device id */
    res = LocalSPM::GetDBManager().GetDevice(OUT deviceID, DTY_USB,"USB-2");
    CPPUNIT_ASSERT(res == 0);

    /* create a file list from a specific directory */
   strncpy_r(path, "/ID3V1", sizeof(path));
   res = LocalSPM::GetListControl().CreateMediaPlayerFileList(
                OUT listID,
                OUT listSize,
                OUT numberOfMediaObjects,
                OUT numberOfPlayLists,
                OUT numberOfFolders,
                IN LTY_FILELIST,
                IN path,
                IN deviceID);
    CPPUNIT_ASSERT(res == 0);

    /* get the list slice */
    vector<tMediaObject> mediaObjectVector;
    res= LocalSPM::GetListControl().RequestMediaPlayerIndexedListSlice(OUT mediaObjectVector, IN listID, 0, listSize);
    CPPUNIT_ASSERT(res == 0);

    for(unsigned int iter = 0 ; iter < mediaObjectVector.size() ; iter++)
    {
        tURL URL;
        strncpy_r(URL, mediaObjectVector[iter].mountPoint, sizeof(URL));
        strncat_r(URL, mediaObjectVector[iter].fileName, sizeof(URL));

        ETG_TRACE_USR1(("USBControlTest::ExtractAudioTags_ID3V1 for file -> :%s",URL));
        TimeTrace ticks("ExtractAudioTags_ID3V1");

        string target((char *)URL);
        Info * info = Info::create_tag_info(target);

        if(info )
        {
            if(info->read())
            {
                ETG_TRACE_USR1(("USBControlTest::ExtractAudioTags_ID3V1 => Title              :%s",info->get_title().toCString(true)));
                ETG_TRACE_USR1(("USBControlTest::ExtractAudioTags_ID3V1 => artist             :%s", info->get_artist().toCString(true)));
                ETG_TRACE_USR1(("USBControlTest::ExtractAudioTags_ID3V1 => album              :%s", info->get_album().toCString(true)));
                ETG_TRACE_USR1(("USBControlTest::ExtractAudioTags_ID3V1 => genre              :%s", info->get_genre().toCString(true)));

                ticks.elapsed();

                /*Validate*/
                if(!strcmp(mediaObjectVector[iter].fileName,"/ID3V1/2Artist2Title1Genre.mp3"))
                {
                    // Genre is from ID3V1
                    CPPUNIT_ASSERT(!strcmp(info->get_artist().toCString(TRUE),"Artist ID3V2"));
                    CPPUNIT_ASSERT(!strcmp(info->get_title().toCString(TRUE),"Title ID3V2"));
                    CPPUNIT_ASSERT(!strcmp(info->get_genre().toCString(TRUE),"Sonata"));
                }
                else if(!strcmp(mediaObjectVector[iter].fileName,"/ID3V1/2Album1Artist2Genre.mp3"))
                {
                    // Artist is from ID3V1
                    CPPUNIT_ASSERT(!strcmp(info->get_album().toCString(TRUE),"Album ID3V2"));
                    CPPUNIT_ASSERT(!strcmp(info->get_artist().toCString(TRUE),"Artist ID3V1"));
                    CPPUNIT_ASSERT(!strcmp(info->get_genre().toCString(TRUE),"Genre ID3V2"));
                }
                else if(!strcmp(mediaObjectVector[iter].fileName,"/ID3V1/2Album2Artist1Title2Genre.mp3"))
                {
                    // Title is from ID3V1
                    CPPUNIT_ASSERT(!strcmp(info->get_album().toCString(TRUE),"Album ID3V2"));
                    CPPUNIT_ASSERT(!strcmp(info->get_artist().toCString(TRUE),"Artist ID3V2"));
                    CPPUNIT_ASSERT(!strcmp(info->get_title().toCString(TRUE),"Title ID3V1"));
                    CPPUNIT_ASSERT(!strcmp(info->get_genre().toCString(TRUE),"Genre ID3V2"));
                }
                else if(!strcmp(mediaObjectVector[iter].fileName,"/ID3V1/1Album2Artist2Title2Genre.mp3"))
                {
                    // Album is from ID3V1
                    CPPUNIT_ASSERT(!strcmp(info->get_album().toCString(TRUE),"Album ID3V1"));
                    CPPUNIT_ASSERT(!strcmp(info->get_artist().toCString(TRUE),"Artist ID3V2"));
                    CPPUNIT_ASSERT(!strcmp(info->get_title().toCString(TRUE),"Title ID3V2"));
                    CPPUNIT_ASSERT(!strcmp(info->get_genre().toCString(TRUE),"Genre ID3V2"));
                }
            }
            else
            {
                ETG_TRACE_ERR(("USBControl::ExtractAudioTags_ID3V1 -> metadata read using libtaginfo failed for URL (%s)",URL));
            }
            delete info;
        }
        else
        {
            ETG_TRACE_ERR(("USBControl::ExtractAudioTags_ID3V1 -> creating tag instance using libtaginfo failed for URL (%s)",URL));
        }
    }
}

void USBControlTest::CheckDRMProtection()
{
    ENTRY

    tResult res;

    /* set device id */
    tDeviceID deviceID;
    tDeviceInfo deviceInfo;
    res = LocalSPM::GetDBManager().GetDevice(OUT deviceID, DTY_USB,"USB-2");
    CPPUNIT_ASSERT(res == 0);
    res = LocalSPM::GetDBManager().GetDeviceInfo(OUT deviceInfo, deviceID);
    CPPUNIT_ASSERT(res == 0);

    tURL URL;
    strncpy_r(URL, deviceInfo.mountPoint, sizeof(URL));
    strncat_r(URL, "/DRMProtected.wma", sizeof(URL));
    ETG_TRACE_USR1(("USBControlTest::CheckDRMProtection for file -> :%s",URL));

    string target((char *)URL);
    Info * info = Info::create_tag_info(target);

    if(info )
    {
        if(info->read())
        {
            // get encryption information
            CPPUNIT_ASSERT(info->get_encrypted());
        }
        else
        {
            ETG_TRACE_ERR(("USBControl::CollectData -> metadata read using libtaginfo failed for URL (%s)",URL));
        }
        delete info;
    }
    else
    {
        ETG_TRACE_ERR(("USBControl::CollectData -> creating tag instance using libtaginfo failed for URL (%s)",URL));
    }
}


void USBControlTest::ExtractImageTags()
{
    ENTRY

    tResult res = MP_NO_ERROR;

    /* Create a new indexed list */
    tDeviceID deviceID;
    tListSize listSize = 0;
    tListType listType = LTY_IMAGE;
    tStreaming streaming = false;
    tListID listID = LIST_ID_NONE;
    tFilterTag1 tag1;
    tFilterTag2 tag2;
    tFilterTag3 tag3;
    tFilterTag4 tag4;

    /* set device id */
    res = LocalSPM::GetDBManager().GetDevice(OUT deviceID, DTY_USB,"USB");
    CPPUNIT_ASSERT(res == 0);

    /* Set the filter tags */
    tag1.tag = 0;
    tag2.tag = 0;
    tag3.tag = 0;
    tag4.tag = 0;

    /* Call create list */
    res = LocalSPM::GetListControl().CreateMediaPlayerIndexedList(
            OUT listID,
            OUT listSize,
            IN listType,
            IN streaming,
            IN tag1,
            IN tag2,
            IN tag3,
            IN tag4,
            IN deviceID);

    CPPUNIT_ASSERT(res == 0);
    CPPUNIT_ASSERT(LIST_ID_NONE != listID);

    /* get the list slice */
    vector<tMediaObject> mediaObjectVector;
    res = LocalSPM::GetListControl().RequestMediaPlayerIndexedListSlice(OUT mediaObjectVector, listID, 0, listSize);
    CPPUNIT_ASSERT(res == 0);


    for(unsigned int iter = 0 ; iter < mediaObjectVector.size() ; iter++)
    {
        /*verify metadata for the image */
        if(!strcmp(mediaObjectVector[iter].fileName,"/albumart/Art-MP3-Its_Raining_Again.jpg"))
        {
            CPPUNIT_ASSERT(mediaObjectVector[iter].fileSize == 4230401LL);
#ifndef TARGET_BUILD
            CPPUNIT_ASSERT(!strcmp(mediaObjectVector[iter].dateTime, "2014:03:27 15:29:32"));
#endif
            CPPUNIT_ASSERT(!strcmp(mediaObjectVector[iter].MetadataField2, "3456x5184"));
            CPPUNIT_ASSERT(!strcmp(mediaObjectVector[iter].MetadataField1, "72x72"));

        }
        if(!strcmp(mediaObjectVector[iter].fileName,"/albumart/Art-MP3-Neuland.jpg"))
        {
            CPPUNIT_ASSERT(mediaObjectVector[iter].fileSize == 5432543LL);
#ifndef TARGET_BUILD
            CPPUNIT_ASSERT(!strcmp(mediaObjectVector[iter].dateTime, "2014:03:27 15:29:34"));
#endif
            CPPUNIT_ASSERT(!strcmp(mediaObjectVector[iter].MetadataField2, "2832x4256"));
            CPPUNIT_ASSERT(!strcmp(mediaObjectVector[iter].MetadataField1, "72x72"));

        }
        else
        {
            /* do nothing */
        }

    }
}

tResult USBControlTest::ReadOneVideoMetaData(IN FILE *fpInput, IN char *tmpFileName, const char *format, ...)
{
    ENTRY

    tGeneralString buffer;

    /* read one line */
    char *cerr;
    cerr = fgets(buffer, sizeof(tGeneralString), fpInput);
    if (cerr != buffer) {
        fclose(fpInput);
        if(0 != remove(tmpFileName))
        {
            ETG_TRACE_ERR(("Cannot delete temp file: %s", tmpFileName));
        }
        return -1;
    }

    /* look for the '=' and read the item */
    char *cEqualSign = strstr(buffer, "=");
    if (!cEqualSign) {
        fclose(fpInput);
        if(0 != remove(tmpFileName))
        {
            ETG_TRACE_ERR(("Cannot delete temp file: %s", tmpFileName));
        }
        return -1;
    }
    cEqualSign++; // now pointing to value
    va_list vl;
    va_start(vl, format);
    vsscanf(cEqualSign, format, vl);
    va_end(vl);

    return MP_NO_ERROR;
}

void USBControlTest::GetMediaType()
{
    ENTRY

    tResult res;
    tDeviceID deviceID;
    tListSize listSize;
    tListID listID;
    tNumberOfMediaObjects numberOfMediaObjects;
    tNumberOfPlayLists numberOfPlayLists;
    tNumberOfFolders numberOfFolders;
    tPath path;

    /* set device id */
    res = LocalSPM::GetDBManager().GetDevice(OUT deviceID, IN DTY_USB, IN "USB");
    CPPUNIT_ASSERT(res == 0);

    /* create a file list from a specific directory */
    strncpy_r(OUT path, IN "/video", IN sizeof(path));
    res = LocalSPM::GetListControl().CreateMediaPlayerFileList(
            OUT listID,
            OUT listSize,
            OUT numberOfMediaObjects,
            OUT numberOfPlayLists,
            OUT numberOfFolders,
            IN LTY_FILELIST,
            IN path,
            IN deviceID);
    CPPUNIT_ASSERT(res == 0);

    /* get the list slice */
    vector<tMediaObject> mediaObjectVector;
    res= LocalSPM::GetListControl().RequestMediaPlayerIndexedListSlice(OUT mediaObjectVector, IN listID, 0, listSize);
    CPPUNIT_ASSERT(res == 0);

    for(unsigned int iter = 0 ; iter < mediaObjectVector.size() ; iter++)
    {
        tURL URL;
        strncpy_r(OUT URL, IN mediaObjectVector[iter].mountPoint, IN sizeof(URL));
        strncat_r(OUT URL, IN mediaObjectVector[iter].fileName, IN sizeof(URL));

        ETG_TRACE_USR1(("USBControlTest::Get Media type for file -> :%s",URL));
        TimeTrace ticks("GetMediaType");

#if USE_VIDEO_METADATA_READ_PROCESS
        VideoTagInfo * info = new VideoTagInfo(URL);
#else
        string target((char *)URL);
        Info * info = Info::create_tag_info(target,true);
#endif
        if(info )
        {
            if(info->read())
            {
                /*Validate*/
                if(!strcmp(mediaObjectVector[iter].fileName,"/video/128x96.mp4"))
                {
                    CPPUNIT_ASSERT(MEDIA_TYPE_VIDEO == (info->get_media_type()));
                }
                if(!strcmp(mediaObjectVector[iter].fileName,"/video/QuickTime_H264_MP4-AAC.mov"))
                {
                    CPPUNIT_ASSERT(MEDIA_TYPE_VIDEO == (info->get_media_type()));
                }
            }
            delete info;
        }
        else
        {
            ETG_TRACE_ERR(("USBControl::GetMediaType -> get media type using libtaginfo failed for URL (%s)",URL));
        }

        ticks.elapsed();
    }
}

#if 0
void USBControlTest::ScanFilesForFingerprintCalc()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    tMountPoint mountPoint;
    add_testfiles_basepath(OUT mountPoint,IN (char*)"/test", IN sizeof(mountPoint));

    tFileExtensions fileExtensions;
    fileExtensions = LocalSPM::GetDataProvider().GetSupportedFileExtensions();

    tFileExtensions fileFilters;
    fileFilters = LocalSPM::GetDataProvider().GetSupportedFileFilters();

    ReadSupportedFilesFromDevice readFilesObject;

    string tmpFilename="/tmp/printfiles.txt";

    tUInt mountPointLen = strlen_r(mountPoint);
    ret = readFilesObject.Initialize(fileExtensions,tmpFilename,fileFilters,mountPoint);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    ret = readFilesObject.ScanDirectory(string(mountPoint));
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    tUInt fileCounter = readFilesObject.GetFileCounter();
    CPPUNIT_ASSERT(fileCounter != 0);

    //Deleting tmp file
    if (0 != remove(tmpFilename.c_str()))
    {
        ETG_TRACE_ERR(("Cannot delete temp file: %s", tmpFilename.c_str()));
    }
}
#endif

void USBControlTest::ValidatetmpfileForFingerprintCalc()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;
    tMountPoint mountPoint;
    add_testfiles_basepath(OUT mountPoint,IN (char*)"/test", IN sizeof(mountPoint));

    tFileExtensions fileExtensions;
    fileExtensions = LocalSPM::GetDataProvider().GetSupportedFileExtensions();

    tFileExtensions fileFilters;
    fileFilters = LocalSPM::GetDataProvider().GetSupportedFileFilters();

    ReadSupportedFilesFromDevice readFilesObject;

    string tmpFilename="/mediaplayer/printfiles.txt";

    tUInt mountPointLen = 0;
    ret = readFilesObject.Initialize(fileExtensions,tmpFilename,fileFilters,mountPoint);
    CPPUNIT_ASSERT(MP_NO_ERROR != ret);
}

#if 0
void USBControlTest::ValidateMountPointForFingerprintCalc()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    tMountPoint mountPoint;
    add_testfiles_basepath(OUT mountPoint,IN (char*)"/mediafiles", IN sizeof(mountPoint));

    tFileExtensions fileExtensions;
    fileExtensions = LocalSPM::GetDataProvider().GetSupportedFileExtensions();

    tFileExtensions fileFilters;
    fileFilters = LocalSPM::GetDataProvider().GetSupportedFileFilters();

    ReadSupportedFilesFromDevice readFilesObject;

    string tmpFilename="/tmp/printfiles.txt";

    tUInt mountPointLen = strlen_r(mountPoint);
    ret = readFilesObject.Initialize(fileExtensions,tmpFilename,fileFilters,mountPoint);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    ret = readFilesObject.ScanDirectory(string(mountPoint));
    CPPUNIT_ASSERT(MP_NO_ERROR != ret);

    //Deleting tmp file
    if (0 != remove(tmpFilename.c_str()))
    {
        ETG_TRACE_ERR(("Cannot delete temp file: %s", tmpFilename.c_str()));
    }
}
#endif

void USBControlTest::ValidateFileExtensionForFingerprintCalc()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;
    tMountPoint mountPoint;
    add_testfiles_basepath(OUT mountPoint,IN (char*)"/test", IN sizeof(mountPoint));

    tFileExtensions fileExtensions;
    fileExtensions.clear();

    tFileExtensions fileFilters;
    fileFilters = LocalSPM::GetDataProvider().GetSupportedFileFilters();

    ReadSupportedFilesFromDevice readFilesObject;

    string tmpFilename="/tmp/printfiles.txt";

    tUInt mountPointLen = 0;
    ret = readFilesObject.Initialize(fileExtensions,tmpFilename,fileFilters,mountPoint);
    CPPUNIT_ASSERT(MP_NO_ERROR != ret);

    //Deleting tmp file
    if (0 != remove(tmpFilename.c_str()))
    {
        ETG_TRACE_ERR(("Cannot delete temp file: %s", tmpFilename.c_str()));
    }
}

void USBControlTest::ValidateFileFilterForFingerprintCalc()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    tFileExtensions fileExtensions;
    fileExtensions = LocalSPM::GetDataProvider().GetSupportedFileExtensions();

    tFileExtensions fileFilters;
    fileFilters.clear();

    ReadSupportedFilesFromDevice readFilesObject;

    string tmpFilename="/tmp/printfiles.txt";
    tMountPoint mountPoint;
    add_testfiles_basepath(OUT mountPoint,IN (char*)"/test", IN sizeof(mountPoint));
    tUInt mountPointLen = 0;
    ret = readFilesObject.Initialize(fileExtensions,tmpFilename,fileFilters,mountPoint);
    CPPUNIT_ASSERT(MP_NO_ERROR != ret);

    //Deleting tmp file
    if (0 != remove(tmpFilename.c_str()))
    {
        ETG_TRACE_ERR(("Cannot delete temp file: %s", tmpFilename.c_str()));
    }
}

void USBControlTest::ValidateExecuteCommandForDFandMD5SUM()
{
    string tmpFileName = "/tmp/dflog.txt";
    string mountPoint = "/";

    char *commandStr[3] = {(char *)"/bin/df",(char *)(mountPoint.c_str()),NULL};
    int retStatus = executeCommand(commandStr,tmpFileName);
    CPPUNIT_ASSERT(0 == retStatus);

    commandStr[0] = (char *)"/usr/bin/md5sum";
    commandStr[1] = (char *)(tmpFileName.c_str()); // "/tmp/dflog.txt"
    commandStr[2] = NULL;
    string tmpFilenameResult = "/tmp/md5sumlog.txt";
    retStatus = executeCommand(commandStr,tmpFilenameResult);
    CPPUNIT_ASSERT(0 == retStatus);

    if(0 != remove(tmpFileName.c_str()))
    {
        ETG_TRACE_ERR(("Cannot delete temp file: %s", tmpFileName.c_str()));
    }
    if(0 != remove(tmpFilenameResult.c_str()))
    {
        ETG_TRACE_ERR(("Cannot delete temp file: %s", tmpFilenameResult.c_str()));
    }
}
