/* CDDADeviceInterface.cpp
 *
 *  Created on: Jun 11, 2015
 *      Author: thm3hi
 */
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_mp.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_CDDA_CONTROL
#ifdef TARGET_BUILD
#include "trcGenProj/Header/CDDADeviceInterface.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_CDDA_CONTROL
#endif
#endif

#include "LocalSPM.h"
#include "FunctionTracer.h"
#include "MediaPlayer_ErrorCodes.h"
#include "VarTrace.h"
#include "TimeTrace.h"

#include "TypeDefinitions.h"
#include "CDDADeviceInterface.h"

#include <sys/stat.h>
#include <fcntl.h>

#define CD_SCSI_IF_SG_CMD_TIMEOUT_MS       20000

extern int globalFakeTheCDDA;

/* a faked CD */
tFiles fakeCD[] = {
        { true,FT_AUDIO,  1, 4711, "file1.cda", FFT_UNKNOWN, 0, FNP_PLAYABLE, 0, "Babelsberg Live 2012", "Calexico", "Frontera / Trigger"    , "no set", 105842 },
        { true,FT_AUDIO,  2, 4712, "file2.cda", FFT_UNKNOWN, 0, FNP_PLAYABLE, 0, "Babelsberg Live 2012", "Calexico", "Epic"                  , "no set", 118933 },
        { true,FT_AUDIO,  3, 4713, "file3.cda", FFT_UNKNOWN, 0, FNP_PLAYABLE, 0, "Babelsberg Live 2012", "Calexico", "The News About William", "no set",  37496 },
        { true,FT_AUDIO,  4, 4714, "file4.cda", FFT_UNKNOWN, 0, FNP_PLAYABLE, 0, "Babelsberg Live 2012", "Calexico", "Black Heart"           , "no set",  10899 },
        { true,FT_AUDIO,  5, 4715, "file5.cda", FFT_UNKNOWN, 0, FNP_PLAYABLE, 0, "Babelsberg Live 2012", "Calexico", "Minas De Cobre"        , "no set", 101705 },
        { true,FT_AUDIO,  6, 4716, "file6.cda", FFT_UNKNOWN, 0, FNP_PLAYABLE, 0, "Babelsberg Live 2012", "Calexico", "Inspiration"           , "no set", 116685 },
        { true,FT_AUDIO,  7, 4717, "file7.cda", FFT_UNKNOWN, 0, FNP_PLAYABLE, 0, "Babelsberg Live 2012", "Calexico", "Two Silver Trees"      , "no set",  42719 },
        { true,FT_AUDIO,  8, 4718, "file8.cda", FFT_UNKNOWN, 0, FNP_PLAYABLE, 0, "Babelsberg Live 2012", "Calexico", "Para"                  , "no set", 106459 },
        { true,FT_AUDIO,  9, 4719, "file9.cda", FFT_UNKNOWN, 0, FNP_PLAYABLE, 0, "Babelsberg Live 2012", "Calexico", "Quattro"               , "no set",  55378 },
        { true,FT_AUDIO, 10, 4720, "dontcare" , FFT_UNKNOWN, 0, FNP_PLAYABLE, 1, "no set"              , "no set"  , "no set"                , "no set",  79945 },
};

CDDADeviceInterface::CDDADeviceInterface()
{
    mCallbacks = NULL;
    //InitPEState(mCurrentState);
    mCurrentState = cdstate_t();
    mCurrentPlaybackState = PE_PBS_UNKNOWN;
    mPreviousPlaybackState = PE_PBS_UNKNOWN;
    mScanEndElapsedTime = 0;
    mTimerAlreadyRunning = false;
    mDevSr[0] = 0;
    mTocRead = false;
    mCDContents = NULL;
    mTrackCount = 0;
    mCurrentTrackPlaying = 0;
    mAudioStatus = CD_AUDIO_STATUS_NONE;
    mStartPlay = false;
}

CDDADeviceInterface::~CDDADeviceInterface()
{
}

void CDDADeviceInterface::registerCallbacks(CDDADeviceInterfaceCallback *callbacks)
{
    mCallbacks = callbacks;
}

tResult CDDADeviceInterface::init()
{
    ENTRY
    tResult res = MP_NO_ERROR;
    mTocRead = false;

    if(LocalSPM::GetDataProvider().CDDAUseRealDevice())
    {
        globalFakeTheCDDA = 0;
    }
    else
    {
        globalFakeTheCDDA = 1;
    }
    mCdRomType = eCDS_AUDIO;
    return res;
}

void CDDADeviceInterface::extractDeviceFileNames(const tMountPoint _mountpoint)
{
    ENTRY

    tMountPoint mountpoint;
    strncpy_r(mountpoint, _mountpoint, sizeof(mountpoint));

    char *cptrStartOfSr = NULL;

    /* is a device file in the mountpoint or URL ? */
    cptrStartOfSr = strstr(mountpoint, "/dev/sr");
    if (!cptrStartOfSr) {
        return;
    }

    /* search for the end of it */
    /* pattern: "/dev/sr??xxx" (?? = number; xxx = any charactder) */
    char *cptr = cptrStartOfSr;
    bool numberFound = false;
    while(*cptr) {
        if (isdigit(*cptr)) {
            numberFound = true;
        } else if (numberFound) { // end of digit
            *cptr = 0;
            break;
        }
        cptr++;
    }
    if (!numberFound) { // number not found
        return;
    }

    /* copy the device file name to member */
    strncpy_r(mDevSr, cptrStartOfSr, sizeof(mDevSr));
}

int CDDADeviceInterface::run_SG_IO_Command()
{
    ENTRY

    tResult res = MP_NO_ERROR;

    int fd = open(mDevSr, O_RDONLY | O_NONBLOCK);
    if (fd == -1) {
        ETG_TRACE_ERR(("CDDADeviceInterface::run_SG_IO_Command: open error: %d/%s", errno, strerror(errno)));
        return MP_ERR_CDDA_OPEN_SR;
    }

    int ioErr = ioctl(fd, (unsigned long)SG_IO, &mSCSICommandBuffer);
    if (ioErr == -1) {
        ETG_TRACE_ERR(("CDDADeviceInterface::run_SG_IO_Command: ioctl error: %d/%s", errno, strerror(errno)));
        res = MP_ERR_CDDA_IOCTRL;
    }

    close(fd);

    return res;
}

int CDDADeviceInterface::cdIOControl(int command, void *buffer)
{
    ENTRY

    tResult res = MP_NO_ERROR;

    int fd = open(mDevSr, O_RDONLY | O_NONBLOCK);
    if (fd == -1) {
        ETG_TRACE_ERR(("CDDADeviceInterface::cdIOControl: open error: %d/%s", errno, strerror(errno)));
        return MP_ERR_CDDA_OPEN_SR;
    }

    int ioErr = 0;
    if (!buffer) {
        ioErr = ioctl(fd, (unsigned long)command);
    } else {
        ioErr = ioctl(fd, (unsigned long)command, buffer);
    }
    if (ioErr == -1) {
        ETG_TRACE_ERR(("CDDADeviceInterface::cdIOControl: ioctl error: %d/%s", errno, strerror(errno)));
        res = MP_ERR_CDDA_IOCTRL;
    }

    close(fd);

    return res;
}


void CDDADeviceInterface::prepare_SG_IO_Command(bool awaitingResponse, int commandLen)
{
    memset(&mSCSICommandBuffer, 0, sizeof(mSCSICommandBuffer));
    memset(&mCommandBuffer, 0, sizeof(mCommandBuffer));
    memset(&mSenseBuffer, 0, sizeof(mSenseBuffer));
    memset(&mResponseBuffer, 0, sizeof(mResponseBuffer));

    mSCSICommandBuffer.cmdp    = mCommandBuffer;
    mSCSICommandBuffer.cmd_len = commandLen;

    mSCSICommandBuffer.dxferp      = mResponseBuffer;
    mSCSICommandBuffer.dxfer_len   = sizeof(mResponseBuffer);

    mSCSICommandBuffer.sbp         = mSenseBuffer;
    mSCSICommandBuffer.mx_sb_len   = (unsigned char)sizeof(mSenseBuffer);

    mSCSICommandBuffer.interface_id = 'S'; //INTERFACE_ID
    if (awaitingResponse) {
        mSCSICommandBuffer.dxfer_direction = SG_DXFER_FROM_DEV;
    } else {
        mSCSICommandBuffer.dxfer_direction = SG_DXFER_TO_DEV;
    }
    mSCSICommandBuffer.timeout         = CD_SCSI_IF_SG_CMD_TIMEOUT_MS;

    int bufferSizeOffset = -1;
    if (awaitingResponse) {
        bufferSizeOffset = 7;
    }

    if((bufferSizeOffset >= 0)
            && ((size_t)(bufferSizeOffset + 1) < sizeof(mCommandBuffer)))
    {
        int responseSize = sizeof(mResponseBuffer);
        mCommandBuffer[bufferSizeOffset] = (unsigned char)((responseSize >> 8) & 0xFF);
        mCommandBuffer[bufferSizeOffset + 1] = (unsigned char)(responseSize & 0xFF);
    }
}

#if 0 // standard CDROM io control for read toc
tResult CDDADeviceInterface::readToc(IN const tMountPoint mountpoint)
{
    ENTRY
    tResult res = MP_NO_ERROR;

    /* toc already read in? */
    if (mTocRead) {
        return res;
    }

    /* extract the device names for the following cdIOControl command */
    extractDeviceFileNames(mountpoint);

    /* read the TOC */
    struct cdrom_tochdr tochdr;
    struct cdrom_tocentry tocentry;
    cdIOControl(CDROMREADTOCHDR, &tochdr);

    /* put some general info */
    mToc.u8MinTrack = tochdr.cdth_trk0;
    mToc.u8MaxTrack = tochdr.cdth_trk1;
    mTrackCount = (tochdr.cdth_trk1 - tochdr.cdth_trk0) + 1;

    /* parse the TOC and fill int the cddb TOC table */
    for (int i = mToc.u8MinTrack; i <= mToc.u8MaxTrack; i++) {

        /* read one toc entry */
        tocentry.cdte_track = i;
        tocentry.cdte_format = CDROM_MSF;
        cdIOControl(CDROMREADTOCENTRY, &tocentry);

        /* put it into the internal storage */
        mToc.arTrack[i].u8AdrCtrl = tocentry.cdte_adr; // dont know for what this is
        mToc.arTrack[i].u32StartZLBA = CD_MSF2ZLBA(tocentry.cdte_addr.msf.minute,
                tocentry.cdte_addr.msf.second, tocentry.cdte_addr.msf.frame);
    }

    /* last entry */
    tocentry.cdte_track = 0xAA;
    tocentry.cdte_format = CDROM_MSF;
    cdIOControl(CDROMREADTOCENTRY, &tocentry);

    /* put in information about end of CD */
    mToc.u32LastZLBA = CD_MSF2ZLBA(tocentry.cdte_addr.msf.minute,
            tocentry.cdte_addr.msf.second, tocentry.cdte_addr.msf.frame);
    mToc.u8Valid = CD_VALID;

    /* if toc valid, try to read cd text */
    if (mToc.u8Valid) {

        /* create a buffer for all CD info */
        mCDContents = (tFiles *)calloc(sizeof(tFiles), mTrackCount+1); // +1: eof record

        /* fill in some general info */
        for(int i=0; i<mTrackCount; i++) {
            mCDContents[i].type = FT_AUDIO;
            mCDContents[i].trackNumber = i+1;
            snprintf(mCDContents[i].fileName, sizeof(mCDContents[i].fileName), "file%d.cda", i+1);
            mCDContents[i].fileFormat = FFT_UNKNOWN;
            mCDContents[i].notPlayable = FNP_PLAYABLE;

            /* not the very last track: */
            if (i < (mTrackCount-1)) {

                // track 0 is a dummy track with information */
                mCDContents[i].playTime = mToc.arTrack[i+2].u32StartZLBA - mToc.arTrack[i+1].u32StartZLBA;

                /* only for the last track: */
            } else {

                // track 0 is a dummy track with information */
                mCDContents[i].playTime = mToc.u32LastZLBA - mToc.arTrack[i+1].u32StartZLBA;
            }

            mCDContents[i].playTime /= CD_SECTORS_PER_SECOND;   // convert to seconds
            mCDContents[i].playTime *= 1000;                    // in milliseconds
        }

        /* set eof marker */
        mCDContents[mTrackCount].isEOF = 1;

        /* prepare my io buffer */
        prepare_SG_IO_Command(RESPONSE_ON, 9);

        /* define the command */
        mCommandBuffer[0] = READ_TOC; /*0x43*/
        mCommandBuffer[1] = 0x02; //MSF_MSF;
        mCommandBuffer[2] = 0x05; //FORMAT_CDTEXT;

        /* do the device command */
        res = run_SG_IO_Command();
        if (res == MP_NO_ERROR) {

            int dataLen = CD_8TO16(mResponseBuffer[0], mResponseBuffer[1]);
            if (dataLen) { // got CD text:

                /* read that out */
                setCDText(dataLen);
            }
        }
    }

    /* now TOC is valid */
    mTocRead = true;

    return res;
}
#endif

tResult CDDADeviceInterface::readToc(IN const tMountPoint mountpoint)
{
    ENTRY
    tResult res = MP_NO_ERROR;

    /* toc already read in? */
    if (mTocRead) {
        return res;
    }

    extractDeviceFileNames(mountpoint);

    /* prepare my io buffer */
    mCdRomType = (tCdromDiskStatus)cdCtrlInfo(CDROM_DISC_STATUS);
    prepare_SG_IO_Command(RESPONSE_ON, cdGetCmdLen(READ_TOC));

    /* define the command */
    mCommandBuffer[0] = READ_TOC; /*0x43*/
    mCommandBuffer[1] = 0x02; //MSF_MSF;
    mCommandBuffer[2] = 0x00; //FORMAT_TOC;

    /* do the device command */
    res = run_SG_IO_Command();

    /* extract data from io buffer which device command has filled */
    if (res == MP_NO_ERROR) {

        unsigned char *pucBuff = mResponseBuffer;
        tCDTOC *prToc = &mToc;
        int tocLen = (unsigned int)CD_8TO16(pucBuff[0], pucBuff[1]);
        unsigned char u8T;
        unsigned char u8Track;
        unsigned char u8TOCValid = CD_INVALID;

        if((tocLen >= CD_SCSI_IF_TOC_ENTRY_LEN) && (tocLen < CD_SCSI_IF_SG_TOC_BUFFER_LEN))
        {
            prToc->u8MinTrack = pucBuff[2];
            if(eCDS_AUDIO == mCdRomType)
            {
                prToc->u8MaxTrack = pucBuff[3];
            }
            else
            {
                prToc->u8MaxTrack = pucBuff[3]-1;
            }
            mTrackCount = (prToc->u8MaxTrack - prToc->u8MinTrack) + 1;

            /*************** Test traces for field reset issue: To be removed later ****************/
            ETG_TRACE_ERRMEM(("prToc->u8MinTrack: %d",prToc->u8MinTrack));
            ETG_TRACE_ERRMEM(("prToc->u8MaxTrack: %d",prToc->u8MaxTrack));
            ETG_TRACE_USR4(("CDDADeviceInterface::readToc: prToc->u8MinTrack = %d",prToc->u8MinTrack));
            ETG_TRACE_USR4(("CDDADeviceInterface::readToc: prToc->u8MaxTrack = %d",prToc->u8MaxTrack));
            ETG_TRACE_ERRMEM(("mTrackCount:%d",mTrackCount));
            ETG_TRACE_USR4(("CDDADeviceInterface::readToc: mTrackCount = %d",mTrackCount));


            pucBuff = &pucBuff[4];
            for(u8T = 0; u8T < mTrackCount; u8T++)
            {
                if(eCDS_AUDIO == mCdRomType)
                {
                    pucBuff  = &mResponseBuffer[4 + ((unsigned int)u8T * CD_SCSI_IF_TOC_ENTRY_LEN)];
                    u8Track = pucBuff[2];
                }
                else
                {
                    pucBuff  = &mResponseBuffer[12 + ((unsigned int)u8T * CD_SCSI_IF_TOC_ENTRY_LEN)];
                    u8Track = pucBuff[2]-1;
                }
                prToc->arTrack[u8Track].u8AdrCtrl    = pucBuff[1];
                prToc->arTrack[u8Track].u32StartZLBA = CD_MSF2ZLBA(pucBuff[5],
                        pucBuff[6],
                        pucBuff[7]);
            } //for(u8T = 0; u8T < u8TrackCount; u8T++)

            /* read end of disk */
            /* prepare my io buffer */
            prepare_SG_IO_Command(RESPONSE_ON, cdGetCmdLen(READ_CAPACITY));

            /* define the command */
            mCommandBuffer[0] = READ_CAPACITY;

            /* do the device command */
            res = run_SG_IO_Command();
            if (res == MP_NO_ERROR) {

                pucBuff = mResponseBuffer;
                prToc->u32LastZLBA = CD_8TO32(pucBuff[0],
                        pucBuff[1], pucBuff[2], pucBuff[3]);
                prToc->u8Valid = CD_VALID;
                u8TOCValid = prToc->u8Valid;

            } else {
                prToc->u8Valid = CD_INVALID;
                u8TOCValid = prToc->u8Valid;
            }

            if(mTrackCount < 0)
            {
                prToc->u8Valid = CD_INVALID;
                u8TOCValid = prToc->u8Valid;
            }

        } else {
            prToc->u8Valid = CD_INVALID;
            u8TOCValid = prToc->u8Valid;
        }

        if (u8TOCValid) {
            mTocRead = true;
        } else {
            ETG_TRACE_ERR(("CDDADeviceInterface::getToc: invalid TOC"));
        }

        /* if toc valid, try to read cd text */
        if (u8TOCValid) {

            /* create a buffer for all CD info */
            mCDContents = (tFiles *)calloc(sizeof(tFiles), mTrackCount+1); // +1: eof record
            if(!mCDContents)
            {
                ETG_TRACE_USR4(("CDDADeviceInterface::readToc:  calloc failed for mCDContents !!"));
            }
            else
            {
                /* fill in some general info */
                for(int i=0; i<mTrackCount; i++) {
                    mCDContents[i].type = FT_AUDIO;
                    mCDContents[i].trackNumber = i+1;
                    snprintf(mCDContents[i].fileName, sizeof(mCDContents[i].fileName), "file%d.cda", i+1);
                    mCDContents[i].fileFormat = FFT_UNKNOWN;
                    mCDContents[i].notPlayable = FNP_PLAYABLE;

                    /* not the very last track: */
                    if (i < (mTrackCount-1)) {

                        // track 0 is a dummy track with information */
                        mCDContents[i].playTime = prToc->arTrack[i+2].u32StartZLBA - prToc->arTrack[i+1].u32StartZLBA;

                        /* only for the last track: */
                    } else {

                        // track 0 is a dummy track with information */
                        mCDContents[i].playTime = prToc->u32LastZLBA - prToc->arTrack[i+1].u32StartZLBA;
                    }

                    mCDContents[i].playTime /= CD_SECTORS_PER_SECOND;   // convert to seconds
                    mCDContents[i].playTime *= 1000;                    // in milliseconds
                }

                /* set eof marker */
                mCDContents[mTrackCount].isEOF = 1;

                /* prepare my io buffer */
                prepare_SG_IO_Command(RESPONSE_ON, cdGetCmdLen(READ_TOC));

                /* define the command */
                mCommandBuffer[0] = READ_TOC; /*0x43*/
                mCommandBuffer[1] = 0x02; //MSF_MSF;
                mCommandBuffer[2] = 0x05; //FORMAT_CDTEXT;

                /* do the device command */
                res = run_SG_IO_Command();
                if (res == MP_NO_ERROR) {

                    int dataLen = CD_8TO16(mResponseBuffer[0], mResponseBuffer[1]);
                    if (dataLen) { // got CD text:

                        /* read that out */
                        setCDText(dataLen);
                    }
                }
            }
        }

        /* wait for settle down */
        int retry = 60;
        int lastSubChannelIndex = 0;
        struct cdrom_subchnl subchnl = {0};
        while(retry--) {

            subchnl.cdsc_format = CDROM_MSF;
            cdIOControl(CDROMSUBCHNL, &subchnl);
#if 0
            printf("cdsc_audiostatus = %d\n", subchnl.cdsc_audiostatus);
            printf("cdsc_adr = %d\n", subchnl.cdsc_adr);
            printf("cdsc_ctrl = %d\n", subchnl.cdsc_ctrl);
            printf("cdsc_trk = %d\n", subchnl.cdsc_trk);
            printf("cdsc_ind = %d\n", subchnl.cdsc_ind);
#endif

            /* if last index in track is same as before, drive has settled down */
            if (lastSubChannelIndex == subchnl.cdsc_ind) {
                break;
            }
            lastSubChannelIndex = subchnl.cdsc_ind;

            /* wait a little to help drive */
            usleep(100000);
        }

    }

    return res;
}

tResult CDDADeviceInterface::getToc(OUT tFiles *tocDataOneTrack, IN const tMountPoint mountpoint,
        IN const tPath path, IN const tIndex index)
{
    ENTRY
    VARTRACE(mountpoint);
    VARTRACE(path);
    VARTRACE(index);

    tResult res = MP_NO_ERROR;

    extractDeviceFileNames(mountpoint);

    //test
    if (globalFakeTheCDDA) {

        /* ask the fake class for values */
        memset(tocDataOneTrack, 0, sizeof(tFiles));

        /* copy data from faked cd */
        *tocDataOneTrack = fakeCD[index];

        return MP_NO_ERROR;

    } else if (!mTocRead) { // read toc only once in a CDs lifetime

        res = readToc(mountpoint);
        if (res != MP_NO_ERROR) {
            return res;
        }
    }
    if((NULL!=tocDataOneTrack) && (index <= mTrackCount))
    {
        ETG_TRACE_USR4(("CDDADeviceInterface::getToc null check Done "));
    /* create return values */
    memset(tocDataOneTrack, 0, sizeof(tFiles));

    if (mTocRead) { // toc is valid
        /* copy data from cd */
            if(NULL != mCDContents)
            {
        *tocDataOneTrack = mCDContents[index];
            }

    } else { // toc is not valid
          ETG_TRACE_USR4(("CDDADeviceInterface::getToc mTocRead failed "));
            /* create an eof entry to return */
            /*
            tFiles *eofEntry = (tFiles *)calloc(1, sizeof(tFiles));
            eofEntry->isEOF = 1;
            *tocDataOneTrack = *eofEntry;
            */
            tocDataOneTrack->isEOF = 1;
            res = MP_ERR_CDDA_CD_NOT_MOUNTED;
       }
    }
    return res;
}

void CDDADeviceInterface::getParameter(OUT tPEHandle &handle,
        OUT tPEPlaybackState &peState, OUT me::reason_e &reason, OUT me::speed_e &speed)
{
    ENTRY
    handle = mCurrentState.handle;
    peState = mCurrentPlaybackState;
    reason = mCurrentState.reason;
    speed = (me::speed_e)mCurrentState.speed;
}

tResult CDDADeviceInterface::play(IN const tURL url, IN const me::speed_e speed,
        IN const tPlaytime position, IN const tPEHandle handle)
{
    ENTRY
    VARTRACE(url);
    VARTRACE(speed);
    VARTRACE(position);
    VARTRACE(handle);

    tResult res = MP_NO_ERROR;

    /* simulation on? */
    if (globalFakeTheCDDA) {
        /* do nothing in here */

    } else {

        extractDeviceFileNames(url);

        /* is the toc already read in? if not: do it */
        if(!mTocRead) {
            res = readToc(url);
            if (res != MP_NO_ERROR) {
                return res;
            }
        }

        /* start playing */

        /* read out the index from the url given */
        char *cptrFile = strstr((char *)url, (char *)"file");
        if (!cptrFile) {
            return MP_ERR_CDDA_WRONG_URL;
        }
        mCurrentTrackPlaying = atoi(cptrFile+4); /* files are numbered beginning from 1 */
        if (mCurrentTrackPlaying > mTrackCount) {
            return MP_ERR_CDDA_WRONG_URL;
        }

        /* read the subchannel info - but why? */
        struct cdrom_subchnl subchnl = {0};
        subchnl.cdsc_format = CDROM_MSF;
        cdIOControl(CDROMSUBCHNL, &subchnl);

        struct cdrom_multisession ms_info;
        memset(&ms_info, 0, sizeof(ms_info));
        cdIOControl(CDROMMULTISESSION, &ms_info);

        cdIOControl(CDROMSTART, NULL);

        /* calc/set the start and end time */
        tCDMSF startTime = {0};
        tCDMSF endTime = {0};

        /* start time */
        int positionLBA = mToc.arTrack[mCurrentTrackPlaying].u32StartZLBA + ((position / 1000) * CD_SECTORS_PER_SECOND);
        vZLBA2MSF(positionLBA, &startTime);

        /* end time */
        if (mCurrentTrackPlaying < mTrackCount) {
            vZLBA2MSF(mToc.arTrack[mCurrentTrackPlaying+1].u32StartZLBA, &endTime);
        } else {
            vZLBA2MSF(mToc.u32LastZLBA, &endTime);
        }

        ETG_TRACE_USR4(("play: MSF min : %d, sec : %d, frame : %d",startTime.u8Min, startTime.u8Sec,startTime.u8Frm));
        ETG_TRACE_USR4(("play end: MSF min : %d, sec : %d, frame : %d",endTime.u8Min, endTime.u8Sec,endTime.u8Frm));
        if(position != 0)
        {
            ETG_TRACE_USR4(("GPCMD_PLAY_AUDIO_MSF"));
            /* give command to scsi: */
            prepare_SG_IO_Command(RESPONSE_OFF, cdGetCmdLen(GPCMD_PLAY_AUDIO_MSF));

            /* define the command */
            mCommandBuffer[0] = GPCMD_PLAY_AUDIO_MSF;
            mCommandBuffer[3] = startTime.u8Min;
            mCommandBuffer[4] = startTime.u8Sec;
            mCommandBuffer[5] = startTime.u8Frm;
            mCommandBuffer[6] = endTime.u8Min;
            mCommandBuffer[7] = endTime.u8Sec;
            mCommandBuffer[8] = endTime.u8Frm;
        }
        else
        {
            ETG_TRACE_USR4(("GPCMD_PLAY_AUDIO_TI StartTrack:%d EndTrack:%d",mCurrentTrackPlaying,mCurrentTrackPlaying));
            /* give PLAY AUDIO TRACK INDEX command to fix NCG3D-5203 */
            /* give command to scsi: */
            prepare_SG_IO_Command(RESPONSE_OFF, cdGetCmdLen(GPCMD_PLAY_AUDIO_TI));

            /* define the command */
            mCommandBuffer[0] = GPCMD_PLAY_AUDIO_TI;
            if(CDS_AUDIO == mCdRomType)
            {
                mCommandBuffer[4] = mCurrentTrackPlaying;
                mCommandBuffer[7] = mCurrentTrackPlaying;
            }
            else
            {
                /* Play track + 1 for mixedmode and data CD , track1 is always data track*/
                mCommandBuffer[4] = mCurrentTrackPlaying+1;
                mCommandBuffer[7] = mCurrentTrackPlaying+1;
            }
        }
        /* do the device command */
        res = run_SG_IO_Command();
        if (res != MP_NO_ERROR) {
            return res;
        }

    } // no simulation

    /* care about status */
    mCurrentState.handle = handle;
    mCurrentState.playstate = PLAYSTATE_PLAY;
    mCurrentState.reason = REASON_OK;
    mCurrentState.speed = speed;
    mCurrentState.url = url;
    mCurrentPlaybackState = PE_PBS_PLAYINGSTATE;
    // NCG3D-22900. below fucntions will be called after PLAY_ANSWER is sent to PM.
    //update();
    //startUpdateTimer();

    return res;
}

tResult CDDADeviceInterface::pause()
{
    ENTRY

    tResult res = MP_NO_ERROR;

    /* simulation on? */
    if (globalFakeTheCDDA) {
        /* do nothing in here */

    } else {

        /* if not already in pause state: */
        if (mCurrentState.playstate != PLAYSTATE_PAUSE) {

            /* give command to scsi: */
            prepare_SG_IO_Command(RESPONSE_OFF, cdGetCmdLen(GPCMD_PAUSE_RESUME));

            /* define the command */
            mCommandBuffer[0] = GPCMD_PAUSE_RESUME;

            /* do the device command */
            res = run_SG_IO_Command();
            if (res != MP_NO_ERROR) {
                return res;
            }
        }
    }

    mCurrentState.playstate = PLAYSTATE_PAUSE;
    mCurrentPlaybackState = PE_PBS_PAUSEDSTATE;
    mCurrentState.reason = REASON_OK;
    stopUpdateTimer();
    update();

    return res;
}

tResult CDDADeviceInterface::resume()
{
    ENTRY

    tResult res = MP_NO_ERROR;

    /* simulation on? */
    if (globalFakeTheCDDA) {
        /* do nothing in here */

    } else {

        /* if in pause state: */
        if (mCurrentState.playstate == PLAYSTATE_PAUSE) {

            /* this is for masca, taken from code of Andre Storch (does not work for linux cdrom driver as resume) */

            /* give command to scsi: */
            prepare_SG_IO_Command(RESPONSE_OFF, cdGetCmdLen(GPCMD_PAUSE_RESUME));

            /* define the command */
            mCommandBuffer[0] = GPCMD_PAUSE_RESUME;

            /* do the device command */
            res = run_SG_IO_Command();
            if (res != MP_NO_ERROR) {
                return res;
            }

            /* this is standard CDROM */
            res = cdIOControl(CDROMRESUME, NULL);
            if (res != MP_NO_ERROR) {
                return res;
            }
        }
    }

    mCurrentState.playstate = PLAYSTATE_PLAY;
    mCurrentPlaybackState = PE_PBS_PLAYINGSTATE;

    update();
    startUpdateTimer();

    return res;
}

tResult CDDADeviceInterface::stop()
{
    ENTRY
    tResult res = MP_NO_ERROR;

    /* simulation on? */
    if (globalFakeTheCDDA) {
        /* do nothing in here */

    } else {

        /* if in pause state: */
        if (mCurrentState.playstate != PLAYSTATE_STOP) {

            /* give command to scsi: */
            prepare_SG_IO_Command(RESPONSE_OFF, cdGetCmdLen(GPCMD_STOP_PLAY_SCAN));

            /* define the command */
            mCommandBuffer[0] = GPCMD_STOP_PLAY_SCAN;

            /* do the device command */
            res = run_SG_IO_Command();
            if (res != MP_NO_ERROR) {
                return res;
            }
        }
    }

    //mCurrentState.playstate = PLAYSTATE_STOP;
    //mCurrentPlaybackState = PE_PBS_STOPPEDSTATE;

    //stopUpdateTimer();
    //update();

    return res;
}

tResult CDDADeviceInterface::seek(IN const me::speed_e speed, IN const tPlaytime position)
{
    ENTRY
    VARTRACE(speed);
    VARTRACE(position);

    tResult res = MP_NO_ERROR;

    char *url = strdup(mCurrentState.url.c_str()); // done to prevent overlapping strcpy
    if (mCurrentState.playstate == PLAYSTATE_PAUSE) {
        play(url, speed, position, mCurrentState.handle);
        pause();
    } else if (mCurrentState.playstate == PLAYSTATE_PLAY) {
        play(url, speed, position, mCurrentState.handle);
    }
    if (url) free(url);

    mCurrentState.speed = speed;
    mCurrentState.pos_ms = position;
    mCurrentPlaybackState = PE_PBS_SEEKTOSTATE;

    update();
    //startUpdateTimer();

    return res;
}

tResult CDDADeviceInterface::frevStart(tCueingRate rate)
{
    ENTRY

    tResult res = MP_NO_ERROR;

    /* leave this in but does not seem to work */
    SetSpeed(rate);

    /* give command to scsi: */
    prepare_SG_IO_Command(RESPONSE_OFF, cdGetCmdLen(GPCMD_SCAN));

    /* define the command */
    mCommandBuffer[0] = GPCMD_SCAN;
    mCommandBuffer[1] = (0x01 << 4); // backwards

    struct cdrom_subchnl subchnl = {0};
#if 0
    subchnl.cdsc_format = CDROM_LBA;
    cdIOControl(CDROMSUBCHNL, &subchnl);
    int positionLBA = subchnl.cdsc_absaddr.lba;

    mCommandBuffer[2] = CD_32BIT_B3(positionLBA);
    mCommandBuffer[3] = CD_32BIT_B2(positionLBA);
    mCommandBuffer[4] = CD_32BIT_B1(positionLBA);
    mCommandBuffer[5] = CD_32BIT_B0(positionLBA);

    mCommandBuffer[9] = 0x00; //LBA, bit 6,7 = 00; MSF=01, TNO=10
#endif

    subchnl.cdsc_format = CDROM_MSF;
    cdIOControl(CDROMSUBCHNL, &subchnl);
    tCDMSF msfScanPos = {0};
    vMSFBCD2Decimal(subchnl.cdsc_absaddr.msf,msfScanPos);
    ETG_TRACE_USR4(("frevstart: MSF min : %d, sec : %d, frame : %d",msfScanPos.u8Min,msfScanPos.u8Sec,msfScanPos.u8Frm));

    mCommandBuffer[3] = msfScanPos.u8Min;
    mCommandBuffer[4] = msfScanPos.u8Sec;
    mCommandBuffer[5] = msfScanPos.u8Frm;

    mCommandBuffer[9] = 0x40; //LBA, bit 6,7 = 00; MSF=01, TNO=10

    /* do the device command */
    res = run_SG_IO_Command();
    if (res != MP_NO_ERROR) {
        return res;
    }

    mCurrentState.playstate = PLAYSTATE_PLAY;
    mCurrentState.speed = (me::speed_e)rate;
    mPreviousPlaybackState = mCurrentPlaybackState;
    mCurrentPlaybackState = PE_PBS_FASTREVERSESTATE;

    update();
    startUpdateTimer();

    return res;
}

tResult CDDADeviceInterface::frevStop()
{
    ENTRY

    tResult res = MP_NO_ERROR;

    /* reset-CDROM speed */
    SetSpeed(1);

    mStartPlay = true;
    mCurrentState.playstate = PLAYSTATE_PLAY;
    mCurrentState.speed = ME_SPEED_NORMAL;
    mPreviousPlaybackState = mCurrentPlaybackState;
    mCurrentPlaybackState = PE_PBS_PLAYINGSTATE;

    /* stop frev */
    res = stop();
    if(res != MP_NO_ERROR)
    {
        ETG_TRACE_ERR(("Could not stop frev error : %d/%s", errno, strerror(errno)));
    }

    update();
    startUpdateTimer();

    return res;
}

tResult CDDADeviceInterface::ffwdStart(tCueingRate rate)
{
    ENTRY

    tResult res = MP_NO_ERROR;

    /* leave this in but does not seem to work */
    SetSpeed(rate);

    /* give command to scsi: */
    prepare_SG_IO_Command(RESPONSE_OFF, cdGetCmdLen(GPCMD_SCAN));

    /* define the command */
    mCommandBuffer[0] = GPCMD_SCAN;
    mCommandBuffer[1] = 0; // forward

    struct cdrom_subchnl subchnl = {0};
#if 0
    subchnl.cdsc_format = CDROM_LBA;
    cdIOControl(CDROMSUBCHNL, &subchnl);
    int positionLBA = subchnl.cdsc_absaddr.lba;

    mCommandBuffer[2] = CD_32BIT_B3(positionLBA);
    mCommandBuffer[3] = CD_32BIT_B2(positionLBA);
    mCommandBuffer[4] = CD_32BIT_B1(positionLBA);
    mCommandBuffer[5] = CD_32BIT_B0(positionLBA);

    mCommandBuffer[9] = 0x00; //LBA, bit 6,7 = 00; MSF=01, TNO=10
#endif

    subchnl.cdsc_format = CDROM_MSF;
    cdIOControl(CDROMSUBCHNL, &subchnl);
    tCDMSF msfScanPos = {0};
    vMSFBCD2Decimal(subchnl.cdsc_absaddr.msf,msfScanPos);
    ETG_TRACE_USR4(("ffstart: MSF min : %d, sec : %d, frame : %d",msfScanPos.u8Min,msfScanPos.u8Sec,msfScanPos.u8Frm));
    mCommandBuffer[3] = msfScanPos.u8Min;
    mCommandBuffer[4] = msfScanPos.u8Sec;
    mCommandBuffer[5] = msfScanPos.u8Frm;
    mCommandBuffer[9] = 0x40; //LBA, bit 6,7 = 00; MSF=01, TNO=10
    /* do the device command */
    res = run_SG_IO_Command();
    if (res != MP_NO_ERROR) {
        return res;
    }

    mCurrentState.playstate = PLAYSTATE_PLAY;
    mCurrentState.speed = (me::speed_e)rate;
    mPreviousPlaybackState = mCurrentPlaybackState;
    mCurrentPlaybackState = PE_PBS_FASTFORWARDSTATE;

    update();
    startUpdateTimer();

    return res;
}

tResult CDDADeviceInterface::ffwdStop()
{
    ENTRY

    tResult res = MP_NO_ERROR;

    /* reset-CDROM speed */
    SetSpeed(1);

    mStartPlay = true;
    mCurrentState.playstate = PLAYSTATE_PLAY;
    mCurrentState.speed = ME_SPEED_NORMAL;
    mPreviousPlaybackState = mCurrentPlaybackState;
    mCurrentPlaybackState = PE_PBS_PLAYINGSTATE;

    /* stop ffwd */
    res = stop();

    if(res != MP_NO_ERROR)
    {
        ETG_TRACE_ERR(("Could not stop ffwd error : %d/%s", errno, strerror(errno)));
    }

    update();
    startUpdateTimer();

    return res;
}

bool CDDADeviceInterface::TimerCallBack(timer_t /*timerID*/ , void *instance, const void */*userData*/)
{
    ENTRY

    tResult res = MP_NO_ERROR;
    CDDADeviceInterface *_this = (CDDADeviceInterface *)instance;
    tU8 audioStatus = CD_AUDIO_STATUS_INVALID;
    bool trackChanged  = false;

    if (globalFakeTheCDDA) {

        /* modify the current playtime */
        _this->mCurrentState.pos_ms += (300 * _this->mCurrentState.speed);
        //CMG3G-11436: to trigger now playing for first timer call back. This will solve the PlayItemFromCDList unit test.
        audioStatus = CD_AUDIO_STATUS_PROGRESS;

    } else {

        /* read the current playtime from CDROM drive */
        struct cdrom_subchnl subchnl = {0};
        subchnl.cdsc_format = CDROM_MSF;
        tCDMSF msf = {0};

        res = _this->cdIOControl(CDROMSUBCHNL, &subchnl);
        if(res)                                           // NCG3D-7274
        {
            ETG_TRACE_USR4(("TimerCallBack cdIOControl error : %d ",res));
            return res;
        }

#if 0
        printf("cdsc_reladdr.msf.minute = %d\n", subchnl.cdsc_reladdr.msf.minute);
        printf("cdsc_reladdr.msf.second = %d\n", subchnl.cdsc_reladdr.msf.second);
        printf("cdsc_reladdr.msf.frame = %d\n", subchnl.cdsc_reladdr.msf.frame);
#endif
        // NCG3D-3269. convert bcd to decimal
        _this->vMSFBCD2Decimal(subchnl.cdsc_reladdr.msf,msf);

        ETG_TRACE_USR4(("cdsc_reladdr.msf.minute = %d, min = %d", subchnl.cdsc_reladdr.msf.minute,msf.u8Min));
        ETG_TRACE_USR4(("cdsc_reladdr.msf.second = %d, sec = %d", subchnl.cdsc_reladdr.msf.second,msf.u8Sec));
        ETG_TRACE_USR4(("cdsc_reladdr.msf.frame = %d, frame = %d", subchnl.cdsc_reladdr.msf.frame,msf.u8Frm));

        audioStatus = subchnl.cdsc_audiostatus;
        //The subchnl.cdsc_trk is in BCD format convert that to decimal before comparison
        int trackNumber = _this->convertBCDtoDecimal(subchnl.cdsc_trk);
        if(eCDS_AUDIO != _this->mCdRomType)
        {
            // For mixedmode CD track starts from track2 (NCG3D-47512)
            trackNumber = trackNumber - 1;
        }
        trackChanged = (_this->mCurrentTrackPlaying != trackNumber);
        ETG_TRACE_USR4(("TimerCallback Track no: %d, audioStatus: %x",trackNumber,audioStatus));

        /* set current playtime in status struct */
        _this->mCurrentState.pos_ms = (msf.u8Min * 60000L) +
                (msf.u8Sec * 1000L) +
                (((float)msf.u8Frm / 75) * 1000L);
        if(_this->mStartPlay ==true)
        {
            _this->playAfterStop(subchnl);
        }
    }
    if (CD_AUDIO_STATUS_NONE == audioStatus ) //Fix for NCG3D-126735 - when the audio status is none, do not update the playtime
    {
        VARTRACE(audioStatus);
        return 0;
    }
    else
    {
    if (CD_AUDIO_STATUS_COMPLETED == audioStatus )
    {
        if(_this->mPreviousPlaybackState == PE_PBS_FASTFORWARDSTATE || _this->mPreviousPlaybackState == PE_PBS_FASTREVERSESTATE )
        {
            ETG_TRACE_USR4(("CDDADeviceInterface::TimerCallBack: FF/FR Scan completed"));
            _this->mPreviousPlaybackState = PE_PBS_UNKNOWN;
            _this->mScanEndElapsedTime = _this->mCurrentState.pos_ms;
        }
        else if((_this->mCurrentState.pos_ms == 0) || (_this->mScanEndElapsedTime != _this->mCurrentState.pos_ms))
        {
            ETG_TRACE_USR4(("CDDADeviceInterface::TimerCallBack: Current PLAY operation completed"));
            _this->mCurrentState.reason = REASON_EMPTY;
            _this->mCurrentState.pos_ms = 0L;
            _this->stopUpdateTimer();
        }
    }
    else if ((_this->mCurrentPlaybackState == PE_PBS_FASTFORWARDSTATE || 
            _this->mCurrentPlaybackState == PE_PBS_FASTREVERSESTATE) && trackChanged)    // NCG3D-22888, NCG3D-22887
    {
        ETG_TRACE_USR4(("CDDADeviceInterface::TimerCallBack: FFW/FRW reached end of current track !! operation completed !!"));
        _this->mCurrentState.reason = REASON_EMPTY;
        _this->mCurrentState.pos_ms = 0L;
        _this->stopUpdateTimer();
    }

    if(CD_AUDIO_STATUS_PROGRESS != _this->mAudioStatus && CD_AUDIO_STATUS_PROGRESS == audioStatus) // NCG3D-22900
    {
        ETG_TRACE_USR4(("CDDADeviceInterface::TimerCallBack: actual audio started so update Now Playing !!"));
        _this->mAudioStatus = audioStatus;
        _this->mCallbacks->resendNowPlaying(true);
    }

    /* call update callback method in CDDAControl */
    _this->mCallbacks->update(_this->mCurrentState);

    return 0;
    }
}

void CDDADeviceInterface::startUpdateTimer()
{
    ENTRY

    if (mTimerAlreadyRunning) {
        return;
    }
    mTimerAlreadyRunning = true;

    /* fires every 300 ms */
    mUpdateTimer.StartTimer(OUT mUpdateTimerID, IN 300, 300L, IN this, IN &TimerCallBack, IN (void *)NULL);

    printf("startUpdateTimer: mUpdateTimerID=%ld", mUpdateTimerID);

    /* give timer time to fire */
    //sleep(1);
    //Reduced the sleep time to 350 ms
    usleep(350000);
}

void CDDADeviceInterface::stopUpdateTimer()
{
    ENTRY

    if (!mTimerAlreadyRunning) {
        return;
    }
    mTimerAlreadyRunning = false;

    //printf("stopUpdateTimer: mUpdateTimerID=%ld", mUpdateTimerID);

    mUpdateTimer.CancelTimer(IN mUpdateTimerID);
}

void CDDADeviceInterface::update()
{
    mCallbacks->update(mCurrentState);
}

tResult CDDADeviceInterface::setCDText(unsigned int u32ATACDTextLen)
{
    ENTRY
    VARTRACE(u32ATACDTextLen);
    tResult u32Ret = MP_NO_ERROR;

    if(u32ATACDTextLen > CDAUDIO_ATA_CDTEXT_RAW_BUFFER_LEN)
    {
        ETG_TRACE_FATAL(("!!! Overflow for setCDText u32ATACDTextLen %d!!!",u32ATACDTextLen))
        return MP_ERR_CDDA_CDTEXT;
    }
    if(u32ATACDTextLen>=4)
    {
        LocalSPM::GetDataProvider().parseCddaResponseBuffer(&mResponseBuffer[4],u32ATACDTextLen,mTrackCount);

        MapCDDAMetadata Title;
        MapCDDAMetadata Artist;
        string Album;
        LocalSPM::GetDataProvider().getCddaMetaDataForTitle(OUT Album, OUT Title, mTrackCount);
        LocalSPM::GetDataProvider().getCddaMetaDataForArtist(OUT Artist, mTrackCount);

        if(Album.size())// album title
        {
            for(int index = 0; index < mTrackCount; index++)
            {
                strncpy_r(mCDContents[index].metaData4, Album.c_str(), sizeof(mCDContents[index].metaData4));
                LocalSPM::GetDataProvider().Convert2UTF8((FastUTF8::tString)(mCDContents[index].metaData4), sizeof(mCDContents[index].metaData4));
                ETG_TRACE_USR4(("Track %d Album %s ", index, mCDContents[index].metaData4));
            }
        }
        for(MapCDDAMetadata::iterator it = Title.begin();it!=Title.end();it++)
        {
            strncpy_r(mCDContents[it->first].metaData3, it->second.c_str(), sizeof(mCDContents[it->first].metaData3));
            LocalSPM::GetDataProvider().Convert2UTF8((FastUTF8::tString)(mCDContents[it->first].metaData3), sizeof(mCDContents[it->first].metaData3));
            ETG_TRACE_USR4(("Track %d Title %s ", it->first,mCDContents[it->first].metaData3));
        }
        for(MapCDDAMetadata::iterator it = Artist.begin();it!=Artist.end();it++)
        {
            strncpy_r(mCDContents[it->first].metaData2, it->second.c_str(), sizeof(mCDContents[it->first].metaData2));
            LocalSPM::GetDataProvider().Convert2UTF8((FastUTF8::tString)(mCDContents[it->first].metaData2), sizeof(mCDContents[it->first].metaData2));
            ETG_TRACE_USR4(("Track %d Artist %s ", it->first,mCDContents[it->first].metaData2));
        }
#if !defined(TARGET_BUILD)
        printf("Album = %s\n", Album.c_str());
        for(MapCDDAMetadata::iterator it = Title.begin(); it!=Title.end();it++)
        {
            printf("%d Title = %s\n", it->first,it->second.c_str());
        }
        for(MapCDDAMetadata::iterator it = Artist.begin(); it!=Artist.end();it++)
        {
            printf("%d Artist = %s\n", it->first,it->second.c_str());
        }
#endif
    }
    else
    {
        ETG_TRACE_FATAL(("!!!setCDText u32ATACDTextLen %d!!!",u32ATACDTextLen))
    }

    return u32Ret;
}

void CDDADeviceInterface::vZLBA2MSF(unsigned int u32ZLBA, tCDMSF *pMSF)
{
    ENTRY

    unsigned int u32Seconds;

    u32Seconds = (u32ZLBA / CD_SECTORS_PER_SECOND);

    if(NULL != pMSF)
    {
        pMSF->u8Min   = (unsigned char)(u32Seconds / 60);
        pMSF->u8Sec   = (unsigned char)(u32Seconds % 60);
        pMSF->u8Frm   = (unsigned char)(u32ZLBA % CD_SECTORS_PER_SECOND);
    }
    else  //if(NULL != pMSF)
    {
    } //else  //if(NULL != pMSF)
}

void CDDADeviceInterface::vMSFBCD2Decimal(const cdrom_msf0 &bcd, tCDMSF &dec)
{
    ENTRY

    dec.u8Min = (bcd.minute >> 4);
    dec.u8Min *= 10;
    dec.u8Min += (bcd.minute & 0x0f);

    dec.u8Sec = (bcd.second >> 4);
    dec.u8Sec *= 10;
    dec.u8Sec += (bcd.second & 0x0f);

    dec.u8Frm = (bcd.frame >> 4);
    dec.u8Frm *= 10;
    dec.u8Frm += (bcd.frame & 0x0f);
}


int CDDADeviceInterface::cdGetCmdLen(int SGIOCommand)
{
    static const int scsi_cdblen[8] = {6, 10, 10, 12, 12, 12, 10, 10};
    return scsi_cdblen[((SGIOCommand >> 5) & 7)];
}

tResult CDDADeviceInterface::SetSpeed(int speed)
{
    ENTRY

    tResult res = MP_NO_ERROR;

#if 0
#if 0
    /* does not work, but leave this in */
    cdIOControl(CDROM_SELECT_SPEED, (void *)speed);
#else {
    /* prepare my io buffer */
    prepare_SG_IO_Command(RESPONSE_ON, cdGetCmdLen(GPCMD_SET_SPEED));

    /* define the command */
    mCommandBuffer[0] = GPCMD_SET_SPEED;
    mCommandBuffer[1] = CD_32BIT_B1(speed); // just a guess
    mCommandBuffer[2] = CD_32BIT_B0(speed); // just a guess

    /* do the device command */
    res = run_SG_IO_Command();
#endif
#endif

    return res;
}

void CDDADeviceInterface::resetToc()
{
    ENTRY

    mTocRead = false;
    memset(&mToc,0,sizeof(mToc));
    if(mCDContents)
    {
        free(mCDContents);
    }
    mCDContents = NULL;
    mTrackCount = 0;
}

void CDDADeviceInterface::resetAudioStatus()
{
    ENTRY
    mAudioStatus = CD_AUDIO_STATUS_NONE;
}

int CDDADeviceInterface::convertBCDtoDecimal(char bcd)
{
    int dec = -1;
    if((((bcd & 0xF0) >> 4) < 10) && ((bcd & 0x0F)<10))
    {
        dec = ((bcd & 0xF0) >> 4) * 10 + (bcd & 0x0F);
    }
    else
    {
        ETG_TRACE_ERR(("The input :%d is not a valid BCD number",bcd));
    }

    return dec;
}
tResult CDDADeviceInterface::playAfterStop(cdrom_subchnl pSubChnl)
{
    ENTRY

    tResult res = MP_NO_ERROR;
    tCDMSF endTime = {0};
    tCDMSF msfScanPos = {0};

    vMSFBCD2Decimal(pSubChnl.cdsc_absaddr.msf,msfScanPos);
    cdIOControl(CDROMSTART, NULL);

    ETG_TRACE_USR4(("playAfterStop: MSF min : %d, sec : %d, frame : %d",msfScanPos.u8Min, msfScanPos.u8Sec,msfScanPos.u8Frm));
    /* end time */
    if (mCurrentTrackPlaying < mTrackCount) {
        vZLBA2MSF(mToc.arTrack[mCurrentTrackPlaying+1].u32StartZLBA, &endTime);
    } else {
        vZLBA2MSF(mToc.u32LastZLBA, &endTime);
    }
    ETG_TRACE_USR4(("playAfterStop: MSF min : %d, sec : %d, frame : %d",endTime.u8Min,endTime.u8Sec,endTime.u8Frm));

    ETG_TRACE_USR4(("playAfterStop::GPCMD_PLAY_AUDIO_MSF"));
    /* give command to scsi: */
    prepare_SG_IO_Command(RESPONSE_OFF, cdGetCmdLen(GPCMD_PLAY_AUDIO_MSF));

    /* define the command */
    mCommandBuffer[0] = GPCMD_PLAY_AUDIO_MSF;
    mCommandBuffer[3] = msfScanPos.u8Min;
    mCommandBuffer[4] = msfScanPos.u8Sec;
    mCommandBuffer[5] = msfScanPos.u8Frm;
    mCommandBuffer[6] = endTime.u8Min;
    mCommandBuffer[7] = endTime.u8Sec;
    mCommandBuffer[8] = endTime.u8Frm;

    /* do the device command */
    res = run_SG_IO_Command();
    if (res != MP_NO_ERROR) {
        return res;
    }
    mStartPlay = false;
    return res;
}
int CDDADeviceInterface::cdCtrlInfo(int command)
{
    ENTRY
    int fd = open(mDevSr, O_RDONLY | O_NONBLOCK);
    if (fd == -1) {
        ETG_TRACE_ERR(("CDDADeviceInterface::cdCtrlInfo: open error: %d/%s", errno, strerror(errno)));
        return MP_ERR_CDDA_OPEN_SR;
    }
    int ioInfo = 0;
    ioInfo = ioctl(fd, (unsigned long)command, 0);
    if (ioInfo == -1) {
        ETG_TRACE_ERR(("CDDADeviceInterface::cdCtrlInfo: ioctl error: %d/%s", errno, strerror(errno)));
    }
    ETG_TRACE_USR4(("cdCtrlInfo %d", ioInfo));
    close(fd);
    return ioInfo;
}

#if !defined(TARGET_BUILD) 
void CDDADeviceInterface::TestsetCDText(unsigned char *ref, tU16 size, tU8 totalTrack)
{
    ENTRY
    if(ref)
    {
        mTrackCount = totalTrack;
        memset(mResponseBuffer,  0, size);
        memcpy(mResponseBuffer,  ref, size);
        /* create a buffer for all CD info */
        mCDContents = (tFiles *)calloc(sizeof(tFiles), totalTrack+1); // +1: eof record
        setCDText(size);
    }
}
#endif
