#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_mp.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_CDDA_CONTROL
#ifdef TARGET_BUILD
#include "trcGenProj/Header/CDDAControl.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_CDDA_CONTROL
#endif
#endif

#include "LocalSPM.h"
#include "Dispatcher.h"
#include "FunctionTracer.h"
#include "MediaPlayer_ErrorCodes.h"
#include <sys/types.h>
#include <signal.h>
#include "FastUTF8.h"
#include "VarTrace.h"
#include "TimeTrace.h"

#include "CDDAControl.h"
#include "RequestResponseSM.h"
//SPM part
Lock CDDAControl::m_Mutex;
void CDDAControl::Create()
{
    ENTRY

    /* Create the state machine */
    CDDAControlSM::Create();

    CreateDone(0);
}

tResult CDDAControl::Init(tInitReason reason)
{
    ENTRY
    VARTRACE(reason);

    /* Init the state machine */
    CDDAControlSM::Init();
    SetAnswerTimeout(CDDA_CTRL_SM_ANSWER_TIMEOUT_MS); // Set answer timeout

    /* Register state machine with dispatcher */
    Dispatcher::GetInstance().Register(IN this);
    m_response = REASON_OK;
    return InitDone(0);
}

tResult CDDAControl::InitSM()
{
    ENTRY

    /* Initialize variables */
    m_Mutex.lock();
    InitMediaObject(OUT m_PlayMediaObject);
    m_Mutex.unlock();

    m_StartPosition = 0;
    mTocDataOneTrack = NULL;

    /* init the cdda device interface */
    mCDDADeviceInterface = new CDDADeviceInterface();
    mCDDADeviceInterface->init();
    mCDDADeviceInterface->registerCallbacks(&mCallbacks);

    mTocAvailabe = false;
    return MP_NO_ERROR;
}

tResult CDDAControl::Run()
{
    ENTRY

    /*Create and start threads */
    LocalSPM::GetThreadFactory().Do(IN this, 0, NULL); //CDDAControl inclusive state machine

    return RunDone(0);
}

void CDDAControl::Do(int functionID, void *ptr)
{
    ENTRY

    //set the threads name
    LocalSPM::GetThreadFactory().SetName(CDDAControlSM::GetSMNameFull());

    switch(functionID)
    {
        case 0:
        {
            while(CDDAControlSM::STATE_MACHINE_FINISHED != CDDAControlSM::StateMachine_Main()){}
            break;
        }
        default:
        {
            ETG_TRACE_ERR(("CDDAControl::Do: No thread defined for functionID: %d", functionID));
            break;
        }
    }
}

tResult CDDAControl::Stop()
{
    ENTRY
    tResult ret = MP_NO_ERROR;

    /* Send STOP message to own SM */
    ret = SendForceEvent(STOP_SM, (char *)NULL);
    if( MP_NO_ERROR != ret )
    {
        ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
    }

    return ret;
}

tResult CDDAControl::StopEventProcessed()
{
    ENTRY

    /* Send stop done to SPM in the transition to final state in state machine */
    return StopDone(0);
}

tResult CDDAControl::Done()
{
    ENTRY
    tResult ret = MP_NO_ERROR;

    /* Deregister state machine with dispatcher */
    Dispatcher::GetInstance().DeRegister(IN this);

    /* Send DONE message to own SM */
    ret = SendForceEvent(DONE, (char *)NULL);
    if( MP_NO_ERROR != ret )
    {
        ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
    }

    return ret;
}

tResult CDDAControl::DoneEventProcessed()
{
    ENTRY

    if( mTocDataOneTrack )
    {
        delete mTocDataOneTrack;
        mTocDataOneTrack = NULL;
    }
    if( mCDDADeviceInterface )
    {
        delete mCDDADeviceInterface;
        mCDDADeviceInterface = NULL;
    }

    //Cleanup threads
    LocalSPM::GetThreadFactory().CleanUpThreads(this);

    /* Send done done to SPM in the transition to final state in state machine */
    return DoneDone(0);
}

char *CDDAControl::GetSMStateName(tGeneralString stateName, size_t size)
{
    GetCurrentState((char *)stateName, size);
    return stateName;
}


//Playback control part

tResult CDDAControl::StartAllocateAudioInput(const tDeviceID /*deviceID*/, const tMountPoint /*mountPoint*/)
{
    ENTRY

    tResult ret = MP_NO_ERROR;

    /* Fake PlayerEngine response: Send expected event to waiting state machine immediately */
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);
    tReturnValue returnValue = true;

    ret = ParameterMETHOD_RETURN(OUT parameterString, IN size, IN returnValue);
    if( MP_NO_ERROR != ret )
    {
        ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
        parameterString[0] = '\0';
    }

    ret = SendEvent(METHOD_RETURN, IN parameterString);
    if( MP_NO_ERROR != ret )
    {
        ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult CDDAControl::StartDeAllocateAudioInput()
{
    ENTRY

    tResult ret = MP_NO_ERROR;

    /* For CDDA no need to deallocate audio input device */
    /* Fake PlayerEngine response: Send expected event to waiting state machine immediately */
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);
    tReturnValue returnValue = true;

    ret = ParameterMETHOD_RETURN(OUT parameterString, IN size, IN returnValue);
    if( MP_NO_ERROR != ret )
    {
        ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
        parameterString[0] = '\0';
    }

    ret = SendEvent(METHOD_RETURN, IN parameterString);
    if( MP_NO_ERROR != ret )
    {
        ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult CDDAControl::StartSwitchObserver(const tDeviceID /*deviceID*/, const tMountPoint /*mountPoint*/)
{
    ENTRY

    tResult ret = MP_NO_ERROR;

    /* nobody will not answer, so fake the answer */
    ret = SendEvent(METHOD_RETURN, IN "0");
    if( MP_NO_ERROR != ret )
    {
        ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult CDDAControl::StartStreaming(const tDeviceID /*deviceID*/, const tMountPoint /*mountPoint*/)
{
    ENTRY

    tResult ret = MP_NO_ERROR;
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);
    tPEPlaybackState playbackState = PE_PBS_LOADINGSTATE;

    /* For CDDA no need to start streaming */
    if(LocalSPM::GetDataProvider().UseMediaEngine())
    {
        /* Fake MediaEngine response: Send expected event to waiting state machine immediately */
        me::reason_e reason = REASON_OK;
        me::speed_e speed = ME_SPEED_NORMAL;

        m_Mutex.lock();
        tPEHandle handle = (tPEHandle)m_PlayMediaObject.objectID;
        m_Mutex.unlock();

        ret = ParameterSTART_STREAMING_ANSWER(OUT parameterString, IN size, IN handle, IN playbackState, IN reason, IN speed);
        if (MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
            parameterString[0] = '\0';
        }

        ret = SendEvent(START_STREAMING_ANSWER, IN parameterString);
        if (MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
        }
    }
    else
    {
        /* Fake PlayerEngine response: Send expected event to waiting state machine immediately */
        tMetadata metadata = { 0 };

        tObjectID ObjectID = OBJECT_ID_NONE;
        ret = ParameterPLAYBACK_STATUS_RESPONSE(OUT parameterString, IN size, IN playbackState, IN metadata, IN metadata, IN metadata, IN metadata,IN ObjectID);
        if (MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
            parameterString[0] = '\0';
        }

        ret = SendEvent(PLAYBACK_STATUS_RESPONSE, IN parameterString);
        if (MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
        }
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult CDDAControl::StartStopStreaming()
{
    ENTRY

    tResult ret = MP_NO_ERROR;
    tDeviceID deviceID = DEVICE_ID_NOT_SET;
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    /* For CDDA no need to stop streaming
     * Send STOP_STREAMING_ANSWER message to release own waiting state */
    ret = ParameterSTOP_STREAMING_ANSWER(OUT parameterString, IN size, IN deviceID);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
        parameterString[0] = '\0';
    }

    ret = SendEvent(STOP_STREAMING_ANSWER, IN parameterString);
    if(MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult CDDAControl::StopStreamingAnswer(tDeviceID deviceID)
{
    ENTRY
    (void)deviceID;
    tPEPlaybackState playbackState = PE_PBS_STOPPEDSTATE;

    /* Send answer to waiting state machine */
    if( LocalSPM::GetDataProvider().UseMediaEngine() )
    {
        m_Mutex.lock();
        tPEHandle handle = (tPEHandle)m_PlayMediaObject.objectID;
        m_Mutex.unlock();

        PlaybackStatusNew(IN handle, IN playbackState, IN REASON_OK, IN ME_SPEED_NORMAL);
    }
    else
    {
        tMetadata metadata = { 0 };

        PlaybackStatus(IN playbackState, IN metadata, IN metadata, IN metadata, IN metadata);
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult CDDAControl::StartPlay(const tDeviceType deviceType,
                              const tDeviceID deviceID,
                              const tURL URL,
                              const tMountPoint mountPoint,
                              const tUUID uuid,
                              const tPEHandle handle,
                              const tPlaytime position,
                              const tStreaming streaming)
{
    ENTRY
    VARTRACE4(deviceType, deviceID, URL, mountPoint);
    (void)uuid;
    VARTRACE(handle);
    VARTRACE(position);
    VARTRACE(streaming);

    /* Update media object to play */
    m_Mutex.lock();
    InitMediaObject(OUT m_PlayMediaObject);
    m_PlayMediaObject.objectID = (tObjectID)handle;
    m_PlayMediaObject.deviceID = deviceID;
    strncpy_r(OUT m_PlayMediaObject.fileName, IN URL, IN sizeof(m_PlayMediaObject.fileName));
    strncpy_r(OUT m_PlayMediaObject.mountPoint, IN mountPoint, IN sizeof(m_PlayMediaObject.mountPoint));
    m_Mutex.unlock();

    m_StartPosition = position;

    /* Put complete path together (mount point + relative path of the URL)*/
    tURL mountPointURL;
    strncpy_r(OUT mountPointURL, IN mountPoint, IN sizeof(mountPointURL));
    strncat_r(OUT mountPointURL, IN URL, IN sizeof(mountPointURL));

    /* @@@ read metadata */

    /* Send METADATA_FOR_PLAYBACK message to own SM */
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    tResult ret;
    ret = ParameterMETADATA_FOR_PLAYBACK(OUT parameterString, IN size, IN handle, IN mountPointURL);
    if( MP_NO_ERROR != ret )
    {
        ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
        parameterString[0] = '\0';
    }

    ret = SendEvent(METADATA_FOR_PLAYBACK, IN parameterString);
    if( MP_NO_ERROR != ret )
    {
        ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult CDDAControl::StartPlayWithMetadata(const tPEHandle handle, const tURL mountPointURL)
{
    ENTRY
    VARTRACE(handle);
    VARTRACE(mountPointURL);

    tResult ret = MP_NO_ERROR;

    m_Mutex.lock();
    tObjectID objectID = m_PlayMediaObject.objectID;
    tNotPlayable notPlayable = m_PlayMediaObject.notPlayable;
    m_Mutex.unlock();

    if((tObjectID)handle != objectID)
    {
        ETG_TRACE_ERR(("Received metadata of wrong object (objectID (received/expected):%u/%u)", (tObjectID)handle, objectID));

        /* Send action error to own state machine */
        ActionError(REASON_ERROR);
        return MP_NO_ERROR;
    }

    /* In case of file is DRM protected send action error */
    if(FNP_DRM_PROTECTED == notPlayable)
    {
        ETG_TRACE_ERR(("Skip playback of file since it is DRM protected (%s)", mountPointURL));

        /* Send action error to own state machine */
        ActionError(REASON_DRM_ERROR);
        return MP_NO_ERROR;
    }
    else if(FNP_PLAY_RESTRICTION == notPlayable)
    {
        ETG_TRACE_ERR(("Skip playback of file since it is Codec restricted (%s)", mountPointURL));

        /* Send action error to own state machine */
        ActionError(REASON_CODEC_ERROR);
        return MP_NO_ERROR;
    }
    else if(FNP_FORMAT_ERROR == notPlayable)
    {
        ETG_TRACE_ERR(("Skip playback of file since it is not readable (%s)", mountPointURL));

        /* Send action error to own state machine */
        ActionError(REASON_FORMAT_ERROR);
        return MP_NO_ERROR;
    }
    else
    {
        /* start play for this song */
        mCDDADeviceInterface->play(IN mountPointURL, IN (me::speed_e)1, IN m_StartPosition, IN handle);

        /* Send event to own waiting state machine */
        tAllParameters parameterString;
        size_t size = sizeof(parameterString);
        tPEHandle handle;
        tPEPlaybackState playbackState;
        me::reason_e reason;
        me::speed_e speed;

        mCDDADeviceInterface->getParameter(OUT handle, OUT playbackState, OUT reason, OUT speed);

        if(LocalSPM::GetDataProvider().UseMediaEngine())
        {
            /* Fake MediaEngine response: PLAY_ANSWER */
            ret = ParameterPLAY_ANSWER(OUT parameterString, IN size, IN handle, IN playbackState, IN reason, IN speed);
            if (MP_NO_ERROR != ret)
            {
                ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
                parameterString[0] = '\0';
            }

            ret = SendEvent(PLAY_ANSWER, IN parameterString);
            if (MP_NO_ERROR != ret)
            {
                ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
            }
        }
        else
        {
            /* Fake PlayerEngine response: PLAYBACK_STATUS_RESPONSE */
            tMetadata metadata = { 0 };

            tObjectID ObjectID = OBJECT_ID_NONE;
            ret = ParameterPLAYBACK_STATUS_RESPONSE(OUT parameterString, IN size, IN playbackState, IN metadata, IN metadata, IN metadata, IN metadata,IN ObjectID);
            if (MP_NO_ERROR != ret)
            {
                ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
                parameterString[0] = '\0';
            }

            ret = SendEvent(PLAYBACK_STATUS_RESPONSE, IN parameterString);
            if (MP_NO_ERROR != ret)
            {
                ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
            }
        }
        //NCG3D-22900
        if(mCDDADeviceInterface)
        {
            mCDDADeviceInterface->update();
            mCDDADeviceInterface->startUpdateTimer();
            mCDDADeviceInterface->resetAudioStatus();
        }
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult CDDAControl::StartStop(const tDeviceType deviceType,
                              const tDeviceID deviceID,
                              const tURL URL,
                              const tMountPoint mountPoint)
{
    ENTRY
    VARTRACE4(deviceType, deviceID, URL, mountPoint);

    tResult ret = MP_NO_ERROR;

    mCDDADeviceInterface->stop();

    /* fake STOP_ANSWER */
    ret = SendEvent(STOP_ANSWER, NULL); // internal event
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
    }

    /* Reset media object to play */
    ETG_TRACE_USR3(("Reset media object to play"));
    m_Mutex.lock();
    InitMediaObject(OUT m_PlayMediaObject);
    m_Mutex.unlock();

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult CDDAControl::StartPause(const tDeviceType deviceType,
                               const tDeviceID deviceID,
                               const tURL URL,
                               const tMountPoint mountPoint)
{
    ENTRY
    VARTRACE4(deviceType, deviceID, URL, mountPoint);

    tResult ret = MP_NO_ERROR;

    mCDDADeviceInterface->pause();

    /* Send event to own waiting state machine */
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);
    tPEHandle handle;
    tPEPlaybackState playbackState;
    me::reason_e reason;
    me::speed_e speed;

    mCDDADeviceInterface->getParameter(OUT handle, OUT playbackState, OUT reason, OUT speed);

    if(LocalSPM::GetDataProvider().UseMediaEngine())
    {
        /* Fake MediaEngine response: PAUSE_ANSWER */
        ret = ParameterPAUSE_ANSWER(OUT parameterString, IN size, IN handle, IN playbackState, IN reason, IN speed);
        if (MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
            parameterString[0] = '\0';
        }

        ret = SendEvent(PAUSE_ANSWER, IN parameterString);
        if (MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
        }
    }
    else
    {
        /* Fake PlayerEngine response: PLAYBACK_STATUS_RESPONSE */
        tMetadata metadata = { 0 };

        tObjectID ObjectID = OBJECT_ID_NONE;
        ret = ParameterPLAYBACK_STATUS_RESPONSE(OUT parameterString, IN size, IN playbackState, IN metadata, IN metadata, IN metadata, IN metadata,IN ObjectID);
        if (MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
            parameterString[0] = '\0';
        }

        ret = SendEvent(PLAYBACK_STATUS_RESPONSE, IN parameterString);
        if (MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
        }
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult CDDAControl::StartResume(const tDeviceType deviceType,
                                const tDeviceID deviceID,
                                const tURL URL,
                                const tMountPoint mountPoint)
{
    ENTRY
    VARTRACE4(deviceType, deviceID, URL, mountPoint);

    tResult ret = MP_NO_ERROR;

    mCDDADeviceInterface->resume();

    /* Send event to own waiting state machine */
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);
    tPEHandle handle;
    tPEPlaybackState playbackState;
    me::reason_e reason;
    me::speed_e speed;

    mCDDADeviceInterface->getParameter(OUT handle, OUT playbackState, OUT reason, OUT speed);

    if(LocalSPM::GetDataProvider().UseMediaEngine())
    {
        /* Fake MediaEngine response: RESUME_ANSWER */
        ret = ParameterRESUME_ANSWER(OUT parameterString, IN size, IN handle, IN playbackState, IN reason, IN speed);
        if (MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
            parameterString[0] = '\0';
        }

        ret = SendEvent(RESUME_ANSWER, IN parameterString);
        if (MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
        }
    }
    else
    {
        /* Fake PlayerEngine response: PLAYBACK_STATUS_RESPONSE */
        tMetadata metadata = { 0 };

        tObjectID ObjectID = OBJECT_ID_NONE;
        ret = ParameterPLAYBACK_STATUS_RESPONSE(OUT parameterString, IN size, IN playbackState, IN metadata, IN metadata, IN metadata, IN metadata,IN ObjectID);
        if (MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
            parameterString[0] = '\0';
        }

        ret = SendEvent(PLAYBACK_STATUS_RESPONSE, IN parameterString);
        if (MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
        }
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult CDDAControl::StartFfwdStart(const tDeviceType deviceType,
                                   const tDeviceID deviceID,
                                   const tURL URL,
                                   const tMountPoint mountPoint,
                                   const tCueingRate rate,
                                   const speedstate_e IsPlaybackSpeed)
{
    ENTRY
    (void)IsPlaybackSpeed;
    VARTRACE4(deviceType, deviceID, URL, mountPoint);
    VARTRACE(rate);

    tResult ret = MP_NO_ERROR;

    mCDDADeviceInterface->ffwdStart(rate);

    /* Send event to own waiting state machine */
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);
    tPEHandle handle;
    tPEPlaybackState playbackState;
    me::reason_e reason;
    me::speed_e speed;

    mCDDADeviceInterface->getParameter(OUT handle, OUT playbackState, OUT reason, OUT speed);

    if(LocalSPM::GetDataProvider().UseMediaEngine())
    {
        /* Fake MediaEngine response: FFWD_ANSWER */
        ret = ParameterFFWD_ANSWER(OUT parameterString, IN size, IN handle, IN playbackState, IN reason, IN speed);
        if (MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
            parameterString[0] = '\0';
        }

        ret = SendEvent(FFWD_ANSWER, IN parameterString);
        if (MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
        }
    }
    else
    {
        /* Fake PlayerEngine response: PLAYBACK_STATUS_RESPONSE */
        tMetadata metadata = { 0 };

        tObjectID ObjectID = OBJECT_ID_NONE;
        ret = ParameterPLAYBACK_STATUS_RESPONSE(OUT parameterString, IN size, IN playbackState, IN metadata, IN metadata, IN metadata, IN metadata,IN ObjectID);
        if (MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
            parameterString[0] = '\0';
        }

        ret = SendEvent(PLAYBACK_STATUS_RESPONSE, IN parameterString);
        if (MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
        }
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult CDDAControl::StartFfwdStop(const tDeviceType deviceType,
                                  const tDeviceID deviceID,
                                  const tURL URL,
                                  const tMountPoint mountPoint,
                                  const speedstate_e IsPlaybackSpeed)
{
    ENTRY
    (void)IsPlaybackSpeed;
    VARTRACE4(deviceType, deviceID, URL, mountPoint);

    tResult ret = MP_NO_ERROR;

    mCDDADeviceInterface->ffwdStop();

    /* Send event to own waiting state machine */
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);
    tPEHandle handle;
    tPEPlaybackState playbackState;
    me::reason_e reason;
    me::speed_e speed;

    mCDDADeviceInterface->getParameter(OUT handle, OUT playbackState, OUT reason, OUT speed);

    if(LocalSPM::GetDataProvider().UseMediaEngine())
    {
        /* Fake MediaEngine response: FFWD_STOP_ANSWER */
        ret = ParameterFFWD_STOP_ANSWER(OUT parameterString, IN size, IN handle, IN playbackState, IN reason, IN speed);
        if (MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
            parameterString[0] = '\0';
        }

        ret = SendEvent(FFWD_STOP_ANSWER, IN parameterString);
        if (MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
        }
    }
    else
    {
        /* Fake PlayerEngine response: PLAYBACK_STATUS_RESPONSE */
        tMetadata metadata = { 0 };

        tObjectID ObjectID = OBJECT_ID_NONE;
        ret = ParameterPLAYBACK_STATUS_RESPONSE(OUT parameterString, IN size, IN playbackState, IN metadata, IN metadata, IN metadata, IN metadata,IN ObjectID);
        if (MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
            parameterString[0] = '\0';
        }

        ret = SendEvent(PLAYBACK_STATUS_RESPONSE, IN parameterString);
        if (MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
        }
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult CDDAControl::StartFrevStart(const tDeviceType deviceType,
                                   const tDeviceID deviceID,
                                   const tURL URL,
                                   const tMountPoint mountPoint,
                                   const tCueingRate rate,
                                   const speedstate_e IsPlaybackSpeed)
{
    ENTRY
    (void)IsPlaybackSpeed;
    VARTRACE4(deviceType, deviceID, URL, mountPoint);
    VARTRACE(rate);

    tResult ret = MP_NO_ERROR;

    mCDDADeviceInterface->frevStart(rate);

    /* Send event to own waiting state machine */
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);
    tPEHandle handle;
    tPEPlaybackState playbackState;
    me::reason_e reason;
    me::speed_e speed;

    mCDDADeviceInterface->getParameter(OUT handle, OUT playbackState, OUT reason, OUT speed);

    if(LocalSPM::GetDataProvider().UseMediaEngine())
    {
        /* Fake MediaEngine response: FREV_ANSWER */
        ret = ParameterFREV_ANSWER(OUT parameterString, IN size, IN handle, IN playbackState, IN reason, IN speed);
        if (MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
            parameterString[0] = '\0';
        }

        ret = SendEvent(FREV_ANSWER, IN parameterString);
        if (MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
        }
    }
    else
    {
        /* Fake PlayerEngine response: PLAYBACK_STATUS_RESPONSE */
        tMetadata metadata = { 0 };

        tObjectID ObjectID = OBJECT_ID_NONE;
        ret = ParameterPLAYBACK_STATUS_RESPONSE(OUT parameterString, IN size, IN playbackState, IN metadata, IN metadata, IN metadata, IN metadata,IN ObjectID);
        if (MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
            parameterString[0] = '\0';
        }

        ret = SendEvent(PLAYBACK_STATUS_RESPONSE, IN parameterString);
        if (MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
        }
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult CDDAControl::StartFrevStop(const tDeviceType deviceType,
                                  const tDeviceID deviceID,
                                  const tURL URL,
                                  const tMountPoint mountPoint,
                                  const speedstate_e IsPlaybackSpeed)
{
    ENTRY
    (void)IsPlaybackSpeed;
    VARTRACE4(deviceType, deviceID, URL, mountPoint);

    tResult ret = MP_NO_ERROR;

    mCDDADeviceInterface->frevStop();

    /* Send event to own waiting state machine */
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);
    tPEHandle handle;
    tPEPlaybackState playbackState;
    me::reason_e reason;
    me::speed_e speed;

    mCDDADeviceInterface->getParameter(OUT handle, OUT playbackState, OUT reason, OUT speed);

    if(LocalSPM::GetDataProvider().UseMediaEngine())
    {
        /* Fake MediaEngine response: FREV_STOP_ANSWER */
        ret = ParameterFREV_STOP_ANSWER(OUT parameterString, IN size, IN handle, IN playbackState, IN reason, IN speed);
        if (MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
            parameterString[0] = '\0';
        }

        ret = SendEvent(FREV_STOP_ANSWER, IN parameterString);
        if (MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
        }
    }
    else
    {
        /* Fake PlayerEngine response: PLAYBACK_STATUS_RESPONSE */
        tMetadata metadata = { 0 };

        tObjectID ObjectID = OBJECT_ID_NONE;
        ret = ParameterPLAYBACK_STATUS_RESPONSE(OUT parameterString, IN size, IN playbackState, IN metadata, IN metadata, IN metadata, IN metadata,IN ObjectID);
        if (MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
            parameterString[0] = '\0';
        }

        ret = SendEvent(PLAYBACK_STATUS_RESPONSE, IN parameterString);
        if (MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
        }
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult CDDAControl::StartSeekTo(const tDeviceType deviceType,
                                const tDeviceID deviceID,
                                const tURL URL,
                                const tMountPoint mountPoint,
                                const tPlaytime position)
{
    ENTRY
    VARTRACE4(deviceType, deviceID, URL, mountPoint);
    VARTRACE(position);

    tResult ret = MP_NO_ERROR;

    mCDDADeviceInterface->seek((me::speed_e)1, position);

    /* Fake response: SEEK_TO_ANSWER */
    m_Mutex.lock();
    tPEHandle handle = (tPEHandle)m_PlayMediaObject.objectID;
    m_Mutex.unlock();

    tPETimeInfoStruct timeInfoStruct;
    InitPETimeInfoStruct(timeInfoStruct);
    timeInfoStruct.position.ms = (tPEMilliseconds)position;
    timeInfoStruct.duration.ms = (tPEMilliseconds)PLAYTIME_NONE;

    /* Marshal struct of timeInfo into a string */
    tPETimeInfo timeInfoString;
    size_t size = sizeof(timeInfoString);
    SMF::Marshal(OUT timeInfoString,
                 IN size-1,
                 IN DOUBLE_MARSHAL_SEPARATOR,
                 IN tPEBytes_format tPEPercentage_format tPEMilliseconds_format tPEBytes_format tPEMilliseconds_format,
                 IN timeInfoStruct.position.bytes,
                 IN timeInfoStruct.position.pct,
                 IN timeInfoStruct.position.ms,
                 IN timeInfoStruct.duration.bytes,
                 IN timeInfoStruct.duration.ms);

    /* Send SEEK_TO_ANSWER event to own waiting state machine */
    tAllParameters parameterString;
    size = sizeof(parameterString);
    ret = ParameterSEEK_TO_ANSWER(OUT parameterString, size, IN handle, IN timeInfoString);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
        parameterString[0] = '\0';
    }

    ret = SendEvent(SEEK_TO_ANSWER, IN parameterString);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult CDDAControl::StartBuffer(const tDeviceType deviceType,
                                const tDeviceID deviceID,
                                const tURL URL,
                                const tMountPoint mountPoint,
                                const tPEHandle handle)
{
    ENTRY
    VARTRACE4(deviceType, deviceID, URL, mountPoint);
    VARTRACE(handle);

    //@@@ code?

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult CDDAControl::DiscardBuffer()
{
    ENTRY
    tResult ret = MP_NO_ERROR;

    //@@ code??

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return ret;
}

tResult CDDAControl::ActionStatus(const tReturnValue returnValue)
{
    ENTRY
    VARTRACE(returnValue);

    tResult ret = MP_NO_ERROR;

    /* Send answer to waiting state machine */
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    ret = LocalSPM::GetDeviceDispatcher().ParameterMETHOD_RETURN(OUT parameterString, IN size, IN returnValue);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
        parameterString[0] = '\0';
    }

    ret = SendAnswer(IN parameterString);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while sending answer via SMF (ErrorCode:%s)", errorString(ret)));
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult CDDAControl::BufferStatus(const tSuccess success)
{
    ENTRY
    VARTRACE(success);

    tResult ret = MP_NO_ERROR;

    /* Send answer to waiting state machine */
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

#if 0
    /* map reason to success */
    tSuccess success = SC_NO;
    if (reason >= REASON_OK) {
        success = SC_YES;
    }
#endif

    ret = LocalSPM::GetPlayerManager().ParameterBUFFER_ANSWER(OUT parameterString, IN size, IN success);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
        parameterString[0] = '\0';
    }

    ret = SendAnswer(IN parameterString);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while sending answer via SMF (ErrorCode:%s)", errorString(ret)));
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult CDDAControl::PlaybackStatus(const tPEPlaybackState status,
                                   const tMetadata metadata1,
                                   const tMetadata metadata2,
                                   const tMetadata metadata3,
                                   const tMetadata metadata4,
                                   const tObjectID ObjectID)
{
    ENTRY
    VARTRACE6(status, metadata1, metadata2, metadata3, metadata4, MTY_UNKNOWN);

    tResult ret = MP_NO_ERROR;
    m_Mutex.lock();
    tMediaObject mediaObject =  m_PlayMediaObject;
    m_Mutex.unlock();

    /* Send answer to waiting state machine */
    if( PE_PBS_ERRORSTATE == status )
    {
        /* Validate error code */
        me::reason_e reason;
        ret = ValidateErrorCode(OUT reason, IN metadata2, IN mediaObject.mountPoint);

        /* Send action error instead of answer to waiting state machine */
        HandleActionError(IN reason);
    }
    else
    {
        tAllParameters parameterString;
        size_t size = sizeof(parameterString);

        ret = LocalSPM::GetDeviceDispatcher().ParameterPLAY_ANSWER(OUT parameterString,
                IN size,
                IN status,
                IN mediaObject.MetadataField1,
                IN mediaObject.MetadataField2,
                IN mediaObject.MetadataField3,
                IN mediaObject.MetadataField4,
                IN mediaObject.title,
                IN mediaObject.mediaType,
                IN mediaObject.metadataConvertFlag,
                IN mediaObject.UUID);
        if (MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
            parameterString[0] = '\0';
        }

        ret = SendAnswer(IN parameterString);
        if (MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while sending answer via SMF (ErrorCode:%s)", errorString(ret)));
        }
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult CDDAControl::ForwardPlaybackStatus(const tPEPlaybackState status,
                                          const tMetadata metadata1,
                                          const tMetadata metadata2,
                                          const tMetadata metadata3,
                                          const tMetadata metadata4,
                                          const tObjectID ObjectID)
{
    ENTRY
    VARTRACE6(status, metadata1, metadata2, metadata3, metadata4, MTY_UNKNOWN);

    tResult ret = MP_NO_ERROR;
    m_Mutex.lock();
    tMediaObject mediaObject =  m_PlayMediaObject;
    m_Mutex.unlock();

    if( PE_PBS_ERRORSTATE == status )
    {
        /* Validate error code */
        me::reason_e reason;
        ret = ValidateErrorCode(OUT reason, IN metadata2, IN mediaObject.mountPoint);

        /* Forward stream error */
        ForwardStreamError(IN reason);
    }
    else
    {
        /* Send PLAYBACK_STATUS_RESPONSE message to PlayerManger to update playback state */
        char messageString[64];
        strncpy_r(OUT messageString, IN "PlayerManagerSM::PLAYBACK_STATUS_RESPONSE", IN sizeof(messageString));

        tAllParameters parameterString;
        size_t size = sizeof(parameterString);
        tObjectID ObjectID = OBJECT_ID_NONE;

        ret = LocalSPM::GetPlayerManager().ParameterPLAYBACK_STATUS_RESPONSE(OUT parameterString,
                IN size,
                IN status,
                IN mediaObject.MetadataField1,
                IN mediaObject.MetadataField2,
                IN mediaObject.MetadataField3,
                IN mediaObject.MetadataField4,
                IN mediaObject.title,
                IN mediaObject.mediaType,
                IN mediaObject.metadataConvertFlag,
                IN mediaObject.UUID,
                IN ObjectID);

    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
        parameterString[0] = '\0';
    }

    ret = Dispatcher::GetInstance().SendMessage(IN messageString, IN parameterString);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while sending message via SMF (ErrorCode:%s)", errorString(ret)));
        }
    }

    return ret;
}

tResult CDDAControl::PlaytimeStatus(const tPlaytime elapsedPlaytime, const tPlaytime totalPlaytime)
{
    ENTRY
    VARTRACE2(elapsedPlaytime, totalPlaytime);

#if 0
    tResult ret = MP_NO_ERROR;

    /* Send answer to waiting state machine */
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    ret = LocalSPM::GetPlayerManager().ParameterSEEK_TO_ANSWER(OUT parameterString, IN size, IN elapsedPlaytime, IN totalPlaytime);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
        parameterString[0] = '\0';
    }

    ret = SendAnswer(IN parameterString);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while sending message via SMF (ErrorCode:%s)", errorString(ret)));
    }
#else
    m_Mutex.lock();
    tPEHandle handle = (tPEHandle)m_PlayMediaObject.objectID;
    m_Mutex.unlock();

    tPETimeInfoStruct timeInfoStruct;
    InitPETimeInfoStruct(timeInfoStruct);
    timeInfoStruct.position.ms = (tPEMilliseconds)elapsedPlaytime;
    timeInfoStruct.duration.ms = (tPEMilliseconds)totalPlaytime;

    /* Marshal struct of timeInfo into a string */
    tPETimeInfo timeInfoString;
    size_t size = sizeof(timeInfoString);
    SMF::Marshal(OUT timeInfoString,
                 IN size-1,
                 IN DOUBLE_MARSHAL_SEPARATOR,
                 IN tPEBytes_format tPEPercentage_format tPEMilliseconds_format tPEBytes_format tPEMilliseconds_format,
                 IN timeInfoStruct.position.bytes,
                 IN timeInfoStruct.position.pct,
                 IN timeInfoStruct.position.ms,
                 IN timeInfoStruct.duration.bytes,
                 IN timeInfoStruct.duration.ms);

    PlaytimeStatusNew(IN handle, IN timeInfoString);
#endif

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult CDDAControl::ForwardPlaytimeStatus(const tPlaytime elapsedPlaytime, const tPlaytime totalPlaytime)
{
    ENTRY
    VARTRACE2(elapsedPlaytime, totalPlaytime);

    tResult ret = MP_NO_ERROR;

#if 0
    /* Send TICK_TIME_ELAPSED message to PlayerManger to update playtime */
    char messageString[64];
    strncpy_r(OUT messageString, IN "PlayerManagerSM::TICK_TIME_ELAPSED", IN sizeof(messageString));
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    ret = LocalSPM::GetPlayerManager().ParameterTICK_TIME_ELAPSED(OUT parameterString, IN size, IN elapsedPlaytime, IN totalPlaytime);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
        parameterString[0] = '\0';
    }

    ret = Dispatcher::GetInstance().SendMessage(IN messageString, IN parameterString);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while sending message via SMF (ErrorCode:%s)", errorString(ret)));
    }
#else
    m_Mutex.lock();
    tPEHandle handle = (tPEHandle)m_PlayMediaObject.objectID;
    m_Mutex.unlock();

    tPETimeInfoStruct timeInfoStruct;
    InitPETimeInfoStruct(timeInfoStruct);
    timeInfoStruct.position.ms = (tPEMilliseconds)elapsedPlaytime;
    timeInfoStruct.duration.ms = (tPEMilliseconds)totalPlaytime;

    /* Marshal struct of timeInfo into a string */
    tPETimeInfo timeInfoString;
    size_t size = sizeof(timeInfoString);
    SMF::Marshal(OUT timeInfoString,
                 IN size-1,
                 IN DOUBLE_MARSHAL_SEPARATOR,
                 IN tPEBytes_format tPEPercentage_format tPEMilliseconds_format tPEBytes_format tPEMilliseconds_format,
                 IN timeInfoStruct.position.bytes,
                 IN timeInfoStruct.position.pct,
                 IN timeInfoStruct.position.ms,
                 IN timeInfoStruct.duration.bytes,
                 IN timeInfoStruct.duration.ms);

    ret = ForwardPlaytimeStatusNew(IN handle, IN timeInfoString);
#endif

    return ret;
}

tResult CDDAControl::HandleActionError(const me::reason_e reason)
{
    ENTRY_INTERNAL

    tResult ret = MP_NO_ERROR;
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    /* Send action error instead of answer to waiting state machine */
    ret = LocalSPM::GetDeviceDispatcher().ParameterACTION_ERROR(OUT parameterString, IN size, IN reason);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
        parameterString[0] = '\0';
    }

    ret = SendActionError(IN parameterString);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while sending action error via SMF (ErrorCode:%s)", errorString(ret)));
    }

    return ret;
}

tResult CDDAControl::ActionError(const me::reason_e reason)
{
    ENTRY_INTERNAL

    tResult ret = MP_NO_ERROR;
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    /* Send action error to own state machine*/
    ret = ParameterACTION_ERROR(OUT parameterString, IN size, IN reason);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
        parameterString[0] = '\0';
    }

    ret = SendEvent(ACTION_ERROR ,IN parameterString);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while sending action error via SMF (ErrorCode:%s)", errorString(ret)));
    }

    return ret;
}

tResult CDDAControl::ForwardStreamError(const me::reason_e reason) const
{
    ENTRY_INTERNAL

    tResult ret = MP_NO_ERROR;
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    /* Send STREAM_ERROR message to PlayerManger */
    char messageString[64];
    strncpy_r(OUT messageString, IN "PlayerManagerSM::STREAM_ERROR", IN sizeof(messageString));

    ret = LocalSPM::GetPlayerManager().ParameterSTREAM_ERROR(OUT parameterString, IN size, IN reason);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
        parameterString[0] = '\0';
    }

    ret = Dispatcher::GetInstance().SendMessage(IN messageString, IN parameterString);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while sending message via SMF (ErrorCode:%s)", errorString(ret)));
    }

    return ret;
}

tResult CDDAControl::PlaybackStatusNew(const tPEHandle handle, const tPEPlaybackState playbackState, const me::reason_e reason, const me::speed_e speed)
{
    ENTRY
    VARTRACE(handle);
    VARTRACE(playbackState);
    VARTRACE(reason);
    VARTRACE(speed);

    tResult ret = MP_NO_ERROR;

    if(REASON_OK > reason)
    {
        /* Send action error instead of answer to waiting state machine */
        HandleActionError(IN reason);
    }
    else
    {
        /* Send answer to waiting state machine */
        tAllParameters parameterString;
        size_t size = sizeof(parameterString);

        ret = LocalSPM::GetDeviceDispatcher().ParameterPLAY_ANSWER_NEW(OUT parameterString, IN size, IN handle, IN playbackState, IN reason, IN speed);
        if (MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
            parameterString[0] = '\0';
        }

        ret = SendAnswer(IN parameterString);
        if (MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while sending answer via SMF (ErrorCode:%s)", errorString(ret)));
        }
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult CDDAControl::ForwardPlaybackStatusNew(const tPEHandle handle, const tPEPlaybackState playbackState, const me::reason_e reason, const me::speed_e speed)
{
    ENTRY
    VARTRACE(handle);
    VARTRACE(playbackState);
    VARTRACE(reason);
    VARTRACE(speed);

    tResult ret = MP_NO_ERROR;

    m_Mutex.lock();
    tObjectID objectID = m_PlayMediaObject.objectID;
    m_Mutex.unlock();

    if(((tObjectID)handle != objectID)
       &&
       (PE_PBS_STOPPEDSTATE != playbackState))
    {
        ETG_TRACE_ERR(("Obsolete playback status -> Do not forward objectID:%u", (tObjectID)handle));
    }
    else
    {
        //Remark: Interpretation of non error reasons is done in PlayerManager::ForwardPlaybackStatusNew
        if(REASON_OK > reason)
        {
            /* Forward stream error */
            ForwardStreamError(IN reason);
        }
        else
        {
            /* Send PLAYBACK_STATUS message to PlayerManger to update playback status */
            char messageString[64];
            strncpy_r(OUT messageString, IN "PlayerManagerSM::PLAYBACK_STATUS", IN sizeof(messageString));
            tAllParameters parameterString;
            size_t size = sizeof(parameterString);

            ret = LocalSPM::GetPlayerManager().ParameterPLAYBACK_STATUS(OUT parameterString, IN size, IN handle, IN playbackState, IN reason, IN speed);
            if (MP_NO_ERROR != ret)
            {
                ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
                parameterString[0] = '\0';
            }

            ret = Dispatcher::GetInstance().SendMessage(IN messageString, IN parameterString);
            if (MP_NO_ERROR != ret)
            {
                ETG_TRACE_ERR(("Error while sending message via SMF (ErrorCode:%s)", errorString(ret)));
            }
        }
    }

    return ret;
}

tResult CDDAControl::ForwardNowPlayingStatus(const tPEHandle handle, const tURL mountPointURL)
{
    ENTRY
    VARTRACE(handle);
    VARTRACE(mountPointURL);

    tResult ret = MP_NO_ERROR;

    m_Mutex.lock();
    tMediaObject mediaObject = m_PlayMediaObject;
    m_Mutex.unlock();

    if((tObjectID)handle != mediaObject.objectID)
    {
        if( !LocalSPM::GetDataProvider().GaplessPlay() )
        {
            ETG_TRACE_ERR(("Obsolete nowPlaying status -> Do not forward objectID:%u", (tObjectID)handle));
        }
        else
        {
            ETG_TRACE_USR1(("Received nowPlaying status of a new object (objectID (received/last):%u/%u)", (tObjectID)handle, mediaObject.objectID));

            //@@@ code?? : get metadata from current track from toc
        }
    }
    else
    {
        /* Send NOW_PLAYING_STATUS message to PlayerManger to update nowPlaying info of object */
        char messageString[64];
        strncpy_r(OUT messageString, IN "PlayerManagerSM::NOW_PLAYING_STATUS", IN sizeof(messageString));
        tAllParameters parameterString;
        size_t size = sizeof(parameterString);
        mediaObject.mediaType = MTY_AUDIO_STREAM;
        ret = LocalSPM::GetPlayerManager().ParameterNOW_PLAYING_STATUS(OUT parameterString,
                IN size,
                IN handle,
                IN mediaObject.MetadataField1,
                IN mediaObject.MetadataField2,
                IN mediaObject.MetadataField3,
                IN mediaObject.MetadataField4,
                IN mediaObject.title,
                IN mediaObject.mediaType,
                IN mediaObject.bitRate,
                IN mediaObject.sampleRate,
                IN mediaObject.audioChannelFormat,
                IN mediaObject.metadataConvertFlag,
                IN NULL,
                IN NULL);
        if (MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
            parameterString[0] = '\0';
        }

        ret = Dispatcher::GetInstance().SendMessage(IN messageString, IN parameterString);
        if (MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while sending message via SMF (ErrorCode:%s)", errorString(ret)));
        }
    }

    return ret;
}

tResult CDDAControl::ForwardNowPlayingStatusWithMetadata(const tPEHandle handle, const tURL mountPointURL)
{
    ENTRY
    VARTRACE(handle);
    VARTRACE(mountPointURL);

    tResult ret = MP_NO_ERROR;

    m_Mutex.lock();
    tMediaObject mediaObject = m_PlayMediaObject;
    m_Mutex.unlock();

    if((tObjectID)handle != mediaObject.objectID)
    {
        ETG_TRACE_ERR(("Received metadata of wrong object (objectID (received/expected):%u/%u)", (tObjectID)handle, mediaObject.objectID));
        return MP_NO_ERROR;
    }

    /* Send NOW_PLAYING_STATUS message to PlayerManger to update nowPlaying info of object */
    char messageString[64];
    strncpy_r(OUT messageString, IN "PlayerManagerSM::NOW_PLAYING_STATUS", IN sizeof(messageString));
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);
    mediaObject.mediaType = MTY_AUDIO_STREAM;
    ret = LocalSPM::GetPlayerManager().ParameterNOW_PLAYING_STATUS(OUT parameterString,
            IN size,
            IN handle,
            IN mediaObject.MetadataField1,
            IN mediaObject.MetadataField2,
            IN mediaObject.MetadataField3,
            IN mediaObject.MetadataField4,
            IN mediaObject.title,
            IN mediaObject.mediaType,
            IN mediaObject.bitRate,
            IN mediaObject.sampleRate,
            IN mediaObject.audioChannelFormat,
            IN mediaObject.metadataConvertFlag,
            IN NULL,
            IN NULL);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
        parameterString[0] = '\0';
    }

    ret = Dispatcher::GetInstance().SendMessage(IN messageString, IN parameterString);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while sending message via SMF (ErrorCode:%s)", errorString(ret)));
    }

    return ret;
}

tResult CDDAControl::PlaytimeStatusNew(const tPEHandle handle, const tPETimeInfo timeInfo)
{
    ENTRY
    VARTRACE(handle);
    VARTRACE(timeInfo);

    tResult ret = MP_NO_ERROR;

    m_Mutex.lock();
    tDeviceID deviceID = m_PlayMediaObject.deviceID;
    m_Mutex.unlock();

    /* Send answer to waiting state machine */
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    ret = LocalSPM::GetDeviceDispatcher().ParameterPLAYTIME_STATUS(OUT parameterString, IN size, IN handle, IN deviceID, IN timeInfo);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
        parameterString[0] = '\0';
    }

    ret = SendAnswer(IN parameterString);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while sending message via SMF (ErrorCode:%s)", errorString(ret)));
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult CDDAControl::ForwardPlaytimeStatusNew(const tPEHandle handle, const tPETimeInfo timeInfo)
{
    ENTRY
    VARTRACE(handle);
    VARTRACE(timeInfo);

    tResult ret = MP_NO_ERROR;

    m_Mutex.lock();
    tObjectID objectID = m_PlayMediaObject.objectID;
    tDeviceID deviceID = m_PlayMediaObject.deviceID;
    m_Mutex.unlock();

    if ((tObjectID)handle != objectID)
    {
        ETG_TRACE_ERR(("Obsolete playtime status -> Do not forward objectID:%u", (tObjectID)handle));
    }
    else
    {
        /* Send PLAYTIME_STATUS message to PlayerManger to update playtime */
        char messageString[64];
        strncpy_r(OUT messageString, IN "PlayerManagerSM::PLAYTIME_STATUS", IN sizeof(messageString));
        tAllParameters parameterString;
        size_t size = sizeof(parameterString);

        ret = LocalSPM::GetPlayerManager().ParameterPLAYTIME_STATUS(OUT parameterString, IN size, IN handle, IN deviceID, IN timeInfo);
        if (MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
            parameterString[0] = '\0';
        }

        ret = Dispatcher::GetInstance().SendMessage(IN messageString, IN parameterString);
        if (MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while sending message via SMF (ErrorCode:%s)", errorString(ret)));
        }
    }

    return ret;
}

tResult CDDAControl::HandleAnswerTimeout()
{
    ENTRY
    ETG_TRACE_ERR(("CDDAControl::HandleAnswerTimeout"));

    tResult ret = MP_NO_ERROR;

    /* Send releasing event to own waiting state machine */
    ret = ReleaseWaiting();
    if(MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while releasing waiting via SMF (ErrorCode:%s)", errorString(ret)));
    }

    return ret;
}

tResult CDDAControl::MessageNotConsumed()
{
    ENTRY
    ETG_TRACE_ERR(("CDDAControl::MessageNotConsumed"));
#if 0 //Do not SendAnswer without reason, it may cause failure within communication to DeviceDispatcher
    /* Send answer to possible waiting state machine to release own SM in case of message answer pair*/
    tResult ret = SendAnswer(NULL);
    if(MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while sending answer via SMF (ErrorCode:%s)", errorString(ret)));
    }
#endif
    return MP_NO_ERROR;
}

tResult CDDAControl::ValidateErrorCode(me::reason_e &reason, const tMetadata errorCode, const tMountPoint mountPoint) const
{
    ENTRY_INTERNAL

    tUInt errorEnum = atoi(errorCode);

    switch(errorEnum)
    {
    case MEDIAPLAYER_ERROR_DEVICE_NOT_FOUND:
    case MEDIAPLAYER_ERROR_AUDIO_DEVICE_ALREADY_IN_USE:
        reason = REASON_DEVICE_ERROR;
        break;
    case MEDIAPLAYER_ERROR_FILE_DOES_NOT_EXISTS:
    case MEDIAPLAYER_ERROR_CANNOT_DECODE_MEDIA:
    case MEDIAPLAYER_ERROR_UNKNOWN_ERROR:
    default:
        reason = REASON_FORMAT_ERROR;
        break;
    }

    return MP_NO_ERROR;
}

tResult CDDAControl::SetOutputDevice(const tAudioOutputDevice audioOutputDevice)
{
    ENTRY
    (void)audioOutputDevice;

    /* For CDDA no need to store audio output device */

    return MP_NO_ERROR;
}

tResult CDDAControl::SendPlaybackStatus(const tPEHandle handle, const tPEPlaybackState playbackState, const me::reason_e reason, const me::speed_e speed)
{
    ENTRY
    //VARTRACE(handle);
    //VARTRACE(playState);
    //VARTRACE(reason);
    //VARTRACE(speed);

    tResult ret = MP_NO_ERROR;

    /* Send PLAYBACK_STATUS event to own state machine */
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    ret = ParameterPLAYBACK_STATUS(OUT parameterString, IN size, IN handle, IN playbackState, IN reason, IN speed);
    if( MP_NO_ERROR != ret )
    {
        ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
        parameterString[0] = '\0';
    }

    ret = SendEvent(PLAYBACK_STATUS, IN parameterString);
    if( MP_NO_ERROR != ret )
    {
        ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
    }

    return ret;
}

tResult CDDAControl::SendNowPlayingStatus(const tPEHandle handle, const tURL mountPointURL)
{
    ENTRY
    //VARTRACE(handle);
    //VARTRACE(mountPointURL);

    tResult ret = MP_NO_ERROR;

    /* Send NOW_PLAYING_STATUS event to own state machine */
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);
    ret = ParameterNOW_PLAYING_STATUS(OUT parameterString, IN size, IN handle, IN mountPointURL);
    if( MP_NO_ERROR != ret )
    {
        ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
        parameterString[0] = '\0';
    }

    ret = SendEvent(NOW_PLAYING_STATUS, IN parameterString);
    if( MP_NO_ERROR != ret )
    {
        ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
    }

    return ret;
}

tResult CDDAControl::SendPlaytimeStatus(const tPEHandle handle, const tPETimeInfoStruct timeInfoStruct)
{
    ENTRY
    //VARTRACE(handle);
    //VARTRACE(timeInfoStruct);

    tResult ret = MP_NO_ERROR;

    /* Marshal struct of timeInfo into a string */
    tPETimeInfo timeInfoString;
    size_t size = sizeof(timeInfoString);
    SMF::Marshal(OUT timeInfoString,
            IN size-1,
            IN DOUBLE_MARSHAL_SEPARATOR,
            IN tPEBytes_format tPEPercentage_format tPEMilliseconds_format tPEBytes_format tPEMilliseconds_format,
            IN timeInfoStruct.position.bytes,
            IN timeInfoStruct.position.pct,
            IN timeInfoStruct.position.ms,
            IN timeInfoStruct.duration.bytes,
            IN timeInfoStruct.duration.ms);

    /* Send PLAYTIME_STATUS event to own state machine */
    tAllParameters parameterString;
    size = sizeof(parameterString);
    ret = ParameterPLAYTIME_STATUS(OUT parameterString, IN size, IN handle, IN timeInfoString);
    if( MP_NO_ERROR != ret )
    {
        ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
        parameterString[0] = '\0';
    }

    ret = SendEvent(PLAYTIME_STATUS, IN parameterString);
    if( MP_NO_ERROR != ret )
    {
        ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
    }

    return ret;
}


//Indexing part

tResult CDDAControl::IsInitRequired(const tDeviceInfoString deviceInfoString)
{
    ENTRY;
    tDeviceInfo deviceInfo;
    DataProvider::UnMarshalDeviceInfo(deviceInfoString, deviceInfo);
    VARTRACE2(deviceInfo.mountPoint, deviceInfo.deviceID);

    /* In case of CDDA no initialization is necessary -> return NoInitAnswer (former false case) */
    return NoInitAnswer(IN deviceInfoString);
}

tResult CDDAControl::SendInitInit(const tDeviceInfoString deviceInfoString)
{
    ENTRY;
    tDeviceInfo deviceInfo;
    DataProvider::UnMarshalDeviceInfo(deviceInfoString, deviceInfo);
    VARTRACE2(deviceInfo.mountPoint, deviceInfo.deviceID);

    //@@@ code?

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult CDDAControl::NoInitAnswer(const tDeviceInfoString deviceInfoString)
{
    ENTRY;
    tDeviceInfo deviceInfo;
    DataProvider::UnMarshalDeviceInfo(deviceInfoString, deviceInfo);
    VARTRACE2(deviceInfo.mountPoint, deviceInfo.deviceID);

    tResult ret = MP_NO_ERROR;

    /* Send INIT_DEVICE_CONNECTION message to IndexerSM */
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);
    tConnectionState connectionState = CS_CONNECTED;
    if(mCDDADeviceInterface && false == mTocAvailabe)
    {
        mCDDADeviceInterface->resetToc();    //NCG3D-16237
        mTocAvailabe = true; //TOC will reset once while CDA ejected (NCG3D-34407)
    }

    ret = LocalSPM::GetIndexer().ParameterDEVICE_INITIALIZED(OUT parameterString, IN size, IN deviceInfo.deviceName, IN deviceInfo.deviceID, IN connectionState);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
        parameterString[0] = '\0';
    }

    ret = LocalSPM::GetIndexer().SendEventByName("DEVICE_INITIALIZED", IN parameterString);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while sending event via SMF (ErrorCode:%s)", errorString(ret)));
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult CDDAControl::IsInitDeviceConnection(const tMountPoint mountPoint) const
{
   ENTRY
   VARTRACE(mountPoint);

   /* For CDDA no implementation -> return false */
   return false;
}

tResult CDDAControl::Disconnect(const tMountPoint mountPoint) const
{
   ENTRY
   VARTRACE(mountPoint);

   /* For CDDA no implementation */
   return MP_NO_ERROR;
}

tResult CDDAControl::OnDeviceInitialized(const tMountPoint mountPoint, const tConnectionState connectionState) const
{
   ENTRY
   VARTRACE(mountPoint);
   VARTRACE(connectionState);

   /* For CDDA no implementation */
   return MP_NO_ERROR;
}

tResult CDDAControl::GetNumberOfFiles(const tDeviceID deviceID)
{
    //Device type is not supported for indexing, nevertheless indexer requires the total number of playable files from this device
    //Added for VAG iPod streaming

    ENTRY;
    VARTRACE(deviceID);

    //@@@ code
    tNumberOfFiles numberOfFiles = 0; //TODO: return total number of files here

    /* Send ANSWER message to IndexerSM */
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    LocalSPM::GetIndexer().ParameterNUMBER_OF_FILES_ANSWER(OUT parameterString, IN size, IN deviceID, IN numberOfFiles);
    SendAnswer(IN parameterString);

   return MP_NO_ERROR;
}

tResult CDDAControl::CalcFingerprint(const tMountPoint mountPoint, const tDeviceID deviceID, const tFingerprint lastFingerprint)
{
    ENTRY
    VARTRACE2(mountPoint, deviceID);
    VARTRACE(lastFingerprint);

    tResult ret = MP_NO_ERROR;

    return ret;
}

tResult CDDAControl::OnFingerprint(const tFingerprint fingerprint,
                                  const tFingerprintStatus fingerprintStatus,
                                  const tNumberOfFiles numberOfFiles,
                                  const tDeviceID deviceID)
{
    ENTRY
    VARTRACE(fingerprint);
    VARTRACE(fingerprintStatus);
    VARTRACE(numberOfFiles);
    VARTRACE(deviceID);

    tResult ret = MP_NO_ERROR;


    /* Send FINGERPRINT_AVAIL message to IndexerSM */
    char messageString[64];
    strncpy(messageString, "IndexerSM::FINGERPRINT_AVAIL", sizeof(messageString) - 1);
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    ret = LocalSPM::GetIndexer().ParameterFINGERPRINT_AVAIL(OUT parameterString,
            IN size,
            IN fingerprint,
            IN fingerprintStatus,
            IN numberOfFiles,
            IN deviceID);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
        parameterString[0] = '\0';
    }

    //ret = SendAnswer(IN parameterString);
    ret = Dispatcher::GetInstance().SendMessage(IN messageString, IN parameterString);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while sending message via SMF (ErrorCode:%s)", errorString(ret)));
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult CDDAControl::IsNextMetadataAvailable(const tMountPoint mountPoint, const tReadPosition readPosition, const tDeviceID deviceID)
{
    ENTRY
    VARTRACE2(mountPoint, deviceID);
    VARTRACE(readPosition);

    /* Return false always because we have no metadata cache */
    return false;
}

tResult CDDAControl::RetrieveMetadataFromDevice(const tMountPoint mountPoint, const tReadPosition readPosition, const tDeviceID deviceID)
{
    ENTRY
    VARTRACE2(mountPoint, deviceID);
    VARTRACE(readPosition);

    tResult ret = MP_NO_ERROR;
    tMetadataStatus metadataStatus = MDS_SUCCESS;

    /* Get next media object from virtual file list */
    //@@@code get next toc entry

    /* Send PUT_METADATA message to own SM */
    tMediaObjectPtr mediaObjectPtr = NULL;
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    ret = ParameterPUT_METADATA(OUT parameterString, IN size, IN metadataStatus, IN mediaObjectPtr);
    if( MP_NO_ERROR != ret )
    {
        ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
        parameterString[0] = '\0';
    }

    ret = SendEvent(PUT_METADATA, IN parameterString);
    if( MP_NO_ERROR != ret )
    {
        ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult CDDAControl::SendPutMetadata(const tMountPoint mountPoint, const tReadPosition readPosition, const tDeviceID deviceID)
{
    ENTRY
    VARTRACE2(mountPoint, deviceID);
    VARTRACE(readPosition);

    tResult ret = MP_NO_ERROR;

    /* Send PUT_METADATA message to own SM */
    tMetadataStatus metadataStatus = MDS_SUCCESS;
    tMediaObjectPtr mediaObjectPtr = NULL;

    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    ret = ParameterPUT_METADATA(OUT parameterString, IN size, IN metadataStatus, IN mediaObjectPtr);
    if( MP_NO_ERROR != ret )
    {
        ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
        parameterString[0] = '\0';
    }

    ret = SendEvent(PUT_METADATA, IN parameterString);
    if( MP_NO_ERROR != ret )
    {
        ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult CDDAControl::AnswerMetadata(const tMetadataStatus metadataStatus, tMediaObjectPtr mediaObjectPtr)
{
    ENTRY
    VARTRACE(metadataStatus);

    tResult ret = MP_NO_ERROR;

    //@@@ add code for toc data??

    /* Copy media object */
    if(MDS_SUCCESS == metadataStatus)
    {
        if(NULL == mediaObjectPtr)
        {
            mediaObjectPtr = new tMediaObject; //Attention: Has to be deleted by Indexer::CheckMetadata
        }
        InitMediaObject(OUT *mediaObjectPtr);
        //*mediaObjectPtr = m_ScanMediaObject;
    }

    /* Send answer to waiting state machine */
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    ret = LocalSPM::GetIndexer().ParameterMETADATA_ANSWER(OUT parameterString,
            IN size,
            IN mediaObjectPtr, //lint -e429 freed by calling thread
            IN metadataStatus);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
        parameterString[0] = '\0';
        if (NULL != mediaObjectPtr)
        {
            delete mediaObjectPtr;
        }
    }

    ret = SendAnswer(IN parameterString);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while sending answer via SMF (ErrorCode:%s)", errorString(ret)));
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult CDDAControl::RetrieveMetadataFromCache(tMetadata &metadata) const
{
    /* Obsolete: DeviceControl is sending a media object instead of a metatdata string so an assembly is not necessary */
    ENTRY
    (void)metadata;
    return MP_NO_ERROR;
}

tResult CDDAControl::RemoveDeviceConnection(const tMountPoint mountPoint, const tDeviceID deviceID)
{
    ENTRY
    VARTRACE2(mountPoint, deviceID);

    tResult ret = MP_NO_ERROR;

    //@@@ code

    return ret;
}

//AlbumArt part

tResult CDDAControl::GetAlbumArt(const tAlbumArt albumArtString)
{
    ENTRY
    VARTRACE(albumArtString);

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult CDDAControl::HandleGetAlbumArtAnswer(const tAlbumArtObjectPtr ptrToAlbumArtObject)
{
    ENTRY

    /*Send the album art data as answer to request GetAlbumArt (Note : requested from DataProvider)*/
    tAllParameters parameterString = {0};
    SMF::Marshal(IN parameterString, sizeof(parameterString)-1, "p",IN ptrToAlbumArtObject);

    SendAnswer(IN parameterString);

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult CDDAControl::AlbumArtAnswerNotConsumed(const tAlbumArtObjectPtr ptrToAlbumArtObject)
{
    ENTRY;

    return MP_NO_ERROR;
}

tResult CDDAControl::SendAlbumArtAnswer(const tAlbumArtObjectPtr ptrToAlbumArtObject)
{
    ENTRY_INTERNAL;

    tResult ret = MP_NO_ERROR;

    /* Send GET_ALBUM_ART_ANSWER message to own SM */
    tAllParameters parameterString = {0};
    SMF::Marshal(IN parameterString, sizeof(parameterString)-1, "p", IN ptrToAlbumArtObject);

    ret = SendEvent(GET_ALBUM_ART_ANSWER, IN parameterString);

    return ret;
}

tResult CDDAControl::Umount(const tDeviceID deviceID, const tMountPoint mountPoint)
{
    ENTRY;
    VARTRACE2(mountPoint, deviceID);

    tResult ret = MP_NO_ERROR;

    /* Send answer to waiting state machine */
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);
    SMF::Marshal(parameterString, size - 1, "i", 0);

    ret = SendAnswer(IN parameterString);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while sending answer via SMF (ErrorCode:%s)", errorString(ret)));
    }

    return ret;
}

tBoolean CDDAControl::IsBatchPlayable(const tDeviceID deviceID)
{
    ENTRY;
    VARTRACE(deviceID);
    return 0;
}

tResult CDDAControl::GetNextObjectToIndex(tMetadataStatus &metadataStatus,
                                         const tDeviceID deviceID,
                                         const tMountPoint mountPoint,
                                         const tReadPosition readPosition)
{
    ENTRY_INTERNAL
    VARTRACE2(mountPoint, deviceID);
    VARTRACE(readPosition);

    tResult ret = MP_NO_ERROR;
    metadataStatus = MDS_SUCCESS;

    //@@@ code

    return ret;
}

void CDDAControl::TransferMetaTags(void *_info, tMediaObject &mediaObject)
{
    //@@@ nothing to do
}

void CDDAControl::DoReadMetadataForPlaybackThread(const tURL completeFileName)
{
    ENTRY
    VARTRACE(completeFileName);

    tResult ret = MP_NO_ERROR;
    tReturnValue returnValue = TRUE;

    /* Get metadata of media object for playback */
    tMediaObject mediaObject;
    InitMediaObject(OUT mediaObject);

    ret = LocalSPM::GetDBManager().GetFileFormat(OUT mediaObject.fileFormat, OUT mediaObject.fileType, IN completeFileName);
    if(MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("unable to get file format for the url: %s", completeFileName));
        returnValue = FALSE;
    }
    else
    {
        if(FT_VIDEO == mediaObject.fileType )
        {
            /* Read video metadata */
            returnValue = ReadVideoMetadataFromFile(INOUT mediaObject, IN completeFileName);
        }
        else
        {
            /* Read audio metadata */
            returnValue = ReadAudioMetadataFromFile(INOUT mediaObject, IN completeFileName);
        }
    }

    m_Mutex.lock();

    tPEHandle handle = (tPEHandle)m_PlayMediaObject.objectID;
    m_PlayMediaObject.notPlayable = mediaObject.notPlayable;

    if(returnValue)
    {
        /* Copy metadata */
        m_PlayMediaObject.mediaType = mediaObject.mediaType;
        strncpy_r(OUT m_PlayMediaObject.title, IN mediaObject.title, IN sizeof(m_PlayMediaObject.title));
        strncpy_r(OUT m_PlayMediaObject.MetadataField1, IN mediaObject.MetadataField1, IN sizeof(m_PlayMediaObject.MetadataField1));
        strncpy_r(OUT m_PlayMediaObject.MetadataField2, IN mediaObject.MetadataField2, IN sizeof(m_PlayMediaObject.MetadataField2));
        strncpy_r(OUT m_PlayMediaObject.MetadataField3, IN mediaObject.MetadataField3, IN sizeof(m_PlayMediaObject.MetadataField3));
        strncpy_r(OUT m_PlayMediaObject.MetadataField4, IN mediaObject.MetadataField4, IN sizeof(m_PlayMediaObject.MetadataField4));
        m_PlayMediaObject.totalPlaytime = mediaObject.totalPlaytime;
        m_PlayMediaObject.trackNumber = mediaObject.trackNumber;
        m_PlayMediaObject.metadataConvertFlag = mediaObject.metadataConvertFlag;
    }

    m_Mutex.unlock();

    /* Send METADATA_FOR_PLAYBACK message to own SM */
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    ret = ParameterMETADATA_FOR_PLAYBACK(OUT parameterString, IN size, IN handle, IN completeFileName);
    if( MP_NO_ERROR != ret )
    {
        ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
        parameterString[0] = '\0';
    }

    ret = SendEvent(METADATA_FOR_PLAYBACK, IN parameterString);
    if( MP_NO_ERROR != ret )
    {
        ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
    }

    delete[] completeFileName;
}

#define PERF_MEAS 1

tReturnValue CDDAControl::ReadAudioMetadataFromFile(tMediaObject &mediaObject, const tURL completeFileName)
{
    ENTRY
    VARTRACE(completeFileName);

    tReturnValue returnValue = TRUE;

    mediaObject.mediaType = MTY_MUSIC_FILE;

    //@@@: read toc data

#if PERF_MEAS
    TimeTrace ticks("ReadAudioMetadataFromFile");
#endif


#if PERF_MEAS
    ticks.elapsed();
#endif

    if(FALSE == returnValue)
    {
        ETG_TRACE_ERR(("CDDAControl::ReadAudioMetadataFromFile -> metadata could not be retrieved for file : %s",completeFileName));
    }
    else
    {
        VARTRACE5(mediaObject.title, mediaObject.MetadataField1, mediaObject.MetadataField2, mediaObject.MetadataField3, mediaObject.MetadataField4);
        VARTRACE(mediaObject.totalPlaytime);
        VARTRACE(mediaObject.trackNumber);
        VARTRACE(mediaObject.notPlayable);
    }

    return returnValue;
}

tReturnValue CDDAControl::ReadVideoMetadataFromFile(tMediaObject &mediaObject, const tURL completeFileName)
{
    ENTRY
    VARTRACE(completeFileName);
    tReturnValue returnValue = TRUE;

    //@@@ no video data

    return returnValue;
}

tReturnValue CDDAControl::ReadImageMetadataFromFile(tMediaObject &mediaObject, const tURL completeFileName)
{
    ENTRY
    VARTRACE(completeFileName);

    tReturnValue returnValue = TRUE;

    //@@@ no image

    return returnValue;
}

tReturnValue CDDAControl::CollectData(const tMountPoint mountPoint)
{
    ENTRY_INTERNAL
    VARTRACE(mountPoint);

    tReturnValue returnValue = TRUE;

    if(0 == LocalSPM::GetDataProvider().CollectMetadata())
    {
        //ETG_TRACE_USR1(("No metadata extraction configured"));
        return TRUE;
    }

    /* Collect data together */
    /* Only a few parts of the media object are filled if it is taken from a LTY_FILELIST_UNSORTED list
     * mediaObject.objectID
     * mediaObject.fileName
     * mediaObject.fileType
     * mediaObject.fileFormat
     * mediaObject.title (= mediaObject.fileName without path but with extension)
     * mediaObject.deviceID
     * mediaObject.deviceType
     * mediaObject.deviceVersion
     * mediaObject.mountPoint
     * mediaObject.catType
     * mediaObject.mediaType
     * mediaObject.albumArtString
     *
     * The rest has to be added here
     */

    //tph: TODO ReadMetadata in thread

    //@@@: reduce code

    //tURL completeFileName;
    //strncpy_r(OUT completeFileName, IN m_ScanMediaObject.mountPoint, IN sizeof(completeFileName));
    //strncat_r(OUT completeFileName, IN m_ScanMediaObject.fileName, IN sizeof(completeFileName));

    return returnValue;
}

tResult CDDAControl::CDDAGetFolderItem(const tMountPoint mountpoint, const tPath path, const tIndex index)
{
    ENTRY
    tResult ret = MP_NO_ERROR;

    /* create a memory object which holds the toc data */
    if( mTocDataOneTrack )
    {
        delete mTocDataOneTrack; //delete old one
    }
    mTocDataOneTrack = new tFiles();

    /* get the toc data for one track */
    mCDDADeviceInterface->getToc(mTocDataOneTrack, mountpoint, path, index);

    /* forward my own state machine */
    ret = CDDAControlSM::SendEvent(CDDA_GET_FOLDER_ITEM_ANSWER, NULL);
    if( MP_NO_ERROR != ret )
    {
        ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult CDDAControl::CDDAGerFolderItemAnswer()
{
    ENTRY
    tResult ret = MP_NO_ERROR;

    /* answer this request */
    tAllParameters parameters;
    SMF::Marshal(parameters, sizeof(parameters), "p", mTocDataOneTrack);
    ret = SendAnswer(parameters);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while sending answer via SMF (ErrorCode:%s)", errorString(ret)));
        if( mTocDataOneTrack )
        {
            delete mTocDataOneTrack;
        }
    }
    mTocDataOneTrack = NULL;

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

CDDAControl::CDDACallbacks::CDDACallbacks()
{
    mResendPlaybackStatus = 0;
    mResendNowPlayingStatus = 0;
    strncpy_r(mObservingStateMachineName, "CDDAControlSM", sizeof(mObservingStateMachineName));
}


int CDDAControl::CDDACallbacks::ForwardPlaybackStatus(cdstate_t const &currentState)
{
    ENTRY
    //ETG_TRACE_USR2(("MediaEngine::ForwardPlaybackStatus"));

    /* no observer registered -> do not send status updates */
    if (mObservingStateMachineName[0] == 0) {
        return 0;
    }

    CDDAControl::m_Mutex.lock();
    copyCddaState(mNotificationStateNew, currentState);
    CDDAControl::m_Mutex.unlock();

    bool update = false;

    /* check the contents of the notification to figure out what event to send */

    // PLAYBACK_STATUS;
    if (mNotificationState.playstate != mNotificationStateNew.playstate ||
        mNotificationState.speed     != mNotificationStateNew.speed ||
        mNotificationState.reason    != mNotificationStateNew.reason ||
        mResendPlaybackStatus) {

        ETG_TRACE_USR2(("MediaEngine::ForwardPlaybackStatus:PLAYBACK_STATUS"));

        tPEPlaybackState playbackState;
        switch(mNotificationStateNew.playstate) {
        case PLAYSTATE_STOP:
            playbackState = PE_PBS_STOPPEDSTATE;
            break;
        case PLAYSTATE_PAUSE:
            playbackState = PE_PBS_PAUSEDSTATE;
            break;
        case PLAYSTATE_PLAY:
            if(0 > mNotificationStateNew.speed) {
                playbackState = PE_PBS_FASTREVERSESTATE;
            }
            else if(1 < mNotificationStateNew.speed) {
                playbackState = PE_PBS_FASTFORWARDSTATE;
            }
            else {
                playbackState = PE_PBS_PLAYINGSTATE;
            }
            break;
        case PLAYSTATE_NOP:
            playbackState = PE_PBS_LOADINGSTATE;
            break;
        default:
            playbackState = PE_PBS_ERRORSTATE;
            break;
        }

        tGeneralString messageName;
        strncpy_r(OUT messageName, IN mObservingStateMachineName, IN sizeof(messageName));
        strncat_r(OUT messageName, "::PLAYBACK_STATUS", IN sizeof(messageName));

        tAllParameters args;
        Marshal((char *)args, sizeof(args)-1, "lill",
                (tU64)mNotificationStateNew.handle,
                (int)playbackState,
                mNotificationStateNew.reason,
                mNotificationStateNew.speed);

        /* send the PLAYBACK_STATUS to observer */
        Dispatcher::GetInstance().SendMessage(messageName, args);

        /* reset the extra trigger for sending playback status */
        mResendPlaybackStatus = 0;

        update = true; // save notification state as current;
    }

    // NOW_PLAYING_STATUS;
    if (mNotificationState.url != mNotificationStateNew.url || mResendNowPlayingStatus) {

        ETG_TRACE_USR2(("MediaEngine::ForwardPlaybackStatus:NOW_PLAYING_STATUS"));
        VARTRACE(mNotificationStateNew.url.c_str());
        VARTRACE(mNotificationState.url.c_str());
        VARTRACE(mResendNowPlayingStatus);

        tGeneralString messageName;
        strncpy_r(OUT messageName, IN mObservingStateMachineName, IN sizeof(messageName));
        strncat_r(OUT messageName, "::NOW_PLAYING_STATUS", IN sizeof(messageName));

        tAllParameters args;
        Marshal((char *)args, sizeof(args)-1, "lt", (tU64)mNotificationStateNew.handle, mNotificationStateNew.url.c_str());

        /* send the NOW_PLAYING_STATUS to observer */
        Dispatcher::GetInstance().SendMessage(messageName, args);

        /* reset the extra trigger for sending now playing status */
        mResendNowPlayingStatus = 0;

        update = true; // save notification state as current;
    }

    // PLAYTIME_STATUS;
    tBool bPos = (mNotificationState.pos_bytes != mNotificationStateNew.pos_bytes)
                  || (mNotificationState.pos_pct != mNotificationStateNew.pos_pct)
                  || (mNotificationState.pos_ms != mNotificationStateNew.pos_ms);
#if 0
    tBool bDur = (mNotificationState.dur_bytes == mNotificationStateNew.dur_bytes)
                 && (mNotificationState.dur_ms == mNotificationStateNew.dur_ms);
#endif
    
    if(bPos)    //Fix for NCG3D-126735 - when there is a change in position forward playtime status
    {
        //ETG_TRACE_USR2(("MediaEngine::ForwardPlaybackStatus:PLAYTIME_STATUS"));

        //#define MEDIA_ENGINE_USE_PLAYTIME_FILTER

        #if defined(MEDIA_ENGINE_USE_PLAYTIME_FILTER)
            /* reduce the event rate */
            if (llabs(state1.pos.ms  - state0.pos.ms)  >  500 || // for every 500 ms change in current position or
                llabs(state1.pos.pct - state0.pos.pct) >=   2 || // for every 2 percent change in current position or
                state1.pos.ms                          <  500 || // if the current position is near the beginning or
                llabs(state1.dur.ms - state1.pos.ms)   <  500 || // if the current position is near the end of file or
                memcmp(&state0.dur, &state1.dur, sizeof(state0.dur))) // for every duration changes
            {
        #endif
            tGeneralString messageName;
            strncpy_r(OUT messageName, IN mObservingStateMachineName, IN sizeof(messageName));
            strncat_r(OUT messageName, "::PLAYTIME_STATUS", IN sizeof(messageName));

            tPETimeInfo timeInfoString;
            SMF::Marshal(OUT timeInfoString, IN sizeof(timeInfoString)-1, IN DOUBLE_MARSHAL_SEPARATOR, "lllll",
                    IN mNotificationStateNew.pos_bytes,
                    IN mNotificationStateNew.pos_pct,
                    IN mNotificationStateNew.pos_ms,
                    IN mNotificationStateNew.dur_bytes,
                    IN mNotificationStateNew.dur_ms);

            //ETG_TRACE_USR2(("timeInfoString=%s", timeInfoString));

            tAllParameters args;
            Marshal((char *)args, sizeof(args)-1, "lt", (tU64)mNotificationStateNew.handle, IN timeInfoString);

            /* send the PLAYTIME_STATUS to observer */
            Dispatcher::GetInstance().SendMessage(messageName, args);

            update = true; // save notification state as current;

        #if defined(MEDIA_ENGINE_USE_PLAYTIME_FILTER)
            } else {// reduce event rate
                ETG_TRACE_USR1(("PLAYTIME_STATUS filtered out"));
            }
        #endif

    }

    if(update)
    {
        CDDAControl::m_Mutex.lock();
        copyCddaState(mNotificationState, mNotificationStateNew);
        CDDAControl::m_Mutex.unlock();
    }

    /* TODO: ACTIVITY_STATUS? */
    /* there is no activity status in the state */

    return 0;
}

void CDDAControl::CDDACallbacks::update(cdstate_t const &currentState)
{
    ENTRY
    ForwardPlaybackStatus(currentState);
}

void CDDAControl::CDDACallbacks::copyCddaState(cdstate_t & dest, cdstate_t const &src)
{
    ENTRY
    dest.playstate =  src.playstate;
    dest.reason    =  src.reason;
    dest.handle    =  src.handle;
    dest.pos_bytes =  src.pos_bytes;
    dest.pos_pct   =  src.pos_pct;
    dest.pos_ms    =  src.pos_ms;
    dest.dur_bytes =  src.dur_bytes;
    dest.dur_ms    =  src.dur_ms;
    dest.speed     =  src.speed;
    dest.url       =  src.url;
}

void CDDAControl::CDDACallbacks::resendNowPlaying(tBoolean resend)
{
    ENTRY
    mResendNowPlayingStatus = resend;
}
void CDDAControl::cddaEjected()
{
    ENTRY
    mTocAvailabe = false;
}
tResult CDDAControl::SendNoResponseMsg(const tResponseMsg response)
{
    ENTRY
    tResult ret = MP_NO_ERROR;
    ETG_TRACE_USR3(("USBControl::SendNoResponseMsg %d", response));
    if(m_response != response)
    {
        ret = LocalSPM::GetOutputWrapper().UpdateStreamingInfo();
    }
    m_response = response;
    return ret;
}
tResult CDDAControl::getStreamingInfo(OUT tResponseMsg &info)
{
    ENTRY
    info = m_response;
    return MP_NO_ERROR;
}
#if !defined(TARGET_BUILD)
void CDDAControl::TestsetCDText(unsigned char *ref, tU16 size, tU8 totalTrack)
{
    ENTRY
    mCDDADeviceInterface->TestsetCDText(ref, size, totalTrack);
}
#endif
