/**
 * @defgroup iPodControlHelper iPodControlHelper
 * @author Ulrich Deuper
 * 
 * @brief Helper state machine for processing database related actions
 * @{
 */

#ifndef _DEVICECONTROLHELPER_H_
#define _DEVICECONTROLHELPER_H_

#include "iPodControlHelperSM.h"
#include "ILocalSPM.h"
#include "ThreadFactory.h"

using namespace std;

/**
 * definition of class iPodControlHelper
 * encapsulate customer specific actions of the Mediaplayer
 */
 class iPodControlHelper : public iPodControlHelperSM , public ILocalSPM, public TFThread
{

public:

//SPM part

    /**
     * Constructor of component.
     * This function is used by LocalSPM to store the iPodControlHelper componentID in mComponentID.
     *
     * @param[in] componentID component ID assigned by SPM
     * @return void
     */
    iPodControlHelper(tComponentID componentID);

    /**
     * Destructor of component.
     *
     * @return void
     */
    virtual ~iPodControlHelper();

    /**
     * This function is used by LocalSPM to create the iPodControlHelper.
     * @attention: running in SPM thread context
     * Create the iPodControlHelper state machine (including create of message queue).
     * Inform LocalSPM that Create is ready -> CreateDone(0) 
     *
     * @return void
     */
    void Create();

    /**
     * This function is used by LocalSPM to trigger the iPodControlHelper initialization.
     * @attention: running in SPM thread context
     * Init the iPodControlHelper state machine.
     * Register iPodControlHelperSM with dispatcher.
     * Inform LocalSPM that Init is ready -> InitDone(0)
     *
     * @return < 0: error, = 0: OK
     */
    tResult Init(tInitReason reason);

    /**
     * Initialize own member variables
     * Start timer.
     *
     * @return != 0: error, = 0: OK
     */
    tResult InitSM();

    /**
     * This function is used by LocalSPM to start the iPodControlHelper.
     * @attention: running in SPM thread context
     * From now on all other mediaplayer components are available.
     * Start the iPodControlHelper thread and the state machine.
     *
     * @return < 0: error, = 0: OK
     */
    tResult Run();

    /**
     * This function starts a user function in a thread from the thread factory's thread pool.
     * After leaving while loop used thread is stopped and released for other users
     *
     * @return void
     */
    void Do(int functionID, void *ptr);

    /**
     * This function is used by LocalSPM to stop the iPodControlHelper.
     * @attention: running in SPM thread context
     * Store last mode values.
     * Deregister on DB trigger.
     * Set state machine to final state -> SendMessage(STOP_SM).
     * LocalSPM will be informed after STOP_SM event is processed
     *
     * @return != 0: error, = 0: OK
     */
    tResult Stop();

    /**
     * Inform LocalSPM that Stop is ready -> StopDone(0)
     *
     * @return != 0: error, = 0: OK
     */
    tResult StopEventProcessed();

    /**
     * This function is used by LocalSPM to cleanup the iPodControlHelper.
     * @attention: running in SPM thread context
     * Deregister iPodControlHelperSM with dispatcher.
     * Set state machine to final state -> SendMessage(DONE).
     * LocalSPM will be informed after DONE event is processed.
     *
     * @return != 0: error, = 0: OK
     */
    tResult Done();

    /**
     * Inform LocalSPM that Done is ready -> DoneDone(0)
     *
     * @return != 0: error, = 0: OK
     */
    tResult DoneEventProcessed();

    /**
     * Returns the current state the state machine is in (for debugging of shutdown problems)
     * @param[inout] stateName buffer for storing the current state name
     *
     * @return pointer to stateName
     */
    char *GetSMStateName(OUT tGeneralString stateName, IN size_t size);

    /**
    * Returns answer if the component is a state machine
    *
    * @return true or false
    */
    tBoolean IsComponentSM() {return true;}; //component is a state machine

    /**
     * DBChange functions
     * Function is used to access DBManager from another thread than iPodControlSM
     *
     * @param[in] deviceID
     * @return tResult
     */
    tResult OnDBChangeDeviceType(const tDeviceID deviceID, const tDeviceType newDeviceType);
    tResult OnDBChangeDeviceTypeAndMountPoint(const tDeviceID deviceID, const tDeviceType newDeviceType, const tMountPoint mountPoint);
    tResult OnDBChangeDeviceName(const tDeviceID deviceID, const tDeviceName newDeviceName);
    tResult OnDBChangeDiPOSettings(const tDeviceID deviceID, const tDiPOCaps diPOCaps, const tDiPOActive diPOActive, const tDiPOVersion diPOVersion);
    tResult OnDBChangeConnectionState(const tDeviceID deviceID, const tConnectionState newConnectionState);
    tResult OnDBChangeNowPlayingListAvailable(const tDeviceID deviceID, const tNowPlayingListAvailable isNowPlayingListAvailable);
    tResult OnDBSetDeviceUUID(const tDeviceID deviceID, const tUUID deviceUUID);
    tResult OnDBSetAppleDeviceMACAddress(const tDeviceID deviceID, const tMACAddress appleDeviceMACAddress);
    tResult OnDBSetAppleDeviceTransportIdentifiers(const tDeviceID deviceID,const tMACAddress appleDeviceMACAddress,const tUSBSerial appleDeviceUSBSerialNumber);
	tResult OnDBChangeDeviceState(const tDeviceID deviceID, const tDeviceState newDeviceState);

};

#endif  //_DEVICECONTROLHELPER_H_

/** @} */
