/**
 * @addtogroup iPodControl
 * @author Ulrich Deuper
 *
 * iPod handle class for multiple device support
 * @{
 */

#ifndef IPODCONTROLHANDLE_H_
#define IPODCONTROLHANDLE_H_

//basic includes
#include "ElapsedTimer.h"
#include <vector>
#include <queue>

//iPod includes
#include "iPodControlCommon.h"
#include "iPodControlMediaPath.h"
#include "iPodControlIndexer.h"
#include "Lock.h"

/**
 * iPod control handle class
 */

class iPodControlHandle
{

public:
    iPodControlHandle();                    /**< constructor */
    ~iPodControlHandle();                    /**< destructor */

    void Reset(const bool all = true );        /**< Reset member variables */
    void TraceOptions();                    /**< trace out all iPod options */

    bool GetOption(const tiPodOptions option);                     /**< check option */
    void SetOption(const tiPodOptions option, const bool set);     /**< set option */

    void ResetElapsedTime(const tiPodElaspedTime elapsedtime);                /**< Reset elapsed time for delayed iap call */
    unsigned long GetElapsedTimeMS(const tiPodElaspedTime elapsedtime);        /**< get elapsed time for delayed iap call */

    int CreateConfiOSApp(IPOD_IOS_APP * &pApps);                                    /**< get iOS app configuration, generate from internal list */
    static void DeleteConfiOSApp(const IPOD_IOS_APP * pApps, const int numApps);     /**< free buffer for iOS app configuration, generated from internal list */

    tBoolean AddAppInfo(const tProtocolName protocol, const tBundleSeedID bundleID, const tAppName appName, const tAppInfoOption option);    /**< add app info, returns true if added, false if already in list or other error */
    vector<tIPODAppInfo> GetAppInfos(const bool hostmode, const bool nativeTransport) const; /**< return app info list */
    tBoolean RemoveAppInfo(const tAppName appName, tSessionID &sessionID);            /**< remove app info, returns true if removed, false if already in list or other error */

    tBoolean AddAppInfoFromSPI(const tProtocolName protocol, const tBundleSeedID bundleID, const tAppName appName, const tAppInfoOption option);
    vector<tIPODAppInfo> GetAppInfosFromSPI() const;
    void ClearAppInfosFromSPI();

    tSessionID GetSessionIDFromAppName(const tAppName appName);            /**< get session id from app name */
    unsigned int GetProtocolIDFromAppName(const tAppName appName);      /**< get protocol id from app name */
    void GetActiveSessionList(vector<string> &list);                    /**< get active session list */
    void SetSessionIDByIndex(const tSessionID sessionId,
            const unsigned char protocolIndex, tAppName &appName);         /**< set session id by protocol index */
    void ClearSessionID(const tSessionID sessionId, tAppName &appName); /**< clear session id */
    void GetAppNameBySessionID(const tSessionID sessionId, tAppName &appName); /**< get appname by session id*/
    void GetAppNameByProtocolID(const unsigned char protocolId, tAppName &appName,
            tSessionID &sessionId); /**< get appname by protocol id*/
    void GetProtocolNameByProtocolID(const unsigned char protocolId, tProtocolName &protocolName); /**< get ProtocolName by protocol id*/

    tDeviceInfo m_DeviceInfo;               /**< current deviceInfo */
    tMediaObject m_NowPlayingMediaObject;    /**< current nowplaying mediaobject */
    tPlaytime m_ElapsedPlaytime;            /**< current play time */
    tDeviceName m_LaunchApp;                   /**< iPod app to be launched after reauthentication*/
    tBoolean m_TrackFinished;               /**< track has been finished in non-streaming mode, PBS FINISHED send to playerManger */
    tBoolean m_StreamingMode;               /**< streaming mode, remote controlled by device itself */
    tAppName m_FocusApp;                    /**< current focus app */
    tPEHandle m_PEHandle;                    /**< current PE handle */
    tCodecType m_CodecType;                 /**< current BT codec type */
    tBoolean m_LocationInformationUpdate;   /**< enable location info updates */
    tBoolean m_VehicleStatusUpdate;         /**< enable vehicle status updates */
    IPOD_HID_REPORT m_LastHIDReport;        /**< last HID report */

    int m_iPodID;                           /**< iPod id assigned by InitDeviceConnection, platform call */
    int m_Options;                          /**< all iPod feature options */
    int m_NowPlayingTrackIndex;              /**< current playing track index */
    int m_NowPlayingTrackCount;              /**< current playing track count */
    int m_NowPlayingChapterIndex;              /**< current playing chapter index */
    int m_PBTrackIndex;                      /**< requested playing track index */
    int m_AlbumArtID;                        /**< album art formatid (IAP1)/transferid(IAP2) */
    int m_QueueListID;                      /**< queue list transfer id */
    int m_OTGPortNumber;                    /**< OTG Port: 0: target-OTG, 2/3/4 Unwired OTG */
    tPath m_OTGPath;                        /**< OTG Path */
    tPath m_PowerPortPath;                  /**< PowerPort Path */
    tPath m_UDCDevice;                      /**< UDC name */
    tPath m_HubVendorID;                    /**< Hub vendor ID */
    tPath m_HubProductID;                   /**< Hub product ID */
    IPOD_OTG_TYPE m_OTGType;                /**< OTG type */
    tBoolean m_RoleSwitched;                /**< Switch OTG role */
    bool m_IsWaitingforRoleSwitchResponse;  /**< Waiting for roles witch response from SPI */

    void * mp_IAP2Device;                    /**< IAP2 device handle pointer, alias iAP2Device_t */
    void * mp_IAP2InitParameter;            /**< IAP2 init parameter pointer, alias AP2InitParam_t */
    void * mp_IAP2AlbumArtBuffer;            /**< IAP2 album art buffer */
    void * mp_QueueListBuffer;              /**< Queue list buffer */
    Lock m_IAP2Mutex;                          /**< lock access to IAP2 stack */

    int m_PlayerState;                        /**< iPod player state, see iap_types.h */
    tPEPlaybackState m_PlaybackState;        /**< iPod playback state, reflected to mediaplayer */
    tPEPlaybackState m_LastPlaybackState;   /**< last iPod playback state */
    iPodControlMediaPath m_DBPath;             /**< current media DB selection*/
    iPodControlMediaPath m_PBPath;             /**< current media PB selection*/
    iPodControlIndexerContext *mp_Context;    /**< indexer context pointer*/
    Lock m_IndexerMutex;                    /**< lock access to indexer context */

    tPlaybackMode m_PlaybackMode;            /**< playback mode, streaming relevant*/
    tRepeatMode m_RepeatMode;                /**< repeat mode, streaming relevant*/
    bool m_RepeatInitFlag;                  /**< initial device repeatmode override */

    tDiPOCallState m_CallState;             /**< iAP2 call state list */
    tDiPOCommunications m_Communications;   /**< iAP2 communications */
    tBTProfile m_BTProfile;                 /**< iAP2 BT profile */

    int m_PollThreadIndex;                  /**< index of iAP2 poll thread */
    U32 m_CallDuration;                     /** iAP2 call duration */
    tDiPOLocationInfoType m_LocationInfoType;
    tDiPOPower m_PowerUpdate;               /** iAP2 Power Update */
    tDiPORouteGuidanceUpdate m_RGUpdate;
    tDiPORouteGuidanceManeuverUpdate m_RGManeuverUpdate;
    queue<tDiPORouteGuidanceUpdate> m_RGQueueUpdate;
    queue<tDiPORouteGuidanceManeuverUpdate> m_RGManeuverQueueUpdate;
    tDiPOCaps m_diPOCaps;
    tBTLimitationActionState m_btLimitationActionState;
    tConnectionState m_btLimitationConnectionState;
    tBoolean m_isCPWActive;
    tDiPOGPRMCDataStatusValues m_GPRMCDataStatusValues;
    tDiPODeviceTime m_DeviceTimeUpdate;

private:
    map<tiPodElaspedTime, ElapsedTimer*> m_TimerMap;    /**< map of delyed iap calls according to current ATS requirements */
    vector<tIPODAppInfo> m_AppInfos;                    /**< list of iOS application info */
    vector<tIPODAppInfo> m_AppInfosFromSPI;                    /**< list of iOS application info */
};

#endif /*IPODCONTROLHANDLE_H_*/

/** @} */
