/**
 * @addtogroup DeviceControl
 * @author Ulrich Deuper
 *
 * Public interface for iPod control
 * @{
 */

#ifndef IPODCONTROL_H_
#define IPODCONTROL_H_

#include "iPodControlSM.h"
#include "ILocalSPM.h"
#include "ThreadFactory.h"
#include "RequestResponseSM.h"
#include "iPodControlIAP.h"
#include "iPodControlMediaPath.h"
#include <list>
#include <vector>

/**
 * iPod control class
 */
class iPodControl: public iPodControlSM,
        public ILocalSPM,
        public TFThread,
        public iPodControlIAP
{

public:

    //SPM part

    /**
     * Constructor of component.
     * This function is used by LocalSPM to store the iPodControl componentID in mComponentID.
     *
     * @param[in] componentID component ID assigned by SPM
     * @return void
     */
    iPodControl(const tComponentID componentID) :
        ILocalSPM(componentID)
    {
    }
    ;

    /**
     * Destructor of component.
     *
     * @return void
     */
    virtual ~iPodControl()
    {
    }
    ;

    /**
     * This function is used by LocalSPM to create the iPodControl.
     * @attention: running in SPM thread context
     * Create the iPodControl state machine (including create of message queue).
     * Inform LocalSPM that Create is ready -> CreateDone(0)
     *
     * @return void
     */
    void Create();

    /**
     * This function is used by LocalSPM to trigger the iPodControl initialization.
     * @attention: running in SPM thread context
     * Init the iPodControl state machine.
     * Register iPodControlSM with dispatcher.
     * Register for the followning iAP callbacks:
     * - iPodRegisterCBUSBAttach() : attach device signal
     * - iPodRegisterCBUSBDetach() : detach device signals
     * - iPodRegisterCBNotification() : iPod DB changes, focus app changes, etc
     * - iPodRegisterCBNotifyStatus() : track playback state changes
     * - iPodRegisterCBNotifyStateChange() : power state changes
     * - iPodRegisterCBNewiPodTrackInfo() : sample rate and soundvalue changes
     * - iPodRegisterCBOpenDataSession() : opened data session to iOS application
     * - iPodRegisterCBCloseDataSession() : closed data session to iOS application
     * - iPodRegisterCBiPodDataTransfer() : recieved data from iOS application
     * - iPodRegisterCBTrackArtworkData() : recieved data for requested albumArt
     * Inform LocalSPM that Init is ready -> InitDone(0)
     *
     * @return != 0: error, = 0: OK
     */
    tResult Init(tInitReason reason);

    /**
     * Initialize own member variables
     *
     * @return != 0: error, = 0: OK
     */
    tResult InitSM();

    /**
     * This function is used by LocalSPM to start the iPodControl.
     * @attention: running in SPM thread context
     * From now on all other mediaplayer components are available.
     * Start the iPodControl thread and the state machine.
     *
     * @return != 0: error, = 0: OK
     */
    tResult Run();

    /**
     * This function starts a user function in a thread from the thread factory's thread pool.
     * After leaving while loop used thread is stopped and released for other users
     *
     * @return void
     */
    void Do(int functionID, void *ptr);

    /**
     * This function is used by LocalSPM to stop the iPodContorl.
     * @attention: running in SPM thread context
     * Store last mode values.
     * Set state machine to final state -> SendMessage(STOP_SM).
     * LocalSPM will be informed after STOP_SM event is processed
     *
     * @return != 0: error, = 0: OK
     */
    tResult Stop();

    /**
     * Inform LocalSPM that Stop is ready -> StopDone(0)
     *
     * @return != 0: error, = 0: OK
     */
    tResult StopEventProcessed();

    /**
     * This function is used by LocalSPM to cleanup the iPodControl.
     * @attention: running in SPM thread context
     * Deregister iPodControlSM with dispatcher.
     * Set state machine to final state -> SendMessage(DONE).
     * LocalSPM will be informed after DONE event is processed.
     *
     * @return != 0: error, = 0: OK
     */
    tResult Done();

    /**
     * Inform LocalSPM that Done is ready -> DoneDone(0)
     *
     * @return != 0: error, = 0: OK
     */
    tResult DoneEventProcessed();

    /**
     * Returns the current state the state machine is in (for debugging of shutdown problems)
     * @param[inout] stateName buffer for storing the current state name
     *
     * @return pointer to stateName
     */
    char *GetSMStateName(OUT tGeneralString stateName, IN size_t size);

    /**
     * Entry function for IDLE state machine.
    */
    int StartDoTask();

    /**
    * Returns answer if the component is a state machine
    *
    * @return true or false
    */
    tBoolean IsComponentSM() { return true; }; //component is a state machine

    //Playback control part

    /**
     * Function transfers assign_audio_input_device command to PlayerEngine.
     * Send AssignAudioInputDevice to PlayerEngine via DBUS -> AssignAudioInputDevice(m_AudioInputDevice).
     * PlayerEngine command has return value of success.
     *
     * @param[in] deviceID device ID
     * @param[in] mountPoint mount point of device
     * @return != 0: error, = 0: OK
     */
    tResult StartAllocateAudioInput(const tDeviceID deviceID, const tMountPoint mountPoint);

    /**
     * This Function gets Called on receiving the Response from the  AVRCP 1.4 Device
     * for StartAudioPipe Method Call
     * This give the Input Audio Pipe Name,Codec Type,BitRate for the AVRCP 1.4 Device
     *
     * @param[in] pipeName     Input AudioPipe Name
     * @param[in] codecType    CodecType of the Data streamed into the Input AudioPipe
     * @param[in] bitRate      BitRate of the Data streamed into the Input AudioPipe
     * @param[in] message      DBus Error Message(if Any)
     * @return != 0: error, = 0: OK
     */
    tResult StartAudioPipeMethodReturn(tPipeName pipeName,tCodecType codecType,tBitRate bitRate,tErrorMessage message);

    /**
     * Function transfers deallocate_in_device command to PlayerEngine.
     * Send DeAllocateInDevice to PlayerEngine via DBUS.
     * PlayerEngine command has return value of success.
     *
     * @warning Not used yet!
     *
     * Comment: at the moment not used!
     *
     * @return != 0: error, = 0: OK
     */
    tResult StartDeAllocateAudioInput();

    /**
     * Function transfers switch_observer command to PlayerEngine.
     * Switch observer at PlayerEngine via DBUS
     * PlayerEngine command has return value of success.
     *
     * @param[in] deviceID device ID
     * @param[in] mountPoint mount point of device
     * @return != 0: error, = 0: OK
     */
    tResult StartSwitchObserver(const tDeviceID /*deviceID*/, const tMountPoint /*mountPoint*/);

    /**
     * Function transfers playFromDevice command to PlayerEngine.
     * Send playFromDevice to PlayerEngine via DBUS -> playFromDevice(m_AudioInputDevice).
     * PlayerEngine response is sent by PlaybackStatusResponseSlot(CurrentPlaybackStatus=PE_PBS_LOADINGSTATE) later.
     *
     * @param[in] deviceID device ID
     * @param[in] mountPoint mount point of device
     * @return != 0: error, = 0: OK
     */
    tResult StartStreaming(const tDeviceID deviceID, const tMountPoint mountPoint);

    /**
     * Function transfers stopMediaPlayback command to SPI
     * Send stopMediaPlayback to SPI via OutputWrapper -> stopMediaPlayback(deviceID).
     * SPI command has return value of success.
     *
     * @return != 0: error, = 0: OK
     */
    tResult StartStopStreaming();

    /**
     * Function answers of STOP request to waiting state machine via SMF
     * if old PlayerEngine
     * - PlaybackStatus(playbackState=PE_PBS_STOPPEDSTATE, metadata, metadata, metadata, metadata) -> SendAnswer.
     * else //Roadmap 13010
     * - PlaybackStatusNew(handle, playbackState=PE_PBS_STOPPEDSTATE, reason, speed) -> SendAnswer.
     *
     * @param[in] deviceID device ID
     * @return != 0: error, = 0: OK
     */
    tResult StopStreamingAnswer(tDeviceID deviceID);

    /**
     * Function transfers play command to iPod IAP (ADIT layer).
     * Select playback record from iPod device according to URL,
     * start playout selection from position.
     * IAP response is sent by PlayStateChangeSignal(IPOD_CHANGED_PLAY_STATUS).
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] URL file path and name including mount point
     * @param[in] mountPoint mount point of device
     * @param[in] handle handle (= objectID) to map property update to an media object //Roadmap 13010
     * @param[in] position absolute playback start in milliseconds
     * @param[in] streaming streaming flag //Roadmap 13008
     * @return != 0: error, = 0: OK
     */
    tResult StartPlay(const tDeviceType deviceType, const tDeviceID deviceID,
            const tURL URL, const tMountPoint mountPoint,const tUUID uuid,
            const tPEHandle handle, const tPlaytime position=0, const tStreaming streaming=false); //Roadmap 13006 IAP2

    /**
     * Function transfers stop command to PlayerEngine and send iPodPause to iPod IAP (ADIT layer).
     * Send iPodPlayToggle to put iPod device into PAUSED state
     * Send StopSlot to PlayerEngine via DBUS -> StopSlot().
     * PlayerEngine response is sent by PlaybackStatusResponseSlot(CurrentPlaybackStatus=STOPPEDSTATE) later.
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] URL file path and name including mount point
     * @param[in] mountPoint mount point of device
     * @return != 0: error, = 0: OK
     */
    tResult StartStop(const tDeviceType deviceType, const tDeviceID deviceID,
            const tURL URL, const tMountPoint mountPoint = NULL); //Roadmap 13006 IAP2

    /**
     * Function transfers pause command to iPod IAP (ADIT layer).
     * Send iPodPlayToggle to put iPod device into PAUSED state
     * Fake iPod IAP response: Send answer to waiting DeviceControl state machine immediately.
     *    -> SendMessage(PLAYBACK_STATUS_RESPONSE, status, metadata1, metadata2, metadata3, metadata4).
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] URL file path and name including mount point
     * @param[in] mountPoint mount point of device
     * @return != 0: error, = 0: OK
     */
    tResult StartPause(const tDeviceType deviceType, const tDeviceID deviceID,
            const tURL URL, const tMountPoint mountPoint = NULL); //Roadmap 13006 IAP2

    /**
     * Function transfers resume command to iPod IAP (ADIT layer).
     * Send iPodPlayToggle to put iPod device into PLAYING state
     * Fake iPod IAP response: Send answer to waiting DeviceControl state machine immediately.
     *    -> SendMessage(PLAYBACK_STATUS_RESPONSE, status, metadata1, metadata2, metadata3, metadata4).
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] URL file path and name including mount point
     * @param[in] mountPoint mount point of device
     * @return != 0: error, = 0: OK
     */
    tResult StartResume(const tDeviceType deviceType, const tDeviceID deviceID,
            const tURL URL, const tMountPoint mountPoint = NULL); //Roadmap 13006 IAP2

    /**
     * Function transfers ffwd_start command to iPod IAP (ADIT layer).
     * Send iPodPlayControl(START_FF) to iPod IAP.
     * Fake iPod response: Send answer to waiting DeviceControl state machine immediately.
     *    -> SendMessage(PLAYBACK_STATUS_RESPONSE, status, metadata1, metadata2, metadata3, metadata4).
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] URL file path and name including mount point
     * @param[in] mountPoint mount point of device
     * @param[in] rate rate of cueing (10= after every second jump 1s)
     * @return != 0: error, = 0: OK
     */
    tResult StartFfwdStart(const tDeviceType deviceType,
            const tDeviceID deviceID, const tURL URL,
            const tMountPoint mountPoint = NULL,
            const tCueingRate rate = 1, //Roadmap 13006 IAP2
            const speedstate_e IsPlaybackSpeed = ME_SPEEDSTATE_OFF);

    /**
     * Function transfers ffwd_stop command to iPod IAP (ADIT layer).
     * Send iPodPlayControl(END_FF_REW) to iPod IAP.
     * Fake iPod response: Send answer to waiting DeviceControl state machine immediately.
     *    -> SendMessage(PLAYBACK_STATUS_RESPONSE, status, metadata1, metadata2, metadata3, metadata4).
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] URL file path and name including mount point
     * @param[in] mountPoint mount point of device
     * @return != 0: error, = 0: OK
     */
    tResult StartFfwdStop(const tDeviceType deviceType,
            const tDeviceID deviceID, const tURL URL,
            const tMountPoint mountPoint = NULL); //Roadmap 13006 IAP2

    /**
     * Function transfers frev_start command to iPod IAP (ADIT layer).
     * Send iPodPlayControl(START_REW) to iPod IAP.
     * Fake iPod response: Send answer to waiting DeviceControl state machine immediately.
     *    -> SendMessage(PLAYBACK_STATUS_RESPONSE, status, metadata1, metadata2, metadata3, metadata4).
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] URL file path and name including mount point
     * @param[in] mountPoint mount point of device
     * @param[in] rate rate of cueing (10= after every second jump 1s)
     * @return != 0: error, = 0: OK
     */
    tResult StartFrevStart(const tDeviceType deviceType,
            const tDeviceID deviceID,
            const tURL URL,
            const tMountPoint mountPoint = NULL,
            const tCueingRate rate = 1,
            const speedstate_e IsPlaybackSpeed = ME_SPEEDSTATE_OFF); //Roadmap 13006 IAP2

    /**
     * Function transfers frev_stop command to iPod IAP (ADIT layer).
     * Send iPodPlayControl(END_FF_REW) to iPod IAP.
     * Fake iPod response: Send answer to waiting DeviceControl state machine immediately.
     *    -> SendMessage(PLAYBACK_STATUS_RESPONSE, status, metadata1, metadata2, metadata3, metadata4).
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] URL file path and name including mount point
     * @param[in] mountPoint mount point of device
     * @return != 0: error, = 0: OK
     */
    tResult StartFrevStop(const tDeviceType deviceType,
            const tDeviceID deviceID, const tURL URL,
            const tMountPoint mountPoint = NULL); //Roadmap 13006 IAP2

    /**
     * Function transfers seek_to command to device iPod IAP (ADIT layer).
     * Send iPodSetTrackPosition(ABSOLUTE) to iPod IAP.
     * Fake iPod response: Send answer to waiting DeviceControl state machine immediately.
     *    -> SendMessage(PLAYBACK_STATUS_RESPONSE, status, metadata1, metadata2, metadata3, metadata4).
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] URL file path and name including mount point
     * @param[in] mountPoint mount point of device
     * @param[in] position absolute playtime in milliseconds
     * @return != 0: error, = 0: OK
     */
    tResult StartSeekTo(const tDeviceType deviceType, const tDeviceID deviceID,
            const tURL URL, const tMountPoint mountPoint = NULL,
            const tPlaytime position = 0); //Roadmap 13006 IAP2

    /**
     * Function sends answer to waiting state machine triggered by DBUS method return message.
     * Answer via SMF -> SendAnswer(returnValue).
     *
     * @param[in] returnValue success = true, otherwise false.
     * @return != 0: error, = 0: OK
     */
    tResult ActionStatus(const tReturnValue returnValue);

    /**
     * Function sends answer to waiting state machine triggered by DBUS method return message.
     * Answer via SMF -> SendAnswer(success).
     *
     * @param[in] success success of buffer request
     * @return != 0: error, = 0: OK
     */
    tResult BufferStatus(const tSuccess success);

    /**
     * StartAudioDevice
     * Function answers of START_AUDIO_DEVICE request to waiting state machine via SMF
     * Function is used to start mediaengine:play when De-Mute is requested.
     *
     * @return tResult
     */
    tResult StartAudioDevice();

    /**
     * StartAudioDeviceAnswer
     * Function answers of START_AUDIO_DEVICE_ANSWER request to waiting state machine via SMF
     *
     * @param[in] void
     * @return != 0: error, = 0: OK
     */
    tResult StartAudioDeviceAnswer();

    /**
     * StopAudioDevice
     * Function answers of STOP_AUDIO_DEVICE request to waiting state machine via SMF
     * Function is used to stop mediaengine:stop when Mute is requested.
     *
     * @return tResult
     */
    tResult StopAudioDevice();

    /**
     * StopAudioDeviceAnswer
     * Function answers of STOP_AUDIO_DEVICE_ANSWER request to waiting state machine via SMF
     *
     * @param[in] void
     * @return != 0: error, = 0: OK
     */
    tResult StopAudioDeviceAnswer();

    /**
     * Function sends answer to waiting state machine triggered by new playback status.
     * Validate error code -> ValidateErrorCode(metadata2).
     * Answer via SMF -> SendAnswer(status, metadata1, metadata2, metadata3, metadata4, metadataTitle, mediaType).
     *
     * @param[in] status status of current media iPod IAP for current song
     * @param[in] metadata1 coming from IAP with meta data content
     * @param[in] metadata2 coming from IAP with meta data content
     * @param[in] metadata3 coming from IAP with meta data content
     * @param[in] metadata4 coming from IAP with meta data content
     * @return != 0: error, = 0: OK
     */
    tResult PlaybackStatus(const tPEPlaybackState status,
            const tMetadata metadata1,
            const tMetadata metadata2,
            const tMetadata metadata3,
            const tMetadata metadata4,
            const tObjectID ObjectID = OBJECT_ID_NONE);

    /**
     * Function sends playback status to PlayerManager triggered by spontaneous playback status update.
     * Validate error code -> ValidateErrorCode(metadata2).
     * Forward status to PlayerManager -> SendMessage(PLAYBACK_STATUS_RESPONSE, status, metadata1, metadata2, metadata3, metadata4, metadataTitle, mediaType).
     *
     * @param[in] status status of current media iPod IAP for current song
     * @param[in] metadata1 coming from IAP with meta data content
     * @param[in] metadata2 coming from IAP with meta data content
     * @param[in] metadata3 coming from IAP with meta data content
     * @param[in] metadata4 coming from IAP with meta data content
     * @return != 0: error, = 0: OK
     */
    tResult ForwardPlaybackStatus(const tPEPlaybackState status,
            const tMetadata metadata1,
            const tMetadata metadata2,
            const tMetadata metadata3,
            const tMetadata metadata4,
            const tObjectID ObjectID = OBJECT_ID_NONE);

    tResult ForwardNowPlayingStatus(const tPEHandle handle, const tURL url);

    /**
     * Function sends answer to own waiting state machine triggered by SendMessageAnswer request timeout.
     * Release waiting in SMF -> ReleaseWaiting().
     *
     * @return != 0: error, = 0: OK
     */
    tResult HandleAnswerTimeout();

    /**
     * Function sends answer to possible waiting state machine to release own SM in case of message answer pair.
     * Answer to SMF -> SendAnswer().
     *
     * @return != 0: error, = 0: OK
     */
    tResult MessageNotConsumed();
    tResult AlbumArtAnswerNotConsumed(const tAlbumArtObjectPtr ptrToAlbumArtObject);

    tResult SetOutputDevice(const tAudioOutputDevice audioOutputDevice);

    /**
     * Function sends ACTION_ERROR to waiting state machine.
     * Roadmap 13010
     *
     * @param[in] reason error type (REASON_ACTION_ERROR, REASON_DRM_ERROR, REASON_DEVICE_ERROR, ...)
     * @return != 0: error, = 0: OK
     */
    tResult HandleActionError(const me::reason_e reason);//Roadmap 13010: 100%

    /**
     * Function sends ACTION_ERROR to its own state machine
     * Roadmap 13010
     *
     * @param[in] reason error type (REASON_ACTION_ERROR, REASON_DRM_ERROR, REASON_DEVICE_ERROR, ...)
     * @return != 0: error, = 0: OK
     */
    tResult ActionError(const me::reason_e reason);

    /**
     * Function sends STREAM_ERROR to PlayerManager.
     * Roadmap 13010
     *
     * @param[in] reason error type (REASON_ACTION_ERROR, REASON_DRM_ERROR, REASON_DEVICE_ERROR, ...)
     * @return != 0: error, = 0: OK
     */
    tResult ForwardStreamError(const me::reason_e reason) const; //Roadmap 13010: 100%

    /**
     * Function sends answer to waiting state machine triggered by new playback status
     * Answer via SMF -> SendAnswer(handle, status, reason, speed).
     * Roadmap 13008
     *
     * @param[in] handle handle (= objectID) to map property update to an media object
     * @param[in] status status of current media PlayerEngine for current song
     * @param[in] reason reason for status change
     * @param[in] speed playback speed (1=normal play, 10=ffwd, -10=frev)
     * @return != 0: error, = 0: OK
     */
    tResult PlaybackStatusNew(const tPEHandle handle, const tPEPlaybackState playbackState, //Roadmap 13008
            const me::reason_e reason, const me::speed_e speed);

    //Browsing VTIPOD part

    /**
     * Function returns marshaled string from internal VT cache
     * This function will be called by the VTIPOD lib.
     *
     * @param[out] rValue answer string to be unmarshled in VTIPOD/IPodCursor
     * @param[in] mountPoint mount point of device
     * @param[in] listType list type of requested media object
     * @param[in] filterTag1 tag to address the categories depending on listType
     * @param[in] filterTag2 tag to address the categories depending on listType
     * @param[in] filterTag3 tag to address the categories depending on listType
     * @param[in] filterTag4 tag to address the categories depending on listType
     * @param[in] rowID it is the item index within the selected list
     * @param[in] limit it is the limit database value for the selected list slice
     * @param[in] offset it is the offset database value for the selected list slice
     * @return != 0: error, = 0: OK
     */
    tResult GetVTRecordFromCache(string &rValue, const tMountPoint mountPoint,
            const tListType listType, const tFilterTag filterTag1,
            const tFilterTag filterTag2, const tFilterTag filterTag3, const tFilterTag filterTag4,
            const tRowID rowID, const tLimit limit, const tOffset offset);

    /**
     * Function checks metadata of a selected list (type and tags)
     * This function will be called by the VTIPOD lib.
     *
     * @param[in] mountPoint mount point of device
     * @param[in] listType list type of requested media object
     * @param[in] filterTag1 tag to address the categories depending on listType
     * @param[in] filterTag2 tag to address the categories depending on listType
     * @param[in] filterTag3 tag to address the categories depending on listType
     * @param[in] filterTag4 tag to address the categories depending on listType
     * @param[in] rowID it is the item index within the selected list
     * @param[in] limit it is the limit database value for the selected list slice
     * @param[in] offset it is the offset database value for the selected list slice
     * @return != 0: error, = 0: OK
     */
    tResult IsNextVirtualMetadataAvailable(const tMountPoint mountPoint,
            const tListType listType, const tFilterTag filterTag1,
            const tFilterTag filterTag2, const tFilterTag filterTag3, const tFilterTag filterTag4,
            const tRowID rowID, const tLimit limit, const tOffset offset);

    /**
     * Function retrieves metadata of a selected list (type and tags)
     * This function will be called by the VTIPOD lib.
     *
     * @param[in] mountPoint mount point of device
     * @param[in] listType list type of requested media object
     * @param[in] filterTag1 tag to address the categories depending on listType
     * @param[in] filterTag2 tag to address the categories depending on listType
     * @param[in] filterTag3 tag to address the categories depending on listType
     * @param[in] filterTag4 tag to address the categories depending on listType
     * @param[in] rowID it is the item index within the selected list
     * @param[in] limit it is the limit database value for the selected list slice
     * @param[in] offset it is the offset database value for the selected list slice
     * @return != 0: error, = 0: OK
     */
    tResult RetrieveVirtualMetadata(const tMountPoint mountPoint,
            const tListType listType, const tFilterTag filterTag1,
            const tFilterTag filterTag2, const tFilterTag filterTag3, const tFilterTag filterTag4,
            const tRowID rowID, const tLimit limit, const tOffset offset);

    /**
     * Function send metadata of a selected list back to calling smf
     * This function will be used in context of VTIPOD lib.
     *
     * @param[in] mountPoint mount point of device
     * @param[in] listType list type of requested media object
     * @param[in] filterTag1 tag to address the categories depending on listType
     * @param[in] filterTag2 tag to address the categories depending on listType
     * @param[in] filterTag3 tag to address the categories depending on listType
     * @param[in] filterTag4 tag to address the categories depending on listType
     * @param[in] rowID it is the item index within the selected list
     * @param[in] limit it is the limit database value for the selected list slice
     * @param[in] offset it is the offset database value for the selected list slice
     * @return != 0: error, = 0: OK
     */
    tResult SendPutVirtualMetadata(const tMountPoint mountPoint,
            const tListType listType, const tFilterTag filterTag1,
            const tFilterTag filterTag2, const tFilterTag filterTag3, const tFilterTag filterTag4,
            const tRowID rowID, const tLimit limit, const tOffset offset);

    /**
     * Function calls umount to the file system in order to detach it from the system.
     * Roadmap 13003 SDCard
     * NOT SUPPORTED BY IPOD
     *
     * By use of a local RRSM send UMOUNT to DeviceControl via RouteMessageAnswer, synchronous call.
     *
     * @param[in] deviceID deviceID to be unmounted
     * @param[in] mountPoint mountPoint of device
     * @return != 0: error, = 0: OK
     */
    tResult Umount(const tDeviceID deviceID, const tMountPoint mountPoint); //Roadmap 13003

    //Indexing part

    /**
     * Function decides if device needs an initialization when INIT_DEVICE_CONNECTION is signaled.
     * In case of iPod authentication is required with Apple Coprocessor located on T-Engine side.
     * Authentication sequence is run from ADIT layer automatically when InitDeviceConnection(mountPoint)
     * is called. There will be a response callback iPodAttach providing success/error status.
     *
     * @param[in] mountPoint mount point of device
     * @param[in] deviceID device ID
     * @return true or false
     */
    tResult
    IsInitRequired(const tDeviceInfoString deviceInfoString);

    /**
     * Function sends internal signal INIT_DEVICE(mountPoint).
     * For iPod this signal will trigger authentication and device initialization to extended accessory mode.
     *
     * @param[in] mountPoint mount point of device
     * @param[in] deviceID device ID
     * @return != 0: error, = 0: OK
     */
    tResult
    SendInitInit(const tDeviceInfoString deviceInfoString);

    /**
     * Function used to response to caller in order to handle device types
     * that do not need special initialization, e.g. USB or SD card
     * Get device properties from DBManager -> GetDeviceInfo(&deviceInfo, mountPoint)
     * Answer via SMF -> SendAnswer(deviceInfo.deviceName,deviceID,connectionState=CS_CONNECTED).
     *
     * @param[in] mountPoint mount point of device
     * @param[in] deviceID device ID
     * @return != 0: error, = 0: OK
     */
    tResult
    NoInitAnswer(const tDeviceInfoString deviceInfoString);

    /**
     * Function checks if device is already initialized properly or malfunction has been detected.
     * In case of iPod authentication is required with Apple Coprozessor located on T-Engine side.
     * Authentication sequence is run from ADIT layer automatically when InitDeviceConnection(mountPoint)
     * is called. There will be a response callback iPodAttach providing success/error status.
     * Delivers answer to the "isInitDeviceConnection" condition.
     * Store current mount point in an internal handle map-> StoreMountpoint
     * Return true if success, otherwise false.
     *
     * @param[in] mountPoint mount point of device
     * @return true or false
     */
    int IsInitDeviceConnection(const tMountPoint mountPoint, const tInitDeviceProtocol protocol); //Roadmap 13006 IAP2

    /**
     * Function is called if iPodInitDeviceConnection failed.
     * iPod IAP (ADIT layer) needs to be closed if initialization try failed.
     * Device will be marked as malfunction for the current connection session.
     *
     * @param[in] mountPoint mount point of device
     * @return != 0: error, = 0: OK
     */
    tResult Disconnect(const tMountPoint mountPoint, const tInitDeviceProtocol protocol);
    tResult ModeDisconnect(const tMountPoint mountPoint, const tInitDeviceProtocol protocol);

    /**
     * Function entered if authentication procedure succeeded.
     * Here for iPod all feature options will be called and iPod name is stored within handle map
     * Answer via SMF -> SendAnswer(deviceName,deviceID,connectionState).
     *
     * @param[in] mountpoint iPod device ID
     * @param[in] connectionState device connection state [CS_CONNECTED or CS_HW_MALFUNCTION]
     * @return != 0: error, = 0: OK
     */
    tResult OnDeviceInitialized(const tMountPoint mountPoint, const tConnectionState connectionState, const tInitDeviceProtocol protocol);
    tResult OnModeInitialized(const tMountPoint mountPoint, const tConnectionState connectionState, const tInitDeviceProtocol protocol);

    /**
     * Function retrieves the total number of playable files
     *
     * @param[in] deviceID device ID
     * @return != 0: error, = 0: OK
     */
    tResult GetNumberOfFiles(const tDeviceID deviceID);

    /**
     * Function retrieves the fingerprint of the device content in order to check for modifications.
     * In case of iPod IAP the iTunes last sync date is processed.
     * The number of media files for indexing will be counted.
     * Send answer to own waiting state machine after result is available.
     *    -> SendMessage(PUT_FINGERPRINT, fingerprint, fingerprintStatus).
     *
     * @param[in] mountPoint mount point of device
     * @param[in] deviceID device ID
     * @param[in] lastFingerprint status of the last fingerprint read from db
     * @return != 0: error, = 0: OK
     */
    tResult CalcFingerprint(const tMountPoint mountPoint, const tDeviceID deviceID, const tFingerprint lastFingerprint); //Roadmap 13006 IAP2

    /**
     * Function sends answer to waiting state machine triggered by new fingerprint status (PUT_FINGERPRINT).
     * Answer via SMF -> SendAnswer(fingerprint, fingerprintStatus, numberOfMediaFiles).
     *
     * @param[in] fingerprint device identification
     * @param[in] fingerprintStatus status of the fingerprint answer [OK, NOT_AVAIL, ERROR, DELTA]
     * @param[in] numberOfFiles number of files to index
     * @return != 0: error, = 0: OK
     */
    tResult OnFingerprint(const tFingerprint fingerprint,
            const tFingerprintStatus fingerprintStatus,
            const tNumberOfFiles numberOfFiles, const tDeviceID deviceID);

    tDeviceID GetLastFingerprintDeviceID() {return m_LastFingerprintDeviceID;};    /**< Fingerprint timer */

    /**
     * Function delivers answer to condition.
     * Check cached list of metadata records already retrieved from iPod.
     * Return true if cache is not empty, otherwise false.
     *
     * @param[in] mountPoint mount point of device
     * @param[in] readPosition position of next media object to read
     * @param[in] deviceID device ID
     * @return true or false
     */
    tResult IsNextMetadataAvailable(const tMountPoint mountPoint,
            const tReadPosition readPosition,
            const tDeviceID deviceID);

    /**
     * Function retrieves a batch of metadata records from iPod database.
     * Batchsize needs to be dynamically calculated depending on device performance.
     * Extraction logic is depending on the iDevice supported features.
     * New iOS device may use iPodGetDBInfo(), legacy device use fallback RetrieveCategoriezedDBRecords().
     * Based on the readPosition this function is able to continue extraction continuously
     * within one virtual list containg all music tracks, audiobooks, podcast and videos, etc.
     *
     * Select iPod device -> SelectDevice(mountPoint)
     * Collect batch size of meta data from iPod -> LoadDataFromIPod(mountPoint, ...)
     * Fill cache -> FillCache()
     * Send answer to own waiting state machine after collecting meta data is finished.
     *    -> SendMessage(PUT_METADATA, metadataStatus).
     *
     * @param[in] mountPoint mount point of device
     * @param[in] readPosition position of next media object to read
     * @param[in] deviceID device ID
     * @return != 0: error, = 0: OK
     */
    tResult RetrieveMetadataFromDevice(const tMountPoint mountPoint,
            const tReadPosition readPosition,
            const tDeviceID deviceID);

    /**
     * Function sends internal signal PUT_METADATA(metadataStatus).
     * Send answer to own waiting state machine because meta data already available in cache.
     *    -> SendMessage(PUT_METADATA, metadataStatus).
     *
     * @param[in] mountPoint mount point of device
     * @param[in] readPosition position of next media object to read
     * @param[in] deviceID device ID
     * @return != 0: error, = 0: OK
     */
    tResult SendPutMetadata(const tMountPoint mountPoint,
            const tReadPosition readPosition,
            const tDeviceID deviceID);


    /**
     * Function calls SendPutMetadata.
     * Send answer to own waiting state machine because meta data already available in cache.
     *    -> SendMessage(PUT_METADATA, metadataStatus).
     *
     * @param[in] mountPoint mount point of device
     * @param[in] readPosition position of next media object to read
     * @param[in] deviceID device ID
     * @return != 0: error, = 0: OK
     */
    tResult ReceivedMetadata(const tMountPoint mountPoint, //Roadmap 13006 IAP2
            const tReadPosition readPosition,
            const tDeviceID deviceID);

    /**
     * Function sends answer to waiting state machine triggered by metadata available in cache (PUT_METADATA).
     * Get meta data from own cache -> RetrieveMetadataFromCache(&metadata).
     * Answer via SMF -> SendAnswer(metadata, metadataStatus).
     *
     * @param[in] metadataStatus status of meta data collection [SUCCESS, FINISHED, FILE_ERROR, DEVICE_ERROR, REMOVED]
     * @param[in] mediaObjectPtr
     * @return != 0: error, = 0: OK
     */
    tResult AnswerMetadata(const tMetadataStatus metadataStatus, const tMediaObjectPtr mediaObjectPtr); //Roadmap 13006 IAP2

    /**
     * Function retrieves the meta data of one media object from cache
     *
     * @param[out] metadata meta data of one media object
     * @return != 0: error, = 0: OK
     */
    tResult RetrieveMetadataFromCache(tMetadata &metadata);

    /**
     * Function stores indexing last mode for removed device and resets scan context.
     *
     * @param[in] mountPoint mount point of device
     * @param[in] deviceID device ID
     * @return != 0: error, = 0: OK
     */
    tResult RemoveDeviceConnection(const tMountPoint mountPoint, const tDeviceID deviceID);
    tResult RemoveDeviceConnectionForced(const tMountPoint mountPoint, const tDeviceID deviceID);
    tResult RemoveDeviceConnectionOnHold(const tMountPoint mountPoint, const tDeviceID deviceID);
    tResult RoleSwitchDeviceRemoved(const tMountPoint mountPoint, const tDeviceID deviceID);
    tResult RoleSwitchDeviceRemovedForced(const tMountPoint mountPoint, const tDeviceID deviceID);   //NCG3D-89502

    //AlbumArt part

    /**
     * Function triggers DoGetAlbumArt in a seperate working thread
     *
     * DoGetAlbumArt gets the album art from media object and stores it in a member variable.
     * Allocate memory for internal album art object.
     * Get album art from Taglib -> GetAlbumArt(&albumArtObject, albumArtString)
     * Answer via SMF -> SendAnswer(ptrToAlbumArtObject).
     * @attention The caller must free this memory if not used anymore.
     *
     * @param[in] albumArtString file name and path (URL) of image to display
     * @return != 0: error, = 0: OK
     */
    tResult GetAlbumArt(const tAlbumArt albumArtString);

    /**
     * Function sends answer to waiting state machine triggered by answer from DoGetAlbumArt
     * working thread (GET_ALBUM_ART_ANSWER).
     * Answer via SMF -> SendAnswer(ptrToAlbumArtObject).
     *
     * @param[in] ptrToAlbumArtObject pointer to allocated album art object
     * @return != 0: error, = 0: OK
     */
    tResult HandleGetAlbumArtAnswer(const tAlbumArtObjectPtr ptrToAlbumArtObject);

    //App control methods
    /**
     * Function sends event to own sm via RRSM and
     * writes song tag information to iPod storage
     *
     * @param[in] mountPoint mount point of device
     * @param[in] transferTags vector of tags to transfer
     * @param[out] transferStatus tagging status
     * @param[out] untransferredTags vector of tags not transferred
     * @return != 0: error, = 0: OK
     */
    tResult SendTransferTags(const tMountPoint mountPoint,
            const vector<tTagTransfer> transferTags,
            tTagTransferStatus &transferStatus,
            vector<tUntransferredTag> &untransferredTags);

    /**
     * Function writes tag file data to ipod
     *
     * @param[in] mountPoint mount point of device
     * @param[in] tagfile name of tag file
     * @return != 0: error, = 0: OK
     */
    tResult TransferTag(const tMountPoint mountPoint, const tFilename tagfile);

    /**
     * Function sends event about transfer tag status back to waiting state machine
     *
     * @param[in] status status of tag transfer
     * @return != 0: error, = 0: OK
     */
    tResult TransferTagAnswer(const tTagTransferStatus status);

    /**
     * The AppControl Connect method provides the interface for the Internet Radio to tell the
     * iPod device control to connect to the Pandora application using the MediaPlayer.
     *
     * @param[in] mountPoint a string containing the iPhone device HID name.
     * @param[in] protocol a string containing protocol of the IAP application.
     * @param[in] bundleSeedID a string containing the bundle seed ID of the IAP application.
     * @param[in] appName a string containing App name of the IAP application.
     * @return < 0: error, = 0: OK
     */
    tResult SendAppControlConnect(const tMountPoint mountPoint,
            const tProtocolName protocol, const tBundleSeedID bundleSeedID,
            const tAppName appName, const tAppLaunchOption launchOption);

    /**
     * Function adds new app info for iOS application.
     * If info has been added a re-authentication is needed to open a data session to the new app
     * --> RECONNECT()
     * Otherwise LaunchApp will be called by event
     * --> LAUNCH_APP()
     * The app to be launched will be stored to member for use in AppControlConnectReconnected
     *
     * @param[in] mountPoint ?
     * @param[in] protocol ?
     * @param[in] bundleSeedID ?
     * @param[in] appName ?
     * @return < 0: error, = 0: OK
     */
    tResult AddAppControlInfo(const tMountPoint mountPoint, const tProtocolName protocol,
            const tBundleSeedID bundleSeedID, const tAppName appName, const tAppLaunchOption launchOption);

    /**
     * Function removes app info for iOS application
     * If info has been removed a re-authentication is needed to close a data session to the app
     * --> RECONNECT()
     * else no action required
     * --> APPCONTROL_ANSWER()
     *
     * @param[in] mountPoint ?
     * @param[in] appName ?
     * @param[in] sessionID ?
     * @return bool added
     */
    tResult RemoveAppControlInfo(const tMountPoint mountPoint,
            const tAppName appName, const tSessionID sessionID);

    /**
     * Function entered if authentication procedure succeeded in context of app control connect.
     * device has to be identified first, see IdentifyDevice(), followed by a LaunchApp() call
     * App to launch was stored in member by AddAppControlInfo()
     * AppControlAnswer() to release the calling outer SM is called within LaunchApp()
     *
     * @param[in] mountPoint iPod device ID
     * @param[in] connectionState device connection state [CS_CONNECTED or CS_HW_MALFUNCTION]
     * @return != 0: error, = 0: OK
     */
    tResult AppControlConnectReconnected(const tMountPoint mountPoint,
            const tConnectionState connectionState, const tInitDeviceProtocol protocol);

    /**
     * Function entered if authentication procedure succeeded in context of app control close.
     * device has to be identified first, see IdentifyDevice().
     * Finally a call to AppControlAnswer() releases the calling outer SM
     *
     * @param[in] mountPoint iPod device ID
     * @param[in] connectionState device connection state [CS_CONNECTED or CS_HW_MALFUNCTION]
     * @return != 0: error, = 0: OK
     */
    tResult AppControlCloseReconnected(const tMountPoint mountPoint,
            const tConnectionState connectionState, const tInitDeviceProtocol protocol);

    /**
     * Function sends event back to waiting state machine
     *
     * @return != 0: error, = 0: OK
     */
    tResult AppControlAnswer();

    /**
     * The AppControl Command method provides the interface for the Internet Radio to send
     * an IAP command to the MediaPlayer.
     *
     * @param[in] mountPoint a string containing the iPhone device HID name.
     * @param[in] protocol a string containing protocol of the IAP application.
     * @param[in] sessionID The session ID of the connected IAP application.
     * @param[in] sizeOfBuffer size of the command buffer
     * @param[in] commandBuffer IAP command.
     * @return < 0: error, = 0: OK
     */
    tResult SendAppControlCommand(const tMountPoint mountPoint, const tAppName appName,
            const tSessionID sessionID, const tSize sizeOfBuffer,
            const tCommandBufferPtr commandBuffer, const tUserContext userContext);

    /**
     * Function sends command to iOS application
     *
     * @param[in] mountPoint
     * @param[in] appName
     * @param[in] sessionID
     * @param[in] sizeOfBuffer
     * @param[in] commandBuffer
     * @return != 0: error, = 0: OK
     */
    tResult AppControlCommand(const tMountPoint mountPoint,
            const tAppName appName, const tSessionID sessionID, const tSize sizeOfBuffer,
            const tCommandBufferPtr commandBuffer);

    /**
     * The AppControl Close method provides the interface for the Internet Radio to send an IAP
     * command to the MediaPlayer to terminate the IAP session with the Pandora application.
     *
     * @param[in] mountPoint a string containing the iPhone device HID name.
     * @param[in] protocol a string containing protocol of the IAP application.
     * @param[in] sessionID The session ID of the connected IAP application.
     * @return < 0: error, = 0: OK
     */
    tResult SendAppControlClose(const tMountPoint mountPoint, const tAppName appName,
            const tSessionID sessionID);

    /**
     * Function disconnects and calls initDeviceConnection again.
     * App infos have to be kept during disconnect
     * time out has to be defined to handle authentication errors
     * --> ON_DEVICE_ATTACHED()
     *
     * @param[in] mountPoint
     * @return != 0: error, = 0: OK
     */
    tResult Reconnect(const tMountPoint mountPoint);

    /**
     * Function request app launch stored in member by IsAppControlInfoAdded()
     * Calls AppControlAnswer() releases the calling outer SM
     *
     * @param[in] mountPoint
     * @param[in] appName
     * @return != 0: error, = 0: OK
     */
    tResult LaunchApp(const tMountPoint mountPoint, const tAppName appName, const tAppLaunchOption launchOption);

    //iPod specific part

    /**
     * Function delivers flag if selection was changed via IAP since device connection on specific device
     * Function is used in context of iPod last mode resuming
     *
     * @param[out] changed selection has changed
     * @param[in] deviceID device ID of affected device
     * @return != 0: error, = 0: OK
     */
    tResult CurrentSelectionChanged(tBoolean &changed, const tDeviceID deviceID);

    /**
     * Function delivers current playtime on specific device
     * Function is used in context of iPod last mode resuming
     *
     * @param[out] playtime current elapsed playtime
     * @param[in] deviceID device ID of affected device
     * @return != 0: error, = 0: OK
     */
    tResult GetCurrentPlaytime(tPlaytime &playtime, const tDeviceID deviceID);

    /**
     * Function delivers flag if IAP2 is active for the device addressed by mountpoint
     *
     * @param[in] mountPoint mountPoint of affected device
     * @return != 0: error, = 0: OK
     */
    tBoolean IsIAP2(const tMountPoint mountPoint);

    /**
     * Function compares two media object
     * Function is used in context of iPod last mode resuming
     *
     * @param[out] identical media objects
     * @param[in] media object 1 to compare
     * @param[in] playtime1 elapsed playtime of media object 1 to compare
     * @param[in] media object 2 to compare
     * @param[in] playtime2 elapsed playtime of media object 2 to compare
     * @return != 0: error, = 0: OK
     */
    static tResult CompareMediaObjects(tBoolean &identical,
            const tMediaObject media1, const tPlaytime playtime1,
            const tMediaObject media2, const tPlaytime playtime2);


    /**
         * Function extracts filter tags from url
         *
         * @param[out] tag tag1 of objects
         * @param[out] tag tag2 of objects
         * @param[out] tag tag3 of objects
         * @param[out] tag tag4 of objects
         * @param[in] url url of object
         * @return != 0: error, = 0: OK
         */
    static tResult GetVTTagByURL(tTagID &tag, tListType &lty, tMetadata &name, tUUID &uuid, const char* url);


    /**
     * Streaming mode only:
     * Function transfers generic playback command, like NEXT, PREV, etc
     * to iPod IAP (ADIT layer).
     * No answer message required on PlayerManager level.
     * IAP response is sent spontaneously, handled by PlayStateChangeSignal(IPOD_CHANGED_PLAY_STATUS).
     *
     * In Non-Streaming mode this function does nothing.
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] URL file path and name including mount point
     * @param[in] mountPoint mount point of device
     * @param[in] playbackAction playback action
     * @param[in] skipcount, number of tracks to be skipped by NEXT, PREV
     * @return != 0: error, = 0: OK
     */
    tResult StartPlaybackAction(const tDeviceType deviceType, const tDeviceID deviceID, //Roadmap 13008 RemoteControl
            const tURL URL, const tMountPoint mountPoint,
            const tPlaybackAction playbackAction, const tNextPrevSkipCount skipcount=0);

    /**
     * Streaming mode only:
     * Function set the playback mode (shuffle on/off) on the device
     * by calling appropriate iap methods (ADIT layer).
     * No answer message required on PlayerManager level.
     * IAP response is sent spontaneously, handled by RemoteEventNotification.
     *
     * In case that the device setting change failed, an update is triggered
     * to PlayerManager sending the current active setting -> PLAYBACK_MODE_STATUS
     *
     * In Non-Streaming mode this function does nothing
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] mountPoint mount point of device
     * @param[in] playbackMode, shuffle/normal mode
     * @return != 0: error, = 0: OK
     */
    tResult SetPlaybackMode(const tDeviceType deviceType,   //Roadmap 13008 RemoteControl
            const tDeviceID deviceID, const tMountPoint mountPoint, const tPlaybackMode playbackMode);

    /**
     * Streaming mode only:
     * Function set the repeat mode (all,one,off) on the device
     * by calling appropriate iap methods (ADIT layer).
     * No answer message required on PlayerManager level.
     * IAP response is sent spontaneously, handled by RemoteEventNotification.
     *
     * In case that the device setting change failed, an update is triggered
     * to PlayerManager sending the current active setting -> REPEAT_MODE_STATUS
     *
     * In Non-Streaming mode this function does nothing
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] mountPoint mount point of device
     * @param[in] repeatMode, repeat all,one,off
     * @return != 0: error, = 0: OK
     */
    tResult SetRepeatMode(const tDeviceType deviceType,     //Roadmap 13008 RemoteControl
            const tDeviceID deviceID, const tMountPoint mountPoint,
            const tRepeatMode repeatMode);

    /**
     * Streaming mode only:
     * Function sned event REMOTE_ACTIVITY to PlayerManager
     * Within iPodControl context.
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] mountPoint mount point of device
     * @param[in] repeatMode, repeat all,one,off
     * @return != 0: error, = 0: OK
     */
    tResult RemoteActivity(const tMountPoint mountPoint); //Roadmap 13008 RemoteControl

    /**
     * Streaming mode only:
     * Function called by PlayerManager as a response whe entered Streaming mode
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] mountPoint mount point of device
     * @return != 0: error, = 0: OK
     */
    tResult StreamingModeOn(const tDeviceType deviceType,       //Roadmap 13008 RemoteControl
            const tDeviceID deviceID, const tMountPoint mountPoint);

    /**
     * Streaming mode only:
     * Function called by iPodControl SM to clear all live entries in VT cache (LTY_CURRENT_SELECTION
     * Executed within iPodControl context.
     *
     * @param[in] mountPoint mount point of device
     * @return != 0: error, = 0: OK
     */
    tResult ClearVTCache(const tMountPoint mountPoint); //Roadmap 13008 RemoteControl

    tResult SendRemoteActivity(const tMountPoint mountPoint);
    tResult SetRemoteActivity(const tDeviceID deviceID); //CarPlay Mode
    void iAP2CheckRemoteActivity(const tMountPoint mountPoint, const tBoolean isStreaming, const tBoolean hostMode, tBoolean &remoteActivitySent);
    void iAP1CheckRemoteActivity(const tMountPoint mountPoint, const tBoolean isStreaming);

    /**
    * MediaPlayer is informed when a device is used as DiPO device.
    *
    * @param[in] deviceID Unique identifier of the device.
    * @param[in] diPOActive "True" if DiPO session is active. "False" if DiPO session is not active
    * @param[in] userContext user context structure. Is transmitted to OutputWrapper without changes
    * @return < 0: error, = 0: OK
    */
    tResult SendDiPOActiveDevice(const tDeviceID deviceID, const tDiPOActive diPOActive, const tUserContext userContext);

    /**
    * Requests a USB role switch: From USB client to USB host or vice versa
    *
    * @param[in] deviceID Unique identifier of the device.
    * @param[in] diPORoleStatus Role status
    * @param[in] userContext user context structure. Is transmitted to OutputWrapper without changes
    * @return < 0: error, = 0: OK
    */
    tResult SendDiPORoleSwitchRequest(const tDeviceID deviceID, const tDiPORoleStatus diPORoleStatus, const tUserContext userContext, const tAppInfo appInfo);
    tResult OnDiPORoleSwitchRequiredResult(const tDeviceID deviceID, const tDiPOSwitchReqResponse diPOSwitchReqResponse,const tAppInfo appInfo);

    /**
    *  @param[in] mountPoint mountpoint ofApple device for which iAP2 over Wireless should be established.
    */
    tResult SendDiPOStartIAP2OverCarplayWiFi(const tMountPoint mountPoint);
    tResult PrepareIAP2OverCarplayWiFi(const tMountPoint mountPoint);
    /**
     * Function sends event back to waiting state machine
     *
     * @return != 0: error, = 0: OK
     */
    tResult RoleSwitchAnswer();
    tResult PrepareRoleSwitch(const tMountPoint mountPoint, const tDiPORoleStatus diPORoleStatus);

    tResult RoleSwitchRequired(const tDeviceID deviceID);
    tResult RoleSwitchRequiredAnswer(const tDeviceID deviceID, const tDiPOSwitchReqResponse diPOSwitchReqResponse);

    /**
    * Transfers GPS data to DiPO device
    *
    * @param[in] deviceID Unique identifier of the device.
    * @param[in] diPOGPGGAData GPS data in GPGGA format as specified by apple
    * @param[in] diPOGPRMCData GPS data in GPRMC format as specified by apple
    * @param[in] userContext user context structure. Is transmitted to OutputWrapper without changes
    * @return < 0: error, = 0: OK
    */
    tResult SendDiPOTransferGPSData(const tDeviceID deviceID, const tDiPOGPGGAData diPOGPGGAData,const tDiPOGPRMCData diPOGPRMCData, const tDiPOGPGSVData diPOGPGSVData, const tDiPOGPHDTData diPOGPHDTData, const tUserContext userContext);

    /**
       * Generic function - non SM blocking
       * @return < 0: error, = 0: OK
       */
    tResult SignalUserData(const tDeviceID deviceID, const tUserDataType userDataType, const tUserData userData);

    /**
          * Generic function - SM blocking
          * @return < 0: error, = 0: OK
          */
    tResult ProcessUserData(const tDeviceID deviceID, const tUserDataType userDataType, const tUserData userData);
    tResult ProcessUserDataAnswer(const tDeviceID deviceID, const tReturnValue retVal);
    tResult DoProcessUserData(char *parameterString);

    /**
    * Transfers GPS data to DiPO device
    *
    * @param[in] deviceID Unique identifier of the device.
    * @param[in] diPOPASCDData Vehicle speed data arrang in the format as specified by apple
    * @param[in] diPOPAGCDData vehicle gyro data arrang in the format as specified by apple
    * @param[in] userContext user context structure. Is transmitted to OutputWrapper without changes
    * @return < 0: error, = 0: OK
    */
    tResult SendDiPOTransferDRData(const tDeviceID deviceID, const tDiPOPASCDData diPOPASCDData, const tDiPOPAGCDData diPOPAGCDData, const tDiPOPAACDData diPOPAACDData, const tUserContext userContext);

    /**
    * Store the AccessoryWiFiCredentials from SPI.
    *
    * @param[in] wifiCredentials Accessory WiFi Access Point Credentials.
    */
    void AccessoryWiFiCredentials(const tDeviceID deviceId,const tWiFiAPCredentials wifiCredentials);

    /**
     * OutsideTemperatureChanged
     * Function is used to send vehicle status update to iAP2 device during CarPlay
     *
     * @param[in] void
     * @return void
     */
    tResult OutsideTemperatureChanged();

    /**
    * SendDiPOInitiateCall
    * Function is used to Initiate Call
    * @param[in] deviceID
    * @param[in] initiateCall
    * @return tResult; OK=MP_NO_ERROR else NOK.
    */
    tResult SendDiPOInitiateCall(const tDeviceID deviceID, const tDiPOInitiateCall initiateCall);

    /**
    * SendDiPOAcceptCall
    * Function is used to Accept Call
    * @param[in] deviceID
    * @param[in] acceptCall
    * @return tResult; OK=MP_NO_ERROR else NOK.
    */
    tResult SendDiPOAcceptCall(const tDeviceID deviceID, const tDiPOAcceptCall acceptCall);

    /**
    * SendDiPOEndCall
    * Function is used to End Call
    * @param[in] deviceID
    * @param[in] endCall
    * @return tResult; OK=MP_NO_ERROR else NOK.
    */
    tResult SendDiPOEndCall(const tDeviceID deviceID, const tDiPOEndCall endCall);
     /**
     * SendSwapCalls
     * Function is used to Swap Calls ( iAP2_SwapCalls )
     *
     * @param[in] deviceID
     * @return void
     */
    tResult SendSwapCalls(const tDeviceID deviceID);

    /**
     * SendStart/StopAudio
     * Function is used to start/stop iAP2USBDeviceModeAudio
     *
     * @param[in] mountpoint
     * @return void
     */
    tResult SendStartAudio(const tMountPoint mountPoint);
    tResult SendStopAudio(const tMountPoint mountPoint);

    /**
     * SendPlaybackAction
     * Function is used to start playback actions on iPod (from AppControl FI, mySPIN)
     *
     * @param[in] deviceID
     * @return void
     */
    tResult SendPlaybackAction( const tDeviceID deviceID, const tPlaybackAction playbackAction, const tNextPrevSkipCount nextPrevSkipCount);

    /**
     * GetBTProfileInfo
     * Function is used to get BT profile information from iAP2 (host and device mode)
     *
     * @param[out] profileList
     * @return void
     */
    tResult GetBTProfileInfo(tBTProfileList &profileList);

    /**
     * GetDiPONowPlaying
     * Function is used to get nowplaying information from iAP2 (host mode only)
     *
     * @param[out] deviceID device ID
     * @param[out] nowPlaying
     * @return void
     */
    tResult GetDiPONowPlaying(tDeviceID &deviceID, tDiPONowPlaying &nowPlaying);

    /**
     * GetDiPOPlaybackStatus
     * Function is used to get playback status information from iAP2 (host mode only)
     *
     * @param[out] deviceID device ID
     * @param[out] playbackState playback statue
     * @return void
     */
    tResult GetDiPOPlaybackStatus(tDeviceID &deviceID, tHMIPlaybackState &playbackState);

    /**
     * GetDiPOPlaybackShuffleMode
     * Function is used to get shuffle information from iAP2 (host mode only)
     *
     * @param[out] deviceID device ID
     * @param[out] playbackMode playback shuffle statue
     * @return void
     */
    tResult GetDiPOPlaybackShuffleMode(tDeviceID &deviceID, tPlaybackMode &playbackMode);

    /**
     * GetDiPOPlaybackRepeatMode
     * Function is used to get repeat information from iAP2 (host mode only)
     *
     * @param[out] deviceID device ID
     * @param[out] repeatMode playback repeat mode
     * @return < 0: error, = 0: OK
     */
    tResult GetDiPOPlaybackRepeatMode(tDeviceID &deviceID, tRepeatMode &repeatMode);

    /**
     * GetDiPOPlaytime
     * Function is used to get playtime information from iAP2 (host mode only)
     *
     * @param[out] deviceID device ID
     * @param[out] elapsedPlaytime elapsed playtime of current media object
     * @param[out] totalPlaytime total playtime of current media object
     * @return < 0: error, = 0: OK
     */
    tResult GetDiPOPlaytime(tDeviceID &deviceID, tPlaytime &elapsedPlaytime, tPlaytime &totalPlaytime);

    /**
     * GetDiPOCallState
     * Function is used to get CallState information from iAP2 (host mode only)
     *
     * @param[out] deviceID device ID
     * @param[out] callStateList list of call states
     * @return < 0: error, = 0: OK
     */
    tResult GetDiPOCallState(tDeviceID &deviceID, tDiPOCallState &callStateList);

    tResult GetDiPOCallDuration(tDeviceID &deviceID, U32 &callDuration);

    /**
     * GetDiPOCommunications
     * Function is used to get Communications information from iAP2 (host mode only)
     *
     * @param[out] deviceID device ID
     * @param[out] communications communications
     * @return < 0: error, = 0: OK
     */
    tResult GetDiPOCommunications(tDeviceID &deviceID, tDiPOCommunications &communications);

    /**
     * GetDiPOLocationInfo
     * Function is used to get DiPOLocation information from iAP2
     */
    tResult GetDiPOLocationInfo(tDeviceID &deviceID, tBool &startStopLocationInfo, tDiPOLocationInfoType &locationInfoType);

    /**
     * GetDiPOGPRMCDataStatusValues
     * Function is used to get GPRMC Data status values notification(A,V,X) from iAP2
     */
    tResult GetDiPOGPRMCDataStatusValues(tDeviceID &deviceID, tMountPoint &USBSerialNumber, tDiPOGPRMCDataStatusValues &GPRMCDataStatusValues);

    tResult GetRGUpdate(tDeviceID &deviceID, tDiPORouteGuidanceUpdate &RGUpdate);
    tResult GetRGManeuverUpdate(tDeviceID &deviceID, tDiPORouteGuidanceManeuverUpdate &RGManeuverUpdate);
    tResult PullRGUpdate(tDeviceID &deviceID, tDiPORouteGuidanceUpdate &RGUpdate);
    tResult PullRGManeuverUpdate(tDeviceID &deviceID, tDiPORouteGuidanceManeuverUpdate &RGManeuverUpdate);
    /**
     * IsBatchPlayable
     * Function is used to check support for batch list playback support
     *
     * @param[in] deviceID
     * @return tBoolean
     */
    tBoolean IsBatchPlayable(const tDeviceID deviceID);

    /**
     * ReadAuthParameterFromRootFS
     * Function is used read/load iPod authentication
     * values (GPIO reset pin) from root filesystem
     * (/etc/pfcfg/IPOD_AUTH.cfg)
     *
     * @param[in] void
     * @return void
     */

    static void ReadAuthParameterFromRootFS();

    /**
     * Check for CarPlay device
     * Function is used by BTControl::Stop to handle a special Apple sequence
     * NCG3D-15134
     *
     * @param[in] tDeviceName
     * @return bool
     */
    bool IsActiveAsCarPlay(const tDeviceName deviceName);

    /**
     * DBChange functions
     * Function is used to access DBManager from another thread than iPodControlSM
     *
     * @param[in] deviceID
     * @return tResult
     */
    tResult SendDBChangeDeviceType(const tDeviceID deviceID, const tDeviceType newDeviceType);
    tResult SendDBChangeDeviceTypeAndMountPoint(const tDeviceID deviceID, const tDeviceType newDeviceType, const tMountPoint mountPoint);
    tResult SendDBChangeDeviceName(const tDeviceID deviceID, const tDeviceName newDeviceName);
    tResult SendDBChangeDiPOSettings(const tDeviceID deviceID, const tDiPOCaps diPOCaps, const tDiPOActive diPOActive, const tDiPOVersion diPOVersion);
    tResult SendDBChangeConnectionState(const tDeviceID deviceID, const tConnectionState newConnectionState);
    tResult SendDBChangeNowPlayingListAvailable(const tMountPoint mountPoint, const int queueListID, tNowPlayingListAvailable ClearNowPlayingList);
    tResult SendDBSetDeviceUUID(const tDeviceID, const tUUID deviceUUID);
    tResult SendDBSetAppleDeviceMACAddress(const tDeviceID, const tMACAddress appleDeviceMACAddress);
    tResult SendDBsetAppleDeviceTransportIdentifiers(const tDeviceID deviceID,const tMACAddress appleDeviceMACAddress,const tUSBSerial appleDeviceUSBSerialNumber);
    tResult SendDBChangeDeviceState(const tDeviceID deviceID, const tDeviceState newDeviceState);

    tResult SendSetRepeatMode(const tMountPoint mountPoint, tRepeatMode repeatMode);
    tBoolean IsInitializingDevice(const tMountPoint mountPoint,tConnectionState connectionState,tInitDeviceProtocol protocol);
    tResult SendDiPOStartRouteGuidance(const tDeviceID deviceID,tRouteGuidanceDisplayComponentIDList RouteGuidanceDisplayComponentIDs, tBoolean SourceName, tBoolean SourceSupportsRouteGuidance);
    tResult SendDiPOStopRouteGuidance(const tDeviceID deviceID, tRouteGuidanceDisplayComponentIDList RouteGuidanceDisplayComponentIDs);
    // CMG3G-14540
    void SendSetBTLimitationMode(tMACAddress btMacAddress,tBTLimitationActionState btLimitationActionState);
    tResult SendAppleHIDCommand(const tPlaybackHIDCommand playbackHIDCommand,const tBTButtonEvent keyEvent);
    /**
     * iAP callbacks
     * Function is used to just send message back to iPodControl state machine
     *
     * @param[in] various
     * @return void
     */
    tResult OnCBIAP1_USBDetach(const S32 iPodID);                           /**< callback USB detach */
    tResult OnCBIAP1_USBAttach(const S32 success, const S32 iPodID);         /**< callback USB attach */
    tResult OnCBIAP1_Notification(IPOD_NOTIFY_TYPE type, IPOD_NOTIFY_STATUS status,
            const S32 iPodID);                                          /**< callback event notification */
#ifdef TARGET_BUILD_GEN3
    tResult OnCBIAP1_RemoteEventNotification(IPOD_STATE_INFO_TYPE eventNum, IPOD_REMOTE_EVENT_NOTIFY_STATUS eventData, const U32 iPodID);
#endif
    tResult OnCBIAP1_NotifyStatus(IPOD_CHANGED_PLAY_STATUS status, tU64 param,
            const S32 iPodID);                                          /**< callback play status change notification */
    tResult OnCBIAP1_NotifyStateChange(IPOD_STATE_CHANGE stateChange,
            const S32 iPodID);                                          /**< callback power status change notification */
    tResult OnCBIAP1_NewiPodTrackInfo(U32 uNewSampleRate, const S32 iPodID); /**< callback new audio track info */
    tResult OnCBIAP1_OpenDataSession(unsigned char protocolIndex,
                unsigned short sessionId, const S32 iPodID);            /**< callback opened session to app info */
    tResult OnCBIAP1_CloseDataSession(unsigned short sessionId, const S32 iPodID); /**< callback closed session to app */
    tResult OnCBIAP1_DataTransfer(unsigned short sessionId, unsigned char *data,
                unsigned short length, const S32 iPodID);               /**< callback app sent data */
    tResult OnCBIAP1_RetrieveCategorizedDBRecord(U32 index, U8* str IPODCONTROL_ID_END_PARAM);        /**< callback DB Record */
    tResult OnCBIAP1_TrackInfo(U64 index, IPOD_TRACK_INFORMATION_TYPE infoType,
            IPOD_TRACK_INFORMATION_DATA *infoData IPODCONTROL_ID_END_PARAM);    /**< callback TrackInfo */
    tResult OnCBIAP1_PlayingTrackInfo(IPOD_TRACK_INFO_TYPE infoType,
            const IPOD_TRACK_CAP_INFO_DATA *capData,
            const IPOD_TRACK_RELEASE_DATE_DATA *releaseData,
            const IPOD_TRACK_ARTWORK_COUNT_DATA *artworkCountData,
            U8* stringBuf IPODCONTROL_ID_END_PARAM);                            /**< callback IndexedPlayingTrackInfo */
    tResult OnCBIAP1_ArtworkData(IPOD_ALBUM_ARTWORK *artworkData,
            const S32 iPodID);                                          /**< callback IndexedPlayingTrackInfo */

    //IAP2 callbacks
#ifdef IPODCONTROL_IAP2_PF_AVAIL
    tResult OnCBIAP2_DeviceState(/*const*/ iAP2Device_t* iap2Device, const iAP2DeviceState_t dState, void* context); /**< callback IAP2 device state */
    tResult OnCBIAP2_IdentificationAccepted(iAP2Device_t* iap2Device, iAP2IdentificationAcceptedParameter* idParameter, void* context);
    tResult OnCBIAP2_DeviceInformationUpdate(iAP2Device_t* iap2Device, iAP2DeviceInformationUpdateParameter *deviceInformationUpdateParameter, void* context);
    tResult OnCBIAP2_DeviceLanguageUpdate(iAP2Device_t* iap2Device, iAP2DeviceLanguageUpdateParameter *deviceLanguageUpdateParameter, void* context);
    tResult OnCBIAP2_NowPlayingUpdate(iAP2Device_t* iap2Device, iAP2NowPlayingUpdateParameter* nowPlayingUpdateParameter, void* context);
    tResult OnCBIAP2_USBDeviceModeAudioInformation(iAP2Device_t* iap2Device, iAP2USBDeviceModeAudioInformationParameter* usbDeviceModeAudioInformationParameter, void* context);
    tResult OnCBIAP2_FileTransferSetup(const iAP2Device_t* iap2Device, iAP2FileTransferSession_t* iAP2FileXferSession, void* context);
    tResult OnCBIAP2_FileTransferDataRcvd(const iAP2Device_t* iap2Device, iAP2FileTransferSession_t* iAP2FileXferSession, void* context);
    tResult OnCBIAP2_FileTransferSuccess(const iAP2Device_t* iap2Device, iAP2FileTransferSession_t* iAP2FileXferSession, void* context);
    tResult OnCBIAP2_FileTransferFailure(const iAP2Device_t* iap2Device, iAP2FileTransferSession_t* iAP2FileXferSession, void* context);
    tResult OnCBIAP2_FileTransferCancel(const iAP2Device_t* iap2Device, iAP2FileTransferSession_t* iAP2FileXferSession, void* context);
    tResult OnCBIAP2_FileTransferPause(const iAP2Device_t* iap2Device, iAP2FileTransferSession_t* iAP2FileXferSession, void* context);
    tResult OnCBIAP2_FileTransferResume(const iAP2Device_t* iap2Device, iAP2FileTransferSession_t* iAP2FileXferSession, void* context);
    tResult OnCBIAP2_MediaLibraryInfo(iAP2Device_t* iap2Device, iAP2MediaLibraryInformationParameter* mediaLibraryInfoParameter, void* context);
    tResult OnCBIAP2_MediaLibraryUpdates(iAP2Device_t* iap2Device, iAP2MediaLibraryUpdateParameter* mediaLibraryUpdateParameter, void* context);
    tResult OnCBIAP2_StartExternalAccessoryProtocolSession(iAP2Device_t* iap2Device, iAP2StartExternalAccessoryProtocolSessionParameter* startExternalAccessoryProtocolSessionParameter, void* context);
    tResult OnCBIAP2_StopExternalAccessoryProtocolSession(iAP2Device_t* iap2Device, iAP2StopExternalAccessoryProtocolSessionParameter* stopExternalAccessoryProtocolSessionParameter, void* context);
    tResult OnCBIAP2_StartEANativeTransport(iAP2Device_t* iap2Device, U8 iAP2iOSAppIdentifier, U8 sinkEndpoint, U8 sourceEndpoint, void* context);
    tResult OnCBIAP2_StopEANativeTransport(iAP2Device_t* iap2Device, U8 iAP2iOSAppIdentifier, U8 sinkEndpoint, U8 sourceEndpoint, void* context);
    tResult OnCBIAP2_iOSAppDataReceived(iAP2Device_t* iap2Device, U8 iAP2iOSAppIdentifier, U8* iAP2iOSAppDataRxd, U16 iAP2iOSAppDataLength, void* context);
    tResult OnCBIAP2_StartLocationInformation(iAP2Device_t* iap2Device, iAP2StartLocationInformationParameter* startLocationInformationParameter, void* context);
    tResult OnCBIAP2_StopLocationInformation(iAP2Device_t* iap2Device, iAP2StopLocationInformationParameter* stopLocationInformationParameter, void* context);
    tResult OnCBIAP2_GPRMCDataStatusValuesNotification(iAP2Device_t* iap2Device, iAP2GPRMCDataStatusValuesNotificationParameter* GPRMCDataStatusValuesNotificationParameter, void* /*context*/);
    tResult OnCBIAP2_BluetoothConnectionUpdate(iAP2Device_t* iap2Device, iAP2BluetoothConnectionUpdateParameter* bluetoothConnectionUpdateParameter, void* context);
    tResult OnCBIAP2_StartVehicleStatusUpdates(iAP2Device_t* iap2Device, iAP2StartVehicleStatusUpdatesParameter* startVehicleStatusUpdatesParameter, void* context);
    tResult OnCBIAP2_StopVehicleStatusUpdates(iAP2Device_t* iap2Device, iAP2StopVehicleStatusUpdatesParameter* stopVehicleStatusUpdatesParameter, void* context);
#ifdef IPODCONTROL_IAP2_PF_OOBBT_AVAIL
#ifdef TARGET_BUILD
    tResult OnCBIAP2_OOBBTPairingLinkKeyInformation(iAP2Device_t* iap2Device, iAP2OOBBTPairingLinkKeyInformationParameter* oobBTPairingLinkKeyInformationParameter, void* context);
    tResult OnCBIAP2_StartOOBBTPairing(iAP2Device_t* iap2Device, iAP2StartOOBBTPairingParameter* startOOBBTPairingParameter, void* context);
#endif
#endif
#ifdef IPODCONTROL_IAP2_PF_R22
    tResult OnCBIAP2_CallStateUpdate(iAP2Device_t* iap2Device, iAP2CallStateUpdateParameter* callStateUpdateParameter, void* context);
    tResult OnCBIAP2_CommunicationsUpdate(iAP2Device_t* iap2Device, iAP2CommunicationsUpdateParameter* communicationsUpdateParameter, void* context);
#else
    tResult OnCBIAP2_TelephonyCallStateInformation(iAP2Device_t* iap2Device, iAP2TelephonyCallStateInformationParameter* telephonyCallStateInformationParameter, void* context);
    tResult OnCBIAP2_TelephonyUpdate(iAP2Device_t* iap2Device, iAP2TelephonyUpdateParameter* telephonyUpdateParameter, void* context);
#endif //IPODCONTROL_IAP2_PF_R22
    tResult OnCBIAP2_DeviceUUIDUpdate(iAP2Device_t *iap2Device, iAP2DeviceUUIDUpdateParameter* deviceUUIDUpdateParameter, void* context);
    tResult OnCBIAP2_WirelessCarPlayUpdate(iAP2Device_t* iap2Device, iAP2WirelessCarPlayUpdateParameter* wirelessCarPlayUpdateParameter, void* context);
    tResult OnCBIAP2_RequestAccessoryWiFiConfigurationInformation(iAP2Device_t* iap2Device, iAP2RequestAccessoryWiFiConfigurationInformationParameter *requestAccessoryWiFiConfigurationInformationParameter, void* context);
#ifdef IPODCONTROL_IAP2_PF_R26
#ifdef TARGET_BUILD
    tResult OnCBIAP2_DeviceTransportIdentifierNotification(iAP2Device_t* iap2Device,iAP2DeviceTransportIdentifierNotificationParameter *deviceTransportIdentifierNotificationParameter, void* context);
#endif
#endif
    tResult OnCBIAP2_ListUpdate(iAP2Device_t* iap2Device, iAP2ListUpdateParameter* listUpdateParameter, void* context);
#endif //IPODCONTROL_IAP2_PF_AVAIL
    tResult OnCBIAP2_PowerUpdate(iAP2Device_t* iap2Device, iAP2PowerUpdateParameter* cup, void* context);
    tResult OnCBIAP2_RouteGuidanceUpdate(iAP2Device_t* iap2Device, iAP2RouteGuidanceUpdateParameter* cup, void* context);
    tResult OnCBIAP2_RouteGuidanceManeuverUpdate(iAP2Device_t* iap2Device, iAP2RouteGuidanceManeuverUpdateParameter* cup, void* context);
    tResult OnCBIAP2_DeviceTimeUpdate(iAP2Device_t* iap2Device, iAP2DeviceTimeUpdateParameter *deviceTimeParameter, void* /*context*/);
    //Test
    void Test(int value=0);
    tResult GetDiPOPower(tDeviceID &deviceID, tDiPOPower &power);

    tResult AddOobPairedDeviceResult(const tDeviceID deviceID,const tBoolean result);
    tResult SetCurrentLanguage(const tLanguageType language);
    tResult GetDiPODeviceTime(tDeviceID &deviceID, tDiPODeviceTime &deviceTime);

private:
    tResult SendPlaybackStatus(const tMountPoint mountPoint,
            const tPEPlaybackState status = PE_PBS_LOADINGSTATE);       /**< send current playback status to own state machine */
    tResult SendPlaybackStatusToPlayerManager(const tMountPoint mountPoint,
            const tPEPlaybackState status);                             /**< send current playback status to PlayerManager state machine */

    tResult SendNowPlayingStatus(const tPEHandle handle, const tURL url = NULL);
    tResult SendNowPlayingStatusToPlayerManager(const tMountPoint mountPoint);
    tResult SendAlbumArtStatusToPlayerManager( const tMountPoint mountPoint);

    tResult SendPlaybackModeToPlayerManager(const tMountPoint mountPoint);  //Roadmap 13010, 13008
    tResult SendRepeatModeToPlayerManager(const tMountPoint mountPoint);    //Roadmap 13010, 13008

    tResult SendClearVTCache(const tMountPoint mountPoint);

    tResult OnFocusAppChanges(const tMountPoint mountPoint);            /**< calling outputwrapper **/
    tResult SendFinishedStatusToPlayerManager(const tMountPoint mountPoint);    /**< send finished status to PlayerManager state machine */
    tResult SendPlaytimeStatusToPlayerManager(const tMountPoint mountPoint); /**< send current playtime status to PlayerManager state machine */
    tResult SendInitDevice(const tMountPoint mountPoint,
            const tInitDeviceProtocol protocol);                        /**< send init device message to own state machine */
    tResult SendDeviceInitialized(const tMountPoint mountPoint);        /**< send response to indexer state machine */
    tResult SendPauseOnTrackEnd(const tMountPoint mountPoint);          /**< send pause signal to own state machine, synch call */
    tResult SendAlbumArtAnswer(const tAlbumArtObjectPtr pAlbumArtObjectPtr);/**< send album art answer signal to own state machine, synch call */
    tResult SendMethodResult(const tReturnValue returnValue);           /**< send method return as event from playerengine */
    tResult SendSignalUserDataEvent(const tDeviceID deviceID, const tUserDataType& userDataType, const tUserData& userData);

    tResult CreateTagFile(tFilename &tagFile, const tTagTransfer transferTag);              /**< write tag info to tmp file */

    tResult DoAppControlConnect(char *parameterString); /**< dispatched from service by thread factory */
    tResult DoAppControlCommand(char *parameterString); /**< dispatched from service by thread factory */
    tResult DoAppControlClose(char *parameterString);   /**< dispatched from service by thread factory */
    tResult DoStopPlaybackOnDetach(char *mountPoint);   /**< dispatched from service by thread factory */
    void DoIAP2Poll(char* mountPoint);                  /**< dispatched from service by thread factory */
    tResult DoRoleSwitch(char *mountPoint);             /**< dispatched from service by thread factory */
    tResult DoPlaybackAction(char *parameterString);    /**< dispatched from service by thread factory */
    tResult DoChangeSampleRate(char *parameterString);  /**< dispatched from service by thread factory */

    tResult OnSampleRateChange(const tMountPoint mountPoint, const me::samplerate_i newSampleRate);

    void StartFingerprintTimer(const tDeviceID deviceID);
    void StopFingerprintTimer(const tDeviceID deviceID = DEVICE_ID_NOT_SET);
    static bool FingerprintTimerCallBack(timer_t timerID , void* instance ,const void *userData);
    void TrackEndDetection(const tMountPoint mountPoint, const tPEPlaybackState iPodPBStatus, const tPlaytime lastElapsedPlaytime);

    tResult FileCountComp(const tMountPoint mountPoint,const tIndexSession indexSession);
    tResult FileCountCompAnswer(const tDeviceID deviceID, const tFingerprintStatus fingerPrintStatus,const tIndexSession indexSession);
    tResult PlaylistNameComp(const tMountPoint mountPoint,const tDeviceID deviceID,const tIndex index,const tListSize chunk,const tIndexSession indexSession);
    tResult PlaylistNameCompAnswer(const tDeviceID deviceID,const tFingerprintStatus fingerPrintStatus,const tIndex index,const tListSize chunk,const tIndexSession indexSession);
    tResult VideoUIDComp(const tMountPoint mountPoint,const tDeviceID deviceID,const tIndex index,const tListSize chunk,const tIndexSession indexSession);
    tResult VideoUIDCompAnswer(const tDeviceID deviceID,const tFingerprintStatus fingerPrintStatus,const tIndex index,const tListSize chunk,const tIndexSession indexSession);
    tResult TrackUIDComp(const tMountPoint mountPoint, const tDeviceID deviceID, const tIndex index, const tListSize chunk,const tIndexSession indexSession);
    tResult TrackUIDCompAnswer(const tDeviceID deviceID, const tFingerprintStatus fingerPrintStatus, const tIndex index, const tListSize chunk,const tIndexSession indexSession);


    tBool bCheckUdevForExpectedHub(unsigned int uProductID, unsigned int uVendorID, const char *f_strProduct, const char *f_strBcdDevice, tBool bStrBcdDeviceNotEqualThis = FALSE);


    void UpdateAppControl(const tDeviceID deviceID, const tAppControlUpdateType updateType);
    void InitAccessoryWifiCredentials();
    tResult OnUpdateReadyToPlay(const tMountPoint mountPoint);
    void SetIsWirelessCarplayActive(const tMountPoint mountPoint);
    void SendUpdateReadyToPlay(const tMountPoint mountPoint);

    tBoolean StartAlbumArtTimer(const tMountPoint mountPoint);
    void StopAlbumArtTimer();
    static bool AlbumArtTimerCallBack(timer_t /*timerID */, void* instance ,const void * /*userData*/);
    void CheckMultipleIAPSessionsForSameDevice(const tDeviceID deviceID, const tUUID deviceUUID);

    // Fix for 736874
    // In internal state WAITINGFOR_MODEINIT, if INIT_DEVICE event is received due to second device detection,
    // then store the event parameters in iPodControl SM.
    // And when state machine returns to IDLE state, retrigger INIT_DEVICE event with the required parameters.
    tResult StoreInitDeviceEvent(const tMountPoint mountPoint, const tInitDeviceProtocol protocol);

    tMountPoint m_ActivePBMountPoint;       /**< mountpoint of currently playing iPod, with respect to playerengien allocation */
    me::samplerate_i m_ActivePBSampleRate;
    tAudioOutputDevice m_AudioOutputDevice;
    tDeviceID m_RoleSwitchDeviceID;
    tDeviceID m_RemoteActivityDeviceID;

    Timer m_FingerprintTimer;               /**< Fingerprint timer */
    timer_t m_FingerprintTimerID;           /**< Fingerprint timer */
    tDeviceID m_LastFingerprintDeviceID;    /**< Fingerprint timer */
    tIndex m_IndexSession;
    tAudioInputDevice m_AudioInputDevice;   //Muthu: Fix for CAF-2023
    tWiFiAPCredentials m_AccessoryWiFiCredentials;
    Lock m_AccessoryWiFiCredentialsMutex;
    tMountPoint m_DeviceToInitMountPoint;
    tBool m_IsNowPlayingListAvail;          //NCG3D-72271 NCG3D-52542, If nowplayinglist is available from device, It will have value as True else False
    Timer m_AlbumArtTimer;
    timer_t m_AlbumArtTimerID;

    tMountPoint m_mountPointPendingInitDevice;
    tInitDeviceProtocol m_protocolPendingInitDevice;
    tBool m_bPendingInitDevice;

};

/* use a request / response state machine for transfer tag */
class TransferTagRR: public RequestResponseSM {
    int HandleInitRequest() {
        transferStatus = TAG_TRANSFER_SUCCESS;
        return 0;
    }

    int HandleSuccessRequest(const char *allParameters) {

        if (!allParameters || strlen_r(allParameters) == 0) {
            transferStatus = TAG_TRANSFER_COMM_ERROR;
        } else {
            /* read out transfer tag status */
            SMF::UnMarshal(IN allParameters, "i", OUT &transferStatus);
        }
        return 0;
    }

    int HandleEntryWaitingRequest() {
        return 0;
    }
public:
    tTagTransferStatus transferStatus;
};

/* use a request / response state machine for thread jobs control */
class iPodControlRR: public RequestResponseSM {
    int HandleInitRequest() {
        result = 0;
        return 0;
    }

    int HandleSuccessRequest(const char *allParameters) {
        if (allParameters && strlen_r(allParameters) > 0) {
            UnMarshal(allParameters, "i", &result);
        }
        return 0;
    }

    int HandleEntryWaitingRequest() {
        return 0;
    }
public:
    int result;
};

#endif /*IPODCONTROL_H_*/

/** @} */
