/**
 * @addtogroup DeviceControl
 * @author Christian Koechling
 *
 * Public interface for MTP control
 * @{
 */

#ifndef _MTPCONTROL_H_
#define _MTPCONTROL_H_

/*-----------------------------------------------------------------------------*
 * Includes                                                                    *
 *-----------------------------------------------------------------------------*/
#include "MTPControlSM.h"
#include "ILocalSPM.h"
#include "ThreadFactory.h"
#include "MTPLibWrapper.h"

#include <libmtp.h>
#include <vector>

/*-----------------------------------------------------------------------------*
 * Defines                                                                     *
 *-----------------------------------------------------------------------------*/
#define LOCAL_MTPFILE_RESERVEDSIZE     10000 /*Kbyte*/
#define LOCAL_MTPFILE_PATH_TARGET      "/tmp/mtp"                       //"/var/opt/bosch/dynamic/media/mtp"
#define LOCAL_MTPFILE_INFO             "tmpMtpFile_info.txt"
#define LOCAL_MTPFILE_BIN              "tmpMtpFile_info.bin"
#define LOCAL_MTPFILE                  "tmpMtpFile"

//-> used for communication between test and MTPControl
#define STRING_TRESULT_OK              "ok  :res==MP_NO_ERROR"
#define STRING_TRESULT_NOK             "nok :res!=MP_NO_ERROR"
#define STRING_MDS_SUCCESS             "ok  :MDS_SUCCESS"
#define STRING_MDS_FINISHED            "ok  :MDS_FINISHED"
#define STRING_MDS_FILE_SKIP           "ok  :MDS_FILE_SKIP"
#define STRING_MDS_FILE_ERROR          "nok :MDS_FILE_ERROR"
#define STRING_MDS_DEVICE_ERROR        "nok :MDS_DEVICE_ERROR"
//<-

#define DEVTABLE_MAXELEMS 16
#define FILETOPLAY_MAXELEMS 1

#define REALDEVICECONNECTED_YES (int)1
#define REALDEVICECONNECTED_NO  (int)0

#define TEST_MSGQ_NAME                 (char*)"TESTQ"


typedef enum
{
    FOLDER_READ_NOT_STARTED = 0UL,
    FOLDER_READ_PARTIAL,
    FOLDER_READ_SUBFOLDER_STARTED,
    FOLDER_READ_SUBFOLDER_COMPLETE,
    FOLDER_READ_ALL_COMPLETE,
    FOLDER_READ_NEXT_STORAGE
} tFolderReadStatus;

#define MTP_ROOT_FOLDER_ID              -1

/*------------------------------------------------------------------------------*
 * typedef enums
 *------------------------------------------------------------------------------*/
typedef enum MTPWRAPPER_TYPE_E
{
    eMTPWrapper_undefined = 0x0,
    eMTPWrapper_realLib   = 0x1,
    eMTPWrapper_stubLib   = 0x2

}eMtpWrapperType;

static int m_indexDefault;


class MTPControl : public MTPControlSM , public ILocalSPM, public TFThread
{

public:

//SPM part

    /**
     * Constructor of component.
     * This function is used by LocalSPM to store the MTPControl componentID in mComponentID.
     *
     * @param[in] componentID component ID assigned by SPM
     * @param[in] pMTPLibWrapper pointer to MTPLibWrapper
     * @return void
     */
    //MTPControl(const tComponentID componentID = COMPONENT_ID_UNKNOWN):ILocalSPM(componentID){};
    MTPControl(const tComponentID componentID = COMPONENT_ID_UNKNOWN);

    /**
     * Destructor of component.
     *
     * @return void
     */
    virtual ~MTPControl(){};

    /**
     * This function is used by LocalSPM to create the MTPControl.
     * @attention: running in SPM thread context
     * Create the MTPControl state machine (including create of message queue).
     * Inform LocalSPM that Create is ready -> CreateDone(0)
     *
     * @return void
     */
    void Create();

    /**
     * This function is used by LocalSPM to trigger the MTPControl initialization.
     * @attention: running in SPM thread context
     * Init the MTPControl state machine.
     * Register MTPControlSM with dispatcher.
     * Inform LocalSPM that Init is ready -> InitDone(0)
     *
     * @return != 0: error, = 0: OK
     */
    tResult Init(tInitReason reason);

    /**
     * Initialize own member variables
     *
     * @return != 0: error, = 0: OK
     */
    tResult InitSM();

    /**
     * This function is used by LocalSPM to start the MTPControl.
     * @attention: running in SPM thread context
     * From now on all other mediaplayer components are available.
     * Start the MTPControl thread and the state machine.
     *
     * @return != 0: error, = 0: OK
     */
    tResult Run();

    /**
     * This function starts a user function in a thread from the thread factory's thread pool.
     * After leaving while loop used thread is stopped and released for other users
     *
     * @return void
     */
    void Do(int functionID, void *ptr);

    /**
     * This function is used by LocalSPM to stop the MTPControl.
     * @attention: running in SPM thread context
     * Store last mode values.
     * Set state machine to final state -> SendMessage(STOP_SM).
     * LocalSPM will be informed after STOP_SM event is processed
     *
     * @return != 0: error, = 0: OK
     */
    tResult Stop();

    /**
     * Inform LocalSPM that Stop is ready -> StopDone(0)
     *
     * @return != 0: error, = 0: OK
     */
    tResult StopEventProcessed();

    /**
     * This function is used by LocalSPM to cleanup the MTPControl.
     * @attention: running in SPM thread context
     * Deregister MTPControlSM with dispatcher.
     * Set state machine to final state -> SendMessage(DONE).
     * LocalSPM will be informed after DONE event is processed.
     *
     * @return != 0: error, = 0: OK
     */
    tResult Done();

    /**
     * Inform LocalSPM that Done is ready -> DoneDone(0)
     *
     * @return != 0: error, = 0: OK
     */
    tResult DoneEventProcessed();

    /**
     * Returns the current state the state machine is in (for debugging of shutdown problems)
     * @param[inout] stateName buffer for storing the current state name
     *
     * @return pointer to stateName
     */
    char *GetSMStateName(OUT tGeneralString stateName, IN size_t size);

    /**
    * Returns answer if the component is a state machine
    *
    * @return true or false
    */
    tBoolean IsComponentSM() { return true; }; //component is a state machine

//Playback control part

    /**
     * Function transfers assign_audio_input_device command to PlayerEngine.
     * For MTP no need to assign audio input device.
     * Send answer via SMF immediately -> SendAnswer(returnValue = true).
     *
     * @param[in] deviceID device ID
     * @param[in] mountPoint mount point of device
     * @return != 0: error, = 0: OK
     */
    tResult StartAllocateAudioInput(const tDeviceID deviceID, const tMountPoint mountPoint=NULL);

    /**
     * Function transfers deallocate_in_device command to PlayerEngine.
     * For MTP no need to deallocate audio input device.
     * Send answer via SMF immediately -> SendAnswer(returnValue = true).
     *
     * @warning Not used yet!
     *
     * Comment: at the moment not used!
     *
     * @return != 0: error, = 0: OK
     */
    tResult StartDeAllocateAudioInput();

    /**
     * Function transfers switch_observer command to PlayerEngine.
     * Switch observer at PlayerEngine via DBUS.
     * PlayerEngine command has return value of success.
     *
     * @param[in] deviceID device ID
     * @param[in] mountPoint mount point of device
     * @return != 0: error, = 0: OK
     */
    tResult StartSwitchObserver(const tDeviceID /*deviceID*/, const tMountPoint /*mountPoint*/);

    /**
     * Function transfers ...
     * For MTP no need to start streaming.
     * if old PlayerEngine
     * - Send event to own SM immediately -> SendEvent(PLAYBACK_STATUS_RESPONSE).
     * else //Roadmap 13010
     * - Send answer via SMF immediately -> SendAnswer(handle, playbackState=PE_PBS_LOADINGSTATE, reason, speed).
     * - Release own waiting state -> SendEvent(START_STREAMING_ANSWER).
     *
     * @param[in] deviceID device ID
     * @param[in] mountPoint mount point of device
     * @return != 0: error, = 0: OK
     */
    tResult StartStreaming(const tDeviceID deviceID, const tMountPoint mountPoint=NULL);

    /**
     * Function transfers ...
     * For MTP no need to stop streaming
     * Release own waiting state -> SendEvent(STOP_STREAMING_ANSWER, deviceID=DEVICE_ID_NOT_SET).
     *
     * @return != 0: error, = 0: OK
     */
    tResult StartStopStreaming();

    /**
     * Function answers of STOP request to waiting state machine via SMF
     * if old PlayerEngine
     * - PlaybackStatus(playbackState=PE_PBS_STOPPEDSTATE, metadata, metadata, metadata, metadata) -> SendAnswer.
     * else //Roadmap 13010
     * - PlaybackStatusNew(handle, playbackState=PE_PBS_STOPPEDSTATE, reason, speed) -> SendAnswer.
     *
     * @param[in] deviceID device ID
     * @return != 0: error, = 0: OK
     */
    tResult StopStreamingAnswer(tDeviceID deviceID);

    /**
     * Function transfers play command to PlayerEngine.
     * Send NewPlaySlot to PlayerEngine via DBUS -> NewPlaySlot(URL, position, playpointFormat=Absolute).
     * PlayerEngine response is sent by PlaybackStatusResponseSlot(CurrentPlaybackStatus=PLAYINGSTATE) later.
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] URL file path and name including mount point
     * @param[in] mountPoint mount point of device
     * @param[in] handle handle (= objectID) to map property update to an media object //Roadmap 13010
     * @param[in] position absolute playtime in milliseconds
     * @param[in] streaming streaming flag //Roadmap 13008
     * @return != 0: error, = 0: OK
     */
    tResult StartPlay(const tDeviceType deviceType, //Roadmap 13008
        const tDeviceID deviceID,
        const tURL URL,
        const tMountPoint mountPoint,
        const tUUID uuid,
        const tPEHandle handle,
        const tPlaytime position=0,
        const tStreaming streaming=false);

    /**
     * Function transfers stop command to PlayerEngine.
     * Send StopSlot to PlayerEngine via DBUS -> StopSlot().
     * PlayerEngine response is sent by PlaybackStatusResponseSlot(CurrentPlaybackStatus=STOPPEDSTATE) later.
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] URL file path and name including mount point
     * @param[in] mountPoint mount point of device
     * @return != 0: error, = 0: OK
     */
    tResult StartStop(const tDeviceType deviceType,
        const tDeviceID deviceID,
        const tURL URL,
        const tMountPoint mountPoint=NULL);

    /**
     * Function transfers pause command to PlayerEngine.
     * Send PauseSlot to PlayerEngine via DBUS -> PauseSlot().
     * Fake PlayerEngine response: Send answer to waiting DeviceControl state machine immediately.
     *    -> SendMessage(PLAYBACK_STATUS_RESPONSE, status, metadata1, metadata2, metadata3, metadata4).
     * PlayerEngine response is sent by PlaybackStatusResponseSlot(CurrentPlaybackStatus=PAUSEDSTATE) later, but we don't use it.
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] URL file path and name including mount point
     * @param[in] mountPoint mount point of device
     * @return != 0: error, = 0: OK
     */
    tResult StartPause(const tDeviceType deviceType,
        const tDeviceID deviceID,
        const tURL URL,
        const tMountPoint mountPoint=NULL);

    /**
     * Function transfers resume command to PlayerEngine.
     * Send ResumePlaySlot to PlayerEngine via DBUS -> ResumePlaySlot().
     * Fake PlayerEngine response: Send answer to waiting DeviceControl state machine immediately.
     *    -> SendMessage(PLAYBACK_STATUS_RESPONSE, status, metadata1, metadata2, metadata3, metadata4).
     * PlayerEngine response is sent by PlaybackStatusResponseSlot(CurrentPlaybackStatus=PLAYINGSTATE) later, but we don't use it.
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] URL file path and name including mount point
     * @param[in] mountPoint mount point of device
     * @return != 0: error, = 0: OK
     */
    tResult StartResume(const tDeviceType deviceType,
        const tDeviceID deviceID,
        const tURL URL,
        const tMountPoint mountPoint=NULL);

    /**
     * Function transfers ffwd_start command to PlayerEngine.
     * Send StartFfSlot to PlayerEngine via DBUS -> StartFfSlot(rate).
     * Fake PlayerEngine response: Send answer to waiting DeviceControl state machine immediately.
     *    -> SendMessage(PLAYBACK_STATUS_RESPONSE, status, metadata1, metadata2, metadata3, metadata4).
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] URL file path and name including mount point
     * @param[in] mountPoint mount point of device
     * @param[in] rate rate of cueing (10= after every second jump 10s)
     * @return != 0: error, = 0: OK
     */
    tResult StartFfwdStart(const tDeviceType deviceType,
        const tDeviceID deviceID,
        const tURL URL,
        const tMountPoint mountPoint=NULL,
        const tCueingRate rate=10,
        const speedstate_e IsPlaybackSpeed = ME_SPEEDSTATE_OFF);

    /**
     * Function transfers ffwd_stop command to PlayerEngine.
     * Send StopFfSlot to PlayerEngine via DBUS -> StopFfSlot().
     * Fake PlayerEngine response: Send answer to waiting DeviceControl state machine immediately.
     *    -> SendMessage(PLAYBACK_STATUS_RESPONSE, status, metadata1, metadata2, metadata3, metadata4).
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] URL file path and name including mount point
     * @param[in] mountPoint mount point of device
     * @return != 0: error, = 0: OK
     */
    tResult StartFfwdStop(const tDeviceType deviceType,
        const tDeviceID deviceID,
        const tURL URL,
        const tMountPoint mountPoint=NULL,
        const speedstate_e IsPlaybackSpeed = ME_SPEEDSTATE_OFF);

    /**
     * Function transfers frev_start command to PlayerEngine.
     * Send StartFrevSlot to PlayerEngine via DBUS -> StartFrevSlot(rate).
     * Fake PlayerEngine response: Send answer to waiting DeviceControl state machine immediately.
     *    -> SendMessage(PLAYBACK_STATUS_RESPONSE, status, metadata1, metadata2, metadata3, metadata4).
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] URL file path and name including mount point
     * @param[in] mountPoint mount point of device
     * @param[in] rate rate of cueing (10= after every second jump 10s)
     * @return != 0: error, = 0: OK
     */
    tResult StartFrevStart(const tDeviceType deviceType,
        const tDeviceID deviceID,
        const tURL URL,
        const tMountPoint mountPoint=NULL,
        const tCueingRate rate=10,
        const speedstate_e IsPlaybackSpeed = ME_SPEEDSTATE_OFF);

    /**
     * Function transfers frev_stop command to PlayerEngine.
     * Send StopFrevSlot to PlayerEngine via DBUS -> StopFrevSlot().
     * Fake PlayerEngine response: Send answer to waiting DeviceControl state machine immediately.
     *    -> SendMessage(PLAYBACK_STATUS_RESPONSE, status, metadata1, metadata2, metadata3, metadata4).
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] URL file path and name including mount point
     * @param[in] mountPoint mount point of device
     * @return != 0: error, = 0: OK
     */

    tResult StartFrevStop(const tDeviceType deviceType,
        const tDeviceID deviceID,
        const tURL URL,
        const tMountPoint mountPoint=NULL,
        const speedstate_e IsPlaybackSpeed = ME_SPEEDSTATE_OFF);

    /**
     * Function transfers seek_to command to PlayerEngine.
     * Send SeekToSlot to PlayerEngine via DBUS -> SeekToSlot(position, playpointFormat=Absolute).
     * Fake PlayerEngine response: Send answer to waiting DeviceControl state machine immediately.
     *    -> SendMessage(PLAYBACK_STATUS_RESPONSE, status, metadata1, metadata2, metadata3, metadata4).
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] URL file path and name including mount point
     * @param[in] mountPoint mount point of device
     * @param[in] position absolute playtime in milliseconds
     * @return != 0: error, = 0: OK
     */
    tResult StartSeekTo(const tDeviceType deviceType,
        const tDeviceID deviceID,
        const tURL URL,
        const tMountPoint mountPoint=NULL,
        const tPlaytime position=0);

    /**
     * Function sends answer to waiting state machine triggered by DBUS method return message.
     * Answer via SMF -> SendAnswer(returnValue).
     *
     * @param[in] returnValue success = true, otherwise false.
     * @return != 0: error, = 0: OK
     */
    tResult ActionStatus(const tReturnValue returnValue);

    /**
     * Function sends answer to waiting state machine triggered by new playback status.
     * Answer via SMF -> SendAnswer(status, metadata1, metadata2, metadata3, metadata4, metadataTitle, mediaType).
     *
     * @param[in] status status of current media PlayerEngine for current song
     * @param[in] metadata1 coming from PlayerEngine with meta data content
     * @param[in] metadata2 coming from PlayerEngine with meta data content
     * @param[in] metadata3 coming from PlayerEngine with meta data content
     * @param[in] metadata4 coming from PlayerEngine with meta data content
     * @return != 0: error, = 0: OK
     */
    tResult PlaybackStatus(const tPEPlaybackState status,
        const tMetadata metadata1,
        const tMetadata metadata2,
        const tMetadata metadata3,
        const tMetadata metadata4,
        const tObjectID ObjectID = OBJECT_ID_NONE);

    /**
     * Function sends playback status to PlayerManager triggered by spontaneous PlaybackStatusResponse update.
     * Validate error code -> ValidateErrorCode(metadata2).
     * Forward status to PlayerManager -> SendMessage(PLAYBACK_STATUS_RESPONSE, status, metadata1, metadata2, metadata3, metadata4, metadataTitle, mediaType).
     *
     * @param[in] status status of current media PlayerEngine for current song
     * @param[in] metadata1 coming from PlayerEngine with meta data content
     * @param[in] metadata2 coming from PlayerEngine with meta data content
     * @param[in] metadata3 coming from PlayerEngine with meta data content
     * @param[in] metadata4 coming from PlayerEngine with meta data content
     * @return != 0: error, = 0: OK
     */
    tResult ForwardPlaybackStatus(const tPEPlaybackState status,
        const tMetadata metadata1,
        const tMetadata metadata2,
        const tMetadata metadata3,
        const tMetadata metadata4,
        const tObjectID ObjectID = OBJECT_ID_NONE);
    tResult ForwardPlaybackStatusNew(const tPEHandle handle, const tPEPlaybackState playbackState,
            const me::reason_e reason, const me::speed_e speed);

    tResult ForwardNowPlayingStatus(const tPEHandle handle, const tURL URL);

    tResult ReadMetadataFromFile(const tPEHandle handle, const tURL URL);
    void TransferMetaTags(IN void *info, OUT tMediaObject &mediaObject);

    tResult PlaybackStatusNew(const tPEHandle handle, const tPEPlaybackState playbackState, const me::reason_e reason, const me::speed_e speed); //Roadmap 13010: 100%

    /**
     * Function sends ACTION_ERROR to waiting state machine.
     * Roadmap 13010
     *
     * @param[in] reason error type (REASON_ACTION_ERROR, REASON_DRM_ERROR, REASON_DEVICE_ERROR, ...)
     * @return != 0: error, = 0: OK
     */
    tResult HandleActionError(const me::reason_e reason);//Roadmap 13010: 100%

    /**
     * Function sends STREAM_ERROR to DeviceDispatcher.
     * Roadmap 13010
     *
     * @param[in] reason error type (REASON_ACTION_ERROR, REASON_DRM_ERROR, REASON_DEVICE_ERROR, ...)
     * @return != 0: error, = 0: OK
     */
    tResult ForwardStreamError(const me::reason_e reason) const; //Roadmap 13010: 100%

    /**
     * Function sends answer to waiting state machine triggered by new playtime status.
     * Answer via SMF -> SendAnswer(elapsedPlaytime, totalPlaytime).
     *
     * @param[in] elapsedPlaytime elapsed playtime of current media object
     * @param[in] totalPlaytime total playtime of current media object
     * @return != 0: error, = 0: OK
     */
    tResult PlaytimeStatus(const tPlaytime elapsedPlaytime, const tPlaytime totalPlaytime);

    /**
     * Function sends playtime status to PlayerManager triggered by spontaneous Tick_TimeElapsed update.
     * Forward status to PlayerManager -> SendMessage(TICK_TIME_ELAPSED, elapsedPlaytime, totalPlaytime).
     * In case of playtime is at the beginning of the track (=0) call StartOfObject.
     *
     * @param[in] elapsedPlaytime elapsed playtime of current media object
     * @param[in] totalPlaytime total playtime of current media object
     * @return != 0: error, = 0: OK
     */
    tResult ForwardPlaytimeStatus(const tPlaytime elapsedPlaytime, const tPlaytime totalPlaytime);

    /**
     * Function sends answer to waiting state machine triggered by new playtime status.
     * Answer via SMF -> SendAnswer(handle, timeInfo).
     *
     * @param[in] handle handle (= objectID) to map property update to an media object
     * @param[in] timeInfo struct of time position and duration of a media object
     * @return != 0: error, = 0: OK
     */
    tResult PlaytimeStatusNew(const tPEHandle handle, const tPETimeInfo timeInfo);

    /**
     * Function sends playback status to PlayerManager triggered by spontaneous PlaytimeStatus update.
     * Roadmap 13010
     * Forward status to PlayerManager -> SendMessage(PLAYTIME_STATUS, handle, timeInfo).
     *
     * @param[in] handle handle (= objectID) to map property update to an media object
     * @param[in] timeInfo struct of time position and duration of a media object
     * @return != 0: error, = 0: OK
     */
    tResult ForwardPlaytimeStatusNew(const tPEHandle handle, const tPETimeInfo timeInfo);

    /**
     * Function only traces HMI layer sync view status triggered by spontaneous viewStatusDCReply update.
     *
     * @param[in] viewStatus HMI layer sync view status (e.g. ACTIVE, STOPPED, ...)
     * @return != 0: error, = 0: OK
     */
    tResult ForwardViewStatus(const tViewStatus viewStatus){return MP_NO_ERROR;};

    /**
     * Function sends answer to own waiting state machine triggered by IPC request timeout.
     * Answer to SMF -> SendAnswer().
     *
     * @return != 0: error, = 0: OK
     */
    tResult HandleAnswerTimeout();

    /**
     * Function stores audio output device name internally.
     *
     * @param[in] audioOutputDevice device name (ALSA) of audio output
     * @return != 0: error, = 0: OK
     */
    tResult SetOutputDevice(const tAudioOutputDevice audioOutputDevice);

    /**
     * Function checks if device is still present in case of error code "file not found".
     * In case of error code of PlayerEngine (metadata2) is MEDIAPLAYER_ERROR_FILE_DOES_NOT_EXISTS
     * make read request on root directory to check if device is still present.
     * If not possible change error code to MEDIAPLAYER_ERROR_DEVICE_NOT_FOUND.
     *
     * @param[out] reason error reason
     * @param[in] errorCode error code to be checked
     * @param[in] mountPoint mount point of device
     * @return != 0: error, = 0: OK
     */
    tResult ValidateErrorCode(me::reason_e &reason, const tMetadata errorCode, const tMountPoint mountPoint);

    /**
     * Function calls umount to the file system in order to detach it from the system.
     * Roadmap 13003 SDCard
     * NOT SUPPORTED BY MTP
     *
     * By use of a local RRSM send UMOUNT to DeviceControl via RouteMessageAnswer, synchronous call.
     *
     * @param[in] deviceID deviceID to be unmounted
     * @param[in] mountPoint mountPoint of device
     * @return != 0: error, = 0: OK
     */
    tResult Umount(const tDeviceID deviceID, const tMountPoint mountPoint); //Roadmap 13003

//Indexing part

    /**
     * Function delivers answer to condition.
     * It decides if device needs an initialization when INIT_DEVICE_CONNECTION is signaled.
     * In case of MTP no initialization is necessary.
     *
     * @param[in] mountPoint mount point of device
     * @param[in] deviceID device ID
     * @return true or false
     */
    tResult IsInitRequired(const tDeviceInfoString deviceInfoString);

    /**
     * Function sends internal signal INIT_DEVICE(mountPoint).
     * @attention For MTP no implementation
     *
     * @param[in] mountPoint mount point of device
     * @param[in] deviceID device ID
     * @return != 0: error, = 0: OK
     */
    tResult SendInitInit(const tDeviceInfoString deviceInfoString);

    /**
     * Function used to response to caller in order to handle device types
     * that do not need special initialization, e.g. MTP or SD card
     * Get device properties from DBManager -> GetDeviceInfo(&deviceInfo, deviceID)
     * Answer via SMF -> SendAnswer(deviceInfo.deviceName,deviceID,connectionState=CS_CONNECTED).
     *
     * @param[in] mountPoint mount point of device
     * @param[in] deviceID device ID
     * @return != 0: error, = 0: OK
     */
    tResult NoInitAnswer(const tDeviceInfoString deviceInfoString);

    /**
     * Function checks if device is already initialized properly or malfunction has been detected.
     * @attention For MTP no implementation (return value is always false to prevent the waiting state)
     *
     * @param[in] mountPoint mount point of device
     * @return true or false
     */
    tResult IsInitDeviceConnection(const tMountPoint mountPoint);

    /**
     * Function is called if IsInitDeviceConnection delivers false.
     * @attention For MTP no implementation
     *
     * @param[in] mountPoint mount point of device
     * @return != 0: error, = 0: OK
     */
    tResult Disconnect(const tMountPoint mountPoint);

    /**
     * Function entered if authentication procedure succeeded.
     * @attention For MTP no implementation
     *
     * @param[in] mountPoint mount point of device
     * @param[in] connectionState device connection state [CS_CONNECTED or CS_HW_MALFUNCTION]
     * @return != 0: error, = 0: OK
     */
    tResult OnDeviceInitialized(const tMountPoint mountPoint, const tConnectionState connectionState);

    /**
     * Function starts timer to handle authentication malfunction
     * @attention For MTP no implementation
     *
     * @return != 0: error, = 0: OK
     */
    tResult StartTimer();

    /**
      * Function retrieves the total number of playable files
      *
      * @param[in] deviceID device ID
      * @return != 0: error, = 0: OK
      */
     tResult GetNumberOfFiles(const tDeviceID deviceID);

    /**
     * Function retrieves the fingerprint of the device content in order to check for modifications.
     * In case of MTP the hash value of the folder structure is calculated recursively.
     * The number of media files for indexing will be counted.
     * Send answer to own waiting state machine after result is available.
     *    -> SendMessage(PUT_FINGERPRINT, fingerprint, fingerprintStatus).
     *
     * @param[in] mountPoint mount point of device
     * @param[in] deviceID device ID
     * @param[in] lastFingerprint status of the last fingerprint read from db
     * @return != 0: error, = 0: OK
     */
    tResult CalcFingerprint(const tMountPoint mountPoint, const tDeviceID deviceID, const tFingerprint lastFingerprint); //Roadmap 13006 IAP2

    /**
     * Function sends answer to waiting state machine triggered by new fingerprint status (PUT_FINGERPRINT).
     * Answer via SMF -> SendAnswer(fingerprint, fingerprintStatus, numberOfMediaFiles).
     *
     * @param[in] fingerprint device identification
     * @param[in] fingerprintStatus status of the fingerprint answer [OK, NOT_AVAIL, ERROR, DELTA]
     * @param[in] numberOfFiles number of files to index
     * @param[in] deviceID device ID
     * @return != 0: error, = 0: OK
     */
    tResult OnFingerprint(const tFingerprint fingerprint,const tFingerprintStatus fingerprintStatus,const tNumberOfFiles numberOfFiles,const tDeviceID deviceID);


    /**
     * Function delivers answer to condition.
     * Check cached list of metadata records already read from MTP.
     * Return true if cache is not empty, otherwise false.
     *
     * @param[in] mountPoint mount point of device
     * @param[in] readPosition position of next media object to read
     * @param[in] deviceID device ID
     * @return true or false
     */
    tResult IsNextMetadataAvailable(const tMountPoint mountPoint, const tReadPosition readPosition, const tDeviceID deviceID);


    tResult MtpGetMetadata(const tDeviceID deviceID,const tMountPoint mountPoint);

    /**
     * Function retrieves a batch of metadata records from device.
     * Batchsize needs to be dynamically calculated depending on device performance.
     * Extraction logic is depending on ???.
     * Based on the readPosition this function is able to continue extraction continuously wihtin
     * one virtual list containg all music tracks, audiobooks, podcast and videos, etc.
     *
     * Collect batch size of meta data from MTP -> CollectData(&metadataStatus,mountPoint,readPosition)
     * Send answer to own waiting state machine after collecting meta data is finished.
     *    -> SendMessage(PUT_METADATA, metadataStatus).
     *
     * @param[in] mountPoint mount point of device
     * @param[in] readPosition position of next media object to read
     * @param[in] deviceID device ID
     * @return != 0: error, = 0: OK
     */
    tResult RetrieveMetadataFromDevice(const tMountPoint mountPoint, const tReadPosition readPosition, const tDeviceID deviceID);

    /**
     * Function sends internal signal PUT_METADATA(metadataStatus).
     * Send answer to own waiting state machine because meta data already available in cache.
     *    -> SendMessage(PUT_METADATA, metadataStatus).
     *
     * @param[in] mountPoint mount point of device
     * @param[in] readPosition position of next media object to read
     * @param[in] deviceID device ID
     * @return != 0: error, = 0: OK
     */
    tResult SendPutMetadata(const tMountPoint mountPoint, const tReadPosition readPosition, const tDeviceID deviceID);

    /**
     * Function sends internal signal PUT_METADATA(metadataStatus).
     * Send answer to own waiting state machine.
     *    -> SendMessage(PUT_METADATA, metadataStatus).
     *
     * @param[in] tMetadataStatus indexing status of metadada indexing.
     * @param[in] tMediaObjectPtr pointer to indexed mediaobject.
      * @return != 0: error, = 0: OK
     */
    tResult SendMetadata(const tMetadataStatus metadataStatus, const tMediaObjectPtr mediaObjectPtr);

    /**
     * Function sends answer to waiting state machine triggered by metadata available in cache (PUT_METADATA).
     * Get meta data from own cache -> RetrieveMetadataFromCache(&metadata).
     * Answer via SMF -> SendAnswer(metadata, metadataStatus).
     *
     * @param[in] metadataStatus status of meta data collection [SUCCESS, FINISHED, FILE_ERROR, DEVICE_ERROR]
     * @return != 0: error, = 0: OK
     */
    tResult AnswerMetadata(const tMetadataStatus metadataStatus, const tMediaObjectPtr mediaObjectPtr);

    /**
     * Function retrieves the meta data of one media object from cache
     * @attention Obsolete: MTPControl does use a scan context with virtual file lists instead of metadata cache.
     *
     * @param[out] metadata meta data of one media object
     * @return != 0: error, = 0: OK
     */
    tResult RetrieveMetadataFromCache(tMetadata &metadata);

    /**
     * Function stores indexing last mode for removed device and resets scan context.
     *
     * @param[in] mountPoint mount point of device
     * @param[in] deviceID device ID
     * @return != 0: error, = 0: OK
     */
    tResult RemoveDeviceConnection(const tMountPoint mountPoint, const tDeviceID deviceID);

    /**
       * Function gets the album art from separate file
       *
       * @param[out] albumArtObjectPtr pointer to an album art object
       * @param[in] albumArtString file name and path (URL) of image to display
       * @return != 0: error, = 0: OK
       */
    tResult GetAlbumArtFromSeparateFile( tAlbumArtObjectPtr &albumArtObjectPtr, const tAlbumArt albumArtString);

//AlbumArt part

    /**
     * Function triggers DoGetAlbumArt in a seperate working thread
     *
     * DoGetAlbumArt gets the album art from media object and stores it in a member variable.
     * Allocate memory for internal album art object.
     * Get album art from Taglib -> GetAlbumArt(&albumArtObject, albumArtString)
     * or from separate file -> fread(file) + GetAlbumArtFromSeparateFile(&albumArtObject, albumArtString).
     * Answer via SMF -> SendAnswer(ptrToAlbumArtObject).
     * @attention The caller must free this memory if not used anymore.
     *
     * @param[in] albumArtString file name and path (URL) of image to display
     * @return != 0: error, = 0: OK
     */
    tResult GetAlbumArt(const tAlbumArt albumArtString);

    /**
     * Function sends answer to waiting state machine triggered by answer from DoGetAlbumArt
     * working thread (GET_ALBUM_ART_ANSWER).
     * Answer via SMF -> SendAnswer(ptrToAlbumArtObject).
     *
     * @param[in] ptrToAlbumArtObject pointer to allocated album art object
     * @return != 0: error, = 0: OK
     */
    tResult HandleGetAlbumArtAnswer(const tAlbumArtObjectPtr ptrToAlbumArtObject);

    /**
    * Function clears allocated album art buffer
    *
    * @param[in] ptrToAlbumArtObject pointer to allocated album art object
    * @return != 0: error, = 0: OK
    */
    tResult AlbumArtAnswerNotConsumed(const tAlbumArtObjectPtr ptrToAlbumArtObject);

//MTP specific part

    /**
     * This function is used by DBManager to generate a unique mount point from some device info
     *
     * param[in] deviceInfo: holds device informations but no valid mount point
     * param[out] deviceInfo: has an updated member mount point
     * @return != 0: error, = 0: OK
     */
    tResult GenerateMountPoint(INOUT tDeviceInfo &deviceInfo);


    /**
     * IsBatchPlayable
     * Function is used to check support for batch list playback support
     *
     * @param[in] deviceID
     * @return tBoolean
     */
    tBoolean IsBatchPlayable(const tDeviceID deviceID);

    tBoolean IsInitializingDevice(const tMountPoint mountPoint,tConnectionState connectionState,tInitDeviceProtocol protocol);
    /**
     * This function is used by Test to check if InitDevice and RemoveDevice worked
     *
     * param[in]  expectedNumOfDevices: expected number of connected devices
     * param[in]  timoutIntervall: wait until check value
     * param[in]  numOfTrials    : e.g. if timoutIntervall=4s and numOfTrials=3 then wait 3 times 4s and check before leaving function
     * @return num of connected devices
     */
    unsigned int TestIF_GetNumOfDevices(unsigned expectedNumOfDevices,unsigned timoutIntervall,unsigned numOfTrials);
   /* MTP VIdeo display settings to be set as part of sendvideo property */
    tResult SendVideoProperty();

    //@todo add header
    eMtpWrapperType TestIF_getWrapper();
    void TestIF_setWrapper(IN eMtpWrapperType eTypeOfWrapper);
    void TestIF_setTestResultMsgQ(IN MessageQueue *pTestResultQ);
    void TestIF_sendTestResultMsg(IN const char* pMessage);
    void TestIF_sendTestResultMsgResult(IN tResult result);
    void TestIF_sendTestResultMsgMediaStatus(IN tMetadataStatus metadataStatus);
    tResult MtpGetFolderItem(const tMountPoint mountpoint, const char* path, unsigned int index);
    tResult GetLastPlayedPath(IN const tDeviceID deviceID, OUT tPath lastPlayedPath);
    tResult ReadFolder(uint32_t storageID, uint32_t ReqFolderID,const tMountPoint mountPoint,tAvailable isFolderview);
    tFiles* CopyFileElement(uint32_t parentID, uint32_t index,tDeviceID deviceId);
    tFiles* CopyStorageElement(const tMountPoint mountpoint, uint32_t index);
    /**
    * Function sends NO_RESPONSE_MSG to waiting state machine.
    * CMG3G-8460
    * @param[in] response false for NOK true for OK
    * @return != 0: error, = 0: OK
    */
    tResult SendNoResponseMsg(const tResponseMsg response);
    /**
     * This method  getStreamingInfo used to send streaming info
     * @param[in]
     * @param[OUT] device : Device type ,
     * @return < 0: error, = 0: OK
     */
    tResult getStreamingInfo(OUT tResponseMsg& info);

private:
    /**
     * Function reads the metadata of an video media object using the TagInfo lib
     *
     * @param[in] mediaObject to update/fill an media object after extraction
     * @param[in] completeFileName file path and name including mount point
     * @return true or false
     */
    tReturnValue ReadVideoMetadataFromFile(tMediaObject &mediaObject, const tURL completeFileName);

    /**
     * Function sends ACTION_ERROR to its own state machine
     * Roadmap 13010
     *
     * @param[in] reason error type (REASON_ACTION_ERROR, REASON_DRM_ERROR, REASON_DEVICE_ERROR, ...)
     * @return != 0: error, = 0: OK
     */
    tResult ActionError(const me::reason_e reason);

public:

    MTPLibWrapper *GetpMTPLibWrapper();

    typedef enum
    {
        MTPLIB_DEBUGINFO_NONE      = 0x00,
        MTPLIB_DEBUGINFO_ALL       = 0x01
    }MTPLIB_DebugInfo;

    //void setDebugLevelLibMtp(MTPLIB_DebugInfo enTypeOfDebugInfo); ///@todo later
    LIBMTP_mtpdevice_t  *devTblFind(IN const tMountPoint mountPoint, OUT int &index = m_indexDefault);


private:
    Lock m_Mutex;                                       /**< lock access to member variables */
    tResponseMsg m_response;                            /*<save previous response>*/
protected:
    //->used for test
    MessageQueue *m_pTestResultMsgQ;
    //<-
    tBoolean m_DeviceRemoved;
    tResult SendAlbumArtAnswer(const tAlbumArtObjectPtr ptrToAlbumArtObject);

    void DoGetAlbumArtThread(const tAlbumArt albumArtString);
    void DoCalcFingerprintThread(IN const char *parameterString);
    void DoReadFilesFromDeviceThread(IN const char *parameterString);

    /*--------------------------variables----------------------------*/


    typedef struct DevTable
    {
        int                  iNextElemToScan;
        tMountPoint          mountPoint;
        tDeviceID            deviceID;
        LIBMTP_mtpdevice_t   *pDevice;
    }DevTable;

    vector<DevTable> m_devTable;

    char                m_PlayFileTmp[1024];

    LIBMTP_mtpdevice_t *m_device;

    tMediaObject        m_PlayMediaObject;              /**< last media object to play */



    MTPLibWrapper      *m_pMTPLibWrapper;
    eMtpWrapperType     m_eTypeOfWrapper;

    tFolderReadStatus   mReadStatus;

    tAudioOutputDevice  m_AudioOutputDevice;            /**< device name (ALSA) of audio output for MediaEngine */

    vector<string>      m_RegisteredFilterPatterns;

    tPath               m_LastPlayedPath;
    tPath               m_LastStoredPath;



    /*--------------------------functions----------------------------*/
    tResult encodeMountPoint(INOUT tDeviceInfo &deviceInfo);
    tResult decodeMountPoint(IN const tMountPoint mountPoint, OUT tDeviceInfo &deviceInfo);

    /**
        * Function collects next metadata from MTP
        * In case of error code FILE_ERROR make read request on root directory to check if device is still present.
        * If not possible change error code to DEVICE_ERROR.
        *
        * @param[out] metadataStatus status of meta data collection [SUCCESS, FINISHED, FILE_ERROR, DEVICE_ERROR]
        * @param[in] deviceID device ID
        * @param[in] mountPoint mount point of device
        * @param[in] readPosition position of next media object to read
        * @return != 0: error, = 0: OK
        */
    tResult CollectData(tMetadataStatus &metadataStatus, const tDeviceID deviceID, const tMountPoint mountPoint, const tReadPosition readPosition);


    /**
     * This function is used by MTPcontrol to find the device linked to a mount point
     *
     * param[in]  mountPoint: mount point of device
     * param[out] device: device which maps to that mount point
     * @return != 0: error, = 0: OK
     */

    LIBMTP_mtpdevice_t  *devTblFind(IN const tDeviceID deviceID,     OUT int &index = m_indexDefault);

    ///@todo add header
    tResult devTbl_SetNextElemForIndexing(IN  uint32_t  iNoElem, IN int iIndexOfDevice);
    tResult devTbl_GetNextElemForIndexing(OUT uint32_t &iNoElem, IN int iIndexOfDevice);

    /**
     * This function is used by MTPcontrol to add the device and its mount point to a table.
     * If the device is connected this function will be called
     *
     * param[in]  mountPoint: mount point of device
     * param[in] device: device which maps to that mount point
     * @return != 0: error, = 0: OK
     */
    tResult devTbl_Add(IN const tMountPoint mountPoint,IN LIBMTP_mtpdevice_t  *pDevice, IN tDeviceID deviceID);


    /**
       * This function is used by MTPcontrol to delete  the device and its mount point from a table
       * if the device is removed this function will be called
       *
       * param[in]  mountPoint: mount point of device
       * param[in] device: device which maps to that mount point
       * @return != 0: error, = 0: OK
       */

    tResult devTbl_Del(IN const tMountPoint mountPoint,IN const tDeviceID deviceID);
    tResult devTbl_Del(IN const tMountPoint mountPoint);

    /**
     * This function is used by MTPcontrol to delete all devices and its mount points from the table
     *
     *
     * param[in]  -
     * @return != 0: error, = 0: OK
     */
    tResult devTblDeleteAll();

    /**
      *
      *
      *
      * param[in]  -
      * @return != 0: error, = 0: OK
     */
    bool devTblIsElemEmpty(DevTable elem);

    /**
     * This function is used by MTPcontrol to show track information - trace
     *
     *
     * param[in]  -
     * @return != 0: error, = 0: OK
     */
    void dbgShowTrack(LIBMTP_track_t *track); ///@todo add INs and OUTs


    /**
    *
    *
    *
    * param[in]  -
    * @return != 0: error, = 0: OK
    */
    tUInt comparePreviousID(IN const char *cPath, IN const char *cfileName, IN tDeviceID deviceID, IN const tURL URL);

    /**
    *
    *
    *
    * param[in]  -
    * @return != 0: error, = 0: OK
    */
    tResult writeID(IN const char *cPath,IN const char *cfileName, IN const tDeviceID deviceID, IN const tURL URL);

    /**
    *
    *
    *
    * param[in]  -
    * @return != 0: error, = 0: OK
    */
    tResult writeAppendInfo(IN const char*cPath,IN const char*cfileName,
                            IN LIBMTP_filetype_t filetype,
                            IN const tMountPoint mountPoint,
                            IN const tMetadata title,
                            IN const tMetadata album,
                            IN const tMetadata artist,
                            IN const tMetadata genre);



    /**
    *
    *
    *
    * param[in]  -
    * @return != 0: error, = 0: OK
    */
    tResult deleteTmpFile(IN const char *cPath, IN const char *cfileName);

    /**
    *
    *
    *
    * param[in]  -
    * @return != 0: error, = 0: OK
    */
    tResult deleteTmpFile(IN char *fullName);


    /*****************************************LibMTP-Wrapper********************************************/

    /**
      *
      *
      *
      * param[in]  -
      * @return != 0: error, = 0: OK
    */
    tResult fillMediaObject(OUT tMediaObjectPtr mediaObjectPtr, IN LIBMTP_track_t *pTrack, IN const tMountPoint mountPoint,IN tDeviceID deviceID,IN tURL fileName);


    /**
      *
      *
      *
      * param[in]  -
      * @return != 0: error, = 0: OK
    */
    tResult  GetFileFormat(tFileFormat &fileFormat,const tAlbumArt albumArtString);   ///@todo think about shifting this to extra class since USBCtrl does use this too

    bool checkRealDevicesUsed(); ///@todo add header
    virtual void factoryGetLibMTPWrapper();

    void LoadFilterConfiguration();

    bool FolderFilterEntry(const char* entryName);

    void sendPlaybackstatus(tPEPlaybackState status);

    tResult errChkAvailable(const tMountPoint mountPoint);

    tResult  readRecursiveFolder(tURL URL,const tDeviceID deviceID,const tMountPoint mountPoint);


};

#endif

/** @} */
