/**
 * @addtogroup DeviceControl
 * @author Thomas Porsch
 *
 * Public interface for device dispatcher
 * @{
 */

#ifndef DEVICEDISPATCHER_H_
#define DEVICEDISPATCHER_H_

#include "DeviceDispatcherSM.h"
#include "ILocalSPM.h"
#include "ThreadFactory.h"

class DeviceDispatcher : public DeviceDispatcherSM , public ILocalSPM, public TFThread
{

public:

//SPM part

    /**
     * Constructor of component.
     * This function is used by LocalSPM to store the DeviceDispatcher componentID in mComponentID.
     *
     * @param[in] componentID component ID assigned by SPM
     * @return void
     */
    DeviceDispatcher(const tComponentID componentID):ILocalSPM(componentID){};

    /**
     * Destructor of component.
     *
     * @return void
     */
    virtual ~DeviceDispatcher(){};

    /**
     * This function is used by LocalSPM to create the DeviceDispatcher.
     * @attention: running in SPM thread context
     * Create the DeviceDispatcher state machine (including create of message queue).
     * Inform LocalSPM that Create is ready -> CreateDone(0)
     *
     * @return void
     */
    void Create();

    /**
     * This function is used by LocalSPM to trigger the DeviceDispatcher initialization.
     * @attention: running in SPM thread context
     * Init the DeviceDispatcher state machine.
     * Register DeviceDispatcherSM with dispatcher.
     * Inform LocalSPM that Init is ready -> InitDone(0)
     *
     * @return != 0: error, = 0: OK
     */
    tResult Init(tInitReason reason);

    /**
     * Initialize own member variables
     *
     * @return != 0: error, = 0: OK
     */
    tResult InitSM();

    /**
     * This function is used by LocalSPM to start the DeviceDispatcher.
     * @attention: running in SPM thread context
     * From now on all other mediaplayer components are available.
     * Start the DeviceDispatcher thread and the state machine.
     *
     * @return != 0: error, = 0: OK
     */
    tResult Run();

    /**
     * This function starts a user function in a thread from the thread factory's thread pool.
     * After leaving while loop used thread is stopped and released for other users
     *
     * @return void
     */
    void Do(int functionID, void *ptr);

    /**
     * This function is used by LocalSPM to stop the DeviceDispatcher.
     * @attention: running in SPM thread context
     * Store last mode values.
     * Deregister on DB trigger.
     * Set state machine to final state -> SendMessage(STOP_SM).
     * LocalSPM will be informed after STOP_SM event is processed
     *
     * @return != 0: error, = 0: OK
     */
    tResult Stop();

    /**
     * Inform LocalSPM that Stop is ready -> StopDone(0)
     *
     * @return != 0: error, = 0: OK
     */
    tResult StopEventProcessed();

    /**
     * This function is used by LocalSPM to cleanup the DeviceDispatcher.
     * @attention: running in SPM thread context
     * Deregister DeviceDispatcherSM with dispatcher.
     * Set state machine to final state -> SendMessage(DONE).
     * LocalSPM will be informed after DONE event is processed.
     *
     * @return != 0: error, = 0: OK
     */
    tResult Done();

    /**
     * Inform LocalSPM that Done is ready -> DoneDone(0)
     *
     * @return != 0: error, = 0: OK
     */
    tResult DoneEventProcessed();

    /**
     * Returns the current state the state machine is in (for debugging of shutdown problems)
     * @param[inout] stateName buffer for storing the current state name
     *
     * @return pointer to stateName
     */
    char *GetSMStateName(OUT tGeneralString stateName, IN size_t size);

    /**
    * Returns answer if the component is a state machine
    *
    * @return true or false
    */
    tBoolean IsComponentSM() { return true; }; //component is a state machine

//Playback control part

    /**
     * Function transfers assign_audio_output_device command to PlayerEngine.
     * if old PlayerEngine
     * - Send AssignAudioOutputDevice to PlayerEngine via DBUS -> AssignAudioOutputDevice(audioOutputDevice).
     * - PlayerEngine command has return value of success.
     * else //Roadmap 13010
     * - Send Event to each device control -> SendMessage(SET_OUTPUT_DEVICE, audioOutputDevice).
     * - Send answer via SMF immediately -> SendAnswer(returnValue = true).
     *
     * @param[in] audioOutputDevice device name (ALSA) of audio output
     * @return != 0: error, = 0: OK
     */
    tResult StartAllocateAudioOutput(const tAudioOutputDevice audioOutputDevice); //Roadmap 13010: 100%

    /**
     * Function transfers deallocate_out_device command to PlayerEngine.
     * Send DeAllocateOutDevice to PlayerEngine via DBUS.
     * PlayerEngine command has return value of success.
     *
     * @return != 0: error, = 0: OK
     */
    tResult StartDeAllocateAudioOutput();

    /**
     * Intermediate state to link 2 subsequent conditional states.
     * Send parametrless DEVICE_TYPE_ANSWER to itself immediately.
     *
     * @return != 0: error, = 0: OK
     */
   tResult StartIsDeviceTypeDifferent();

    /**
     * Function delivers answer to condition.
     * If currentDeviceType != newDeviceType return true, otherwise false.
     *
     * @return true or false
     */
    tResult IsDeviceTypeDifferent();

    /**
     * Function transfers stop command to device controls.
     * SendMessageAnswer(STOP) to current device control (e.g. USBControl).
     *    -> "STOP m_DeviceType,m_DeviceID,m_URL,m_MountPoint","DEVICE_DISPATCHER STOP_ANSWER".
     *
     * @return != 0: error, = 0: OK
     */
    tResult StartStopViaContext();

    /**
     * Function transfers spause command to device controls.
     * SendMessageAnswer(PAUSE) to current device control (e.g. USBControl).
     *    -> "PAUSE m_DeviceType,m_DeviceID,m_URL,m_MountPoint","DEVICE_DISPATCHER PAUSE_ANSWER".
     *
     * @return != 0: error, = 0: OK
     */
    tResult StartPauseViaContext();

    /**
     * Function transfers assign_audio_input_device command to device controls.
     * SendMessageAnswer(ALLOCATE_AUDIO_INPUT) to new device control (e.g. IPODControl).
     *    -> "ALLOCATE_AUDIO_INPUT m_MountPoint","DEVICE_DISPATCHER ALLOCATE_AUDIO_INPUT_ANSWER".
     *
     * @return != 0: error, = 0: OK
     */
    tResult StartAllocateAudioInput();

    /**
     * Function transfers switch_observer command to device controls.
     * SendMessageAnswer(SWITCH_OBSERVER) to new device control (e.g. IPODControl).
     *    -> "SWITCH_OBSERVER","DEVICE_DISPATCHER SWITCH_OBSERVER_ANSWER".
     *
     * @return != 0: error, = 0: OK
     */
    tResult StartSwitchObserver();

    /**
     * Function transfers play command to device controls.
     * SendMessageAnswer(PLAY) to new device control (e.g. IPODControl).
     *    -> "PLAY m_DeviceType,m_DeviceID,m_URL,m_MountPoint,m_Position","DEVICE_DISPATCHER PLAY_ANSWER".
     *
     * @return != 0: error, = 0: OK
     */
    tResult StartPlay();

    /**
     * Function transfers start_streaming command to device controls.
     * SendMessageAnswer(START_STREAMING) to new device control (e.g. IPODControl).
     *    -> "START_STREAMING m_MountPoint","DEVICE_DISPATCHER START_STREAMING_ANSWER".
     *
     * @return != 0: error, = 0: OK
     */
    tResult StartStreaming();

    /**
     * Function transfers mute command to audio management via OutputWrapper.
     * Roadmap 13018
     * Set wait time for mute to 3s -> SetAnswerTimeout(3000L)
     * Request mute at AudioManagement via OutputWrapper -> RequestMuteState(MUTED).
     *
     * @return != 0: error, = 0: OK
     */
    tResult StartMute(); //Roadmap 13018: 100%

    /**
     * Function transfers demute command to audio management via OutputWrapper.
     * Roadmap 13018
     * Request demute at AudioManagement via OutputWrapper -> RequestMuteState(DEMUTED).
     *
     * @return != 0: error, = 0: OK
     */
    tResult StartDemute();

    /**
     * Function transfers stop command to device controls.
     * SendMessageAnswer(STOP) to device control (e.g. IPODControl).
     *    -> "STOP deviceType,deviceID,URL,mountPoint","DEVICE_DISPATCHER PLAYBACK_STATUS_RESPONSE".
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] URL file path and name including mount point
     * @param[in] mountPoint mount point of device
     * @return != 0: error, = 0: OK
     */
    tResult StartStop(const tDeviceType deviceType,
        const tDeviceID deviceID,
        const tURL URL,
        const tMountPoint mountPoint);

    /**
     * Function transfers pause command to device controls.
     * SendMessageAnswer(PAUSE) to device control (e.g. IPODControl).
     *    -> "PAUSE deviceType,deviceID,URL,mountPoint","DEVICE_DISPATCHER PLAYBACK_STATUS_RESPONSE".
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] URL file path and name including mount point
     * @param[in] mountPoint mount point of device
     * @return != 0: error, = 0: OK
     */
    tResult StartPause(const tDeviceType deviceType,
        const tDeviceID deviceID,
        const tURL URL,
        const tMountPoint mountPoint);
    /**
     * Function transfers pause command to device controls.
     * SendMessageAnswer(PAUSE) to device control (e.g. IPODControl). Parameters to be sent are fetched from stored context
     *    -> "PAUSE deviceType,deviceID,URL,mountPoint","DEVICE_DISPATCHER PLAYBACK_STATUS_RESPONSE".
     * @return != 0: error, = 0: OK
     */
    tResult StartPause();

    /**
     * Function transfers resume command to device controls.
     * SendMessageAnswer(RESUME) to device control (e.g. IPODControl).
     *    -> "RESUME deviceType,deviceID,URL,mountPoint","DEVICE_DISPATCHER PLAYBACK_STATUS_RESPONSE".
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] URL file path and name including mount point
     * @param[in] mountPoint mount point of device
     * @return != 0: error, = 0: OK
     */
    tResult StartResume(const tDeviceType deviceType,
        const tDeviceID deviceID,
        const tURL URL,
        const tMountPoint mountPoint);

    /**
     * Function transfers fast forward start command to device controls.
     * SendMessageAnswer(FFWD) to device control (e.g. IPODControl).
     *    -> "FFWD deviceType,deviceID,URL,mountPoint,rate","DEVICE_DISPATCHER PLAYBACK_STATUS_RESPONSE".
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] URL file path and name including mount point
     * @param[in] mountPoint mount point of device
     * @param[in] rate rate of cueing (10= after every second jump 10s)
     * @return != 0: error, = 0: OK
     */
    tResult StartFfwdStart(const tDeviceType deviceType,
        const tDeviceID deviceID,
        const tURL URL,
        const tMountPoint mountPoint,
        const tCueingRate rate=10,
        const speedstate_e IsPlaybackSpeed= ME_SPEEDSTATE_OFF);

    /**
     * Function transfers fast forward stop command to device controls.
     * SendMessageAnswer(FFWD_STOP) to device control (e.g. IPODControl).
     *    -> "FFWD_STOP deviceType,deviceID,URL,mountPoint","DEVICE_DISPATCHER PLAYBACK_STATUS_RESPONSE".
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] URL file path and name including mount point
     * @param[in] mountPoint mount point of device
     * @return != 0: error, = 0: OK
     */
    tResult StartFfwdStop(const tDeviceType deviceType,
        const tDeviceID deviceID,
        const tURL URL,
        const tMountPoint mountPoint,
        const speedstate_e IsPlaybackSpeed = ME_SPEEDSTATE_OFF);

    /**
     * Function transfers fast reverse start command to device controls.
     * SendMessageAnswer(FREV) to device control (e.g. IPODControl).
     *    -> "FREV deviceType,deviceID,URL,mountPoint,rate","DEVICE_DISPATCHER PLAYBACK_STATUS_RESPONSE".
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] URL file path and name including mount point
     * @param[in] mountPoint mount point of device
     * @param[in] rate rate of cueing (10= after every second jump 10s)
     * @return != 0: error, = 0: OK
     */
    tResult StartFrevStart(const tDeviceType deviceType,
        const tDeviceID deviceID,
        const tURL URL,
        const tMountPoint mountPoint,
        const tCueingRate rate=10,
        const speedstate_e IsPlaybackSpeed= ME_SPEEDSTATE_OFF);

    /**
     * Function transfers fast reverse stop command to device controls.
     * SendMessageAnswer(FREV_STOP) to device control (e.g. IPODControl).
     *    -> "FREV_STOP deviceType,deviceID,URL,mountPoint","DEVICE_DISPATCHER PLAYBACK_STATUS_RESPONSE".
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] URL file path and name including mount point
     * @param[in] mountPoint mount point of device
     * @return != 0: error, = 0: OK
     */
    tResult StartFrevStop(const tDeviceType deviceType,
        const tDeviceID deviceID,
        const tURL URL,
        const tMountPoint mountPoint,
        const speedstate_e IsPlaybackSpeed= ME_SPEEDSTATE_OFF);

    /**
     * Function transfers seek_to command to device controls.
     * SendMessageAnswer(SEEK_TO) to device control (e.g. IPODControl).
     *    -> "SEEK_TO deviceType,deviceID,URL,mountPoint,position","DEVICE_DISPATCHER PLAYBACK_STATUS_RESPONSE".
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] URL file path and name including mount point
     * @param[in] mountPoint mount point of device
     * @param[in] position absolute playtime in milliseconds
     * @return != 0: error, = 0: OK
     */
    tResult StartSeekTo(const tDeviceType deviceType,
        const tDeviceID deviceID,
        const tURL URL,
        const tMountPoint mountPoint,
        const tPlaytime position=0);

    /**
     * Function transfers buffer command to device controls.
     * SendMessageAnswer(BUFFER) to device control (e.g. USBControl).
     *    -> "BUFFER deviceType,deviceID,URL,mountPoint,handle","DEVICE_DISPATCHER BUFFER_ANSWER".
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] URL file path and name including mount point
     * @param[in] mountPoint mount point of device
     * @param[in] handle handle (= objectID) to map property update to an media object
     * @return != 0: error, = 0: OK
     */
    tResult StartBuffer(const tDeviceType deviceType,
        const tDeviceID deviceID,
        const tURL URL,
        const tMountPoint mountPoint,
        const tPEHandle handle);

    /**
     * Function sends answer to waiting state machine triggered by DBUS method return message.
     * Answer to SMF -> SendAnswer(returnValue).
     *
     * @param[in] returnValue success = true, otherwise false.
     * @return != 0: error, = 0: OK
     */
    tResult ActionStatus(const tReturnValue returnValue);

    /**
     * Function sends answer to waiting state machine triggered by new playback status.
     * Calls PlaybackStatusPlay with status and empty other parameters.
     *
     * @param[in] status status of current media PlayerEngine for current song
     * @return != 0: error, = 0: OK
     */
    tResult PlaybackStatus(const tPEPlaybackState status);

    /**
     * Function sends answer to waiting state machine triggered by new playback status.
     * Request demute at AudioManagement via OutputWrapper -> RequestMuteState(DEMUTED). //Roadmap 13018
     * Answer to SMF -> SendAnswer(status, metadata1, metadata2, metadata3, metadata4, metadataTitle, mediaType).
     *
     * @param[in] status status of current media PlayerEngine for current song
     * @param[in] metadata1 coming from PlayerEngine with meta data content
     * @param[in] metadata2 coming from PlayerEngine with meta data content
     * @param[in] metadata3 coming from PlayerEngine with meta data content
     * @param[in] metadata4 coming from PlayerEngine with meta data content
     * @param[in] metadataTitle coming from PlayerEngine with meta data content
     * @param[in] mediaType current media type sent by PlayerEngine
     * @return != 0: error, = 0: OK
     */
    tResult PlaybackStatusPlay(const tPEPlaybackState status,  //Roadmap 13018:100%
        const tMetadata metadata1,
        const tMetadata metadata2,
        const tMetadata metadata3,
        const tMetadata metadata4,
        const tTitle metadataTitle,
        const tMediaType mediaType,
        const tConvertFlag metadataConvertFlag = 0,
        const tUUID uuid=NULL);

    /**
     * Function sends answer to waiting state machine triggered by new playback status.
     * Roadmap 13010
     * Request demute at AudioManagement via OutputWrapper -> RequestMuteState(DEMUTED). //Roadmap 13018
     * Answer via SMF -> SendAnswer(handle, playbackState, reason, speed).
     *
     * @param[in] handle handle (= objectID) to map property update to an media object
     * @param[in] playbackState playback status of current media PlayerEngine for current song
     * @param[in] reason reason for status change
     * @param[in] speed playback speed (1=normal play, 10=ffwd, -10=frev)
     * @return != 0: error, = 0: OK
     */
    tResult PlaybackStatusNew(const tPEHandle handle, //Roadmap 13010: 100%, 13018:100%
        const tPEPlaybackState playbackState,
        const me::reason_e reason,
        const me::speed_e speed);

#if 0
    /**
     * Function sends answer to waiting state machine triggered by new playtime status.
     * Answer via SMF -> SendAnswer(elapsedPlaytime, totalPlaytime).
     *
     * @param[in] elapsedPlaytime elapsed playtime of current media object
     * @param[in] totalPlaytime total playtime of current media object
     * @return != 0: error, = 0: OK
     */
    tResult PlaytimeStatus(const tPlaytime elapsedPlaytime, const tPlaytime totalPlaytime);
#endif

    /**
     * Function sends answer to waiting state machine triggered by new playtime status.
     * Answer via SMF -> SendAnswer(handle, timeInfo).
     *
     * @param[in] handle handle (= objectID) to map property update to an media object
     * @param[in] deviceID device ID of updating device
     * @param[in] timeInfo struct of time position and duration of a media object
     * @return != 0: error, = 0: OK
     */
    tResult PlaytimeStatusNew(const tPEHandle handle, const tDeviceID deviceID, const tPETimeInfo timeInfo);

    /**
     * Function sends answer to waiting state machine triggered by new buffer status.
     * Roadmap 13010
     * Answer via SMF -> SendAnswer(success).
     *
     * @param[in] success success of buffer request
     * @return != 0: error, = 0: OK
     */
    tResult BufferStatus(const tSuccess success); //Roadmap 13010: 100%

    /**
     * Function handles the mute status from the audio management.
     * Roadmap 13018
     * Sends the internal event MUTE_ANSWER if status is MUTED.
     *
     * @param[in] muteState mute state (DEMUTED, MUTED, DEMUTE_RUNNING)
     * @return != 0: error, = 0: OK
     */
    tResult MuteStatus(const tMuteState muteState); //Roadmap 13018: 100%

    /**
     * Function sends MUTE_STATUS event to own state machine.
     * Roadmap 13018
     * Call SendMessage(MUTE_STATUS, muteState).
     *
     * @param[in] muteState mute state (DEMUTED, MUTED, DEMUTE_RUNNING)
     * @return != 0: error, = 0: OK
     */
    tResult SendMuteStatus(const tMuteState muteState); //Roadmap 13018: 100%

    /**
     * Function sends answer to own waiting state machine triggered by SendMessageAnswer request timeout.
     * Release waiting in SMF -> ReleaseWaiting().
     *
     * @return != 0: error, = 0: OK
     */
    tResult HandleAnswerTimeout();

    /**
     * Function sends answer to possible waiting state machine to release own SM in case of message answer pair.
     * Answer to SMF -> SendAnswer().
     *
     * @return != 0: error, = 0: OK
     */
    tResult MessageNotConsumed();

    /**
     * Function handles a stream error situation.
     * Clear release event and disarm timer of own state machine in case of still waiting -> ClearReleaseEvent().
     * Reset context parameters -> ResetContext()
     * Forward action error to PlayerManager -> SendMessage(ACTION_ERROR, reason).
     *
     * @param[in] reason reason for error
     * @return != 0: error, = 0: OK
     */
    tResult HandleStreamError(const me::reason_e reason);

    /**
     * Function handles an error after triggering an action.
     * Roadmap 13010
     * Clear release event and disarm timer of own state machine in case of still waiting -> ClearReleaseEvent().
     * Reset context parameters -> ResetContext()
     * Forward action error to PlayerManager -> SendMessage(ACTION_ERROR, reason).
     *
     * @param[in] reason reason for error
     * @return != 0: error, = 0: OK
     */
    tResult HandleActionError(const me::reason_e reason); //Roadmap 13010: 100%

    /**
     * Function delivers answer to condition.
     * Reset answer timeout to default -> SetAnswerTimeout() //Roadmap 13018
     * If currentDeviceID != newDeviceID return true, otherwise false.
     *
     * @return true or false
     */
    tResult IsDeviceDifferent(); //Roadmap 13018:100%

    /**
     * Function delivers answer to condition.
     * If allocation was successful return true, otherwise false.
     *
     * @return true or false
     */
    tResult IsAllocateSuccessful(const tReturnValue returnValue);

    /**
     * Function delivers answer to condition.
     * If deallocation was successful return true, otherwise false.
     * In case of success reset context parameters -> ResetContext()
     *
     * @return true or false
     */
    tResult IsDeAllocateSuccessful(const tReturnValue returnValue);

    /**
     * Function routes message to correct device control.
     * Get device control name by means of deviceType -> GetDeviceControlName(deviceControlName,deviceType)
     * Call SendMessageAnswer(deviceControlName msg,msgAnswer)
     *  -> e.g. SendMessageAnswer("USB_CONTROL PAUSE deviceType,deviceID,URL,mountPoint","PLAYER_MANAGER PAUSE_ANSWER")
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] message message to send like "PAUSE"
     * @param[in] parameter parameter string to send like "deviceType,deviceID,URL,mountPoint"
     * @param[in] answer answer message like "PLAYER_MANAGER PAUSE_ANSWER"
     * @return != 0: error, = 0: OK
     */
    tResult RouteMessageAnswer(const tDeviceType deviceType, const char *message, const char *parameter, const char *answer) const;

    /**
     * Function routes message to correct device control.
     * Get device control name by means of deviceType -> GetDeviceControlName(deviceControlName,deviceType)
     * Call SendMessage(deviceControlName msg)
     *  -> e.g. SendMessage("USB_CONTROL INIT_DEVICE_CONNECTION mountPoint,deviceID")
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] message message to send like "INIT_DEVICE_CONNECTION"
     * @param[in] parameter parameter string to send like "mountPoint,deviceID"
     * @return != 0: error, = 0: OK
     */
    tResult RouteMessage(const tDeviceType deviceType, const char *message, const char *parameter) const;

    /**
     * Function delivers device control name from deviceType.
     *
     * @param[out] name name of a device control (e.g. USB_CONTROL)
     * @param[in] deviceType type of device on which the media object is stored
     * @return != 0: error, = 0: OK
     */
    tResult GetDeviceControlName(tDeviceControlName &name, const tDeviceType deviceType) const;

    /**
     * Function stores play context parameters temporary in m_NewContext for later use.
     * Additionally the current active context is stored too as backup in m_OldContext.
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] URL file path and name including mount point
     * @param[in] mountPoint mount point of device
     * @param[in] handle handle (= objectID) to map property update to an media object //Roadmap 13010
     * @param[in] position absolute playtime in milliseconds
     * @param[in] streaming streaming flag //Roadmap 13008
     * @return != 0: error, = 0: OK
     */
    tResult StoreContext(const tDeviceType deviceType, //Roadmap 13010: 100%, 13008: 100%
        const tDeviceID deviceID,
        const tURL URL,
        const tMountPoint mountPoint,
        const tUUID uuid=0,
        const tPEHandle handle=0,
        const tPlaytime position=0,
        const tStreaming streaming=false);

    /**
     * Function resets play context parameters.
     *
     * @return != 0: error, = 0: OK
     */
    tResult ResetContext();

    /**
     * Function transfers umount command to DeviceControl.
     * Roadmap 13003 SDCard
     * Get device info from DBManager::GetDeviceInfo(&deviceInfo, deviceID).
     * By use of a local RRSM send UMOUNT to DeviceControl via RouteMessageAnswer, synchronous call.
     * DeviceControl returns errno.
     *
     * @param[out] errorCode errorCode of command umount
     * @param[in] deviceID deviceID to be unmounted
     * @return != 0: error, = 0: OK
     */
    tResult SendUmount(tErrorCode &errorCode, const tDeviceID deviceID) const; //Roadmap 13003

    /**
     * IsBatchPlayable
     * Function is used to check support for batch list playback support
     *
     * @param[in] deviceID
     * @return tBoolean
     */
    tBoolean IsBatchPlayable(const tDeviceID deviceID);

    tBoolean GetLastPlayedPath(const tDeviceID deviceID,OUT tPath lastPlayedPath);



    /**
     * CheckCarPlayResume
     * Function is used to check if CarPlay needs to get resumed after mediaplayer got back active state
     *
     * @return != 0: error, = 0: OK
     */
    tResult CheckCarPlayResume();

    /**
     * setMuteState
     * Function is used to store the mute state requested to audio
     *
     * @return void
     */
    void setMuteState(const tMuteState muteState);

    /**
     * getMuteState
     * Function is used to retrieve the stored mute state
     *
     * @return mute state
     */
    tMuteState getMuteState();

private:

    tPlayContext m_NewContext; /**< new play context */
    tPlayContext m_OldContext; /**< previous play context */
    me::reason_e m_LastError;
    tMuteState m_CurrentMuteState;
};

#endif //DEVICEDISPATCHER_H_

/** @} */
