/**
 * @addtogroup DeviceControl
 * @author Ulrich Deuper
 *
 * Public interface for Content Sharing control
 * @{
 */

#ifndef _CSCONTROL_H_
#define _CSCONTROL_H_

#include "CSControlSM.h"
#include "ILocalSPM.h"
#include "ThreadFactory.h"
#include <vector>

class CSControl : public CSControlSM , public ILocalSPM, public TFThread
{

public:

//SPM part

    /**
     * Constructor of component.
     * This function is used by LocalSPM to store the CSControl componentID in mComponentID.
     *
     * @param[in] componentID component ID assigned by SPM
     * @return void
     */
    CSControl(const tComponentID componentID):ILocalSPM(componentID){};

    /**
     * Destructor of component.
     *
     * @return void
     */
    virtual ~CSControl(){};

    /**
     * This function is used by LocalSPM to create the CSControl.
     * @attention: running in SPM thread context
     * Create the CSControl state machine (including create of message queue).
     * Inform LocalSPM that Create is ready -> CreateDone(0)
     *
     * @return void
     */
    void Create();

    /**
     * This function is used by LocalSPM to trigger the CSControl initialization.
     * @attention: running in SPM thread context
     * Init the CSControl state machine.
     * Register CSControlSM with dispatcher.
     * Inform LocalSPM that Init is ready -> InitDone(0)
     *
     * @return != 0: error, = 0: OK
     */
    tResult Init(tInitReason reason);

    /**
     * Initialize own member variables
     *
     * @return != 0: error, = 0: OK
     */
    tResult InitSM();

    /**
     * This function is used by LocalSPM to start the CSControl.
     * @attention: running in SPM thread context
     * From now on all other mediaplayer components are available.
     * Start the CSControl thread and the state machine.
     *
     * @return != 0: error, = 0: OK
     */
    tResult Run();

    /**
     * This function starts a user function in a thread from the thread factory's thread pool.
     * After leaving while loop used thread is stopped and released for other users
     *
     * @return void
     */
    void Do(int functionID, void *ptr);

    /**
     * This function is used by LocalSPM to stop the CSControl.
     * @attention: running in SPM thread context
     * Store last mode values.
     * Set state machine to final state -> SendMessage(STOP_SM).
     * LocalSPM will be informed after STOP_SM event is processed
     *
     * @return != 0: error, = 0: OK
     */
    tResult Stop();

    /**
     * Inform LocalSPM that Stop is ready -> StopDone(0)
     *
     * @return != 0: error, = 0: OK
     */
    tResult StopEventProcessed();

    /**
     * This function is used by LocalSPM to cleanup the CSControl.
     * @attention: running in SPM thread context
     * Deregister CSControlSM with dispatcher.
     * Set state machine to final state -> SendMessage(DONE).
     * LocalSPM will be informed after DONE event is processed.
     *
     * @return != 0: error, = 0: OK
     */
    tResult Done();

    /**
     * Inform LocalSPM that Done is ready -> DoneDone(0)
     *
     * @return != 0: error, = 0: OK
     */
    tResult DoneEventProcessed();

    /**
     * Returns the current state the state machine is in (for debugging of shutdown problems)
     * @param[inout] stateName buffer for storing the current state name
     *
     * @return pointer to stateName
     */
    char *GetSMStateName(tGeneralString stateName, size_t size);

    /**
    * Returns answer if the component is a state machine
    *
    * @return true or false
    */
    tBoolean IsComponentSM() { return true; }; //component is a state machine


//Playback control part

    /**
     * Function transfers assign_audio_input_device command to PlayerEngine.
     * For USB no need to assign audio input device.
     * Send event to own SM immediately -> SendEvent(METHOD_RETURN, returnValue = true).
     *
     * @param[in] deviceID device ID
     * @param[in] mountPoint mount point of device
     * @return != 0: error, = 0: OK
     */
    tResult StartAllocateAudioInput(const tDeviceID deviceID, const tMountPoint mountPoint=NULL);

    /**
     * Function transfers deallocate_in_device command to PlayerEngine.
     * For USB no need to deallocate audio input device.
     * Send event to own SM immediately -> SendEvent(METHOD_RETURN, returnValue = true).
     *
     * @warning Not used yet!
     *
     * Comment: at the moment not used!
     *
     * @return != 0: error, = 0: OK
     */
    tResult StartDeAllocateAudioInput();

    /**
     * Function transfers switch_observer command to PlayerEngine.
     * Switch observer at PlayerEngine via DBUS.
     * PlayerEngine command has return value of success.
     *
     * @param[in] deviceID device ID
     * @param[in] mountPoint mount point of device
     * @return != 0: error, = 0: OK
     */
    tResult StartSwitchObserver(const tDeviceID /*deviceID*/, const tMountPoint /*mountPoint*/);

    /**
     * Function transfers ...
     * For USB no need to start streaming.
     * if old PlayerEngine
     * - Send event to own SM immediately -> SendEvent(PLAYBACK_STATUS_RESPONSE).
     * else //Roadmap 13010
     * - Send answer via SMF immediately -> SendAnswer(handle, playbackState=PE_PBS_LOADINGSTATE, reason, speed).
     * - Release own waiting state -> SendEvent(START_STREAMING_ANSWER).
     *
     * @param[in] deviceID device ID
     * @param[in] mountPoint mount point of device
     * @return != 0: error, = 0: OK
     */
    tResult StartStreaming(const tDeviceID deviceID, const tMountPoint mountPoint=NULL); //Roadmap 13010: 100%

    /**
     * Function transfers ...
     * For USB no need to stop streaming
     * Release own waiting state -> SendEvent(STOP_STREAMING_ANSWER, deviceID=DEVICE_ID_NOT_SET).
     *
     * @return != 0: error, = 0: OK
     */
    tResult StartStopStreaming();

    /**
     * Function answers of STOP request to waiting state machine via SMF
     * if old PlayerEngine
     * - PlaybackStatus(playbackState=PE_PBS_STOPPEDSTATE, metadata, metadata, metadata, metadata) -> SendAnswer.
     * else //Roadmap 13010
     * - PlaybackStatusNew(handle, playbackState=PE_PBS_STOPPEDSTATE, reason, speed) -> SendAnswer.
     *
     * @param[in] deviceID device ID
     * @return != 0: error, = 0: OK
     */
    tResult StopStreamingAnswer(tDeviceID deviceID);

    /**
     * Function transfers play command to PlayerEngine (1st part).
     * Update media object to play via TagInfo
     * Verify genre name via DataProvider::VerifyGenreName(&genreName) //Roadmap 13014
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] URL file path and name including mount point
     * @param[in] mountPoint mount point of device
     * @param[in] handle handle (= objectID) to map property update to an media object //Roadmap 13010
     * @param[in] position absolute playtime in milliseconds
     * @param[in] streaming streaming flag //Roadmap 13008
     * @return != 0: error, = 0: OK
     */
    tResult StartPlay(const tDeviceType deviceType, //Roadmap 13010: 100%, 13014, 13008: 100%
        const tDeviceID deviceID,
        const tURL URL,
        const tMountPoint mountPoint,
        const tUUID uuid,
        const tPEHandle handle,
        const tPlaytime position=0,
        const tStreaming streaming=false);

    tResult StartPlaybackAction(const tDeviceType deviceType, const tDeviceID deviceID,
                const tURL URL, const tMountPoint mountPoint,
                const tPlaybackAction playbackAction, const tNextPrevSkipCount skipcount=0);

    /**
     * Function transfers play command to PlayerEngine (2nd part).
     * if DRM protected //Roadmap 13010
     * - Forward action error to Device Dispatcher -> SendMessage(ACTION_ERROR, reason = DRM_ERROR).
     * if old PlayerEngine
     * - Send NewPlaySlot to PlayerEngine via DBUS -> NewPlaySlot(URL, position, playpointFormat=Absolute).
     *   PlayerEngine response is sent by PlaybackStatusResponseSlot(CurrentPlaybackStatus=PLAYINGSTATE) later.
     * else //Roadmap 13010
     * - Release own waiting state -> SendEvent(PLAY_ANSWER).
     * - Call PEClient::Action(&returnValue, peState with playState=PLAY, handle, m_OutputDevice, URL, speed=1)
     * - if (returnValue == true)
     *   - Send answer via SMF immediately -> SendAnswer(handle, playbackState=PE_PBS_PLAYINGSTATE, reason, speed).
     *   - Forward status to PlayerManager -> SendMessage(NOW_PLAYING_STATUS, handle, metadata1, metadata2, metadata3, metadata4, mediaType).
     * - else
     *   - Forward action error to Device Dispatcher -> SendMessage(ACTION_ERROR, reason).
     *
     * @param[in] handle handle (= objectID) to map metadata to an media object
     * @param[in] mountPointURL file path and name including mount point
     * @return != 0: error, = 0: OK
     */
    tResult StartPlayWithMetadata(const tPEHandle handle, const tURL mountPointURL);

    /**
     * Function transfers stop command to PlayerEngine.
     * if old PlayerEngine
     * - Send StopSlot to PlayerEngine via DBUS -> StopSlot().
     *   PlayerEngine response is sent by PlaybackStatusResponseSlot(CurrentPlaybackStatus=STOPPEDSTATE) later.
     * else //Roadmap 13010
     * - Release own waiting state -> SendEvent(STOP_ANSWER).
     * - Call PEClient::Action(&returnValue, peState with playState=STOP)
     * - if (returnValue == true)
     *   - Send answer via SMF immediately -> SendAnswer(handle, playbackState=PE_PBS_STOPPEDSTATE, reason, speed).
     * - else
     *   - Forward action error to Device Dispatcher -> SendMessage(ACTION_ERROR, reason).
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] URL file path and name including mount point
     * @param[in] mountPoint mount point of device
     * @return != 0: error, = 0: OK
     */
    tResult StartStop(const tDeviceType deviceType, //Roadmap 13010: 100%
        const tDeviceID deviceID,
        const tURL URL,
        const tMountPoint mountPoint=NULL);

    /**
     * Function transfers pause command to PlayerEngine.
     * if old PlayerEngine
     * - Send PauseSlot to PlayerEngine via DBUS -> PauseSlot().
     *   PlayerEngine response is sent by PlaybackStatusResponseSlot(CurrentPlaybackStatus=PAUSEDSTATE) later, but we don't use it.
     * else //Roadmap 13010
     * - Release own waiting state -> SendEvent(PAUSE_ANSWER).
     * - Call PEClient::Action(&returnValue, peState with playState=PAUSE)
     * - if (returnValue == true)
     *   - Send answer via SMF immediately -> SendAnswer(handle, playbackState=PE_PBS_PAUSEDSTATE, reason, speed).
     * - else
     *   - Forward action error to Device Dispatcher -> SendMessage(ACTION_ERROR, reason).
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] URL file path and name including mount point
     * @param[in] mountPoint mount point of device
     * @return != 0: error, = 0: OK
     */
    tResult StartPause(const tDeviceType deviceType, //Roadmap 13010: 100%
        const tDeviceID deviceID,
        const tURL URL,
        const tMountPoint mountPoint=NULL);

    /**
     * Function transfers resume command to PlayerEngine.
     * if old PlayerEngine
     * - Send ResumePlaySlot to PlayerEngine via DBUS -> ResumePlaySlot().
     *   PlayerEngine response is sent by PlaybackStatusResponseSlot(CurrentPlaybackStatus=PLAYINGSTATE) later, but we don't use it.
     * else //Roadmap 13010
     * - Release own waiting state -> SendEvent(RESUME_ANSWER).
     * - Call PEClient::Action(&returnValue, peState with playState=PLAY, speed=1)
     * - if (returnValue == true)
     *   - Send answer via SMF immediately -> SendAnswer(handle, playbackState=PE_PBS_PLAYINGSTATE, reason, speed).
     * - else
     *   - Forward action error to Device Dispatcher -> SendMessage(ACTION_ERROR, reason).
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] URL file path and name including mount point
     * @param[in] mountPoint mount point of device
     * @return != 0: error, = 0: OK
     */
    tResult StartResume(const tDeviceType deviceType, //Roadmap 13010: 100%
        const tDeviceID deviceID,
        const tURL URL,
        const tMountPoint mountPoint=NULL);

    /**
     * Function transfers ffwd_start command to PlayerEngine.
     * if old PlayerEngine
     * - Send StartFfSlot to PlayerEngine via DBUS -> StartFfSlot(rate).
     * - Fake PlayerEngine response: Send event to own SM immediately -> SendEvent(PLAYBACK_STATUS_RESPONSE).
     * else //Roadmap 13010
     * - Release own waiting state -> SendEvent(FFWD_ANSWER).
     * - Call PEClient::Action(&returnValue, peState with playState=PLAY, speed=rate)
     * - if (returnValue == true)
     *   - Send answer via SMF immediately -> SendAnswer(handle, playbackState=PE_PBS_FASTFORWARDSTATE, reason, speed).
     * - else
     *   - Forward action error to Device Dispatcher -> SendMessage(ACTION_ERROR, reason).
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] URL file path and name including mount point
     * @param[in] mountPoint mount point of device
     * @param[in] rate rate of cueing (10= after every second jump 10s)
     * @return != 0: error, = 0: OK
     */
    tResult StartFfwdStart(const tDeviceType deviceType, //Roadmap 13010: 100%
        const tDeviceID deviceID,
        const tURL URL,
        const tMountPoint mountPoint=NULL,
        const tCueingRate rate=10,
        const speedstate_e IsPlaybackSpeed = ME_SPEEDSTATE_OFF);

    /**
     * Function transfers ffwd_stop command to PlayerEngine.
     * if old PlayerEngine
     * - Send StopFfSlot to PlayerEngine via DBUS -> StopFfSlot().
     * - Fake PlayerEngine response: Send event to own SM immediately -> SendEvent(PLAYBACK_STATUS_RESPONSE).
     * else //Roadmap 13010
     * - Release own waiting state -> SendEvent(FFWD_STOP_ANSWER).
     * - Call PEClient::Action(&returnValue, peState with playState=PLAY, speed=1)
     * - if (returnValue == true)
     *   - Send answer via SMF immediately -> SendAnswer(handle, playbackState=PE_PBS_PLAYINGSTATE, reason, speed).
     * - else
     *   - Forward action error to Device Dispatcher -> SendMessage(ACTION_ERROR, reason).
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] URL file path and name including mount point
     * @param[in] mountPoint mount point of device
     * @return != 0: error, = 0: OK
     */
    tResult StartFfwdStop(const tDeviceType deviceType, //Roadmap 13010: 100%
        const tDeviceID deviceID,
        const tURL URL,
        const tMountPoint mountPoint=NULL);

    /**
     * Function transfers frev_start command to PlayerEngine.
     * if old PlayerEngine
     * - Send StartFrevSlot to PlayerEngine via DBUS -> StartFrevSlot(rate).
     * - Fake PlayerEngine response: Send event to own SM immediately -> SendEvent(PLAYBACK_STATUS_RESPONSE).
     * else //Roadmap 13010
     * - Release own waiting state -> SendEvent(FREV_ANSWER).
     * - Call PEClient::Action(&returnValue, peState with playState=PLAY, speed=-rate)
     * - if (returnValue == true)
     *   - Send answer via SMF immediately -> SendAnswer(handle, playbackState=PE_PBS_FASTREVERSESTATE, reason, speed).
     * - else
     *   - Forward action error to Device Dispatcher -> SendMessage(ACTION_ERROR, reason).
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] URL file path and name including mount point
     * @param[in] mountPoint mount point of device
     * @param[in] rate rate of cueing (10= after every second jump 10s)
     * @return != 0: error, = 0: OK
     */
    tResult StartFrevStart(const tDeviceType deviceType, //Roadmap 13010: 100%
        const tDeviceID deviceID,
        const tURL URL,
        const tMountPoint mountPoint=NULL,
        const tCueingRate rate=10,
        const speedstate_e IsPlaybackSpeed = ME_SPEEDSTATE_OFF);

    /**
     * Function transfers frev_stop command to PlayerEngine.
     * if old PlayerEngine
     * - Send StopFrevSlot to PlayerEngine via DBUS -> StopFrevSlot().
     * - Fake PlayerEngine response: Send event to own SM immediately -> SendEvent(PLAYBACK_STATUS_RESPONSE).
     * else //Roadmap 13010
     * - Release own waiting state -> SendEvent(FREV_STOP_ANSWER).
     * - Call PEClient::Action(&returnValue, peState with playState=PLAY, speed=1)
     * - if (returnValue == true)
     *   - Send answer via SMF immediately -> SendAnswer(handle, playbackState=PE_PBS_PLAYINGSTATE, reason, speed).
     * - else
     *   - Forward action error to Device Dispatcher -> SendMessage(ACTION_ERROR, reason).
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] URL file path and name including mount point
     * @param[in] mountPoint mount point of device
     * @return != 0: error, = 0: OK
     */
    tResult StartFrevStop(const tDeviceType deviceType, //Roadmap 13010: 100%
        const tDeviceID deviceID,
        const tURL URL,
        const tMountPoint mountPoint=NULL);

    /**
     * Function transfers seek_to command to PlayerEngine.
     * if old PlayerEngine
     * - Send SeekToSlot to PlayerEngine via DBUS -> SeekToSlot(position, playpointFormat=Absolute).
     * - Fake PlayerEngine response: Send event to own SM immediately
     *    -> SendEvent(TICK_TIME_ELAPSED, elapsedPlaytime=position, totalPlaytime=PLAYTIME_NONE).
     * else //Roadmap 13010
     * - Release own waiting state -> SendEvent(SEEK_TO_ANSWER).
     * - Call PEClient::Action(&returnValue, peState with playState=PLAY, timeInfo=position)
     * - if (returnValue == true)
     *   - Send answer via SMF immediately -> SendAnswer().
     *   - Forward status to PlayerManager -> SendMessage(PLAYTIME_STATUS, handle, timeInfo).
     * - else
     *   - Forward action error to Device Dispatcher -> SendMessage(ACTION_ERROR, reason).
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] URL file path and name including mount point
     * @param[in] mountPoint mount point of device
     * @param[in] position absolute playtime in milliseconds
     * @return != 0: error, = 0: OK
     */
    tResult StartSeekTo(const tDeviceType deviceType, //Roadmap 13010: 100%
        const tDeviceID deviceID,
        const tURL URL,
        const tMountPoint mountPoint=NULL,
        const tPlaytime position=0);

    /**
     * Function transfers start_buffering command to PlayerEngine.
     * Roadmap 13010 Gapless Play
     * Call PEClient::Action(&returnValue, peState with reason=BUFFER)
     * if (returnValue == true)
     * - Send answer via SMF immediately -> SendAnswer(success = SC_YES).
     * else
     * - Send answer via SMF immediately -> SendAnswer(success = SC_NO).
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] URL file path and name including mount point
     * @param[in] mountPoint mount point of device
     * @param[in] handle handle (= objectID) to map property update to an media object
     * @return != 0: error, = 0: OK
     */
    tResult StartBuffer(const tDeviceType deviceType, //Roadmap 13010: 100%
        const tDeviceID deviceID,
        const tURL URL,
        const tMountPoint mountPoint,
        const tPEHandle handle);

    /**
     * Function transfers discard_buffer command to PlayerEngine.
     * Roadmap 13017
     * Call PEClient::Action(&returnValue, peState with reason=FLUSH)
     *
     * @return != 0: error, = 0: OK
     */
    tResult DiscardBuffer(); //Roadmap 13017: 100%

    /**
     * Function sends answer to waiting state machine triggered by DBUS method return message.
     * Answer via SMF -> SendAnswer(returnValue).
     *
     * @param[in] returnValue success = true, otherwise false.
     * @return != 0: error, = 0: OK
     */
    tResult ActionStatus(const tReturnValue returnValue);

    /**
     * Function sends answer to waiting state machine triggered by DBUS method return message.
     * Roadmap 13010
     * Answer via SMF -> SendAnswer(success).
     *
     * @param[in] success success of buffer request
     * @return != 0: error, = 0: OK
     */
    tResult BufferStatus(const tSuccess success); //Roadmap 13010: 100%

    /**
     * Function sends answer to waiting state machine triggered by new playback status.
     * Answer via SMF -> SendAnswer(status, metadata1, metadata2, metadata3, metadata4, metadataTitle, mediaType).
     *
     * @param[in] status status of current media PlayerEngine for current song
     * @param[in] metadata1 coming from PlayerEngine with meta data content
     * @param[in] metadata2 coming from PlayerEngine with meta data content
     * @param[in] metadata3 coming from PlayerEngine with meta data content
     * @param[in] metadata4 coming from PlayerEngine with meta data content
     * @return != 0: error, = 0: OK
     */
    tResult PlaybackStatus(const tPEPlaybackState status,
        const tMetadata metadata1,
        const tMetadata metadata2,
        const tMetadata metadata3,
        const tMetadata metadata4,
        const tObjectID ObjectID = OBJECT_ID_NONE);

    /**
     * Function sends playback status to PlayerManager triggered by spontaneous PlaybackStatusResponse update.
     * Validate error code -> ValidateErrorCode(metadata2).
     * Forward status to PlayerManager -> SendMessage(PLAYBACK_STATUS_RESPONSE, status, metadata1, metadata2, metadata3, metadata4, metadataTitle, mediaType).
     *
     * @param[in] status status of current media PlayerEngine for current song
     * @param[in] metadata1 coming from PlayerEngine with meta data content
     * @param[in] metadata2 coming from PlayerEngine with meta data content
     * @param[in] metadata3 coming from PlayerEngine with meta data content
     * @param[in] metadata4 coming from PlayerEngine with meta data content
     * @return != 0: error, = 0: OK
     */
    tResult ForwardPlaybackStatus(const tPEPlaybackState status,
        const tMetadata metadata1,
        const tMetadata metadata2,
        const tMetadata metadata3,
        const tMetadata metadata4,
        const tObjectID ObjectID = OBJECT_ID_NONE);

    /**
     * Function sends answer to waiting state machine triggered by new playtime status.
     * Answer via SMF -> SendAnswer(elapsedPlaytime, totalPlaytime).
     *
     * @param[in] elapsedPlaytime elapsed playtime of current media object
     * @param[in] totalPlaytime total playtime of current media object
     * @return != 0: error, = 0: OK
     */
    tResult PlaytimeStatus(const tPlaytime elapsedPlaytime, const tPlaytime totalPlaytime);

    /**
     * Function sends playtime status to PlayerManager triggered by spontaneous Tick_TimeElapsed update.
     * Forward status to PlayerManager -> SendMessage(TICK_TIME_ELAPSED, elapsedPlaytime, totalPlaytime).
     * In case of playtime is at the beginning of the track (=0) call StartOfObject.
     *
     * @param[in] elapsedPlaytime elapsed playtime of current media object
     * @param[in] totalPlaytime total playtime of current media object
     * @return != 0: error, = 0: OK
     */
    tResult ForwardPlaytimeStatus(const tPlaytime elapsedPlaytime, const tPlaytime totalPlaytime);

    /**
     * Function only traces HMI layer sync view status triggered by spontaneous viewStatusDCReply update.
     *
     * @param[in] viewStatus HMI layer sync view status (e.g. ACTIVE, STOPPED, ...)
     * @return != 0: error, = 0: OK
     */
    tResult ForwardViewStatus(const tViewStatus viewStatus){return MP_NO_ERROR;};

    /**
     * Function sends ACTION_ERROR to waiting state machine.
     * Roadmap 13010
     *
     * @param[in] reason error type (REASON_ACTION_ERROR, REASON_DRM_ERROR, REASON_DEVICE_ERROR, ...)
     * @return != 0: error, = 0: OK
     */
    tResult HandleActionError(const me::reason_e reason);//Roadmap 13010: 100%

    /**
     * Function sends ACTION_ERROR to its own state machine
     * Roadmap 13010
     *
     * @param[in] reason error type (REASON_ACTION_ERROR, REASON_DRM_ERROR, REASON_DEVICE_ERROR, ...)
     * @return != 0: error, = 0: OK
     */
    tResult ActionError(const me::reason_e reason);

    /**
     * Function sends STREAM_ERROR to DeviceDispatcher.
     * Roadmap 13010
     *
     * @param[in] reason error type (REASON_ACTION_ERROR, REASON_DRM_ERROR, REASON_DEVICE_ERROR, ...)
     * @return != 0: error, = 0: OK
     */
    tResult ForwardStreamError(const me::reason_e reason) const; //Roadmap 13010: 100%

    /**
     * Function sends answer to waiting state machine triggered by new playback status.
     * Roadmap 13010
     * Answer via SMF -> SendAnswer(handle, playbackState, reason, speed).
     *
     * @param[in] handle handle (= objectID) to map property update to an media object
     * @param[in] playbackState playback status of current media PlayerEngine for current song
     * @param[in] reason reason for status change
     * @param[in] speed playback speed (1=normal play, 10=ffwd, -10=frev)
     * @return != 0: error, = 0: OK
     */
    tResult PlaybackStatusNew(const tPEHandle handle, const tPEPlaybackState playbackState, const me::reason_e reason, const me::speed_e speed); //Roadmap 13010: 100%

    /**
     * Function sends playback status to PlayerManager triggered by spontaneous PlaybackStatus update.
     * Roadmap 13010
     * Forward status to PlayerManager -> SendMessage(PLAYBACK_STATUS, handle, playbackState, reason, speed).
     *
     * @param[in] handle handle (= objectID) to map property update to an media object
     * @param[in] playbackState playback status of current media PlayerEngine for current song
     * @param[in] reason reason for status change
     * @param[in] speed playback speed (1=normal play, 10=ffwd, -10=frev)
     * @return != 0: error, = 0: OK
     */
    tResult ForwardPlaybackStatusNew(const tPEHandle handle, const tPEPlaybackState playbackState, const me::reason_e reason, const me::speed_e speed); //Roadmap 13010: 100%

    /**
     * Function sends playback status to PlayerManager triggered by spontaneous NowPlayingStatus update.
     * Roadmap 13010
     * Read meta data from TagInfo
     * Forward status to PlayerManager -> SendMessage(NOW_PLAYING_STATUS, handle, metadata1, metadata2, metadata3, metadata4, mediaType).
     *
     * @param[in] handle handle (= objectID) to map property update to an media object
     * @param[in] mountPointURL file path and name including mount point
     * @return != 0: error, = 0: OK
     */
    tResult ForwardNowPlayingStatus(const tPEHandle handle, const tURL mountPointURL); //Roadmap 13010: 100%

    /**
     * Function sends playback status to PlayerManager triggered by spontaneous NowPlayingStatus update.
     * Roadmap 13010
     * Read meta data from TagInfo
     * Forward status to PlayerManager -> SendMessage(NOW_PLAYING_STATUS, handle, metadata1, metadata2, metadata3, metadata4, mediaType).
     *
     * @param[in] handle handle (= objectID) to map property update to an media object
     * @param[in] mountPointURL file path and name including mount point
     * @return != 0: error, = 0: OK
     */
    tResult ForwardNowPlayingStatusWithMetadata(const tPEHandle handle, const tURL mountPointURL); //Roadmap 13010: 100%

    /**
     * Function sends answer to waiting state machine triggered by new playtime status.
     * Answer via SMF -> SendAnswer(handle, timeInfo).
     *
     * @param[in] handle handle (= objectID) to map property update to an media object
     * @param[in] timeInfo struct of time position and duration of a media object
     * @return != 0: error, = 0: OK
     */
    tResult PlaytimeStatusNew(const tPEHandle handle, const tPETimeInfo timeInfo);

    /**
     * Function sends playback status to PlayerManager triggered by spontaneous PlaytimeStatus update.
     * Roadmap 13010
     * Forward status to PlayerManager -> SendMessage(PLAYTIME_STATUS, handle, timeInfo).
     *
     * @param[in] handle handle (= objectID) to map property update to an media object
     * @param[in] timeInfo struct of time position and duration of a media object
     * @return != 0: error, = 0: OK
     */
    tResult ForwardPlaytimeStatusNew(const tPEHandle handle, const tPETimeInfo timeInfo); //Roadmap 13010: 100%

    /**
     * Function sends answer to own waiting state machine triggered by SendMessageAnswer request timeout.
     * Release waiting in SMF -> ReleaseWaiting().
     *
     * @return != 0: error, = 0: OK
     */
    tResult HandleAnswerTimeout();

    /**
     * Function sends answer to possible waiting state machine to release own SM in case of message answer pair.
     * Answer to SMF -> SendAnswer().
     *
     * @return != 0: error, = 0: OK
     */
    tResult MessageNotConsumed();

    /**
     * Function stores audio output device name internally.
     * Roadmap 13010
     *
     * @param[in] audioOutputDevice device name (ALSA) of audio output
     * @return != 0: error, = 0: OK
     */
    tResult SetOutputDevice(const tAudioOutputDevice audioOutputDevice); //Roadmap 13010: 100%

    /**
     * Function transfers playback mode to external device.
     * Roadmap 13008
     * For USB no need set the playback mode.
     * Restriction: Do it only in streaming mode
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] mountPoint mount point of device
     * @param[in] playbackMode current playing sequence is executed in sequential or random order
     * @return != 0: error, = 0: OK
     */
    tResult SetPlaybackMode(const tDeviceType deviceType, //Roadmap 13008
        const tDeviceID deviceID,
        const tMountPoint mountPoint,
        const tPlaybackMode playbackMode){return MP_NO_ERROR;};

    /**
     * Function transfers repeat mode to external device.
     * Roadmap 13008
     * For USB no need set the playback mode.
     * Restriction: Do it only in streaming mode
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] mountPoint mount point of device
     * @param[in] repeatMode repeat the current playing track or list or nothing
     * @return != 0: error, = 0: OK
     */
    tResult SetRepeatMode(const tDeviceType deviceType, //Roadmap 13008
        const tDeviceID deviceID,
        const tMountPoint mountPoint,
        const tRepeatMode repeatMode){return MP_NO_ERROR;};

    /**
     * Function sends internal signal PLAYBACK_STATUS(handle, status, reason, speed).
     * Roadmap 13010
     *
     * @param[in] handle handle (= objectID) to map property update to an media object
     * @param[in] playbackState status of current media PlayerEngine for current song
     * @param[in] reason reason for status change
     * @param[in] speed playback speed (1=normal play, 10=ffwd, -10=frev)
     * @return != 0: error, = 0: OK
     */
    tResult SendPlaybackStatus(const tPEHandle handle, const tPEPlaybackState playbackState, const me::reason_e reason, const me::speed_e speed); //Roadmap 13010: 100%

    /**
     * Function sends internal signal NOW_PLAYING_STATUS(handle, URL).
     * Roadmap 13010
     *
     * @param[in] handle handle (= objectID) to map property update to an media object
     * @param[in] mountPointURL file path and name including mount point
     * @return != 0: error, = 0: OK
     */
    tResult SendNowPlayingStatus(const tPEHandle handle, const tURL mountPointURL); //Roadmap 13010: 100%

    /**
     * Function sends internal signal PLAYTIME_STATUS(handle, timeInfo).
     * Roadmap 13010
     *
     * @param[in] handle handle (= objectID) to map property update to an media object
     * @param[in] timeInfoStruct struct of time position and duration of a media object
     * @return != 0: error, = 0: OK
     */
    tResult SendPlaytimeStatus(const tPEHandle handle, const tPETimeInfoStruct timeInfoStruct); //Roadmap 13010: 100%

    /**
     * Function calls umount to the file system in order to detach it from the system.
     * Roadmap 13003 SDCard
     * In case of having multiple partitions on that device umount is applied to the pysical device path instead.
     * By use of a local RRSM send UMOUNT to DeviceControl via RouteMessageAnswer, synchronous call.
     * DeviceControl returns errno back to calling SM via UMOUNT_ANSWER.
     *
     * @param[in] deviceID deviceID to be unmounted
     * @param[in] mountPoint mountPoint of device
     * @return != 0: error, = 0: OK
     */
    tResult Umount(const tDeviceID deviceID, const tMountPoint mountPoint); //Roadmap 13003: 100%

    /**
     * IsBatchPlayable
     * Function is used to check support for batch list playback support
     *
     * @param[in] deviceID
     * @return tBoolean
     */
    tBoolean IsBatchPlayable(const tDeviceID deviceID);


//Indexing part

    /**
     * Function delivers answer to condition.
     * It decides if device needs an initialization when INIT_DEVICE_CONNECTION is signaled.
     * In case of USB no initialization is necessary.
     *
     * @param[in] mountPoint mount point of device
     * @param[in] deviceID device ID
     * @return true or false
     */
    tResult IsInitRequired(const tDeviceInfoString deviceInfoString);

    /**
     * Function sends internal signal INIT_DEVICE(mountPoint).
     * @attention For USB no implementation
     *
     * @param[in] mountPoint mount point of device
     * @param[in] deviceID device ID
     * @return != 0: error, = 0: OK
     */
    tResult SendInitInit(const tDeviceInfoString deviceInfoString);

    /**
     * Function used to response to caller in order to handle device types
     * that do not need special initialization, e.g. USB or SD card
     * Get device properties from DBManager -> GetDeviceInfo(&deviceInfo, deviceID)
     * Answer via SMF -> SendAnswer(deviceInfo.deviceName,deviceID,connectionState=CS_CONNECTED).
     *
     * @param[in] mountPoint mount point of device
     * @param[in] deviceID device ID
     * @return != 0: error, = 0: OK
     */
    tResult NoInitAnswer(const tDeviceInfoString deviceInfoString);

    /**
     * Function checks if device is already initialized properly or malfunction has been detected.
     * @attention For USB no implementation (return value is always false to prevent the waiting state)
     *
     * @param[in] mountPoint mount point of device
     * @return true or false
     */
    tResult IsInitDeviceConnection(const tMountPoint mountPoint) const;

    /**
     * Function is called if IsInitDeviceConnection delivers false.
     * @attention For USB no implementation
     *
     * @param[in] mountPoint mount point of device
     * @return != 0: error, = 0: OK
     */
    tResult Disconnect(const tMountPoint mountPoint) const;

    /**
     * Function entered if authentication procedure succeeded.
     * @attention For USB no implementation
     *
     * @param[in] mountPoint mountpoint of device
     * @param[in] connectionState device connection state [CS_CONNECTED or CS_HW_MALFUNCTION]
     * @return != 0: error, = 0: OK
     */
    tResult OnDeviceInitialized(const tMountPoint mountPoint, const tConnectionState connectionState) const;

    /**
      * Function retrieves the total number of playable files
      *
      * @param[in] deviceID device ID
      * @return != 0: error, = 0: OK
      */
     tResult GetNumberOfFiles(const tDeviceID deviceID);

    /**
     * Function retrieves the fingerprint of the device content in order to check for modifications.
     * In case of USB the hash value of the folder structure is calculated recursively.
     * The number of media files for indexing will be counted.
     * Send answer to own waiting state machine after result is available.
     *    -> SendMessage(PUT_FINGERPRINT, fingerprint, fingerprintStatus).
     *
     * @param[in] mountPoint mount point of device
     * @param[in] deviceID device ID
     * @param[in] lastFingerprint status of the last fingerprint read from db
     * @return != 0: error, = 0: OK
     */
    tResult CalcFingerprint(const tMountPoint mountPoint, const tDeviceID deviceID, const tFingerprint lastFingerprint); //Roadmap 13006 IAP2

    /**
     * Function sends answer to waiting state machine triggered by new fingerprint status (PUT_FINGERPRINT).
     * Answer via SMF -> SendAnswer(fingerprint, fingerprintStatus, numberOfMediaFiles).
     *
     * @param[in] fingerprint device identification
     * @param[in] fingerprintStatus status of the fingerprint answer [OK, NOT_AVAIL, ERROR, DELTA]
     * @param[in] numberOfFiles number of files to index
     * @param[in] deviceID device ID
     * @return != 0: error, = 0: OK
     */
    tResult OnFingerprint(const tFingerprint fingerprint,
        const tFingerprintStatus fingerprintStatus,
        const tNumberOfFiles numberOfFiles,
        const tDeviceID deviceID);

    /**
     * Function delivers answer to condition.
     * Check cached list of metadata records already read from USB.
     * Return true if cache is not empty, otherwise false.
     *
     * @param[in] mountPoint mount point of device
     * @param[in] readPosition position of next media object to read
     * @param[in] deviceID device ID
     * @return true or false
     */
    tResult IsNextMetadataAvailable(const tMountPoint mountPoint, const tReadPosition readPosition, const tDeviceID deviceID);

    /**
     * Function retrieves a metadata record from device.
     * Based on the readPosition this function is able to continue extraction continuously within
     * one virtual list containing all music tracks, audiobooks, podcast and videos, etc.
     *
     * Get next media object from virtual file list -> GetNextObjectToIndex(&metadataStatus,deviceID,mountPoint,readPosition)
     * Send answer to own waiting state machine after collecting meta data is finished.
     *    -> SendMessage(PUT_METADATA, metadataStatus).
     *
     * @param[in] mountPoint mount point of device
     * @param[in] readPosition position of next media object to read
     * @param[in] deviceID device ID
     * @return != 0: error, = 0: OK
     */
    tResult RetrieveMetadataFromDevice(const tMountPoint mountPoint, const tReadPosition readPosition, const tDeviceID deviceID);

    /**
     * Function sends internal signal PUT_METADATA(metadataStatus).
     * Send answer to own waiting state machine because meta data already available in cache.
     *    -> SendMessage(PUT_METADATA, metadataStatus).
     *
     * @param[in] mountPoint mount point of device
     * @param[in] readPosition position of next media object to read
     * @param[in] deviceID device ID
     * @return != 0: error, = 0: OK
     */
    tResult SendPutMetadata(const tMountPoint mountPoint, const tReadPosition readPosition, const tDeviceID deviceID);

    /**
     * Function sends answer to waiting state machine triggered by metadata available in cache (PUT_METADATA).
     * Get meta data from own cache -> RetrieveMetadataFromCache(&metadata).
     * Answer via SMF -> SendAnswer(metadata, metadataStatus).
     *
     * @param[in] metadataStatus status of meta data collection [SUCCESS, FINISHED, FILE_ERROR, DEVICE_ERROR]
     * @param[in] mediaObjectPtr
     * @return != 0: error, = 0: OK
     */
    tResult AnswerMetadata(const tMetadataStatus metadataStatus, tMediaObjectPtr mediaObjectPtr);

    /**
     * Function retrieves the meta data of one media object from cache
     * @attention Obsolete: CSControl does use a scan context with virtual file lists instead of metadata cache.
     *
     * @param[out] metadata meta data of one media object
     * @return != 0: error, = 0: OK
     */
    tResult RetrieveMetadataFromCache(tMetadata &metadata) const;

    /**
     * Function stores indexing last mode for removed device and resets scan context.
     *
     * @param[in] mountPoint mount point of device
     * @param[in] deviceID device ID
     * @return != 0: error, = 0: OK
     */
    tResult RemoveDeviceConnection(const tMountPoint mountPoint, const tDeviceID deviceID);


//AlbumArt part

    /**
     * Function triggers DoGetAlbumArt in a seperate working thread
     *
     * DoGetAlbumArt gets the album art from media object and stores it in a member variable.
     * Allocate memory for internal album art object.
     * Get album art from Taglib -> GetAlbumArt(&albumArtObject, albumArtString)
     * or from separate file -> fread(file) + GetAlbumArtFromSeparateFile(&albumArtObject, albumArtString).
     * Answer via SMF -> SendAnswer(ptrToAlbumArtObject).
     * @attention The caller must free this memory if not used anymore.
     *
     * @param[in] albumArtString file name and path (URL) of image to display
     * @return != 0: error, = 0: OK
     */
    tResult GetAlbumArt(const tAlbumArt albumArtString);

    /**
     * Function sends answer to waiting state machine triggered by answer from DoGetAlbumArt
     * working thread (GET_ALBUM_ART_ANSWER).
     * Answer via SMF -> SendAnswer(ptrToAlbumArtObject).
     *
     * @param[in] ptrToAlbumArtObject pointer to allocated album art object
     * @return != 0: error, = 0: OK
     */
    tResult HandleGetAlbumArtAnswer(const tAlbumArtObjectPtr ptrToAlbumArtObject);

    /**
     * Function clears allocated album art buffer
     *
     * @param[in] ptrToAlbumArtObject pointer to allocated album art object
     * @return != 0: error, = 0: OK
     */
    tResult AlbumArtAnswerNotConsumed(const tAlbumArtObjectPtr ptrToAlbumArtObject);


//CS specific part
private:

    tResult SendAlbumArtAnswer(const tAlbumArtObjectPtr ptrToAlbumArtObject);

    /*internal thread functions*/
    void DoGetAlbumArtThread(const tAlbumArt albumArtString);
    void DoCalcFingerprintThread(const char *inputString);
    void DoReadMetadataForPlaybackThread(const tURL completeFileName);

    /**
     * Function checks if device is still present in case of error code "file not found".
     * In case of error code of PlayerEngine (metadata2) is MEDIAPLAYER_ERROR_FILE_DOES_NOT_EXISTS
     * make read request on root directory to check if device is still present.
     * If not possible change error code to MEDIAPLAYER_ERROR_DEVICE_NOT_FOUND.
     *
     * @param[out] reason error reason
     * @param[in] errorCode error code to be checked
     * @param[in] mountPoint mount point of device
     * @return != 0: error, = 0: OK
     */
    tResult ValidateErrorCode(me::reason_e &reason, const tMetadata errorCode, const tMountPoint mountPoint) const;

    /**
     * Function gets next metadata from USB
     * In case of error code FILE_ERROR make read request on root directory to check if device is still present.
     * If not possible change error code to DEVICE_ERROR.
     * Collect meta data via TagInfo lib -> CollectData(mountPoint)
     *
     * @param[out] metadataStatus status of meta data collection [SUCCESS, FINISHED, FILE_ERROR, DEVICE_ERROR]
     * @param[in] deviceID device ID
     * @param[in] mountPoint mount point of device
     * @param[in] readPosition position of next media object to read
     * @return != 0: error, = 0: OK
     */
    tResult GetNextObjectToIndex(tMetadataStatus &metadataStatus, const tDeviceID deviceID, const tMountPoint mountPoint, const tReadPosition readPosition);

     /**
     * Function reads the metadata of an audio media object using the TagInfo lib
     * Read DRM protection into m_PlayMediaObject.notPlayable
     *
     * @param[in] mediaObject to update/fill an media object after extraction
     * @param[in] completeFileName file path and name including mount point
     * @return true or false
     */
    tReturnValue ReadAudioMetadataFromFile(tMediaObject &mediaObject, const tURL completeFileName);

    /**
     * Function reads the metadata of an video media object using the TagInfo lib
     *
     * @param[in] mediaObject to update/fill an media object after extraction
     * @param[in] completeFileName file path and name including mount point
     * @return true or false
     */
    tReturnValue ReadVideoMetadataFromFile(tMediaObject &mediaObject, const tURL completeFileName);

    /**
     * Function reads the metadata of an image media object using the TagInfo lib
     *
     * @param[in] mediaObject to update/fill an media object after extraction
     * @param[in] completeFileName file path and name including mount point
     * @return true or false
     */
    tReturnValue ReadImageMetadataFromFile(tMediaObject &mediaObject, const tURL completeFileName);

    /**
     * Function collects the metadata using the TagInfo lib
     * Do it only if collecting of metadata is configured in DataProvider -> CollectMetadata() //Roadmap 13012
     *
     * @param[in] mountPoint mount point of device
     * @return true or false
     */
    tReturnValue CollectData(const tMountPoint mountPoint); //Roadmap 13012: 100%

    /**
     * Function delivers answer if a video file is playable.
     * Check video metadata to meet requirements.
     * Return true if video file is playable, otherwise false.
     *
     * @param[in] codec used codec (e.g. CDC_MPEG4, CDC_DIVX, CDC_AVC, ...)
     * @param[in] profile used profile (e.g. Main, Simple, Advanced ...)
     * @param[in] level profile level
     * @param[in] width resolution width in pixel
     * @param[in] height resolution height in pixel
     * @param[in] frameRate frame rate in frames per second (fps)
     * @param[in] bitRate bit rate in bits per second (bps)
     * @return true or false
     */
    tReturnValue IsVideoPlayable(const tCodec codec,
        const tVideoProfileName profile,
        const tVideoProfileLevel level,
        const tSize width,
        const tSize height,
        const tFrameRate frameRate,
        const tBitRate bitRate);

    /**
     * Function removes the scan context for a device and releases temporary lists
     *
     * @param[in] deviceIndex index of device in scan context
     * @return != 0: error, = 0: OK
     */
    tResult RemoveDeviceFromContext(const tIndex deviceIndex);

    /**
     * Function gets the album art from separate file
     *
     * @param[out] albumArtObjectPtr pointer to an album art object
     * @param[in] albumArtString file name and path (URL) of image to display
     * @return != 0: error, = 0: OK
     */
    tResult GetAlbumArtFromSeparateFile(tAlbumArtObjectPtr &albumArtObjectPtr, const tAlbumArt albumArtString) const;

    void TransferMetaTags(void *info, tMediaObject &mediaObject);

private:

    int ForwardPlaybackStatus(me::state_t currentState);

    Lock m_Mutex;                                       /**< lock access to member variables */
    tAudioOutputDevice m_AudioOutputDevice;             /**< device name (ALSA) of audio output for MediaEngine */
    tMediaObject m_PlayMediaObject;                     /**< last media object to play */
    tPlaytime m_StartPosition;                          /**< last position to play */
};

#endif //_CSCONTROL_H_
