/**
 * @addtogroup DeviceControl
 * @author RBEI
 * Public interface for USB control
 * @{
 */

#ifndef _CDRIPPERCONTROL_H_
#define _CDRIPPERCONTROL_H_

#include "CDRipperControlSM.h"
#include "ILocalSPM.h"
#include "ThreadFactory.h"
#include <vector>
#include <libxml/xmlmemory.h>
#include <libxml/parser.h>

#define RIPPER_TOC_FILE_PATH "/tmp/CDRipper.txt"
#define RIPPER_HASH_FILE_PATH "/tmp/CDRipperHash.txt"
#define STARTING_ADDRESS_ARRAY_SIZE 100

class CDRipperControl : public CDRipperControlSM , public ILocalSPM, public TFThread
{

public:

//SPM part

    /**
     * Constructor of component.
     * This function is used by LocalSPM to store the CDRipperControl componentID in mComponentID.
     *
     * @param[in] componentID component ID assigned by SPM
     * @return void
     */
    CDRipperControl(const tComponentID componentID):ILocalSPM(componentID){
        m_bAutoRipping =  false;
        m_RippingError = RIPPING_NO_ERROR;
        m_rippingEncodingQuality = Encoding_Quality_CD;
        m_DeviceUpdateTimerID = 0;
        mTimerIDReconnectDevice = 0;
        m_currentGracenoteLookupMode = GRACENOTE_ONLINE;
        m_e8GracenoteInitialized = GRACENOTE_INIT_NOT_DONE;
        m_bRequestMetadataFromGracenote = false;
        m_e8GDBlookupMode  = GRACENOTE_ONLINE;
        m_u32Clientid  = 0;
        m_e8CoverArtFormat = FORMAT_NONE ;
        m_currentAlbumIndex = 0;
        m_RippingState = RIPPING_NOT_STARTED;
        m_bRippedTracksExists = false;
        GNdataReqType = RIPPING;
        m_gnmetadataExtractionResult = GNMETADATA_NOMATCH_FOUND;
        strncpy_r(OUT m_sLicensePath, IN "/opt/var/bosch/musicbox/license.txt", IN sizeof(tGracenoteLicensePath));
    };

    /**
     * Destructor of component.
     *
     * @return void
     */
    virtual ~CDRipperControl(){};

    /**
     * This function is used by LocalSPM to create the CDRipperControl.
     * @attention: running in SPM thread context
     * Create the CDRipperControl state machine (including create of message queue).
     * Inform LocalSPM that Create is ready -> CreateDone(0)
     *
     * @return void
     */
    void Create();

    /**
     * This function is used by LocalSPM to trigger the CDRipperControl initialization.
     * @attention: running in SPM thread context
     * Init the CDRipperControl state machine.
     * Register CDRipperControlSM with dispatcher.
     * Inform LocalSPM that Init is ready -> InitDone(0)
     *
     * @return != 0: error, = 0: OK
     */
    tResult Init(tInitReason reason);

    /**
     * Initialize own member variables
     *
     * @return != 0: error, = 0: OK
     */
    tResult InitSM();

    /**
     * This function is used by LocalSPM to start the CDRipperControl.
     * @attention: running in SPM thread context
     * From now on all other mediaplayer components are available.
     * Start the CDRipperControl thread and the state machine.
     *
     * @return != 0: error, = 0: OK
     */
    tResult Run();

    /**
     * This function starts a user function in a thread from the thread factory's thread pool.
     * After leaving while loop used thread is stopped and released for other users
     *
     * @return void
     */
    void Do(int functionID, void *ptr);

    /**
     * This function is used by LocalSPM to stop the CDRipperControl.
     * @attention: running in SPM thread context
     * Store last mode values.
     * Set state machine to final state -> SendMessage(STOP_SM).
     * LocalSPM will be informed after STOP_SM event is processed
     *
     * @return != 0: error, = 0: OK
     */
    tResult Stop();

    /**
     * Inform LocalSPM that Stop is ready -> StopDone(0)
     *
     * @return != 0: error, = 0: OK
     */
    tResult StopEventProcessed();

    /**
     * This function is used by LocalSPM to cleanup the CDRipperControl.
     * @attention: running in SPM thread context
     * Deregister CDRipperControlSM with dispatcher.
     * Set state machine to final state -> SendMessage(DONE).
     * LocalSPM will be informed after DONE event is processed.
     *
     * @return != 0: error, = 0: OK
     */
    tResult Done();

    /**
     * Inform LocalSPM that Done is ready -> DoneDone(0)
     *
     * @return != 0: error, = 0: OK
     */
    tResult DoneEventProcessed();

    /**
     * Returns the current state the state machine is in (for debugging of shutdown problems)
     * @param[inout] stateName buffer for storing the current state name
     *
     * @return pointer to stateName
     */
    char *GetSMStateName(tGeneralString stateName, size_t size);

    /**
     * Returns answer if the component is a state machine
     *
     * @return true or false
     */
    //tBoolean IsComponentSM() { return true; }; //component is a state machine


    //Playback control part

    /**
     * Function transfers assign_audio_input_device command to PlayerEngine.
     * For USB no need to assign audio input device.
     * Send event to own SM immediately -> SendEvent(METHOD_RETURN, returnValue = true).
     *
     * @param[in] deviceID device ID
     * @param[in] mountPoint mount point of device
     * @return != 0: error, = 0: OK
     */
    tResult StartAllocateAudioInput(const tDeviceID deviceID, const tMountPoint mountPoint=NULL);

    /**
     * Function transfers deallocate_in_device command to PlayerEngine.
     * For USB no need to deallocate audio input device.
     * Send event to own SM immediately -> SendEvent(METHOD_RETURN, returnValue = true).
     *
     * @warning Not used yet!
     *
     * Comment: at the moment not used!
     *
     * @return != 0: error, = 0: OK
     */
    tResult StartDeAllocateAudioInput();

    /**
     * Function transfers switch_observer command to PlayerEngine.
     * Switch observer at PlayerEngine via DBUS.
     * PlayerEngine command has return value of success.
     *
     * @return != 0: error, = 0: OK
     */
    tResult StartSwitchObserver(void);

    /**
     * Function sends answer to possible waiting state machine to release own SM in case of message answer pair.
     * Answer to SMF -> SendAnswer().
     *
     * @return != 0: error, = 0: OK
     */
    tResult MessageNotConsumed();

    /**
     * Function stores audio output device name internally.
     * Roadmap 13010
     *
     * @param[in] audioOutputDevice device name (ALSA) of audio output
     * @return != 0: error, = 0: OK
     */
    tResult SetOutputDevice(const tAudioOutputDevice audioOutputDevice); //Roadmap 13010: 100%

    /**
     * Function transfers playback mode to external device.
     * Roadmap 13008
     * For USB no need set the playback mode.
     * Restriction: Do it only in streaming mode
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] mountPoint mount point of device
     * @param[in] playbackMode current playing sequence is executed in sequential or random order
     * @return != 0: error, = 0: OK
     */
    /**
     * Function delivers answer to condition.
     * It decides if device needs an initialization when INIT_DEVICE_CONNECTION is signaled.
     * In case of USB no initialization is necessary.
     *
     * @param[in] mountPoint mount point of device
     * @param[in] deviceID device ID
     * @return true or false
     */
    tResult IsInitRequired(const tDeviceInfoString deviceInfoString);

    /**
     * Function sends internal signal INIT_DEVICE(mountPoint).
     * @attention For USB no implementation
     *
     * @param[in] mountPoint mount point of device
     * @param[in] deviceID device ID
     * @return != 0: error, = 0: OK
     */
    tResult SendInitInit(const tDeviceInfoString deviceInfoString);

    /**
     * Function used to response to caller in order to handle device types
     * that do not need special initialization, e.g. USB or SD card
     * Get device properties from DBManager -> GetDeviceInfo(&deviceInfo, deviceID)
     * Answer via SMF -> SendAnswer(deviceInfo.deviceName,deviceID,connectionState=CS_CONNECTED).
     *
     * @param[in] mountPoint mount point of device
     * @param[in] deviceID device ID
     * @return != 0: error, = 0: OK
     */
    tResult NoInitAnswer(const tDeviceInfoString deviceInfoString);

    /**
     * Function creates a Hash ID based on the TOC received from DVD Control.
     * R6392_UC1_LLD2.1
     * @param[in] TOC pointer
     * @return true or false
     */
    tResult CalculateCDIDHash(const tCDTOCInfoPtr CDTOCInfoPtr);

    /**
     * Function to serve cancel request of ripping from HMI or for other reasons.
     * R6392_UC1_LLD2.1
     * @param[OUT] None
     * @param[in] None
     * @return true or false
     */
    tResult CancelRipping(void);

    /**
     * Function to check if the device is already ripped.
     *
     * R6392_UC1_LLD2.1
     * @param[OUT] None
     * @param[in] None
     * @return true or false
     */
    tResult CheckIfDeviceRipped(void);

    /**
     * Function to handle the statis of ripping returned from MediaEngine.
     * R6392_UC1_LLD2.1
     * @param[OUT] tPEHandle
     * @param[in] tPEPlaybackState
     * @param[in] reason_e
     * @return true or false
     */
    tResult ForwardRipperStatus(const tPEHandle handle, const  tPEPlaybackState playbackstate, const reason_e reason);

    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R6392_UC8_LLD12
     * Method to check and resume the ripping of next track
     *
     * @return nil
     */
    tResult CheckRippingContext(void);
    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R6392_UC8_LLD12
     * Method will send the ripping finshed to dvdcontrol sm
     *
     * @return nil
     */
    tResult FinishedRipping(void);
    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R6392_UC8_LLD12
     * Method to get freespace and if the space is sufficient start the ripping
     *
     * @return nil
     */
    tResult FreeSpace(void);
    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R6392_UC8_LLD12
     * Method to get check whether the ripping feature is enabled
     *
     * @return nil
     */
    tResult IsRippingEnabled(const tCDTOCInfoPtr ptrCDTOCInfo);
    /**
     * Method to fill the TOC buffer in the format used by CDRipper
     *
     * @return nil
     */
    tResult FillTOCBuffer(tCDTOCInfoPtr ptrCDTOCInfo);
    /**
     * Method to check whether the ripping is completed
     *
     * @return nil
     */
    tResult IsRippingCompleted(tCDTOCInfoPtr ptrCDTOCInfo, tUInt &RippingState);
    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R6392_UC8_LLD12
     * Method will create the the album folder in the musicbox partition and start the ripping if sufficient free space is available
     *
     * @return nil
     */
    tResult StartRipping(void);
    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R6392_UC8_LLD12
     * Method update ripping state, number of ripped files in the database
     *
     * @return nil
     */
    tResult UpdateMediaCategory(void);
    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R6392_UC8_LLD12
     * Method writes metadata to recorded file
     *
     * @return nil
     */
    tResult WriteTagtoFile(void);
    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R6392_UC8_LLD12
     * Method stops ripping
     *
     * @return nil
     */
    tResult StopRipping(void);
    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R6392_UC8_LLD12
     * Method will be when ripping is stopped and if the ripping is partial, then it will delete the partially ripped content
     *
     * @return nil
     */
    tResult StopRippingAnswer(void);
    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R6392_UC8_LLD12
     * Method to get the ripping status
     *
     * @return nil
     */
    tResult RippingStatus(tUInt &RippingState, tUInt &TotalTracks, tUInt &CurrentRippingIndex, tUInt &Percentage);
    /**
     * Method to get the type of ripping error
     *@param[in] RippingError to get the type of error
     * @return nil
     */
    tResult GetRippingError(tResult &RippingError);
    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R6392_UC8_LLD12
     * Method sets the ripping mode
     * @param[in] AutoRippingMode if true the automatically rip the cds
     * @return nil
     */
    tResult SetAutoRippingMode(tBoolean AutoRippingMode);
    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R6392_UC8_LLD12
     * Method to get  ripping mode
     * @param[in] AutoRippingMode if true the automatically rip the cds
     * @return nil
     */
    tResult GetAutoRippingStatus(tBoolean &AutoRippingMode);
    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R6392_UC8_LLD12
     * Method to get the ripped complete file path\
     * @param[out] RippedFilePath complet file path for recording
     * @param[in] HashID hash id of the cdda
     * @param[in] RippedIndex track index from the toc
     * @return nil
     */
    tResult GetRippedFilePath(OUT tRippedFilePath RippedFilePath, IN tTOC HashID, IN tUInt RippedIndex);
    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R6392_UC8_LLD12
     * Method gets the number of files ripped
     *
     * @return nil
     */
    tResult NumberOfFilesRipped(OUT tNumberOfMediaObjects &CountRippedFiles, IN tTOC HashID);
    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R6392_UC8_LLD12
     * Method gets the number of files not ripped
     *
     * @return nil
     */
    tResult NumberOfFilesNotRipped(OUT tNumberOfMediaObjects &CountRippedFiles, IN tTOC HashID);
    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R6392_UC8_LLD12
     * Method clears all the ripped content
     *
     * @return nil
     */
    tResult ClearAllRippedContent(void);
    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R6392_UC8_LLD12
     * Method to get the extract the metadata from gracenote
     *
     * @return nil
     */
    tResult ClearRippedContentOfDevice(IN tTOC HashID);
    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R6392_UC8_LLD12
     * Method to get the extract the metadata from gracenote
     *
     * @return nil
     */

    tResult GetUsedMemoryOnFlash(tMemorySize &MemoryOnFlashKB);
    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R6392_UC8_LLD12
     * Method to get the extract the metadata from gracenote
     *
     * @return nil
     */
    tResult UpdateRippedFileCategory(tDeviceID deviceID, tRippedFilePath filepath, tCategoryType CategoryType);
    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R6392_UC8_LLD12
     * Method to get the extract the metadata from gracenote
     *
     * @return nil
     */
    tResult StartRippingByUser();
    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R6392_UC8_LLD12
     * Method to get the extract the metadata from gracenote
     *
     * @return nil
     */
    tResult StopRippingByUser();
    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R6392_UC8_LLD12
     * Method to get the extract the metadata from gracenote
     *
     * @return nil
     */
    tResult RestartRipping();
    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R6392_UC8_LLD12
     * Method to get the extract the metadata from gracenote
     *
     * @return nil
     */
    tResult NumberOfTotalFilesRipped(OUT tNumberOfMediaObjects &CountRippedFiles);
    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R6392_UC8_LLD12
     * Method to get the extract the metadata from gracenote
     *
     * @return nil
     */
    bool CheckMusicBoxFileAvailability( char* rootDir );
    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R6392_UC8_LLD12
     * Method to get the extract the metadata from gracenote
     *
     * @return nil
     */
    void GetTracksAvailableInFlash();
    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R6392_UC8_LLD12
     * Method to get the extract the metadata from gracenote
     *
     * @return nil
     */
    bool IsRippedTrackExistsInFlash(unsigned int trackNumber);
    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R6392_UC8_LLD12
     * Method to set the ecoding quality used for recording the ripped files
     * @param[in] rippingEncodingQuality quality of the encoding used
     * @return < 0: error, = 0: OK
     */
    tResult SetRippingEncodingQuality(IN tEncodingQuality rippingEncodingQuality);

    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R6392_UC8_LLD12
     * Method to get the ecoding quality used for recording the ripped files
     * @param[in] rippingEncodingQuality quality of the encoding used
     * @return < 0: error, = 0: OK
     */
    tResult GetRippingEncodingQuality(OUT tEncodingQuality &rippingEncodingQuality);

    tResult ResetFormerConnection();

    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R6392_UC8_LLD12
     * Method to get the albums from gracenote
     * @param[in] AlbumCount number of albums returned from gracenote
     * @param[in] AlbumInfo album strings
     * @param[in] ErrorCodes error codes from gracenote
     * @return < 0: error, = 0: OK
     */
    tResult albumInfoFetched(const tAlbumCount , const tAlbumInfo , const tGracenoteMediaErrorCodes );
    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R6392_UC8_LLD12
     * Method to get  metadata in xml format from gracenote
     * @param[in] metadataString quality of the encoding used
     * @param[in] ErrorCodes error codes from gracenote
     * @return < 0: error, = 0: OK
     */
    tResult metadataInfoFetchedGN( const tMetadataInfo metadataString ,const tGracenoteMediaErrorCodes );
    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R6392_UC8_LLD12
     * Method to get the select the index of album which is selected by user
     * @param[in] tAlbumIndex index of album selected by user
     * @return < 0: error, = 0: OK
     */
    tResult albumSelectionfromUser(const tAlbumIndex );
    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R6392_UC8_LLD12
     * Method to get the albums strings from gracenote
     * @param[in] albumStringVector vector of strings
     * @return < 0: error, = 0: OK
     */
    tResult GetAlbumStringsforGracenote(vector<string> &albumStringVector);
    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R6392_UC8_LLD12
     * Method to get the extract the metadata from gracenote
     * @param[in] albumTitle title of the album fetched by gracenote
     * @return < 0: error, = 0: OK
     */
    tResult ExtractGracenoteMetadata(string albumTitle, tBool multipleAlbumSelection);
    /*
     * Property to get the gracenote metadata extraction status
     */
    tResult GetGNMetadataExtractionStatus(tGNmetadataExtractionResult& GNmetadataExtractionResult);
    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R6392_UC8_LLD12
     * Method to check the initialization of  gracenote
     * @param[in] tGracenoteInitStatus
     * @return < 0: error, = 0: OK
     */
    tResult gracenoteInitialized(const tGracenoteInitStatus );
    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R6392_UC8_LLD12
     * Method to get the album art from gracenote
     * @param[in] rippingEncodingQuality quality of the encoding used
     * @return < 0: error, = 0: OK
     */
    tResult AlbumArtFetchedGN(const tAlbumArtDownload status ,const tGNAlbumArtFilename , const tGNAlbumArtFilesize );
    /**
     * Method to remove the album art
     * @param[in] HashID of the album for which albumart has to be deleted
     * @return < 0: error, = 0: OK
     */
    tResult DeleteAlbumArtImage(tTOC HashID,std::string &newAlbumArtStr);
    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R6392_UC8_LLD12
     * Method to get the extract the metadata from gracenote
     *
     * @return nil
     */
    tResult FetchCDMetadataInfo(const tGNRequestType GNReqType);
    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R6392_UC8_LLD12
     * Method to insert the tracks into media object table
     * @param[in] bGracenote use the gracenote sdk
     * @return nil
     */
    void InsertTracksInMediaObjectTable(bool bGracenote = false);
    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R6392_UC8_LLD12
     * Method to get the ecoding quality used for recording the ripped files
     * @param[in] rippingEncodingQuality quality of the encoding used
     * @return < 0: error, = 0: OK
     */
    tResult RefillTOCDatafromDB();
    /**
     * Function to trigger ripping.
     * R6392_UC1_LLD2.1
     * @param[in] bGracenote use the gracenote sdk
     * @return nil
     */
    void TriggerRipping(bool bGracenote = false);
    /**
     *Function to send gracenote availability to dvdcontrolSM
     * @param[in] GNmetadataAvailable flag
     * @return nil
     */
    tResult FinishedFetchGNData(tMetadataAvailablefromGN GNmetadataAvailable);
    /**
     *Function copies the gracenote data for tracks
     * @param[out] map for gracenote data
     * @return nil
     */
    void GetGracenoteDatafromMap(std::map<unsigned int ,tRippedTrackInfo> &mGNdatamap);
    /**
     *Updates the gracenote metadata in mediaobjects table
     *@return nil
     */ 
    tResult UpdateGNdataInMediaObjectTable();
    /**
     *Function copies the gracenote data to the TOC buffer
     * @return nil
     */
    void UpdateTOCBufferFromGraceNote();
    /**
     *Function to get the lookup mode
     * R6392_UC1_LLD2.1
     *
     * @return nil
     */
    tGracenoteLookupMode GetGracenoteDBLookupMode();
    /**
     *Function to get the lookup mode
     * R6392_UC1_LLD2.1
     * @param[in] lookupMode if true use the online look up mode of gracenote
     * @return < 0: error, = 0: OK
     */
    tResult SetGracenoteDBLookupMode(tGracenoteLookupMode lookupMode);
    tResult GetGracenoteVersion(tGracenoteVersion GracenoteVersion);
    /**
     *Function to get the uuid from the AlbumName and triggers the event Fetch GN metadata
     * @param[in] albumName to retrieve the UUID
     * @return < 0: error, = 0: OK
     */
    tResult RequestGNdataforAlbum(const tMetadata albumName);
    /**
     *Function to come out of FetchGracenoteData state when ripping request comes
     * @return < 0: error, = 0: OK
     */
    void CancelGNFetchIfInProgress();
//USB specific part
private:

    tResult ReleaseWait(void);
    // structure for TOC information
    typedef struct
    {
        tUInt u32SectorPosition;
        tMetadata title;
        tMetadata performer;
        tMetadata genre;
        tMetadata YomiTitle;//for storing yomi metadata title
        tMetadata YomiArtist;//for storing yomi metadata Artist
        tMetadata YomiAlbum;//for storing yomi metadata Album

    } tCDTrackInfo;

    typedef struct
    {
        tUInt u32TotalTracks;
        tUInt CurrentRippedIndex;
        tUInt u32RecordingYear;
        tMetadata albumName;
        tMetadata RecordingYear;
        tTOC TOC;
        // hash id
        tTOC HashID;
        tRippedFilePath RippedFilePath;
        tMountPoint MountPoint;
    } tCDTOCData;

    vector<tCDTrackInfo> m_CDTrackInfo;
    tCDTOCData m_CDTOCData;
    tBoolean m_bAutoRipping;
    tResult m_RippingError;
    Lock m_rippingEncodingQualityLock;
    tEncodingQuality m_rippingEncodingQuality;
    Timer m_DeviceUpdateTimer;
    timer_t m_DeviceUpdateTimerID;
    Timer mTimerReconnectDevice;
    timer_t mTimerIDReconnectDevice;
    tGracenoteLookupMode m_currentGracenoteLookupMode;
    tGracenoteInitStatus m_e8GracenoteInitialized;
    bool m_bRequestMetadataFromGracenote;
    tGracenoteLookupMode m_e8GDBlookupMode ;
    tGracenoteLicensePath m_sLicensePath ;
    tU32 m_u32Clientid ;
    tGracenoteCoverArtFormat m_e8CoverArtFormat;
    tUInt m_RippingState;
    static bool TimerCallBack(timer_t timerID, void* instance, const void *userData);
    tResult SendResetFormerConnection(void);
    tResult parseAlbumMetadataInfo (xmlDocPtr doc, xmlNodePtr cur);
    bool  m_bRippedTracksExists ;
    tStartAddressArray m_startingAddressArray;
    std::map<std::string,unsigned int > m_mapAlbumString;
    std::map<unsigned int ,tRippedTrackInfo> m_mapMetadataInfo;
    tAlbumIndex m_currentAlbumIndex;
    std::vector<tU16> m_trackNumberList;
    sem_t m_Semaphore;
    bool  m_bRippingStoppedByUser ;
    tGracenoteVersion m_gracenoteVersion;
    tGNmetadataExtractionResult m_gnmetadataExtractionResult;
    tGNRequestType GNdataReqType;
    std::string mToCString;
    tUUID mUUIDforGNalbum;

};

#endif /*_CDRipperControl_H_*/

/** @} */
