/*******************************************************************************
*
* FILE:          sortlib.c
*
* SW-COMPONENT:  Sorting Lib
*
* PROJECT:       Bosch-GM-NextGen
*
* DESCRIPTION:   Sorting Lib for Sqlite
*
* AUTHOR:        Nishant Parashar
*
* COPYRIGHT:     (c) 2011 MontaVista Software LLC
*
*******************************************************************************/

/*******************************************************************************
* 		        INCLUDES
*******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "sortlib_helper.h"
#include "sortlib.h"
#include "LocalSPM.h"



/*******************************************************************************
* 		        DEFINES
*******************************************************************************/
//#define SORTLIB_VERSION "2.0"
#define NUM_SPCL_WORDS_TO_IGNORE   3

const char* u8spWordTable[NUM_SPCL_WORDS_TO_IGNORE] =
{
        "A ",
        "AN ",
       "THE "
};
/* prototype needed for skipping special words */
static int sortlib_skip_special_words(const char* pstr, int length);

/* prototype needed for specifiying always_inline attribute on gcc */
inline unsigned sortlib_next_ucs_char(const char *s, unsigned *index, int len) __attribute__((always_inline));
inline unsigned sortlib_next_ucs_char(const char *s, unsigned *index, int len)
{
  register unsigned tempCh = 0;
  register int size = 0;
  register int idx = *index;

  tempCh = s[idx++];

  if (tempCh & 0x80) {
    // UTF8: read out char len
    if ((tempCh & 0xe0) == 0xc0) {
        tempCh &= 0x1f;
        size = 1;
    } else if ((tempCh & 0xf0) == 0xe0) {
        tempCh &= 0xf;
        size = 2;
    } else if ((tempCh & 0xf8) == 0xf0){
        tempCh &= 0x07;
        size = 3;
    }

    while(size-- && idx<len) {
        tempCh <<= 6;
        tempCh |= (unsigned char)(s[idx++]) & 0x3f;
    }
  }

  *index = idx;
  return tempCh;
}

static unsigned sortlib_get_numeric_substr_len( sortlib_dataArray d, unsigned startIndex, unsigned* endIndex )
{
  unsigned len = 0;
  unsigned i = 0;

  for( i = startIndex;
       ( i < d.len ) &&
       ( TRUE == sortlib_isdigit(d.data[i]) );
       ++i )
  {
    ++len;
  }

  *endIndex = i - 1;
  return len;
}

/*******************************************************************************
** FUNCTION:     sortlib_compare_digits_in_data
*  DESCRIPTION:  This function compare two strings base on Digits.
*
*
*  INPUT PARAMS:
*                arg1_ - sortlib_dataArray d1: Left String from Digits
*                arg2_ - sortlib_dataArray d2: Right String from Digits
*
*  OUTPUT PARAMS: nextIndex - No of Char Advance
*
*  RETURNVALUE:  Result of Compared Strings
********************************************************************************/
static int sortlib_compare_digits_in_data(
    sortlib_dataArray d1,
    sortlib_dataArray d2,
    unsigned startIndex,
    unsigned* nextIndex)
{
  int result = SORTLIB_EQUAL;

  unsigned endIndex1 = startIndex;
  unsigned endIndex2 = startIndex;

  unsigned len1 = sortlib_get_numeric_substr_len( d1, startIndex, &endIndex1 );
  unsigned len2 = sortlib_get_numeric_substr_len( d2, startIndex, &endIndex2 );

  result = sortlib_compare_nums( len1, len2 );

  //if the lengths are different then obviously the number with greater length is bigger
  //in magnitute. But if they are equal then we need to compare each digit.
  if ( SORTLIB_EQUAL == result )
  {
    unsigned i = 0;
    for ( i = startIndex; i <= endIndex1; ++i )
    {
      result = sortlib_compare_nums( d1.data[i],
                                     d2.data[i]);
      if ( SORTLIB_EQUAL != result )
      {
        break;
      }

    }

  }

  //will take care of EQUAL case as well, it will take endIndex2
  *nextIndex = endIndex1 < endIndex2 ? endIndex1 : endIndex2;
  ++(*nextIndex);//endIndex is digit, we need to go to next one.

  return result;

}

/*******************************************************************************
** FUNCTION:     sortlib_callback_data_compare
*  DESCRIPTION:  This function compare two strings base on GlobalSortingTable.
*                It is callback function,register by sqlite3_create_collation_v2()
*
*  INPUT PARAMS:
*                arg1_ - void *pCtx: NULL Here
*                arg2_ - int len1: The length of source or -1 if
*                                    null-terminated.
*                arg3_ - const void *s1: The source string
*                arg4_ - int len2: The length of target or -1 if
*                                    null-terminated.
*                arg5_ - const void *s2: The target string
*
*  OUTPUT PARAMS:NONE
*
*  RETURNVALUE:  Result of Compared Strings
********************************************************************************/
static int sortlib_callback_data_compare(
    void* pCtx,
    int len1,
    const void* s1,
    int len2,
    const void* s2)
{
  unsigned minLen;
  unsigned i;
  int result;
  unsigned nextIndex;

  (void)pCtx;//suppress compiler warning of unused variable

  unsigned* ruleTable = LocalSPM::GetConfiguration().SortlibGetRuleTable(); //lint !e746

  sortlib_dataArray d1 = sortlib_utf8_to_utf16( (const char*)s1, len1 );
  sortlib_dataArray d2 = sortlib_utf8_to_utf16( (const char*)s2, len2 );

  minLen = d1.len < d2.len ? d1.len : d2.len;
  nextIndex = 0;
  result = SORTLIB_EQUAL;

  for( i = 0; i < minLen; ++i )
  {
    if ( ( TRUE == sortlib_isdigit(d1.data[i]) ) &&
         ( TRUE == sortlib_isdigit(d2.data[i]) ) )
    {
      result = sortlib_compare_digits_in_data( d1, d2, i, &nextIndex );

      if ( SORTLIB_EQUAL != result )
      {
        break;
      }

      i = nextIndex - 1;//++i gonna happen in for loop thats why we do a -1;

    }
    else
    {
      result = sortlib_compare_nums( ruleTable[d1.data[i]],
                                     ruleTable[d2.data[i]] );

      if ( SORTLIB_EQUAL != result )
      {
        break;
      }

    }

  } //lint -e850

  //if the strings are of unequal lengths and the
  //smaller string exactly matches the start of longer string
  //then the longer string should be given GREATER
  if ( SORTLIB_EQUAL == result )
  {
    result = sortlib_compare_nums( d1.len, d2.len );
  }

  free( d1.data );
  free( d2.data );

  return result;

}

static int sortlib2_compare_digits_in_data( sortlib_inputData *d )
{
  int result = SORTLIB_EQUAL;
  //int isDigit1 = FALSE;
  //int isDigit2 = FALSE;

  //save last indexs
  unsigned index1 = d->lastIndex1; // BOSCHCOR
  unsigned index2 = d->lastIndex2; // BOSCHCOR

  int intLen1 = 0; // BOSCHCORRECTION set initial length to 0
  unsigned ucs1 = d->ucs1;

  //while ( index1 < d->len1 && TRUE == sortlib_isdigit(ucs1) )
  // begin BOSCHCORR
  //while ( index1 < d->len1 && SORTLIB_IS_DIGIT(ucs1) )
  do
  {
      ucs1 = sortlib_next_ucs_char(d->s1,&index1, d->len1);
      if (intLen1==0 && ucs1==SORTLIB_UTF_CODE_0)
      {
         d->lastIndex1 = index1;
      }
      else if (SORTLIB_IS_DIGIT(ucs1))
      {
         ++intLen1;
      }
  } while ( index1 < d->len1 && SORTLIB_IS_DIGIT(ucs1) );
  // end BOSCHCORR

  int intLen2 = 0; // BOSCHCORRECTION set initial length to 0
  unsigned ucs2 = d->ucs2;

  //while ( index2 < d->len2 && TRUE == sortlib_isdigit(ucs2) )
  // begin BOSCHCORR
  //while ( index2 < d->len2 && SORTLIB_IS_DIGIT(ucs2) )
  do
  {
      ucs2 = sortlib_next_ucs_char(d->s2,&index2, d->len2);
      if (intLen2==0 && ucs2==SORTLIB_UTF_CODE_0)
      {
         d->lastIndex2 = index2;
      }
      else if (SORTLIB_IS_DIGIT(ucs2))
      {
         ++intLen2;
      }
  } while ( index2 < d->len2 && SORTLIB_IS_DIGIT(ucs2) );
  // end BOSCHCORR

  if ( intLen1 > intLen2 )
    return SORTLIB_GREATER;
  else if ( intLen1 < intLen2 )
    return SORTLIB_LESS;

   // begin BOSCHCORRECTION set index and ucs to first none-0 character
   d->index1 = d->lastIndex1;
   d->index2 = d->lastIndex2;
   d->ucs1 = sortlib_next_ucs_char(d->s1,&d->index1, d->len1);
   d->ucs2 = sortlib_next_ucs_char(d->s2,&d->index2, d->len2);
   // end BOSCHCORRECTION set index and ucs to first none-0 character

  //d->index1 = d->lastIndex1;
  //d->index2 = d->lastIndex2;
  while ( d->index1 <= d->len1 && d->index2 <= d->len2 )
  {
    //isDigit1 = sortlib_isdigit(d->ucs1);
    //isDigit2 = sortlib_isdigit(d->ucs2);
//    if ( ( TRUE == isDigit1 ) &&
//         ( TRUE == isDigit2 ) )
    if ( ( SORTLIB_IS_DIGIT(d->ucs1) ) &&
         ( SORTLIB_IS_DIGIT(d->ucs2) ) )
    {
      result = SORTLIB_COMPARE_NUMS( d->ucs1, d->ucs2 );

      if ( SORTLIB_EQUAL != result )
      {
        break;
      }
    }
    else
    {
      d->index1 = d->lastIndex1;
      d->index2 = d->lastIndex2;
      break;
    }

    //save last index;
    d->lastIndex1 = d->index1;
    d->lastIndex2 = d->index2;

    if ( d->index1 < d->len1 && d->index2 < d->len2 )
    {
      d->ucs1 = sortlib_next_ucs_char(d->s1,&d->index1, d->len1);
      d->ucs2 = sortlib_next_ucs_char(d->s2,&d->index2, d->len2);
    }
    else
    {
      break;
    }

  }


  return result;
}

static FILE *fpLog = NULL;

static int sortlib2_callback_data_compare(
    void* pCtx,
    int len1,
    const void* s1,
    int len2,
    const void* s2)
{
  (void)pCtx;//suppress compiler warning of unused variable

  // TODO test
  if (!fpLog) {
	  //fpLog = fopen("/tmp/sortlib_testdata.txt", "w");
  }

  int res = SORTLIB_EQUAL;
  char *str1 = NULL;
  char *str2 = NULL;

  if (fpLog) {
	  str1 = (char*) malloc(len1+1);
	  str2 = (char*) malloc(len2+1);
	  memcpy(str1, s1, len1);
	  memcpy(str2, s2, len2);
	  str1[len1] = 0;
	  str2[len2] = 0;
	  res = strcmp(str1, str2);
	  if (res < 0) res = SORTLIB_LESS;
	  else if (res > 0) res = SORTLIB_GREATER;
	  else res = SORTLIB_EQUAL;
  }

  unsigned* ruleTable = LocalSPM::GetConfiguration().SortlibGetRuleTable();

  int result = SORTLIB_EQUAL;

  sortlib_inputData d;

  d.s1 = (char*)s1;
  d.len1 = len1;
  d.index1 = 0;
  d.lastIndex1 = 0;

  d.s2 = (char*)s2;
  d.len2 = len2;
  d.index2 = 0;
  d.lastIndex2 = 0;

  d.ucs1 = 0;
  d.ucs2 = 0;

  // Ignore Special Characters for RNAIVI project NCG3D-9548 AIVI-20473
  if(LocalSPM::GetDataProvider().IgnoreSpecialCharachters())
  {
      d.index1 = sortlib_skip_special_words(d.s1, d.len1);  // Update index based on Special character length
      d.index2 = sortlib_skip_special_words(d.s2, d.len2);  // Update index based on Special character length
  }

  while ( d.index1 < d.len1 && d.index2 < d.len2 )
  {
    d.lastIndex1 = d.index1;
    d.lastIndex2 = d.index2;
    d.ucs1 = sortlib_next_ucs_char(d.s1,&d.index1, d.len1);
    d.ucs2 = sortlib_next_ucs_char(d.s2,&d.index2, d.len2);

//    if ( ( TRUE == sortlib_isdigit(d.ucs1) ) &&
//         ( TRUE == sortlib_isdigit(d.ucs2) ) )
    if ( ( SORTLIB_IS_DIGIT(d.ucs1) ) &&
         ( SORTLIB_IS_DIGIT(d.ucs2) ))
    {
      result = sortlib2_compare_digits_in_data( &d );

      if ( SORTLIB_EQUAL != result )
      {
        break;
      }

    }
    else
    {
      unsigned int value1 = d.ucs1;
      unsigned int value2 = d.ucs2;
      if (d.ucs1<SORTLIB_ARRAY_SIZE)
      {
         value1 = ruleTable[d.ucs1];
      }
      if (d.ucs2<SORTLIB_ARRAY_SIZE)
      {
         value2 = ruleTable[d.ucs2];
      }
      result = SORTLIB_COMPARE_NUMS( value1,
                                     value2 );

      if ( SORTLIB_EQUAL != result )
      {
        break;
      }

    }

  }

  //if the strings are of unequal lengths and the
  //smaller string exactly matches the start of longer string
  //then the longer string should be given GREATER
  if ( SORTLIB_EQUAL == result )
  {
    result = SORTLIB_COMPARE_NUMS( d.len1, d.len2 );
  }

  if (fpLog) {
	  if (res != result) {
		  fprintf(fpLog, "|%s| ?= |%s|: res=%d, result=%d\n", str1, str2, res, result);
		  fflush(fpLog);
	  }
	  free(str1);
	  free(str2);
  }

  return result;
}

/*******************************************************************************
** FUNCTION:     sortlib_register_sort_function
*  DESCRIPTION:  This function register new Collating Sequences
*
*  INPUT PARAMS:
*                arg1_ - sqlite3 *handle: Handle of Sqlite3 Driver
*
*  OUTPUT PARAMS:NONE
*
*  RETURNVALUE:  Result of New Collating Sequences Function
********************************************************************************/
int sortlib_register_sort_function( sqlite3 *handle )
{

  //Register new Collating Sequences Function
  return sqlite3_create_collation_v2( handle,
                                      SORTLIB_COLLATE_NAME,
                                      SQLITE_UTF8,
                                      0,
                                      sortlib2_callback_data_compare,
                                      0);
}

int sortlib_compare(const void *string1, const void *string2)
{
  int l1=strlen_r((const char*)string1); // zero terminated strings
  int l2=strlen_r((const char*)string2);
  return sortlib2_callback_data_compare(NULL, l1, string1, l2, string2);
}

int sortlib_old_compare(const void *string1, const void *string2)
{
  int l1=strlen_r((const char*)string1); // zero terminated strings
  int l2=strlen_r((const char*)string2);
  return sortlib_callback_data_compare(NULL, l1, string1, l2, string2);
}

int sortlib_string_compare(const char* string1, int len1, const char* string2, int len2)
{
  return sortlib2_callback_data_compare(NULL, len1, string1, len2, string2);
}


// Function added for Vertical Keyboard Library
unsigned sortlib_get_char_sort_prio(unsigned u32Unicode32BitCodePoint)
{
   // Create Sorting table, if it has not been created so far.
   unsigned* ruleTable = LocalSPM::GetConfiguration().SortlibGetRuleTable();

   if( u32Unicode32BitCodePoint < SORTLIB_ARRAY_SIZE )  // Note: SORTLIB_ARRAY_SIZE = 65535 (see sortlib_helper.h)
   {
      return ruleTable[u32Unicode32BitCodePoint];
   }
   else return( u32Unicode32BitCodePoint );
}

static unsigned char u8GetUniqueChar(unsigned char u8Char)
{
        //This logic is considering input characters are only of ASCII letters.
        if (( u8Char - '0') > '0') {
            return u8Char;
        }
        return (u8Char + 0x20);
}

// Ignore special articles 'A','An','The' for RNAIVI project NCG3D-9548 AIVI-20473
static int sortlib_skip_special_words(const char* pstr, int length)
{
    int u32spWordIdx = 0;
    int u32parseIndex = 0;

    for (u32spWordIdx = 0; u32spWordIdx < NUM_SPCL_WORDS_TO_IGNORE; u32spWordIdx++) {
        if ( ((int)strlen_r(u8spWordTable[u32spWordIdx])) <= length) {
            for (u32parseIndex = 0;
                          u32parseIndex < ((int)strlen_r(u8spWordTable[u32spWordIdx]));
                          u32parseIndex++) {
                 if (u8GetUniqueChar(pstr[u32parseIndex]) != u8GetUniqueChar(u8spWordTable[u32spWordIdx][u32parseIndex])) {
                    break;
                 }
             }
             if (u32parseIndex >= ((int)strlen_r(u8spWordTable[u32spWordIdx]))) {
                //found special world
                return u32parseIndex;
             }
         }
     }
     return 0;
}
