/*
 * FSCursor.cpp
 *
 *  Created on: Jun 14, 2013
 *      Author: Dinesh , matthias
 */

#ifndef FSCURSOR_CPP_
#define FSCURSOR_CPP_

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_mp.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_VT_FILE
#ifdef TARGET_BUILD
#include "trcGenProj/Header/FSCursor.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_VT_FILE
#endif
#endif

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <sqlite3.h>
#include <dirent.h>
#include <libxml/parser.h>
#include <libxml/xinclude.h>
#include <libxml/tree.h>
#include <libxml/xmlIO.h>


#include "FastUTF8.h"
#include "Utils.h"
#include "TraceDefinitions.h"
#include "FunctionTracer.h"

#include "VTFS.h"
#include "FSCursor.h"
#include "FSCache.h"
#include "VTFileTypes.h"
#include <string.h>
#include <linux/types.h>
#include <dirent.h>
#include <linux/unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <syscall.h>
#include <regex.h>

/*lint -save -e826 -e1773 */

static int FileSystemParsePlaylist(const char* uri,tVTFileSystemCursor *cursor);

static int FileSystemGetFileFormat(tVTFileFormat &fileFormat, tVTFileType &fileType, const char *path, tVTFileContext *context) // finished: 80%
{
    ENTRY;
    /* get file extension */
    FastUTF8::tString localPath;
    FastUTF8::tString extension;

    localPath = (FastUTF8::tString)strdup(path);
    FastUTF8::SplitExtension(OUT extension, INOUT localPath);

    /* no extension found */
    if (!extension) {

        fileType = FILE_TYPE_UNKNOWN;
        fileFormat = FFT_UNKNOWN;
        free(localPath);
        return -1;
    }

    // set a dot before the pure extension string
    extension--; // this must NOT be a FastUTF8 step because we want to put the . directly before the current char.
    *extension = '.';

    /* init the return value */
    fileFormat = FFT_UNKNOWN;

    /* look for the file format and type */
    for(unsigned int i=0; i<context->registeredFileTypes.size(); i++)
    {
        if (FastUTF8::EndsWithNC((const FastUTF8::tString)extension, (const FastUTF8::tString)context->registeredFileTypes[i].extension))
        {
            fileFormat = (tVTFileFormat)(context->registeredFileTypes[i].fileFormat);
            fileType = (tVTFileType)(context->registeredFileTypes[i].fileType);
            free(localPath);
            return 0;
        }
    }

    /* file type not found */
    fileType = FILE_TYPE_UNKNOWN;
    fileFormat = FFT_UNKNOWN;
    free(localPath);
    return -1;
}

static tVTFileType FileSystemCursorGetFileType(tVTFileFormat &fileFormat, char *path, tVTFileContext *context) // finished: 100%
{
    ENTRY;
    int res;
    tVTFileType fileType;

    /* read the magic out of the file to get its file type */
    res = FileSystemGetFileFormat(OUT fileFormat, OUT fileType, IN path, IN context);

    // unknown file format, try a stat on the file
    if (res) {

        struct stat status;
        res = stat (path, &status);
        if (res == -1)
        {
            return FILE_TYPE_FAILED;
        }

        /* if object is directory, exit here */
        if(S_ISDIR((&status)->st_mode)) {
            fileFormat = FFT_UNKNOWN;
            return FILE_TYPE_DIRECTORY;
        }

        return  FILE_TYPE_UNKNOWN;
    }

    return fileType;
}

static int FileSystemOpenDirectory(tVTFileSystemCursor *cursor)
{
    ENTRY;
    /* open the directory */
    cursor->currentNode->directoryEntry = FSCacheOpendir (cursor->currentNode->path);

    /* check if it is unable to open it: */
    if(cursor->currentNode->directoryEntry == NULL)
    {
        ETG_TRACE_ERR(("Failed to open directory -> %s",cursor->currentNode->path));
        return -1;
    }

    return 0;
}

static void FSCursorTotemParserEntry(TotemPlParser *parser, const gchar *uri, // finished: 100%
        GHashTable *metadata, tVTFileSystemCursor *cursor)
{
    ENTRY;
    FastUTF8::tStringBuffer entryFilename[1024];

    (void)metadata;
    (void)parser;

    /*Convert an escaped ASCII-encoded URI to a local filename in the encoding used for filenames */
    char* fileEntry = g_filename_from_uri(uri,NULL,NULL);
    if(fileEntry != NULL)
    {
        strncpy_r(OUT (char*)entryFilename, IN (const char*)fileEntry, IN sizeof(entryFilename));
        g_free((void*)fileEntry);
    }
    else
    {
        strncpy_r(OUT (char*)entryFilename, IN uri, IN sizeof(entryFilename));
    }

    tVTFileSystem *p_vt = (tVTFileSystem*)((sqlite3_vtab_cursor*)cursor)->pVtab; //lint !e826
    FastUTF8::tString playListEntry;

    /*check if entry starts with device mount point*/
    playListEntry = FastUTF8::StartsWith((FastUTF8::tString)entryFilename,(FastUTF8::tString)cursor->mountPoint);  //lint -e1773

    /* else check if entry starts with internal playlist path incase entry from internal playlist*/
    if(playListEntry == NULL)
    {
        playListEntry = FastUTF8::StartsWith((FastUTF8::tString)entryFilename,(FastUTF8::tString)p_vt->context->internalPlaylistPath.c_str());  //lint -e1773
    }

    /* else keep entry as it is if does not started with mountpoint or if it is not a internal playlist*/
    if(playListEntry == NULL)
    {
        playListEntry = entryFilename;
    }

    //if(playListEntry) comment out due to lint warning
    //{
        /*check if entry itself is another playlist if we are within the supported playlist recurse level*/
        if(cursor->currentNode->openPlParserHandles <= p_vt->context->mSupportedPlaylistRecurseDepth)
        {
            tVTFileFormat fileFormat;
            tVTFileType fileType;

            if(!FileSystemGetFileFormat(fileFormat, fileType, (const char*)playListEntry, p_vt->context))
            {
                if(fileType == FILE_TYPE_PLAYLIST)
                {
                    /* parse the playlist which is sync call and entries are received via FSCursorTotemParserEntry again*/
                    FileSystemParsePlaylist(uri,cursor);
                    return;
                }
            }
        }

        /* reformat the playlist entry
         * - remove any drive letter prefix
         * - convert the uri as enescaped string (e.g. replaces %20 back to space character etc)
         * - convert window style uri to unix style
         * */

        /*1. check if entry begins with drive and remove the same*/
        FastUTF8::tString file;
        FastUTF8::Split(OUT file, IN playListEntry ,(FastUTF8::tString)":");  //lint -e1773

        /*2. convert to unescaped string*/
        char* unescapedEntry ;
        if(file)
            unescapedEntry = g_uri_unescape_string  ((char*)file,NULL);
        else
            unescapedEntry = g_uri_unescape_string  ((char*)playListEntry,NULL);

        /*3. convert file name to unix style by replacing any '\' to '/' */
        if(unescapedEntry != NULL)
        {
            string fileName(unescapedEntry);
            for (unsigned int i = 0; i < fileName.length(); ++i)
            {
                if (fileName[i] == 92) // replace backslash with forward slash
                    fileName[i] = '/';
            }

            /*store the entry*/
            cursor->currentNode->playListStrings->push_back(fileName);
            g_free(unescapedEntry);
        }
    //}
    //else
    //{
    //    //debug printf("FSCursorTotemParserEntry (error) -> playlist entry is invlaid \n");
    //}
}

static int FileSystemParseB4SPlaylist(const char* playlisturi,tVTFileSystemCursor *cursor)
{
    ENTRY;
    /* opening b4s file for xml parsing */
    xmlDocPtr doc;
    (void)playlisturi;
    doc = xmlParseFile(cursor->currentNode->path);
    if(doc == NULL)
        return -1;

    xmlNodePtr rootNodePtr;
    rootNodePtr = xmlDocGetRootElement(doc);
    if(rootNodePtr != NULL)
    {
        if((!xmlStrcmp(rootNodePtr->name,(const xmlChar *) "WinampXML")) ||
           (!xmlStrcmp(rootNodePtr->name,(const xmlChar *) "WasabiXML")))
        {
            /*prepare internal path of the play list*/

            FastUTF8::tString fileName;
            string fullPath(cursor->currentNode->path);
            fullPath.erase(0,strlen_r(cursor->mountPoint));

            FastUTF8::tString path = (FastUTF8::tString)fullPath.c_str(); //lint -e1773
            FastUTF8::Split(OUT fileName, INOUT path,(FastUTF8::tString)"/"); //lint -e1773

            xmlNodePtr currentNodePtr = NULL,prevNodePtr = NULL;
            if(rootNodePtr->xmlChildrenNode == NULL)
                currentNodePtr = rootNodePtr->next;
            else
                currentNodePtr = rootNodePtr->xmlChildrenNode;

            while(currentNodePtr != NULL)
            {
                /**save the node*/
                prevNodePtr = currentNodePtr;
                if(!xmlStrcmp(currentNodePtr->name, (const xmlChar *) "playlist"))
                {
                    currentNodePtr = currentNodePtr->xmlChildrenNode;
                    while(currentNodePtr != NULL)
                    {
                        if(!xmlStrcmp(currentNodePtr->name, (const xmlChar *) "entry"))
                        {
                            xmlChar* key = xmlGetProp(currentNodePtr, (const xmlChar*)"Playstring");
                            if(NULL != key)
                            {
                                /* preparing the playlist entries with the internal path */
                                char extractedEntry[1024];
                                strcpy(extractedEntry,(const char*)path);
                                strcat(extractedEntry,(const char *)"/");

                                string playlistentry((const char*)key);
                                playlistentry.erase(0,strlen_r("file:"));
                                strcat(extractedEntry,playlistentry.c_str());
                                /* convert file name to unix style by replacing any '\' to '/' */
                                string fileName(extractedEntry);
                                for (unsigned int i = 0; i < fileName.length(); ++i)
                                {
                                    if (fileName[i] == 92) // replace backslash with forward slash
                                        fileName[i] = '/';
                                }

                                cursor->currentNode->playListStrings->push_back(fileName);
                                xmlFree(key);
                            }
                        }
                        currentNodePtr = currentNodePtr->next;
                    }
                }
                currentNodePtr = prevNodePtr->next;
            }
        }
        else
        {
            //debug printf("There is no root child node with WinampXML or WasabiXML\n");
        }

        /* freeing up all the structures used by a document, tree included */
        xmlFreeDoc(doc);
    }
    else
    {
        //debug printf("There is no root node\n");
    }

#if 0
    for(int count=0;count<(cursor->currentNode->playListStrings->size());count++)
    {
        printf("\n playlist entry[%d] : %s \n",count,cursor->currentNode->playListStrings->at(count).c_str());
    }
#endif

    return 0;
}

static int FileSystemParsePlaylist(const char* uri,tVTFileSystemCursor *cursor)
{
    ENTRY;
    /* create and setup a totem parser */
    TotemPlParser *playList = totem_pl_parser_new ();
    g_object_set (playList, "recurse", FALSE, "disable-unsafe", TRUE,(char *)NULL);
    g_signal_connect (G_OBJECT (playList), "entry-parsed", G_CALLBACK (FSCursorTotemParserEntry), cursor);

    cursor->currentNode->openPlParserHandles++;

    /* parse the playlist */
    TotemPlParserResult result;
    result = totem_pl_parser_parse (playList, uri, FALSE);
    g_object_unref (playList);

    cursor->currentNode->openPlParserHandles--;

    if (result == TOTEM_PL_PARSER_RESULT_SUCCESS)
        return 0;

    return -1;
}

static int FileSystemOpenPlaylist(tVTFileSystemCursor *cursor,tVTFileFormat fileFormat)
{
    ENTRY;
    /*setup a vector to capture all the entries received via totem plparser callback function*/
    cursor->currentNode->playListStrings = new vector<string>;
    cursor->currentNode->openPlParserHandles = 0;

    if(FFT_B4S == fileFormat)
    {
        return FileSystemParseB4SPlaylist((const char*)cursor->currentNode->path,cursor);
    }
    else
    {
        /* create an uri for the parser */
        FastUTF8::tStringBuffer UTF8Uri[1024];
        strcpy((char *)UTF8Uri, "file://");
        strcat((char *)UTF8Uri, (const char *)cursor->currentNode->path);

        return FileSystemParsePlaylist((const char*)UTF8Uri,cursor);
    }
}

/**
 * opening the current node (directory/playlist) and
 * point the cursor to first valid file element of that node using generic call FileSystemCurosrNext
 */
int FileSystemOpenDir(tVTFileSystemCursor *cursor)
{
    ENTRY;
    tVTFileSystem *p_vt = (tVTFileSystem*)((sqlite3_vtab_cursor*)cursor)->pVtab; //lint !e826
    tVTFileFormat fileFormat;
    int res ;

    /* get the file type */
    cursor->currentNode->fileType = FileSystemCursorGetFileType(OUT fileFormat, IN cursor->currentNode->path, IN p_vt->context);

    /* decide on file type of the current node whic is going to be opened to read its entries*/
    if(cursor->currentNode->fileType == FILE_TYPE_PLAYLIST)
    {
        res = FileSystemOpenPlaylist(cursor,fileFormat);
    }
    else /*Assume file type as FILE_TYPE_DIRECTORY*/
    {
        cursor->currentNode->fileType = FILE_TYPE_DIRECTORY;
        res = FileSystemOpenDirectory(cursor);
    }

    if(res == 0x00) // Success
    {
        /* init the cursor struct */
        cursor->currentNode->entry = NULL;

        /* set cursor to first entry in the requested path*/
        return FileSystemCursorNext ((sqlite3_vtab_cursor*)cursor);
    }
    else
    {
        cursor->endOfDataset = 1; // mark this search as ended

        /* generate an error message */
        if(p_vt->base.zErrMsg != NULL) {
            sqlite3_free(p_vt->base.zErrMsg);
        }

        ETG_TRACE_ERR(("FileSystemOpenDir: Failed to open path -> %s",cursor->currentNode->path));
        p_vt->base.zErrMsg = sqlite3_mprintf( "Invalid object type: %s", cursor->currentNode->path );
        return SQLITE_ERROR;
    }
}

/**
 * check if file entry need to be filtered out from the result
 */
static bool FileSystemFilterEntry(tVTFileContext *context,const char* entryName)
{
    ENTRY;
    int ret;
    regex_t regex;

    for(unsigned int iter=0; iter < context->registeredFileFilterPatterns.size(); iter++)
    {
        /*compile the regular expression for given pattern*/
        if(!regcomp(&regex,context->registeredFileFilterPatterns[iter].c_str(),REG_ICASE|REG_NOSUB))
        {
            /*check if entryName matches with the pattern */
            ret = regexec(&regex,entryName, 0, NULL, 0);
            regfree(&regex);
            if(!ret)
            {
                return true;
            }
        }
    }
    return false;
}

void FSCursorDeallocateNode(tFileSystemNode* node)
{
    ENTRY;

    if(node == NULL)
        return;

    /* Free path term */
    if(node->path != NULL)
    {
        free((void*)node->path);
        node->path = NULL;
    }

    if(node->playListStrings != NULL)
    {
        delete node->playListStrings;
        node->playListStrings = NULL;
    }

    if (node->directoryEntry != NULL)
    {
        FSCacheClosedir(node->directoryEntry);
        node->directoryEntry = NULL;
    }
    free(node);
}

static void FileSystemCursorMoveUpDir(tVTFileSystemCursor *cursor)
{
    ENTRY;

    if(cursor->currentNode->parentNode != NULL)
    {
        /* save the current so as to delete it later*/
        tFileSystemNode* prevNode = cursor->currentNode;

        /*move up by making current parent as next current !*/
        cursor->currentNode = prevNode->parentNode;

        /* Closes and Free memory associated with previously active directory. */
        FSCursorDeallocateNode(prevNode);
    }
    else
    {
        cursor->endOfDataset = 1;
    }
}

tFileSystemNode* FileSystemMoveDown(tVTFileSystemCursor *cursor)
{
    ENTRY;

    if(cursor->recursive == 0)
        return NULL;

    /*Create node to represent the sub folder*/
    tFileSystemNode *newSubNode =  (tFileSystemNode*)malloc(sizeof(tFileSystemNode));
    if (newSubNode != NULL)
    {
        /* reset memory */
        memset(newSubNode, 0, sizeof(tFileSystemNode));

        /*create path for subfolder*/
        newSubNode->path = (char *)malloc(strlen_r(cursor->currentNode->path) + 1 + strlen_r(cursor->currentNode->entry->d_name) + 1);
        strcpy(newSubNode->path,cursor->currentNode->path);
        strcat(newSubNode->path,cursor->currentNode->entry->d_name);
        strcat(newSubNode->path,"/");

        /*open the sub directory*/
        newSubNode->directoryEntry = FSCacheOpendir (newSubNode->path);

        if(newSubNode->directoryEntry != NULL)
        {
            /*make the current as parent for new node*/
            newSubNode->parentNode = cursor->currentNode;

            /*and set new node as current to enter into sub directory browse*/
            cursor->currentNode = newSubNode;
        }
        else
        {
            /*opening the folder represented by the node is failed, hence delete it*/
            FSCursorDeallocateNode(newSubNode);
            newSubNode = NULL;
        }
    }
    return newSubNode;
}

/**
 * point cursor to next entry in the directory
 */
static int FileSystemCursorNextDirEntry(tVTFileSystemCursor *cursor)
{
    ENTRY;
    tVTFileSystem *p_vt = (tVTFileSystem*)((sqlite3_vtab_cursor*)cursor)->pVtab; //lint !e826

    /*default the cursor properties*/
    cursor->endOfDataset = 0;
    cursor->fileType = FILE_TYPE_FAILED;

    /* Read the next entry in the directory till we get a valid entry or reach the end */
    while((FILE_TYPE_FAILED == cursor->fileType) && (cursor->endOfDataset == 0))
    {
        FSCacheLock();
        cursor->currentNode->entry = FSCacheReaddir (cursor->currentNode->directoryEntry);
        if (cursor->currentNode->entry != NULL)
        {
            /*check if entry need to be filtered out based on configuration*/
            if(((cursor->currentNode->entry->d_type == DT_DIR) ||
                (cursor->currentNode->entry->d_type == DT_REG) ||
                (cursor->currentNode->entry->d_type == DT_UNKNOWN)) &&
                FileSystemFilterEntry(p_vt->context,cursor->currentNode->entry->d_name))
            {
                /* Skip the entry since it matches the filter for file skipping*/
                cursor->fileType = FILE_TYPE_FAILED;
            }
            /*entry is a DIRECTORY*/
            else if(cursor->currentNode->entry->d_type == DT_DIR)
            {
                /*go down the sub folder if allowed*/
                if(FileSystemMoveDown(cursor))
                {
                    //continue;
                }
                else
                {
                    cursor->fileType = FILE_TYPE_DIRECTORY;
                    cursor->fileFormat = FFT_UNKNOWN;
                }
            }
            /*entry is a file and find its type(like audio,video or playlist) and format (like mp3,avi,pls)*/
            else if((cursor->currentNode->entry->d_type == DT_REG) ||
                    (cursor->currentNode->entry->d_type == DT_UNKNOWN))
            {
                /* create the complete filename */
                FastUTF8::tStringBuffer UTF8Path[1024];
                strcpy((char *)UTF8Path, cursor->currentNode->path);
                if(NULL == FastUTF8::EndsWithNC(UTF8Path,(const FastUTF8::tString)"/")) //lint -e1773
                {
                    strcat((char *)UTF8Path, (const char *)"/");
                }
                strcat((char *)UTF8Path, cursor->currentNode->entry->d_name);

                /* get the file type and its format only for regular files or incase dirent is not provding the type*/
                cursor->fileType = FileSystemCursorGetFileType(OUT cursor->fileFormat,IN (char *)UTF8Path, IN p_vt->context);

                /*if entry is a directory check possibility to brwose sub levels*/
                if((cursor->fileType == FILE_TYPE_DIRECTORY) && FileSystemMoveDown(cursor))
                {
                    cursor->fileType = FILE_TYPE_FAILED;
                    //continue;
                }
            }
        }
        else
        {
            FileSystemCursorMoveUpDir(cursor);
        }
        FSCacheUnlock();
    }

    /* Increment the current row count. */
    cursor->objectCount_rowID += 1;

    return SQLITE_OK;
}

/**
 * point the cursor to next entry in the playlist
 */
static int FileSystemCursorNextPlaylistEntry(tVTFileSystemCursor *cursor)
{
    ENTRY;
    tVTFileSystem *p_vt = (tVTFileSystem*)((sqlite3_vtab_cursor*)cursor)->pVtab; //lint !e826

    if ((unsigned int)cursor->objectCount_rowID == cursor->currentNode->playListStrings->size())
    {
        cursor->endOfDataset = 1; // mark end of dataset
        return SQLITE_OK;
    }

/*-----------------------------------------------------------------------------------------------------------------------*/
    FastUTF8::tStringBuffer absolutePath[1024];

    /*prepare absolute path for the play list entry*/
    char* playlistEntry = g_uri_unescape_string  ((*cursor->currentNode->playListStrings)[cursor->objectCount_rowID].c_str(),NULL);
    strcpy((char *)absolutePath, cursor->mountPoint);
    strcat((char *)absolutePath, playlistEntry);

    /* Take only the purefile name for the title*/
    FastUTF8::tString fullPath = NULL;
    FastUTF8::tString fileName = NULL;
    fullPath = (FastUTF8::tString)strdup(playlistEntry);
    FastUTF8::Split(OUT fileName, INOUT fullPath); //Split full path into filename and base path

    if(NULL != fileName)
    {
        fileName = (FastUTF8::tString)strdup((char*)fileName);
    }

    /*release path from previous entry if any (allocate by prev strdup(playlistEntry) )*/
    if(cursor->playListEntryName != NULL)
    {
        free((void*)cursor->playListEntryName); //lint -e1773
    }
    cursor->playListEntryName = (char*)fileName; //pure file name of the playlist entry

    if(NULL != fileName && NULL != fullPath)
    {
        //replace "/" that got lost during Split()
        strcat((char *)fullPath,(const char *)"/");
    }

    if(cursor->playListEntryPath != NULL)
    {
        free((void*)cursor->playListEntryPath); //lint -e1773
    }
    cursor->playListEntryPath = (char*)fullPath; //device internal base for the playlist entry



    /*check if playlist entry is valid/accessible*/
    bool isValidEntry = true;
    if (-1 == access ((const char*)absolutePath, F_OK))
    {
        /*set the files which are not accessible as UNKNOWN format and type Failed*/
        isValidEntry = false;
        cursor->fileType = FILE_TYPE_FAILED;
        cursor->fileFormat = FFT_UNKNOWN;

        //debug printf("playlist entry not accessible : %s\n",absolutePath);
        ETG_TRACE_ERR(("FileSystemCursorNextPlaylistEntry: entry not accessible->%s",(char*)absolutePath));
    }

    if(isValidEntry || p_vt->context->mShowAllPlaylistEntries)
    {
        /* get the file type and format */
        cursor->fileType = FileSystemCursorGetFileType(OUT cursor->fileFormat, IN (char *)absolutePath, IN p_vt->context);
    }

    /* Increment the current row count. */
    cursor->objectCount_rowID += 1;

    free(playlistEntry);

    return SQLITE_OK;
}

/**
 * Point the cursor to the next entry in the current node. (Node can be a playlist or a directory)
 */
int FileSystemCursorNext(sqlite3_vtab_cursor *cur)
{
    ENTRY;
    tVTFileSystemCursor *cursor = (tVTFileSystemCursor*)cur; //lint !e826

    if (cursor->currentNode->fileType == FILE_TYPE_PLAYLIST)
    {
        /* special treatment for play lists */
        return FileSystemCursorNextPlaylistEntry(cursor);
    }
    else if (cursor->currentNode->fileType == FILE_TYPE_DIRECTORY)
    {
        return FileSystemCursorNextDirEntry(cursor);
    }
    else
    {
        ETG_TRACE_ERR(("FileSystemCursorNext:Unknown file type:%d",cursor->currentNode->fileType));
        return -1;
    }
}

#endif /* FSCURSOR_CPP_ */
