/*
 * FSCache.cpp
 *
 *  Created on: 19.12.2014
 *      Author: Thömel
 */

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#ifndef FS_CACHE_CPP_
#define FS_CACHE_CPP_

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_mp.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_VT_FILE
#ifdef TARGET_BUILD
#include "trcGenProj/Header/FSCache.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_VT_FILE
#endif
#endif

#include "FSCache.h"
#include "Utils.h"
#include "TraceDefinitions.h"
#include "FunctionTracer.h"
#include "Lock.h"

#include <string>
#include <vector>
#include <map>
#include <string.h>
#include <string>
#include <fstream>
#include <stdlib.h>
#include <errno.h>

/*
 * cached directory entry
 */
typedef struct {
    dirent dirEntry;
} tDirScanCacheLine;

/*
 * list of (partly) cached directories
 */
typedef struct {
    int age;                    // age of scan context
    int used;                   // is this entry free for re-use?
    int ref;                    // how often is this context referenced by a current open statement
    int fullyCached;            // =0: directory not fully cached
    char *dirName;              // name of directory
    long lastScannedFile;       // number of last scanned file if not fully cached, to be able to continue readdir from there (0: before first file)
    long numberOfFiles;         // number of files total in directory
    long currentDirPosition;    // last read position from telldir
    tDirScanCacheLine *cache;   // pointer to directory item cache for this directory
} tDirScanContext;

typedef struct {
    int scanContextIndex;       // index into original directory scan context
    DIR *dirp;                  // context of low level opendir if directory not fully cached
    long currentReadPosition;   // number of file to be read next (=0: before first file)
} tOpenDirContext;

#define FORCE_LIMIT 0           // to test the limit function, set this bigger then 0
#define DEBUG_ON 0              // switch on file debugging to /tmp/FSCache.log

/*
 * globals:
 */
Lock gFSCacheMutex; /**< lock access to FSCache */
int FSCacheLock()
{
    return gFSCacheMutex.lock();
}
void FSCacheUnlock()
{
    gFSCacheMutex.unlock();
}

tDirScanContext *DirScanContext = NULL;
int noOfDirScanContexts = 0;
#if FORCE_LIMIT
int limitOfDirScanContexts = FORCE_LIMIT;
#else
int limitOfDirScanContexts = 10000;
#endif
int ageDirScanContexts = 0;
FILE *logFile = NULL;

/*
 * functions:
 */
DIR *FSCacheOpendir(const char *name)
{
    ENTRY;
    Locker locker(&gFSCacheMutex);
#if DEBUG_ON // debug
    if (!logFile) {
        logFile = fopen("/tmp/FSCache.log", "w");
    }
#endif

    /* is no cache created? */
    if (!DirScanContext) {

        /* create initial cache for this directory */
        DirScanContext = (tDirScanContext *)calloc(1, sizeof(tDirScanContext));
        if (!DirScanContext) return NULL;
        noOfDirScanContexts = 1;
        gNumberOfVTCacheElements++;
    }

    /* check if the requested directory is already cached */
    int entryFound = -1;
    for(int i=0; i<noOfDirScanContexts; i++) {

        if (DirScanContext[i].used && !strcmp(DirScanContext[i].dirName, name)) {

            /* found the entry */
            entryFound = i;

            break;
        } else if (!DirScanContext[i].used && entryFound == -1) {

            /* found first unused entry */
            entryFound = i;
        }
    }

    /* no free entry and no entry found? */
    if (entryFound == -1) {

        /* create a new entry */
        noOfDirScanContexts++;
        DirScanContext = (tDirScanContext *)realloc(DirScanContext, noOfDirScanContexts * sizeof(tDirScanContext));
        if (!DirScanContext) return NULL;
        entryFound = noOfDirScanContexts - 1;
        gNumberOfVTCacheElements++;

        /* zero the new entry */
        memset(&(DirScanContext[entryFound]), 0, sizeof(tDirScanContext));

        if (logFile) {
            fprintf(logFile, "FSCacheOpendir: create context for dirName:%s\n", name);
            fflush(logFile);
        }
    }

    /* if it is not used yet, set some values */
    if (!DirScanContext[entryFound].used) {
        DirScanContext[entryFound].dirName = strdup(name);
        DirScanContext[entryFound].used = 1;
    }

    /* reset the age of the context */
    ageDirScanContexts++;
    DirScanContext[entryFound].age = ageDirScanContexts;

    /* create an open-context */
    tOpenDirContext *openContext = (tOpenDirContext *)calloc(1, sizeof(tOpenDirContext));
    if (!openContext) {
        DirScanContext[entryFound].used = 0;
        free(DirScanContext[entryFound].dirName);
        return NULL;
    }

    /* set reference to scan context */
    openContext->scanContextIndex = entryFound;
    tDirScanContext *dirScanContext = &(DirScanContext[entryFound]);
    dirScanContext->ref++;

    /* check if the directory scan was NOT completed */
    if (!dirScanContext->fullyCached) {

        /* open the real directory for (continuing) scan */
        errno = 0;
        openContext->dirp = opendir(dirScanContext->dirName);

        /* opendir failed, return NULL and mark this entry as not used */
        if (!openContext->dirp) {

            if (logFile) {
                fprintf(logFile, "errno:%d=%s\n", errno, strerror(errno));
                fflush(logFile);
            }

            dirScanContext->used = 0;
            free(dirScanContext->dirName);
            free(openContext);
            return NULL;
        }

        /* seek to the readdir position for next read */
        seekdir(openContext->dirp, dirScanContext->currentDirPosition);

        if (logFile) {
            fprintf(logFile, "%p: FSCacheOpendir: %p=opendir(%s), seekdir(%ld)\n", openContext,
                    openContext->dirp,
                    dirScanContext->dirName,
                    dirScanContext->currentDirPosition);
            fflush(logFile);
        }
    }

    /* no readdir call done yet */
    openContext->currentReadPosition = 0;

    if (logFile) {
        fprintf(logFile, "%p: FSCacheOpendir(%s)\n", openContext, name);
        fflush(logFile);
    }

    /* return the index into the directory contexts (NOT the opendir handle!) */
    return (DIR *)openContext;
}

struct dirent *FSCacheReaddir(DIR *dirp)
{
    ENTRY;
    //Locker locker(&gFSCacheMutex); //Locked from calling function, see FSCacheLock
    tOpenDirContext *openContext = (tOpenDirContext *)dirp;
    tDirScanContext *dirScanContext = &(DirScanContext[openContext->scanContextIndex]);

    /* is directory fully cached? */
    if (dirScanContext->fullyCached) {

        /* end of directory reached? */
        if (openContext->currentReadPosition == dirScanContext->numberOfFiles) {

            if (logFile) {
                fprintf(logFile, "%p: End of Directory\n", openContext);
                fflush(logFile);
            }

            return NULL;
        }
    }

    /* possible to read from cache? */
    if (openContext->currentReadPosition < dirScanContext->lastScannedFile) {

        /* switch to next line */
        openContext->currentReadPosition++;

        if (logFile) {
            fprintf(logFile, "%p: hit(%s)\n", openContext, dirScanContext->cache[openContext->currentReadPosition-1].dirEntry.d_name);
            fflush(logFile);
        }
        gCacheHits++;

        /* but return the line before */
        return &(dirScanContext->cache[openContext->currentReadPosition-1].dirEntry);
    }

    /* get next directory entry for cache and for returning it */
    struct dirent *dirEntry = readdir(openContext->dirp);

    /* is the end of the directory reached? */
    if (!dirEntry) {
        dirScanContext->fullyCached = 1;
        openContext->currentReadPosition = dirScanContext->numberOfFiles;

        if (logFile) {
            fprintf(logFile, "%p: End of Directory\n", openContext);
            fflush(logFile);
        }

        return NULL;
    }

    /* add a directory entry cache line */
    dirScanContext->cache = (tDirScanCacheLine *)realloc(dirScanContext->cache, (dirScanContext->lastScannedFile+1) * sizeof(tDirScanCacheLine));
    if (!dirScanContext->cache) {
        return NULL;
    }
    gNumberOfVTCacheElements++;

    /* simplifier */
    tDirScanCacheLine *cacheLine = &(dirScanContext->cache[dirScanContext->lastScannedFile]);

    /* now one more cache line is inserted */
    dirScanContext->lastScannedFile++;

    /* get the current directory position to do a seekdir on a pafrallel open/reopen */
    dirScanContext->currentDirPosition = telldir(openContext->dirp);

    /* zero the new entry */
    memset(cacheLine, 0, sizeof(tDirScanCacheLine));

    /*
     * not the end of the directory:
     *
     * copy the result into the cache line
     */
    memcpy(&(cacheLine->dirEntry), dirEntry, sizeof(struct dirent));

    /* care about numbers */
    openContext->currentReadPosition++;
    dirScanContext->numberOfFiles++;

    if (logFile) {
        fprintf(logFile, "%p: miss(%s)\n", openContext, cacheLine->dirEntry.d_name);
        fflush(logFile);
    }
    gCacheMisses++;

    return &(cacheLine->dirEntry);
}

int FSCacheClosedir(DIR *dirp)
{
    ENTRY;
    Locker locker(&gFSCacheMutex);
    tOpenDirContext *openContext = (tOpenDirContext *)dirp;
    tDirScanContext *dirScanContext = &(DirScanContext[openContext->scanContextIndex]);

    /* is a directory open? */
    if (openContext->dirp) {
        if (logFile) {
            fprintf(logFile, "%p: FSCacheClosedir: closedir(%p)\n", openContext, openContext->dirp);
            fflush(logFile);
        }
        closedir(openContext->dirp);
        openContext->dirp = NULL;
    }

    /* free the open context */
    free(openContext);

    /* scan context not referenced anymore */
    dirScanContext->ref--;

    /* check if limit reached and clean some old caches up */
    if (gNumberOfVTCacheElements > limitOfDirScanContexts) {

        if (logFile) {
            fprintf(logFile, "NULL: FSCacheClosedir: limit (%d>%d) reached: cleanup some old contexts\n",
                    /*openContext,*/ gNumberOfVTCacheElements, limitOfDirScanContexts); //CID 18218 (#1 of 1): Use after free (USE_AFTER_FREE)
            fflush(logFile);
        }

        /* set the number of entries to be deleted at least */
        int entriesToBeDeletedAtLeast = limitOfDirScanContexts/2;

        /* delete loop */
        while(gNumberOfVTCacheElements > (limitOfDirScanContexts - entriesToBeDeletedAtLeast)) {

            /* search the oldest context */
            int oldestContext = -1;
            int oldestAgeFound = 0;
            for(int i=0; i<noOfDirScanContexts; i++) {

                /* context entry used and not referenced by an open statement? */
                if (DirScanContext[i].used && !DirScanContext[i].ref) {

                    /* is its age older than the ones before? */
                    if ((ageDirScanContexts - DirScanContext[i].age) > oldestAgeFound) {

                        /* remember this */
                        oldestAgeFound = (DirScanContext[i].age - ageDirScanContexts);
                        oldestContext = i;
                    }
                }
            }

            /* found NO context? */
            if (oldestContext == -1) {
                break; // cleanup not possible
            }

            /* cleanup the oldest context now */
            char *dirName = strdup(DirScanContext[oldestContext].dirName);
            if (dirName) {
                FSCacheDelete(dirName);
                free(dirName);
            } else {
                break;
            }

        } // delete loop
    }

    return 0;
}

void FSCacheReset(void)
{
    ENTRY;
    Locker locker(&gFSCacheMutex);
    if (logFile) {
        fprintf(logFile, "FSCacheReset\n");
        fflush(logFile);
    }
    gNumberOfVTCacheElements = 0;
}

bool FSCacheDelete(const char *path)
{
    ENTRY;
    Locker locker(&gFSCacheMutex);
    if (logFile) {
        fprintf(logFile, "FSCacheDelete(%s)\n", path);
        fflush(logFile);
    }

    /* look for all scan contexts that caches the desired directory and its sub directories and delete the contexts */
    for(int i=0; i<noOfDirScanContexts; i++) {

        /* scan context used and the name contains the searched string? */
        if (DirScanContext[i].used && strstr(DirScanContext[i].dirName, path)) {

            if (logFile) {
                fprintf(logFile, "FSCacheDelete: delete context of dirName:%s\n", DirScanContext[i].dirName);
                fflush(logFile);
            }

            /* free the cache */
            free(DirScanContext[i].cache);
            gNumberOfVTCacheElements -= DirScanContext[i].numberOfFiles;

            /* free the dirName */
            free(DirScanContext[i].dirName);

            /* set context to zero */
            memset(&(DirScanContext[i]), 0, sizeof(tDirScanContext));
            gNumberOfVTCacheElements--;
        }
    }

    return true;
}

void FSCacheSetLimit(int limit)
{
    ENTRY;
    Locker locker(&gFSCacheMutex);
#if !FORCE_LIMIT // just for test
    limitOfDirScanContexts = limit;
#endif

#if DEBUG_ON // debug
    if (!logFile) {
        logFile = fopen("/tmp/FSCache.log", "w");
    }
#endif

    if (logFile) {
        fprintf(logFile, "FSCacheSetLimit(%d)\n", limitOfDirScanContexts);
        fflush(logFile);
    }
}

#endif /* FS_CACHE_CPP_ */
