/*
 * MyMediaUpdateManager.cpp
 *
 *  Created on: Jun 13, 2013
 *      Author: Thömel
 */

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_mp.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_DB_MANAGER
#ifdef TARGET_BUILD
#include "trcGenProj/Header/MyMediaUpdateManager.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_DB_MANAGER
#endif
#endif

#include "DBManager.h"
#include "FunctionTracer.h"
#include "ThreadFactory.h"
#include "LocalSPM.h"
#include "DataProvider.h"
#include "Database.h"
#include <TimeTrace.h>
#include "ElapsedTimer.h"
#include "MyMediaUpdateManager.h"

#define MMUMGR_POLLING_INTERVALL_US 50000

/*lint -save -e801 */

void MyMediaUpdateManager::Do(int functionID, void *ptr)
{
	/* task scheduler */
	if (functionID == TSK_SCHEDULE) {

		ETG_TRACE_USR2(("MMUpdt: start scheduler"));
		LocalSPM::GetThreadFactory().SetName("TSK_SCHEDULE");

		/* focus: scheduler for all device dependent tasks, starting the first in vector */

		/* endless loop */
		while(1) {

			/* wait for a new task */
			//ETG_TRACE_USR2(("MMUpdt: wait for new tasks"));
			newTask.lock();
			if (stop) {
				ETG_TRACE_USR2(("MMUpdt: stop scheduler"));
				newTask.unlock();
				return;
			}
			//ETG_TRACE_USR2(("MMUpdt: check for new tasks"));

			tasksLock.lock();

			/* scan the tasks vector for tasks */
			for(unsigned int i=0; i<tasks.size(); i++) {

				/* check if device already started */
				unsigned int j;
				for(j=0; j<startedDevices.size(); j++) {
					if (startedDevices[j] == tasks[i]->deviceID) {
						break;
					}
				}
				if (j < startedDevices.size()) {
					ETG_TRACE_USR2(("MMUpdt: device busy: task=%d, devicdID=%d", tasks[i]->task, tasks[i]->deviceID));
					continue; // next entry in task list
				}

				/* get the current task */
				tTask *task = tasks[i];

				/* remember this device as started */
				startedDevices.push_back(task->deviceID);

				/* start the subtask */
				ETG_TRACE_USR2(("MMUpdt: launching: task=%d, devicdID=%d", task->task, task->deviceID));
				taskID = LocalSPM::GetThreadFactory().Do(this, task->task, (void *)task);
			}

			tasksLock.unlock();
		}

	/* all subtasks */
	} else {

	    LocalSPM::GetThreadFactory().SetName("MMUpdt");

		/* focus: only one task for one device */

		tTask *task = (tTask *)ptr;
		tResult res = MP_NO_ERROR;

		ETG_TRACE_USR2(("MMUpdt: starting: task=%d, devicdID=%d", task->task, task->deviceID));

        /* reset result in last-result table */
		tasksLock.lock();
        lastResults[task->deviceID] = MP_ERR_DB_UPD_MYMEDIA_DB_RUNNING;
        tasksLock.unlock();

		/* do the task */
		switch(task->task) {
		case TSK_ADD_DEVICE:
			res = AddDeviceThread(task);
			break;
		case TSK_REMOVE_DEVICE:
			res = RemoveDeviceThread(task);
			break;
		case TSK_SPEED_UP:
			res = SpeedupThread(task);
			break;
		default:
			break;
		}

		tasksLock.lock();

	    /* store result in last-result table */
		lastResults[task->deviceID] = res;

		tTask debugTask = *task; // only for printing

		/* remove this device from started devices list */
		for(unsigned int j=0; j<startedDevices.size(); j++) {
			if (startedDevices[j] == task->deviceID) {
				startedDevices.erase(startedDevices.begin() + j);
				ETG_TRACE_USR2(("MMUpdt: removed device from startedDevices: devicdID=%d", task->deviceID));
				break;
			}
		}

		/* remove the task from the tasks list */
		for(unsigned int i=0; i<tasks.size(); i++) {
			if (tasks[i] == task) {
				ETG_TRACE_USR2(("MMUpdt: removed task from tasks: task=%d, deviceID=%d", task->task, task->deviceID));
				tasks.erase(tasks.begin() + i);
				delete task;
				break;
			}
		}

		tasksLock.unlock();

		ETG_TRACE_USR2(("MMUpdt: finished: task=%d, devicdID=%d", debugTask.task, debugTask.deviceID));

		/* restart scheduler */
		newTask.unlock();
	}
}

void MyMediaUpdateManager::Init()
{
    tasksLock.lock();
    lastResults.clear();
    tasksLock.unlock();

	newTask.setNotReantrant();
	newTask.lock();

	/* start the scheduler */
	stop = 0;
	taskID = LocalSPM::GetThreadFactory().Do(this, TSK_SCHEDULE, NULL);

	/* manager is active now */
	deactive = 0;
}

void MyMediaUpdateManager::Stop()
{
	/* manager is deactive now */
	deactive = 1;

	/* stop all ongoing tasks */
	AbortAllTasks();

	/* stop the thread */
	stop = 1;
	newTask.unlock();
}

tResult MyMediaUpdateManager::AddDeviceThread(void *ptr) const
{
	ENTRY
	tResult res = MP_NO_ERROR;
	Query query;
	tTask *task = (tTask *)ptr;
	TimeTrace ticks("AddDeviceThread");

	/* do a complete update */
	if (task->deviceID == MY_MEDIA) {

		/* remove the complete my media contents */
		res = task->db->DeleteMyMediaDatabase(IN task->db->Handle());
		if (res) {
		    goto end; // lint !e801
		}

		/* setup a select for all connected devices */
		res = query.Select(LTY_DEVICES, MY_MEDIA, IN "ii",IN 0, IN CS_CONNECTED);
		if (res) {
		    goto end; //lint !e801
		}

		/* loop over all connected devices */
		while(1) {

			tDeviceID deviceID;

			/* get the device id */
			res = query.Get(IN "i", OUT &deviceID);
			if (res) break;

			ETG_TRACE_USR1(("UpdateMyMediaDatabase(%d)", deviceID));

			/* check the stop flag */
			if (task->stopFlag) {
			    res = MP_ERR_DB_UPD_MYMEDIA_DB_STOPPED;
			    break;
			}

			/* add this device to my media */
			res = task->db->UpdateMyMediaDatabase(IN &task->stopFlag, IN task->db->Handle(), IN deviceID);
			if (res) break;
		}

		/* end the query */
		query.End();

		if (res == MP_ERR_DB_END_OF_LIST) {
			res = MP_NO_ERROR;
		}

	/* do a update only for one device */
	} else {

		ETG_TRACE_USR1(("UpdateMyMediaDatabase(%d)", task->deviceID));

		/* check the stop flag */
		if (task->stopFlag) {
		    res = MP_ERR_DB_UPD_MYMEDIA_DB_STOPPED;
		    goto end; //lint !e801
		}

		/* add this device to my media */
		res = task->db->UpdateMyMediaDatabase(IN &task->stopFlag, IN task->db->Handle(), IN task->deviceID);
	}

	/* update all my media lists in the hmi */
	LocalSPM::GetListControl().CheckLists(IN MY_MEDIA);

end:
	ETG_TRACE_USR1(("UpdateMyMediaDatabase(%d): End, stopFlag=%d", task->deviceID, task->stopFlag));

	if (res) {
		ETG_TRACE_ERR(("error=%d/%s", res, errorString(res)));
	}

	ticks.elapsed();
	return res;
}

tResult MyMediaUpdateManager::RemoveDeviceThread(void *ptr) const
{
	ENTRY
	tResult res = MP_NO_ERROR;
	tTask *task = (tTask *)ptr;

	TimeTrace	 	 ticks("RemoveDeviceThread");

	res = task->db->RemoveDeviceFromMyMedia(IN &task->stopFlag, IN task->db->Handle(), IN task->deviceID);

	/* update all my media lists in the hmi */
	LocalSPM::GetListControl().CheckLists(IN MY_MEDIA);

    ticks.elapsed();
    return res;
}

tResult MyMediaUpdateManager::SpeedupThread(void *ptr) const
{
	ENTRY
	tResult res = MP_NO_ERROR;
	tTask *task = (tTask *)ptr;
	TimeTrace ticks("SpeedupThread");

	/* speedup all device dependent db functions (the my media is done by update my media thread) */
	res = task->db->SpeedupDatabase(IN task->db->Handle(), IN !MY_MEDIA);
	if (res) {
		ETG_TRACE_ERR(("error=%d/%s", res, errorString(res)));
	}

	ticks.elapsed();
	return res;
}

tResult MyMediaUpdateManager::AddDevice(Database *db, const tDeviceID deviceID)
{
	ENTRY
	tResult res = MP_NO_ERROR;

	if (deactive) return -1;

	/* abort the current running task for that device */
	res = AbortCurrentTask(IN deviceID);
	if (res) return res;

	tTask *task = new tTask(); //lint -e429 task will be freed after Do of subtask
	if (!task) return -1;

	task->task = TSK_ADD_DEVICE;
	task->db = db;
	task->deviceID = deviceID;
	task->stopFlag = 0;

	tasksLock.lock();
	if(!deactive)
	{
	    tasks.push_back(task);
	}
	else
	{
	    delete task;
	}
	tasksLock.unlock();

	newTask.unlock();

	return MP_NO_ERROR;
}

tResult MyMediaUpdateManager::RemoveDevice(Database *db, const tDeviceID deviceID)
{
	ENTRY
	tResult res = MP_NO_ERROR;

	if (deactive) return -1;

	/* abort the current running task for that device */
	res = AbortCurrentTask(IN deviceID);
	if (res) return res;

	tTask *task = new tTask(); //lint -e429 task will be freed after Do of subtask
	if (!task) return -1;

	task->task = TSK_REMOVE_DEVICE;
	task->db = db;
	task->deviceID = deviceID;
	task->stopFlag = 0;

	tasksLock.lock();
	if(!deactive)
	{
	    tasks.push_back(task);
	}
	else
	{
	    delete task;
	}
	tasksLock.unlock();

	newTask.unlock();

	return MP_NO_ERROR;
}

tResult MyMediaUpdateManager::Speedup(Database *db)
{
	ENTRY

	if (deactive) return -1;

	tTask *task = new tTask(); //lint -e429 task will be freed after Do of subtask
	if (!task) return -1;

	task->task = TSK_SPEED_UP;
	task->db = db;
	task->deviceID = MY_MEDIA;
	task->stopFlag = 0;

	tasksLock.lock();
	if(!deactive)
	{
	    tasks.push_back(task);
	}
	else
	{
	    delete task;
	}
	tasksLock.unlock();

	newTask.unlock();

	return MP_NO_ERROR;
}

tResult MyMediaUpdateManager::SlowDownCurrentTask(Database *db) const
{
	ENTRY
	db->SlowDownUpdateMyMedia();
	return MP_NO_ERROR;
}

tResult MyMediaUpdateManager::WaitForFinishedInternal(tTask *task)
{
	ENTRY
	//tResult res = MP_NO_ERROR;

	while(1) {

		/* search for the task pointer */
		int found = 0;
		tasksLock.lock();
		for(unsigned int i=0; i<tasks.size(); i++) {
			if (tasks[i] == task) {
				found = 1;
				break;
			}
		}
		tasksLock.unlock();

		/* not found? task end */
		if (!found) return MP_NO_ERROR;

		usleep(MMUMGR_POLLING_INTERVALL_US);
	}

	return MP_NO_ERROR;
}

tResult MyMediaUpdateManager::GetLastResult(const tDeviceID deviceID)
{
    ENTRY

    tResult res = MP_NO_ERROR;
    tasksLock.lock();

    tLastResultsIter lastResultsIter;
    lastResultsIter = lastResults.find(deviceID);

    if (lastResultsIter != lastResults.end()) {
        res = lastResultsIter->second;
    }

    tasksLock.unlock();

    return res;
}

tResult MyMediaUpdateManager::WaitForFinished(const tDeviceID deviceID)
{
	ENTRY
	tResult res = MP_NO_ERROR;
	tTask *task;

	/* poll for task */
	while(1) {

		/* find the first entry for device in vector */
		tasksLock.lock();
		res = GetCurrentTask(OUT &task, IN deviceID);
		tasksLock.unlock();
		if (res) {
			return GetLastResult(deviceID);
		}

		usleep(MMUMGR_POLLING_INTERVALL_US);
	}
}

tResult MyMediaUpdateManager::WaitForFinished()
{
	ENTRY

	/* poll for any task */
	while(1) {

		/* get the current tasks size */
		tasksLock.lock();
		int tasksSize = tasks.size();
		tasksLock.unlock();

		if (!tasksSize) {
		    return MP_NO_ERROR;
		}

		usleep(MMUMGR_POLLING_INTERVALL_US);
	}
}

tResult MyMediaUpdateManager::AbortCurrentTask(tDeviceID deviceID)
{
	ENTRY
	tResult res = MP_NO_ERROR;
	tTask *task;

	ETG_TRACE_USR2(("MMUpdt: request abort: deviceID=%d", deviceID));

	/* find the first entry for device in vector */
	tasksLock.lock();
	res = GetCurrentTask(OUT &task, IN deviceID);
	if (res) {
		tasksLock.unlock();
		ETG_TRACE_USR2(("MMUpdt: already aborted: deviceID=%d", deviceID));
		return MP_NO_ERROR; // no current task
	}

	/* signal the stop wanted */
	task->stopFlag = 1;

	tTask debugTask = *task; // only for printing

	tasksLock.unlock();

	/* wait for end of this task */
	WaitForFinishedInternal(IN task);
	ETG_TRACE_USR2(("MMUpdt: abort successful: task=%d, deviceID=%d", debugTask.task, debugTask.deviceID));

	return MP_NO_ERROR;
}

tResult MyMediaUpdateManager::GetCurrentTask(tTask **task, const tDeviceID deviceID)
{
	ENTRY

	for(unsigned int i=0; i<tasks.size(); i++) {
		if (tasks[i]->deviceID == deviceID) {
			*task = tasks[i];
			return MP_NO_ERROR;
		}
	}

	return -1;
}

tResult MyMediaUpdateManager::AbortAllTasks()
{
	ENTRY
	//tTask *task;

	ETG_TRACE_USR2(("MMUpdt: try to abort all tasks"));

	/* stop all ongoing tasks */
	tasksLock.lock();

	/* set all stop flags */
	for(unsigned int i=0; i<tasks.size(); i++) {
		tasks[i]->stopFlag = 1;
	}

	/* wait for an empty tasks vector / all tasks gone (with retry timeout) */
	//GMMY17-15631
	//long retry = (3/*sec*/ * 1000000L/*in usec*/) / MMUMGR_POLLING_INTERVALL_US;
	while(tasks.size()/* && retry-- */) {
		tasksLock.unlock();
		usleep(MMUMGR_POLLING_INTERVALL_US);
		tasksLock.lock();
	}

	tasksLock.unlock();

	return MP_NO_ERROR;
}
