#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_mp.h"

#define TSK_FLUSH_MEDIA_OBJECTS 2
#define TSK_CHECK_VALID_PARTITION 3
#define FILTER_FILE_LIST_VIA_CTY 0 // Attention! same switch as in Queries.h
#define NO_FOLDER_HIERARCHY 0 // Attention! same switch as in Queries.h

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_DB_MANAGER
#ifdef TARGET_BUILD
#include "trcGenProj/Header/DBManager.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_DB_MANAGER
#endif
#endif

#include "DBManager.h"
#include "FunctionTracer.h"
#include "ThreadFactory.h"
#include "LocalSPM.h"
#include "DataProvider.h"
#include "Database.h"
#include "SearchKeyboard.h"
#include <TimeTrace.h>
#include "ElapsedTimer.h"
#include "DBTriggerManager.h"
#include "VarTrace.h"

#include <stdarg.h>
#include <cstring>
#include <sys/stat.h>
#include <fstream>
#include <time.h>
#include <sys/time.h>
#include <ctype.h>
#include <glib.h>

using std::__basic_file;

//#define ENTRY           FunctionTracer  _t(ETG_DEFAULT_TRACE_CLASS,__PRETTY_FUNCTION__);

/*lint -save -e774 */

// Methods implementation
void DBManager::Create()
{
    ENTRY

    mFlushLock.setNotReantrant();

    mDB = NULL;

    return ILocalSPM::Create();
}

tResult DBManager::Init(tInitReason reason)
{
    ENTRY

    /* init the my media update manager */
    mMMUpdt.Init();

    /* init the DB (if needed) */
    if(!mDB) {
        mDB = new Database();

        /* create internal flash devices */
        mDB->CreateInternalDevices();
    }

    /* set members */
    mObjectCountFlushBorder = 1;
    mDBStoreMediaObjectMaxTime_ms = LocalSPM::GetDataProvider().DBStoreMediaObjectMaxTimeMS();

    mActiveLanguage = (tLanguageType)LocalSPM::GetDataProvider().DBDefaultLanguage();
    mActiveMediaContext = (tMediaContext)LocalSPM::GetDataProvider().DBDefaultMediaContext();

    /* remove all triggers from the data base from the run before if any !*/
    DBTriggerManager::GetInstance().TriggerCleanupDB();

    /* set all devices to disconnected */
    ResetDevices(reason);

    if(LocalSPM::GetDataProvider().FavoritesSupported()) /*RESET all Favorites to Non-available*/
    {
        /* set all favorites to non-available */
        ResetAllFavorites();
    }

    /* reset all live tags */
    ResetLiveTags();

    /* reset all virual devices */
    ResetVirtualDevices();

    /*Portnumber Map clear */
    mPortNumberInfo.clear();

    /*ValidPartition Map clear*/
    mValidPartitionInfo.clear();

    /*DeviceSysPath Map clear*/
    mDeviceSysPathInfo.clear();

    /* set BT MAC address in DataProvider initially */
    tBTMACAddress address;
    if( MP_NO_ERROR == GetBTMACAddress(OUT address) ) {
        LocalSPM::GetDataProvider().iPodControlBTMACAddress = string(address);
    }

    if(mDB)
    {
        /* create internal flash devices */
        mDB->CreateInternalDevices();
    }


    // Perform the parent class (ILocalSPM) specific initialization.
    return ILocalSPM::Init(reason);
}

tResult DBManager::Run()
{
    ENTRY
    return ILocalSPM::Run();
}

tResult DBManager::Stop()
{
    ENTRY

    /* cancel the timer */
    mTimerResetDevices.CancelTimer(IN mTimerIDResetDevices);

    /* de-init the my media update manager */
    mMMUpdt.Stop();

    /* wait until the last cache flush has done */
    WaitForFlushMediaObjectCacheEnd();

    return ILocalSPM::Stop();
}

tResult DBManager::Done()
{
    ENTRY

    /* shutdown the database progresses (may take up to 3 seconds) */
    if(mDB) {
        mDB->Shutdown(1);
    }

    // release all lists and close all database connections
    ReleaseAllDBLists();

    if(mDB) {
        // de init the data base (triggers will be removed, all cached connections will be closed, trigger manager will be detached)
        mDB->DeInit();

        // last call...
        delete mDB;
        mDB = NULL;
    }

    // Perform the parent class (ILocalSPM) specific cleanup.
    return ILocalSPM::Done();
}


bool DBManager::MemoryUsageTimerCallBack(timer_t timerID , void* instance ,const void *userData)
{
    ENTRY;
    //ETG_TRACE_ERRMEM(("DBManager::MemoryUsageTimerCallBack"));
    (void)timerID;
    (void)userData;
    if (instance != NULL)
    {
        DBManager *self = (DBManager*)instance;
        if(self)
        {
            //Timeout
            self->OnMemoryUsageTimeout();
        }
    }
    return MP_NO_ERROR;
}

void DBManager::OnMemoryUsageTimeout()
{
    if (mDB)
    {
        ofstream fileObj;
        fileObj.open ("/var/opt/bosch/dynamic/media/db/MemoryUsageLog.csv",std::ofstream::out | std::ofstream::app);

        static int check = 1;

        if(check){
            fileObj << "SQLITE_DBSTATUS_CACHE_USED" << ",," << "SQLITE_DBSTATUS_CACHE_HIT" << ",," << "SQLITE_DBSTATUS_CACHE_WRITE" << ",," << "SQLITE_DBSTATUS_STMT_USED"
                    << ",," << "AllocatedMemory: Used Memory" << ",," << "PrepareStatementCount: OPEN STATEMENT COUNT" << ",," << "Number Of Playable Objects In DB"
                    << ",," << "Memory usage(MB)" << "\n";
            check = 0;
        }

        fileObj << mDB->DatabaseStatus(mDB->Handle(), SQLITE_DBSTATUS_CACHE_USED) << ",,";
        fileObj << mDB->DatabaseStatus(mDB->Handle(), SQLITE_DBSTATUS_CACHE_HIT) << ",,";
        fileObj << mDB->DatabaseStatus(mDB->Handle(), SQLITE_DBSTATUS_CACHE_WRITE) << ",,";
        fileObj << mDB->DatabaseStatus(mDB->Handle(), SQLITE_DBSTATUS_STMT_USED) << ",,";
        fileObj << mDB->AllocatedMemory() << ",,";
        fileObj << mDB->PrepareStatementCount(mDB->Handle()) << ",,";

        tDeviceID deviceID;
        GetActiveDevice(deviceID);

        tNumberOfMediaObjects numberOfAudioObjectsInDB = 0;
        tNumberOfMediaObjects numberOfVideoObjectsInDB = 0;
        GetNumberOfPlayableObjectsInDB(OUT numberOfAudioObjectsInDB, OUT numberOfVideoObjectsInDB, IN deviceID);
        fileObj << numberOfAudioObjectsInDB << ",,";
        fileObj << numberOfVideoObjectsInDB << ",,";

        size_t memory_usage_MB = getCurrentRSS()/1024/1024;
        fileObj << memory_usage_MB << ",,";

        fileObj <<"\n";
        fileObj.close();

    }
}

int DBManager::DatabaseStatistics(void)
{
    long timeoutValue = 5000;
    m_MemoryUsageTimer.StartTimer(OUT m_MemoryUsageTimerID, IN timeoutValue, timeoutValue, &LocalSPM::GetDBManager(), &MemoryUsageTimerCallBack, NULL);
    return MP_NO_ERROR;
}


int DBManager::Statistics(OUT tStatistics stat)
{
    stat[0]=0;

    int usedEntries = 0;
    int size = 0;
    mDBListsLock.lock();
    size = mDBLists.size();
    for(unsigned int i=0; i<mDBLists.size(); i++) {
        if (mDBLists[i]->mListID != LIST_ID_NONE) usedEntries++;
    }
    mDBListsLock.unlock();

    /* get VT info */
    int cacheEntries = 0;
    int cacheHits = 0;
    int cacheMisses = 0;

    if(mDB) {
        mDB->GetNumberOfVTCacheElements(&cacheEntries, &cacheHits, &cacheMisses);
    }

    /* create some statistic information */
    snprintf(stat, sizeof(tStatistics), "VTCacheElem=%d(h:%d/m:%d) dbLists=%d/%d QueryCacheElem=%d",
            cacheEntries, cacheHits, cacheMisses, usedEntries, size, Query::GetQueryCacheCount());

    return 0;
}

tResult DBManager::MultiPartitionDeviceChanged(vector<tDeviceInfo> deviceInfos)
{
    ENTRY
    vector<tDeviceInfo> *deviceInfosCache;
    deviceInfosCache = new vector<tDeviceInfo>(deviceInfos.begin(), deviceInfos.end());
    //(*deviceInfosCache) = deviceInfos;
    LocalSPM::GetThreadFactory().Do(this, TSK_CHECK_VALID_PARTITION,deviceInfosCache);
    return 0;
}

void DBManager::Do(int functionID, void *ptr)
{
    switch(functionID) {
    case TSK_FLUSH_MEDIA_OBJECTS:
    {
        LocalSPM::GetThreadFactory().SetName("TSK_FLUSH_MEDIA_OBJECTS");
        vector<tMediaObject> *objectCache = (vector<tMediaObject> *)ptr;
        FlushMediaObjectCacheInternal(IN objectCache);
        delete objectCache;
        mFlushLock.unlock();
        break;
    }
    case TSK_CHECK_VALID_PARTITION:
    {
        LocalSPM::GetThreadFactory().SetName("TSK_CHECK_VALID_PARTITION");
        vector<tDeviceInfo> *deviceInfos = (static_cast<vector<tDeviceInfo>*>(ptr));
        FindLargestValidPartition(INOUT *deviceInfos);
        DeviceChanged(IN *deviceInfos);
        delete deviceInfos;
        break;
    }
    default:
    {
        ETG_TRACE_ERR(("DBManager::Do: No thread defined for functionID: %d", functionID));
        break;
    }}
}

tResult DBManager::FindLargestValidPartition(vector<tDeviceInfo>& deviceInfos)
{
    ENTRY
    tResult res;
    Query query;
    tDeviceSerialNumber serialNumber = {0};
    tSearchString searchString = {0};
    tBool bValid = false;
    tConnectionState dbConnectionState;
    vector<tDeviceInfo>::iterator itDeviceInfo;

    vector<tDeviceInfo>::iterator it = deviceInfos.end(); it--;  //Iterator points to Last element
    ETG_TRACE_USR3(("FindLargestValidPartition  Last Element: it->serialNumber =%s", it->serialNumber));
    strncat_r(OUT searchString, IN it->accessoryName2, IN sizeof(searchString));
    strncat_r(OUT searchString, IN "%", IN sizeof(searchString));

    res = query.Select(IN LTY_DEVICES_LATEST_VALID_PARTITION,IN MY_MEDIA, IN "it", IN 0, IN searchString);
    if(res) return res;

    res = query.Get("Ti", OUT serialNumber, IN sizeof(serialNumber), OUT &dbConnectionState);
    query.End();

    /* If serialNumber of multipartition device exist in DB */
    if(strlen(serialNumber)>0)
    {
        tAccessoryName2 tempAN2 = {0};
        itDeviceInfo = std::find_if(deviceInfos.begin(),deviceInfos.end(),tFindDeviceBySerialNumber(serialNumber,tempAN2));
        if(itDeviceInfo!=deviceInfos.end())
        {
            //serialNumber of multipartition device exist in DB and found in deviceInfos
            ETG_TRACE_USR3(("dbConnectionState =%d", dbConnectionState));
            ETG_TRACE_USR3(("Incoming connectionState =%d", itDeviceInfo->connectionState));
            ETG_TRACE_USR3(("Incoming serialNumber =%s", itDeviceInfo->serialNumber));

            if(dbConnectionState==CS_CONNECTED && itDeviceInfo->connectionState==CS_DISCONNECTED)
            {
                //For Device Disconnection
                bValid = true;
                mValidPartitionInfo[it->accessoryName2] = itDeviceInfo->serialNumber;
                ETG_TRACE_USR3(("Valid multi partition entry updated to MAP for disconnection: itDeviceInfo->serialNumber %s", itDeviceInfo->serialNumber));

                //Remove all invalid multipartition entries from deviceinfos. (i.e) only valid entry of accessoryName2 is retained in deviceInfos.
                deviceInfos.erase(std::remove_if(deviceInfos.begin(),deviceInfos.end(),tFindDeviceBySerialNumber(itDeviceInfo->serialNumber,it->accessoryName2,true)),deviceInfos.end());
            }
            else if((dbConnectionState==CS_CONNECTED && itDeviceInfo->connectionState==CS_ATTACHED) || (dbConnectionState==CS_DISCONNECTED && itDeviceInfo->connectionState==CS_DISCONNECTED))
            {
                //Device to be connected/disconnected is already in connection/disconnection state, so remove all multipartition entries from deviceInfos.
                ETG_TRACE_USR3(("device to be connected/disconnected is already in connection/disconnection state"));
                deviceInfos.erase(std::remove_if(deviceInfos.begin(),deviceInfos.end(),tFindDeviceBySerialNumber(itDeviceInfo->serialNumber,it->accessoryName2,false,true)),deviceInfos.end());

                return MP_NO_ERROR;
            }
        }
    }

    if(bValid==false)
    {
        itDeviceInfo=std::find_if(deviceInfos.begin(),deviceInfos.end(),[&it](const tDeviceInfo& deviceInfo)
        {
            return strcmp(deviceInfo.accessoryName2,it->accessoryName2)==0 && deviceInfo.connectionState==CS_DISCONNECTED;
        });
        if(itDeviceInfo!=deviceInfos.end())
        {
            //Incoming connection state is CS_DISCONNECTED for device and device doesn't exist in DB, so remove all multipartition entries from deviceInfos.
            ETG_TRACE_USR3(("incoming connection state is CS_DISCONNECTED for multipartition device and device doesn't exist in DB"));
            deviceInfos.erase(std::remove_if(deviceInfos.begin(),deviceInfos.end(),tFindDeviceBySerialNumber(itDeviceInfo->serialNumber,it->accessoryName2,false,true)),deviceInfos.end());

            return MP_NO_ERROR;
        }

        //For Device Connection
        int nCount = 0;
        for(tUInt nIndex = 0; nIndex < deviceInfos.size();nIndex++)
        {
            if(strncmp(deviceInfos[nIndex].accessoryName2,it->accessoryName2,sizeof(deviceInfos[nIndex].accessoryName2))==0)
            {
                nCount++;
                if (!LocalSPM::GetDataProvider().IsFileSystemSupported(IN deviceInfos[nIndex].fileSystemType)) /*check for unsupported file system*/
                {
                    bValid = false;
                    ETG_TRACE_USR3(("Unsupported multipartition file system"));
                }
                else
                {
                    bValid = LocalSPM::GetDataProvider().IsValidContentAvilable(deviceInfos[nIndex].mountPoint);
                }
                if(bValid)
                {
                    //Remove all invalid multipartition entries from deviceinfos. (i.e) only valid entry of accessoryName2 is retained in deviceInfos.
                    deviceInfos.erase(std::remove_if(deviceInfos.begin(),deviceInfos.end(),tFindDeviceBySerialNumber(deviceInfos[nIndex].serialNumber,deviceInfos[nIndex].accessoryName2,true)),deviceInfos.end());
                    mValidPartitionInfo[deviceInfos[nIndex].accessoryName2] = deviceInfos[nIndex].serialNumber;
                    ETG_TRACE_USR3(("Valid multi partition entry added to MAP for connection: deviceInfos[nIndex].serialNumber %s", deviceInfos[nIndex].serialNumber));
                    break;
                }
                if(nCount >= (tUInt)LocalSPM::GetDataProvider().SupportedPartionNumbers())
                {
                    break;
                }
            }
        }
    }

    /*If no valid partition is found choose the default partition by default*/
    if(!bValid)
    {
        it = deviceInfos.end(); it--;  //Iterator points to Last element
        mValidPartitionInfo[it->accessoryName2] = it->serialNumber;
        ETG_TRACE_USR3(("Valid multi partition entry added to MAP : it->serialNumber %s", it->serialNumber));
        deviceInfos.erase(std::remove_if(deviceInfos.begin(),deviceInfos.end(),tFindDeviceBySerialNumber(it->serialNumber,it->accessoryName2,true)),deviceInfos.end());
    }

    //DB update is required only during connection of existing device in case if serialNum from DB and serialNum of validpartition is not matching.
    if(strlen(serialNumber)>0 && strcmp(serialNumber,(--mValidPartitionInfo.end())->second.c_str()))
    {
        res=RemoveMultiPartitionDevice(serialNumber);
        if(res!=MP_NO_ERROR)
        {
            ETG_TRACE_ERR(("RemoveMultiPartitionDevice: Error=%d/%s", res, errorString(res)));
        }
    }

    return MP_NO_ERROR;
}

tResult DBManager::ScanDirForTest(const tDeviceID deviceID)
{
    ENTRY

    if (!mDB) return MP_ERR_DB_NOT_OPEN;

    return mDB->ScanDirForTest(IN deviceID);
}

tResult DBManager::ClearMediaPlayerData(void)
{
    ENTRY
    tResult res;

    /* wait for flush thread to be finished */
    mFlushLock.lock();

    /* clear cache */
    mMediaObjectCache.clear();

    if(mDB) {
        mDB->Shutdown(0);
    }

    res = ReleaseAllDBLists();
    if (res) return res;

    /* delete the own member */
    // de init the data base (triggers will be removed)
    if(mDB) {

        mDB->DeInit();

        // close the db
        delete mDB;
        mDB = NULL;
    }

    /* recreate the db */
    Database *DB = new Database();
    res = DB->RecreateDatabase();
    DB->DeInit();
    delete DB;

    /* reopen it */
    mDB = new Database();

    /* create internal flash devices */
    mDB->CreateInternalDevices();

    /* now flushing is allowed again */
    mFlushLock.unlock();

    return res;
}

tResult DBManager::CreateTestDatabase(const tBoolean createDevices, const tBoolean createSongs)
{
    ENTRY_INTERNAL

    if (!mDB) return MP_ERR_DB_NOT_OPEN;

    return mDB->CreateTestDatabase(IN createDevices, IN createSongs);
}

DBManager::DBList *DBManager::CreateDBList()
{
    ENTRY_INTERNAL
    static int _uniqueDBListID = 1;
    DBList *dbList = new DBList();
    if (!dbList) return NULL;

    /* fill out the context */
    dbList->mListID = _uniqueDBListID;
    _uniqueDBListID++;

    /* create the query */
    dbList->mQuery = new Query();
    if (!dbList->mQuery) {
        delete dbList;
        return NULL;
    }

    mDBListsLock.lock();
    tUInt i;

    /* look for an empty entry in the vector */
    for(i = 0; i<mDBLists.size(); i++) {
        if ((mDBLists[i]->mListID == LIST_ID_NONE) && (mDBLists[i]->mRefCounter < 1)) break;
    }

    // found one
    if (i < mDBLists.size()) {

        /* delete the old memory */
        if (mDBLists[i]->mQuery) delete mDBLists[i]->mQuery;
        delete mDBLists[i];

        /* replace entry with new one */
        mDBLists[i] = dbList;

    } else { // found no free entry, add a new one:

        /* add the new list to the vector */
        mDBLists.push_back(dbList);
    }
    mDBListsLock.unlock();

    return dbList;
}

/* Attention: Use macro on same level/focus like related variable DBList*
 * because decrease of ref counter is done after leaving this level/focus (PSARCCB-7043) */
#define TOKENPASTE(x, y) x ## y
#define TOKENPASTE2(x, y) TOKENPASTE(x, y)
#define GET_DBLIST_POINTER(listPtrName, dbListID) \
        AutoDeRefDBListPtr TOKENPASTE2(adrlp,__LINE__)(listPtrName = DBManager::LookupDBList(dbListID));

DBManager::DBList *DBManager::LookupDBList(const tListID dbListID)
{
    ENTRY_INTERNAL
    DBList *listPtr;

    if (dbListID == LIST_ID_NONE) return NULL;

    mDBListsLock.lock();

    /* search for the element */
    for(tUInt i=0; i<mDBLists.size(); i++) {
        if (mDBLists[i]->mListID == dbListID) {
            listPtr = mDBLists[i];
            listPtr->mRefCounter++;
            mDBListsLock.unlock();
            return listPtr;
        }
    }

    /* element not found */
    mDBListsLock.unlock();

    ETG_TRACE_ERR(("No Entry for dbListID %d found", dbListID));

    return NULL;
}

tResult DBManager::ReleaseDBList(const tListID dbListID)
{
    ENTRY
    VARTRACE(dbListID);
    tResult res = MP_ERR_DB_LIST_NOT_FOUND;

    if (dbListID == LIST_ID_NONE) return res;

    mDBListsLock.lock();
    for(tUInt i=0; i<mDBLists.size(); i++) {
        if (mDBLists[i]->mListID == dbListID) {

            /* mark list entry as not used */
            mDBLists[i]->mListID = LIST_ID_NONE;

            /* delete the memory only if reference counter zero */
            if (mDBLists[i]->mRefCounter < 1) {

                if (mDBLists[i]->mQuery) delete mDBLists[i]->mQuery;
                delete mDBLists[i];

                /* remove list from vector */
                mDBLists.erase(mDBLists.begin() + i);
            }

            res = MP_NO_ERROR; //element found
            break;
        }
    }
    mDBListsLock.unlock();

    return res;
}

tResult DBManager::ResetObjectCacheOfDBList(const tDeviceID deviceID)
{
    ENTRY_INTERNAL
    mDBListsLock.lock();
    for(unsigned int i=0; i<mDBLists.size(); i++) {
        if((mDBLists[i]->mDeviceID == deviceID) || (mDBLists[i]->mDeviceID == MY_MEDIA)) {
            mDBLists[i]->mCurrentRow = ROW_NUMBER_NONE;
        }
    }
    mDBListsLock.unlock();

    return MP_NO_ERROR;
}

tResult DBManager::ReleaseAllDBLists()
{
    ENTRY
    mDBListsLock.lock();
    for(unsigned int i=0; i<mDBLists.size(); i++) {

        if (mDBLists[i]->mQuery) delete mDBLists[i]->mQuery;
        delete mDBLists[i];
    }

    //mDBLists.clear();
    if( 0 < mDBLists.capacity() )
    {
        vector<DBList *>().swap(mDBLists); //used instead of clear to guarantee a memory reallocation
    }
    mDBListsLock.unlock();

    return MP_NO_ERROR;
}

tResult DBManager::IsVTIPODList(
        const tFilterTag1 filterTag1,
        const tFilterTag2 filterTag2,
        const tFilterTag3 filterTag3,
        const tFilterTag4 filterTag4,
        const tListType listType,
        const tDeviceInfo deviceInfo,
        const tBoolean lastPlayedList) const
{
    /* check VT iPod lists, IsVTIPODList should be enabled based on the MARKER bit,
    * it is NOT depending on IDS state when creating a last played list */

    if((filterTag1.tag & IPOD_MARKER_BIT) ||
           (filterTag2.tag & IPOD_MARKER_BIT) ||
           (filterTag3.tag & IPOD_MARKER_BIT) ||
       (filterTag4.tag & IPOD_MARKER_BIT)) {
        // A) check filter tags for IPOD_MARKER_BIT
        return true;
    } else if(IsAppleDevice(deviceInfo.deviceType)) {
        // B) check DTY apple
        if(listType == LTY_CURRENT_SELECTION) {
            // B1) check DTY apple, current selection always from VTIPOD
            return true;
        } else if(!LocalSPM::GetIPODControl().IsIAP2(deviceInfo.mountPoint)) {
            // B2) no VTIPOD for IAP2, apart from LTY_CURRENT_SELECTION
            if(deviceInfo.indexedState != IDS_COMPLETE && !lastPlayedList) {
                // B21) check DTY apple, indexing in progress, not last played list
            return true;
            } else if(listType == LTY_PLAYLIST_SONG || listType == LTY_PODCAST || listType == LTY_AUDIOBOOK) {  //SUZUKI-26056
                // B22) check DTY apple, playlist always fomr VTIPOD
            return true;
        }
        //NCG3D-56937: commented below condition to solve VR not playing issue with iAP1 devices. This fix changes the list creation & access(list creation from DB instead of device list for iAP1 devices ).
        /*else if(LocalSPM::GetDataProvider().IsPermanentStreamingActive(deviceInfo.deviceType)) {
                // B23) check DTY apple, permanent streaming configured
          return true;
        }*/
    }
    }
    return false;
}

tResult DBManager::IsVTList(const tListID listID)
{
    ENTRY
    tResult res = 0;
    DBList *dbList = NULL;

    /* get list pointer */
    //dbList = LookupDBList(IN listID);
    GET_DBLIST_POINTER(dbList, listID);
    if (!dbList) return res;

    if ((dbList->mIsVTIPODList) ||
        (LTY_FILELIST == dbList->mListType) ||
        (LTY_FILELIST_UNSORTED == dbList->mListType) ||
        (LTY_FILELIST_SORTED_FOR_INDEXING == dbList->mListType) ||
        (LTY_FILELIST_PLAYLIST == dbList->mListType) ||
        (LTY_FILELIST_MEDIAOBJECTS == dbList->mListType) ||
        (LTY_FILELIST_MEDIAOBJECTS_WITH_SUBFOLDERS == dbList->mListType) ||
        (LTY_MTP_FILELIST == dbList->mListType) ||
        (LTY_BLUETOOTH_FILELIST == dbList->mListType) ||
        (LTY_CD == dbList->mListType))
    {
        res = 1;
    }

    return res;
}

tResult DBManager::CreateList(tListID &dbListID, const tObjectID objectID,
        const tListType listType, const tPlayContinuation playContinuation, const tDeviceID deviceID)
{
    ENTRY
    tResult res;
    tFilterTag1 filterTag1;
    tFilterTag2 filterTag2;
    tFilterTag3 filterTag3;
    tFilterTag4 filterTag4;

    /* reset all tags */
    filterTag1.tag = 0;
    filterTag2.tag = 0;
    filterTag3.tag = 0;
    filterTag4.tag = 0;

    /* set the object id as a given tag */
    switch(listType) {
    case LTY_SONG:
    case LTY_EPISODE_OF_VIDEO:
        break;
    case LTY_GENRE_SONG:
        filterTag1.genre = objectID;
        break;
    case LTY_ARTIST_SONG:
        filterTag2.artist = objectID;
        break;
    case LTY_ALBUM_SONG:
    case LTY_COMPILATION_SONG:
        filterTag4.album = objectID;
        break;
    case LTY_COMPOSER_SONG:
        filterTag3.composer = objectID;
        break;
    case LTY_PODCAST_EPISODE:
        filterTag1.podcast = objectID;
        break;
    case LTY_VIDEO_EPISODE:
        filterTag1.video = objectID;
        break;
    case LTY_BOOKTITLE_CHAPTER:
        filterTag2.audiobook = objectID;
        break;
    case LTY_PLAYLIST_SONG:
        filterTag1.playlist = objectID;
        break;
    default:
        return MP_ERR_DB_UNSUPPORTED_LIST_TYPE;
    }

    /* just create a list with the generic method */
    res =  CreateList(OUT dbListID,
            IN deviceID,
            IN listType,
            IN playContinuation,
            IN filterTag1,
            IN filterTag2,
            IN filterTag3,
            IN filterTag4);
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::CreateList(tListID &listID, const tDeviceID deviceID, const tListType listType, const tPlayContinuation playContinuation,
        const tFilterTag1 filterTag1, const tFilterTag2 filterTag2, const tFilterTag3 filterTag3, const tFilterTag4 filterTag4,
        const tStreaming streaming /*=false*/, const tBoolean checkVTIPOD /*=true*/, const tBoolean lastPlayedList /*=false*/, const tPath lastPlayedPath,
        const char *pSearchString /*= NULL*/, tUserID userID /*= 0*/) // additional param Roadmap 17001 'Personalization'
{
    ENTRY;
    VARTRACE(listType);
    tResult res;
    DBList *dbList = NULL;
    UnlockOnExit unlockQuery(&dbList);

    /* create a list context
     * mQuery = new Query() + mQuery->DoLock() is done in CreateDBList() */
    dbList = CreateDBList();
    if (!dbList) return MP_ERR_DB_CREATE_LIST_CONTEXT;

    /* set the returning list id */
    listID = dbList->mListID;

    /* take real deviceID in case of a MyMedia playlist */
    tDeviceID realDeviceID = deviceID;
    if ((MY_MEDIA == deviceID)
        &&
        (listType == LTY_PLAYLIST_SONG)) {

        res = GetObjectDeviceID(OUT realDeviceID, IN filterTag1.tag, CTY_PLAYLIST);
        if (res) return res;
    }

    /* remember the list creation parms for later storage in last mode table */
    dbList->mDeviceID = realDeviceID;
    dbList->mListType = listType;
    dbList->mPlayContinuation = playContinuation;
    dbList->mFilterTag1.tag = filterTag1.tag;
    dbList->mFilterTag2.tag = filterTag2.tag;
    dbList->mFilterTag3.tag = filterTag3.tag;
    dbList->mFilterTag4.tag = filterTag4.tag;
    dbList->mStreaming = streaming;
    dbList->mUserID = userID;


    if(pSearchString)
    {
        //for the LIKE in Queries: e.g. text so search is 'em' then LIKE demands '%em%'
        snprintf(dbList->mSearchString,sizeof(tSearchString),"%s",pSearchString); // additional param Roadmap 160014
    }

    /* get the device information*/
    tDeviceInfo deviceInfo;
    res = GetDeviceInfo(OUT deviceInfo, IN realDeviceID);
    if (res) return res;

    if(DTY_MTP == deviceInfo.deviceType){
        if(lastPlayedList) {
            strncpy_r(OUT dbList->mPath, IN lastPlayedPath, IN sizeof(dbList->mPath));
        }
    }

    /* mask out all filtertags not matching the columns according to list type */
    res = MaskOutFilterTagsByListType(INOUT dbList->mFilterTag1, INOUT dbList->mFilterTag2, INOUT dbList->mFilterTag3, INOUT dbList->mFilterTag4, IN listType);
    if (res) return res;

    /* If check is denied set the new list fix to false */
    if (checkVTIPOD) {
    /* is this list a hidden IPOD access? */
    dbList->mIsVTIPODList = IsVTIPODList(IN dbList->mFilterTag1, IN dbList->mFilterTag2, IN dbList->mFilterTag3, IN dbList->mFilterTag4, IN listType, IN deviceInfo, IN lastPlayedList);
    }
    else {
        dbList->mIsVTIPODList = false;
    }
    ETG_TRACE_USR3(("Created list is VTIPODList: %u", dbList->mIsVTIPODList));

    /* if it is an IPOD list, do another query */
    if (dbList->mIsVTIPODList) {

        tFilterTag1 IPODFilterTag1;
        tFilterTag2 IPODFilterTag2;
        tFilterTag3 IPODFilterTag3;
        tFilterTag3 IPODFilterTag4;

        IPODFilterTag1.tag = dbList->mFilterTag1.tag;
        IPODFilterTag2.tag = dbList->mFilterTag2.tag;
        IPODFilterTag3.tag = dbList->mFilterTag3.tag;
        IPODFilterTag4.tag = dbList->mFilterTag4.tag;

        /* Playlist items retrieved always via VTIPOD */
        if (listType == LTY_PLAYLIST_SONG && !(filterTag1.tag & IPOD_MARKER_BIT)) {
            //it's an indexed playlist tagID
            tTagID playlistTag = filterTag1.tag;
            res = GetVTTagForIPODPlaylist(INOUT playlistTag);
            if (res) return res;
            IPODFilterTag1.tag = playlistTag;
        } else {
            res = GetVTTagForIPOD(INOUT IPODFilterTag1.tag);
            if (res) return res;
        }
        res = GetVTTagForIPOD(INOUT IPODFilterTag2.tag);
        if (res) return res;
        res = GetVTTagForIPOD(INOUT IPODFilterTag3.tag);
        if (res) return res;
        res = GetVTTagForIPOD(INOUT IPODFilterTag4.tag);
        if (res) return res;

        res = dbList->mQuery->Select(LTY_IPOD_ACCESS, MY_MEDIA, "tiiiii",
                IN deviceInfo.mountPoint,
                IN listType,
                IN IPODFilterTag1.tag,
                IN IPODFilterTag2.tag,
                IN IPODFilterTag3.tag,
                IN IPODFilterTag4.tag);
        if (res) return res;
    }
    else if ((LocalSPM::GetDataProvider().PersonalizationSupported()) && IsUserFavoriteList(listType))          //Favorite Lists
    {
        ETG_TRACE_USR2(("DBManager::CreateList Favorites userID : %d",dbList->mUserID));

        if (0==userID)
        {
             GetCurrentUser(userID);
             ETG_TRACE_USR2(("DBManager::CreateList Favorites userID : %d",userID));
             dbList->mUserID = userID;
             ETG_TRACE_USR2(("DBManager::CreateList Favorites userID : %d",dbList->mUserID));

        }

        //tGeneralString formatBuffer;

        if (LTY_USER_FAVORITES_SONG_SEARCH == listType)
        {
            res = dbList->mQuery->Select(IN listType, IN realDeviceID,"iit",IN 0,IN dbList->mUserID,IN pSearchString);
        }
        else
        {
            res = dbList->mQuery->Select(IN listType, IN realDeviceID,"i",IN dbList->mUserID,IN pSearchString); //additional param Roadmap 160014 'Full text search'
        }
        if (res) return res;
    }
    else { // standard list

        /* prepare the sql statement for the metadata list */
        tGeneralString formatBuffer;
        SelectFormatString(INOUT formatBuffer, IN dbList->mFilterTag1, IN dbList->mFilterTag2, IN dbList->mFilterTag3, IN dbList->mFilterTag4,
                           IN dbList->mSearchString); //additional param Roadmap 160014 'Full text search'


        res = dbList->mQuery->Select(IN listType, IN realDeviceID, IN formatBuffer,
                IN filterTag1.tag,
                IN filterTag2.tag,
                IN filterTag3.tag,
                IN filterTag4.tag,
                IN pSearchString); //additional param Roadmap 160014 'Full text search'
        if (res) return res;
    }

    return MP_NO_ERROR;
}

tResult DBManager::GetMediaObjectDoStep(tMediaObject &mediaObject, const tListID listID, const tIndex rowNumber, const tBool uuidOnly)
{
    ENTRY
    /* locking of the query is done from outside */
    return GetMediaObjectInternal(OUT mediaObject, IN true/*doStep*/, IN listID, IN rowNumber, IN uuidOnly);
}

tResult DBManager::GetMediaObject(tMediaObject &mediaObject, const tListID listID, const tIndex rowNumber)
{
    ENTRY
    DBList *dbList = NULL;
    UnlockOnExit unlockQuery(&dbList);

    /* get list pointer */
    //dbList = LookupDBList(IN listID);
    GET_DBLIST_POINTER(dbList, listID);
    if (!dbList) return MP_ERR_DB_LIST_NOT_FOUND;

    /* lock the query */
    dbList->mQuery->DoLock();

    return GetMediaObjectInternal(OUT mediaObject, IN false/*doStep*/, IN listID, IN rowNumber, IN 0);
}

/*
 * update the ObjectID of the mediaobject with relavant tagID
 * e.g update objectID of Album element with MetaDataTag3 etc
 */
tResult DBManager::UpdateObjectID(tMediaObject &mediaObject, const tListType listType) const
{
    switch(listType)
    {
        case LTY_GENRE:
        case LTY_VIDEO:
        case LTY_PODCAST:
        case LTY_PLAYLIST:
        case LTY_AUTHOR:
            mediaObject.objectID = mediaObject.MetadataTag1;
            break;

        case LTY_ARTIST:
        case LTY_GENRE_ARTIST:
        case LTY_AUDIOBOOK:
        case LTY_AUTHOR_BOOKTITLE:
            mediaObject.objectID = mediaObject.MetadataTag2;
            break;

        case LTY_COMPOSER:
            mediaObject.objectID = mediaObject.MetadataTag3;
            break;

        case LTY_ALBUM:
        case LTY_ALBUM_UNKNOWN_ALBUMART:
        case LTY_GENRE_ARTIST_ALBUM:
        case LTY_GENRE_ALBUM :
        case LTY_YEAR_ALBUM :
        case LTY_ARTIST_ALBUM:
        case LTY_COMPOSER_ALBUM:
        case LTY_COMPILATION:
            mediaObject.objectID = mediaObject.MetadataTag4;
            break;
        case LTY_YEAR:
            mediaObject.objectID = mediaObject.yearID;
            break;
        default:
            /*For other list types keep the objectID as delivered by the data base . e.g LTY_SONGS , LTY_*_*_SONGS, *_EPISODE, *_CHAPTER etc*/
            break;
    }

    return MP_NO_ERROR;
}

tResult DBManager::GetMediaObjectInternal(tMediaObject &mediaObject, const tBoolean doStep, const tListID listID, const tIndex rowNumber, const tBool uuidOnly)
{
    ENTRY
    tResult res;
    DBList *dbList;
    tRowNumber row;
    int fillDeviceData = 0;
    InitMediaObject(OUT mediaObject);

    /* get list pointer */
    //dbList = LookupDBList(IN listID);
    GET_DBLIST_POINTER(dbList, listID);
    if (!dbList) return MP_ERR_DB_LIST_NOT_FOUND;

    res = dbList->mQuery->GetRow(OUT row);
    if (res) return res;


    if (rowNumber != INDEX_LAST_ACTIVE_OBJECT && row != rowNumber) {

        /* set the row to read */
        res = dbList->mQuery->SetRow(IN (tRowNumber)rowNumber);
        if (res) return res;
    } else if ((rowNumber == INDEX_LAST_ACTIVE_OBJECT)
               &&
               (!dbList->mIsVTIPODList)
               &&
               (dbList->mCurrentRow == row)) {

        // JAC2-4883 patch begin: the stored current object might be outdated, need to get the newest information from database
        // This code snippet is the same to bellow 'fillDeviceData' part
        // get the db info for the device
        tDeviceInfo deviceInfo;
        res = GetDeviceInfo(OUT deviceInfo, IN dbList->mCurrentMediaObject.deviceID);
        if (!res){
            /* put the device information into the media object */
            strncpy_r(dbList->mCurrentMediaObject.deviceVersion, deviceInfo.deviceVersion, sizeof(dbList->mCurrentMediaObject.deviceVersion));
            strncpy_r(dbList->mCurrentMediaObject.mountPoint, deviceInfo.mountPoint, sizeof(dbList->mCurrentMediaObject.mountPoint));
            dbList->mCurrentMediaObject.deviceType = deviceInfo.deviceType;
        }
        // JAC2-4883 patch end

        /* return the stored current object */
        mediaObject = dbList->mCurrentMediaObject;
        ETG_TRACE_USR3(("Return stored current object: %u (row: %u, dbListID: %u)", mediaObject.objectID, row, listID));
        return MP_NO_ERROR;
    }

    /* fill the media object */

    /* check if it is an IPOD list */
    if (dbList->mIsVTIPODList) {

        res = dbList->mQuery->Get(IN doStep, IN "iTTTii",
                 OUT &mediaObject.objectID,
                 OUT mediaObject.title, IN sizeof(mediaObject.title),
                 OUT mediaObject.fileName, IN sizeof(mediaObject.fileName),
                 OUT mediaObject.UUID, IN sizeof(mediaObject.UUID),
                 OUT &mediaObject.active,
                 OUT &mediaObject.trackNumber);
        if (!doStep) {
            dbList->mQuery->End();
        }
        if (res) return res;

        //VARTRACE(mediaObject.title);
        //VARTRACE(mediaObject.trackNumber);
        //take over static parameter
        /* take real deviceID in case of a MyMedia playlist */
        if ((MY_MEDIA == dbList->mDeviceID)
            &&
            (dbList->mListType == LTY_PLAYLIST_SONG))
        {
            res = GetObjectDeviceID(OUT mediaObject.deviceID, IN dbList->mFilterTag1.tag, CTY_PLAYLIST);
            if (res) return res;
        }
        else
        {
            mediaObject.deviceID = dbList->mDeviceID;
        }

        tBoolean found = FALSE;

        //fill up empty titles based on UUID (iOS8 queue list items)
        if ((mediaObject.UUID[0] != 0)
            &&
            (dbList->mListType == LTY_CURRENT_SELECTION)) {

            //Backup mediaObject to be able to restore it if no object is found in DB
            tMediaObject backupMediaObject = mediaObject;

            res = GetIPODMediaObjectByUUID(INOUT mediaObject, IN mediaObject.deviceID, IN mediaObject.UUID);
            if (res) {
                mediaObject = backupMediaObject;
            } else {
                found = TRUE;
                strncpy_r(mediaObject.fileName, backupMediaObject.fileName, sizeof(mediaObject.fileName));
            }
        }

        tTagID tagID = 0;

        /* add URL to live tags... */
        res = StoreLiveTagsElement(IN mediaObject.fileName, IN CTY_VTIPOD, IN mediaObject.deviceID, IN mediaObject.notPlayable);
        if (res) return res;

        /* ...and get back real id from live tags table */
        res = GetLiveTagsElement(OUT tagID, IN CTY_VTIPOD, IN mediaObject.fileName);
        if (res) return res;

        /* add vtipod marker bit to new object id*/
        mediaObject.objectID = tagID | IPOD_MARKER_BIT;

        if(!found)
        {
            //No total playtime from VTIPOD, refer to GMNGA-57544
            mediaObject.totalPlaytime = PLAYTIME_NONE;
            mediaObject.catType = GetCTYByLTY(dbList->mListType);

            /* map meta fields and tags according to LTY - copy available title to related metadata field */
            switch (dbList->mListType) {
            case LTY_GENRE:
            case LTY_VIDEO:
            case LTY_PODCAST:
            case LTY_PLAYLIST:
            case LTY_AUTHOR:
                mediaObject.MetadataTag1 = mediaObject.objectID;
                strncpy_r(mediaObject.MetadataField1, mediaObject.title, sizeof(mediaObject.MetadataField1));
                //mediaObject.objectID = 0;  // Fix for PSARCCB-4506
                mediaObject.title[0] = 0;
                break;
            case LTY_ARTIST:
            case LTY_GENRE_ARTIST:
            case LTY_AUDIOBOOK:
            case LTY_AUTHOR_BOOKTITLE:
                mediaObject.MetadataTag2 = mediaObject.objectID;
                strncpy_r(mediaObject.MetadataField2, mediaObject.title, sizeof(mediaObject.MetadataField2));
                //mediaObject.objectID = 0; // Fix for PSARCCB-4506
                mediaObject.title[0] = 0;
                break;
            case LTY_COMPOSER:
                mediaObject.MetadataTag3 = mediaObject.objectID;
                strncpy_r(mediaObject.MetadataField3, mediaObject.title, sizeof(mediaObject.MetadataField3));
                //mediaObject.objectID = 0; // Fix for PSARCCB-4506
                mediaObject.title[0] = 0;
                break;
            case LTY_ALBUM:
            case LTY_ALBUM_UNKNOWN_ALBUMART:
            case LTY_GENRE_ARTIST_ALBUM:
            case LTY_GENRE_ALBUM:
            case LTY_ARTIST_ALBUM:
            case LTY_YEAR_ALBUM:
            case LTY_COMPOSER_ALBUM:
            case LTY_COMPILATION:
                mediaObject.MetadataTag4 = mediaObject.objectID;
                strncpy_r(mediaObject.MetadataField4, mediaObject.title, sizeof(mediaObject.MetadataField4));
                //mediaObject.objectID = 0; // Fix for PSARCCB-4506
                mediaObject.title[0] = 0;
                break;
            case LTY_SONG:
            case LTY_GENRE_ARTIST_ALBUM_SONG:
            case LTY_GENRE_ARTIST_SONG:
            case LTY_GENRE_ALBUM_SONG:
            case LTY_YEAR_ALBUM_SONG:
            case LTY_GENRE_SONG:
            case LTY_ARTIST_ALBUM_SONG:
            case LTY_ARTIST_SONG:
            case LTY_ALBUM_SONG:
            case LTY_COMPOSER_ALBUM_SONG:
            case LTY_COMPOSER_SONG:
            case LTY_PLAYLIST_SONG:
            case LTY_CURRENT_SELECTION:
            case LTY_PODCAST_EPISODE:
            case LTY_VIDEO_EPISODE:
            case LTY_EPISODE_OF_VIDEO:
            case LTY_BOOKTITLE_CHAPTER:
            case LTY_AUTHOR_BOOKTITLE_CHAPTER:
            case LTY_IMAGE_FOLDER:
            case LTY_IMAGE_FOLDER_ITEM:
            case LTY_COMPILATION_SONG:
                //no mapping required
                break;
            default:
                ETG_TRACE_ERR(("VTIPOD - Invalid LTY %d", dbList->mListType));
                break;
            }
        }

        mediaObject.mediaType = GetMTYByCTY(mediaObject.catType);
        mediaObject.fileType = GetFTByMTY(mediaObject.mediaType);
        strncpy_r(mediaObject.albumArtString, mediaObject.fileName, sizeof(mediaObject.albumArtString));

        fillDeviceData = 1;

    } else if ((dbList->mListType == LTY_FILELIST) ||
               (dbList->mListType == LTY_FILELIST_UNSORTED) ||
               (dbList->mListType == LTY_FILELIST_SORTED_FOR_INDEXING) ||
               (dbList->mListType == LTY_FILELIST_PLAYLIST) ||
               (dbList->mListType == LTY_FILELIST_MEDIAOBJECTS) ||
               (dbList->mListType == LTY_FILELIST_MEDIAOBJECTS_WITH_SUBFOLDERS) ||
               (dbList->mListType == LTY_MTP_FILELIST) ||
               (dbList->mListType == LTY_BLUETOOTH_FILELIST)
               )
    {
        /*for mtp and bt file lists retrieve also the item id of the element*/
        if(dbList->mListType == LTY_MTP_FILELIST)
        {
            tU32 uuid = 0;
            res = dbList->mQuery->Get(IN doStep, IN "ii-TTii",
                    OUT &mediaObject.objectID,
                    OUT &uuid,
                    OUT mediaObject.path, IN sizeof(mediaObject.path),
                    OUT mediaObject.title, IN sizeof(mediaObject.title),
                    OUT &mediaObject.fileType,
                    OUT &mediaObject.fileFormat);
            snprintf(mediaObject.UUID,sizeof(mediaObject.UUID),"%d",uuid);
        }
        else if (dbList->mListType == LTY_BLUETOOTH_FILELIST)
        {
            res = dbList->mQuery->Get(IN doStep, IN "iT-TTii",
                    OUT &mediaObject.objectID,
                    OUT mediaObject.UUID, IN sizeof(mediaObject.UUID),
                    OUT mediaObject.path, IN sizeof(mediaObject.path),
                    OUT mediaObject.title, IN sizeof(mediaObject.title),
                    OUT &mediaObject.fileType,
                    OUT &mediaObject.fileFormat);
        }
        else
        {
            res = dbList->mQuery->Get(IN doStep, IN "i-TTii",
                    OUT &mediaObject.objectID,
                    OUT mediaObject.path, IN sizeof(mediaObject.path),
                    OUT mediaObject.title, IN sizeof(mediaObject.title),
                    OUT &mediaObject.fileType,
                    OUT &mediaObject.fileFormat);
            if((dbList->mDeviceID == 1) && (dbList->mListType == LTY_FILELIST_PLAYLIST))
            {
                Query query;
                tURL PlaylistURL;
                tRippedTrackInfo trackInfo;

                //Form the URL from path and title(Track_1.mp3)
                strncpy_r(PlaylistURL, mediaObject.path, sizeof(tURL));
                strncat_r(PlaylistURL,mediaObject.title,sizeof(tURL));

                /* get the metadata info */
                res = query.Select(LTY_MEDIAOBJECTS_RIPPED_FILE_METADATA, IN dbList->mDeviceID, "t" , IN PlaylistURL);
                if (res) return res;

                res = query.Get("----T--",
                        OUT trackInfo.title, IN sizeof(tMetadata));
                query.End();
                if (res) return res;

                strncpy_r(OUT mediaObject.MetadataField4, IN trackInfo.title, IN sizeof(mediaObject.MetadataField4));
            }
        }
        if (!doStep) {
            dbList->mQuery->End();
        }
        if (res) return res;

        /* take real deviceID in case of a MyMedia playlist */
        if ((MY_MEDIA == dbList->mDeviceID)
            &&
            (dbList->mListType == LTY_FILELIST_PLAYLIST))
        {
            res = GetObjectDeviceID(OUT mediaObject.deviceID, IN dbList->mFilterTag1.tag, CTY_PLAYLIST);
            if (res) return res;
        }
        else
        {
            mediaObject.deviceID = dbList->mDeviceID;
        }

#if 0 // remove: only for test for chinese characters in m3u playlist
        static FILE *fpLog = NULL;
        if (!fpLog) {
            fpLog = fopen("/tmp/DBManager.txt", "w");
        }
        if (fpLog) {

            fprintf(fpLog, "playlist: dbList->mListID=%d, dbList->mPath=%s\n", dbList->mListID, dbList->mPath);
            fprintf(fpLog, "playlist: mediaObject.objectID=%d\n", mediaObject.objectID);
            fprintf(fpLog, "playlist: mediaObject.path=%s\n", mediaObject.path);
            hexDump(fpLog, (const char *)"playlist: mediaObject.path", mediaObject.path, strlen_r(mediaObject.path));
        }

        if (dbList->mListType == LTY_FILELIST_PLAYLIST) {

            gchar *toCodeset = (gchar *)"ISO-8859-1"; // default / ANSI

            if (LocalSPM::GetDataProvider().GBKSupported()) {
                toCodeset = (gchar *)"GBK";
            } else if (LocalSPM::GetDataProvider().EUC_KRSupported()) {
                toCodeset = (gchar *)"EUC-KR";
            }

            if (fpLog) {
                hexDump(fpLog, (const char *)"playlist conversion 1: mediaObject.title", mediaObject.title, strlen_r(mediaObject.title));
            }

            /* the title is in UTF-8 because the VTFile delivers it EVERY TIME in UTF-8 */

            /* interpret it as UTF-8 and store it as ANSI (just remove the UTF-8 framing) */
            gchar *newString;
            newString = g_convert((gchar *)mediaObject.title, -1, "ISO-8859-1", "UTF-8", NULL, NULL, NULL);
            if (newString) {
                if (fpLog) {
                    hexDump(fpLog, (const char *)"playlist conversion 1: newString", newString, strlen_r(newString));
                }
                strncpy_r(OUT (char *)mediaObject.title, IN newString, IN sizeof(mediaObject.title));
                g_free(newString);
            }

            /* interpret it as GBK2312 (EUC-CN) and convert it to UTF-8. Here EUC-CN is used and NOT GBK! Why?*/
            newString = g_convert((gchar *)mediaObject.title, -1, "UTF-8", "EUC-CN", NULL, NULL, NULL);
            if (newString) {
                if (fpLog) {
                    //hexDump(fpLog, (const char *)"playlist conversion 3: mediaObject.title", mediaObject.title, strlen_r(mediaObject.title));
                    hexDump(fpLog, (const char *)"playlist conversion 3: newString", newString, strlen_r(newString));
                }
                strncpy_r(OUT (char *)mediaObject.title, IN newString, IN sizeof(mediaObject.title));
                g_free(newString);
            }

        } else {
            if (fpLog) {
                hexDump(fpLog, (const char *)"NO playlist: mediaObject.title", mediaObject.title, strlen_r(mediaObject.title));
            }
        }
#endif

        switch(mediaObject.fileType) {
        case FT_AUDIO:
            mediaObject.catType = CTY_SONG;
            mediaObject.mediaType = MTY_MUSIC_FILE;
            break;
        case FT_FOLDER:
            mediaObject.catType = CTY_DIR;
            mediaObject.mediaType = MTY_UNKNOWN;
            break;
        case FT_PLAYLIST:
            mediaObject.catType = CTY_PLAYLIST;
            mediaObject.mediaType = MTY_PLAYLIST;
            break;
        case FT_VIDEO:
            mediaObject.catType = CTY_VIDEO;
            mediaObject.mediaType = MTY_VIDEO;
            break;
        case FT_IMAGE:
           mediaObject.catType = CTY_IMAGE;
           mediaObject.mediaType = MTY_IMAGE;
            break;
        default:
            break;
        }

        /*create complete path(combine basepath<path> and pure filename<title> ) and set it to mediaObject.fileName*/
        if(NULL == FastUTF8::StartsWith((const FastUTF8::tString)mediaObject.path, (const FastUTF8::tString)"/")) //lint !e1773
        {
            strncpy_r(mediaObject.fileName, "/", sizeof(mediaObject.fileName));
            strncat_r(mediaObject.fileName, mediaObject.path, sizeof(mediaObject.fileName));
        }
        else
        {
            strncpy_r(mediaObject.fileName, mediaObject.path, sizeof(mediaObject.fileName));
        }

        if(strlen_r(mediaObject.path))
        {
            if(NULL == FastUTF8::EndsWithNC((const FastUTF8::tString)mediaObject.path, (const FastUTF8::tString)"/")) //lint !e1773
            {
                strncat_r(mediaObject.fileName, "/", sizeof(mediaObject.fileName));
            }
        }
        else
        {
            strncpy_r(mediaObject.path, "/", sizeof(mediaObject.path));
        }


        strncat_r(mediaObject.fileName, mediaObject.title, sizeof(mediaObject.fileName));

        if((mediaObject.fileType == FT_AUDIO) || (mediaObject.fileType == FT_VIDEO))
        {
            strncpy_r(mediaObject.albumArtString, mediaObject.fileName, sizeof(mediaObject.fileName));
        }


        /* store file name to live tags to enable favorite tagging and last mode for not indexed objects in future */
        if ((dbList->mListType != LTY_FILELIST_UNSORTED) //do it only for HMI list, not for the unsorted list which is used to index a device
            &&
            (dbList->mListType != LTY_FILELIST_SORTED_FOR_INDEXING)
            &&
            (mediaObject.catType != CTY_NONE)
            &&
            (dbList->mListType != LTY_BLUETOOTH_FILELIST))
        {
            /* Store element to live tags only if it is not already in DB.
             * Do it for all browsed and played objects.
             * Do it although the object is already in the MediaObjects table to be able to get the object position in a file list via GetPositionInList */
            mediaObject.objectID = OBJECT_ID_NONE;
            res = GetLiveTagsElement(OUT mediaObject.objectID, IN mediaObject.catType, IN mediaObject.fileName);
            //res = GetObjectID(OUT mediaObject.objectID, IN mediaObject.deviceID, IN mediaObject.fileName);
            if (res && res != MP_ERR_DB_END_OF_LIST) return res;

            if (mediaObject.objectID == OBJECT_ID_NONE)
            {
                /* add URL to live tags... */
                res = StoreLiveTagsElement(IN mediaObject.fileName, IN mediaObject.catType, IN mediaObject.deviceID, IN mediaObject.notPlayable);
                if (res) return res;

                /* ...and get back real id from live tags table */
                res = GetLiveTagsElement(OUT mediaObject.objectID, IN mediaObject.catType, IN mediaObject.fileName);
                if (res) return res;
            }
        }

        fillDeviceData = 1;

    /* CDDA list types */
    } else if (dbList->mListType == LTY_CD) {

        // itemid,filename,type,format,metadata1,metadata2,metadata3,metadata4,playtime
        res = dbList->mQuery->Get(IN doStep, IN "iTiiTTTTi",
                OUT &mediaObject.objectID,
                OUT mediaObject.fileName, IN sizeof(mediaObject.fileName),
                OUT &mediaObject.fileType,
                OUT &mediaObject.fileFormat,
                OUT mediaObject.MetadataField1, IN sizeof(mediaObject.MetadataField1),
                OUT mediaObject.MetadataField2, IN sizeof(mediaObject.MetadataField2),
                OUT mediaObject.MetadataField3, IN sizeof(mediaObject.MetadataField3),
                OUT mediaObject.MetadataField4, IN sizeof(mediaObject.MetadataField4),
                OUT &mediaObject.totalPlaytime);
        if (!doStep) {
            dbList->mQuery->End();
        }
        if (res) return res;

        strncpy_r(OUT mediaObject.title, IN mediaObject.MetadataField3, IN sizeof(mediaObject.title));
        mediaObject.catType = CTY_SONG;
        mediaObject.mediaType = MTY_MUSIC_FILE;

        /* initiate the fill of device data to the media object */
        mediaObject.deviceID = dbList->mDeviceID;
        fillDeviceData = 1;

    } else { // standard List

        /* get the complete media object or fill only the UUID field */
        if (!uuidOnly) {

        res = dbList->mQuery->Get(IN doStep, IN "iiiTTTiiTiTiTiTiiiTTiilTiiiiiTTi",
                 OUT &mediaObject.objectID,
                 OUT &mediaObject.catType,
                 OUT &mediaObject.deviceID,
                 OUT mediaObject.fileName, IN sizeof(mediaObject.fileName),
                 OUT mediaObject.UUID, IN sizeof(mediaObject.UUID),
                 OUT mediaObject.title, IN sizeof(mediaObject.title),
                 OUT &mediaObject.trackNumber,
                 OUT &mediaObject.MetadataTag1,
                 OUT mediaObject.MetadataField1, IN sizeof(mediaObject.MetadataField1),
                 OUT &mediaObject.MetadataTag2,
                 OUT mediaObject.MetadataField2, IN sizeof(mediaObject.MetadataField2),
                 OUT &mediaObject.MetadataTag3,
                 OUT mediaObject.MetadataField3, IN sizeof(mediaObject.MetadataField3),
                 OUT &mediaObject.MetadataTag4,
                 OUT mediaObject.MetadataField4, IN sizeof(mediaObject.MetadataField4),
                 OUT &mediaObject.metadataConvertFlag,
                 OUT &mediaObject.compilationFlag,
                 OUT &mediaObject.totalPlaytime,
                 OUT mediaObject.albumArtString, IN sizeof(mediaObject.albumArtString),
                 OUT mediaObject.path, IN sizeof(mediaObject.path),
                 OUT &mediaObject.notPlayable,
                 OUT &mediaObject.mediaType,
                 OUT &mediaObject.fileSize,
                 OUT mediaObject.dateTime, IN sizeof(mediaObject.dateTime),
                 OUT &mediaObject.fileMode,
                 OUT &mediaObject.userID,
                 OUT &mediaObject.groupID,
                 OUT &mediaObject.year,
                 OUT &mediaObject.yearID,
                 OUT mediaObject.deviceVersion, IN sizeof(mediaObject.deviceVersion),
                 OUT mediaObject.mountPoint, IN sizeof(mediaObject.mountPoint),
                 OUT &mediaObject.deviceType);
        } else { // uuiOnly:

            res = dbList->mQuery->Get(IN doStep, IN "----T", OUT mediaObject.UUID, IN sizeof(mediaObject.UUID));

        }
        if (!doStep) {
            dbList->mQuery->End();
        }
        if (res) return res;

        /* do some post processing only if the complete media object was read out */
        if (!uuidOnly) {

            /*update objectID of the media object based on the meta data tag*/
            UpdateObjectID(OUT mediaObject, IN dbList->mListType);

            /* get file attributes from configuration via file extension */
            if(CTY_DIR == mediaObject.catType)
            {
                mediaObject.fileFormat = FFT_UNKNOWN;
                mediaObject.fileType = FT_FOLDER;
            }
            else
            {
                /* get category type from list type (except for browse lists) */
                if ((LTY_BROWSELIST != dbList->mListType)
                    &&
                    (LTY_BROWSELIST_MEDIAOBJECTS != dbList->mListType)
                    &&
                    (LTY_BROWSELIST_MEDIAOBJECTS_WITH_SUBFOLDERS != dbList->mListType))
                {
                    mediaObject.catType = GetCTYByLTY(dbList->mListType);
                }
                mediaObject.mediaType = GetMTYByCTY(mediaObject.catType);

                res = GetFileFormat(OUT mediaObject.fileFormat, OUT mediaObject.fileType, IN mediaObject.fileName);
            }
        } // !uuidOnly
    }

    /* do some post processing only if the complete media object was read out */
    if (!uuidOnly) {

        if (fillDeviceData) {

            // get the db info for the device
            tDeviceInfo deviceInfo;
            res = GetDeviceInfo(OUT deviceInfo, IN mediaObject.deviceID);
            if (res) return res;

            /* put the device information into the media object */
            strncpy_r(mediaObject.deviceVersion, deviceInfo.deviceVersion, sizeof(mediaObject.deviceVersion));
            strncpy_r(mediaObject.mountPoint, deviceInfo.mountPoint, sizeof(mediaObject.mountPoint));
            mediaObject.deviceType = deviceInfo.deviceType;
        }

        /* set the album art path with complete mount point */
        tAlbumArt newAlbumArt;
        if (strlen_r(mediaObject.albumArtString))
        {
            if(LocalSPM::GetDataProvider().IsAlbumArtSupported(mediaObject.deviceType))
            {
                strncpy_r(newAlbumArt, mediaObject.mountPoint, sizeof(newAlbumArt));
                strncat_r(newAlbumArt, mediaObject.albumArtString, sizeof(newAlbumArt));
            }
            else
            {
                strncpy_r(newAlbumArt, "" , sizeof(newAlbumArt));
            }
            strncpy_r(mediaObject.albumArtString, newAlbumArt, sizeof(newAlbumArt));
        }
    } // !uuidOnly

    //<--Roadmap 17001 : 'Personalization'
    if (LocalSPM::GetDataProvider().PersonalizationSupported())
    {
        if (!uuidOnly && (DTY_USB == mediaObject.deviceType))
        {
            mediaObject.isFavorite = IsUserFavoriteObject(IN mediaObject.objectID);
            ETG_TRACE_USR3(("mediaObject.isFavorite : %d",mediaObject.isFavorite));
        }
    }
    //<--Roadmap 17001 : 'Personalization'

    /* if no move desired, reset the current row number */
    if (!doStep) {
        res = dbList->mQuery->SetRow(IN row);
        if (res) return res;
    }

    if (!uuidOnly) {
        /* store the object as current one (for caching further requests) */
        dbList->mCurrentMediaObject = mediaObject;
        if (rowNumber == INDEX_LAST_ACTIVE_OBJECT )
        {
            dbList->mCurrentRow = row;
        }
        else
        {
            dbList->mCurrentRow = rowNumber;
        }
        ETG_TRACE_USR3(("Store object as current one. dbListID: %u, row: %u, objectID: %u", dbList->mListID, dbList->mCurrentRow, dbList->mCurrentMediaObject.objectID));
    }

    return MP_NO_ERROR;
}

#if 0 //Do not use it because the mount point can change in between due to auto mounter
tResult DBManager::GetDeviceInfo(tDeviceInfo &deviceInfo, const tMountPoint mountPoint)
{
    ENTRY
    tResult res;
    tDeviceID deviceID;

    /* get the device to the mount point */
    res = GetDevice(OUT deviceID, IN mountPoint);
    if (res) return res;

    /* get the device info for that */
    res = GetDeviceInfo(OUT deviceInfo, IN deviceID);
    if (res) return res;

    return MP_NO_ERROR;
}
#endif

tResult DBManager::GetDeviceInfo(tDeviceInfo &deviceInfo, const tDeviceID deviceID, const tBoolean virtualDevice/*=false*/)
{
    ENTRY
    Query query;
    tResult res;

    InitDeviceInfo(OUT deviceInfo);

    /* prepare the sql statement for the devices list */
    if(virtualDevice)
    {
        res = query.Select(LTY_VIRTUAL_DEVICES, IN deviceID, "");
    }
    else
    {
        res = query.Select(LTY_DEVICES, IN deviceID, "");
    }
    if (res) return res;


         res = query.Get("iTTTTTi--i----iiii-iTiiiiiiiiiiiTiiii---TT",
            OUT &deviceInfo.deviceID,
            OUT deviceInfo.UUID, IN sizeof(deviceInfo.UUID),
            OUT deviceInfo.serialNumber, IN sizeof(deviceInfo.serialNumber),
            OUT deviceInfo.deviceVersion, IN sizeof(deviceInfo.deviceVersion), //FirmwareVersion
            OUT deviceInfo.deviceName, IN sizeof(deviceInfo.deviceName),       //FriendlyName
            OUT deviceInfo.mountPoint, IN sizeof(deviceInfo.mountPoint),
            OUT &deviceInfo.connectionCount,
            OUT &deviceInfo.activeSource,
            OUT &deviceInfo.deviceType,
            OUT &deviceInfo.discType,
            OUT &deviceInfo.indexedState,
            OUT &deviceInfo.connectionState,
            OUT &deviceInfo.formerConnectionState,
            OUT deviceInfo.accessoryName, IN sizeof(deviceInfo.accessoryName),
            OUT &deviceInfo.deviceState,
            OUT &deviceInfo.fileSystemType,
            OUT &deviceInfo.partitionNumber,
            OUT &deviceInfo.totalSize,
            OUT &deviceInfo.freeSize,
            OUT &deviceInfo.productID,
            OUT &deviceInfo.numberOfAudioFiles,
            OUT &deviceInfo.numberOfVideoFiles,
            OUT &deviceInfo.numberOfImageFiles,
            OUT &deviceInfo.diPOCaps,
            OUT &deviceInfo.diPOActive,
            OUT deviceInfo.diPOVersion, IN sizeof(deviceInfo.diPOVersion),
            OUT &deviceInfo.connectionType,
            OUT &deviceInfo.disconnectReason,
            OUT &deviceInfo.isShuffleSupported,
            OUT &deviceInfo.isRepeatSupported,
            OUT &deviceInfo.appleDeviceMACAddress,IN sizeof(deviceInfo.appleDeviceMACAddress),
            OUT &deviceInfo.appleDeviceUSBSerialNumber,IN sizeof(deviceInfo.appleDeviceUSBSerialNumber));
    query.End();
    if (res) return res;

    if(!LocalSPM::GetDataProvider().AVRCPLowerProfileSupported())
    {
        deviceInfo.isShuffleSupported = FALSE;
        deviceInfo.isRepeatSupported = FALSE;
    }

    if ((CS_CONNECTED == deviceInfo.connectionState)
        ||
        (CS_UNSUPPORTED == deviceInfo.connectionState))
    {
        deviceInfo.connected = 1;
    }

    return MP_NO_ERROR;
}

tResult DBManager::GenerateMountPoint(tDeviceInfo &deviceInfo) const
{
    ENTRY
    tResult res = MP_NO_ERROR;

    if (DTY_MTP == deviceInfo.deviceType) {
        res = LocalSPM::GetMTPControl().GenerateMountPoint(INOUT deviceInfo);
    }

    return res;
}

tResult DBManager::AddDevice(tDeviceInfo &deviceInfo)
{
    ENTRY
    Query query;
    tResult res;
    tDeviceInfo _deviceInfo = deviceInfo;
    tDeviceID lastDeviceID = 0;
    tDeviceID deviceID = 0;

    /* check total number of devices in db */
    tDeviceCount totalDeviceCountDB;
    res = query.Select(LTY_DEVICES_COUNT, MY_MEDIA, "");
    if (res) return res;
    res = query.Get(IN "i", OUT &totalDeviceCountDB);
    query.End();
    if (res) return res;

    /* check totla number of Indexable devices in db */
    tDeviceCount totalIndexableDeviceCountDB;
    res = query.Select(LTY_INDEXABLEDEVICES_COUNT, MY_MEDIA, "");
    if (res) return res;
    res = query.Get(IN "i", OUT &totalIndexableDeviceCountDB);
    query.End();
    if (res) return res;

    /* Checking connected device type(Indexable or Non-Indexable) */
    if(LocalSPM::GetDataProvider().IsIndexableDevice(IN _deviceInfo.deviceType, IN _deviceInfo.connectionType)){

        /* Check indexable device count in db */
        if(totalIndexableDeviceCountDB >= LocalSPM::GetDataProvider().MaxNoOfIndexableDevicesInDB()){

            /* remove oldest indexable device and return the device id which is reusable afterwards */
            res = RemoveOldestDevice(OUT deviceID, true);
            if (res) return res;
        }
        /* Check total device count in db */
        else if(totalDeviceCountDB >= LocalSPM::GetDataProvider().MaxNoOfDevicesInDB()){

            /* remove oldest device and return the device id which is reusable afterwards */
            res = RemoveOldestDevice(OUT deviceID, false);
            if (res) return res;
        }
    }
    /* Check total device count in db */
    else if(totalDeviceCountDB >= LocalSPM::GetDataProvider().MaxNoOfDevicesInDB()){

        /* remove oldest device and return the device id which is reusable afterwards */
        res = RemoveOldestDevice(OUT deviceID, false);
        if (res) return res;
    }

    /* generate */
    res = GenerateMountPoint(INOUT _deviceInfo);
    if (res) return res;

    /* if a device was removed before, a free id is already set, a search is not needed */
    if (deviceID == 0) {

        /* figure out a free id */
        res = query.Select(LTY_DEVICES, MY_MEDIA, "");
        if (res) return res;

        while(1) {

            res = query.Get("i", OUT &deviceID);
            if (res == MP_ERR_DB_END_OF_LIST) {

                /* the next id is free */
                deviceID++;
                break;
            }

            /* is a gap between this and the former id? */
            if ((deviceID-lastDeviceID) > 1) {

                /* fill this gap */
                deviceID = lastDeviceID+1;
                break;
            }

            lastDeviceID = deviceID;
        }
        query.End();
    }

    /* prepare the sql statement for the devices list */
    res = query.Insert(LTY_DEVICES_INSERT);
    if (res) return res;

    res = query.Put(IN "itttttiiitiiiiiiit",
            IN deviceID,
            IN deviceInfo.UUID,
            IN deviceInfo.serialNumber,
            IN deviceInfo.deviceVersion,
            IN deviceInfo.deviceName,
            IN _deviceInfo.mountPoint,
            IN deviceInfo.deviceType,
            IN deviceInfo.discType,
            IN deviceInfo.connectionState,
            IN deviceInfo.accessoryName,
            IN deviceInfo.fileSystemType,
            IN deviceInfo.partitionNumber,
            IN deviceInfo.totalSize,
            IN deviceInfo.freeSize,
            IN deviceInfo.productID,
            IN deviceInfo.connectionType,
            IN deviceInfo.deviceState,
            IN deviceInfo.appleDeviceMACAddress);
    if (res) return res;

    /* set the default repeat mode */
    res = SetRepeatMode(IN deviceID, IN (tRepeatMode)LocalSPM::GetDataProvider().DBDefaultRepeatMode());
    if (res) return res;

    /* get the device information*/
    res = GetDeviceInfo(OUT deviceInfo, IN deviceID);
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::FreeSpace()
{
    ENTRY
    tDeviceID freeDeviceID;

    return RemoveOldestDevice(OUT freeDeviceID, true);
}

tResult DBManager::RemoveOldestDevice(tDeviceID &freeDeviceID, bool indexableDevice)
{
    ENTRY
    tResult res;
    Query query;
    tMACAddress appleDeviceMACAddress;
    tConnectionType connectionType;
    tNumberOfDevices numberOfDevices;
    vector<tDeviceInfo> deviceInfos;
    tDeviceID freeDCT_WIFIDeviceID = 00;
    bool isDone = false;
    int count=0;

    if(indexableDevice){
        res = query.Select(LTY_INDEXABLEDEVICES_OLDEST, MY_MEDIA, "");
        if (res) return res;

        res = query.Get(IN "i", OUT &freeDeviceID);
        query.End();
    }
    else{
        res = query.Select(LTY_NONINDEXABLEDEVICES_OLDEST, MY_MEDIA, "");
        if (res) return res;

        res = query.Get(IN "i--------------------------------i------T", OUT &freeDeviceID, OUT &connectionType, OUT &appleDeviceMACAddress,IN sizeof(appleDeviceMACAddress));
        query.End();

        if(DCT_BLUETOOTH == connectionType){

            res = GetMediaplayerDeviceConnections(numberOfDevices,deviceInfos,true);
            if (res) return res;

            for(unsigned int i=0; i<deviceInfos.size(); i++) {
                if( (!(strcmp(deviceInfos[i].appleDeviceMACAddress, appleDeviceMACAddress)))&& (deviceInfos[i].connectionType == DCT_WIFI)){

                    freeDCT_WIFIDeviceID = deviceInfos[i].deviceID;
                    break;
                }
            }
        }
    }

    tResult albumArtRemovalResult = MP_NO_ERROR;

    do{
        if (MP_NO_ERROR == res) {

            if(count > 0){
                freeDeviceID = freeDCT_WIFIDeviceID;
                isDone = true;
            }

            /* wait for flush thread to be finished */
            ETG_TRACE_USR3(("RemoveOldestDevice: Lock FlushLock"));
            mFlushLock.lock();
            /* delete all media objects and the device itself */
            ETG_TRACE_USR3(("RemoveOldestDevice deviceID:%u", freeDeviceID));
            RemoveMediaObjects(IN freeDeviceID);
            RemoveDevice(IN freeDeviceID);
            RemoveLastPlayed(IN freeDeviceID); // NCG3D-25376.
            albumArtRemovalResult = RemoveAlbumArts(IN freeDeviceID); //NCG3D-134745
            /* Unlock it here */
            mFlushLock.unlock();

            if(albumArtRemovalResult != MP_NO_ERROR)
            {
                ETG_TRACE_ERR(("RemoveAlbumArts failed with Error=%d/%s",albumArtRemovalResult,errorString(albumArtRemovalResult)));
            }

            /* remove cover art directory of device */
            if (LocalSPM::GetDataProvider().CoverArtFlow()) {

                tPath devicePath; //e.g. "/var/opt/bosch/dynamic/media/coverart/dev03"
                snprintf(devicePath, sizeof(devicePath)-1, "%sdev%02d", LocalSPM::GetDataProvider().CoverArtDirPath().c_str(), freeDeviceID);
                if((exists_directory(devicePath))
                   &&
                   (23 < strlen_r(devicePath))) { //do not delete directory recursively with a path length below "/var/opt/bosch/dynamic/"

                    if(0 != remove_directory(devicePath)) {
                        ETG_TRACE_ERR(("remove_directory: error: %d/%s", errno, strerror(errno)));
                    }
                }
            }

            //todo(pee1cob):Remove the Video Thumbnail storage in flash
        }
        else {
                freeDeviceID = DEVICE_ID_NOT_SET;
        }

        if(freeDCT_WIFIDeviceID){
            count++;
        }

    }while(freeDCT_WIFIDeviceID != 00  && false == isDone);
    return MP_NO_ERROR;
}

tResult DBManager::GetFileListSize(tListSize &listSize, tNumberOfMediaObjects &numberOfMediaObjects, tNumberOfPlayLists &numberOfPlayLists,
        tNumberOfFolders &numberOfFolders, const tListID listID)
{
    ENTRY
    tResult res;
    Query query;
    tUInt objectCount;
    tNumberOfMediaObjects numberOfImages = 0,numberOfAudioObjects = 0,numberOfVideoObjects = 0;
    numberOfMediaObjects = 0;
    numberOfFolders = 0;
    numberOfPlayLists = 0;

    /* get list pointer */
    DBList *dbList = NULL;
    //dbList = LookupDBList(IN listID);
    GET_DBLIST_POINTER(dbList, listID);
    if (!dbList) return MP_ERR_DB_LIST_NOT_FOUND;

    /* get the counts */
    if ((LTY_BROWSELIST == dbList->mListType)
        ||
        (LTY_BROWSELIST_MEDIAOBJECTS == dbList->mListType)
        ||
        (LTY_BROWSELIST_MEDIAOBJECTS_WITH_SUBFOLDERS == dbList->mListType)
        ||
        (LTY_IMAGE_FOLDER_ITEM == dbList->mListType))
    {
        if(LTY_BROWSELIST_MEDIAOBJECTS_WITH_SUBFOLDERS == dbList->mListType)
        {
            tCategoryType catType1 = CTY_DIR;
            tCategoryType catType2 = CTY_DIR;
            tCategoryType catType3 = CTY_DIR;
            tCategoryType catType4 = CTY_DIR;
            GetCTYByFTS(OUT catType1, OUT catType2, OUT catType3, OUT catType4, IN dbList->mFileTypeSelection, IN true/*onlyPlayableObjects*/);

            if( strlen_r(dbList->mPath) > 1 )
            {
#if NO_FOLDER_HIERARCHY
                res = query.Select(LTY_BROWSELIST_MEDIAOBJECTS_COUNT_WITH_SUBFOLDERS, IN dbList->mDeviceID, "iitii",
                        IN ((strlen_r(dbList->mPath) > 1) ? 0 : 1),
                        IN strlen_r(dbList->mPath),
                        IN dbList->mPath,
                        IN catType1, IN catType2);
#else
                /* trim last character if it is "/" */
                tURL folderName;
                strncpy_r(OUT folderName, IN dbList->mPath, IN sizeof(folderName));
                if(FastUTF8::EndsWithNC((const FastUTF8::tString)folderName, (const FastUTF8::tString)"/")) //lint !e1773
                {
                    unsigned char *pureFile;
                    FastUTF8::Split(OUT pureFile, INOUT (unsigned char *)folderName); //lint !e1773
                }
                VARTRACE(folderName);

                res = query.Select(LTY_BROWSELIST_MEDIAOBJECTS_COUNT_WITH_SUBFOLDERS, IN dbList->mDeviceID, "tii",
                        IN folderName,
                        IN catType1, IN catType2);
#endif // #if NO_FOLDER_HIERARCHY
            }
            else // root folder -> select all / do not make string compare on path
            {
                res = query.Select(LTY_BROWSELIST_MEDIAOBJECTS_COUNT_ALL, IN dbList->mDeviceID, "ii",
                        IN catType1, IN catType2);
            }
        }
        else if(LTY_IMAGE_FOLDER_ITEM == dbList->mListType)
        {
            res = query.Select(LTY_IMAGE_FOLDER_ITEM_COUNT, IN dbList->mDeviceID, "t", IN dbList->mPath);
        }
        else if(LTY_BROWSELIST_MEDIAOBJECTS == dbList->mListType)
        {
#if FILTER_FILE_LIST_VIA_CTY
            tCategoryType catType1 = CTY_DIR;
            tCategoryType catType2 = CTY_DIR;
            tCategoryType catType3 = CTY_DIR;
            tCategoryType catType4 = CTY_DIR;
            GetCTYByFTS(OUT catType1, OUT catType2, OUT catType3, OUT catType4, IN dbList->mFileTypeSelection, IN true/*onlyPlayableObjects*/);

            res = query.Select(LTY_BROWSELIST_MEDIAOBJECTS_COUNT, IN dbList->mDeviceID, "tiii",
                    IN dbList->mPath,
                    IN catType1, IN catType2, IN CTY_DIR);
#else
            tMediaContent mediaContent = MTY_UNKNOWN;
            GetMediaContentByFTS(OUT mediaContent, IN dbList->mFileTypeSelection, IN true/*onlyPlayableObjects*/);

            res = query.Select(LTY_BROWSELIST_MEDIAOBJECTS_COUNT, IN dbList->mDeviceID, "ti",
                    IN dbList->mPath,
                    IN IN mediaContent);
#endif // #if FILTER_FILE_LIST_VIA_CTY
        }
        else //LTY_BROWSELIST
        {
            /* Set the media context filter related to FTS */
            if(FTS_ALL != dbList->mFileTypeSelection) //not FTS_ALL
            {
#if FILTER_FILE_LIST_VIA_CTY
                tCategoryType catType1 = CTY_DIR;
                tCategoryType catType2 = CTY_DIR;
                tCategoryType catType3 = CTY_DIR;
                tCategoryType catType4 = CTY_DIR;
                GetCTYByFTS(OUT catType1, OUT catType2, OUT catType3, OUT catType4, IN dbList->mFileTypeSelection, IN false/*onlyPlayableObjects*/);

                res = query.Select(LTY_BROWSELIST_COUNT, IN dbList->mDeviceID, "tiiiiii",
                        IN dbList->mPath,
                        IN 0, IN catType1, IN catType2, IN catType3, IN catType4, IN CTY_DIR);
#else
                tMediaContent mediaContent = MTY_UNKNOWN;
                GetMediaContentByFTS(OUT mediaContent, IN dbList->mFileTypeSelection, IN false/*onlyPlayableObjects*/);

                res = query.Select(LTY_BROWSELIST_COUNT, IN dbList->mDeviceID, "tii",
                        IN dbList->mPath,
                        IN 0, IN mediaContent);
#endif // #if FILTER_FILE_LIST_VIA_CTY
            }
            else
            {
                res = query.Select(LTY_BROWSELIST_COUNT, IN dbList->mDeviceID, "t", IN dbList->mPath);
            }
        }
        if (res) return res;

        while(1) {
            tCategoryType catType;
            res = query.Get("ii", OUT &catType, OUT &objectCount);
            if (res) break;

            switch(catType) {
            case CTY_DIR:
                numberOfFolders = objectCount;
                break;
            case CTY_SONG:
            case CTY_FAVORITES:
                numberOfAudioObjects = objectCount;
                numberOfMediaObjects += objectCount;
                break;
            case CTY_VIDEO:
                numberOfVideoObjects = objectCount;
                numberOfMediaObjects += objectCount;
                break;
            case CTY_PLAYLIST:
            case CTY_PLAYLIST_INTERNAL:
                numberOfPlayLists += objectCount;
                break;
            case CTY_IMAGE:
                numberOfImages = objectCount;
                break;
            default:
                break;
            }
        }
    }
    else
    {
        /* take real deviceID in case of a MyMedia playlist */
        tDeviceID realDeviceID = dbList->mDeviceID;
        if ((MY_MEDIA == dbList->mDeviceID)
            &&
            (dbList->mListType == LTY_FILELIST_PLAYLIST)) {

            res = GetObjectDeviceID(OUT realDeviceID, IN dbList->mFilterTag1.tag, CTY_PLAYLIST);
            if (res) return res;
        }

        /* get the device information*/
        tDeviceInfo deviceInfo;
        res = GetDeviceInfo(OUT deviceInfo, IN realDeviceID);
        if (res) return res;

        if(LTY_BLUETOOTH_FILELIST == dbList->mListType)
        {
            res = query.Select(LTY_BLUETOOTH_FILELIST_COUNT, MY_MEDIA, "tti", IN deviceInfo.mountPoint, IN dbList->mPath,IN LTY_BLUETOOTH_FILELIST_COUNT);
        }
        else if (LTY_CD == dbList->mListType )
        {
            res = query.Select(LTY_CD_COUNT, MY_MEDIA, "tt", IN IN deviceInfo.mountPoint, IN dbList->mPath);
        }
        else if(LTY_MTP_FILELIST == dbList->mListType)
        {
            res = query.Select(LTY_MTP_FILELIST_COUNT, MY_MEDIA, "tt", IN deviceInfo.mountPoint, IN dbList->mPath);
        }
        else if(LTY_FILELIST_MEDIAOBJECTS == dbList->mListType)
        {
            /* Set the media context filter related to FTS */
            tVTFileType fileType1 = FILE_TYPE_DIRECTORY;
            tVTFileType fileType2 = FILE_TYPE_DIRECTORY;
            tVTFileType fileType3 = FILE_TYPE_DIRECTORY;
            tVTFileType fileType4 = FILE_TYPE_DIRECTORY;

            GetVFTByFTS(OUT fileType1, OUT fileType2, OUT fileType3, OUT fileType4, IN dbList->mFileTypeSelection, IN true/*onlyPlayableObjects*/);

            res = query.Select(LTY_FILELIST_MEDIAOBJECTS_COUNT, MY_MEDIA, "ttiii",
                IN deviceInfo.mountPoint,
                IN dbList->mPath,
                IN fileType1,
                IN fileType2,
                IN FILE_TYPE_DIRECTORY);
        }
        else if(LTY_FILELIST_MEDIAOBJECTS_WITH_SUBFOLDERS == dbList->mListType)
        {
            /*prefix the path to indicate recursive request to VTFile*/
            tPath path;
            strncpy_r(OUT path, IN "recursive:", IN sizeof(path));
            strncat_r(OUT path, IN dbList->mPath, IN sizeof(path));

            /* Set the media context filter related to FTS */
            tVTFileType fileType1 = FILE_TYPE_DIRECTORY;
            tVTFileType fileType2 = FILE_TYPE_DIRECTORY;
            tVTFileType fileType3 = FILE_TYPE_DIRECTORY;
            tVTFileType fileType4 = FILE_TYPE_DIRECTORY;

            GetVFTByFTS(OUT fileType1, OUT fileType2, OUT fileType3, OUT fileType4, IN dbList->mFileTypeSelection, IN true/*onlyPlayableObjects*/);

            res = query.Select(LTY_FILELIST_MEDIAOBJECTS_COUNT_WITH_SUBFOLDERS, MY_MEDIA, "ttii",
                IN deviceInfo.mountPoint,
                IN path,
                IN fileType1,
                IN fileType2);
        }
        else //LTY_FILELIST or LTY_FILELIST_UNSORTED or LTY_FILELIST_SORTED_FOR_INDEXING or LTY_FILELIST_PLAYLIST
        {
            res = query.Select(LTY_FILELIST_COUNT, MY_MEDIA, "tt", IN deviceInfo.mountPoint, IN dbList->mPath);
        }
        if (res) return res;

        if(LTY_BLUETOOTH_FILELIST == dbList->mListType)
        {
            res = query.Get("i", OUT &objectCount);
        }
        else{
            while(1) {
                tFileType fileType;
                res = query.Get("ii", OUT &fileType, OUT &objectCount);
                if (res) break;

                switch(fileType) {
                    case FT_FOLDER:
                        numberOfFolders = objectCount;
                        break;
                    case FT_AUDIO:
                        numberOfAudioObjects = objectCount;
                        numberOfMediaObjects += objectCount;
                        break;
                    case FT_VIDEO:
                        numberOfVideoObjects = objectCount;
                        numberOfMediaObjects += objectCount;
                        break;
                    case FT_PLAYLIST:
                        numberOfPlayLists = objectCount;
                        break;
                    case FT_IMAGE:
                        numberOfImages = objectCount;
                        break;
                    case FT_UNKNOWN:
                    default:
                        break;
                }
            }
        }
    }

    query.End();

    /* in case of filelist with only media objects update the list size as mediaobjects count */
    if((LTY_FILELIST_MEDIAOBJECTS_WITH_SUBFOLDERS == dbList->mListType)
       ||
       (LTY_BROWSELIST_MEDIAOBJECTS_WITH_SUBFOLDERS == dbList->mListType)
       ||
       (LTY_FILELIST_PLAYLIST == dbList->mListType))
    {
        listSize = numberOfMediaObjects;
    }
    else if((LTY_FILELIST_MEDIAOBJECTS == dbList->mListType)
            ||
            (LTY_BROWSELIST_MEDIAOBJECTS == dbList->mListType))
    {
        listSize = numberOfFolders + numberOfMediaObjects;
    }
    else if(LTY_IMAGE_FOLDER_ITEM == dbList->mListType)
    {
        listSize = numberOfImages;
    }
    else if((LTY_FILELIST == dbList->mListType)
            ||
            (LTY_BROWSELIST == dbList->mListType))
    {
        switch(dbList->mFileTypeSelection)
        {
            case FTS_AUDIO_VIDEO_PLAYLIST:
                listSize = numberOfFolders + numberOfMediaObjects + numberOfPlayLists;
                break;
            case FTS_AUDIO:
                listSize = numberOfFolders + numberOfAudioObjects;
                break;
            case FTS_VIDEO:
                listSize = numberOfFolders + numberOfVideoObjects;
                break;
            case FTS_PLAYLIST:
                listSize = numberOfFolders + numberOfPlayLists;
                break;
            case FTS_IMAGE:
                listSize = numberOfFolders + numberOfImages;
                break;
            case FTS_AUDIO_PLAYLIST:
                listSize = numberOfFolders + numberOfAudioObjects + numberOfPlayLists;
                numberOfMediaObjects = numberOfAudioObjects;
                break;
            case FTS_ALL:
            default:
                listSize = numberOfFolders + numberOfMediaObjects + numberOfPlayLists + numberOfImages;
                break;
        }
    }
    else if(LTY_BLUETOOTH_FILELIST == dbList->mListType)
    {
        listSize = objectCount;
    }
    else    // otherwise list size is sum of all items count e.g LTY_FILELIST_UNSORTED
    {
        listSize = numberOfFolders + numberOfMediaObjects + numberOfPlayLists + numberOfImages;
    }

    return MP_NO_ERROR;
}

tResult DBManager::CreateMediaPlayerFileList(tListID &listID, tListType &listType, const tPath path, const tDeviceID deviceID,
        const tStreaming streaming/*=false*/, const tFileTypeSelection fileTypeSelection/*=FTS_AUDIO_VIDEO_PLAYLIST*/)
{
    ENTRY

    /* check requested list type */
    if((LTY_FILELIST != listType)
       &&
       (LTY_FILELIST_UNSORTED != listType)
       &&
       (LTY_FILELIST_SORTED_FOR_INDEXING != listType)
       &&
       (LTY_FILELIST_PLAYLIST != listType)
       &&
       (LTY_PLAYLIST_SONG != listType)
       &&
       (LTY_BROWSELIST != listType)
       &&
       (LTY_MTP_FILELIST != listType)
       &&
       (LTY_BLUETOOTH_FILELIST != listType)
       &&
       (LTY_CD != listType)
       &&
       (LTY_IMAGE_FOLDER_ITEM != listType))
    {
        return MP_ERR_DB_UNSUPPORTED_LIST_TYPE;
    }

    tResult res;
    DBList *dbList = NULL;
    UnlockOnExit unlockQuery(&dbList);
    tBoolean useOldList = false;

    if(LIST_ID_NONE == listID) {

        /* create a list context
         * mQuery = new Query() + mQuery->DoLock() is done in CreateDBList() */
        dbList = CreateDBList();
        if (!dbList) return MP_ERR_DB_CREATE_LIST_CONTEXT;

        /* set the returning listID */
        listID = dbList->mListID;
    }
    else { //use old list
        useOldList = true;
    }

    /* get list pointer */
    //dbList = LookupDBList(IN listID);
    GET_DBLIST_POINTER(dbList, listID);
    if (!dbList) return MP_ERR_DB_LIST_NOT_FOUND;

    if(useOldList) { //use old list

        /* lock the query */
        dbList->mQuery->DoLock();

        /* save the current row */
        res = dbList->mQuery->Save();
        if (res) return res;
    }

    /* remember the list creation parms for later storage in last mode table */
    dbList->mDeviceID = deviceID;
    dbList->mListType = listType;
    dbList->mPlayContinuation = PC_NO_REPEAT;
    strncpy_r(dbList->mPath, path, sizeof(dbList->mPath));
    dbList->mFileTypeSelection = fileTypeSelection;
    dbList->mStreaming = streaming;

    /* mapping of playlist */
    if (LTY_PLAYLIST_SONG == dbList->mListType) {
        dbList->mListType = LTY_FILELIST_PLAYLIST;
    }

    /* get the device information*/
    tDeviceInfo deviceInfo;
    res = GetDeviceInfo(OUT deviceInfo, IN deviceID);
    if (res) return res;

    /* in case of device is already indexed use folder database instead of VTFile */
    if(LTY_BROWSELIST == listType || LocalSPM::GetDataProvider().IsFileBrowsingByDb(IN deviceInfo,IN path,IN dbList->mListType))
    {
        dbList->mListType = LTY_BROWSELIST;

        /* add "/" if missing */
        if(NULL == FastUTF8::EndsWithNC((const FastUTF8::tString)path, (const FastUTF8::tString)"/")) //lint !e1773
        {
            strncat_r(dbList->mPath, "/", sizeof(dbList->mPath));
        }

        /* Set the media context filter related to FTS */
        if(FTS_ALL != fileTypeSelection) //not FTS_ALL
        {
#if FILTER_FILE_LIST_VIA_CTY
            tCategoryType catType1 = CTY_DIR;
            tCategoryType catType2 = CTY_DIR;
            tCategoryType catType3 = CTY_DIR;
            tCategoryType catType4 = CTY_DIR;
            GetCTYByFTS(OUT catType1, OUT catType2, OUT catType3, OUT catType4, IN fileTypeSelection, IN false/*onlyPlayableObjects*/);

            res = dbList->mQuery->Select(IN dbList->mListType, IN deviceID, "tiiiiii",
                    IN dbList->mPath,
                    IN 0, IN catType1, IN catType2, IN catType3, IN catType4, IN CTY_DIR);
#else
            tMediaContent mediaContent = MTY_UNKNOWN;
            GetMediaContentByFTS(OUT mediaContent, IN fileTypeSelection, IN false/*onlyPlayableObjects*/);

            res = dbList->mQuery->Select(IN dbList->mListType, IN deviceID, "tii",
                    IN dbList->mPath,
                    IN 0, IN mediaContent);
#endif // #if FILTER_FILE_LIST_VIA_CTY
        }
        else
        {
            res = dbList->mQuery->Select(IN dbList->mListType, IN deviceID, "t", IN dbList->mPath);
        }
        if (res) return res;
    }
    else if(LTY_MTP_FILELIST == dbList->mListType)
    {
        res = dbList->mQuery->Select(IN dbList->mListType, MY_MEDIA, "tti", IN deviceInfo.mountPoint, IN path, IN FILE_TYPE_UNKNOWN);
        if (res) return res;
    }
    else if(LTY_BLUETOOTH_FILELIST == dbList->mListType)
    {
        res = dbList->mQuery->Select(IN dbList->mListType, MY_MEDIA, "ttii", IN deviceInfo.mountPoint, IN path,IN LTY_BLUETOOTH_FILELIST,IN FILE_TYPE_UNKNOWN);
        if (res) return res;
    }
    else if((LTY_FILELIST_UNSORTED == dbList->mListType)
            ||
            (LTY_FILELIST_PLAYLIST == dbList->mListType))
    {
        res = dbList->mQuery->Select(IN LTY_FILELIST_UNSORTED, MY_MEDIA, "tti", IN deviceInfo.mountPoint, IN path, IN FILE_TYPE_UNKNOWN);
        if (res) return res;
    }
    else if(LTY_FILELIST_SORTED_FOR_INDEXING == dbList->mListType)
    {
        /* Set the media context filter related to FTS */
        tVTFileType fileType1 = FILE_TYPE_DIRECTORY;
        tVTFileType fileType2 = FILE_TYPE_DIRECTORY;
        tVTFileType fileType3 = FILE_TYPE_DIRECTORY;
        tVTFileType fileType4 = FILE_TYPE_DIRECTORY;

        GetVFTByFTS(OUT fileType1, OUT fileType2, OUT fileType3, OUT fileType4, IN fileTypeSelection, IN false/*onlyPlayableObjects*/);

        res = dbList->mQuery->Select(IN dbList->mListType, MY_MEDIA, "ttiiiiii",
                IN deviceInfo.mountPoint,
                IN path,
                IN 0, IN fileType1, IN fileType2, IN fileType3, IN fileType4, IN FILE_TYPE_DIRECTORY);
        if (res) return res;
    }
    else if(LTY_CD == dbList->mListType)
    {
        res = dbList->mQuery->Select(IN dbList->mListType, MY_MEDIA, "tt", IN deviceInfo.mountPoint, IN path);
        if (res) return res;
    }
    else if(LTY_IMAGE_FOLDER_ITEM == dbList->mListType)
    {
        ETG_TRACE_USR2(("Gettitle: deviceID=%d", deviceID));
        ETG_TRACE_USR2(("Getpath: Path=%s", path));
        res = dbList->mQuery->Select(IN dbList->mListType, IN deviceID , "t", IN path);
        if (res) return res;
    }
    else //LTY_FILELIST
    {
        dbList->mListType = LTY_FILELIST;

        /* Set the media context filter related to FTS */
        tVTFileType fileType1 = FILE_TYPE_DIRECTORY;
        tVTFileType fileType2 = FILE_TYPE_DIRECTORY;
        tVTFileType fileType3 = FILE_TYPE_DIRECTORY;
        tVTFileType fileType4 = FILE_TYPE_DIRECTORY;

        GetVFTByFTS(OUT fileType1, OUT fileType2, OUT fileType3, OUT fileType4, IN fileTypeSelection, IN false/*onlyPlayableObjects*/);

        res = dbList->mQuery->Select(IN dbList->mListType, MY_MEDIA, "ttiiiiii",
                IN deviceInfo.mountPoint,
                IN path,
                IN 0, IN fileType1, IN fileType2, IN fileType3, IN fileType4, IN FILE_TYPE_DIRECTORY);
        if (res) return res;
    }

    if(useOldList) { //use old list

        /* restore the current row */
        res = dbList->mQuery->Restore();
        if (res) return res;
    }

    /* set the returning listType (but don't change playlist type) */
    if (LTY_PLAYLIST_SONG != listType) {
        listType = dbList->mListType;
    }

    return MP_NO_ERROR;
}

tResult DBManager::CreateFileListOfMediaObjects(tListID &listID, tListType &listType, const tPath path,
        const tDeviceID deviceID, const tStreaming streaming /*=false*/, const tFileTypeSelection fileTypeSelection/*=FTS_AUDIO_VIDEO_PLAYLIST*/)
{
    ENTRY

    /* check requested list type */
    if((LTY_FILELIST_MEDIAOBJECTS != listType)
       &&
       (LTY_BROWSELIST_MEDIAOBJECTS != listType)
       &&
       (LTY_FILELIST_MEDIAOBJECTS_WITH_SUBFOLDERS != listType)
       &&
       (LTY_BROWSELIST_MEDIAOBJECTS_WITH_SUBFOLDERS != listType))
    {
        return MP_ERR_DB_UNSUPPORTED_LIST_TYPE;
    }

    /* check requested file type selection */
    if((FTS_AUDIO_VIDEO_PLAYLIST != fileTypeSelection)
       &&
       (FTS_AUDIO != fileTypeSelection)
       &&
       (FTS_VIDEO != fileTypeSelection)
       &&
       (FTS_AUDIO_PLAYLIST != fileTypeSelection)
       &&
       (FTS_ALL != fileTypeSelection))
    {
        return MP_ERR_DB_INVALID_FTS;
    }

    tResult res;
    DBList *dbList = NULL;
    UnlockOnExit unlockQuery(&dbList);
    tBoolean useOldList = false;

    if(LIST_ID_NONE == listID) {

        /* create a list context
         * mQuery = new Query() + mQuery->DoLock() is done in CreateDBList() */
        dbList = CreateDBList();
        if (!dbList) return MP_ERR_DB_CREATE_LIST_CONTEXT;

        /* set the returning listID */
        listID = dbList->mListID;
    }
    else { //use old list
        useOldList = true;
    }

    /* get list pointer */
    //dbList = LookupDBList(IN listID);
    GET_DBLIST_POINTER(dbList, listID);
    if (!dbList) return MP_ERR_DB_LIST_NOT_FOUND;

    if(useOldList) { //use old list

        /* lock the query */
        dbList->mQuery->DoLock();

        /* save the current row */
        res = dbList->mQuery->Save();
        if (res) return res;
    }

    /* remember the list creation parameters for later storage */
    dbList->mDeviceID = deviceID;
    dbList->mListType = listType;
    dbList->mPlayContinuation = PC_NO_REPEAT;
    strncpy_r(dbList->mPath, path, sizeof(dbList->mPath));
    dbList->mFileTypeSelection = fileTypeSelection;
    dbList->mStreaming = streaming;

    /* get the device information*/
    tDeviceInfo deviceInfo;
    res = GetDeviceInfo(OUT deviceInfo, IN deviceID);
    if (res) return res;

    /* in case of device is already indexed use folder database instead of VTFile */
    if((LocalSPM::GetDataProvider().DBFileBrowsingByDB())
       &&
       ((LocalSPM::GetDataProvider().DBFileListWhileIndexing()) || (IDS_COMPLETE == deviceInfo.indexedState) || (IDS_COMPLETE_FULL_DB == deviceInfo.indexedState)))
    {
        /* add "/" if missing */
        if(NULL == FastUTF8::EndsWithNC((const FastUTF8::tString)path, (const FastUTF8::tString)"/")) //lint !e1773
        {
            strncat_r(dbList->mPath, "/", sizeof(dbList->mPath));
        }

        tCategoryType catType1 = CTY_DIR;
        tCategoryType catType2 = CTY_DIR;
        tCategoryType catType3 = CTY_DIR;
        tCategoryType catType4 = CTY_DIR;
        GetCTYByFTS(OUT catType1, OUT catType2, OUT catType3, OUT catType4, IN fileTypeSelection, IN true/*onlyPlayableObjects*/);

        if((LTY_FILELIST_MEDIAOBJECTS_WITH_SUBFOLDERS == listType)
           ||
           (LTY_BROWSELIST_MEDIAOBJECTS_WITH_SUBFOLDERS == listType))
        {
            dbList->mListType = LTY_BROWSELIST_MEDIAOBJECTS_WITH_SUBFOLDERS;

            if( strlen_r(dbList->mPath) > 1 )
            {
#if NO_FOLDER_HIERARCHY
                res = dbList->mQuery->Select(IN dbList->mListType, IN deviceID, "iitii",
                        IN ((strlen_r(dbList->mPath) > 1) ? 0 : 1),
                        IN strlen_r(dbList->mPath),
                        IN dbList->mPath,
                        IN catType1, IN catType2);
#else
                /* trim last character if it is "/" */
                tURL folderName;
                strncpy_r(OUT folderName, IN dbList->mPath, IN sizeof(folderName));
                if(FastUTF8::EndsWithNC((const FastUTF8::tString)folderName, (const FastUTF8::tString)"/")) //lint !e1773
                {
                    unsigned char *pureFile;
                    FastUTF8::Split(OUT pureFile, INOUT (unsigned char *)folderName); //lint !e1773
                }
                VARTRACE(folderName);

                res = dbList->mQuery->Select(IN dbList->mListType, IN deviceID, "tii",
                        IN folderName,
                        IN catType1, IN catType2);
#endif // #if NO_FOLDER_HIERARCHY
            }
            else // root folder -> select all / do not make string compare on path
            {
                res = dbList->mQuery->Select(LTY_BROWSELIST_MEDIAOBJECTS_ALL, IN deviceID, "ii",
                        IN catType1, IN catType2);
            }
        }
        else
        {
            dbList->mListType = LTY_BROWSELIST_MEDIAOBJECTS;

#if FILTER_FILE_LIST_VIA_CTY
            res = dbList->mQuery->Select(IN dbList->mListType, IN deviceID, "tiii",
                    IN dbList->mPath,
                    IN catType1, IN catType2, IN CTY_DIR);
#else
            tMediaContent mediaContent = MTY_UNKNOWN;
            GetMediaContentByFTS(OUT mediaContent, IN fileTypeSelection, IN true/*onlyPlayableObjects*/);

            res = dbList->mQuery->Select(IN dbList->mListType, IN deviceID, "ti",
                    IN dbList->mPath,
                    IN IN mediaContent);
#endif // #if FILTER_FILE_LIST_VIA_CTY
        }
        if (res) return res;
    }
    else
    {
        /* Set the media context filter related to FTS */
        tVTFileType fileType1 = FILE_TYPE_DIRECTORY;
        tVTFileType fileType2 = FILE_TYPE_DIRECTORY;
        tVTFileType fileType3 = FILE_TYPE_DIRECTORY;
        tVTFileType fileType4 = FILE_TYPE_DIRECTORY;

        GetVFTByFTS(OUT fileType1, OUT fileType2, OUT fileType3, OUT fileType4, IN fileTypeSelection, IN true/*onlyPlayableObjects*/);

        if((LTY_FILELIST_MEDIAOBJECTS_WITH_SUBFOLDERS == listType)
           ||
           (LTY_BROWSELIST_MEDIAOBJECTS_WITH_SUBFOLDERS == listType))
        {
            /*prefix the path to indicate recursive request to VTFile*/
            tPath newPath;
            strncpy_r(newPath, "recursive:" ,sizeof(newPath));
            strncat_r(newPath, dbList->mPath,sizeof(newPath));
            dbList->mListType = LTY_FILELIST_MEDIAOBJECTS_WITH_SUBFOLDERS;

            res = dbList->mQuery->Select(IN dbList->mListType, IN deviceID, "ttii",
                    IN deviceInfo.mountPoint,
                    IN newPath,
                    IN fileType1,
                    IN fileType2);
        }
        else
        {
            dbList->mListType = LTY_FILELIST_MEDIAOBJECTS;

            res = dbList->mQuery->Select(IN dbList->mListType, IN deviceID, "ttiii",
                    IN deviceInfo.mountPoint,
                    IN dbList->mPath,
                    IN fileType1,
                    IN fileType2,
                    IN FILE_TYPE_DIRECTORY);
        }
        if (res) return res;
    }

    if(useOldList) { //use old list

        /* restore the current row */
        res = dbList->mQuery->Restore();
        if (res) return res;
    }

    /* set the returning listType */
    listType = dbList->mListType;

    return MP_NO_ERROR;
}

tResult DBManager::DeleteVTFileCache(const tDeviceID deviceID)
{
    ENTRY
    tResult res;

    /*fetch device details*/
    tDeviceInfo deviceInfo;
    res = GetDeviceInfo(OUT deviceInfo, IN deviceID);
    if (res) return res;

#if 0

    Query query;

    /* query delete cache from VTFile */
    res = query.Delete(LTY_FILELIST_DELETE);
    if (res) return res;

    res = query.Remove("it",0,deviceInfo.mountPoint);
    if (res) return res;
#else
    // delete the cache directly till we find a way to delete cache using sql delete for virtual table (xUpdate)
    extern bool FSCacheDelete(const char *path);
    FSCacheDelete(deviceInfo.mountPoint);
#endif

    return MP_NO_ERROR;
}

tResult DBManager::GetSearchKeyList(vector<tSearchKey> &searchKeys, const tListID listID)
{
    //TODO: update query for all list types
    ENTRY
    tResult res;
    DBList *dbList = NULL;
    Query query;

    /* get list pointer */
    //dbList = LookupDBList(IN listID);
    GET_DBLIST_POINTER(dbList, listID);
    if (!dbList) return MP_ERR_DB_LIST_NOT_FOUND;

    /* No search keyboard support on VTIPOD lists */
    if(dbList->mIsVTIPODList) {
        return MP_ERR_DB_SEARCHKEYS_NOT_SUPPORTED;
    }

    /* select and issue a suitable query to get the list of all the first letter along with occurance of each letter */
    tListType searchKeysListType;
    switch(dbList->mListType)
    {
        case LTY_SONG:
            searchKeysListType = LTY_SONG_SEARCH_KEY;
            break;
        case LTY_ARTIST:
            searchKeysListType = LTY_ARTIST_SEARCH_KEY;
            break;
        case LTY_GENRE:
            searchKeysListType = LTY_GENRE_SEARCH_KEY;
            break;
        case LTY_ALBUM:
            searchKeysListType = LTY_ALBUM_SEARCH_KEY;
            break;
        case LTY_GENRE_SONG:
            searchKeysListType = LTY_GENRE_SONG_SEARCH_KEY;
            break;
        case LTY_COMPOSER:
            searchKeysListType = LTY_COMPOSER_SEARCH_KEY;
            break;
        case LTY_GENRE_ARTIST:
            searchKeysListType = LTY_GENRE_ARTIST_SEARCH_KEY;
            break;
        case LTY_GENRE_ARTIST_ALBUM:
            searchKeysListType = LTY_GENRE_ARTIST_ALBUM_SEARCH_KEY;
            break;
        case LTY_GENRE_ARTIST_SONG:
            searchKeysListType = LTY_GENRE_ARTIST_SONG_SEARCH_KEY;
            break;
        case LTY_GENRE_ALBUM:
            searchKeysListType = LTY_GENRE_ALBUM_SEARCH_KEY;
            break;
        case LTY_YEAR_ALBUM:
            searchKeysListType = LTY_YEAR_ALBUM_SEARCH_KEY;
            break;
        case LTY_ARTIST_ALBUM:
            searchKeysListType = LTY_ARTIST_ALBUM_SEARCH_KEY;
            break;
        case LTY_ARTIST_SONG:
            searchKeysListType = LTY_ARTIST_SONG_SEARCH_KEY;
            break;
        case LTY_PLAYLIST:
            searchKeysListType = LTY_PLAYLIST_SEARCH_KEY;
            break;
        case LTY_COMPOSER_ALBUM:
            searchKeysListType = LTY_COMPOSER_ALBUM_SEARCH_KEY;
            break;
        case LTY_COMPOSER_SONG:
            searchKeysListType = LTY_COMPOSER_SONG_SEARCH_KEY;
            break;
        case LTY_PODCAST:
            searchKeysListType = LTY_PODCAST_SEARCH_KEY;
            break;
        case LTY_AUDIOBOOK:
            searchKeysListType = LTY_AUDIOBOOK_SEARCH_KEY;
            break;
        case LTY_VIDEO:
            searchKeysListType = LTY_VIDEO_SEARCH_KEY;
            break;
        case LTY_VIDEO_EPISODE:
            searchKeysListType = LTY_VIDEO_EPISODE_SEARCH_KEY;
            break;
        case LTY_COMPILATION:
            searchKeysListType = LTY_COMPILATION_SEARCH_KEY;
            break;
        case LTY_AUTHOR:
        case LTY_AUTHOR_BOOKTITLE:
        case LTY_GENRE_ARTIST_ALBUM_SONG:   //Keep native order of album tracks
        case LTY_GENRE_ALBUM_SONG:
        case LTY_YEAR_ALBUM_SONG:
        case LTY_ARTIST_ALBUM_SONG:
        case LTY_ALBUM_SONG:
        case LTY_COMPOSER_ALBUM_SONG:
        case LTY_PLAYLIST_SONG:             //Not for songs of playlist because it has no alphabetic ordering
        case LTY_FILELIST_PLAYLIST:         //Not for songs of playlist because it has no alphabetic ordering
        case LTY_PODCAST_EPISODE:           //query available , but disabled for Podcast episodes
        case LTY_EPISODE_OF_VIDEO:
        case LTY_BOOKTITLE_CHAPTER:
        case LTY_AUTHOR_BOOKTITLE_CHAPTER:
        case LTY_CURRENT_SELECTION:         //Unknown order
        case LTY_FILELIST:                  //Not for FILELIST because of ordering by file type
        case LTY_FILELIST_UNSORTED:
        case LTY_FILELIST_MEDIAOBJECTS:
        case LTY_FILELIST_MEDIAOBJECTS_WITH_SUBFOLDERS:
        case LTY_BROWSELIST:                //Not for BROWSELIST because of ordering by category type
        case LTY_BROWSELIST_MEDIAOBJECTS:
        case LTY_BROWSELIST_MEDIAOBJECTS_WITH_SUBFOLDERS:
        case LTY_END_OF_LTY:
        case LTY_MTP_FILELIST:
        case LTY_ALBUM_UNKNOWN_ALBUMART:
        case LTY_YEAR:
            //No searchkeyboard for any of the above LTYs
            return MP_ERR_DB_SEARCHKEYS_NOT_SUPPORTED;
            break;
        default:
            return MP_ERR_DB_UNSUPPORTED_LIST_TYPE;
            break;
    }

    /* prepare the sql statement to get the list*/
    tGeneralString formatBuffer;
    SelectFormatString(INOUT formatBuffer, IN dbList->mFilterTag1, IN dbList->mFilterTag2, IN dbList->mFilterTag3, IN dbList->mFilterTag4,IN dbList->mSearchString);
    res = query.Select(IN searchKeysListType, IN dbList->mDeviceID, IN formatBuffer,
            IN dbList->mFilterTag1.tag,
            IN dbList->mFilterTag2.tag,
            IN dbList->mFilterTag3.tag,
            IN dbList->mFilterTag4.tag);
    if (res) return res;

    /* clear the vector */
    searchKeys.clear();

    /* get all infos */
    while(1)
    {
        /* get the count */
        tSearchKey charItem;
        res = query.Get("Ti",
                OUT charItem.letter, IN sizeof(charItem.letter),
                OUT &charItem.count);
        if (res) break;

        searchKeys.push_back(charItem);
    }
    query.End();

    if (res && (res != MP_ERR_DB_END_OF_LIST)) return res;

    return MP_NO_ERROR;
}

tResult DBManager::GetListSize(tListSize &listSize, const tListID listID, const unsigned int uMinimalExpectedSize /*=DB_UNDEFINED_VALUE*/,const unsigned int uOffsetForSearch /*=DB_UNDEFINED_VALUE*/)
{
    ENTRY
    tResult res;
    DBList *dbList = NULL;
    Query query;

    tBool bIsSearchList = false;

    /* get list pointer */
    //dbList = LookupDBList(IN listID);
    GET_DBLIST_POINTER(dbList, listID);
    if (!dbList) return MP_ERR_DB_LIST_NOT_FOUND;

    /* issue a query to get the count for a given list type */
    tListType countListType;
    switch(dbList->mListType) {

    //----------------------------------------
    case LTY_GENRE:
        countListType = LTY_GENRE_COUNT;
        break;
    //Note: Search:All entities  with 'LTY..._SEARCH'' have been added for 'Roadmap 16014 full text'
    case LTY_GENRE_SEARCH:
        countListType = LTY_GENRE_SEARCH_COUNT;
        bIsSearchList = true;
        break;
    //----------------------------------------
    case LTY_ARTIST:
        countListType = LTY_ARTIST_COUNT;
        break;
    case LTY_ARTIST_SEARCH:
        countListType = LTY_ARTIST_SEARCH_COUNT;
        bIsSearchList = true;
        break;
    //----------------------------------------
    case LTY_ALBUM:
        countListType = LTY_ALBUM_COUNT;
        break;
    case LTY_ALBUM_SEARCH:
        countListType = LTY_ALBUM_SEARCH_COUNT;
        bIsSearchList = true;
        break;
    //-----------------------------------------
    case LTY_YEAR:
        countListType = LTY_YEAR_COUNT;
        break;
    case LTY_YEAR_ALBUM:
        countListType = LTY_YEAR_ALBUM_COUNT;
        break;
    case LTY_YEAR_ALBUM_SONG:
        countListType = LTY_YEAR_ALBUM_SONG_COUNT;
        break;
    case LTY_YEAR_ALBUM_SEARCH:
        countListType = LTY_YEAR_ALBUM_SEARCH_COUNT;
        bIsSearchList = true;
        break;
    case LTY_YEAR_ALBUM_SONG_SEARCH:
        countListType = LTY_YEAR_ALBUM_SONG_SEARCH_COUNT;
        bIsSearchList = true;
        break;
    //----------------------------------------
    case LTY_SONG:
        countListType = LTY_SONG_COUNT;
        break;
    case LTY_SONG_SEARCH:
        countListType = LTY_SONG_SEARCH_COUNT;
        bIsSearchList = true;
        break;
    //----------------------------------------
    case LTY_ALBUM_UNKNOWN_ALBUMART:
        countListType = LTY_ALBUM_UNKNOWN_ALBUMART_COUNT;
        break;
    //Note: not fts
    //----------------------------------------
    case LTY_GENRE_ARTIST:
        countListType = LTY_GENRE_ARTIST_COUNT;
        break;
    case LTY_GENRE_ARTIST_SEARCH:
         countListType = LTY_GENRE_ARTIST_SEARCH_COUNT;
         bIsSearchList = true;
         break;
    //----------------------------------------
    case LTY_GENRE_ARTIST_ALBUM:
        countListType = LTY_GENRE_ARTIST_ALBUM_COUNT;
        break;
    case LTY_GENRE_ARTIST_ALBUM_SEARCH:
           countListType = LTY_GENRE_ARTIST_ALBUM_SEARCH_COUNT;
           bIsSearchList = true;
           break;
    //-------------------------------------
    case LTY_GENRE_ARTIST_ALBUM_SONG:
        countListType = LTY_GENRE_ARTIST_ALBUM_SONG_COUNT;
        break;
    case LTY_GENRE_ARTIST_ALBUM_SONG_SEARCH:
        countListType = LTY_GENRE_ARTIST_ALBUM_SONG_SEARCH_COUNT;
        bIsSearchList = true;
        break;
   //-------------------------------------
    case LTY_GENRE_ARTIST_SONG:
        countListType = LTY_GENRE_ARTIST_SONG_COUNT;
        break;
    case LTY_GENRE_ARTIST_SONG_SEARCH:
        countListType = LTY_GENRE_ARTIST_SONG_SEARCH_COUNT;
        bIsSearchList = true;
        break;
     //-------------------------------------
    case LTY_GENRE_ALBUM:
        countListType = LTY_GENRE_ALBUM_COUNT;
        break;
    case LTY_GENRE_ALBUM_SEARCH:
        countListType = LTY_GENRE_ALBUM_SEARCH_COUNT;
        bIsSearchList = true;
        break;
    //-------------------------------------
    case LTY_GENRE_ALBUM_SONG:
        countListType = LTY_GENRE_ALBUM_SONG_COUNT;
        break;
    case LTY_GENRE_ALBUM_SONG_SEARCH:
         countListType = LTY_GENRE_ALBUM_SONG_SEARCH_COUNT;
         bIsSearchList = true;
         break;
    //-------------------------------------
    case LTY_GENRE_SONG:
        countListType = LTY_GENRE_SONG_COUNT;
        break;
    case LTY_GENRE_SONG_SEARCH:
        countListType = LTY_GENRE_SONG_SEARCH_COUNT;
        bIsSearchList = true;
        break;
    //-------------------------------------
    case LTY_ARTIST_ALBUM:
        countListType = LTY_ARTIST_ALBUM_COUNT;
        break;
    case LTY_ARTIST_ALBUM_SEARCH:
        countListType = LTY_ARTIST_ALBUM_SEARCH_COUNT;
        bIsSearchList = true;
        break;
    //-------------------------------------
    case LTY_ARTIST_ALBUM_SONG:
        countListType = LTY_ARTIST_ALBUM_SONG_COUNT;
        break;
    case LTY_ARTIST_ALBUM_SONG_SEARCH:
        countListType = LTY_ARTIST_ALBUM_SONG_SEARCH_COUNT;
        bIsSearchList = true;
        break;
    //-------------------------------------
    case LTY_ARTIST_SONG:
        countListType = LTY_ARTIST_SONG_COUNT;
        break;
    case LTY_ARTIST_SONG_SEARCH:
        countListType = LTY_ARTIST_SONG_SEARCH_COUNT;
        bIsSearchList = true;
        break;
    //-------------------------------------
    case LTY_ALBUM_SONG:
        countListType = LTY_ALBUM_SONG_COUNT;
        break;
    case LTY_ALBUM_SONG_SEARCH:
        countListType = LTY_ALBUM_SONG_SEARCH_COUNT;
        bIsSearchList = true;
        break;
    //-------------------------------------
    case LTY_PODCAST:
        countListType = LTY_PODCAST_COUNT;
        break;
    case LTY_PODCAST_SEARCH:
        countListType = LTY_PODCAST_SEARCH_COUNT;
        bIsSearchList = true;
        break;
    //-------------------------------------
    case LTY_PODCAST_EPISODE:
        countListType = LTY_PODCAST_EPISODE_COUNT;
        break;
    case LTY_PODCAST_EPISODE_SEARCH:
        countListType = LTY_PODCAST_EPISODE_SEARCH_COUNT;
        bIsSearchList = true;
        break;
    //-------------------------------------
    case LTY_AUDIOBOOK:
        countListType = LTY_AUDIOBOOK_COUNT;
        break;
    case LTY_AUDIOBOOK_SEARCH:
        countListType = LTY_AUDIOBOOK_SEARCH_COUNT;
        bIsSearchList = true;
        break;
    //-------------------------------------
    case LTY_BOOKTITLE_CHAPTER:
        countListType = LTY_BOOKTITLE_CHAPTER_COUNT;
        break;
    case LTY_BOOKTITLE_CHAPTER_SEARCH:
        countListType = LTY_BOOKTITLE_CHAPTER_SEARCH_COUNT;
        bIsSearchList = true;
        break;
    //-------------------------------------
    case LTY_AUTHOR:
        countListType = LTY_AUTHOR_COUNT;
        break;
    case LTY_AUTHOR_SEARCH:
         countListType = LTY_AUTHOR_SEARCH_COUNT;
         bIsSearchList = true;
         break;
    //-------------------------------------
    case LTY_AUTHOR_BOOKTITLE:
        countListType = LTY_AUTHOR_BOOKTITLE_COUNT;
        break;
    case LTY_AUTHOR_BOOKTITLE_SEARCH:
        countListType = LTY_AUTHOR_BOOKTITLE_SEARCH_COUNT;
        bIsSearchList = true;
        break;
    //-------------------------------------
    case LTY_AUTHOR_BOOKTITLE_CHAPTER:
        countListType = LTY_AUTHOR_BOOKTITLE_CHAPTER_COUNT;
        break;
    case LTY_AUTHOR_BOOKTITLE_CHAPTER_SEARCH:
        countListType = LTY_AUTHOR_BOOKTITLE_CHAPTER_SEARCH_COUNT;
        bIsSearchList = true;
        break;
    //-------------------------------------
    case LTY_COMPOSER:
        countListType = LTY_COMPOSER_COUNT;
        break;
    case LTY_COMPOSER_SEARCH:
        countListType = LTY_COMPOSER_SEARCH_COUNT;
        bIsSearchList = true;
        break;
    //-------------------------------------
    case LTY_COMPOSER_ALBUM:
        countListType = LTY_COMPOSER_ALBUM_COUNT;
        break;
    case LTY_COMPOSER_ALBUM_SEARCH:
        countListType = LTY_COMPOSER_ALBUM_SEARCH_COUNT;
        bIsSearchList = true;
        break;
    //-------------------------------------
    case LTY_COMPOSER_ALBUM_SONG:
        countListType = LTY_COMPOSER_ALBUM_SONG_COUNT;
        break;
    case LTY_COMPOSER_ALBUM_SONG_SEARCH:
        countListType = LTY_COMPOSER_ALBUM_SONG_SEARCH_COUNT;
        bIsSearchList = true;
        break;
    //-------------------------------------
    case LTY_COMPOSER_SONG:
        countListType = LTY_COMPOSER_SONG_COUNT;
        break;
    case LTY_COMPOSER_SONG_SEARCH:
        countListType = LTY_COMPOSER_SONG_SEARCH_COUNT;
        bIsSearchList = true;
        break;
    //-------------------------------------
    case LTY_VIDEO:
        countListType = LTY_VIDEO_COUNT;
        break;
    case LTY_VIDEO_SEARCH:
       countListType = LTY_VIDEO_SEARCH_COUNT;
       bIsSearchList = true;
       break;
    //-------------------------------------
    case LTY_VIDEO_EPISODE:
        countListType = LTY_VIDEO_EPISODE_COUNT;
        break;
    case LTY_VIDEO_EPISODE_SEARCH:
        countListType = LTY_VIDEO_EPISODE_COUNT;
        bIsSearchList = true;
        break;
    //-------------------------------------
    case LTY_EPISODE_OF_VIDEO:
        countListType = LTY_EPISODE_OF_VIDEO_COUNT;
        break;
    case LTY_EPISODE_OF_VIDEO_SEARCH:
        countListType = LTY_EPISODE_OF_VIDEO_SEARCH_COUNT;
        bIsSearchList = true;
        break;
    //-------------------------------------
    case LTY_PLAYLIST:
        countListType = LTY_PLAYLIST_COUNT;
        break;
    case LTY_PLAYLIST_SEARCH:
        countListType = LTY_PLAYLIST_COUNT;
        bIsSearchList = true;
        break;
    //-------------------------------------
    case LTY_PLAYLIST_INTERNAL:
        countListType = LTY_PLAYLIST_INTERNAL_COUNT;
        break;
    case LTY_PLAYLIST_INTERNAL_SEARCH:
        countListType = LTY_PLAYLIST_INTERNAL_SEARCH_COUNT;
        bIsSearchList = true;
        break;
    //-------------------------------------
    case LTY_PLAYLIST_SONG:
        countListType = LTY_PLAYLIST_SONG_COUNT;
        break;
    case LTY_PLAYLIST_SONG_SEARCH:
        countListType = LTY_PLAYLIST_SONG_SEARCH_COUNT;
        bIsSearchList = true;
        break;
    //-------------------------------------
    case LTY_FILELIST:
    case LTY_FILELIST_UNSORTED:
    case LTY_FILELIST_SORTED_FOR_INDEXING:
    case LTY_FILELIST_PLAYLIST:
        countListType = LTY_FILELIST_COUNT;
        break;
    case LTY_FILELIST_MEDIAOBJECTS:
        countListType = LTY_FILELIST_MEDIAOBJECTS_COUNT;
        break;
    case LTY_FILELIST_MEDIAOBJECTS_WITH_SUBFOLDERS:
        countListType = LTY_FILELIST_MEDIAOBJECTS_COUNT_WITH_SUBFOLDERS;
        break;
    case LTY_BLUETOOTH_FILELIST:
        countListType = LTY_BLUETOOTH_FILELIST_COUNT;
        break;
    case LTY_MTP_FILELIST:
        countListType = LTY_MTP_FILELIST_COUNT;
        break;
    case LTY_BROWSELIST:
        countListType = LTY_BROWSELIST_COUNT;
        break;
    case LTY_BROWSELIST_MEDIAOBJECTS:
        countListType = LTY_BROWSELIST_MEDIAOBJECTS_COUNT;
        break;
    case LTY_BROWSELIST_MEDIAOBJECTS_WITH_SUBFOLDERS:
        countListType = LTY_BROWSELIST_MEDIAOBJECTS_COUNT_WITH_SUBFOLDERS;
        break;
    case LTY_CURRENT_SELECTION:
        countListType = LTY_CURRENT_SELECTION_COUNT;
        break;
    case LTY_END_OF_LTY:
        countListType = LTY_END_OF_LTY_COUNT;
        break;
    case LTY_CD:
        countListType = LTY_CD_COUNT;
        break;
    case LTY_IMAGE_FOLDER:
         countListType = LTY_IMAGE_FOLDER_COUNT;
         break;
    case LTY_IMAGE_FOLDER_ITEM:
        countListType = LTY_IMAGE_FOLDER_ITEM_COUNT;
        break;
    //-------------------------------------
    case LTY_IMAGE:
        countListType = LTY_IMAGE_COUNT;
        break;
    case LTY_IMAGE_SEARCH:
       countListType = LTY_IMAGE_SEARCH_COUNT;
       bIsSearchList = true;
       break;
    //-------------------------------------
    case LTY_COMPILATION:
        countListType = LTY_COMPILATION_COUNT;
        break;
    case LTY_COMPILATION_SEARCH:
        countListType = LTY_COMPILATION_SEARCH_COUNT;
        bIsSearchList = true;
        break;
    //-------------------------------------
    case LTY_COMPILATION_SONG:
        countListType = LTY_COMPILATION_SONG_COUNT;
        break;
    case LTY_COMPILATION_SONG_SEARCH:
        countListType = LTY_COMPILATION_SONG_SEARCH_COUNT;
        bIsSearchList = true;
        break;
    case LTY_USER_FAVORITES_SONG:
        countListType = LTY_USER_FAVORITES_SONG_COUNT;
        break;
    case LTY_USER_FAVORITES_SONG_SEARCH:
        countListType = LTY_USER_FAVORITES_SONG_SEARCH_COUNT;
        bIsSearchList = true;
        break;
    default:
        return MP_ERR_DB_UNSUPPORTED_LIST_TYPE;
    }




    /* take real deviceID in case of a MyMedia playlist */
    tDeviceID realDeviceID = dbList->mDeviceID;
    if ((MY_MEDIA == dbList->mDeviceID)
        &&
        (countListType == LTY_PLAYLIST_SONG_COUNT)) {

        res = GetObjectDeviceID(OUT realDeviceID, IN dbList->mFilterTag1.tag, CTY_PLAYLIST);
        if (res) return res;
    }

    /* issue the old query for the count statement */
    /* if it is an IPOD list, do another query */
    if (dbList->mIsVTIPODList) {

        /* get the device information*/
        tDeviceInfo deviceInfo;
        res = GetDeviceInfo(OUT deviceInfo, IN realDeviceID);
        if (res) return res;

        tFilterTag1 IPODFilterTag1;
        tFilterTag2 IPODFilterTag2;
        tFilterTag3 IPODFilterTag3;
        tFilterTag4 IPODFilterTag4;

        IPODFilterTag1.tag = dbList->mFilterTag1.tag;
        IPODFilterTag2.tag = dbList->mFilterTag2.tag;
        IPODFilterTag3.tag = dbList->mFilterTag3.tag;
        IPODFilterTag4.tag = dbList->mFilterTag4.tag;

        /* Playlist items retrieved always via VTIPOD */
        if (countListType == LTY_PLAYLIST_SONG_COUNT && !(dbList->mFilterTag1.tag & IPOD_MARKER_BIT)) {
            //it's an indexed playlist tagID
            tTagID playlistTag = dbList->mFilterTag1.tag;
            res = GetVTTagForIPODPlaylist(INOUT playlistTag);
            if (res) return res;
            IPODFilterTag1.tag = playlistTag;
        } else {
            res = GetVTTagForIPOD(INOUT IPODFilterTag1.tag);
            if (res) return res;
        }
        res = GetVTTagForIPOD(INOUT IPODFilterTag2.tag);
        if (res) return res;
        res = GetVTTagForIPOD(INOUT IPODFilterTag3.tag);
        if (res) return res;
        res = GetVTTagForIPOD(INOUT IPODFilterTag4.tag);
        if (res) return res;

        res = query.Select(LTY_IPOD_ACCESS_COUNT, MY_MEDIA, "tiiiii",
                IN deviceInfo.mountPoint,
                IN countListType,
                IN IPODFilterTag1.tag,
                IN IPODFilterTag2.tag,
                IN IPODFilterTag3.tag,
                IN IPODFilterTag4.tag);
        if (res) return res;

    } else if (IsFileList(dbList->mListType)) {

        /* get the counts */
        tNumberOfMediaObjects numberOfMediaObjects;
        tNumberOfPlayLists numberOfPlayLists;
        tNumberOfFolders numberOfFolders;
        res = GetFileListSize(OUT listSize, OUT numberOfMediaObjects, OUT numberOfPlayLists, OUT numberOfFolders, IN listID);

        return res; // for file list counts, the method returns already a list size

    }
    else if ((LocalSPM::GetDataProvider().PersonalizationSupported()) && IsUserFavoriteList(dbList->mListType))
    {
                ETG_TRACE_USR2(("DBManager::GetListSize countListType %d realDeviceID %d dbList->mUserID : %d",countListType,realDeviceID,dbList->mUserID));

                if(dbList->mListType == LTY_USER_FAVORITES_SONG_SEARCH)
                {
                    unsigned int uSize = uMinimalExpectedSize;
                    if(DB_UNDEFINED_VALUE == uMinimalExpectedSize)
                    {
                        uSize = INDEX_NO_LIMIT;
                    }
                    unsigned int uOffset = uOffsetForSearch;
                    if(DB_UNDEFINED_VALUE ==uOffsetForSearch)
                    {
                        uOffset = 0;
                    }

                    res = query.Select(IN countListType, IN realDeviceID, /*Start from Offset*/uOffset,uSize,"iit",IN 0,IN dbList->mUserID,IN dbList->mSearchString);
                }
                else
                {
                res = query.Select(IN countListType, IN realDeviceID,"ii",IN 0,IN dbList->mUserID);
                }
            if (res) return res;
    }
        else { // standard list

        /* prepare the sql statement for the metadata list */
        tGeneralString formatBuffer;


        SelectFormatString(INOUT formatBuffer, IN dbList->mFilterTag1, IN dbList->mFilterTag2, IN dbList->mFilterTag3, IN dbList->mFilterTag4, IN dbList->mSearchString);


        if(bIsSearchList) //Roadmap 16014 full text Search
        {
            unsigned int uSize = uMinimalExpectedSize;
            if(DB_UNDEFINED_VALUE == uMinimalExpectedSize)
            {
                uSize = INDEX_NO_LIMIT;
            }
            unsigned int uOffset = uOffsetForSearch;
            if(DB_UNDEFINED_VALUE ==uOffsetForSearch)
            {
                uOffset = 0;
            }

            res = query.Select(IN countListType, IN realDeviceID, /*Start from Offset*/uOffset,uSize,IN formatBuffer,
                    IN dbList->mFilterTag1.tag,
                    IN dbList->mFilterTag2.tag,
                    IN dbList->mFilterTag3.tag,
                    IN dbList->mFilterTag4.tag,
                    IN dbList->mSearchString);
        }
        else
        {
        res = query.Select(IN countListType, IN realDeviceID, IN formatBuffer,
                IN dbList->mFilterTag1.tag,
                IN dbList->mFilterTag2.tag,
                IN dbList->mFilterTag3.tag,
                IN dbList->mFilterTag4.tag);

        }

        if (res) return res;
    }

    /* get the count */
    res = query.Get("i", OUT &listSize);
    query.End();

    if (res == MP_ERR_DB_END_OF_LIST) { // if DB returns no entry: list is empty
        listSize = 0;
    } else if (res) return res;

    VARTRACE(countListType);
    VARTRACE(listSize);
    return MP_NO_ERROR;
}

char *DBManager::SelectFormatString(char *buffer, const tFilterTag1 tag1, const tFilterTag2 tag2, const tFilterTag3 tag3, const tFilterTag4 tag4, const tSearchString searchString /*Roadmap 16014 'full Text Search'*/) const
{
    ENTRY_INTERNAL
    char *cptr = buffer;

    if (tag1.tag) *cptr++ = 'i';
    else *cptr++ = 's';
    if (tag2.tag) *cptr++ = 'i';
    else *cptr++ = 's';
    if (tag3.tag) *cptr++ = 'i';
    else *cptr++ = 's';
    if (tag4.tag) *cptr++ = 'i';
    else *cptr++ = 's';
    if (strlen_r(searchString)) *cptr++ = 't'; /*Roadmap 16014 'full Text Search'*/
    else *cptr++ = 's';


    *cptr = 0;
    return buffer;
}

tResult DBManager::GetMediaplayerDeviceConnections(tNumberOfDevices &numberOfDevices, vector<tDeviceInfo> &deviceInfos,
        const tBoolean allConnectionStates/*= false*/, const tBoolean virtualDevice/*=false*/, const tConnectionState connectionState/*=CS_CONNECTED*/)
{
    ENTRY
    Query query;
    tResult res;

    /* prepare the sql statement for the devices list */
    if(virtualDevice)
    {
        res = query.Select(LTY_VIRTUAL_DEVICES, MY_MEDIA, "ii", IN (int)allConnectionStates, IN connectionState);
    }
    else
    {
        res = query.Select(LTY_DEVICES, MY_MEDIA, "ii", IN (int)allConnectionStates, IN connectionState);
    }
    if (res) return res;

    /* clear the vector */
    deviceInfos.clear();

    /* get all infos */
    numberOfDevices = 0;
    while(1) {

        tDeviceInfo deviceInfo;
        InitDeviceInfo(OUT deviceInfo);

        res = query.Get("iTTTTTi--i----iiii-iTiiiiiiiiiiiTiiii-iTTT",
                OUT &deviceInfo.deviceID,
                OUT deviceInfo.UUID, IN sizeof(deviceInfo.UUID),
                OUT deviceInfo.serialNumber, IN sizeof(deviceInfo.serialNumber),
                OUT deviceInfo.deviceVersion, IN sizeof(deviceInfo.deviceVersion), //FirmwareVersion
                OUT deviceInfo.deviceName, IN sizeof(deviceInfo.deviceName),       //FriendlyName
                OUT deviceInfo.mountPoint, IN sizeof(deviceInfo.mountPoint),
                OUT &deviceInfo.connectionCount,
                OUT &deviceInfo.activeSource,
                OUT &deviceInfo.deviceType,
                OUT &deviceInfo.discType,
                OUT &deviceInfo.indexedState,
                OUT &deviceInfo.connectionState,
                OUT &deviceInfo.formerConnectionState,
                OUT deviceInfo.accessoryName, IN sizeof(deviceInfo.accessoryName),
                OUT &deviceInfo.deviceState,
                OUT &deviceInfo.fileSystemType,
                OUT &deviceInfo.partitionNumber,
                OUT &deviceInfo.totalSize,
                OUT &deviceInfo.freeSize,
                OUT &deviceInfo.productID,
                OUT &deviceInfo.numberOfAudioFiles,
                OUT &deviceInfo.numberOfVideoFiles,
                OUT &deviceInfo.numberOfImageFiles,
                OUT &deviceInfo.diPOCaps,
                OUT &deviceInfo.diPOActive,
                OUT deviceInfo.diPOVersion, IN sizeof(deviceInfo.diPOVersion),
                OUT &deviceInfo.connectionType,
                OUT &deviceInfo.disconnectReason,
                OUT &deviceInfo.isShuffleSupported,
                OUT &deviceInfo.isRepeatSupported,
                OUT &deviceInfo.isNowPlayingListAvailable,
                OUT &deviceInfo.deviceUUID,IN sizeof(deviceInfo.deviceUUID),
                OUT &deviceInfo.appleDeviceMACAddress,IN sizeof(deviceInfo.appleDeviceMACAddress),
                OUT &deviceInfo.appleDeviceUSBSerialNumber,IN sizeof(deviceInfo.appleDeviceUSBSerialNumber));
        if (res) break;

        #if 0
        if(!LocalSPM::GetDataProvider().AVRCPLowerProfileSupported())
        {
            deviceInfo.isShuffleSupported = FALSE;
            deviceInfo.isRepeatSupported = FALSE;
        }
        #endif
        // set the connected flag (it is a query over connected devices only)
        if ((CS_CONNECTED == deviceInfo.connectionState)
            ||
            (CS_UNSUPPORTED == deviceInfo.connectionState))
        {
            deviceInfo.connected = 1;
        }

        // if MyMedia shall be visible to HMI then add all devices; if MyMedia is not supported then add only normal devices (!=0)
        if (LocalSPM::GetDataProvider().MyMediaVisible() == 1 || deviceInfo.deviceID != 0)
        {
            deviceInfos.push_back(deviceInfo);
            numberOfDevices++;
        }
    }
    query.End();

    return MP_NO_ERROR;
}

tResult DBManager::GetConnectedDeviceCount(tNumberOfDevices &numberOfDevices)
{
    ENTRY
    Query query;
    tResult res;

    /* prepare the sql statement for the devices list */
    res = query.Select(LTY_DEVICES_COUNT, MY_MEDIA, "ii",IN 0, IN CS_CONNECTED);
    if (res) return res;

    /* get all infos */
    numberOfDevices = 0;
    res = query.Get("i", OUT &numberOfDevices);
    query.End();

    if (res == MP_ERR_DB_END_OF_LIST)
    {
        numberOfDevices = 0;// if DB returns no entry: list is empty
    }
    else if (res) return res;

    return MP_NO_ERROR;
}

tTriggerID DBManager::OnInsertTrigger(const char* smName, const char* msgName, const char* tbl_name, const char* clause, const char* cbArg1, const char* cbArg2)
{
    ENTRY

    if (!mDB) return MP_ERR_DB_NOT_OPEN;
    if (!mDB->Handle()) return MP_ERR_DB_INVALID_HANDLE;

    // TODO: move the DbTrigMgr object to the spm
    return DBTriggerManager::GetInstance().regisCallback_onInsert(IN smName, IN msgName , IN mDB->Handle(), IN tbl_name, IN clause, IN cbArg1, IN cbArg2);
}

tTriggerID DBManager::OnDeleteTrigger(const char* smName, const char* msgName, const char* tbl_name, const char* clause, const char* cbArg1, const char* cbArg2)
{
    ENTRY

    if (!mDB) return MP_ERR_DB_NOT_OPEN;
    if (!mDB->Handle()) return MP_ERR_DB_INVALID_HANDLE;

    return DBTriggerManager::GetInstance().regisCallback_onDelete(IN smName, IN msgName , IN mDB->Handle(), IN tbl_name, IN clause, IN cbArg1, IN cbArg2);
}

tTriggerID DBManager::OnUpdateTrigger(const char* smName, const char* msgName, const char* tbl_name, const char* col_name, const char* clause, const char* cbArg1, const char* cbArg2)
{
    ENTRY

    if (!mDB) return MP_ERR_DB_NOT_OPEN;
    if (!mDB->Handle()) return MP_ERR_DB_INVALID_HANDLE;

    return DBTriggerManager::GetInstance().regisCallback_onUpdate(IN smName, IN msgName, IN mDB->Handle(), IN tbl_name, IN col_name, IN clause, IN cbArg1, IN cbArg2);
}

tResult DBManager::OnInsertTrigger(tTriggerID &triggerID, const char* smName, const char* msgName, const char* tbl_name, const char* clause, const char* cbArg1, const char* cbArg2)
{
    ENTRY

    if (!mDB) return MP_ERR_DB_NOT_OPEN;
    if (!mDB->Handle()) return MP_ERR_DB_INVALID_HANDLE;

    // TODO: move the DbTrigMgr object to the spm
    triggerID = DBTriggerManager::GetInstance().regisCallback_onInsert(IN smName, IN msgName , IN mDB->Handle(), IN tbl_name, IN clause, IN cbArg1, IN cbArg2);
    if(0 == triggerID) return MP_ERR_DB_REGISTER_FAILED;

    return MP_NO_ERROR;
}

tResult DBManager::OnDeleteTrigger(tTriggerID &triggerID, const char* smName, const char* msgName, const char* tbl_name, const char* clause, const char* cbArg1, const char* cbArg2)
{
    ENTRY

    if (!mDB) return MP_ERR_DB_NOT_OPEN;
    if (!mDB->Handle()) return MP_ERR_DB_INVALID_HANDLE;

    triggerID = DBTriggerManager::GetInstance().regisCallback_onDelete(IN smName, IN msgName , IN mDB->Handle(), IN tbl_name, IN clause, IN cbArg1, IN cbArg2);
    if(0 == triggerID) return MP_ERR_DB_REGISTER_FAILED;

    return MP_NO_ERROR;
}

tResult DBManager::OnUpdateTrigger(tTriggerID &triggerID, const char* smName, const char* msgName, const char* tbl_name, const char* col_name, const char* clause, const char* cbArg1, const char* cbArg2)
{
    ENTRY

    if (!mDB) return MP_ERR_DB_NOT_OPEN;
    if (!mDB->Handle()) return MP_ERR_DB_INVALID_HANDLE;

    triggerID = DBTriggerManager::GetInstance().regisCallback_onUpdate(IN smName, IN msgName, IN mDB->Handle(), IN tbl_name, IN col_name, IN clause, IN cbArg1, IN cbArg2);
    if(0 == triggerID) return MP_ERR_DB_REGISTER_FAILED;

    return MP_NO_ERROR;
}

tResult DBManager::UnregisterTrigger(const tTriggerID triggerID) const
{
    ENTRY
    tBoolean bRet;

    bRet = DBTriggerManager::GetInstance().DeRegister(IN triggerID);
    if(false == bRet) return MP_ERR_DB_UNREGISTER_FAILED;

    return MP_NO_ERROR;
}

tResult DBManager::RemoveDevice(const tDeviceID deviceID, const tBoolean virtualDevice/*=false*/)
{
    ENTRY
    tResult res;
    Query query;

    /* delete the device from DB */
    if(virtualDevice)
    {
        res = query.Delete(LTY_VIRTUAL_DEVICES_DELETE);
    }
    else
    {
        res = query.Delete(LTY_DEVICES_DELETE);
    }
    if (res) return res;

    res = query.Remove("ii", IN 0, IN deviceID);
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::UpdateMediaObjectInternal(Query &queryMain, const tMediaObject mediaObject)
{
    ENTRY
    tResult res;

    ETG_TRACE_USR3(("Update media object URL: %s", mediaObject.fileName));

    /* try to remove the given object */
    //queryMain.Remove("it", IN mediaObject.deviceID, IN mediaObject.fileName);

    /* try to update the given object */
    res = queryMain.Change("ititititititiiiiiiitiiiiiiilitiiiiiiiiiiit",
            IN 0, IN ((mediaObject.catType == CTY_PLAYLIST && mediaObject.UUID[0] == 0) ? mediaObject.fileName : mediaObject.UUID), // CMG3GB-1087 - Unique Playlist (multiple playlists with same name on same device is possible)
            IN 0, IN mediaObject.title,
            IN 0, IN mediaObject.MetadataField1,
            IN 0, IN mediaObject.MetadataField2,
            IN 0, IN mediaObject.MetadataField3,
            IN 0, IN mediaObject.MetadataField4,
            IN 0, IN mediaObject.metadataConvertFlag,
            IN 0, IN mediaObject.compilationFlag,
            IN 0, IN mediaObject.totalPlaytime,
            IN 0, IN mediaObject.albumArtString,
            IN 0, IN mediaObject.trackNumber,
            IN 0, IN mediaObject.notPlayable,
            IN 0, IN mediaObject.mediaType,
            IN 0, IN mediaObject.fileSize,
            IN 0, IN mediaObject.dateTime,
            IN 0, IN mediaObject.fileMode,
            IN 0, IN mediaObject.userID,
            IN 0, IN mediaObject.groupID,
            IN 0, IN mediaObject.year,
            IN 0, IN mediaObject.deviceID,       //for WHERE clause
            IN 0, IN mediaObject.fileName);      //for WHERE clause
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::StoreMediaObjectInternal(Query &queryMain, Query &queryMyMe, const tMediaObject mediaObject)
{
    ENTRY
    tResult res;

    /* store the directory elements first */
    if((LocalSPM::GetDataProvider().DBFileBrowsingByDB())
       &&
       (IsMassStorageDevice(IN mediaObject.deviceType)) || (DTY_MTP == mediaObject.deviceType)) //DTY_USB, DTY_SD, DTY_CDROM, DTY_FLASH,DTY_MTP
    {
        StoreDirectoriesInternal(INOUT queryMain, INOUT queryMyMe, IN mediaObject);
    }

    ETG_TRACE_USR3(("Store media object URL: %s", mediaObject.fileName));

    /* put in the data */
    res = queryMain.Put(IN "iittttttttiiititiiltiiii",
            IN mediaObject.deviceID,
            IN mediaObject.catType,
            IN mediaObject.fileName,
            IN ((mediaObject.catType == CTY_PLAYLIST && mediaObject.UUID[0] == 0) ? mediaObject.fileName : mediaObject.UUID), // CMG3GB-1087 - Unique Playlist (multiple playlists with same name on same device is possible)
            IN mediaObject.parentUUID,
            IN mediaObject.title,          // song, chapter, episode
            IN mediaObject.MetadataField1, // genre, author, podcastName, videoFileName, playlistName
            IN mediaObject.MetadataField2, // artist, bookTitle
            IN mediaObject.MetadataField3, // composer
            IN mediaObject.MetadataField4, // album
            IN mediaObject.metadataConvertFlag,
            IN mediaObject.compilationFlag,
            IN mediaObject.totalPlaytime,
            IN mediaObject.albumArtString,
            IN mediaObject.trackNumber,
            IN mediaObject.path,
            IN mediaObject.notPlayable,
            IN mediaObject.mediaType,
            IN mediaObject.fileSize,
            IN mediaObject.dateTime,
            IN mediaObject.fileMode,
            IN mediaObject.userID,
            IN mediaObject.groupID,
            IN mediaObject.year
            );
    if (res) return res;

    /* do this only if MyMedia shall be supported.
       Comment out because MyMedia is needed for cleanup Main DB after reindexing */
    //if (LocalSPM::GetDataProvider().MyMediaSupported()) {

        /* update the my media db by copying the stored object to myme database */
        res = queryMyMe.Put("it", IN mediaObject.deviceID, IN mediaObject.fileName);
        if (res) return res;
    //}

    return MP_NO_ERROR;
}

tResult DBManager::StoreDirectoriesInternal(Query &queryMain, Query &queryMyMe, const tMediaObject mediaObject)
{
    ENTRY
    tResult res;

    tMediaObject dirMediaObject = mediaObject;

    tURL fullName;
    unsigned char *pureFile = NULL;
    strncpy_r(OUT fullName, IN dirMediaObject.path, IN sizeof(fullName));

    /* trim last character if it is "/" */
    if (FastUTF8::EndsWith((const FastUTF8::tString)fullName, (const FastUTF8::tString)"/")) //lint !e1773
    {
        FastUTF8::Split(OUT pureFile, INOUT (unsigned char *)fullName);
    }
    strncpy_r(OUT (char *)dirMediaObject.fileName, IN (const char *)fullName, IN sizeof(dirMediaObject.fileName));

    FastUTF8::Split(OUT pureFile, INOUT (unsigned char *)fullName);
    if (pureFile)
    {
        strncpy_r(OUT (char *)dirMediaObject.path, IN (const char *)fullName, IN sizeof(dirMediaObject.path));
        strncpy_r(OUT (char *)dirMediaObject.title, IN (const char *)pureFile, IN sizeof(dirMediaObject.title));

        StoreDirectoriesInternal(INOUT queryMain, INOUT queryMyMe, IN dirMediaObject);

        dirMediaObject.catType = CTY_DIR;
        strncat_r(OUT (char *)dirMediaObject.path, "/", IN sizeof(dirMediaObject.path));

        ETG_TRACE_USR3(("Store directory URL: %s", dirMediaObject.fileName));
        //ETG_TRACE_USR3(("Store directory path: %s", dirMediaObject.path));
        //ETG_TRACE_USR3(("Store directory title: %s", dirMediaObject.title));

        /* put in the data */
        res = queryMain.Put(IN "iittttttttiiititiiltiii",
                IN dirMediaObject.deviceID,
                IN dirMediaObject.catType,
                IN dirMediaObject.fileName, // URL of directory
                IN "",                      // UUID
                IN "",                      // ParentUUID
                IN dirMediaObject.title,    // directory name
                IN "", // genre, author, podcaseName, videFileName, playListName
                IN "", // artist, bookTitle
                IN "", // composer
                IN "", // album
                IN 0,                       // metadataConvertFlag
                IN 0,                       // compilationFlag
                IN 0,                       // totalPlaytime
                IN "",                      // albumArtString
                IN 0,                       // trackNumber
                IN dirMediaObject.path,     // path of directory
                IN FNP_PLAYABLE,
                IN MTY_UNKNOWN,
                IN 0LL,                     // fileSize
                IN "",                      // dateTime
                IN 511,                     // fileMode
                IN 0,                       // userID
                IN 0);                      // groupID
        if (res) return res;

        tMediaContent mediaContent = MTY_UNKNOWN;
        UpdateFolderMediaContent(INOUT queryMain, IN dirMediaObject);

        res = queryMain.Insert(LTY_MEDIAOBJECTS_INSERT);
        if(res)
        {
            ETG_TRACE_ERR(("StoreDirectoriesInternal:Prepare(LTY_MEDIAOBJECTS_INSERT)  Error=%d/%s", res, errorString(res)));
            return res;
        }

        /* do this only if MyMedia shall be supported.
           Comment out because MyMedia is needed for cleanup Main DB after reindexing */
        //if (LocalSPM::GetDataProvider().MyMediaSupported()) {

            /* update the my media db by copying the stored object to myme database */
            res = queryMyMe.Put("it", IN dirMediaObject.deviceID, IN dirMediaObject.fileName);
            if (res) return res;
        //}
    }

    return MP_NO_ERROR;
}

#define TRANSACTION_ON 1
#define PERF_MEAS 0

tResult DBManager::FlushMediaObjectCache()
{
    ENTRY;
    tResult res;

    if (!mMediaObjectCache.empty())
    {
        /* reset object cache of all lists for this device because position of object may change */
        res = ResetObjectCacheOfDBList(IN mMediaObjectCache.at(0).deviceID);
        if (res) return res;

        if (!LocalSPM::GetDataProvider().DBFlushMediaObjectsAsThread()) {

            res = FlushMediaObjectCacheInternal(IN &mMediaObjectCache);
            if (res) return res;

        } else {

            /* copy the object cache */
            vector<tMediaObject> *objectCacheCopy; //lint -e429 freed and unlocked by calling thread
            objectCacheCopy = new vector<tMediaObject>(mMediaObjectCache.rbegin(), mMediaObjectCache.rend());
            (*objectCacheCopy) = mMediaObjectCache;

            /* wait for previous run to be completed */
            mFlushLock.lock();
            if(LocalSPM::GetDataProvider().ControlCPUThtreadLoad())
            {
                /* start the flush as thread with low priority*/
                LocalSPM::GetThreadFactoryLowprio().Do(this, TSK_FLUSH_MEDIA_OBJECTS, objectCacheCopy);
            }
            else
            {
                /* start the flush as thread */
                LocalSPM::GetThreadFactory().Do(this, TSK_FLUSH_MEDIA_OBJECTS, objectCacheCopy);
            }
        }
    }

    return MP_NO_ERROR;
}

tResult DBManager::WaitForFlushMediaObjectCacheEnd()
{
    ENTRY;
    mFlushLock.lock();
    mFlushLock.unlock();
    return MP_NO_ERROR;
}

tResult DBManager::FlushMediaObjectCacheInternal(vector<tMediaObject> *objectCache)
{
    ENTRY
    ElapsedTimer measTime;
    Query *queryMain; //of [Main] DB
    queryMain = new Query();
    Query *queryMyMe; //of [MyMe] DB
    queryMyMe = new Query();
    tResult res = MP_NO_ERROR;

#if PERF_MEAS
    TimeTrace        ticks("FlushMediaObjectCache");
#endif
#if TRANSACTION_ON
    queryMain->BeginTransaction(LTY_SONG);
#if PERF_MEAS
    ticks.elapsed();
#endif // #if PERF_MEAS
#endif // #if TRANSACTION_ON

#define INSERT_OR_REPLACE 0 // Attention! same switch as in Queries.h
#if !INSERT_OR_REPLACE
    /* delete all media objects */
    //queryMain->Delete(LTY_MEDIAOBJECTS_DELETE);

    /* update all media objects */
    queryMain->Update(IN LTY_MEDIAOBJECTS_UPDATE);

    for(unsigned int i=0; i<objectCache->size(); i++) {

#if PERF_MEAS
        ticks.restart();
#endif
        res = UpdateMediaObjectInternal(IN *queryMain, IN objectCache->at(i));
        if (res) {
            ETG_TRACE_ERR(("UpdateMediaObjectInternal: Error=%d/%s", res, errorString(res)));
            res = MP_NO_ERROR;
        }
#if PERF_MEAS
        ticks.elapsed();
#endif
    }
#endif // #if !INSERT_OR_REPLACE

    /* insert new media objects to Main DB */
    queryMain->Insert(IN LTY_MEDIAOBJECTS_INSERT);

    /* copy media objects to MyMe DB */
    queryMyMe->Insert(IN LTY_MEDIAOBJECTS_MYME_DEV_URL_COPY);

    /* flush all media objects */
    for(unsigned int i=0; i<objectCache->size(); i++) {

#if PERF_MEAS
        ticks.restart();
#endif
        res = StoreMediaObjectInternal(IN *queryMain, IN *queryMyMe, IN objectCache->at(i));
        if (res) {
            ETG_TRACE_ERR(("StoreMediaObjectInternal: Error=%d/%s", res, errorString(res)));
            res = MP_NO_ERROR;
        }
#if PERF_MEAS
        ticks.elapsed();
#endif
    }

    /* End Transaction */
#if TRANSACTION_ON
#if PERF_MEAS
    ticks.restart();
#endif
    queryMain->EndTransaction();
#if PERF_MEAS
    ticks.elapsed();
#endif
#endif

    delete queryMain;
    delete queryMyMe;

    if(res == MP_NO_ERROR) {
        //calc new flush border based on last performance
        CalcCountFlushBorder(objectCache->size(), measTime.ElapsedUS());
    }

    return res;
}

tResult DBManager::StoreMediaObject(const tMediaObject mediaObject)
{
    ENTRY
    tResult res = MP_NO_ERROR;

    /* UNKNOWN macro: if string is empty take UnknownText from configuration */
    static char unknown[64] = {0};
    if (!unknown[0])
    {
        strncpy_r(unknown, LocalSPM::GetDataProvider().DBUnknownText().c_str(), sizeof(unknown));
    }
    #define UNKNOWN(a) (strlen_r(a) ? (strtrim((char *)a)) : unknown)

    /* set UnknownText for empty metadata fields */
    tMediaObject localMediaObject = mediaObject;
    strncpy_r(OUT localMediaObject.MetadataField1, IN UNKNOWN(mediaObject.MetadataField1), IN sizeof(localMediaObject.MetadataField1));
    strncpy_r(OUT localMediaObject.MetadataField2, IN UNKNOWN(mediaObject.MetadataField2), IN sizeof(localMediaObject.MetadataField2));
    strncpy_r(OUT localMediaObject.MetadataField3, IN UNKNOWN(mediaObject.MetadataField3), IN sizeof(localMediaObject.MetadataField3));
    strncpy_r(OUT localMediaObject.MetadataField4, IN UNKNOWN(mediaObject.MetadataField4), IN sizeof(localMediaObject.MetadataField4));

    /* set path of media object */
    tURL fullName;
    unsigned char *pureFile;
    strncpy_r(OUT fullName, IN localMediaObject.fileName, IN sizeof(fullName));
    FastUTF8::Split(OUT pureFile, INOUT (unsigned char *)fullName);
    if (pureFile)
    {
        strncpy_r(OUT (char *)localMediaObject.path, IN (const char *)fullName, IN sizeof(localMediaObject.path));
    }
    strncat_r(OUT (char *)localMediaObject.path, "/", IN sizeof(localMediaObject.path));

    /* if title is empty replace it with the pure file name of the file */
    if (!strlen_r(localMediaObject.title))
    {
        /* no pure file name found? seems to be an IPOD string, try it with IPOD seperator */
        if (!pureFile)
        {
            char IPODSeperator[2];
            IPODSeperator[0] = 0xfe;
            IPODSeperator[1] = 0x0;
            FastUTF8::Split(OUT pureFile, INOUT (unsigned char *)fullName, IN (unsigned char *)IPODSeperator);
        }
        if (pureFile)
        {
            /* remove the extension, if wanted */
            if (LocalSPM::GetDataProvider().DBRemoveExtensionFromFilename()) {

                unsigned char *extension;
                FastUTF8::SplitExtension(OUT extension, INOUT (unsigned char *)pureFile);
            }

            strncpy_r(OUT (char *)localMediaObject.title, IN (const char *)pureFile, IN sizeof(localMediaObject.title));
        }
    }

#if 0
    /* set the category type from media type */
    tCategoryType catType = GetCTYByMTY(IN localMediaObject.mediaType);
    if (CTY_NONE != catType)
    {
        localMediaObject.catType = catType;
    }
    else
    {
        return MP_ERR_DB_UNSUPPORTED_CAT_TYPE;
    }
#endif

    /* put the media object into the cache */
    mMediaObjectCache.push_back(localMediaObject);

    /* check if a flush has to be taken */
    if(mMediaObjectCache.size() >= GetCountFlushBorder()) {

        res = FlushMediaObjectCache();

        /* clear cache */
        mMediaObjectCache.clear();
    }

    return res;
}

tResult DBManager::StoreMediaObjectEnd(const bool speedup)
{
    ENTRY
    tResult res = MP_NO_ERROR;

    /* flush the rest of the objects (in a thread) */
    res = FlushMediaObjectCache();
    if (res) return res;

    /* wait for end of the flush thread */
    WaitForFlushMediaObjectCacheEnd();

    /* clear cache */
    mMediaObjectCache.clear();

    if(speedup) {
        /* speedup the DB (also in a thread) */
        if(mDB) {
            res = mMMUpdt.Speedup(IN mDB);
            if (res) return res;
        }
        else {
            res = MP_ERR_DB_NOT_OPEN;
        }
    }
    return res;
}

tResult DBManager::GetNumberOfMediaObjectsInDB(tNumberOfMediaObjects &numberObjectsInDB) const
{
    ENTRY
    Query query;
    tResult res;

    numberObjectsInDB = 0;

    /* prepare the sql statement for the metadata list */
    res = query.Select(LTY_MEDIAOBJECTS_COUNT, IN 1 /* dummy */, "");
    if (res) return res;

    /* get the count */
    res = query.Get("i", OUT &numberObjectsInDB);
    query.End();
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::GetNumberOfPlayableObjectsInDB(tNumberOfMediaObjects &numberOfAudioObjectsInDB, tNumberOfMediaObjects &numberOfVideoObjectsInDB, const tDeviceID deviceID) const
{
    ENTRY
    Query query;
    tResult res;
    numberOfAudioObjectsInDB = 0;
    numberOfVideoObjectsInDB = 0;

    /* prepare the sql statement for the metadata list */
    res = query.Select(LTY_MEDIAOBJECTS_PLAYABLE_FILTER_COUNT, IN deviceID, "iiiiii",
                IN 0, IN CTY_SONG, IN CTY_PODCAST, IN CTY_AUDIOBOOK,
                IN 0, IN FNP_PLAYABLE);
    if (res) return res;

    /* get the count */
    res = query.Get("i", OUT &numberOfAudioObjectsInDB);
    query.End();
    if (res) return res;

    /* prepare the sql statement for the metadata list */
    res = query.Select(LTY_MEDIAOBJECTS_PLAYABLE_FILTER_COUNT, IN deviceID, "iiii",
                IN 0, IN CTY_VIDEO,
                IN 0, IN FNP_PLAYABLE);
    if (res) return res;

    /* get the count */
    res = query.Get("i", OUT &numberOfVideoObjectsInDB);
    query.End();
    if (res) return res;
    VARTRACE(numberOfAudioObjectsInDB);
    VARTRACE(numberOfVideoObjectsInDB);

    return MP_NO_ERROR;
}

tResult DBManager::GetNumberOfRippedFile(tNumberOfMediaObjects &numberOfRippedFile, const tDeviceID deviceID, IN tTOC HashID, tCategoryType CategoryType) const
{
    ENTRY
    Query query;
    tResult res;


    /* prepare the sql statement for the metadata list */
    res = query.Select(LTY_MEDIAOBJECTS_RIPPED_FILE, IN deviceID, "tii", IN HashID, IN 0, IN CategoryType);
    if (res) return res;

    /* get the count */
    res = query.Get("i", OUT &numberOfRippedFile);
    query.End();
    if (res) return res;
    ETG_TRACE_USR4(("[INFO] GetNumberOfRippedFile:FileCount:%d",numberOfRippedFile));
    return MP_NO_ERROR;
}

tResult DBManager::GetMetadataOfRippedFile(tRippedTrackInfo &trackInfo, const tDeviceID deviceID, IN tTOC HashID, IN tU32 trackIndex) const
{
    ENTRY
    Query query;
    tResult res;

    //Form URL
    tRippedFilePath RippedFilePath;
    char data[8] = {0};
    strncpy_r(OUT RippedFilePath, IN HashID, IN sizeof(RippedFilePath));
    strncat_r(OUT RippedFilePath, IN "/Track_", IN sizeof(RippedFilePath));
    snprintf(data, sizeof(data), "%d", trackIndex);
    strncat_r(OUT RippedFilePath, IN data, IN sizeof(RippedFilePath));
    strncat_r(OUT RippedFilePath, IN ".mp3", IN sizeof(RippedFilePath));

    /* prepare the sql statement for the metadata list */
    res = query.Select(LTY_MEDIAOBJECTS_RIPPED_FILE_METADATA, IN deviceID, "t" , IN RippedFilePath);
    if (res) return res;
    ETG_TRACE_USR4(("[INFO] SELECT Success : GetMetadataOfRippedFile:Trackindex:%d url %s",trackIndex , RippedFilePath));
    /* get the metadata info */
    res = query.Get("TTTTTii",
            OUT trackInfo.MetadataField1, IN sizeof(tMetadata),
            OUT trackInfo.MetadataField2, IN sizeof(tMetadata),
            OUT trackInfo.MetadataField3, IN sizeof(tMetadata),
            OUT trackInfo.MetadataField4 , IN sizeof(tMetadata),
            OUT trackInfo.title, IN sizeof(tMetadata),
            OUT &trackInfo.CompilationFlag,
            OUT &trackInfo.Year);
    query.End();

    if (res) return res;
  ETG_TRACE_USR4(("[INFO] GetMetadataOfRippedFile:Trackindex:%d ",trackIndex));
  /* ETG_TRACE_USR4(("[INFO] GetMetadataOfRippedFile:MetadataField1:%s ",trackInfo.MetadataField1));
    ETG_TRACE_USR4(("[INFO] GetMetadataOfRippedFile:MetadataField2:%s ",trackInfo.MetadataField2));
    ETG_TRACE_USR4(("[INFO] GetMetadataOfRippedFile:MetadataField3:%s ",trackInfo.MetadataField3));
    ETG_TRACE_USR4(("[INFO] GetMetadataOfRippedFile:MetadataField4:%s ",trackInfo.MetadataField4));
    ETG_TRACE_USR4(("[INFO] GetMetadataOfRippedFile:title:%s ",trackInfo.title));*/
    return MP_NO_ERROR;
}

tResult DBManager::GetNumberOfTotalRippedFile(tNumberOfMediaObjects &numberOfTotalRippedFile, const tDeviceID deviceID, tCategoryType CategoryType) const
{
    ENTRY
    Query query;
    tResult res;


    /* prepare the sql statement for the metadata list */
    res = query.Select(LTY_MEDIAOBJECTS_TOTAL_RIPPED_FILE, IN deviceID, "ii", IN 0, IN CategoryType);
    if (res) return res;

    /* get the count */
    res = query.Get("i", OUT &numberOfTotalRippedFile);
    query.End();
    if (res) return res;
    ETG_TRACE_USR4(("[INFO] GetNumberOfTotalRippedFile:FileCount:%d",numberOfTotalRippedFile));
    return MP_NO_ERROR;
}

tResult DBManager::GetIndexingState(vector<tIndexingStateResult> &indexingState)
{
    ENTRY
    Query query;
    tResult res;
    tIndexingStateResult indexingStateResult;

    /* loop over all devices */
    res = query.Select(LTY_DEVICES, MY_MEDIA, "");
    if (res) return res;

    /* clear the vector */
    indexingState.clear();

    while(1) {

        /* clear object */
        InitIndexingStateResult(OUT indexingStateResult);

        /* get next device */
        res = query.Get("i---------------i-i",
                OUT &indexingStateResult.deviceID,
                OUT &indexingStateResult.state,
                OUT &indexingStateResult.percent);
        if (res) break;

        /* push it to the vector */
        indexingState.push_back(indexingStateResult);
    }
    query.End();

    if (res != MP_ERR_DB_END_OF_LIST) return res;

    return MP_NO_ERROR;
}

tResult DBManager::GetIndexingState(tIndexingState &indexingState, const tDeviceID deviceID)
{
    ENTRY
    Query query;
    tResult res;

    res = query.Select(LTY_DEVICES, IN deviceID, "");
    if (res) return res;

    res = query.Get("----------------i", OUT &indexingState);
    query.End();

    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::SetIndexingState(const tDeviceID deviceID, const tIndexingState indexingState, const tIndexingPercentComplete percent)
{
    ENTRY
    Query query;
    tResult res;

    res = query.Update(LTY_DEVICES_UPDATE);
    if (res) return res;

    /* set the given devices indexing state */
    res = query.Change("----------------ii--ii--------------------------------------------ii",
            IN 0, IN indexingState,
            IN 0, IN percent,
            IN 0, IN deviceID);
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::GetFingerprint(tFingerprint &fingerprint, const tDeviceID deviceID)
{
    ENTRY
    Query query;
    tResult res;

    res = query.Select(LTY_DEVICES, IN deviceID, "");
    if (res) return res;

    res = query.Get("--------T", OUT fingerprint, IN sizeof(fingerprint));
    query.End();

    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::SetFingerprint(IN const tDeviceID& deviceID, IN const tFingerprint fingerprint)
{
    ENTRY
    Query query;
    tResult res;

    res = query.Update(LTY_DEVICES_UPDATE);
    if (res)
    {
        return res;
    }

    /* set the given devices check sum */
    res = query.Change("----it------------------------------------------------------------ii", IN 0, IN fingerprint, IN 0, IN deviceID);

    if (res)
    {
        return res;
    }
    return MP_NO_ERROR;

}

tResult DBManager::SetFingerprint(const tDeviceID deviceID, const tFingerprint fingerprint, const tNumberOfFiles numberOfAudioFiles, const tNumberOfFiles numberOfVideoFiles)
{
    ENTRY
    Query query;
    tResult res;
    VARTRACE(numberOfAudioFiles);
    VARTRACE(numberOfVideoFiles);

    res = query.Update(LTY_DEVICES_UPDATE);
    if (res) return res;

    /* set the given devices check sum */
    res = query.Change("----it------------------------iiii--------------------------------ii", IN 0, IN fingerprint, IN 0, IN numberOfAudioFiles, IN 0, IN numberOfVideoFiles, IN 0, IN deviceID);
    if (res) return res;

    /* clear not playable flag */
    tFileErrorHandling fileErrorHandling = (tFileErrorHandling)LocalSPM::GetDataProvider().FileErrorHandling();
    if ((FEH_MARK_OBJECT == fileErrorHandling)
        &&
        (0 < ((int)numberOfAudioFiles) || (0 < (int)numberOfVideoFiles)))
    {
        ClearNotPlayableFlag(IN deviceID);//TODO(kmv5cob):Handle Audio and Video separately
    }

    return MP_NO_ERROR;
}

tResult DBManager::GetNumberOfAudioFiles(tNumberOfFiles &numberOfAudioFiles, const tDeviceID deviceID) const
{
    ENTRY
    Query query;
    tResult res;

    res = query.Select(LTY_DEVICES, IN deviceID, "");
    if (res) return res;

    res = query.Get("---------------------------i", OUT &numberOfAudioFiles);

    query.End();

    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::GetNumberOfPlayableFiles(tNumberOfFiles &numberOfPlayableFiles, const tDeviceID deviceID) const
{
    ENTRY
    Query query;
    tResult res;
    tNumberOfFiles numberOfAudioFiles, numberOfVideoFiles;
    numberOfAudioFiles = numberOfVideoFiles = 0;

    res = query.Select(LTY_DEVICES, IN deviceID, "");
    if (res) return res;

    res = query.Get("---------------------------ii", OUT &numberOfAudioFiles,OUT &numberOfVideoFiles);

    query.End();
    if (res) return res;

    if(numberOfAudioFiles != NUMBER_OF_FILES_NONE && numberOfVideoFiles != NUMBER_OF_FILES_NONE)
    {
        numberOfPlayableFiles = numberOfAudioFiles + numberOfVideoFiles;//(Audio,Video)==> (>0,>0)(0,>0)(>0,0)(0,0)
    }
    else if ((numberOfAudioFiles == NUMBER_OF_FILES_NONE && numberOfVideoFiles == 0 ) || (numberOfAudioFiles == 0 && numberOfVideoFiles == NUMBER_OF_FILES_NONE) || (numberOfAudioFiles == NUMBER_OF_FILES_NONE && numberOfAudioFiles == NUMBER_OF_FILES_NONE))

    {
        numberOfPlayableFiles = -1;//(Audio,Video)==> (-1,-1) (-1,0) (0,-1)
    }
    else if (numberOfAudioFiles!= NUMBER_OF_FILES_NONE)
    {
        numberOfPlayableFiles = numberOfAudioFiles;//(Audio,Video)==> >(>0,-1)
    }
    else
    {
        numberOfPlayableFiles = numberOfVideoFiles;//(Audio,Video)==> (-1,>0)
    }

    VARTRACE(numberOfAudioFiles);
    VARTRACE(numberOfVideoFiles);
    VARTRACE(numberOfPlayableFiles);

    return MP_NO_ERROR;
}


tResult DBManager::SetNumberOfAudioFiles(const tDeviceID deviceID, const tNumberOfFiles numberOfAudioFiles)
{
    ENTRY
    Query query;
    tResult res;
    VARTRACE(numberOfAudioFiles);

    res = query.Update(LTY_DEVICES_UPDATE);
    if (res) return res;

    res = query.Change("------------------------------ii----------------------------------ii", IN 0, IN numberOfAudioFiles, IN 0, IN deviceID);
    if (res) return res;

    /* clear not playable flag */
    tFileErrorHandling fileErrorHandling = (tFileErrorHandling)LocalSPM::GetDataProvider().FileErrorHandling();
    if ((FEH_MARK_OBJECT == fileErrorHandling)
        &&
        (0 < (int)numberOfAudioFiles))
    {
        ClearNotPlayableFlag(IN deviceID);//TODO(Kmv5cob):Clear flag only for audio files
    }

    return MP_NO_ERROR;
}
tResult DBManager::SetNumberOfVideoFiles(const tDeviceID deviceID, const tNumberOfFiles numberOfVideoFiles)
{
    ENTRY
    Query query;
    tResult res;
    VARTRACE(numberOfVideoFiles);

    res = query.Update(LTY_DEVICES_UPDATE);
    if (res) return res;

     res = query.Change("--------------------------------ii--------------------------------ii", IN 0, IN numberOfVideoFiles, IN 0, IN deviceID);
    if (res) return res;

    /* clear not playable flag */
    tFileErrorHandling fileErrorHandling = (tFileErrorHandling)LocalSPM::GetDataProvider().FileErrorHandling();
    if ((FEH_MARK_OBJECT == fileErrorHandling)
        &&
        (0 < (int)numberOfVideoFiles))
    {
        ClearNotPlayableFlag(IN deviceID);//TODO(Kmv5cob):Clear flag only for Video files
    }


    return MP_NO_ERROR;
}
tResult DBManager::SetNumberOfImageFiles(const tDeviceID deviceID, const tNumberOfFiles numberOfImageFiles)
{
    ENTRY
    Query query;
    tResult res;

    res = query.Update(LTY_DEVICES_UPDATE);
    if (res) return res;
    res = query.Change("----------------------------------ii------------------------------ii", IN 0, IN numberOfImageFiles, IN 0, IN deviceID);
    if (res) return res;



    return MP_NO_ERROR;
}

tResult DBManager::RecalculateNumberOfFiles(const tDeviceID deviceID) const
{
    ENTRY
    Query query;
    tResult res;

    tNumberOfMediaObjects numberOfAudioObjectsInDB = 0;
    tNumberOfMediaObjects numberOfVideoObjectsInDB = 0;
    res = GetNumberOfPlayableObjectsInDB(OUT numberOfAudioObjectsInDB, OUT numberOfVideoObjectsInDB, IN deviceID);
    if (res) return res;

    res = query.Update(LTY_DEVICES_UPDATE);
    if (res) return res;

    res = query.Change("------------------------------iiii--------------------------------ii", IN 0, IN numberOfAudioObjectsInDB, IN 0, IN numberOfVideoObjectsInDB, IN 0, IN deviceID);
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::GetRepeatMode(tRepeatMode &repeatMode, const tDeviceID deviceID)
{
    ENTRY
    Query query;
    tResult res;

    res = query.Select(LTY_DEVICES, IN deviceID, "");
    if (res) return res;

    tMediaContext mediaContext;
    GetMediaContext( OUT mediaContext);
    /* Get the given devices repeat mode based on the media context*/
    if(MC_VIDEO == mediaContext)
        res = query.Get("-----------i", OUT &repeatMode);/*repeat mode for MC_VIDEO*/
    else
        res = query.Get("----------i", OUT &repeatMode);/*repeat mode for MC_AUDIO and MC_ALL*/
    query.End();

    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::ResetAllRepeatModes(const tDeviceID deviceID)
{
    ENTRY
    Query query;
    tResult res;

    res = query.Update(LTY_DEVICES_UPDATE);
    if (res) return res;

    res = query.Change("--------ii------------------------------------------------------iiii", IN 0, IN (tRepeatMode)LocalSPM::GetDataProvider().DBDefaultRepeatMode(), IN 0, IN (tRepeatMode)LocalSPM::GetDataProvider().DBDefaultVideoRepeatMode(), IN 0, IN deviceID);/*repeat mode for both video and audio*/
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::SetRepeatMode(const tDeviceID deviceID, const tRepeatMode repeatMode)
{
    ENTRY
    Query query;
    tResult res;

    res = query.Update(LTY_DEVICES_UPDATE);
    if (res) return res;

    tMediaContext mediaContext;
    GetMediaContext( OUT mediaContext);
    /* set the given devices repeat mode based on the media context*/
    if(MC_VIDEO == mediaContext)
        res = query.Change("----------------------------------------------------------------iiii", IN 0, IN repeatMode, IN 0, IN deviceID);/*repeat mode for MC_VIDEO*/
    else
        res = query.Change("--------ii--------------------------------------------------------ii", IN 0, IN repeatMode, IN 0, IN deviceID);/*repeat mode for MC_AUDIO and MC_ALL*/
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::GetPlaybackMode(tPlaybackMode &playbackMode, const tDeviceID deviceID,const tFileTypeSelection fileTypeSelection)
{
    ENTRY
    Query query;
    tResult res;

    res = query.Select(LTY_DEVICES, IN deviceID, "");
    if (res) return res;
    tMediaContext mediaContext;
    GetMediaContext( OUT mediaContext);
    /* Get the given devices Playback Mode based on the media context*/
    if(MC_VIDEO == mediaContext || FTS_VIDEO == fileTypeSelection)
        res = query.Get("-------------i", OUT &playbackMode);
    else
        res = query.Get("------------i", OUT &playbackMode);
    query.End();

    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::SetPlaybackMode(const tDeviceID deviceID, const tPlaybackMode playbackMode)
{
    ENTRY
    Query query;
    tResult res;

    res = query.Update(LTY_DEVICES_UPDATE_PLAYBACKMODE);
    if (res) return res;
    tMediaContext mediaContext;
    GetMediaContext( OUT mediaContext);
    /* set the given devices Playback Mode based on the media context*/
    if(MC_VIDEO == mediaContext)
        res = query.Change("--iiii", IN 0, IN playbackMode, IN 0, IN deviceID);
    else
        res = query.Change("ii--ii", IN 0, IN playbackMode, IN 0, IN deviceID);
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::ResetAllPlaybackModes(const tDeviceID deviceID)
{
    ENTRY
    Query query;
    tResult res;

    res = query.Update(LTY_DEVICES_UPDATE_PLAYBACKMODE);
    if (res) return res;

    res = query.Change("iiiiii", IN 0, IN PBM_NORMAL, IN 0, IN PBM_NORMAL, IN 0, IN deviceID);
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::SetDeviceNameAndConnectionState(const tDeviceID deviceID, const tDeviceName deviceName,
    const tConnectionState connectionState, const tDeviceState deviceState)
{
    ENTRY;
    VARTRACE(deviceID);
    VARTRACE(deviceName);
    VARTRACE(connectionState);
    VARTRACE(deviceState);

    Query query;
    tResult res;

    res = query.Update(LTY_DEVICES_UPDATE);
    if (res) return res;

    /* set the given devices connection state and name */
    res = query.Change("--it--------------ii--------ii------------------------------------ii",
            IN 0, IN deviceName,
            IN 0, IN connectionState,
            IN 0, IN deviceState,
            IN 0, IN deviceID);
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::SetDeviceType(const tDeviceID deviceID, const tDeviceType deviceType)
{
    ENTRY
    Query query;
    tResult res;

    res = query.Update(LTY_DEVICES_UPDATE);
    if (res) return res;

    /* set the given devices type */
    res = query.Change("------------ii----------------------------------------------------ii", IN 0, IN deviceType, IN 0, IN deviceID);
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::SetDeviceTypeAndMountPoint(const tDeviceID deviceID, const tDeviceType deviceType, const tMountPoint mountPoint)
{
    ENTRY
    Query query;
    tResult res;

    res = query.Update(LTY_DEVICES_UPDATE);
    if (res) return res;

    /* set the given devices type */
    res = query.Change("------------ii------------it--------------------------------------ii", IN 0, IN deviceType, IN 0, IN mountPoint, IN 0, IN deviceID);
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::SetDeviceName(const tDeviceID deviceID, const tDeviceName deviceName)
{
    ENTRY
    Query query;
    tResult res;

    res = query.Update(LTY_DEVICES_UPDATE);
    if (res) return res;

    /* set the given devices name */
    res = query.Change("--it--------------------------------------------------------------ii", IN 0, IN deviceName, IN 0, IN deviceID);
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::SetDiPOParameters(const tDeviceID deviceID, const tDiPOCaps diPOCaps, const tDiPOActive diPOActive, const tDiPOVersion diPOVersion)
{
    ENTRY;
    VARTRACE(deviceID);
    VARTRACE(diPOCaps);
    VARTRACE(diPOActive);
    VARTRACE(diPOVersion);

    Query query;
    tResult res;

    res = query.Update(LTY_DEVICES_UPDATE);
    if (res) return res;

    /* set the given devices DiPO parameters */
    res = query.Change("------------------------------------iiiiit------------------------ii",
            IN 0, IN diPOCaps,
            IN 0, IN diPOActive,
            IN 0, IN diPOVersion,
            IN 0, IN deviceID);
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::GetAlbumArtIndexingComplete(tAlbumArtIndexingComplete &complete, const tDeviceID deviceID)
{
    ENTRY
    Query query;
    tResult res;

    res = query.Select(LTY_DEVICES, IN deviceID, "");
    if (res) return res;

    res = query.Get("-------------------------------------i", OUT &complete);
    query.End();

    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::SetAlbumArtIndexingComplete(const tDeviceID deviceID, const tAlbumArtIndexingComplete complete)
{
    ENTRY;
    VARTRACE(deviceID);
    VARTRACE(complete);

    Query query;
    tResult res;

    res = query.Update(LTY_DEVICES_UPDATE);
    if (res) return res;

    /* set the given devices album art indexing complete flag */
    res = query.Change("--------------------------------------------------ii--------------ii", IN 0, IN complete, IN 0, IN deviceID);
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::DeviceDisconnected(const tDeviceID deviceID) //THIS is a test funtion only!
{
    ENTRY
    Query query;
    tResult res;

    res = query.Update(LTY_DEVICES_UPDATE);
    if (res) return res;

    /* set the given devices connection state to disconnected */
    res = query.Change("------------------ii--ii--------------------ii--------------------ii",
            IN 0, IN CS_DISCONNECTED,
            IN 0, IN CS_DISCONNECTED,       // set former connection state to disconnect
            IN 0, IN DR_REMOVED,
            IN 0, IN deviceID);
    if (res) return res;

    /* update the my media data base */
    res = RemoveDeviceFromMyMedia(IN deviceID);
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::DeviceConnected(const tDeviceID deviceID) //THIS is a test funtion only!
{
    ENTRY
    Query query;
    tResult res;

    res = query.Update(LTY_DEVICES_UPDATE);
    if (res) return res;

    /* set the given devices connection state to connected */
    res = query.Change("------------------ii----------------------------------------------ii", IN 0, IN CS_CONNECTED, IN 0, IN deviceID);
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::DeviceAttached(const tDeviceID deviceID) //THIS is a test funtion only!
{
    ENTRY
    Query query;
    tResult res;

    res = query.Update(LTY_DEVICES_UPDATE);
    if (res) return res;

    /* set the given devices connection state to attached */
    res = query.Change("------------------ii----------------------------------------------ii", IN 0, IN CS_ATTACHED, IN 0, IN deviceID);
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::ResetDevices(const tInitReason initReason)
{
    ENTRY
    Query query;
    tResult res;
    tConnectionState newConnectionState,newFormerConnectionState;

    /* loop over all connected devices and copy the old connection state to the "FormerConnectionState" column */
    res = query.Select(LTY_DEVICES, MY_MEDIA, "");
    if (res) return res;

    /* read all states */
    vector<tDeviceID> deviceIDs;
    vector<tDeviceType> deviceTypes;
    vector<tConnectionState> oldConnectionStates,oldFormerConnectionStates;
    while(1) {

        tDeviceID deviceID;
        tDeviceType deviceType;
        tConnectionState oldConnectionState,oldFormerConnectionState;

        /* read one entry */
        res = query.Get("i-------------i--i-i", OUT &deviceID, OUT &deviceType, OUT &oldConnectionState,OUT &oldFormerConnectionState);
        if (res) break;

        deviceIDs.push_back(deviceID);
        deviceTypes.push_back(deviceType);
        oldConnectionStates.push_back(oldConnectionState);
        oldFormerConnectionStates.push_back(oldFormerConnectionState);
    }

    query.End();
    if (res && res != MP_ERR_DB_END_OF_LIST) return res;

    /* write all old states to DB and set all current connection states to disconnected */
    for(unsigned int i=0; i<deviceIDs.size(); i++) {

        /* if device was connected before undervoltage set it to CS_UNDERVOLTAGE */
        if((IR_UNDERVOLTAGE == initReason)
           &&
           (CS_CONNECTED == oldConnectionStates[i]))
        {
            newConnectionState = CS_UNDERVOLTAGE;
            newFormerConnectionState = oldConnectionStates[i];
        }
        else
        {
            newConnectionState = CS_DISCONNECTED;

            /* NCG3D-2815:During Startup,if a device found to be auto-reinserted then,
             * If still not connected,Donot Forget its autoReinsertion.Else,forget its autoReinsertion
             *  */
            if((oldFormerConnectionStates[i] == CS_AUTO_REINSERTED) && (CS_ATTACHED == oldConnectionStates[i] || CS_DISCONNECTED == oldConnectionStates[i]))
            {
                newFormerConnectionState = oldFormerConnectionStates[i];
            }
            else
            {
                newFormerConnectionState = oldConnectionStates[i];
            }
        }

        const tDiPOVersion diPOVersion = {0};

        res = query.Update(LTY_DEVICES_UPDATE);
        if (res) return res;


        if(LocalSPM::GetDataProvider().SeparateMediaContent())
        {
            /* set the given devices connection state to disconnected */
            res = query.Change("------iiii--------ii--ii----ii------iiiiit----------------------iiii",
                   IN 0, IN 0,                     //active = false
                    IN LocalSPM::GetDataProvider().RepeatModeAdjustable(), IN (tRepeatMode)LocalSPM::GetDataProvider().DBDefaultRepeatMode(),
                    IN 0, IN newConnectionState,
                    IN 0, IN newFormerConnectionState,
                    IN 0, IN DS_NONE,
                    IN 0, IN tDiPOCaps_init,        //diPOCaps = DIPO_CAP_NONE
                    IN 0, IN 0,                     //diPOActive = false
                    IN 0, IN diPOVersion,           //diPOVersion = "\0"
                    IN LocalSPM::GetDataProvider().RepeatModeAdjustable(), IN (tRepeatMode)LocalSPM::GetDataProvider().DBDefaultVideoRepeatMode(),
                    IN 0, IN deviceIDs[i]);
            if (res) return res;
        }
        else
        {
            /* set the given devices connection state to disconnected */
            res = query.Change("------iiii--------ii--ii----ii------iiiiit------------------------ii",
                   IN 0, IN 0,                     //active = false
                    IN LocalSPM::GetDataProvider().RepeatModeAdjustable(), IN (tRepeatMode)LocalSPM::GetDataProvider().DBDefaultRepeatMode(),
                    IN 0, IN newConnectionState,
                    IN 0, IN newFormerConnectionState,
                    IN 0, IN DS_NONE,
                    IN 0, IN tDiPOCaps_init,        //diPOCaps = DIPO_CAP_NONE
                    IN 0, IN 0,                     //diPOActive = false
                    IN 0, IN diPOVersion,           //diPOVersion = "\0"
                    IN 0, IN deviceIDs[i]);
            if (res) return res;
        }
    }
    /* start a timer to reset the former connection state column */
    long timeoutInMS = LocalSPM::GetDataProvider().StartupTimeMS();

    mTimerResetDevices.StartTimer(OUT mTimerIDResetDevices, timeoutInMS, 0L, &LocalSPM::GetCustomControl(),
            &TimerCallBack, (char *)"ResetFormerConnectionTimer");  //lint !e1773

    return MP_NO_ERROR;
}

tResult DBManager::ResetAllFavorites()
{
    ENTRY
    tResult res;
    Query query;

    /* update the favorites  */
    res = query.Update(LTY_FAVORITES_UPDATE);
    if (res) return res;

    /* Update All Available and Active as false*/
    res = query.Change("iiii", IN 0, IN false, IN 0, IN false);
    if (res) return res;

    return MP_NO_ERROR;
}

bool DBManager::TimerCallBack(timer_t timerID , void* instance ,const void *userData)
{
    ENTRY_INTERNAL
    (void)timerID;
    CustomControl *customControl = (CustomControl*)instance;
    (void)userData;
    tResult res;

    res = customControl->SendResetFormerConnection();
    if (res) {
        ETG_TRACE_ERR(("TimerCallBack: SendResetFormerConnection returned %d", res));
    }

    return MP_NO_ERROR;
}

tResult DBManager::ResetFormerConnection(void)
{
    ENTRY
    Query query;
    tResult res;

    // cancel the timer
    mTimerResetDevices.CancelTimer(IN mTimerIDResetDevices);

    /* loop over all devices and set the  "FormerConnectionState" to CS_DISCONNECTED if current state is not CS_CONNECTED */
    res = query.Select(LTY_DEVICES, MY_MEDIA, "");
    if (res) return res;

    /* read all deviceIDs and currentConnectionState */
    vector<tDeviceID> deviceIDs;
    vector<tConnectionState> currentConnectionStates,formerConnectionStates;
    while(1) {

        tDeviceID deviceID;
        tConnectionState currentConnectionState,formerConnectionState;

        /* read one entry */
        res = query.Get("i----------------i-i", OUT &deviceID, OUT &currentConnectionState,OUT &formerConnectionState);
        if (res) break;

        deviceIDs.push_back(deviceID);
        currentConnectionStates.push_back(currentConnectionState);
        formerConnectionStates.push_back(formerConnectionState);
    }
    query.End();
    if (res && res != MP_ERR_DB_END_OF_LIST) return res;

    /* write CS_DISCONNECT into FormerConnectionState if ConnectionState != CS_CONNECTED  */
    for(unsigned int i=0; i<deviceIDs.size(); i++) {

        res = query.Update(LTY_DEVICES_UPDATE);
        if (res) return res;
        if (currentConnectionStates[i]!=CS_CONNECTED && formerConnectionStates[i] != CS_AUTO_REINSERTED) //When AutoReinsert occured,donot overwrite FormerConnectionstate -NCG3D-2815
        {
            /* set the given devices connection state to disconnected */
            res = query.Change("----------------------ii------------------------------------------ii", IN 0, IN CS_DISCONNECTED, IN 0, IN deviceIDs[i]);
            if (res) return res;
        }
    }
    return MP_NO_ERROR;
}

tResult DBManager::SetInternalDevicesToAttached(void)
{
    ENTRY
    Query query;
    tResult res;

    /* loop over all devices */
    res = query.Select(LTY_DEVICES, MY_MEDIA, "");
    if (res) return res;

    /* read all deviceIDs and deviceTypes */
    vector<tDeviceID> deviceIDs;
    vector<tDeviceType> deviceTypes;
    while(1) {

        tDeviceID deviceID;
        tDeviceType deviceType;

        /* read one entry */
        res = query.Get("i-------------i", OUT &deviceID, OUT &deviceType);
        if (res) break;

        deviceIDs.push_back(deviceID);
        deviceTypes.push_back(deviceType);
    }
    query.End();
    if (res && res != MP_ERR_DB_END_OF_LIST) return res;

    /* set device state to CS_ATTACHED for all internal devices */
    for(unsigned int i=0; i<deviceIDs.size(); i++)
    {
        if (deviceTypes[i] == DTY_FLASH || deviceTypes[i] == DTY_CS)
        {
            res = query.Update(LTY_DEVICES_UPDATE);
            if (res) return res;
            /* set the given devices connection state to attached */
            res = query.Change("------------------ii----------------------------------------------ii", IN 0, IN CS_ATTACHED, IN 0, IN deviceIDs[i]);
            if (res) return res;
        }
    }
    return MP_NO_ERROR;
}


tResult DBManager::GetActiveDevice(tDeviceID &deviceID)
{
    ENTRY
    Query query;
    tResult res;
    tActive active;

    /* loop over all devices */
    res = query.Select(LTY_DEVICES, MY_MEDIA, "");
    if (res) return res;

    while(1) {

        /* get next device */
        res = query.Get("i--------i", OUT &deviceID, OUT &active);
        if (res) break;
        if (active) {
            query.End();
            return MP_NO_ERROR;
        }
    }
    query.End();

    deviceID = DEVICE_ID_NOT_SET;
    return res;
}

tResult DBManager::SetActiveDevice(const tDeviceID deviceID)
{
    ENTRY
    return ActiveMediaDeviceSet(IN deviceID, IN true);
}

tResult DBManager::ActiveMediaDeviceSet(const tDeviceID deviceID, const tBoolean active)
{
    ENTRY
    tResult res;
    Query query;
    //time_t currentTime;

    /* get last active device */
    tDeviceID lastActiveDeviceID;
    GetActiveDevice(OUT lastActiveDeviceID);

    /* set all device to be not active */
    res = query.Update(LTY_DEVICES_UPDATE);
    if (res) return res;

    /* set all devices to not active */
    if (FALSE != active)
    {
        res = query.Change("------ii", IN 0, IN 0);
        if (res) return res;
    }

    /* get the current date for time stamp */
    //currentTime = time(NULL);

    /* set the given device as active */
    ETG_TRACE_USR2(("ActiveMediaDeviceSet: deviceID=%d, active=%d", deviceID, active));
    res = query.Change("------ii----------------------------------------------------------ii", IN 0, IN active, IN 0, IN deviceID);
    if (res) {
        if (FALSE != active) {
            ETG_TRACE_ERR(("Restore last active device=%d after DB error", lastActiveDeviceID));
            query.Change("------ii----------------------------------------------------------ii", IN 0, IN active, IN 0, IN lastActiveDeviceID);
        }
        return res;
    }

    return MP_NO_ERROR;
}

tResult DBManager::GetLastActiveDevice(tDeviceID &deviceID)
{
    ENTRY
    Query query;
    tResult res;

    /* look for the last active device */
    tActive active = true;
    res = query.Select(LTY_LASTMODE, MY_MEDIA, "ii", IN 0, IN active);
    if (res) return res;

    /* get the device id */
    res =query.Get("i", OUT &deviceID);
    query.End();

    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::GetLastModeDevices(tNumberOfDevices &numberOfDevices, vector<tDeviceID> &deviceIDs)
{
    ENTRY
    Query query;
    tResult res;

    /* prepare the sql statement for the last mode list */
    res = query.Select(LTY_LASTMODE, MY_MEDIA, "");
    if (res) return res;

    /* clear the vector */
    deviceIDs.clear();

    /* get all deviceIDs */
    tDeviceID deviceID;
    numberOfDevices = 0;
    while(1)
    {
        res = query.Get("i", OUT &deviceID);
        if (res) break;

        deviceIDs.push_back(deviceID);
        numberOfDevices++;
    }
    query.End();

    return MP_NO_ERROR;
}

tResult DBManager::GetCurrentMediaObject(tMediaObject &mediaObject, const tListID listID)
{
    ENTRY
    tResult res;
    DBList *dbList = NULL;
    UnlockOnExit unlockQuery(&dbList);

    /* get list pointer */
    //dbList = LookupDBList(IN listID);
    GET_DBLIST_POINTER(dbList, listID);
    if (!dbList) return MP_ERR_DB_LIST_NOT_FOUND;

    /* lock the query */
    dbList->mQuery->DoLock();

    /* get the last active object of the list without moving cursor forward */
    res =   GetMediaObjectInternal(OUT mediaObject, IN false/*doStep*/, IN listID, IN INDEX_LAST_ACTIVE_OBJECT);
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::GetObjectID(tObjectID &objectID, const tDeviceID deviceID, const tURL url)
{
    ENTRY
    Query query;
    tResult res;

    res = query.Select(LTY_MEDIAOBJECTS, IN deviceID, "it", IN 0, IN url);
    if (res) return res;

    res = query.Get("i", OUT &objectID);
    query.End();

    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::GetObjectDeviceID(tDeviceID &deviceID, const tObjectID objectID, const tCategoryType objectType)
{
    tResult res = -1;
    Query query;

    /* setup a query to get the deviceID from the objectid based on its type*/
    switch(objectType)
    {
        case CTY_SONG:          // audio object
        case CTY_AUDIOBOOK:     // audiobook object
        case CTY_VIDEO:         // video object
        case CTY_PODCAST:       // podcast object
        case CTY_CHAPTER:       // audio book chapter name
        case CTY_EPISODE:       // podcast or video episode name
        case CTY_PLAYLIST_ITEM:
            res = query.Select(LTY_MEDIAOBJECTS_DEVICEID_OBJECTID_FILTER, MY_MEDIA, "ii", IN 0, IN objectID);
            break;

        case CTY_GENRE:         // genre name
        case CTY_AUTHOR:        // audiobook author name
        case CTY_NAME:          // video or podcast name
        case CTY_PLAYLIST:      // playlist name
            res = query.Select(LTY_MEDIAOBJECTS_DEVICEID_OBJECTID_FILTER, MY_MEDIA, "--ii", IN 0, IN objectID);
            break;

        case CTY_ARTIST:        // artist name
        case CTY_TITLE:         // audiobook title
            res = query.Select(LTY_MEDIAOBJECTS_DEVICEID_OBJECTID_FILTER, MY_MEDIA, "----ii", IN 0, IN objectID);
            break;

        case CTY_COMPOSER:
            res = query.Select(LTY_MEDIAOBJECTS_DEVICEID_OBJECTID_FILTER, MY_MEDIA, "------ii", IN 0, IN objectID);
            break;

        case CTY_ALBUM:
            res = query.Select(LTY_MEDIAOBJECTS_DEVICEID_OBJECTID_FILTER, MY_MEDIA, "--------ii", IN 0, IN objectID);
            break;

        default:
            break;
    }
    if (res) return res;

    res = query.Get("i", OUT &deviceID);
    query.End();

    return res;
}


tResult DBManager::GetObjectDeviceID(vector<tDeviceID> &deviceIDs, const tObjectID objectID, const tCategoryType objectType)
{
    ENTRY
    tResult res = MP_NO_ERROR;
    Query query;
    /*Create new temporary variables, retobjectID and retobjectType for Object type retrieval*/
    tObjectID retobjectID = objectID;
    tCategoryType retobjectType = objectType;
    GetObjectType(OUT retobjectType, INOUT retobjectID);

    /* setup a query to get the deviceID from the objectid based on its type*/
    switch(retobjectType)
    {
        case CTY_SONG:          // audio object
        case CTY_AUDIOBOOK:     // audiobook object
        case CTY_VIDEO:         // video object
        case CTY_PODCAST:       // podcast object
        case CTY_CHAPTER:       // audio book chapter name
        case CTY_EPISODE:       // podcast or video episode name
        case CTY_PLAYLIST_ITEM:
            res = query.Select(LTY_MEDIAOBJECTS_DEVICEID_OBJECTID_FILTER, MY_MEDIA, "ii", IN 0, IN objectID);
            break;

        case CTY_GENRE:         // genre name
        case CTY_AUTHOR:        // audiobook author name
        case CTY_NAME:          // video or podcast name
        case CTY_PLAYLIST:      // playlist name
            res = query.Select(LTY_MEDIAOBJECTS_DEVICEID_OBJECTID_FILTER, MY_MEDIA, "--ii", IN 0, IN objectID);
            break;

        case CTY_ARTIST:        // artist name
        case CTY_TITLE:         // audiobook title
            res = query.Select(LTY_MEDIAOBJECTS_DEVICEID_OBJECTID_FILTER, MY_MEDIA, "----ii", IN 0, IN objectID);
            break;

        case CTY_COMPOSER:
            res = query.Select(LTY_MEDIAOBJECTS_DEVICEID_OBJECTID_FILTER, MY_MEDIA, "------ii", IN 0, IN objectID);
            break;

        case CTY_ALBUM:
            res = query.Select(LTY_MEDIAOBJECTS_DEVICEID_OBJECTID_FILTER, MY_MEDIA, "--------ii", IN 0, IN objectID);
            break;

        default:
            break;
    }
    if (res) return res;

    /* clear the vector */
    deviceIDs.clear();

     //get all deviceIDs
     tDeviceID deviceID;
     while(1) {
        deviceID = DEVICE_ID_NOT_SET;
        res = query.Get("i", OUT &deviceID);
        if (res) break;
        deviceIDs.push_back(deviceID);
    }
    query.End();

    if (res && (res != MP_ERR_DB_END_OF_LIST)) return res;

    return MP_NO_ERROR;
}

tResult DBManager::GetObjectType(tCategoryType &objectType, tObjectID &objectID)
{
    ENTRY
    tResult res;

    if(!(objectID & IPOD_MARKER_BIT))
    {
    Query query;

    /* setup a query to get the category from the object id */
    res = query.Select(LTY_OBJECTS, IN objectID, IN 0, IN 1, "");
    if (res) return res;

    res = query.Get("--i", OUT &objectType);
    query.End();

    /* if object id was not found, try to search the live tags table */
    if (res == MP_ERR_DB_END_OF_LIST) {

        /* look up the tag in the live tags table */
        tTag tag;
        tDeviceID deviceID;
        tNotPlayable notPlayable;
        res = GetLiveTagsElement(OUT tag, OUT objectType, OUT deviceID, OUT notPlayable, IN objectID);

        /* if also this failed: object is unknown */
        if (res) return res;

        /* lookup this tag in the regular db (in all possible columns) */
        res = query.Select(LTY_MEDIAOBJECTS_OBJECTID_BY_CATEGORY_FILTER, IN deviceID, "ittttt",
                IN objectType,
                IN tag,
                IN tag,
                IN tag,
                IN tag,
                IN tag);
        if (res) return res;

        /* read the real object ID and overwrite given one */
        res = query.Get("i", OUT &objectID);
        query.End();

        /* if this failed: object is unknown */
        if (res) return res;

    } else if (res) {
        return res;
    }
    }
    else
    {
        //remove marker bit
        tObjectID pureObjectID = objectID & IPOD_MARKER_MASK;

        //get URL from live tags
        tTag tag;
        tDeviceID deviceID;
        tNotPlayable notPlayable;
        res = GetLiveTagsElement(OUT tag, OUT objectType, OUT deviceID, OUT notPlayable, IN pureObjectID);
        if (res) return res;
        if (objectType != CTY_VTIPOD) {
            return MP_ERR_IPOD_METADATA;
        }

        //get CTP by LTY
        tTagID tagID;
        tListType listType = tListType_init;
        tTitle name;
        tUUID uuid;
        res = LocalSPM::GetIPODControl().GetVTTagByURL(OUT tagID, OUT listType, OUT name, OUT uuid, IN tag);
        if (res) return res;
        objectType = GetCTYByLTY(IN listType);
    }

    return MP_NO_ERROR;
}

tResult DBManager::GetMediaObject(tMediaObject &mediaObject, tMediaObjectID &mediaObjectID)
{
    ENTRY

    tResult res;
    tCategoryType categoryType;

    /* get the object type first */
    res = GetObjectType(OUT categoryType, INOUT mediaObjectID);
    if (res) return res;

    VARTRACE(categoryType);

    /* get the object */
    return GetMediaObject(OUT mediaObject, IN mediaObjectID, IN categoryType);
}

tResult DBManager::GetMediaObject(tMediaObject &mediaObject, const tMediaObjectID mediaObjectID,
        const tCategoryType _categoryType, const tDeviceID deviceID /*= MY_MEDIA*/,const tBoolean isRequestFromInterface /* =false*/)
{
    ENTRY
    tResult res;
    VARTRACE(mediaObjectID);
    VARTRACE(_categoryType);
    VARTRACE(deviceID);

    /* if no category type given, try to find out */
    tCategoryType categoryType = _categoryType;

    // if ID is not an ipod object then use the direct media object select from DB using objectID as input
    // avoid list creation for non-ipod object fetch
    if(!(mediaObjectID & IPOD_MARKER_BIT))
    {
        Query query;
        InitMediaObject(OUT mediaObject);

        /* Get media object from DB using direct select */
        res = query.Select(LTY_MEDIAOBJECTS_OBJECTID_FILTER, IN deviceID, "iiiii",
                IN mediaObjectID,  //objectID
                IN mediaObjectID,  //filterTag1
                IN mediaObjectID,  //filterTag2
                IN mediaObjectID,  //filterTag3
                IN mediaObjectID); //filterTag4
        if (res) return res;

        res = query.Get("iiiTTTiiTiTiTiTiiiTTiilTiiiiiTTi",
                OUT &mediaObject.objectID,
                OUT &mediaObject.catType,
                OUT &mediaObject.deviceID,
                OUT mediaObject.fileName, IN sizeof(mediaObject.fileName),
                OUT mediaObject.UUID, IN sizeof(mediaObject.UUID),
                OUT mediaObject.title, IN sizeof(mediaObject.title),
                OUT &mediaObject.trackNumber,
                OUT &mediaObject.MetadataTag1,
                OUT mediaObject.MetadataField1, IN sizeof(mediaObject.MetadataField1),
                OUT &mediaObject.MetadataTag2,
                OUT mediaObject.MetadataField2, IN sizeof(mediaObject.MetadataField2),
                OUT &mediaObject.MetadataTag3,
                OUT mediaObject.MetadataField3, IN sizeof(mediaObject.MetadataField3),
                OUT &mediaObject.MetadataTag4,
                OUT mediaObject.MetadataField4, IN sizeof(mediaObject.MetadataField4),
                OUT &mediaObject.metadataConvertFlag,
                OUT &mediaObject.compilationFlag,
                OUT &mediaObject.totalPlaytime,
                OUT mediaObject.albumArtString, IN sizeof(mediaObject.albumArtString),
                OUT mediaObject.path, IN sizeof(mediaObject.path),
                OUT &mediaObject.notPlayable,
                OUT &mediaObject.mediaType,
                OUT &mediaObject.fileSize,
                OUT mediaObject.dateTime, IN sizeof(mediaObject.dateTime),
                OUT &mediaObject.fileMode,
                OUT &mediaObject.userID,
                OUT &mediaObject.groupID,
                OUT &mediaObject.year,
                OUT &mediaObject.yearID,
                OUT mediaObject.deviceVersion, IN sizeof(mediaObject.deviceVersion),
                OUT mediaObject.mountPoint, IN sizeof(mediaObject.mountPoint),
                OUT &mediaObject.deviceType);
        query.End();
        if (res) return res;

        if(strlen_r(mediaObject.albumArtString))
        {
            /*  -------- fill additional fields which are not part of query result -------- */
            /* set the album art path as complete url including mount point */
            tAlbumArt completeAlbumArtString;
            strncpy_r(completeAlbumArtString, mediaObject.mountPoint, sizeof(completeAlbumArtString));
            strncat_r(completeAlbumArtString, mediaObject.albumArtString, sizeof(completeAlbumArtString));
            strncpy_r(mediaObject.albumArtString, completeAlbumArtString, sizeof(completeAlbumArtString));
        }

        //Fix GMMY16-27707
        if(CTY_PLAYLIST == mediaObject.catType)
        {
                    mediaObject.objectID = mediaObject.MetadataTag1;
        }
        else //Fix GMMY17-13354
        {
            if(isRequestFromInterface)
            {
                //Prepare mediaObject with requested ObjectID's information
                if (categoryType == CTY_NONE)
                {
                    tObjectID objectID = (tObjectID)mediaObjectID;
                    res = GetObjectType(OUT categoryType, INOUT objectID);
                    if (res) return res;
                }
                mediaObject.catType = categoryType;

                //Set the requested objectID
                mediaObject.objectID = mediaObjectID;
            }
            else
            {
                /*e.g Within GMP,If objectID of CTY_NAME requested,it may be to resolve whether CTY_NAME represents a video episode or a podcast episode.
                 * Hence,return the mediaObject whose any of the filterTagID is mediaObjectID.Hence returned mediaObject represents CTY_VIDEO or CTY_PODCAST
                 */
            }
        }
        //set mediatype based on it category type
        mediaObject.mediaType = GetMTYByCTY(mediaObject.catType);
    }
    else
    {
        if (categoryType == CTY_NONE) {

            /* get the category type */
            tObjectID objectID = (tObjectID)mediaObjectID;
            res = GetObjectType(OUT categoryType, INOUT objectID);
            if (res) return res;
            if (categoryType == CTY_NONE) return MP_ERR_DB_UNSUPPORTED_CAT_TYPE;
        }

        tListID listID;
        tPosition position = POSITION_NOT_SET;

        /* create a temporary list */
        res = CreateList(OUT listID, IN mediaObjectID, IN GetLTYByCTY(categoryType), IN PC_NO_REPEAT, IN deviceID);
        if (res) return res;

        if (categoryType == CTY_SONG) {
            /* find the object in the list */
            res = GetPositionInList(OUT position, IN listID, IN mediaObjectID);
            if (res) return res;

            if (position == POSITION_NOT_SET) {
                ReleaseDBList(IN listID);
                return MP_ERR_DB_END_OF_LIST;
            }
        }
        /* get the object */
        res = GetMediaObjectInternal(OUT mediaObject, IN false/*doStep*/, IN listID, IN position);
        if (res) return res;

        /* release the temporary list */
        ReleaseDBList(IN listID);
    }

    return MP_NO_ERROR;
}

tResult DBManager::UpdateMediaObject(const tMediaObject mediaObject)
{
    ENTRY;
    tResult res;
    Query query;

    /* update the MediaObjects table */
    res = query.Update(LTY_MEDIAOBJECTS_UPDATE);
    if (res) return res;

    /* find match for mediaObjectID and update notPlayable flag */
    res = query.Change("----------------ii----ii------------------ii",
            IN 0, mediaObject.totalPlaytime,
            IN 0, IN mediaObject.notPlayable,
            IN 0, IN mediaObject.objectID);     //for WHERE clause
    if (res) return res;

    /* decrement numberOfAudioFiles and numberOfVideoFiles for the device is done by internal DB trigger now */

    return MP_NO_ERROR;
}

tResult DBManager::SetCurrentRow(const tListID listID, const tRowNumber newRow)
{
    ENTRY
    VARTRACE(listID);
    VARTRACE(newRow);

    Query query;
    tResult res;
    DBList *dbList = NULL;
    UnlockOnExit unlockQuery(&dbList);

    /* get list pointer */
    //dbList = LookupDBList(IN listID);
    GET_DBLIST_POINTER(dbList, listID);
    if (!dbList) return MP_ERR_DB_LIST_NOT_FOUND;

    /* lock the query */
    dbList->mQuery->DoLock();

    /* set the current media object id in the list */
    res = dbList->mQuery->SetRow(IN newRow);
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::GetCurrentRow(tRowNumber &currentRow, const tListID listID)
{
    ENTRY
    tResult res;
    DBList *dbList = NULL;
    UnlockOnExit unlockQuery(&dbList);

    /* get list pointer */
    //dbList = LookupDBList(IN listID);
    GET_DBLIST_POINTER(dbList, listID);
    if (!dbList) return MP_ERR_DB_LIST_NOT_FOUND;

    /* lock the query */
    dbList->mQuery->DoLock();

    /* get the current row */
    res = dbList->mQuery->GetRow(OUT currentRow);
    VARTRACE(currentRow);
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::SetPosition(const tObjectID objectID, const tListID listID)
{
    tPosition position;
    return SetPosition(OUT position, IN objectID, IN listID);
}

tResult DBManager::SetPosition(tPosition &position, const tObjectID objectID, const tListID listID)
{
    ENTRY
    tResult res;
    DBList *dbList = NULL;
    UnlockOnExit unlockQuery(&dbList);

    /* get list pointer */
    //dbList = LookupDBList(IN listID);
    GET_DBLIST_POINTER(dbList, listID);
    if (!dbList) return MP_ERR_DB_LIST_NOT_FOUND;

    /* check the list type */
    if (false == IsPlayableList(dbList->mListType))
    {
        return MP_ERR_DB_UNSUPPORTED_LIST_TYPE;
    }

    if((dbList->mListType == LTY_FILELIST_PLAYLIST) ||
            (dbList->mListType == LTY_PLAYLIST_SONG))
    {
        /* Playlist may contain duplication of songs, thus getting the row position */
        res = GetCurrentRow(OUT position, IN listID);
        if (res) return res;
    }
    else
    {
        /* get the position in the list */
        res = GetPositionInList(OUT position, IN listID, IN objectID);
        if (res) return res;
    }

    if (position == POSITION_NOT_SET) return MP_ERR_DB_END_OF_LIST;

    /* lock the query */
    dbList->mQuery->DoLock();

    /* and set this as current position */
    res = dbList->mQuery->SetRow(IN position);
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::SetPlaylistAttributes(const tListID listID, const tDeviceID deviceID, const tFilterTag1 filterTag1)
{
    ENTRY
    DBList *dbList = NULL;

    /* get list pointer */
    //dbList = LookupDBList(IN listID);
    GET_DBLIST_POINTER(dbList, listID);
    if (!dbList) return MP_ERR_DB_LIST_NOT_FOUND;

    /* check the list type */
    if( (LTY_PLAYLIST_SONG != dbList->mListType)
        &&
        (LTY_FILELIST_PLAYLIST != dbList->mListType))
    {
        return MP_ERR_DB_UNSUPPORTED_LIST_TYPE;
    }

    dbList->mDeviceID = deviceID;
    dbList->mFilterTag1.tag = filterTag1.tag;

    //ETG_TRACE_USR3(("DBManager::SetPlaylistAttributes dbListID:%u, deviceID:%u, filterTag1:%u", listID, dbList->mDeviceID, dbList->mFilterTag1.tag));

    return MP_NO_ERROR;
}

tResult DBManager::RemoveMediaObject(const tMediaObjectID mediaObjectID)
{
    ENTRY
    tResult res;
    Query query;

    res = query.Delete(LTY_SONGS_DELETE);
    if (res) return res;

    res = query.Remove("ii", IN 0, IN mediaObjectID);
    if (res) return res;

    /* decrement numberOfAudioFiles and numberOfVideoFiles for the device is done by internal DB trigger now */

    res = query.Delete(LTY_MEDIAOBJECTS_MYME_DELETE);
    if (res) return res;

    res = query.Remove("ii", IN 0, IN mediaObjectID);
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::RemoveMediaObjectByUUID(const tDeviceID deviceID, const tUUID UUID) const
{
    ENTRY
    tResult res;
    Query query;

    res = query.Delete(LTY_SONGS_DELETE);
    if (res) return res;

    res = query.Remove("--itii", IN 0, IN UUID, IN 0, IN deviceID);
    if (res) return res;

    /* decrement numberOfAudioFiles and numberOfVideoFiles for the device is done by internal DB trigger now */

    res = query.Delete(LTY_MEDIAOBJECTS_MYME_DELETE);
    if (res) return res;

    res = query.Remove("--itii", IN 0, IN UUID, IN 0, IN deviceID);
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::RemoveMediaObjects(const tDeviceID deviceID)
{
    ENTRY
    tResult res;
    Query query;

    /* delete all elements in song table for a specific device id */
    res = query.Delete(LTY_SONGS_DELETE);
    if (res) return res;

    res = query.Remove("----ii", IN 0, IN deviceID);
    if (res) return res;

    /* recreate my media */
    res = RemoveDeviceFromMyMedia(IN deviceID);
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::GetListInfo(tListInfo &listInfo, const tListID listID, const tBoolean withListSize)
{
    ENTRY
    DBList *dbList = NULL;
    tListSize listSize = 0;

    /* get list pointer */
    //dbList = LookupDBList(IN listID);
    GET_DBLIST_POINTER(dbList, listID);
    if (!dbList) return MP_ERR_DB_LIST_NOT_FOUND;

    /* fill in the struct */
    InitListInfo(OUT listInfo);

    listInfo.listID = listID;
    listInfo.listType = dbList->mListType;
    listInfo.deviceID = dbList->mDeviceID;
    listInfo.playContinuation = dbList->mPlayContinuation;
    listInfo.filterTag1.tag = dbList->mFilterTag1.tag;
    listInfo.filterTag2.tag = dbList->mFilterTag2.tag;
    listInfo.filterTag3.tag = dbList->mFilterTag3.tag;
    listInfo.filterTag4.tag = dbList->mFilterTag4.tag;
    strncpy_r(listInfo.path, dbList->mPath, sizeof(listInfo.path));
    listInfo.fileTypeSelection = dbList->mFileTypeSelection;
    listInfo.isVTIPODList = dbList->mIsVTIPODList;
    listInfo.streaming = dbList->mStreaming;

    /* replace list type in case of filelist which might be a playlist*/
    //if(listInfo.listType == LTY_FILELIST)
    //    listInfo.listType = listPtr->listType;
    if( LTY_FILELIST_PLAYLIST == listInfo.listType )
    {
        listInfo.listType = LTY_PLAYLIST_SONG;
    }

    if (false == withListSize) return MP_NO_ERROR;

    /* get the list size */
    tResult res;
    res = GetListSize(OUT listSize, IN dbList->mListID);
    if (res) return res;

    listInfo.listSize = listSize;

    return MP_NO_ERROR;
}

tResult DBManager::GetPositionInList(tPosition &position, const tListID listID, const tObjectID objectID)
{
    ENTRY
    VARTRACE(listID);
    VARTRACE(objectID);
    tResult res = MP_NO_ERROR;
    position = POSITION_NOT_SET;
    DBList *dbList = NULL;
    UnlockOnExit unlockQuery(&dbList);

    /* get list pointer */
    //dbList = LookupDBList(IN listID);
    GET_DBLIST_POINTER(dbList, listID);
    if (!dbList) return MP_ERR_DB_LIST_NOT_FOUND;

    /* lock the query */
    dbList->mQuery->DoLock();

    /* check the list type */
    if (false == IsPlayableList(dbList->mListType))
    {
        return MP_ERR_DB_UNSUPPORTED_LIST_TYPE;
    }

    /* special handling for IPOD object ID's */
    if (dbList->mIsVTIPODList) {
        /* get back VT tag from live tags table */
        tTagID vtTagID = objectID; // starts with 1
        res = GetVTTagForIPOD(vtTagID);
        if (res) return res;

        position = (vtTagID & IPOD_MARKER_MASK) -1;
        VARTRACE(position);
        return MP_NO_ERROR;
    }

    /* to use a sub query is not fast for every list type */
    switch(dbList->mListType) {

    /* do a simple position search */
    case LTY_FILELIST:
    case LTY_FILELIST_UNSORTED:
    case LTY_FILELIST_PLAYLIST:
    case LTY_FILELIST_MEDIAOBJECTS:
    case LTY_FILELIST_MEDIAOBJECTS_WITH_SUBFOLDERS:
    case LTY_MTP_FILELIST:
    {
        /* get the file name represented by objectID from LiveTags */
        tDeviceID deviceID = MY_MEDIA;
        tTag fileName;
        tCategoryType catType = CTY_FILE;
        tNotPlayable notPlayable = FNP_PLAYABLE;
        res = GetLiveTagsElement(OUT fileName, OUT catType, OUT deviceID, OUT notPlayable, IN objectID);
        if (res)
        {
            /* if objectID is not in LiveTags table search in MediaObjects table*/
            res = GetURLOfMediaObject(OUT fileName, IN (tTagID)objectID, IN CTY_SONG);
            if (res) return res;
        }
        VARTRACE(fileName)


        /* search for the filename in the file list to find its position in the list */
        TimeTrace ticks("GetPositionInFileList");

        res = StartListAccess(IN listID, IN 0, INDEX_NO_LIMIT);
        if (res) return res;

        /* loop over all elements of this list */
        tMediaObject mediaObject;
        tBoolean doStep = true;
        for(tPosition row=0; 1; row++)
        {
            /* get one VT entry */
            InitMediaObject(OUT mediaObject);

            if(dbList->mListType == LTY_MTP_FILELIST)
            {
                res = dbList->mQuery->Get(IN doStep, IN "i--TT",
                        OUT &mediaObject.objectID,
                        OUT mediaObject.path, IN sizeof(mediaObject.path),
                        OUT mediaObject.title, IN sizeof(mediaObject.title));
            }
            else
            {
                res = dbList->mQuery->Get(IN doStep, IN "i-TT",
                        OUT &mediaObject.objectID,
                        OUT mediaObject.path, IN sizeof(mediaObject.path),
                        OUT mediaObject.title, IN sizeof(mediaObject.title));
            }
            if (res) break;

            /*create complete path(combine basepath<path> and pure filename<title> ) and set it to mediaObject.fileName*/
            if(NULL == FastUTF8::StartsWith((const FastUTF8::tString)mediaObject.path, (const FastUTF8::tString)"/"))  //lint !e1773
            {
                strncpy_r(mediaObject.fileName, "/", sizeof(mediaObject.fileName));
                strncat_r(mediaObject.fileName, mediaObject.path, sizeof(mediaObject.fileName));
            }
            else
            {
                strncpy_r(mediaObject.fileName, mediaObject.path, sizeof(mediaObject.fileName));
            }
            if (strlen_r(mediaObject.path)) {
                if(NULL == FastUTF8::EndsWithNC((const FastUTF8::tString)mediaObject.path, (const FastUTF8::tString)"/"))  //lint !e1773
                {
                    strncat_r(mediaObject.fileName, "/", sizeof(mediaObject.fileName));
                }
            }
            strncat_r(mediaObject.fileName, mediaObject.title, sizeof(mediaObject.fileName));
            if (strcmp(mediaObject.fileName, fileName) == 0)
            {
                position = row;
                break;
            }
        }

        EndListAccess(IN listID);

        ticks.elapsed();
        break;
    }
    case LTY_BLUETOOTH_FILELIST:
    {
        /*BT Device's MediaObjects are not being stored in LiveTag or MediaObjects Table.
                 Hence only way to check the position of the objectID is to query the vtable of
                 bluetooth and to check whether the ObjectID available in the List */

        //VTBluetooth the objectID is prepared from rowid.
        //eg if objectID=3,then  rowid=2.Hence query to know the row exists.
        //KNOWN BUG!!-row may exist but item in row is different from user selection.Hence,mediaObject.UUID be used to get position.
        if((objectID > 0) && (OBJECT_ID_NONE != objectID))
        {
            tPosition expectedRow = objectID-1;
            res = StartListAccess(IN listID, IN expectedRow, 1);

            if(MP_ERR_DB_LIST_NOT_FOUND == res)
            {
                break;
            }

            if(MP_NO_ERROR == res)
            {
                tMediaObject mediaObject;
                tBoolean doStep = true;

                /* get one VT entry */
                InitMediaObject(OUT mediaObject);

                res = dbList->mQuery->Get(IN doStep, IN "i",
                        OUT &mediaObject.objectID);

                if (MP_NO_ERROR == res)
                {
                    position = expectedRow;
                }
            }

            tResult resEndList = EndListAccess(IN listID);
            if(MP_NO_ERROR != resEndList)
            {
                ETG_TRACE_USR4(("GetPositionInList-> EndListAccess for BT returned %d",resEndList));
            }
        }
        break;
    }
    case LTY_CD:
    {
        res = StartListAccess(IN listID, IN 0, INDEX_NO_LIMIT);
        if (res) return res;

        /* loop over all elements of this list */
        tMediaObject mediaObject;
        tBoolean doStep = true;
        for(tPosition row=0; 1; row++)
        {
            /* get one VT entry */
            InitMediaObject(OUT mediaObject);

            res = dbList->mQuery->Get(IN doStep, IN "i",
                    OUT &mediaObject.objectID);
            if (res) break;

            if (mediaObject.objectID == objectID)
            {
                position = row;
                break;
            }
        }

        EndListAccess(IN listID);

        break;
    }
    case LTY_BROWSELIST:
    case LTY_BROWSELIST_MEDIAOBJECTS:
    case LTY_BROWSELIST_MEDIAOBJECTS_WITH_SUBFOLDERS:
    {
        Query query;
        VARTRACE(dbList->mDeviceID);

        if(LTY_BROWSELIST_MEDIAOBJECTS_WITH_SUBFOLDERS == dbList->mListType)
        {
            tCategoryType catType1 = CTY_DIR;
            tCategoryType catType2 = CTY_DIR;
            tCategoryType catType3 = CTY_DIR;
            tCategoryType catType4 = CTY_DIR;
            GetCTYByFTS(OUT catType1, OUT catType2, OUT catType3, OUT catType4, IN dbList->mFileTypeSelection, IN true/*onlyPlayableObjects*/);

            if( strlen_r(dbList->mPath) > 1 )
            {
#if NO_FOLDER_HIERARCHY
                res = query.PositionSelect(IN dbList->mQuery, IN dbList->mDeviceID, "iitii",
                        IN ((strlen_r(dbList->mPath) > 1) ? 0 : 1),
                        IN strlen_r(dbList->mPath),
                        IN dbList->mPath,
                        IN catType1, IN catType2);
#else
                /* trim last character if it is "/" */
                tURL folderName;
                strncpy_r(OUT folderName, IN dbList->mPath, IN sizeof(folderName));
                if(FastUTF8::EndsWithNC((const FastUTF8::tString)folderName, (const FastUTF8::tString)"/")) //lint !e1773
                {
                    unsigned char *pureFile;
                    FastUTF8::Split(OUT pureFile, INOUT (unsigned char *)folderName); //lint !e1773
                }
                VARTRACE(folderName);

                res = query.PositionSelect(IN dbList->mQuery, IN dbList->mDeviceID, "tii",
                        IN folderName,
                        IN catType1, IN catType2);
#endif // #if NO_FOLDER_HIERARCHY
            }
            else // root folder -> select all / do not make string compare on path
            {
                res = query.PositionSelect(IN dbList->mQuery, IN dbList->mDeviceID, "ii",
                        IN catType1, IN catType2);
            }
        }
        else if(LTY_BROWSELIST_MEDIAOBJECTS == dbList->mListType)
        {
            VARTRACE(dbList->mPath);

#if FILTER_FILE_LIST_VIA_CTY
            tCategoryType catType1 = CTY_DIR;
            tCategoryType catType2 = CTY_DIR;
            tCategoryType catType3 = CTY_DIR;
            tCategoryType catType4 = CTY_DIR;
            GetCTYByFTS(OUT catType1, OUT catType2, OUT catType3, OUT catType4, IN dbList->mFileTypeSelection, IN true/*onlyPlayableObjects*/);

            res = query.PositionSelect(IN dbList->mQuery, IN dbList->mDeviceID, "tiii",
                    IN dbList->mPath,
                    IN catType1, IN catType2, IN CTY_DIR);
#else
            tMediaContent mediaContent = MTY_UNKNOWN;
            GetMediaContentByFTS(OUT mediaContent, IN dbList->mFileTypeSelection, IN true/*onlyPlayableObjects*/);

            res = query.PositionSelect(IN dbList->mQuery, IN dbList->mDeviceID, "ti",
                    IN dbList->mPath,
                    IN IN mediaContent);
#endif // #if FILTER_FILE_LIST_VIA_CTY
        }
        else //LTY_BROWSELIST
        {
            VARTRACE(dbList->mPath);

            /* Set the media context filter related to FTS */
            if(FTS_ALL != dbList->mFileTypeSelection) //not FTS_ALL
            {
#if FILTER_FILE_LIST_VIA_CTY
                tCategoryType catType1 = CTY_DIR;
                tCategoryType catType2 = CTY_DIR;
                tCategoryType catType3 = CTY_DIR;
                tCategoryType catType4 = CTY_DIR;
                GetCTYByFTS(OUT catType1, OUT catType2, OUT catType3, OUT catType4, IN dbList->mFileTypeSelection, IN false/*onlyPlayableObjects*/);

                res = query.PositionSelect(IN dbList->mQuery, IN dbList->mDeviceID, "tiiiiii",
                        IN dbList->mPath,
                        IN 0, IN catType1, IN catType2, IN catType3, IN catType4, IN CTY_DIR);
#else
                tMediaContent mediaContent = MTY_UNKNOWN;
                GetMediaContentByFTS(OUT mediaContent, IN dbList->mFileTypeSelection, IN false/*onlyPlayableObjects*/);

                res = query.PositionSelect(IN dbList->mQuery, IN dbList->mDeviceID, "tii",
                        IN dbList->mPath,
                        IN 0, IN mediaContent);
#endif // #if FILTER_FILE_LIST_VIA_CTY
            }
            else
            {
                res = query.PositionSelect(IN dbList->mQuery, IN dbList->mDeviceID, "t", IN dbList->mPath);
            }
        }
        if (res) return res;

        /* get the position */
        res = query.PositionGet(OUT position, IN objectID);
        query.End();

        break;
    }

    case LTY_IMAGE_FOLDER_ITEM:
    {
        Query query;

        res = query.PositionSelect(IN dbList->mQuery, dbList->mDeviceID, IN "t", dbList->mPath);
        if (res) return res;
        /* get the position */
        res = query.PositionGet(OUT position, IN objectID);
        query.End();

        break;
    }

    case LTY_ALBUM:
    {
        Query selectQuery;
        tUInt filterTag4Id;
        res = selectQuery.Select(LTY_MEDIAOBJECTS_FILTERTAG4ID_OBJECTID_FILTER, dbList->mDeviceID, "i", IN objectID);
        if (res) return res;

        res = selectQuery.Get("i", OUT &filterTag4Id);
        if (res) return res;
        selectQuery.End();
        VARTRACE(filterTag4Id);

        Query positionQuery;
        res = positionQuery.PositionSelect(IN LTY_ALBUM_POSITION,IN dbList->mDeviceID,IN "");
        if (res)
            return res;
        /* get the position */
        res = positionQuery.PositionGet(OUT position, IN filterTag4Id);
        positionQuery.End();

        break;
    }

    /* for all others: do a sub query search */
    default:
    {
        Query query;

        /* take real deviceID in case of a MyMedia playlist */
        tDeviceID realDeviceID = dbList->mDeviceID;
        if ((MY_MEDIA == dbList->mDeviceID)
            &&
            (dbList->mListType == LTY_PLAYLIST_SONG)) {

            res = GetObjectDeviceID(OUT realDeviceID, IN dbList->mFilterTag1.tag, CTY_PLAYLIST);
            if (res) return res;
        }

        if((LocalSPM::GetDataProvider().PersonalizationSupported()) && IsUserFavoriteList(dbList->mListType))
         {
          /* prepare the sql statement for object position select */
          res = query.PositionSelect(IN dbList->mQuery, IN realDeviceID, IN "ii",IN 0, IN dbList->mUserID);
          ETG_TRACE_USR3(("PositionSelect  Error=%d/%s", res, errorString(res)));
          if (res) return res;
         }
        else
         {
        /* prepare the sql statement for object position select */
        tGeneralString formatBuffer;
        SelectFormatString(INOUT formatBuffer, IN dbList->mFilterTag1, IN dbList->mFilterTag2, IN dbList->mFilterTag3, IN dbList->mFilterTag4, IN dbList->mSearchString);
        res = query.PositionSelect(IN dbList->mQuery, IN realDeviceID, IN formatBuffer,
                IN dbList->mFilterTag1.tag,
                IN dbList->mFilterTag2.tag,
                IN dbList->mFilterTag3.tag,
                IN dbList->mFilterTag4.tag);
        if (res) return res;
        }

        /* get the position */
        res = query.PositionGet(OUT position, IN objectID);
        query.End();

        break;
    }}

    if (res == MP_ERR_DB_END_OF_LIST) { // if DB returns no entry: list is empty
        position = POSITION_NOT_SET;
        res = MP_NO_ERROR;
    }

    VARTRACE(position);
    return res;
}

tResult DBManager::GetPositionInParentFileList(tPosition &position, const tListID listID, const tPath nowPlayingFolder)
{
    ENTRY
    VARTRACE(nowPlayingFolder);
    tResult res = MP_NO_ERROR;
    position = POSITION_NOT_SET;
    DBList *dbList = NULL;
    UnlockOnExit unlockQuery(&dbList);

    /* get list pointer */
    //dbList = LookupDBList(IN listID);
    GET_DBLIST_POINTER(dbList, listID);
    if (!dbList) return MP_ERR_DB_LIST_NOT_FOUND;

    /* check the list type */
    if ((dbList->mListType != LTY_FILELIST)
        &&
        (dbList->mListType != LTY_MTP_FILELIST)
        &&
        (dbList->mListType != LTY_BROWSELIST))
    {
        return MP_ERR_DB_UNSUPPORTED_LIST_TYPE;
    }

    /* to use a sub query is not fast for every list type */
    switch(dbList->mListType) {

    /* do a simple position search */
    case LTY_FILELIST:
    case LTY_MTP_FILELIST:
    {
        /*search for the filename in the file list to find its position in the list*/
        TimeTrace ticks("GetPositionInParentFileList");

        res = StartListAccess(IN listID, IN 0, INDEX_NO_LIMIT);
        if (res) return res;

        /* loop over all elements of this list */
        tMediaObject mediaObject;
        tBoolean doStep = true;
        for(tPosition row=0; 1; row++)
        {
            /* get one VT entry */
            InitMediaObject(OUT mediaObject);

            if(dbList->mListType == LTY_MTP_FILELIST)
            {
                res = dbList->mQuery->Get(IN doStep, IN "i--TTi",
                        OUT &mediaObject.objectID,
                        OUT mediaObject.path, IN sizeof(mediaObject.path),
                        OUT mediaObject.title, IN sizeof(mediaObject.title),
                        OUT &mediaObject.fileType);
            }
            else
            {
                res = dbList->mQuery->Get(IN doStep, IN "i-TTi",
                        OUT &mediaObject.objectID,
                        OUT mediaObject.path, IN sizeof(mediaObject.path),
                        OUT mediaObject.title, IN sizeof(mediaObject.title),
                        OUT &mediaObject.fileType);
            }
            if (res) break;

            /*create complete path(combine basepath<path> and pure filename<title> ) and set it to mediaObject.fileName*/
            if(NULL == FastUTF8::StartsWith((const FastUTF8::tString)mediaObject.path, (const FastUTF8::tString)"/"))  //lint !e1773
            {
                strncpy_r(OUT mediaObject.fileName, IN "/", IN sizeof(mediaObject.fileName));
                strncat_r(OUT mediaObject.fileName, IN mediaObject.path, IN sizeof(mediaObject.fileName));
            }
            else
            {
                strncpy_r(OUT mediaObject.fileName, IN mediaObject.path, IN sizeof(mediaObject.fileName));
            }
            if (strlen_r(mediaObject.path)) {
                if(NULL == FastUTF8::EndsWithNC((const FastUTF8::tString)mediaObject.path, (const FastUTF8::tString)"/"))  //lint !e1773
                {
                    strncat_r(OUT mediaObject.fileName, IN "/", IN sizeof(mediaObject.fileName));
                }
            }
            strncat_r(OUT mediaObject.fileName, IN mediaObject.title, IN sizeof(mediaObject.fileName));

            if(FT_FOLDER == mediaObject.fileType)
            {
                tPath compareFolder;
                unsigned char *folder;
                strncpy_r(OUT compareFolder, IN nowPlayingFolder, IN sizeof(compareFolder));
                while (strlen_r(compareFolder) > strlen_r(mediaObject.fileName))
                {
                    FastUTF8::Split(OUT folder, INOUT (unsigned char *)compareFolder);
                    if( NULL == folder ) break;
                }

                if(strcmp(mediaObject.fileName, compareFolder) == 0)
                {
                    //ETG_TRACE_USR3(("DBManager::GetPositionInParentFileList row:%u, compareFolder:%40s, mediaObject.fileName:%40s", row, compareFolder, mediaObject.fileName));
                    position = row;
                    break;
                }
            }
        }

        EndListAccess(IN listID);

        ticks.elapsed();
        break;
    }
    case LTY_BROWSELIST:
    {
        Query query1;
        tObjectID parentID;
        tUInt depth;
        tPath compareFolder;
        strncpy_r(OUT compareFolder, IN nowPlayingFolder, IN sizeof(compareFolder));
        if(FastUTF8::EndsWithNC((const FastUTF8::tString)compareFolder, (const FastUTF8::tString)"/")) //lint !e1773
        {
            unsigned char *folder;
            FastUTF8::Split(OUT folder, INOUT (unsigned char *)compareFolder); //lint !e1773
        }
        res = query1.Select(LTY_FOLDERHIERARCHY_PARENTID_BY_CHILD_NAME, IN dbList->mDeviceID, "t", IN compareFolder);
        if (res) return res;

        Query query2;
        //VARTRACE(dbList->mDeviceID);
        //VARTRACE(dbList->mPath);
        /* Set the media context filter related to FTS */
        if(FTS_ALL != dbList->mFileTypeSelection) //not FTS_ALL
        {
#if FILTER_FILE_LIST_VIA_CTY
            tCategoryType catType1 = CTY_DIR;
            tCategoryType catType2 = CTY_DIR;
            tCategoryType catType3 = CTY_DIR;
            tCategoryType catType4 = CTY_DIR;
            GetCTYByFTS(OUT catType1, OUT catType2, OUT catType3, OUT catType4, IN dbList->mFileTypeSelection, IN false/*onlyPlayableObjects*/);

            res = query2.PositionSelect(IN dbList->mQuery, IN dbList->mDeviceID, "tiiiiii",
                    IN dbList->mPath,
                    IN 0, IN catType1, IN catType2, IN catType3, IN catType4, IN CTY_DIR);
#else
            tMediaContent mediaContent = MTY_UNKNOWN;
            GetMediaContentByFTS(OUT mediaContent, IN dbList->mFileTypeSelection, IN false/*onlyPlayableObjects*/);

            res = query2.PositionSelect(IN dbList->mQuery, IN dbList->mDeviceID, "tii",
                    IN dbList->mPath,
                    IN 0, IN mediaContent);
#endif // #if FILTER_FILE_LIST_VIA_CTY
        }
        else
        {
            res = query2.PositionSelect(IN dbList->mQuery, IN dbList->mDeviceID, "t", IN dbList->mPath);
        }
        if (res) return res;

        do {
            /* get all parentIDs of the child folder */
            res = query1.Get("ii", OUT &parentID, OUT &depth);
            VARTRACE(parentID);
            if (res) break;

            /* get the position of the parent folder */
            res = query2.PositionGet(OUT position, IN parentID);
        } while (res);
        query1.End();
        query2.End();

        break;
    }
    default:
    {
        ETG_TRACE_ERR(("DBManager::GetPositionInParentFileList: Not supported list type %u", dbList->mListType));
        break;
    }}

    VARTRACE(position);
    return MP_NO_ERROR;
}

tResult DBManager::SetLanguage(const tLanguageType language)
{
    ENTRY

    mActiveLanguage = language;

    return MP_NO_ERROR;
}

tResult DBManager::GetLanguage(tLanguageType &language)
{
    ENTRY

    language = mActiveLanguage;

    return MP_NO_ERROR;
}


tResult DBManager::GetSearchKeyboard(vector<tSearchKeyboard> &searchKeyboardList, const tListID listID)
{
    ENTRY

    tResult res;

    /*prepare the start character list along with it occurrence for a given list ID*/
    vector<tSearchKey> searchKeys;
    res = GetSearchKeyList(OUT searchKeys, IN listID);
    if (res) return res;

#define debug
#ifdef debug
    // debug
    unsigned int i;
    for(i=0; i<searchKeys.size(); i++) {
        ETG_TRACE_USR1(("search letter=%c, count=%d", searchKeys[i].letter[0],searchKeys[i].count));
    }
#endif

    /*get the corresponding Search keyboard using sortlib character priorities*/
    res = SearchKeyboard::GetInstance().GetSearchKeyboard(OUT searchKeyboardList, IN searchKeys, IN mActiveLanguage );
    if (res) return res;

#ifdef debug
    // debug
    for(i=0; i<searchKeyboardList.size(); i++) {
        ETG_TRACE_USR1(("skb letter=%c, active=%d, start=%d, end=%d", searchKeyboardList[i].letter[0], searchKeyboardList[i].available,searchKeyboardList[i].startIndex,searchKeyboardList[i].endIndex));
    }
#endif
    return MP_NO_ERROR;
}

tResult DBManager::SetMediaContext(const tMediaContext mediaContext)
{
    ENTRY

    mActiveMediaContext = mediaContext;

    ETG_TRACE_USR4(("SetMediaContext:%d",ETG_CENUM(tMediaContext,mActiveMediaContext)));

    return MP_NO_ERROR;
}

tResult DBManager::GetMediaContext(tMediaContext &mediaContext)
{
    ENTRY

    mediaContext = mActiveMediaContext;

    //ETG_TRACE_USR4(("GetMediaContext:%d",ETG_CENUM(tMediaContext,mediaContext)));

    return MP_NO_ERROR;
}

tFileTypeSelection DBManager::GetFTSByActiveMediaContext() const
{
    ENTRY_INTERNAL

    return GetFTSByMC(IN mActiveMediaContext);
}

tFileTypeSelection DBManager::GetFTSByDefaultMediaContext() const
{
    ENTRY_INTERNAL

    tMediaContext mediaContext = (tMediaContext)LocalSPM::GetDataProvider().DBDefaultMediaContext();

    return GetFTSByMC(IN mediaContext);
}

tFileTypeSelection DBManager::GetFTSByMC(const tMediaContext mediaContext) const
{
    ENTRY

    tFileTypeSelection fileTypeSelection;

    if(MC_AUDIO == mediaContext)
    {
        fileTypeSelection = FTS_AUDIO_PLAYLIST;
    }
    else if(MC_VIDEO == mediaContext)
    {
        fileTypeSelection = FTS_VIDEO;
    }
    else //MC_ALL
    {
        fileTypeSelection = FTS_AUDIO_VIDEO_PLAYLIST;
    }

    //ETG_TRACE_USR4(("GetFTSByMC:%d",ETG_CENUM(tFileTypeSelection,fileTypeSelection)));

    return fileTypeSelection;
}


tResult DBManager::DeviceChanged(const vector<tDeviceInfo>deviceInfos)
{
    ENTRY
    tResult res;
    Query query;
    tDeviceInfo changedDeviceInfoElem;
    tDeviceInfo deviceInfo;
    tDeviceID deviceID;
    tBoolean isMountPointSame = false;

    /* loop over all devices in the vector and adjust their status in the database */
    for(tUInt i=0; i<deviceInfos.size(); i++)
    {
        if(deviceInfos[i].deviceType == DTY_NOT_USED_BY_GMP) //BTA/TCU is e.g. DTY_NOT_USED_BY_GMP and thus not used not notified by mediaplayer
        {
            ETG_TRACE_USR4(("[INFO] DeviceChanged:Device is DTY_NOT_USED_BY_GMP: deviceID:%d serialNumber:%s",deviceInfos[i].deviceID,deviceInfos[i].serialNumber));
        }
        else
        {
        InitDeviceInfo(INOUT deviceInfo);
        deviceID = DEVICE_ID_NOT_SET;

        changedDeviceInfoElem = deviceInfos[i];

#if 0   //Only a test
        /* Implementation test: Map DTY_USB to DTY_CDROM */
        if (changedDeviceInfoElem.deviceType == DTY_USB)
        {
            changedDeviceInfoElem.deviceType = DTY_CDROM;
        }
        VARTRACE(changedDeviceInfoElem.deviceType);
#endif

            // PSA BT SmartApp issue.
            // Check if the USB Apple device has a deviceName already available in DB
            //Also Check for Disconnect Reason
        if(IsAppleDevice(changedDeviceInfoElem.deviceType))
        {
            res = query.Select(LTY_DEVICES, MY_MEDIA, "--it", IN 0, IN changedDeviceInfoElem.serialNumber);
            if (res) return res;

            /* get the device info */
            VARTRACE(changedDeviceInfoElem.disconnectReason);
            tDeviceName deviceName;
            tDisconnectReason disconnectReason;
            tBoolean isConnectedOverUSB = false;
            res = query.Get("----T-----------------------------i", OUT &deviceName, IN sizeof(deviceName), OUT &disconnectReason);
            query.End();
            if(!res)
            {
                ETG_TRACE_USR4(("deviceName for connected device from DB: %s",deviceName));
                strncpy_r(changedDeviceInfoElem.deviceName, deviceName, sizeof(changedDeviceInfoElem.deviceName));

                /*NCG3D-65285*/
                VARTRACE(disconnectReason);
                if ( DR_HW_MALFUNC != changedDeviceInfoElem.disconnectReason)
                {
                    changedDeviceInfoElem.disconnectReason = disconnectReason;
                }
            }

            if(LocalSPM::GetDataProvider().iAP2BTDisconnectOniAP2USBConnection())
            {
                tNumberOfDevices numberOfDevices;
                vector<tDeviceInfo> deviceInfoList;

                GetMediaplayerDeviceConnections(numberOfDevices,deviceInfoList,true);

                if (numberOfDevices > 0 )
                {
                    for(tUInt j = 0; j < deviceInfoList.size(); j++)
                    {
                        ETG_TRACE_USR4(("deviceName: %s",deviceInfoList[j].deviceName));
                        ETG_TRACE_USR4(("deviceType : %d", deviceInfoList[j].deviceType));
                        ETG_TRACE_USR4(("device Connection Type: %d ",deviceInfoList[j].connectionType));
                        ETG_TRACE_USR4(("connection State: %d",deviceInfoList[j].connectionState));
                        ETG_TRACE_USR4(("disconnect Reason: %d",deviceInfoList[j].disconnectReason));

                        //IAP2BT device insersion.
                        //Check and avaoid updating DB if the same device is connected via USB for IAP or CP.
                        if(IsAppleDevice(changedDeviceInfoElem.deviceType) && (DCT_BLUETOOTH == changedDeviceInfoElem.connectionType)
                            && (CS_ATTACHED == changedDeviceInfoElem.connectionState) && (DCT_USB == deviceInfoList[j].connectionType)
                            && !strncmp(changedDeviceInfoElem.deviceName,deviceInfoList[j].deviceName, sizeof(changedDeviceInfoElem.deviceName)))
                        {
                            if((CS_DISCONNECTED != deviceInfoList[j].connectionState) || (DIPO_CAP_CARPLAY == deviceInfoList[j].diPOCaps)
                               || (DIPO_CAP_NATIVE_TRANSPORT == deviceInfoList[j].diPOCaps) || (DIPO_CAP_CARLIFE == deviceInfoList[j].diPOCaps))
                            {
                            ETG_TRACE_USR4(("Same device already connected via USB deviceID: %d",deviceInfoList[j].deviceID));
                            isConnectedOverUSB = true;
                            break;
                            }

                        }

                        if( IsAppleDevice(deviceInfoList[j].deviceType)/*(DTY_IPHONE == deviceInfoList[j].deviceType || DTY_IPOD == deviceInfoList[j].deviceType)*/
                                && DCT_BLUETOOTH == deviceInfoList[j].connectionType
                                && !strncmp(changedDeviceInfoElem.deviceName,deviceInfoList[j].deviceName, sizeof(changedDeviceInfoElem.deviceName)))
                        {
                            //TODO(pee1cob):Without checking the incoming DCT_USB apple device ConnectionState,how the DCT_BT apple device connection state altered ?
                            if(CS_CONNECTED == deviceInfoList[j].connectionState)
                            {
                                ETG_TRACE_USR4(("Disconnecting IAP2 BT device: %s as it is connected via USB",deviceInfoList[j].deviceName));
                                res = SetConnectionState(deviceInfoList[j].deviceID,CS_DISCONNECTED,DR_REMOVED_BY_MP);
                                if(res) return res;
                                ETG_TRACE_USR4(("Disconnected IAP2 BT device:%s",deviceInfoList[j].deviceName));
                            }
                            else if(CS_DISCONNECTED == deviceInfoList[j].connectionState && deviceInfoList[j].disconnectReason == DR_REMOVED_BY_MP)
                            {
                                ETG_TRACE_USR4(("Connecting IAP2 BT device back: %s as it is disconnected via USB",deviceInfoList[j].deviceName));
                                res = SetConnectionState(deviceInfoList[j].deviceID,CS_ATTACHED,DR_REMOVED);
                                if(res) return res;
                                ETG_TRACE_USR4(("Connected IAP2 BT device:%s",deviceInfoList[j].deviceName));
                            }
                        }
                    }
                    if(isConnectedOverUSB)      ///to ignore the BT connection when same device is connected over USB.
                    {
                       continue;
                    }
                }
            }
        }

         /*TODO :://Remove this WorkAroud and Provide a ProperFix
         Issue:For iAP1 connected over BT,If IPOD Authentication fails it fallback to DTY_BLUETOOTH from DTY_IPHONE.This fallback to DTY_BLUETOOTH not Known to the ClientHandler_BTSettings.
         Proper Fix:Proper fix should make it known ClientHandler_BTSettings.So ClientHandler_BTSettings notifies with proper deviceType.So Below Workaround No more needed
         WorkAround:Due to this missing logic,ClientHandler_BTSettings always notifies DTY_IPHONE.
         Mediaplayer internally tries to Interpret whether it has arrived for the DTY_BLUETOOTH or DTY_IPHONE.
         case 1: If CS_ATTACH arrives for DTY_IPHONE,if Same device available as DTY_BLUETOOTH - CS_ATTACHED /CS_CONNECTED,This update is IGNORED
         case 2: If CS_DISCONNECT arrives for DTY_IPHONE,if Same device available as DTY_BLUETOOTH - CS_ATTACHED /CS_CONNECTED,it is not known whether to disconnect this device.
                Due to this,DR_SPP_DISCONNECTED introduced.
        */

        if(IsAppleDevice(changedDeviceInfoElem.deviceType) && DCT_BLUETOOTH == changedDeviceInfoElem.connectionType &&
               ( (CS_DISCONNECTED != changedDeviceInfoElem.connectionState)|| (DR_SPP_DISCONNECTED == changedDeviceInfoElem.disconnectReason)))
        {
            res = query.Select(LTY_DEVICES, MY_MEDIA, "--it", IN 0, IN changedDeviceInfoElem.serialNumber);
            if (res) return res;

            tDeviceType deviceType;
            tConnectionState connectionState;
            res = query.Get("--------------i--i", OUT &deviceType,OUT &connectionState);
            query.End();
            if(!res)
            {
                if(DTY_BLUETOOTH == deviceType && CS_DISCONNECTED != connectionState)
                {
                    ETG_TRACE_USR4(("The device:%s is fallback to Blutooth.So ignoring the Device update",changedDeviceInfoElem.deviceName));
                    return MP_NO_ERROR;
                }

            }

        }

        /* first check if device is already in the database */
        res = query.Select(LTY_DEVICES, MY_MEDIA, "--it", IN 0, IN changedDeviceInfoElem.serialNumber);
        if (res) return res;

        /* get the device info */
        res = query.Get("i", OUT &deviceID);
        query.End();

        /*if((LocalSPM::GetDataProvider().MultiPartitionSupported())
                && (0 != strncmp(deviceInfos[i].accessoryName2,"",strlen_r(deviceInfos[i].accessoryName2)) )
                && ((deviceInfos[i].deviceType==DTY_USB) ||(deviceInfos[i].deviceType==DTY_SD )))
            {
                If alread added multi partition device change the connection status except for valid partition as CS_UNDEFINED)
                tNumberOfPartitions nPartitionCount = 0;
                nPartitionCount = count_if(deviceInfos.begin(),
                                           deviceInfos.end(),
                                           tFindDeviceBySerialNumber(deviceInfos[i].serialNumber,deviceInfos[i].accessoryName2,false,true));
                if(nPartitionCount > 1)
                {
                    if(mValidPartitionInfo.find(deviceInfos[i].accessoryName2)->second.c_str())  //NULLCHECK PSARCC30-2584
                    {
                        if(strncmp(deviceInfos[i].serialNumber,mValidPartitionInfo.find(deviceInfos[i].accessoryName2)->second.c_str(),strlen_r(deviceInfos[i].serialNumber))!=0)
                        {
                            continue;
                        }
                        else
                        {
                            ETG_TRACE_USR2(("Valid Multipartition entry : Serial number:%s", deviceInfos[i].serialNumber));
                        }
                    }

                }
            }*/

        if (res == MP_ERR_DB_END_OF_LIST) { // device not known, seems to be a new one:

            if(changedDeviceInfoElem.connectionState == CS_ATTACHED) {

                //Update BTLimitationmode in iPodControl if device supports iAP2BT & CarPlay wifi is enabled.
                if(IsAppleDevice(changedDeviceInfoElem.deviceType) && (DCT_BLUETOOTH == changedDeviceInfoElem.connectionType) && LocalSPM::GetDataProvider().iPodControlCarPlayWifiEnabled() )
                {
                    ETG_TRACE_USR2(("DeviceChanged: device not in DB, Setting Limitation mode in iPodControl before connection"));
                    LocalSPM::GetIPODControl().SendSetBTLimitationMode(changedDeviceInfoElem.appleDeviceMACAddress,changedDeviceInfoElem.btLimitationActionState);
                }

                strncpy_r(deviceInfo.UUID, changedDeviceInfoElem.UUID, sizeof(deviceInfo.UUID));
                strncpy_r(deviceInfo.serialNumber, changedDeviceInfoElem.serialNumber, sizeof(deviceInfo.serialNumber));
                strncpy_r(deviceInfo.deviceVersion, changedDeviceInfoElem.deviceVersion, sizeof(deviceInfo.deviceVersion));
                strncpy_r(deviceInfo.deviceName, changedDeviceInfoElem.deviceName, sizeof(deviceInfo.deviceName));
                strncpy_r(deviceInfo.mountPoint, changedDeviceInfoElem.mountPoint, sizeof(deviceInfo.mountPoint));
                strncpy_r(deviceInfo.accessoryName, changedDeviceInfoElem.accessoryName, sizeof(deviceInfo.accessoryName));
                //CARPLAY_WIFI
                //Condition is changed as Macaddress is required for all Bluetooth devices as a unique identifier across all components (especially mediaplayer and bluetooth) .
                if((DCT_BLUETOOTH == changedDeviceInfoElem.connectionType) || (DCT_WIFI == changedDeviceInfoElem.connectionType && IsAppleDevice(changedDeviceInfoElem.deviceType)))
                {
                    strncpy_r(deviceInfo.appleDeviceMACAddress, changedDeviceInfoElem.appleDeviceMACAddress, sizeof(deviceInfo.appleDeviceMACAddress));
                }
                deviceInfo.deviceType = changedDeviceInfoElem.deviceType;
                deviceInfo.connectionState = changedDeviceInfoElem.connectionState;
                deviceInfo.fileSystemType  = changedDeviceInfoElem.fileSystemType;
                deviceInfo.partitionNumber = changedDeviceInfoElem.partitionNumber;
                deviceInfo.totalSize       = changedDeviceInfoElem.totalSize;
                deviceInfo.freeSize        = changedDeviceInfoElem.freeSize;
                deviceInfo.productID       = changedDeviceInfoElem.productID;
                deviceInfo.connectionType  = changedDeviceInfoElem.connectionType;
                deviceInfo.deviceState     = DS_INITIALIZING;

                res = AddDevice(INOUT deviceInfo);
            } else {
                ETG_TRACE_ERR(("DeviceChanged: Add device failed - Invalid connection state: connectionState=%d, deviceName=%s",
                        changedDeviceInfoElem.connectionState, changedDeviceInfoElem.deviceName));
            }
        } else { // device is in DB, change the properties

            /* get the current state of the device */
            res = GetDeviceInfo(OUT deviceInfo, IN deviceID);
            if (res) return res;

            //CMG3G-14540 Update BTLimitationmode and Trigger Reconnection.
            if(IsAppleDevice(deviceInfo.deviceType) && (DCT_BLUETOOTH == deviceInfo.connectionType)
                    && LocalSPM::GetDataProvider().iPodControlCarPlayWifiEnabled() && deviceInfo.connectionState == CS_ATTACHED)
            {
                ETG_TRACE_USR2(("DeviceChanged: device in DB, Setting Limitation mode in iPodControl & triggering reconnection"));
                LocalSPM::GetIPODControl().SendSetBTLimitationMode(changedDeviceInfoElem.appleDeviceMACAddress,changedDeviceInfoElem.btLimitationActionState);
            }

            // update the status
            Query update;
            res = update.Update(LTY_DEVICES_UPDATE);
            if (res) return res;

            /* Avoid updating Deviceinfo IAPBT device that is in CS_ATTACHED state. */
            if(LocalSPM::GetDataProvider().IAPBTConnectionOnlyForSmartApp() && (DCT_BLUETOOTH == deviceInfo.connectionType) && (DTY_IPHONE == deviceInfo.deviceType)
                    &&  (deviceInfo.connectionState == CS_ATTACHED) && (changedDeviceInfoElem.connectionState == CS_ATTACHED))
            {
                continue;
            }

            if (((deviceInfo.connectionState == CS_CONNECTED || deviceInfo.connectionState == CS_ON_HOLD)&&
                 changedDeviceInfoElem.connectionState == CS_ATTACHED))  //No  Error Marked by GMP when read occured: May be Marked by Now
            {
                /*For mass storage devices if the total size is changed , update in db. For other devices this will not happen*/
                if(IsMassStorageDevice(deviceInfo.deviceType) && (changedDeviceInfoElem.totalSize != 0) && (deviceInfo.freeSize != changedDeviceInfoElem.freeSize))
                {
                    res = update.Change("------------------------------------------------------------iiii--ii",
                    IN 0, IN changedDeviceInfoElem.totalSize,
                    IN 0, IN changedDeviceInfoElem.freeSize,
                    IN 0, IN deviceID);
                }
                //case a:If mountpoint Changed,we will Disconnect the Device.So Error Marking will erase.Error marking will NOT occur over this value
                if(isMountPointSame = (0 == strncmp(deviceInfo.mountPoint,changedDeviceInfoElem.mountPoint,sizeof(changedDeviceInfoElem.mountPoint))) /*&& (deviceInfo.deviceState != DS_COMMUNICATION_ERROR) */) //Mediaplyer will list this dev as Playable Source.
                {
                    //Skip will not happen If AVRCPLowerProfile supported and DTY_BLUETOOTH has come as CS_ATTACHED which is already in CS_CONNECTED in database
                    if(!LocalSPM::GetDataProvider().AVRCPLowerProfileSupported() || (DTY_BLUETOOTH != deviceInfo.deviceType))
                    {
                        ETG_TRACE_USR4(("DeviceChanged:IGNORE CS_ATTACH as it has the same mountpoint for deviceID:%d",deviceID));
                        continue;
                    }
                }
                else //Mountpoint of the Device has changed!!!!!. To Notify Internal GMP about MountPoint change,it is needed to Disconnect the device & connect again
                {
                    ETG_TRACE_USR4(("DeviceChanged:OLD mountPoint:%s",deviceInfo.mountPoint));
                    ETG_TRACE_USR4(("DeviceChanged:NEW mountPoint:%s",changedDeviceInfoElem.mountPoint));

                    res = update.Change("------------------ii--iiititiiiiii----------ii--------------------ii",
                    IN 0, IN CS_DISCONNECTED,
                    IN 0, IN CS_DISCONNECTED,
                    IN 0, IN deviceInfo.accessoryName,
                    IN 0, IN deviceInfo.mountPoint,
                    IN 0, IN DS_NONE,
                    IN 0, IN (deviceInfo.numberOfAudioFiles>0 ? deviceInfo.numberOfAudioFiles : NUMBER_OF_FILES_NONE),
                    IN 0, IN (deviceInfo.numberOfVideoFiles>0 ? deviceInfo.numberOfVideoFiles : NUMBER_OF_FILES_NONE),
                    IN 0, IN DR_REMOVED,
                    IN 0, IN deviceID);
                }
            }

            if ((deviceInfo.connectionState == changedDeviceInfoElem.connectionState) && (changedDeviceInfoElem.disconnectReason == deviceInfo.disconnectReason))
            {
                ETG_TRACE_USR4(("DeviceChanged:IGNORE DevMgr update for this device as it is already in same State internally"));
                continue;
            }

            /* change connection state */
            ETG_TRACE_USR2(("DeviceChanged: deviceID=%d, old connectionState=%d, new connectionState=%d",
                    deviceID, deviceInfo.connectionState, changedDeviceInfoElem.connectionState));

            if(DTY_UNKNOWN != changedDeviceInfoElem.deviceType) {
                //update deviceType, see usecase in BT IAP handling
                SetDeviceType(IN deviceID, IN changedDeviceInfoElem.deviceType);
            }

            if((LocalSPM::GetDataProvider().AVRCPLowerProfileSupported())
              && (DTY_BLUETOOTH == deviceInfo.deviceType)
              && (string(deviceInfo.deviceVersion) == "1.0")
              && (string(changedDeviceInfoElem.deviceVersion) == "1.4" || string(changedDeviceInfoElem.deviceVersion) == "1.3"))
            {
                res = update.Change("it----------------------------------------------------------------ii", IN 0, IN changedDeviceInfoElem.deviceVersion, IN 0, IN deviceID);

                return MP_NO_ERROR;
            }

            if(changedDeviceInfoElem.connectionState == CS_DISCONNECTED)  // set also former connection state to CS_DISCONNECTED
            {
                res = update.Change("------------------ii--ii----iiiiii----------ii------ii------------ii",
                        IN 0, IN changedDeviceInfoElem.connectionState,
                        IN 0, IN CS_DISCONNECTED,
                        IN 0, IN DS_NONE,
                        IN 0, IN (deviceInfo.numberOfAudioFiles>0 ? deviceInfo.numberOfAudioFiles : NUMBER_OF_FILES_NONE),
                        IN 0, IN (deviceInfo.numberOfVideoFiles>0 ? deviceInfo.numberOfVideoFiles : NUMBER_OF_FILES_NONE),
                        IN 0, IN changedDeviceInfoElem.disconnectReason,
                        IN 0, IN tNowPlayingListAvailable_init,
                        IN 0, IN deviceID);
                //if (res) return res; go on in loop even in case of IC violation
            }
            else
            {
                /* Set the former connection statte as CS_CONNECTED if the previous disconnection reason is DR_HW_MALFUNC */
                if(CS_DISCONNECTED == deviceInfo.connectionState && CS_ATTACHED == changedDeviceInfoElem.connectionState && DR_HW_MALFUNC == deviceInfo.disconnectReason)
                {
                    res = update.Change("itit--------------ii--iiititiiiiii--------------ii----------------ii",
                            IN 0, IN changedDeviceInfoElem.deviceVersion,
                            IN 0, IN changedDeviceInfoElem.deviceName[0] != 0 ? changedDeviceInfoElem.deviceName : deviceInfo.deviceName,                                 //GMMY16-17003
                            IN 0, IN changedDeviceInfoElem.connectionState,
                            IN 0, IN CS_CONNECTED,
                            IN 0, IN changedDeviceInfoElem.accessoryName,
                            IN 0, IN changedDeviceInfoElem.mountPoint,
                            IN 0, IN changedDeviceInfoElem.connectionState == CS_ATTACHED ? DS_INITIALIZING : deviceInfo.deviceState,
                            IN 0, IN (deviceInfo.numberOfAudioFiles>0 ? deviceInfo.numberOfAudioFiles : NUMBER_OF_FILES_NONE),
                            IN 0, IN (deviceInfo.numberOfVideoFiles>0 ? deviceInfo.numberOfVideoFiles : NUMBER_OF_FILES_NONE),
                            IN 0, IN changedDeviceInfoElem.disconnectReason,
                            IN 0, IN deviceID);
                }
                else
                {
                    /* BTMacAddress should not be cleared from DB when connection type is USB for Carplay */
                    VARTRACE(changedDeviceInfoElem.connectionType);
                    if(IsAppleDevice(changedDeviceInfoElem.deviceType) && changedDeviceInfoElem.connectionType == DCT_USB)
                    {
                        res = update.Change("itit--------------ii----ititiiiiii----------ii--------------------ii",
                            IN 0, IN changedDeviceInfoElem.deviceVersion,
                            IN 0, IN changedDeviceInfoElem.deviceName[0] != 0 ? changedDeviceInfoElem.deviceName : deviceInfo.deviceName,                                 //GMMY16-17003
                            IN 0, IN changedDeviceInfoElem.connectionState,
                            IN 0, IN changedDeviceInfoElem.accessoryName,
                            IN 0, IN changedDeviceInfoElem.mountPoint,
                            IN 0, IN changedDeviceInfoElem.connectionState == CS_ATTACHED ? DS_INITIALIZING : deviceInfo.deviceState,
                            IN 0, IN (deviceInfo.numberOfAudioFiles>0 ? deviceInfo.numberOfAudioFiles : NUMBER_OF_FILES_NONE),
                            IN 0, IN (deviceInfo.numberOfVideoFiles>0 ? deviceInfo.numberOfVideoFiles : NUMBER_OF_FILES_NONE),
                            IN 0, IN changedDeviceInfoElem.disconnectReason,
//                          IN 0, IN changedDeviceInfoElem.appleDeviceMACAddress, // not cleared for apple USB device.(NCG3D-128928)
                            IN 0, IN deviceID);
                    }
                    else
                    {
                        res = update.Change("itit--------------ii----ititiiiiii----------ii----------it--------ii",
                            IN 0, IN changedDeviceInfoElem.deviceVersion,
                            IN 0, IN changedDeviceInfoElem.deviceName[0] != 0 ? changedDeviceInfoElem.deviceName : deviceInfo.deviceName,                                 //GMMY16-17003
                            IN 0, IN changedDeviceInfoElem.connectionState,
                            IN 0, IN changedDeviceInfoElem.accessoryName,
                            IN 0, IN changedDeviceInfoElem.mountPoint,
                            IN 0, IN changedDeviceInfoElem.connectionState == CS_ATTACHED ? DS_INITIALIZING : deviceInfo.deviceState,
                            IN 0, IN (deviceInfo.numberOfAudioFiles>0 ? deviceInfo.numberOfAudioFiles : NUMBER_OF_FILES_NONE),
                            IN 0, IN (deviceInfo.numberOfVideoFiles>0 ? deviceInfo.numberOfVideoFiles : NUMBER_OF_FILES_NONE),
                            IN 0, IN changedDeviceInfoElem.disconnectReason,
                            IN 0, IN changedDeviceInfoElem.appleDeviceMACAddress,
                            IN 0, IN deviceID);
                    }
                }
                //if (res) return res; go on in loop even in case of IC violation
            }

#if 0 //Done in ClearLastMode now
            /* if device was disconnected: */
            if (changedDeviceInfoElem.connectionState == CS_DISCONNECTED) {

                /* reset shuffle mode */
                res = SetPlaybackMode(IN deviceID, IN PBM_NORMAL);
                if (res) return res;
            }
#endif
        }

        /*
         * Update Device Portnumber CAFI06-122
         */
        if((LocalSPM::GetDataProvider().AddPortNumberToDeviceName()) && (DCT_USB == changedDeviceInfoElem.connectionType))
        {
            ETG_TRACE_USR3(("DBManager::DeviceChanged changedDeviceInfoElem.portNumber:%s",changedDeviceInfoElem.portNumber));
            UpdatePortNumberInfo(IN deviceInfo.deviceID,IN changedDeviceInfoElem.portNumber,IN changedDeviceInfoElem.connectionState);
        }

        /*
         * Copy SysPath to Map CMG3G-12660
         */
        ETG_TRACE_USR3(("DBManager::DeviceChanged changedDeviceInfoElem.deviceSyspath:%s",changedDeviceInfoElem.deviceSyspath));
        if((LocalSPM::GetDataProvider().AddDeviceSysPath()) && (DCT_USB == changedDeviceInfoElem.connectionType))
        {
            UpdateSysPathInfoToMap(IN deviceInfo.deviceID,IN changedDeviceInfoElem.deviceSyspath,IN changedDeviceInfoElem.connectionState);
        }
    }
    }


    return MP_NO_ERROR;
}

tResult DBManager::UpdatePortNumberInfo(tDeviceID deviceID, tUSBPortNumber portNumber, const tConnectionState connectionstate)
{
    ENTRY
    VARTRACE(deviceID);
    VARTRACE(portNumber);
    VARTRACE(connectionstate);
    tUSBPortNumber retPortNumber;
    strncpy_r(retPortNumber,PORTNUM_DEFAULT,sizeof(retPortNumber));

    if(connectionstate == CS_DISCONNECTED) // Remove Map entry
    {
        map<tDeviceID,string>::iterator iterPortNumberInfo = mPortNumberInfo.find(deviceID);
        if(iterPortNumberInfo != mPortNumberInfo.end())
        {
            mPortNumberInfo.erase(iterPortNumberInfo);
        }
    }
    else // Add/Map portnumber to device ID if device is connected
    {
        // Find if deviceID is already mapped with portnumber
        map<tDeviceID,string>::iterator iterPortNumberInfo = mPortNumberInfo.find(deviceID);
        if(iterPortNumberInfo != mPortNumberInfo.end())
        {
            if(!strncmp(portNumber,iterPortNumberInfo->second.c_str(),sizeof(tUSBPortNumber)))
            {
                ETG_TRACE_USR3(("DBManager::UpdatePortNumberInfo-- Portnumber is already mapped with the deviceID "));
            }
            else
            {
                iterPortNumberInfo->second = portNumber;// update existing device ID with new portnumber
            }
        }
        else
        {
            mPortNumberInfo[deviceID]=portNumber;// Add a new map of device ID with portnumber
        }
    }
    return MP_NO_ERROR;
}

tResult DBManager::GetDevicePortNumberInfo(tDeviceID deviceID, tUSBPortNumber &retPortNumber)
{
    ENTRY
    VARTRACE(deviceID);

    strncpy_r(retPortNumber,PORTNUM_DEFAULT,sizeof(retPortNumber));

    map<tDeviceID,string>::iterator iterPortNumberInfo = mPortNumberInfo.find(deviceID);
    if(iterPortNumberInfo != mPortNumberInfo.end())
    {
        ETG_TRACE_USR3(("DBManager::GetDevicePortNumberInfo DeviceID:%u ,PortNumber:%s",deviceID,iterPortNumberInfo->second.c_str()));
        strncpy_r(retPortNumber, iterPortNumberInfo->second.c_str(), sizeof(retPortNumber));
    }
    ETG_TRACE_USR3(("DBManager::GetDevicePortNumberInfo DeviceID:%u : retPortNumber:%s",deviceID,retPortNumber));
    return MP_NO_ERROR;
}


tResult DBManager::UpdateSysPathInfoToMap(tDeviceID deviceID, tPath deviceSyspath, const tConnectionState connectionstate)
{
    ENTRY
    VARTRACE(deviceID);
    VARTRACE(deviceSyspath);
    VARTRACE(connectionstate);

    if(connectionstate == CS_DISCONNECTED) // Remove Map entry
    {
        map<tDeviceID,string>::iterator iterSysPathInfo = mDeviceSysPathInfo.find(deviceID);
        if(iterSysPathInfo != mDeviceSysPathInfo.end())
        {
            mDeviceSysPathInfo.erase(iterSysPathInfo);
        }
    }
    else // Add/Map deviceSyspath to device ID if device is connected
    {
        // Find if deviceID is already mapped with deviceSyspath
        map<tDeviceID,string>::iterator iterSysPathInfo = mDeviceSysPathInfo.find(deviceID);
        if(iterSysPathInfo != mDeviceSysPathInfo.end())
        {
            if(!strncmp(deviceSyspath,iterSysPathInfo->second.c_str(),sizeof(tPath)))
            {
                ETG_TRACE_USR3(("DBManager::UpdateSysPathInfoToMap-- deviceSyspath is already mapped with the deviceID "));
            }
            else
            {
                iterSysPathInfo->second = deviceSyspath;// update existing device ID with new deviceSyspath
            }
        }
        else
        {
            mDeviceSysPathInfo[deviceID]=deviceSyspath;// Add a new map of device ID with deviceSyspath
        }
    }
    return MP_NO_ERROR;
}

tResult DBManager::GetSysPathInfoFromMap(tDeviceID deviceID, tPath &retDeviceSyspath)
{
    ENTRY
    VARTRACE(deviceID);

    retDeviceSyspath[0] = '\0';

    map<tDeviceID,string>::iterator iterSysPathInfo = mDeviceSysPathInfo.find(deviceID);
    if(iterSysPathInfo != mDeviceSysPathInfo.end())
    {
        ETG_TRACE_USR3(("DBManager::GetSysPathInfoFromMap DeviceID:%u ,DeviceSyspath:%s",deviceID,iterSysPathInfo->second.c_str()));
        strncpy_r(retDeviceSyspath, iterSysPathInfo->second.c_str(), sizeof(retDeviceSyspath));
    }
    ETG_TRACE_USR3(("DBManager::GetSysPathInfoFromMap DeviceID:%u : retDeviceSyspath:%s",deviceID,retDeviceSyspath));
    return MP_NO_ERROR;
}

tResult DBManager::StoreLastPlayed(const tListID listID, const tObjectID objectID, const tPosition position, const tPlaytime playtime)
{
    ENTRY
    tResult res;
    Query query;
    DBList *dbList = NULL;
    tMediaContext mediaContext;

    /* get list pointer */
    //dbList = LookupDBList(IN listID);
    GET_DBLIST_POINTER(dbList, listID);
    if (!dbList) return MP_ERR_DB_LIST_NOT_FOUND;

    /* get the current row in the list */
    tRowNumber rowNumber = (tRowNumber)position;
    //res = GetCurrentRow(OUT rowNumber, IN listID);
    //if (res) return res;

    /* reset all active flags */
    res = query.Update(LTY_LASTMODE_UPDATE);
    if (res) return res;

    /* set all entries to be not active */
    tActive active = false;
    res = query.Change("ii", IN 0, IN active);
    if (res) return res;

    GetMediaContext( OUT mediaContext);
    /* add it to the last mode table */
    res = query.Insert(LTY_LASTMODE_INSERT);
    if (res) return res;

        res = query.Put("iiiiiiiiiiitii",
            IN dbList->mDeviceID,
            IN dbList->mListType,
            IN dbList->mPlayContinuation,
            IN dbList->mFilterTag1.tag,
            IN dbList->mFilterTag2.tag,
            IN dbList->mFilterTag3.tag,
            IN dbList->mFilterTag4.tag,
            IN playtime,
            IN objectID,
            IN rowNumber,
            IN true,                //Active
            IN dbList->mPath,
              IN dbList->mStreaming,
              IN mediaContext);
    if (res) return res;

    ETG_TRACE_USR3(("DBManager::StoreLastPlayed deviceID:%u, objectID:%u, rowNumber:%u, playtime:%u, listType:%u, streaming: %u, path:%s",
            dbList->mDeviceID, objectID, rowNumber, playtime, dbList->mListType, dbList->mStreaming, dbList->mPath));

    return MP_NO_ERROR;
}

tResult DBManager::RemoveLastPlayed(const tDeviceID deviceID)
{
    ENTRY
    tResult res;
    Query query;

    /* delete the last mode entry from DB */
    res = query.Delete(IN LTY_LASTMODE_DELETE);
    if (res) return res;

    res = query.Remove("ii", IN 0, IN deviceID);
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::GetLastPlayed(tListID &listID, tObjectID &objectID, tPlaytime &playtime, const tDeviceID deviceID)
{
    ENTRY
    tResult res;
    Query query;
    tListType listType;
    tPlayContinuation playContinuation;
    tFilterTag1 filterTag1;
    tFilterTag2 filterTag2;
    tFilterTag3 filterTag3;
    tFilterTag4 filterTag4;
    tRowNumber rowNumber;
    tPath path;
    listID = LIST_ID_NONE;
    tDeviceID realDeviceID = deviceID;
    tStreaming streaming;
    tDeviceID readDeviceID;
    tMediaContext mediaContext = MC_ALL; //TODO: Read context from LastMode table

    GetMediaContext( OUT mediaContext);
    /* find lastmode entry for given device */
    res =   query.Select(LTY_LASTMODE, IN deviceID, "--ii",IN 0, IN mediaContext);
    if (res) return res;

    res = query.Get("iiiiiiiiii-Ti",
            OUT &readDeviceID,
            OUT &listType,
            OUT &playContinuation,
            OUT &filterTag1.tag,
            OUT &filterTag2.tag,
            OUT &filterTag3.tag,
            OUT &filterTag4.tag,
            OUT &playtime,
            OUT &objectID,
            OUT &rowNumber,
            OUT path, IN sizeof(path),
            OUT &streaming);
    query.End();
    if (res) return res;

    /* check if correct device was read out */
    if (readDeviceID != deviceID) {
        return MP_ERR_DB_END_OF_LIST;
    }

    ETG_TRACE_USR3(("DBManager::GetLastPlayed deviceID:%u, mediaContext:%u, objectID:%u, rowNumber:%u, playtime:%u, listType:%u, streaming: %u, path:%s",
            deviceID, mediaContext, objectID, rowNumber, playtime, listType, streaming, path));

    tFileTypeSelection fileTypeSelection = GetFTSByMC(IN mediaContext);

    /* do a special for file lists */
    switch(listType)
    {
    case LTY_FILELIST:
    case LTY_FILELIST_UNSORTED:
    case LTY_BROWSELIST:
    case LTY_MTP_FILELIST:
    case LTY_CD:
        res = CreateMediaPlayerFileList(INOUT listID, INOUT listType, IN path, IN deviceID, IN streaming, IN fileTypeSelection);
        break;
    case LTY_FILELIST_MEDIAOBJECTS:
    case LTY_BROWSELIST_MEDIAOBJECTS:
    case LTY_FILELIST_MEDIAOBJECTS_WITH_SUBFOLDERS:
    case LTY_BROWSELIST_MEDIAOBJECTS_WITH_SUBFOLDERS:
        res = CreateFileListOfMediaObjects(INOUT listID, INOUT listType, IN path, IN deviceID, IN streaming, IN fileTypeSelection);
        break;
    case LTY_FILELIST_PLAYLIST:
    case LTY_PLAYLIST_SONG:
    {
        /* take real deviceID in case of a MyMedia playlist */
        if(MY_MEDIA == deviceID)
        {
            res = GetObjectDeviceID(OUT realDeviceID, IN filterTag1.tag, CTY_PLAYLIST);
            if (res) return res;
        }

        if(LTY_FILELIST_PLAYLIST == listType)
        {
            res = CreateMediaPlayerFileList(INOUT listID, INOUT listType, IN path, IN realDeviceID, IN streaming, IN fileTypeSelection);
        }
        else
        {
            res = CreateList(OUT listID,
                    IN realDeviceID,
                    IN listType,
                    IN playContinuation,
                    IN filterTag1,
                    IN filterTag2,
                    IN filterTag3,
                    IN filterTag4,
                    IN streaming,
                    IN true,  //checkVTIPOD
                    IN true); //lastPlayedList
        }
        if (res) return res;

        /* store playlist attributes in DB list too */
        res = SetPlaylistAttributes(IN listID, IN deviceID, IN filterTag1);
        break;
    }
    default:
        res = CreateList(OUT listID,
                IN deviceID,
                IN listType,
                IN playContinuation,
                IN filterTag1,
                IN filterTag2,
                IN filterTag3,
                IN filterTag4,
                IN streaming,
                IN true,  //checkVTIPOD
                IN true,  //lastPlayedList
                IN path); //lastPlayedPath
        break;
    }
    if (res) return res;

    /* position to the last active object */
    if (objectID != OBJECT_ID_NONE) {

        tMediaObject mediaObject;
        InitMediaObject(OUT mediaObject);

        if (realDeviceID != MY_MEDIA) {

            /* set the row and check if the desired object is met */
            res = SetCurrentRow(IN listID, IN rowNumber);
            if (res && res != MP_ERR_DB_END_OF_LIST) return res;

            if (res == MP_NO_ERROR) {
                /* get the current object */
                res = GetCurrentMediaObject(OUT mediaObject, IN listID);
                if (res && res != MP_ERR_DB_END_OF_LIST) return res;
            }

            /* is the desired media object NOT met? */
            if (mediaObject.objectID != objectID || res == MP_ERR_DB_END_OF_LIST) {

                /* try to set the position to the given object (slow) */
                res = SetPosition(IN objectID, IN listID);
                if (res) return res;
            }
        } else {

            /* For MyMedia we cannot set the object via the row because the content changes by connecting and disconnecting devices */
            /* try to set the position to the given object (slow) */
            res = SetPosition(IN objectID, IN listID);

            /* if it was not possible to set the objectID, create this single object in the MyMedia table */
            if (res) {

                ETG_TRACE_USR3(("DBManager::GetLastPlayed: Insert missing object in MyMedia DB explicitly"));

                /* add this object to MyMedia DB */
                res = query.Insert(LTY_MEDIAOBJECTS_MYME_COPY);
                if (res) return res;

                res = query.Put("i", IN objectID);
                if (res) return res;

                /* next try to set the position to the given object */
                res = SetPosition(IN objectID, IN listID);
                if (res) return res;
            }

            /* get object with objectID */
            res = GetMediaObject(OUT mediaObject, INOUT objectID);
            if (res) return res;

            /* get the device info of the current object */
            tDeviceInfo deviceInfo;
            res = GetDeviceInfo(OUT deviceInfo, IN mediaObject.deviceID);
            if (res) return res;

            /* if device is not connected remove object from the MyMedia table again and return an error */
            if (deviceInfo.connectionState != CS_CONNECTED) {

                res = query.Delete(LTY_MEDIAOBJECTS_MYME_DELETE);
                if (res) return res;

                res = query.Remove("ii", IN 0, IN objectID);
                if (res) return res;

                return MP_ERR_DB_NOT_CONNECTED;
            }
        }

        ETG_TRACE_USR3(("DBManager::GetLastPlayed after SetPosition, objectID:%u, title:%s", mediaObject.objectID, mediaObject.title));
    }

    return MP_NO_ERROR;
}

tResult DBManager::StoreLastPlaytime(const tObjectID objectID, const tPlaytime playtime)
{
    ENTRY
    ETG_TRACE_USR3(("DBManager::StoreLastPlaytime objectID:%u, playtime:%u", objectID, playtime));

    if (0 == playtime) return RemoveLastPlaytime(IN objectID);

    tResult res;
    Query query;

    /* add or replace it to the last playtime table */
    res = query.Insert(LTY_LAST_PLAYTIME_INSERT);
    if (res) return res;

    res = query.Put("ii",
            IN objectID,
            IN playtime);
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::RemoveLastPlaytime(const tObjectID objectID)
{
    ENTRY
    tResult res;
    Query query;

    /* delete the last playtime entry from DB */
    res = query.Delete(IN LTY_LAST_PLAYTIME_DELETE);
    if (res) return res;

    res = query.Remove("ii", IN 0, IN objectID);
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::GetLastPlaytime(tPlaytime &playtime, const tObjectID objectID)
{
    ENTRY
    tResult res;
    Query query;
    tObjectID readObjectID;

    /* find last playtime entry for given object */
    res = query.Select(LTY_LAST_PLAYTIME, IN objectID, "");
    if (res) return res;

    res = query.Get("ii",
            OUT &readObjectID,
            OUT &playtime);
    query.End();
    if (res) return res;

    /* check if correct device was read out */
    if (readObjectID != objectID) {
        return MP_ERR_DB_END_OF_LIST;
    }

    ETG_TRACE_USR3(("DBManager::GetLastPlayed objectID:%u, playtime:%u", objectID, playtime));
    return MP_NO_ERROR;
}

tResult DBManager::ClearLastMode(const tDeviceID deviceID)
{
    ENTRY
    tResult res;
    Query query;

    /* reset shuffle mode */
    if (LocalSPM::GetDataProvider().ClearPlaybackMode())
    {
        if(LocalSPM::GetDataProvider().SeparateMediaContent())/* reset both audio and video shuffle modes */
            res = ResetAllPlaybackModes(IN deviceID);
        else
            res = SetPlaybackMode(IN deviceID, IN PBM_NORMAL);/*reset only audio shuffle mode*/
        if (res) return res;
    }

    /* reset repeat mode */
    if ((LocalSPM::GetDataProvider().ClearRepeatMode())
        ||
        (!LocalSPM::GetDataProvider().RepeatModeAdjustable()))
    {
        if(LocalSPM::GetDataProvider().SeparateMediaContent())
            res = ResetAllRepeatModes(IN deviceID);/*reset repeat modes for both audio and video*/
        else
            res = SetRepeatMode(IN deviceID, IN (tRepeatMode)LocalSPM::GetDataProvider().DBDefaultRepeatMode());/*reset repeat modes for audio only*/
        if (res) return res;
    }

    /* clear not playable flag */
    tFileErrorHandling fileErrorHandling = (tFileErrorHandling)LocalSPM::GetDataProvider().FileErrorHandling();
    if (FEH_MARK_OBJECT == fileErrorHandling)
    {
        res = ClearNotPlayableFlag(IN deviceID);
        if (res) return res;
    }

    /* delete last mode (played list and object) for device */
    if (LocalSPM::GetDataProvider().ClearLastMode())
    {
        res = RemoveLastPlayed(IN deviceID);
        if (res) return res;
    }

    return MP_NO_ERROR;
}

tResult DBManager::ClearNotPlayableFlag(const tDeviceID deviceID)
{
    ENTRY
    tResult res;
    Query query;

    /* update the MediaObjects table */
    res = query.Update(LTY_MEDIAOBJECTS_UPDATE);
    if (res) return res;

    /* find match for deviceID and reset notPlayable flag */
    tNotPlayable notPlayable = FNP_PLAYABLE;
    res = query.Change("----------------------ii------------ii------ii",
            IN 0, IN notPlayable,
            IN 0, IN deviceID,      //for WHERE clause
            IN 0, IN notPlayable);  //for WHERE clause
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::StoreBTMACAddress(const tBTMACAddress address)
{
    ENTRY
    tResult res;
    Query query;

    /* add it to the system table */
    res = query.Insert(IN LTY_SYSTEM_INSERT);
    if (res) return res;

    res = query.Put("t", IN address);
    if (res) return res;

    /* store it also in DataProvider */
    LocalSPM::GetDataProvider().iPodControlBTMACAddress = string(address);

    return MP_NO_ERROR;
}

tResult DBManager::GetBTMACAddress(tBTMACAddress &address)
{
    ENTRY
    tResult res;
    Query query;

    res = query.Select(LTY_SYSTEM, 1 /* dummy */, "");
    if (res) return res;

    /* get system infos */
    res = query.Get("T", OUT address, IN sizeof(address));
    query.End();
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::StoreScanContext(const tScanContext scanContext)
{
    ENTRY
    tResult res;
    Query query;

    /* add it to the scan context table */
    res = query.Insert(IN LTY_SCAN_CONTEXT_INSERT);
    if (res) return res;

    for( tUInt i = 0; i < scanContext.dirContext.size(); i++ )
    {
        res = query.Put("iiti",
                IN scanContext.deviceID,
                IN scanContext.deep,
                IN scanContext.dirContext[i].path,
                IN scanContext.dirContext[i].row);
        if (res) return res;
    }

    return MP_NO_ERROR;
}

tResult DBManager::RemoveScanContext(const tDeviceID deviceID)
{
    ENTRY
    tResult res;
    Query query;

    /* delete the scan context from DB */
    res = query.Delete(IN LTY_SCAN_CONTEXT_DELETE);
    if (res) return res;

    res = query.Remove("ii", IN 0, IN deviceID);
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::GetScanContext(tScanContext &scanContext, const tDeviceID deviceID)
{
    ENTRY
    tResult res;
    Query query;

    res = query.Select(LTY_SCAN_CONTEXT, IN deviceID , "");
    if (res) return res;

    tUInt deep = 0;
    tURL path = {0};
    tRowNumber row = 0;
    tUInt i=0;
    /* max depth value for scanning is in accordance with max stored number of directory context elements */
    tUInt maxDirectories = LocalSPM::GetDataProvider().DBVTFileMaxDepthForScan() + 10;

    /* get all infos */
    for (i=0; i<maxDirectories; i++) {

        tDirectoryContext directoryContextElem;
        InitDirectoryContext(OUT directoryContextElem);

        res = query.Get("--iTi",
                OUT &deep,
                OUT path, IN sizeof(path),
                OUT &row);
        if (res) break;

        strncpy_r(directoryContextElem.path, path, sizeof(directoryContextElem.path));
        directoryContextElem.listID = LIST_ID_NONE;
        directoryContextElem.row = row;

        scanContext.dirContext.push_back(directoryContextElem);
    }
    query.End();

    if (i >= maxDirectories)
    {
        /* write a file onto the disk to issue a data base file delete on next restart of system */
        FILE *fp;
        fp = fopen(Database::mDatabaseRecreateFile, "r");
        /* if no recreate file exists create a new one */
        if (!fp) {
        fp = fopen(Database::mDatabaseRecreateFile, "w");
        ETG_TRACE_FATAL (("push back more elements into scan context vector than possible -> recreate MediaPlayer DB at next startup"));
        ETG_TRACE_ERRMEM(("push back more elements into scan context vector than possible -> recreate MediaPlayer DB at next startup"));
    }
        fclose(fp);
    }

    scanContext.deviceID = deviceID;
    scanContext.deep = deep;

    return MP_NO_ERROR;
}

tResult DBManager::StoreAlbumArt(const tDeviceID deviceID, const tAlbumID albumID, const tTitle title,const tAlbumArt coverArtPath, const tImageBlob thumbnailBlob,const tDateTime lastModifiedTime)
{
    ENTRY
    tResult res;
    Query query;

    tBlob blob;
    blob.length = thumbnailBlob.imageSize;
    blob.dataPtr = thumbnailBlob.imageData;
    tBlob *pBlob = &blob;

    /* add it to the album art table */
    res = query.Insert(IN LTY_ALBUMART_INSERT);
    if (res) return res;

    res = query.Put("iittbt",
            IN deviceID,
            IN albumID,
            IN title,
            IN coverArtPath,
    //        IN &blob);
            IN pBlob,
            IN lastModifiedTime); //Do it via pointer to blob struct instead of blob struct itself to satisfy lint
    if (res) return res;

    return MP_NO_ERROR;
}
tResult DBManager::StoreDVDResumeInfo(const tDVDInfoBlob resumeBlob)
{
    ENTRY
    tResult res;
    Query query;

    tBlob blob;
    blob.length = resumeBlob.dataSize;
    blob.dataPtr = resumeBlob.data;
    tBlob *pBlob = &blob;

    /* add it to the album art table */
    res = query.Insert(IN LTY_DVDINFO_RESUME_INSERT);
    if (res) return res;

    res = query.Put("b",
            IN pBlob); //Do it via pointer to blob struct instead of blob struct itself to satisfy lint
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::GetDVDResumeInfo(tDVDInfoBlob &resumeInfo)
{
    ENTRY
    tResult res;
    Query query;
    tBlob blob;

    res = query.Select(LTY_DVDINFO_RESUME,MY_MEDIA,"");

    if (res) return res;

    /* get system infos */
    res = query.Get("b", OUT &blob);

    query.End();
    if (res) return res;

    resumeInfo.dataSize = (tDVDInfoSize)blob.length;
    memcpy(resumeInfo.data ,blob.dataPtr , blob.length); //memory is allocated by Query::Get

    return MP_NO_ERROR;
}
tResult DBManager::StoreDVDSetupInfo(const tDVDInfoBlob setupInfo)
{
    ENTRY
    tResult res;
    Query query;

    tBlob blob;
    blob.length = setupInfo.dataSize;
    blob.dataPtr = setupInfo.data;
    tBlob *pBlob = &blob;

    /* add it to the album art table */
    res = query.Insert(IN LTY_DVDINFO_SETUP_INSERT);
    if (res) return res;

    res = query.Put("b",
            IN pBlob); //Do it via pointer to blob struct instead of blob struct itself to satisfy lint
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::GetDVDSetupInfo(tDVDInfoBlob &setupInfo)
{
    ENTRY
    tResult res;
    Query query;
    tBlob blob;

    res = query.Select(LTY_DVDINFO_SETUP,MY_MEDIA,"");

    if (res) return res;

    /* get system infos */
    res = query.Get("b", OUT &blob);

    query.End();
    if (res) return res;

    setupInfo.dataSize = (tDVDInfoSize)blob.length;
    memcpy(setupInfo.data ,blob.dataPtr , blob.length); //memory is allocated by Query::Get

    return MP_NO_ERROR;
}

tResult DBManager::RemoveAlbumArts(const tDeviceID deviceID)
{
    ENTRY
    tResult res;
    Query query;

    /* delete all elements in album art table for a specific device ID */
    res = query.Delete(IN LTY_ALBUMART_DELETE);
    if (res) return res;

    res = query.Remove("ii", IN 0, IN deviceID);
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::GetCoverArtPath(tAlbumArt &coverArtPath, const tDeviceID deviceID, const tAlbumID albumID)
{
    ENTRY
    tResult res;
    Query query;

    res = query.Select(LTY_ALBUMART, IN deviceID, "ii", IN 0, IN albumID);
    if (res) return res;

    res = query.Get("---T", OUT coverArtPath, IN sizeof(coverArtPath));
    query.End();

    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::GetCoverArtPathByObjectId(tAlbumArt &coverArtPath, const tDeviceID deviceID, const tObjectID objectID)
{
    ENTRY
    tResult res;
    Query query;

    res = query.Select(LTY_ALBUMART_COVERARTPATH_OBJECTID_FILTER, IN deviceID, "ii",IN 0, IN objectID);
    if (res) return res;

    res = query.Get("T", OUT coverArtPath, IN sizeof(coverArtPath));
    query.End();

    if (res) return res;

    return MP_NO_ERROR;
}


tResult DBManager::GetThumbnailImage(tImageBlob &thumbnailBlob, const tDeviceID deviceID, const tAlbumID albumID)
{
    ENTRY
    tResult res;
    Query query;
    tBlob blob;

    res = query.Select(LTY_ALBUMART, IN deviceID, "ii", IN 0, IN albumID);
    if (res) return res;

    res = query.Get("----b", OUT &blob);
    query.End();

    if (res) return res;

    thumbnailBlob.imageSize = (tImageSize)blob.length;
    thumbnailBlob.imageData = (tImageData)blob.dataPtr; //memory is allocated by Query::Get

    return MP_NO_ERROR;
}

tResult DBManager::GetOldDevices(tNumberOfDevices &numberOfDevices, vector<tDeviceID> &deviceIDs)
{
    ENTRY
    tResult res;
    Query query;
    tDeviceID deviceID;

    /* find the devices ordered by LastConnected timestamp */
    res = query.Select(LTY_DEVICES_OLDEST, MY_MEDIA, "");
    if (res) return res;

    /* clear the vector */
    deviceIDs.clear();

    /* get all devices */
    numberOfDevices = 0;

    while(1) {

        res = query.Get(IN "i", OUT &deviceID);
        if (res) break;

        deviceIDs.push_back(deviceID);
        numberOfDevices++;
    }
    query.End();

    return MP_NO_ERROR;
}

tResult DBManager::GetDevice(tDeviceID &deviceID, const tDeviceType deviceType, const tBoolean allDevices/*=false*/,const tBoolean virtualDevice/*=false*/, const tConnectionState connectionState/*=CS_CONNECTED*/)
{
    tResult res;
    tNumberOfDevices numberOfDevices;
    vector<tDeviceInfo> deviceInfos;

    deviceID = DEVICE_ID_NOT_SET;

    res = GetMediaplayerDeviceConnections(OUT numberOfDevices, OUT deviceInfos, IN allDevices,IN virtualDevice,IN connectionState);
    if (res) return res;

    for(unsigned int i=0; i<deviceInfos.size(); i++) {
        if (deviceInfos[i].deviceType == deviceType) {
            deviceID = deviceInfos[i].deviceID;
            break;
        }
    }
    if (deviceID == DEVICE_ID_NOT_SET) return MP_ERR_DB_END_OF_LIST;
    return MP_NO_ERROR;
}

tResult DBManager::GetDevice(tDeviceID &deviceID, const tDeviceType deviceType, const tDeviceName deviceName, const tBoolean allDevices/*=false*/,
        const tConnectionState connectionState/*= CS_CONNECTED*/, const tConnectionType connectionType/*=DCT_UNKNOWN*/)
{
    tResult res;
    tNumberOfDevices numberOfDevices;
    vector<tDeviceInfo> deviceInfos;
    tConnectionType localConnectionType = DCT_UNKNOWN;

    deviceID = DEVICE_ID_NOT_SET;

    /* Get all device connections */
    res = GetMediaplayerDeviceConnections(OUT numberOfDevices, OUT deviceInfos, IN allDevices, IN false, IN connectionState);
    if (res) return res;

    for(unsigned int i=0; i<deviceInfos.size(); i++) {
        /* Take parameter connectionType only into consideration if it is not DCT_UNKNOWN */
        if(DCT_UNKNOWN ==connectionType)
        {
            localConnectionType = deviceInfos[i].connectionType;
        }
        else
        {
            localConnectionType = connectionType;
        }

        if ((deviceInfos[i].deviceType == deviceType)
            &&
            (!strncmp(deviceName, deviceInfos[i].deviceName, strlen_r(deviceName)))
            &&
            (deviceInfos[i].connectionType == localConnectionType)) {
            deviceID = deviceInfos[i].deviceID;
            break;
        }
    }
    if (deviceID == DEVICE_ID_NOT_SET) return MP_ERR_DB_END_OF_LIST;
    return MP_NO_ERROR;
}

tResult DBManager::GetDevice(tDeviceID &deviceID, const tMountPoint mountPoint, const tBoolean allDevices/*=false*/)
{
    tResult res;
    tNumberOfDevices numberOfDevices;
    vector<tDeviceInfo> deviceInfos;

    deviceID = DEVICE_ID_NOT_SET;

    /* Get all device connections */
    res = GetMediaplayerDeviceConnections(OUT numberOfDevices, OUT deviceInfos, IN allDevices);
    if (res) return res;

    for(unsigned int i=0; i<deviceInfos.size(); i++)
    {
        //ETG_TRACE_USR1(("mountPoint               =%s",mountPoint));
        //ETG_TRACE_USR1(("deviceInfos[i].mountPoint=%s",deviceInfos[i].mountPoint));
        if(0 /*equal*/ == strcmp(mountPoint,deviceInfos[i].mountPoint))
        {
            deviceID = deviceInfos[i].deviceID;
            break;
        }
    }
    if (deviceID == DEVICE_ID_NOT_SET) return MP_ERR_DB_END_OF_LIST;
    return MP_NO_ERROR;
}

tResult DBManager::GetDeviceFromAlbumArtString(tDeviceID &deviceID, const tAlbumArt albumArtString)
{
    tResult res;
    tNumberOfDevices numberOfDevices;
    vector<tDeviceInfo> deviceInfos;

    deviceID = DEVICE_ID_NOT_SET;

    /* Get only connected device connections */
    res = GetMediaplayerDeviceConnections(OUT numberOfDevices, OUT deviceInfos, IN true); // PSARCC21-1781
    if (res) return res;

    for(unsigned int i=0; i<deviceInfos.size(); i++)
    {
        if(0 /*equal*/ == strncmp(albumArtString,deviceInfos[i].mountPoint, strlen(deviceInfos[i].mountPoint)))
        {
            deviceID = deviceInfos[i].deviceID;
            break;
        }
    }
    if (deviceID == DEVICE_ID_NOT_SET) return MP_ERR_DB_END_OF_LIST;
    return MP_NO_ERROR;
}

tResult DBManager::ActivateFavorite(const tFavoriteID favoriteID) const
{
    ENTRY
    tResult res;
    Query query;

    /* update the favorites  */
    res = query.Update(LTY_FAVORITES_UPDATE);
    if (res) return res;

    /* find match for favoriteID and update its status to new values (isAvailable,isActive) */
    tBoolean active = true;
    res = query.Change("--iiii", IN 0, IN active, IN 0, IN favoriteID);
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::StoreFavorite(tFavoriteID &retFavID, const tObjectID objectID, const tCategoryType objectType, const tLabelText labelText, const tDescriptorText descriptorText) const
{
    ENTRY
    tResult res;
    Query query;

    tBoolean active = false;
    tBoolean available = true;

    /* add the favorite entry into favorite's table */
    res = query.Insert(LTY_FAVORITES_INSERT);
    if (res) return res;

    res = query.Put("iittii",
            IN objectID,
            IN objectType,
            IN labelText,
            IN descriptorText,
            IN active,
            IN available);
    if (res) return res;

    /*Get the favoriteID generated for the last entry */
    tFavoriteID favoriteID;
    res = query.Select(LTY_FAVORITES, MY_MEDIA , "ii", IN 0, IN objectID);
    if (res) return res;

    while(1) // iterate till last entry matching the given objectID
    {
        res = query.Get("i",OUT &favoriteID);
        if(res == MP_ERR_DB_END_OF_LIST)
        {
            retFavID = favoriteID;
            break;
        }
        if ( res ) return res;
    }
    query.End();

    return MP_NO_ERROR;
}

tResult DBManager::DeleteFavorite(const tFavoriteID favoriteID) const
{
    ENTRY
    tResult res;
    Query query;

    res = query.Delete(LTY_FAVORITES_DELETE);
    if (res) return res;

    res = query.Remove("ii", IN 0, IN favoriteID);
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::ResetFavorites(void) const
{
    ENTRY
    tResult res;
    Query query;

    /* delete all the favorite records */
    res = query.Delete(LTY_FAVORITES_DELETE);
    if (res) return res;

    res = query.Remove("");

    return MP_NO_ERROR;
}

tResult DBManager::UpdateFavoriteStatus(const tFavoriteID favoriteID, const tBool isAvailable, const tBool isActive) const
{
    ENTRY
    tResult res;
    Query query;

    ETG_TRACE_USR3(("DBManager::UpdateFavoriteStatus ->favoriteID:%d , isAvailable:%d , isActive:%d",favoriteID,isAvailable,isActive));

    /* update the favorites  */
    res = query.Update(LTY_FAVORITES_UPDATE);
    if (res) return res;

    /* find match for favoriteID and update its status to new values (isAvailable,isActive) */
    res = query.Change("iiiiii", IN 0, IN isAvailable, IN 0, IN isActive, IN 0, IN favoriteID);
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::DeActivateAllFavorites(void) const
{
    ENTRY
    tResult res;
    Query query;

    /* set all the favorites as not active */
    res = query.Update(LTY_FAVORITES_UPDATE);
    if (res) return res;

    tBoolean active = false;
    res = query.Change("--ii", IN 0, IN active);
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::GetFavoriteInfo(tFavoriteInfo &favoriteInfo, const tFavoriteID favID) const
{
    ENTRY
    tResult res;
    Query query;

    res = query.Select(LTY_FAVORITES, IN favID, "");
    if (res) return res;

    res = query.Get("iiiTTii",
            OUT &favoriteInfo.favoriteId,
            OUT &favoriteInfo.objectID,
            OUT &favoriteInfo.objectType,
            OUT favoriteInfo.labelText, IN sizeof(favoriteInfo.labelText),
            OUT favoriteInfo.descriptorText, IN sizeof(favoriteInfo.descriptorText),
            OUT &favoriteInfo.active,
            OUT &favoriteInfo.available);
    query.End();

    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::GetAllFavoriteInfo(vector<tFavoriteInfo> &favoriteInfos) const
{
    ENTRY
    Query query;
    tResult res;

    /* prepare the sql statement for the devices list */
    res = query.Select(LTY_FAVORITES, MY_MEDIA, "");
    if (res) return res;

    /* clear the vector */
    favoriteInfos.clear();

    /* get all infos */
    while(1) {

        tFavoriteInfo favoriteInfo;

        //InitFavoriteInfo(OUT deviceInfo);

        res = query.Get("iiiTTii",
                OUT &favoriteInfo.favoriteId,
                OUT &favoriteInfo.objectID,
                OUT &favoriteInfo.objectType,
                OUT favoriteInfo.labelText, IN sizeof(favoriteInfo.labelText),
                OUT favoriteInfo.descriptorText, IN sizeof(favoriteInfo.descriptorText),
                OUT &favoriteInfo.active,
                OUT &favoriteInfo.available);
        if (res) break;

        favoriteInfos.push_back(favoriteInfo);
    }
    query.End();

    return MP_NO_ERROR;
}

tResult DBManager::GetObjectIdOfFavorite(tObjectID &objectId, tCategoryType &objectType, const tFavoriteID favoriteId) const
{
    ENTRY
    tResult res;
    Query query;

    res = query.Select(LTY_FAVORITES, IN favoriteId, "");
    if (res) return res;

    res = query.Get("-ii", OUT &objectId, OUT &objectType);
    query.End();

    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::StartListAccess(const tListID listID, const tIndex startIndex, const tIndex noOfRows)
{
    ENTRY
    tResult res;
    DBList *dbList;

    /* get list pointer */
    //dbList = LookupDBList(IN listID);
    GET_DBLIST_POINTER(dbList, listID);
    if (!dbList) return MP_ERR_DB_LIST_NOT_FOUND;

    /* lock the query */
    dbList->mQuery->DoLock();

    /* save the current row */
    res = dbList->mQuery->Save();
    if (res) return res;

    /* set offset and limit */
    res = dbList->mQuery->SetLimitAndRow(IN noOfRows, IN startIndex);
    if (res) return res;

    return MP_NO_ERROR;
}


tResult DBManager::EndListAccess(const tListID listID)
{
    ENTRY
    DBList *dbList;
    tResult res;

    /* get list pointer */
    //dbList = LookupDBList(IN listID);
    GET_DBLIST_POINTER(dbList, listID);
    if (!dbList) return MP_ERR_DB_LIST_NOT_FOUND;

    /* restore the current row */
    res = dbList->mQuery->Restore(); // does implicitly an End() call
    if (res) return res;

    /* unlock the query */
    dbList->mQuery->UnLock();

    return MP_NO_ERROR;
}

tResult DBManager::ReleaseAccess(const tListID listID)
{
    ENTRY
    DBList *dbList;
    tResult res;

    /* get list pointer */
    //dbList = LookupDBList(IN listID);
    GET_DBLIST_POINTER(dbList, listID);
    if (!dbList) return MP_ERR_DB_LIST_NOT_FOUND;

    /* restore the current row */
    res = dbList->mQuery->End();
    if (res) return res;

    /* unlock the query */
    dbList->mQuery->UnLock();

    return MP_NO_ERROR;
}

tResult DBManager::WaitForUpdateMyMediaFinished(const tDeviceID deviceID)
{
    ENTRY

    return mMMUpdt.WaitForFinished(deviceID);
}

tResult DBManager::WaitForUpdateMyMediaFinished()
{
    ENTRY

    return mMMUpdt.WaitForFinished();
}

tResult DBManager::UpdateMyMediaDatabase()
{
    // This is only for test purposes, use UpdateMyMediaDatabase(const tDeviceID deviceID) instead
    ENTRY

    /* do this only if MyMedia shall be supported */
    if(mDB && LocalSPM::GetDataProvider().MyMediaSupported()) {
        mMMUpdt.AddDevice(IN mDB, IN MY_MEDIA);
    }

    return MP_NO_ERROR;
}

tResult DBManager::UpdateMyMediaDatabase(const tDeviceID deviceID)
{
    ENTRY

    /* do this only if MyMedia shall be supported */
    if(mDB && LocalSPM::GetDataProvider().MyMediaSupported()) {
        mMMUpdt.AddDevice(IN mDB, IN deviceID);
    }

    return MP_NO_ERROR;
}

tResult DBManager::DeleteMyMedia()
{
    ENTRY

    if (!mDB) return MP_ERR_DB_NOT_OPEN;
    if (!mDB->Handle()) return MP_ERR_DB_INVALID_HANDLE;

    tResult res;
    res = mDB->DeleteMyMediaDatabase(IN mDB->Handle());
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::WaitForSpeedupDBFinished()
{
    ENTRY

    mMMUpdt.WaitForFinished(IN MY_MEDIA);

    return MP_NO_ERROR;
}

tResult DBManager::SpeedupDB()
{
    ENTRY

    if (!mDB) return MP_ERR_DB_NOT_OPEN;

    mMMUpdt.Speedup(IN mDB);

    return MP_NO_ERROR;
}

tResult DBManager::StopSpeedupDB()
{
    ENTRY

    mMMUpdt.AbortCurrentTask(IN MY_MEDIA);

    return MP_NO_ERROR;
}

tResult DBManager::Sync()
{
    ENTRY_INTERNAL;

    if (!mDB) return MP_ERR_DB_NOT_OPEN;
    if (!mDB->Handle()) return MP_ERR_DB_INVALID_HANDLE;

    return mDB->Sync(IN mDB->Handle());
}

tResult DBManager::RemoveDeviceFromMyMedia(const tDeviceID deviceID)
{
    ENTRY;

    if (!mDB) return MP_ERR_DB_NOT_OPEN;

    /* do this only if MyMedia shall be supported.
       Comment out because MyMedia is needed for cleanup Main DB after reindexing */
    //if (LocalSPM::GetDataProvider().MyMediaSupported()) {
        mMMUpdt.RemoveDevice(IN mDB, IN deviceID);
    //}

    return MP_NO_ERROR;
}

tResult DBManager::UpdateMain(const tDeviceID deviceID)
{
    ENTRY;

    if (!mDB) return MP_ERR_DB_NOT_OPEN;
    if (!mDB->Handle()) return MP_ERR_DB_INVALID_HANDLE;

    tResult res;
    tDeviceInfo deviceInfo;
    res = GetDeviceInfo(OUT deviceInfo, IN deviceID);

    TimeTrace ticks("UpdateMain");

    /* remove all objects in main db which are not found in my media db */
    res = mDB->UpdateMain(IN mDB->Handle(), IN deviceID);
    if (res) return res;

    ticks.elapsed();

    /* remove all objects in the LiveTags table which are already in Objects table */
    res = mDB->CleanupLiveTags(IN mDB->Handle());
    if (res) return res;

    ticks.restart();

    if((LocalSPM::GetDataProvider().DBFileBrowsingByDB())
       &&
       (!IsMassStorageDevice(IN deviceInfo.deviceType))) //DTY_USB, DTY_SD, DTY_CDROM, DTY_FLASH
       {
           /* determine the media content (media types) of the folders inclusive there subfolders */
            res = mDB->DetermineFolderContent(IN mDB->Handle(), IN deviceID);
            if (res) return res;
       }

    ticks.elapsed();

    /* recalculate number of playable files for device and set it */
    res = RecalculateNumberOfFiles(IN deviceID);
    if (res) return res;

    // Clear the free memory of sqlite
    res = mDB->ReleaseDatabaseMemory(mDB->Handle());

    return MP_NO_ERROR;
}

tResult DBManager::MoveDeviceToFromVirtualTable(const tBoolean fromVirtual, const tBoolean toVirtual,
        const tDeviceID deviceID, tDeviceID newDeviceID/*=DEVICE_ID_NOT_SET*/)
{
    ENTRY;
    tResult res;

    if (!mDB) return MP_ERR_DB_NOT_OPEN;
    if (!mDB->Handle()) return MP_ERR_DB_INVALID_HANDLE;

    /* copy device from device table to virtual device table */
    res = mDB->CopyDeviceToFromVirtualTable(IN mDB->Handle(), IN fromVirtual, IN toVirtual, IN deviceID, IN newDeviceID);
    if (res) return res;

    res = RemoveDevice(IN deviceID, IN fromVirtual /*virtualDevice*/);
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::SetDeviceID(const tDeviceID oldDeviceID, const tDeviceID newDeviceID, const tBoolean virtualDevice/*=false*/)
{
    ENTRY
    Query query;
    tResult res;

    /* in case of newDeviceID is already available in DB don't go on */
    tDeviceInfo deviceInfo;
    res = GetDeviceInfo(OUT deviceInfo, IN newDeviceID, IN virtualDevice);
    if (!res) return MP_ERR_DB_UPDATE_FAILED;

    MoveDeviceToFromVirtualTable(IN virtualDevice /*fromVirtual*/, IN virtualDevice /*toVirtual*/, IN oldDeviceID, IN newDeviceID);

    return MP_NO_ERROR;
}

tBoolean DBManager::IsInBothDeviceTables(const tDeviceName deviceName)
{
    ENTRY;
    tBoolean inBothTables = false;
    tBoolean inDevicesTable = false;
    tBoolean inVirtualDevicesTable = false;
    tResult res;
    tNumberOfDevices numberOfDevices;
    vector<tDeviceInfo> deviceInfos;
    vector<tDeviceInfo> virtualDeviceInfos;

    /* search device in devices table */
    res = GetMediaplayerDeviceConnections(OUT numberOfDevices, OUT deviceInfos, IN true/*allDevices*/, IN false/*virtualDevice*/);
    if (res) return inBothTables;

    for(unsigned int i=0; i<deviceInfos.size(); i++) {
        if (strcmp(deviceInfos[i].deviceName, deviceName) == 0) {
            inDevicesTable = true;
            break;
        }
    }

    /* search device in virtual devices table */
    res = GetMediaplayerDeviceConnections(OUT numberOfDevices, OUT virtualDeviceInfos, IN true/*allDevices*/, IN true/*virtualDevice*/);
    if (res) return inBothTables;

    for(unsigned int i=0; i<virtualDeviceInfos.size(); i++) {
        if (strcmp(virtualDeviceInfos[i].deviceName, deviceName) == 0) {
            inVirtualDevicesTable = true;
            break;
        }
    }

    if (inDevicesTable && inVirtualDevicesTable)
    {
        inBothTables = true;
    }

    return inBothTables;
}

tResult DBManager::ResetVirtualDevices(void) const
{
    ENTRY
    tResult res;
    Query query;

    res = query.Delete(LTY_VIRTUAL_DEVICES_DELETE);
    if (res) return res;

    res = query.Remove("");
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::ResetLiveTags(void) const
{
    ENTRY
    tResult res;

    if (!mDB) return MP_ERR_DB_NOT_OPEN;
    if (!mDB->Handle()) return MP_ERR_DB_INVALID_HANDLE;

#if 1 //startup performance issue if LiveTags table grows unlimited - GMMY17-11923
    Query query;

    res = query.Delete(LTY_LIVETAGS_DELETE);
    if (res) return res;

    res = query.Remove("");
    if (res) return res;
#else
    /* remove all objects in the LiveTags table which are already in Objects table */
    res = mDB->CleanupLiveTags(IN mDB->Handle());
    if (res) return res;
#endif

    return MP_NO_ERROR;
}

#if 0
Not used yet but maybe in future. Comment out since LiveTag table is persistent
tResult DBManager::RemoveLiveTags(const tDeviceID deviceID) const
{
    ENTRY
    tResult res;
    Query query;

    res = query.Delete(LTY_LIVETAGS_DELETE);
    if (res) return res;

    res = query.Remove("ii", IN 0, IN deviceID);
    if (res) return res;

    return MP_NO_ERROR;
}
#endif

tResult DBManager::FillupMetaDataIDs(tMediaObject &mediaObject)
{
    ENTRY
    tResult res;
    tTagID objectID = mediaObject.objectID;

    if(MTY_AUDIO_STREAM == mediaObject.mediaType) return MP_NO_ERROR;

    /* found an ID, try to fetch the object */
    if (objectID != OBJECT_ID_NONE)
    {
        /*check object is already part of the indexed DB*/
        tMediaObject sourceMediaObject;
        res = GetMediaObject(OUT sourceMediaObject, INOUT objectID);
        if (res == MP_NO_ERROR) {

            /* fill out all missing ID's */
            if (!mediaObject.MetadataTag1) mediaObject.MetadataTag1 = sourceMediaObject.MetadataTag1;
            if (!mediaObject.MetadataTag2) mediaObject.MetadataTag2 = sourceMediaObject.MetadataTag2;
            if (!mediaObject.MetadataTag3) mediaObject.MetadataTag3 = sourceMediaObject.MetadataTag3;
            if (!mediaObject.MetadataTag4) mediaObject.MetadataTag4 = sourceMediaObject.MetadataTag4;

            /*Synchronize/overwrite the objectID as it might contain temporary ID from VT */
            mediaObject.objectID = sourceMediaObject.objectID;

            if(IsAppleDevice(mediaObject.deviceType) && mediaObject.catType == CTY_SONG){

                /* To check mediaobject catergorytype is CTY_PLAYLIST_ITEM */
                tCategoryType objectType = CTY_NONE;
                GetObjectType(OUT objectType, INOUT mediaObject.MetadataTag1);
                /* To check mediaobject catergorytype is CTY_PLAYLIST_ITEM */
                if(objectType == CTY_PLAYLIST){
                    /* metadata tags are not filled for CTY_PLAYLIST_ITEM to get metadatatags*/
                    GetMetatdataIDs(INOUT mediaObject);
                }
            }
            return MP_NO_ERROR;
        }
        else if (res && res != MP_ERR_DB_END_OF_LIST && res != MP_ERR_DB_UNSUPPORTED_CAT_TYPE) {
            return res;
        }

        /* element as object not found. enter all tags from the player engine into the live tags table */
        switch (mediaObject.mediaType) {
        case MTY_MUSIC_FILE:
            res = StoreLiveTagsElement(IN mediaObject.MetadataField1, IN CTY_GENRE, IN MY_MEDIA, IN FNP_PLAYABLE);
            res = StoreLiveTagsElement(IN mediaObject.MetadataField2, IN CTY_ARTIST, IN MY_MEDIA, IN FNP_PLAYABLE);
            res = StoreLiveTagsElement(IN mediaObject.MetadataField3, IN CTY_COMPOSER, IN MY_MEDIA, IN FNP_PLAYABLE);
            res = StoreLiveTagsElement(IN mediaObject.MetadataField4, IN CTY_ALBUM, IN MY_MEDIA, IN FNP_PLAYABLE);
            //res = StoreLiveTagsElement(IN mediaObject.title, CTY_SONG, IN mediaObject.deviceID, IN mediaObject.notPlayable);
            /* Update notPlayable flag */
            res = UpdateLiveTagsElement(IN objectID, IN mediaObject.notPlayable); //CTY_SONG
            break;
        case MTY_PODCAST:
            res = StoreLiveTagsElement(IN mediaObject.MetadataField1, IN CTY_NAME, IN MY_MEDIA, IN FNP_PLAYABLE);
            //res = StoreLiveTagsElement(IN mediaObject.title, CTY_PODCAST, IN mediaObject.deviceID, IN mediaObject.notPlayable);
            /* Update notPlayable flag */
            res = UpdateLiveTagsElement(IN objectID, IN mediaObject.notPlayable); //CTY_PODCAST
            break;
        case MTY_AUDIOBOOK:
            res = StoreLiveTagsElement(IN mediaObject.MetadataField1, IN CTY_AUTHOR, IN MY_MEDIA, IN FNP_PLAYABLE);
            res = StoreLiveTagsElement(IN mediaObject.MetadataField2, IN CTY_TITLE, IN MY_MEDIA, IN FNP_PLAYABLE);
            //res = StoreLiveTagsElement(IN mediaObject.title, CTY_AUDIOBOOK, IN mediaObject.deviceID, IN mediaObject.notPlayable);
            /* Update notPlayable flag */
            res = UpdateLiveTagsElement(IN objectID, IN mediaObject.notPlayable); //CTY_AUDIOBOOK
            break;
        case MTY_VIDEO:
            res = StoreLiveTagsElement(IN mediaObject.MetadataField1, IN CTY_NAME, IN MY_MEDIA, IN FNP_PLAYABLE);
            //res = StoreLiveTagsElement(IN mediaObject.title, CTY_VIDEO, IN mediaObject.deviceID, IN mediaObject.notPlayable);
            /* Update notPlayable flag */
            res = UpdateLiveTagsElement(IN objectID, IN mediaObject.notPlayable); //CTY_VIDEO
            break;
        case MTY_PLAYLIST:
        default:
            return MP_ERR_DB_UNSUPPORTED_CAT_TYPE;
        }
    }

    /* no ID found: use the live tags table to find the tag id's */
    //if (!mediaObject.MetadataTag1) {
        res = GetLiveTagsElement(OUT mediaObject.MetadataTag1, CTY_GENRE, IN mediaObject.MetadataField1);
        if (res && res != MP_ERR_DB_END_OF_LIST) return res;
    //}
    //if (!mediaObject.MetadataTag1) {
        res = GetLiveTagsElement(OUT mediaObject.MetadataTag1, CTY_AUTHOR, IN mediaObject.MetadataField1);
        if (res && res != MP_ERR_DB_END_OF_LIST) return res;
    //}
    //if (!mediaObject.MetadataTag1) {
        res = GetLiveTagsElement(OUT mediaObject.MetadataTag1, CTY_NAME, IN mediaObject.MetadataField1);
        if (res && res != MP_ERR_DB_END_OF_LIST) return res;
    //}

    //if (!mediaObject.MetadataTag2) {
        res = GetLiveTagsElement(OUT mediaObject.MetadataTag2, CTY_ARTIST, IN mediaObject.MetadataField2);
        if (res && res != MP_ERR_DB_END_OF_LIST) return res;
    //}
    //if (!mediaObject.MetadataTag2) {
        res = GetLiveTagsElement(OUT mediaObject.MetadataTag2, CTY_TITLE, IN mediaObject.MetadataField2);
        if (res && res != MP_ERR_DB_END_OF_LIST) return res;
    //}

    //if (!mediaObject.MetadataTag3) {
        res = GetLiveTagsElement(OUT mediaObject.MetadataTag3, CTY_COMPOSER, IN mediaObject.MetadataField3);
        if (res && res != MP_ERR_DB_END_OF_LIST) return res;
    //}

    //if (!mediaObject.MetadataTag4) {
        res = GetLiveTagsElement(OUT mediaObject.MetadataTag4, CTY_ALBUM, IN mediaObject.MetadataField4);
        if (res && res != MP_ERR_DB_END_OF_LIST) return res;
    //}

    return MP_NO_ERROR;
}

tResult DBManager::StoreLiveTagsElement(const tTag tag, const tCategoryType catType, const tDeviceID deviceID, const tNotPlayable notPlayable) const
{
    ENTRY;
    VARTRACE(tag);
    VARTRACE(catType);
    VARTRACE(deviceID);
    VARTRACE(notPlayable);

    //store only valid tag strings
    if (strlen (tag)) {
        tResult res;
        Query query;

        res = query.Insert(LTY_LIVETAGS_INSERT);
        if (res) return res;

        tTag trimmedTag;
        strncpy_r(trimmedTag, tag, sizeof(trimmedTag));

        res = query.Put("iiti", IN deviceID, IN catType, IN strtrim(trimmedTag), IN notPlayable);
        if (res) return res;
    }
    return MP_NO_ERROR;
}

tResult DBManager::UpdateLiveTagsElement(const tTagID tagID, const tNotPlayable notPlayable) const
{
    ENTRY;
    VARTRACE(tagID);
    VARTRACE(notPlayable);
    tResult res;
    Query query;

    /* update the LiveTags table */
    res = query.Update(LTY_LIVETAGS_UPDATE);
    if (res) return res;

    /* find match for tagID and update notPlayable flag */
    res = query.Change("iiii", IN 0, IN notPlayable, IN 0, IN tagID);
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::GetLiveTagsElement(tTag &tag, tCategoryType &catType, tDeviceID &deviceID, tNotPlayable &notPlayable, const tTagID tagID) const
{
    ENTRY;
    tResult res;
    Query query;

    res = query.Select(LTY_LIVETAGS_SELECT, MY_MEDIA, "ii", IN 0, IN tagID);
    if (res) return res;

    res = query.Get("-iiTi",
            OUT &deviceID,
            OUT &catType,
            OUT tag, IN sizeof(tag),
            OUT &notPlayable);
    query.End();
    if (res) return res;

    ETG_TRACE_USR3(("GetLiveTagsElement tagID: %u, deviceID: %u, catType: %u, notPlayable: %u, tag: %s", tagID, deviceID, catType, notPlayable, tag));
    return MP_NO_ERROR;
}

tResult DBManager::GetLiveTagsElement(tTagID &tagID, const tCategoryType catType, const tTag tag) const
{
    ENTRY;
    tResult res;
    Query query;

    res = query.Select(LTY_LIVETAGS_SELECT, MY_MEDIA, "--iiit", IN 0, IN catType, IN 0, IN tag);
    if (res) return res;

    res = query.Get("i", OUT &tagID);
    query.End();
    if (res) return res;

    ETG_TRACE_USR3(("GetLiveTagsElement tagID: %u, catType: %u, tag: %s", tagID, catType, tag));
    return MP_NO_ERROR;
}

tResult DBManager::GetURLOfMediaObject(tTag &url, const tTagID tagID, const tCategoryType catType) const
{
    ENTRY;
    tResult res;
    Query query;

    if(CTY_PLAYLIST == catType) //TagID = FilterTag1ID
    {
        res = query.Select(LTY_MEDIAOBJECTS_URL, MY_MEDIA, "--ii", IN 0, IN tagID);
    }
    else                        //TagID = MediaObjectID
    {
        res = query.Select(LTY_MEDIAOBJECTS_URL, MY_MEDIA, "ii", IN 0, IN tagID);
    }
    if (res) return res;

    res = query.Get("T", OUT url, IN sizeof(url));
    query.End();
    if (res) return res;

    return MP_NO_ERROR;
}



/* mask out all filtertags not matching the columns according to list type */
tResult DBManager::MaskOutFilterTagsByListType(tFilterTag1 &filterTag1,
        tFilterTag2 &filterTag2, tFilterTag3 &filterTag3,
        tFilterTag4 &filterTag4, const tListType listType)
{
    ENTRY_INTERNAL;

    //matching filtertags according to
    //GIS-351-2340 Fist Filter Types
    //all entities with 'LTY_...SEARCH' added for Roadmap 16014 'full text search'


    switch(listType) {
    case LTY_GENRE:
    case LTY_GENRE_SEARCH:
    case LTY_ARTIST:
    case LTY_ARTIST_SEARCH:
    case LTY_ALBUM:
    case LTY_ALBUM_SEARCH:
    case LTY_ALBUM_UNKNOWN_ALBUMART:     /*tbd.:fts*/
    case LTY_SONG:
    case LTY_SONG_SEARCH:
    case LTY_PODCAST:
    case LTY_PODCAST_SEARCH:
    case LTY_AUDIOBOOK:
    case LTY_AUDIOBOOK_SEARCH:
    case LTY_AUTHOR:
    case LTY_AUTHOR_SEARCH:
    case LTY_COMPOSER:
    case LTY_COMPOSER_SEARCH:
    case LTY_VIDEO:
    case LTY_VIDEO_SEARCH:
    case LTY_PLAYLIST:
    case LTY_PLAYLIST_SEARCH:
    case LTY_PLAYLIST_INTERNAL:
    case LTY_PLAYLIST_INTERNAL_SEARCH:
    case LTY_CURRENT_SELECTION:
    case LTY_IMAGE:
    case LTY_IMAGE_SEARCH:
    case LTY_IMAGE_FOLDER:             /*tbd.:fts*/
    case LTY_IMAGE_FOLDER_ITEM:        /*tbd.:fts*/
    case LTY_COMPILATION:
    case LTY_COMPILATION_SEARCH:
    case LTY_END_OF_LTY:              /*tbd.:fts*/
    case LTY_CD:                      /*tbd.:fts*/
    case LTY_YEAR:
    case LTY_USER_FAVORITES_SONG:
    case LTY_USER_FAVORITES_SONG_SEARCH:
        filterTag1.tag = 0;
        filterTag2.tag = 0;
        filterTag3.tag = 0;
        filterTag4.tag = 0;
        break;
    case LTY_COMPOSER_ALBUM:
    case LTY_COMPOSER_ALBUM_SEARCH:
    case LTY_COMPOSER_SONG:
    case LTY_COMPOSER_SONG_SEARCH:
        filterTag1.tag = 0;
        filterTag2.tag = 0;
        filterTag4.tag = 0;
        break;
    case LTY_COMPOSER_ALBUM_SONG:
    case LTY_COMPOSER_ALBUM_SONG_SEARCH:
        filterTag1.tag = 0;
        filterTag2.tag = 0;
        break;
    case LTY_GENRE_ARTIST:
    case LTY_GENRE_ARTIST_SEARCH:
    case LTY_GENRE_ALBUM:
    case LTY_GENRE_ALBUM_SEARCH:
    case LTY_GENRE_SONG:
    case LTY_GENRE_SONG_SEARCH:
    case LTY_PODCAST_EPISODE:
    case LTY_PODCAST_EPISODE_SEARCH:
    case LTY_AUTHOR_BOOKTITLE:
    case LTY_AUTHOR_BOOKTITLE_SEARCH:
    case LTY_VIDEO_EPISODE:
    case LTY_VIDEO_EPISODE_SEARCH:
    case LTY_EPISODE_OF_VIDEO:
    case LTY_EPISODE_OF_VIDEO_SEARCH:
    case LTY_PLAYLIST_SONG:
    case LTY_PLAYLIST_SONG_SEARCH:
    case LTY_YEAR_ALBUM:
        filterTag2.tag = 0;
        filterTag3.tag = 0;
        filterTag4.tag = 0;
        break;
    case LTY_GENRE_ARTIST_ALBUM:
    case LTY_GENRE_ARTIST_ALBUM_SEARCH:
    case LTY_GENRE_ARTIST_SONG:
    case LTY_GENRE_ARTIST_SONG_SEARCH:
    case LTY_AUTHOR_BOOKTITLE_CHAPTER:
    case LTY_AUTHOR_BOOKTITLE_CHAPTER_SEARCH:
        filterTag3.tag = 0;
        filterTag4.tag = 0;
        break;
    case LTY_GENRE_ARTIST_ALBUM_SONG:
    case LTY_GENRE_ARTIST_ALBUM_SONG_SEARCH:
        filterTag3.tag = 0;
        break;
    case LTY_GENRE_ALBUM_SONG:
    case LTY_GENRE_ALBUM_SONG_SEARCH:
        filterTag2.tag = 0;
        filterTag3.tag = 0;
        break;
    case LTY_ARTIST_ALBUM:
    case LTY_ARTIST_ALBUM_SEARCH:
    case LTY_ARTIST_SONG:
    case LTY_ARTIST_SONG_SEARCH:
    case LTY_BOOKTITLE_CHAPTER:
    case LTY_BOOKTITLE_CHAPTER_SEARCH:
        filterTag1.tag = 0;
        filterTag3.tag = 0;
        filterTag4.tag = 0;
        break;
    case LTY_YEAR_ALBUM_SONG:
    case LTY_YEAR_ALBUM_SONG_SEARCH:
           filterTag2.tag = 0;
           filterTag3.tag = 0;
           break;
    case LTY_ARTIST_ALBUM_SONG:
    case LTY_ARTIST_ALBUM_SONG_SEARCH:
        filterTag1.tag = 0;
        filterTag3.tag = 0;
        break;
    case LTY_ALBUM_SONG:
    case LTY_ALBUM_SONG_SEARCH:
        break;
    case LTY_COMPILATION_SONG:
    case LTY_COMPILATION_SONG_SEARCH:
        filterTag1.tag = 0;
        filterTag2.tag = 0;
        filterTag3.tag = 0;
        break;

    default:
        return MP_ERR_DB_UNSUPPORTED_LIST_TYPE;
    }

    return MP_NO_ERROR;
}



tResult DBManager::GetFileFormat(tFileFormat &fileFormat, tFileType &fileType, const tFilename fileName) const
{
    ENTRY_INTERNAL;
    tResult res;

    if (!mDB) return MP_ERR_DB_NOT_OPEN;

    /* get file attributes from configuration via file extension */
    res = mDB->GetFileFormat(OUT fileFormat, OUT fileType, IN fileName);
    if(LocalSPM::GetDataProvider().InformVideoFileAsAudio())  //TODO(pee1cob): checking InformVideoFileAsAudio() be done  only when res is MP_NO_ERROR
    {
        if(fileType == FT_VIDEO)
        {
            fileType = FT_AUDIO;
        }
    }
    if (res) return res;

    return MP_NO_ERROR;
}

tCategoryType DBManager::GetCTYByLTY(const tListType listType) const
{
    ENTRY_INTERNAL;

    switch (listType) {
    case LTY_GENRE:
    case LTY_GENRE_SEARCH:
        return CTY_GENRE;
    //------------------------------
    case LTY_ARTIST:
    case LTY_ARTIST_SEARCH:
    case LTY_END_OF_LTY:
    case LTY_GENRE_ARTIST:
    case LTY_GENRE_ARTIST_SEARCH:
        return CTY_ARTIST;
    //------------------------------
    case LTY_ALBUM:
    case LTY_ALBUM_SEARCH:
    case LTY_ALBUM_UNKNOWN_ALBUMART:
    case LTY_GENRE_ARTIST_ALBUM:
    case LTY_GENRE_ARTIST_ALBUM_SEARCH:
    case LTY_GENRE_ALBUM:
    case LTY_GENRE_ALBUM_SEARCH:
    case LTY_ARTIST_ALBUM:
    case LTY_ARTIST_ALBUM_SEARCH:
    case LTY_COMPOSER_ALBUM:
    case LTY_YEAR_ALBUM:
    case LTY_COMPOSER_ALBUM_SEARCH:
    case LTY_COMPILATION:
    case LTY_COMPILATION_SEARCH:
        return CTY_ALBUM;
    //------------------------------
    //------------------------------
    case LTY_YEAR:
        return CTY_YEAR;
    //------------------------------
    case LTY_SONG:
    case LTY_SONG_SEARCH:
    case LTY_GENRE_ARTIST_ALBUM_SONG:
    case LTY_GENRE_ARTIST_ALBUM_SONG_SEARCH:
    case LTY_GENRE_ARTIST_SONG:
    case LTY_GENRE_ARTIST_SONG_SEARCH:
    case LTY_GENRE_ALBUM_SONG:
    case LTY_GENRE_ALBUM_SONG_SEARCH:
    case LTY_YEAR_ALBUM_SONG:
    case LTY_YEAR_ALBUM_SONG_SEARCH:
    case LTY_GENRE_SONG:
    case LTY_GENRE_SONG_SEARCH:
    case LTY_ARTIST_ALBUM_SONG:
    case LTY_ARTIST_ALBUM_SONG_SEARCH:
    case LTY_ARTIST_SONG:
    case LTY_ARTIST_SONG_SEARCH:
    case LTY_ALBUM_SONG:
    case LTY_ALBUM_SONG_SEARCH:
    case LTY_COMPOSER_ALBUM_SONG:
    case LTY_COMPOSER_ALBUM_SONG_SEARCH:
    case LTY_COMPOSER_SONG:
    case LTY_COMPOSER_SONG_SEARCH:
    case LTY_PLAYLIST_SONG:
    case LTY_PLAYLIST_SONG_SEARCH:

    case LTY_FILELIST_PLAYLIST:
    case LTY_CURRENT_SELECTION:                     //best guess
    case LTY_FILELIST:                //best guess
    case LTY_FILELIST_MEDIAOBJECTS:   //best guess
    case LTY_FILELIST_MEDIAOBJECTS_WITH_SUBFOLDERS: //best guess
    case LTY_BROWSELIST:              //best guess
    case LTY_BROWSELIST_MEDIAOBJECTS: //best guess
    case LTY_BROWSELIST_MEDIAOBJECTS_WITH_SUBFOLDERS: //best guess
    case LTY_COMPILATION_SONG:
    case LTY_COMPILATION_SONG_SEARCH:
    case LTY_USER_FAVORITES_SONG:
    case LTY_USER_FAVORITES_SONG_SEARCH:
        return CTY_SONG;
    //------------------------------
    case LTY_PODCAST_EPISODE:
    case LTY_PODCAST_EPISODE_SEARCH:
        return CTY_PODCAST;
    //------------------------------
    case LTY_VIDEO_EPISODE:
    case LTY_VIDEO_EPISODE_SEARCH:
    case LTY_EPISODE_OF_VIDEO:
    case LTY_EPISODE_OF_VIDEO_SEARCH:
        return CTY_VIDEO;
    //------------------------------
    case LTY_AUDIOBOOK:
    case LTY_AUDIOBOOK_SEARCH:
    case LTY_AUTHOR_BOOKTITLE:
    case LTY_AUTHOR_BOOKTITLE_SEARCH:
        return CTY_TITLE;
    //------------------------------
    case LTY_BOOKTITLE_CHAPTER:
    case LTY_BOOKTITLE_CHAPTER_SEARCH:
    case LTY_AUTHOR_BOOKTITLE_CHAPTER:
    case LTY_AUTHOR_BOOKTITLE_CHAPTER_SEARCH:
        return CTY_AUDIOBOOK;
    //------------------------------
    case LTY_AUTHOR:
    case LTY_AUTHOR_SEARCH:
        return CTY_AUTHOR;
    //------------------------------
    case LTY_COMPOSER:
    case LTY_COMPOSER_SEARCH:
        return CTY_COMPOSER;
    //------------------------------
    case LTY_VIDEO:
    case LTY_VIDEO_SEARCH:
    case LTY_PODCAST:
    case LTY_PODCAST_SEARCH:
        return CTY_NAME;
    //------------------------------
    case LTY_PLAYLIST:
    case LTY_PLAYLIST_SEARCH:
        return CTY_PLAYLIST;
    //------------------------------
    case LTY_PLAYLIST_INTERNAL:
    case LTY_PLAYLIST_INTERNAL_SEARCH:
        return CTY_PLAYLIST_INTERNAL;
    //------------------------------
    case LTY_IMAGE:
    case LTY_IMAGE_SEARCH:
    case LTY_IMAGE_FOLDER:
    case LTY_IMAGE_FOLDER_ITEM:
        return CTY_IMAGE;
    //------------------------------
    case LTY_CD:
        return CTY_SONG;
    //------------------------------
    default:
        break;
    }

    ETG_TRACE_ERR(("GetCTYByLTY() - Invalid LTY %d", listType));
    return CTY_NONE;
}

tCategoryType DBManager::GetCTYByMTY(const tMediaType mediaType) const
{
    ENTRY_INTERNAL;
    switch (mediaType) {
    case MTY_MUSIC_FILE:
        return CTY_SONG;
    case MTY_PLAYLIST:
        return CTY_PLAYLIST;
    case MTY_PLAYLIST_INTERNAL:
        return CTY_PLAYLIST_INTERNAL;
    case MTY_PODCAST:
        return CTY_PODCAST;
    case MTY_VIDEO:
        return CTY_VIDEO;
    case MTY_AUDIOBOOK:
        return CTY_AUDIOBOOK;
    case MTY_IMAGE:
        return CTY_IMAGE;
    case MTY_AUDIO_STREAM:
        return CTY_SONG;
    case MTY_UNKNOWN:
    default:
        break;
    }

    ETG_TRACE_ERR(("GetCTYByMTY() - Invalid MTY %d", mediaType));
    return CTY_NONE;
}

tMediaType DBManager::GetMTYByCTY(const tCategoryType catType) const
{
    ENTRY_INTERNAL;
    switch (catType) {
    case CTY_GENRE:
    case CTY_ARTIST:
    case CTY_ALBUM:
    case CTY_SONG:
    case CTY_COMPOSER:
    case CTY_YEAR:
        return MTY_MUSIC_FILE;
    case CTY_PODCAST:
        return MTY_PODCAST;
    case CTY_VIDEO:
    case CTY_EPISODE: //maybe MTY_PODCAST
    case CTY_NAME:    //maybe MTY_PODCAST
        return MTY_VIDEO;
    case CTY_PLAYLIST:
    case CTY_PLAYLIST_ITEM:
        return MTY_PLAYLIST;
    case CTY_PLAYLIST_INTERNAL:
        return MTY_PLAYLIST_INTERNAL;
    case CTY_AUDIOBOOK:
    case CTY_AUTHOR:
    case CTY_TITLE:
    case CTY_CHAPTER:
        return MTY_AUDIOBOOK;
    case CTY_IMAGE:
        return MTY_IMAGE;
    default:
        break;
    }

    ETG_TRACE_ERR(("GetMTYByCTY() - Invalid CTY %d", catType));
    return MTY_UNKNOWN;
}

tFileType DBManager::GetFTByMTY(const tMediaType mediaType) const
{
    ENTRY_INTERNAL;
    switch (mediaType) {
    case MTY_MUSIC_FILE:
    case MTY_PODCAST:       //can be audio or video
    case MTY_AUDIOBOOK:
    case MTY_AUDIO_STREAM:
        return FT_AUDIO;
    case MTY_VIDEO:
        return FT_VIDEO;
    case MTY_PLAYLIST:
    case MTY_PLAYLIST_INTERNAL:
        return FT_PLAYLIST;
    case MTY_UNKNOWN:
    case MTY_IMAGE:
        return FT_IMAGE;
    default:
        break;
    }

    ETG_TRACE_ERR(("GetFTByMTY() - Invalid MTY %d", mediaType));
    return FT_UNKNOWN;
}

tListType DBManager::GetLTYByCTY(const tCategoryType catType) const
{
    ENTRY_INTERNAL;

    /* set the list type based on category type */
    switch(catType) {
    case CTY_SONG:
//  case CTY_AUDIOBOOK:
    case CTY_VIDEO:
//  case CTY_PODCAST:
    case CTY_CHAPTER:
    case CTY_EPISODE:
        return LTY_SONG;
    case CTY_GENRE:
        return LTY_GENRE_SONG;
    case CTY_ARTIST:
        return LTY_ARTIST_SONG;
    case CTY_ALBUM:
        return LTY_ALBUM_SONG;
    case CTY_COMPOSER:
        return LTY_COMPOSER_SONG;
    case CTY_NAME:    // meant is a video
        return LTY_VIDEO_EPISODE;
    case CTY_PODCAST:
        return LTY_PODCAST_EPISODE;
    case CTY_AUDIOBOOK:
    case CTY_TITLE:
        return LTY_BOOKTITLE_CHAPTER;
    case CTY_PLAYLIST:
        return LTY_PLAYLIST_SONG;
    default:
        break;
    }

    ETG_TRACE_ERR(("GetLTYByCTY() - Invalid CTY %d", catType));
    return LTY_END_OF_EXTERNAL_LIST_TYPES;
}

tResult DBManager::GetVFTByFTS(tVTFileType &fileType1, tVTFileType &fileType2, tVTFileType &fileType3, tVTFileType &fileType4,
        const tFileTypeSelection fileTypeSelection, const tBoolean onlyPlayableObjects) const
{
    ENTRY_INTERNAL;

    fileType1 = FILE_TYPE_DIRECTORY;
    fileType2 = FILE_TYPE_DIRECTORY;
    fileType3 = FILE_TYPE_DIRECTORY;
    fileType4 = FILE_TYPE_DIRECTORY;

    /* set the file type based on file type selection */
    switch(fileTypeSelection)
    {
    case FTS_AUDIO_VIDEO_PLAYLIST:
        fileType1 = FILE_TYPE_AUDIOOBJECT;
        fileType2 = FILE_TYPE_VIDEOOBJECT;
        if(!onlyPlayableObjects) fileType3 = FILE_TYPE_PLAYLIST;
        break;
    case FTS_AUDIO:
        fileType1 = FILE_TYPE_AUDIOOBJECT;
        fileType2 = FILE_TYPE_AUDIOOBJECT;
        break;
    case FTS_VIDEO:
        fileType1 = FILE_TYPE_VIDEOOBJECT;
        fileType2 = FILE_TYPE_VIDEOOBJECT;
        break;
    case FTS_PLAYLIST:
        if(!onlyPlayableObjects) fileType1 = FILE_TYPE_PLAYLIST;
        break;
    case FTS_IMAGE:
        if(!onlyPlayableObjects) fileType1 = FILE_TYPE_IMAGE;
        break;
    case FTS_AUDIO_PLAYLIST:
        fileType1 = FILE_TYPE_AUDIOOBJECT;
        if(!onlyPlayableObjects) fileType2 = FILE_TYPE_PLAYLIST;
        break;
    case FTS_ALL:
    default :
        fileType1 = FILE_TYPE_AUDIOOBJECT;
        fileType2 = FILE_TYPE_VIDEOOBJECT;
        if(!onlyPlayableObjects) fileType3 = FILE_TYPE_PLAYLIST;
        if(!onlyPlayableObjects) fileType4 = FILE_TYPE_IMAGE;
        break;
    }

    return MP_NO_ERROR;
}

tResult DBManager::GetCTYByFTS(tCategoryType &catType1, tCategoryType &catType2, tCategoryType &catType3, tCategoryType &catType4,
        const tFileTypeSelection fileTypeSelection, const tBoolean onlyPlayableObjects) const
{
    ENTRY_INTERNAL;

    catType1 = CTY_DIR;
    catType2 = CTY_DIR;
    catType3 = CTY_DIR;
    catType4 = CTY_DIR;

    /* set the file type based on file type selection */
    switch(fileTypeSelection)
    {
    case FTS_AUDIO_VIDEO_PLAYLIST:
        catType1 = CTY_SONG;
        catType2 = CTY_VIDEO;
        if(!onlyPlayableObjects) catType3 = CTY_PLAYLIST;
        if(!onlyPlayableObjects) catType4 = CTY_PLAYLIST_INTERNAL;
        break;
    case FTS_AUDIO:
        catType1 = CTY_SONG;
        catType2 = CTY_SONG;
        break;
    case FTS_VIDEO:
        catType1 = CTY_VIDEO;
        catType2 = CTY_VIDEO;
        break;
    case FTS_PLAYLIST:
        if(!onlyPlayableObjects) catType1 = CTY_PLAYLIST;
        if(!onlyPlayableObjects) catType2 = CTY_PLAYLIST_INTERNAL;
        break;
    case FTS_IMAGE:
        if(!onlyPlayableObjects) catType1 = CTY_IMAGE;
        break;
    case FTS_AUDIO_PLAYLIST:
        catType1 = CTY_SONG;
        if(!onlyPlayableObjects) catType2 = CTY_PLAYLIST;
        else catType2 = CTY_SONG;
        if(!onlyPlayableObjects) catType3 = CTY_PLAYLIST_INTERNAL;
        break;
    case FTS_ALL:
    default :
        catType1 = CTY_SONG;
        catType2 = CTY_VIDEO;
        if(!onlyPlayableObjects) catType3 = CTY_PLAYLIST;
        if(!onlyPlayableObjects) catType4 = CTY_IMAGE;
        break;
    }

    return MP_NO_ERROR;
}

tResult DBManager::GetMediaContentByFTS(tMediaContent &mediaContent,
        const tFileTypeSelection fileTypeSelection, const tBoolean onlyPlayableObjects) const
{
    ENTRY_INTERNAL;

    mediaContent = MTY_UNKNOWN;

    /* set the file type based on file type selection */
    switch(fileTypeSelection)
    {
    case FTS_AUDIO_VIDEO_PLAYLIST:
        if(onlyPlayableObjects) {
            mediaContent = (MTY_MUSIC_FILE | MTY_VIDEO);
        }
        else {
            mediaContent = (MTY_MUSIC_FILE | MTY_VIDEO | MTY_PLAYLIST | MTY_PLAYLIST_INTERNAL);
        }
        break;
    case FTS_AUDIO:
        mediaContent = MTY_MUSIC_FILE;
        break;
    case FTS_VIDEO:
        mediaContent = MTY_VIDEO;
        break;
    case FTS_PLAYLIST:
        if(!onlyPlayableObjects) {
            mediaContent = (MTY_PLAYLIST | MTY_PLAYLIST_INTERNAL);
        }
        break;
    case FTS_IMAGE:
        if(!onlyPlayableObjects) mediaContent = MTY_IMAGE;
        break;
    case FTS_AUDIO_PLAYLIST:
        if(onlyPlayableObjects) {
            mediaContent = MTY_MUSIC_FILE;
        }
        else {
            mediaContent = (MTY_MUSIC_FILE | MTY_PLAYLIST | MTY_PLAYLIST_INTERNAL);
        }
        break;
    case FTS_ALL:
    default :
        if(onlyPlayableObjects) {
            mediaContent = (MTY_MUSIC_FILE | MTY_VIDEO);
        }
        else {
            mediaContent = (MTY_MUSIC_FILE | MTY_VIDEO | MTY_PLAYLIST | MTY_PLAYLIST_INTERNAL | MTY_IMAGE);
        }
        break;
    }

    return MP_NO_ERROR;
}

tResult DBManager::GetVTTagForIPODPlaylist(tTagID &playlistTag) const
{
    ENTRY;
    tResult res;

    /* Playlist items retrieved always via VTIPOD */
    /* get the VTIPOD tag for playlist parent form URL of playlist */
    /* filterTag1.tag currently holds the DB object id that does not match the VT */

    //get the URL
    tTag playlistUrl = {0};
    res = GetURLOfMediaObject(OUT playlistUrl, IN playlistTag, IN CTY_PLAYLIST);
    if (res) return res;
    ETG_TRACE_USR3(("VTIPOD - Got URL of Playlist"));
    VARTRACE(playlistTag);
    VARTRACE(playlistUrl);

    //extract tags from url
    tListType listType;
    tMetadata name = {0};
    tUUID uuid = {0};
    res = LocalSPM::GetIPODControl().GetVTTagByURL(OUT playlistTag, OUT listType, OUT name, OUT uuid, IN playlistUrl);
    if (res) return res;

    playlistTag = playlistTag | IPOD_MARKER_BIT;
    ETG_TRACE_USR3(("VTIPOD - Got Tags from playlist URL"));
    VARTRACE(playlistTag);

    return MP_NO_ERROR;
}

tResult DBManager::GetVTTagForIPOD(tTagID &tagID) const
{
    ENTRY;
    tResult res;

    if (tagID & IPOD_MARKER_MASK) {
        tTag tag;
        tCategoryType catType;
        tDeviceID deviceID;
        tNotPlayable notPlayable;
        res = GetLiveTagsElement(OUT tag, OUT catType, OUT deviceID, OUT notPlayable, IN (tagID & IPOD_MARKER_MASK));
        if (res) return res;

        //double check CTY from live tag table
        if (catType != CTY_VTIPOD) {
            res = MP_ERR_DB_UNSUPPORTED_CAT_TYPE;
            return res;
        }

        tListType listType = LTY_END_OF_EXTERNAL_LIST_TYPES;
        tMetadata name = {0};
        tUUID uuid = {0};
        res = LocalSPM::GetIPODControl().GetVTTagByURL(OUT tagID, OUT listType, OUT name, OUT uuid, IN tag);
        if (res)return res;
    }

    tagID = tagID | IPOD_MARKER_BIT;
    return MP_NO_ERROR;
}

tResult DBManager::GetFavoriteNameIPOD(tTitle &name, tObjectID &objectID, tCategoryType &objectType)
{
    ENTRY;
    tResult res;

    //remove marker bit
    objectID = objectID & IPOD_MARKER_MASK;

    //get URL from live tags
    tTag tag;
    tDeviceID deviceID;
    tNotPlayable notPlayable;
    res = GetLiveTagsElement(OUT tag, OUT objectType, OUT deviceID, OUT notPlayable, IN objectID);
    if (res) return res;
    if (objectType != CTY_VTIPOD) {
        return MP_ERR_IPOD_METADATA;
    }

    //get CTP by LTY
    tTagID tagID;
    tListType listType = tListType_init;
    tUUID uuid;
    res = LocalSPM::GetIPODControl().GetVTTagByURL(OUT tagID, OUT listType, OUT name, OUT uuid, IN tag);
    if (res) return res;
    objectType = GetCTYByLTY(listType);

    //handle CTYs
    if((objectType == CTY_SONG) ||
       (objectType == CTY_AUDIOBOOK) ||
       (objectType == CTY_VIDEO) ||
       (objectType == CTY_PODCAST)) {


         /* look for the media object ID in the songs */
        if(uuid[0] != 0) {
            tMediaObject mediaObject;
            InitMediaObject(OUT mediaObject);
            res = GetIPODMediaObjectByUUID(INOUT mediaObject, IN deviceID, IN uuid);
            if (res) return res;
            strncpy_r(name, mediaObject.title, sizeof(name));
            objectID = mediaObject.objectID;
            objectType = mediaObject.catType;
        } else {
            Query query;
            res = query.Select(LTY_MEDIAOBJECTS_OBJECTID_CATEGORYID_BY_TITLE, IN deviceID, "t", IN name);
         if (res) return res;
         res = query.Get("ii", OUT &objectID, OUT &objectType);
         query.End();
         if (res) return res;
        }

    } else {
        /* use the live tags table to find the tag id's */
        res = StoreLiveTagsElement(IN name, IN objectType, IN deviceID, IN FNP_PLAYABLE);
        if (res) return res;

        /* ...and get back real id from live tags table */
        res = GetLiveTagsElement(OUT objectID, IN objectType, IN name);
        if (res) return res;
    }

    VARTRACE(objectType);
    VARTRACE(objectID);
    VARTRACE(name);
    return MP_NO_ERROR;
}

tResult DBManager::UpdateInternalPlaylist(const tPath playlistPath, const tPlaylistName playlistName, const tDeviceID deviceID)
{
    ENTRY;
    tResult res = MP_NO_ERROR;
    Query query;
    int PlaylistNameCount = 0;
    Locker l_FlushLocker(&mFlushLock);
    /*Check if the playlist name exists already - if exists,throw an error*/
    if(deviceID == MUSICBOX_DEVICE_ID)
    {
        CheckifPlaylistExists(PlaylistNameCount,deviceID,playlistName);
    }

    if(PlaylistNameCount == 0)
    {
        /* update the MediaObjects table */
        res = query.Update(LTY_MEDIAOBJECTS_UPDATE);
        if (res) return res;

        /* update the title to <playlistName> for internal playlist with URL <playlistPath>*/
        res = query.Change("--itit----------------------------------it",
                IN 0, IN playlistName,
                IN 0, IN playlistName,
                IN 0, IN playlistPath); //for WHERE clause
        if (res) return res;
    }
    else
    {
        res = MP_ERR_LC_PLAYLIST_MULTIPLE_ENTRIS_FOUND;
        ETG_TRACE_ERR(("Playlist name exists already"));
    }

    return res;
}


tResult DBManager::DeleteVTBluetoothCache(const tMountPoint mountPoint)
{
    ENTRY

    // delete the cache directly till we find a way to delete cache using sql delete for virtual table (xUpdate)
    extern bool BTCacheDelete(const char *path);
    BTCacheDelete(mountPoint);

    return MP_NO_ERROR;
}


tResult DBManager::setFormerConnectionState(const tDeviceID deviceID, const tConnectionState formerConnectionState)
{
    ENTRY
    Query query;
    tResult res;
    res = query.Update(LTY_DEVICES_UPDATE);
    if (res) return res;

    VARTRACE(formerConnectionState);

    res = query.Change("----------------------ii------------------------------------------ii",  IN 0, IN formerConnectionState, IN 0, IN deviceID);

    if (res) return res;

    return MP_NO_ERROR;
}


tResult DBManager::SetConnectionState(const tDeviceID deviceID, const tConnectionState connectionState,tDisconnectReason disconnectReason /*= DR_REMOVED*/) const
{
    ENTRY
    Query query;
    tResult res;
    tDeviceInfo deviceInfo;
    LocalSPM::GetDBManager().GetDeviceInfo(OUT deviceInfo, IN deviceID);
    res = query.Update(LTY_DEVICES_UPDATE);
    if (res) return res;

    if(CS_DISCONNECTED == connectionState)
    {
        if(DTY_DVD_DRIVE == deviceInfo.deviceType)  // set former connection state also to disconnected if its DVD_DRIVE
        {
            res = query.Change("------------------ii--ii----ii--------------ii--------------------ii", IN 0, IN connectionState, IN 0, IN connectionState, IN 0, IN DS_NONE, IN 0, IN disconnectReason, IN 0, IN deviceID);
        }
        else
        {
            res = query.Change("------------------ii--------ii--------------ii--------------------ii", IN 0, IN connectionState, IN 0, IN DS_NONE, IN 0, IN disconnectReason, IN 0, IN deviceID);
        }
    }
    else if(CS_ON_HOLD == connectionState)
    {
        res = query.Change("------------------ii------------------------ii--------------------ii", IN 0, IN connectionState, IN 0, IN disconnectReason,IN 0, IN deviceID);
    }
    else
    {
        res = query.Change("------------------ii----------------------------------------------ii", IN 0, IN connectionState, IN 0, IN deviceID);
    }

    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::SetDeviceState(const tDeviceID deviceID, const tDeviceState deviceState,tDisconnectReason disconnectReason /*= DR_REMOVED*/) const
{
    ENTRY
    Query query;
    tResult res;
    res = query.Update(LTY_DEVICES_UPDATE);
    if (res) return res;

    if(DS_NONE == deviceState)
    {
        res = query.Change("----------------------------ii--------------ii--------------------ii", IN 0, IN DS_NONE, IN 0, IN disconnectReason, IN 0, IN deviceID);
    }

    if (res) return res;

    return MP_NO_ERROR;
}
tResult DBManager::GetNumberofTracks(tNumberOfMediaObjects &numberObjectsInDB, const tDeviceID deviceID)
{
    ENTRY
    Query query;
    tResult res;

    numberObjectsInDB = 0;

    /* prepare the sql statement for the metadata list */
    res = query.Select(LTY_TRACK_COUNT, IN deviceID, "");
    if (res) return res;

    /* get the count */
    res = query.Get("i", OUT &numberObjectsInDB);
    query.End();
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::GetNumberOfPlaylist(tNumberOfMediaObjects &numberObjectsInDB,const tDeviceID deviceID)
{
    ENTRY
    Query query;
    tResult res;
    numberObjectsInDB = 0;
    res = query.Select(LTY_PLAYLIST_COUNT, IN deviceID,"");
    if (res) return res;
        /* get the count */
    res = query.Get("i", OUT &numberObjectsInDB);
    query.End();
    if (res) return res;
    return MP_NO_ERROR;
}

tResult DBManager::GetNumberOfVideos(tNumberOfMediaObjects &numberObjectsInDB,const tDeviceID deviceID)
{
    ENTRY
    Query query;
    tResult res;
    numberObjectsInDB = 0;
    res = query.Select(LTY_VIDEO_TRACK_COUNT, IN deviceID,"");
    if (res) return res;
        /* get the count */
    res = query.Get("i", OUT &numberObjectsInDB);
    query.End();
    if (res) return res;
    return MP_NO_ERROR;
}

tResult DBManager::CreateTrackTempTable()
{
    ENTRY
    tResult res;
    Query query;

    /* add it to the scan context table */
    res = query.Insert(IN LTY_TRACK_TEMP_CREATE);
    if (res) return res;

    res = query.Put("");
    query.End();
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::GetNumberofTrackTempEntry(tNumberOfMediaObjects &numberObjectsInDB)
{
    Query query;
    tResult res;

    numberObjectsInDB = 0;

    /* prepare the sql statement for the metadata list */
    res = query.Select(LTY_TRACK_TEMP_COUNT, IN 1 /* dummy */, "");
    if (res) return res;

    /* get the count */
    res = query.Get("i", OUT &numberObjectsInDB);
    query.End();
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::DropTrackTempTable()
{
    ENTRY
    tResult res;
    Query query;

    /* add it to the scan context table */
    res = query.Delete(IN LTY_TRACK_TEMP_DROP);
    if (res) return res;

    res = query.Remove("");
    query.End();
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::FillTrackTempTable(const tDeviceID deviceID)
{
    ENTRY
    tResult res;
    Query query;

    /* add it to the scan context table */
    res = query.Insert(IN LTY_TRACK_TEMP_INSERT);
    if (res) return res;

    res = query.Put("i",IN deviceID);
    query.End();
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::FillTrackTempTableWithVideoUID(const tDeviceID deviceID)
{
    ENTRY
    tResult res;
    Query query;

    /* add it to the scan context table */
    res = query.Insert(IN LTY_VIDEO_TEMP_INSERT);
    if (res) return res;

    res = query.Put("i",IN deviceID);
    query.End();
    if (res) return res;

    return MP_NO_ERROR;
}


tResult DBManager::RemoveTrackEntryFromTemp(const tUUID UUid)
{
    tResult res;
    Query query;

    /* add it to the scan context table */
    res =  query.Delete(IN LTY_TRACK_TEMP_DELETE);
    if (res) return res;

    res = query.Remove("it",IN 0,IN UUid);
    query.End();
    if (res) return res;

    return MP_NO_ERROR;
}


tResult DBManager::CreatePlaylistTempTable()
{
    ENTRY
    tResult res;
    Query query;

    /* add it to the scan context table */
    res = query.Insert(IN LTY_PLAYLIST_TEMP_CREATE);
    if (res) return res;

    res = query.Put("");
    query.End();
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::FillPlaylistTempTable(const tDeviceID deviceID)
{
    ENTRY
    tResult res;
    Query query;

    /* add it to the scan context table */
    res = query.Insert(IN LTY_PLAYLIST_TEMP_INSERT);
    if (res) return res;

    res = query.Put("i",IN deviceID);
    query.End();
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::RemovePlaylistEntryFromTemp(const tPlaylistName playListName)
{
    tResult res;
    Query query;

    /* add it to the scan context table */
    res =  query.Delete(IN LTY_PLAYLIST_TEMP_DELETE);
    if (res) return res;

    res = query.Remove("it",IN 0,IN playListName);
    query.End();
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::GetNumberofPlaylistTempEntry(tNumberOfMediaObjects &numberObjectsInDB)
{
    Query query;
    tResult res;

    numberObjectsInDB = 0;

    /* prepare the sql statement for the metadata list */
    res = query.Select(LTY_PLAYLIST_TEMP_COUNT, IN 1 /* dummy */, "");
    if (res) return res;

    /* get the count */
    res = query.Get("i", OUT &numberObjectsInDB);
    query.End();
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::DropPlaylistTempTable()
{
    ENTRY
    tResult res;
    Query query;

    /* add it to the scan context table */
    res = query.Delete(IN LTY_PLAYLIST_TEMP_DROP);
    if (res) return res;

    res = query.Remove("");
    query.End();
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::GetMetatdataIDs(tMediaObject &mediaObject)
{
    ENTRY
    Query query;
    tResult res;
    VARTRACE(mediaObject.deviceID);
    VARTRACE(mediaObject.UUID);

    res = query.Select(LTY_SONGS_FILTERTAGIDS_BY_UUID, IN mediaObject.deviceID, "t", IN mediaObject.UUID);
    if (res) return res;

    res = query.Get(IN "iiii",
            OUT &mediaObject.MetadataTag1,
            OUT &mediaObject.MetadataTag2,
            OUT &mediaObject.MetadataTag3,
            OUT &mediaObject.MetadataTag4);
    query.End();
    VARTRACE(mediaObject.MetadataTag1);
    VARTRACE(mediaObject.MetadataTag2);
    VARTRACE(mediaObject.MetadataTag3);
    VARTRACE(mediaObject.MetadataTag4);
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::GetObjectUUID(tDeviceID &deviceID, tUUID &UUID, const tObjectID objectID)
{
    ENTRY
    Query query;
    tResult res;

    res = query.Select(LTY_MEDIAOBJECTS_DEVICEID_UUID_BY_OBJECTID, IN objectID, IN 0, IN 1, "");
    if (res) return res;

    res = query.Get("iT",
            OUT &deviceID,
            OUT UUID, IN sizeof(UUID));
    query.End();

    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::GetObjectIDByUUID(tObjectID &objectID, const tDeviceID deviceID, const tUUID UUID)
{
    ENTRY
    Query query;
    tResult res;
    VARTRACE(deviceID);
    VARTRACE(UUID);

    res = query.Select(LTY_SONGS_OBJECTID_BY_UUID, IN deviceID, "t", IN UUID);
    if (res) return res;

    res = query.Get("i", OUT &objectID);
    query.End();

    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::GetIPODMediaObjectByUUID(tMediaObject& mediaObject, const tDeviceID deviceID, const tUUID UUID)
{
    ENTRY
    Query query;
    tResult res;
    VARTRACE(deviceID);
    VARTRACE(UUID);

    res = query.Select(LTY_IPOD_MEDIAOBJECTS_BY_UUID, IN deviceID, "t", IN UUID);
    if (res) return res;

    res = query.Get("iiiTTTiiTiTiTiTiiiTTiilTiiiiiTTi",
             OUT &mediaObject.objectID,
             OUT &mediaObject.catType,
             OUT &mediaObject.deviceID,
             OUT mediaObject.fileName, IN sizeof(mediaObject.fileName),
             OUT mediaObject.UUID, IN sizeof(mediaObject.UUID),
             OUT mediaObject.title, IN sizeof(mediaObject.title),
             OUT &mediaObject.trackNumber,
             OUT &mediaObject.MetadataTag1,
             OUT mediaObject.MetadataField1, IN sizeof(mediaObject.MetadataField1),
             OUT &mediaObject.MetadataTag2,
             OUT mediaObject.MetadataField2, IN sizeof(mediaObject.MetadataField2),
             OUT &mediaObject.MetadataTag3,
             OUT mediaObject.MetadataField3, IN sizeof(mediaObject.MetadataField3),
             OUT &mediaObject.MetadataTag4,
             OUT mediaObject.MetadataField4, IN sizeof(mediaObject.MetadataField4),
             OUT &mediaObject.metadataConvertFlag,
             OUT &mediaObject.compilationFlag,
             OUT &mediaObject.totalPlaytime,
             OUT mediaObject.albumArtString, IN sizeof(mediaObject.albumArtString),
             OUT mediaObject.path, IN sizeof(mediaObject.path),
             OUT &mediaObject.notPlayable,
             OUT &mediaObject.mediaType,
             OUT &mediaObject.fileSize,
             OUT mediaObject.dateTime, IN sizeof(mediaObject.dateTime),
             OUT &mediaObject.fileMode,
             OUT &mediaObject.userID,
             OUT &mediaObject.groupID,
             OUT &mediaObject.year,
             OUT &mediaObject.yearID,
             OUT mediaObject.deviceVersion, IN sizeof(mediaObject.deviceVersion),
             OUT mediaObject.mountPoint, IN sizeof(mediaObject.mountPoint),
             OUT &mediaObject.deviceType);
    query.End();

    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::UpdateShuffleSupport(const tDeviceID deviceID, const tShuffleSupport isShuffleSupported)
{
    ENTRY
    Query query;
    tResult res;

    res = query.Update(LTY_DEVICES_UPDATE);
    if (res) return res;

    /* update the given device's ShuffleSupport */
    res = query.Change("----------------------------------------------ii------------------ii", IN 0, IN isShuffleSupported, IN 0, IN deviceID);
    if (res) return res;

    return MP_NO_ERROR;

}

tResult DBManager::UpdateRepeatSupport(const tDeviceID deviceID, const tRepeatSupport isRepeatSupported)
{
    ENTRY
    Query query;
    tResult res;

    res = query.Update(LTY_DEVICES_UPDATE);
    if (res) return res;

    /* set the given device's RepeatSupport */
    res = query.Change("------------------------------------------------ii----------------ii", IN 0, IN isRepeatSupported, IN 0, IN deviceID);
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::UpdateDeviceVersion(const tDeviceID deviceID,const tDeviceVersion deviceVersion)
{
    ENTRY
    Query query;
    tResult res = MP_NO_ERROR;

    res = query.Update(LTY_DEVICES_UPDATE);
    if (res) return res;

    /* set the given device's version */
    res = query.Change("it----------------------------------------------------------------ii", IN 0, IN deviceVersion, IN 0, IN deviceID); // goa5kor ( length was 23 instead of 24. Now corrected to 26 )
    return res;
}
tResult DBManager::UpdateDiscType(const tDeviceID deviceID,const tDiscType discType)
{
    ENTRY
    Query query;
    tResult res = MP_NO_ERROR;
    res = query.Update(LTY_DEVICES_UPDATE);
    if (res) return res;
    res = query.Change("--------------ii--------------------------------------------------ii", IN 0, IN discType, IN 0, IN deviceID);
    return res;
}

tResult DBManager::UpdateNowPlayingListAvailable(const tDeviceID deviceID, const tNowPlayingListAvailable isNowPlayingListAvailable)
{
    ENTRY;
    Query query;
    tResult res;

    res = query.Update(LTY_DEVICES_UPDATE);
    if(res) return res;

    /*set the NowPlayingListAvailability from Apple devices in Devices table*/
    res = query.Change("----------------------------------------------------ii------------ii", IN 0, IN isNowPlayingListAvailable, IN 0, IN deviceID);
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::UpdateDeviceUUID(const tDeviceID deviceID, const tUUID deviceUUID)
{
    ENTRY;
    Query query;
    tResult res;

    res = query.Update(LTY_DEVICES_UPDATE);
    if(res) return res;

    /* set the deviceUUID of apple device */
    res = query.Change("------------------------------------------------------it----------ii", IN 0, IN deviceUUID, IN 0, IN deviceID);
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::UpdateAppleDeviceMACAddress(const tDeviceID deviceID, const tMACAddress appleDeviceMacAddress)
{
    ENTRY;
    Query query;
    tResult res;

    res = query.Update(LTY_DEVICES_UPDATE);
    if(res) return res;

    /* set the MAC Address of apple device */
    res = query.Change("--------------------------------------------------------it--------ii", IN 0, IN appleDeviceMacAddress, IN 0, IN deviceID);
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::GetNumberofImages(tNumberOfMediaObjects &numberOfImages, const tDeviceID deviceID)
{
    ENTRY
    Query query;
    tResult res;
    VARTRACE(deviceID);

    numberOfImages = 0;

    res = query.Select(LTY_IMAGE_COUNT, IN deviceID, "");
    if (res) return res;

    res = query.Get("i", OUT &numberOfImages);
    query.End();
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::GetNumberofImagesByPath(tNumberOfMediaObjects &numberOfImages, const tDeviceID deviceID, const tPath path)
{
    ENTRY
    Query query;
    tResult res;
    VARTRACE(deviceID);
    VARTRACE(path);

    numberOfImages = 0;

    res = query.Select(LTY_IMAGE_FOLDER_ITEM_COUNT, IN deviceID, "t", IN path);
    if (res) return res;

    tCategoryType catType;
    res = query.Get("ii", OUT &catType, OUT &numberOfImages);
    query.End();
    if (res) return res;

    return MP_NO_ERROR;
}

unsigned int DBManager::GetCountFlushBorder()
{
    ENTRY_INTERNAL;
    Locker locker(&mFlushBorderMutex);
    return mObjectCountFlushBorder;
}

void DBManager::CalcCountFlushBorder(unsigned int cacheSize, unsigned long durationUS)
{
    ENTRY;
    VARTRACE(cacheSize);
    VARTRACE((int)durationUS);
    Locker locker(&mFlushBorderMutex);

    /* recalc the cache size for next stores */
    int objectCountFlushBorderOld = mObjectCountFlushBorder;
    VARTRACE(objectCountFlushBorderOld);

    int objectCountFlushBorderNew = (int)((double)mDBStoreMediaObjectMaxTime_ms / ((durationUS / 1000.) / cacheSize));
    VARTRACE(objectCountFlushBorderNew);

    /* add integral part*/
    mObjectCountFlushBorder = objectCountFlushBorderOld + ((objectCountFlushBorderNew - objectCountFlushBorderOld) / 2);
    VARTRACE(mObjectCountFlushBorder);

    /* cut to flush border upper limit */
    unsigned int maxObjectBorder = (unsigned int) LocalSPM::GetDataProvider().DBStoreMediaObjectMaxCount();
    if (mObjectCountFlushBorder > maxObjectBorder) {
        mObjectCountFlushBorder = maxObjectBorder;
    }
    VARTRACE(mObjectCountFlushBorder);
}

tResult DBManager::UpdateDBListPath(const tDeviceID deviceID ,const tListID listID, const tPath lastPlayedPath)
{

    ENTRY
    Query query;
    DBList *dbList = NULL;

    /* get list pointer */
    //dbList = LookupDBList(IN listID);
    GET_DBLIST_POINTER(dbList, listID);
    if (!dbList) return MP_ERR_DB_LIST_NOT_FOUND;

    if(dbList->mDeviceID == deviceID) {
        strncpy_r(dbList->mPath, lastPlayedPath, sizeof(dbList->mPath));
    }
    return MP_NO_ERROR;
}

tResult DBManager::CheckDatabaseIntegrity(const bool write_recreate_file)
{
    ENTRY
    if (!mDB) return MP_ERR_DB_NOT_OPEN;
    if (!mDB->Handle()) return MP_ERR_DB_INVALID_HANDLE;

    tResult res = mDB->CheckDatabaseIntegrity(mDB->Handle());

    if(write_recreate_file && (MP_NO_ERROR != res)) {
        FILE *fp;
        fp = fopen(Database::mDatabaseRecreateFile, "w");
        fclose(fp);
        ETG_TRACE_USR1(("created: %s", Database::mDatabaseRecreateFile));
    }
    return res;
}

tResult DBManager::GetVideoBrightness(OUT tVideoBrightness &videoBrightness)
{
    ENTRY
    tResult res;
    Query query;
    tDeviceInfo deviceInfo;
    tDeviceID activeDeviceID = DEVICE_ID_NOT_SET;
    GetActiveDevice(OUT activeDeviceID);
    res = GetDeviceInfo(OUT deviceInfo, IN activeDeviceID);

    if(DTY_USB == deviceInfo.deviceType)
        res = query.Select(LTY_VIDEOPROPERTY_BRIGHTNESS, MY_MEDIA, "");
    else if(DTY_DVD_DRIVE == deviceInfo.deviceType)
        res = query.Select(LTY_DVDVIDEOPROPERTY_BRIGHTNESS, MY_MEDIA, "");
    else
        res = query.Select(LTY_MTPVIDEOPROPERTY_BRIGHTNESS, MY_MEDIA, "");
    if (res) return res;

    res = query.Get("i",OUT &videoBrightness);
    query.End();

    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::GetVideoHue(OUT tVideoHue &videoHue)
{
    ENTRY
    tResult res;
    Query query;
    tDeviceInfo deviceInfo;
    tDeviceID activeDeviceID = DEVICE_ID_NOT_SET;
    GetActiveDevice(OUT activeDeviceID);
    GetDeviceInfo(OUT deviceInfo, IN activeDeviceID);

    if(DTY_USB == deviceInfo.deviceType)
        res = query.Select(LTY_VIDEOPROPERTY_HUE, MY_MEDIA, "");
    else if(DTY_DVD_DRIVE == deviceInfo.deviceType)
        res = query.Select(LTY_DVDVIDEOPROPERTY_HUE, MY_MEDIA, "");
    else
        res = query.Select(LTY_MTPVIDEOPROPERTY_HUE, MY_MEDIA, "");
    if (res) return res;

    res = query.Get("i",OUT &videoHue);
    query.End();

    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::GetVideoSaturation(OUT tVideoSaturation &videoSaturation)
{
    ENTRY
    tResult res;
    Query query;

    tDeviceInfo deviceInfo;
    tDeviceID activeDeviceID = DEVICE_ID_NOT_SET;
    GetActiveDevice(OUT activeDeviceID);
    GetDeviceInfo(OUT deviceInfo, IN activeDeviceID);

    if(DTY_USB == deviceInfo.deviceType)
        res = query.Select(LTY_VIDEOPROPERTY_SATURATION, MY_MEDIA, "");
    else if(DTY_DVD_DRIVE == deviceInfo.deviceType)
        res = query.Select(LTY_DVDVIDEOPROPERTY_SATURATION, MY_MEDIA, "");
    else
        res = query.Select(LTY_MTPVIDEOPROPERTY_SATURATION, MY_MEDIA, "");
    if (res) return res;

    res = query.Get("i",OUT &videoSaturation);
    query.End();

    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::GetVideoContrast(OUT tVideoContrast &videoContrast)
{
    ENTRY
    tResult res;
    Query query;

    tDeviceInfo deviceInfo;
    tDeviceID activeDeviceID = DEVICE_ID_NOT_SET;
    GetActiveDevice(OUT activeDeviceID);
    GetDeviceInfo(OUT deviceInfo, IN activeDeviceID);

    if(DTY_USB == deviceInfo.deviceType)
        res = query.Select(LTY_VIDEOPROPERTY_CONTRAST, MY_MEDIA, "");
    else if(DTY_DVD_DRIVE == deviceInfo.deviceType)
        res = query.Select(LTY_DVDVIDEOPROPERTY_CONTRAST, MY_MEDIA, "");
    else
        res = query.Select(LTY_MTPVIDEOPROPERTY_CONTRAST, MY_MEDIA, "");
    if (res) return res;

    res = query.Get("i",OUT &videoContrast);
    query.End();

    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::GetVideoBrightnessOffset(OUT tVideoBrightnessOffset &videoBrightnessOffset)
{
    ENTRY
    tResult res;
    Query query;

    tDeviceInfo deviceInfo;
    tDeviceID activeDeviceID = DEVICE_ID_NOT_SET;
    GetActiveDevice(OUT activeDeviceID);
    GetDeviceInfo(OUT deviceInfo, IN activeDeviceID);

    if(DTY_USB == deviceInfo.deviceType)
        res = query.Select(LTY_VIDEOPROPERTY_BRIGHTNESSOFFSET, MY_MEDIA, "");
    else if(DTY_DVD_DRIVE == deviceInfo.deviceType)
        res = query.Select(LTY_DVDVIDEOPROPERTY_BRIGHTNESSOFFSET, MY_MEDIA, "");
    else
        res = query.Select(LTY_MTPVIDEOPROPERTY_BRIGHTNESSOFFSET, MY_MEDIA, "");
    if (res) return res;

    res = query.Get("i",OUT &videoBrightnessOffset);
    query.End();

    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::GetVideoHueOffset(OUT tVideoHueOffset &videoHueOffset)
{
    ENTRY
    tResult res;
    Query query;

    tDeviceInfo deviceInfo;
    tDeviceID activeDeviceID = DEVICE_ID_NOT_SET;
    GetActiveDevice(OUT activeDeviceID);
    GetDeviceInfo(OUT deviceInfo, IN activeDeviceID);

    if(DTY_USB == deviceInfo.deviceType)
        res = query.Select(LTY_VIDEOPROPERTY_HUEOFFSET, MY_MEDIA, "");
    else if(DTY_DVD_DRIVE == deviceInfo.deviceType)
        res = query.Select(LTY_DVDVIDEOPROPERTY_HUEOFFSET, MY_MEDIA, "");
    else
        res = query.Select(LTY_MTPVIDEOPROPERTY_HUEOFFSET, MY_MEDIA, "");
    if (res) return res;

    res = query.Get("i",OUT &videoHueOffset);
    query.End();

    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::GetVideoSaturationOffset(OUT tVideoSaturationOffset &videoSaturationOffset)
{
    ENTRY
    tResult res;
    Query query;

    tDeviceInfo deviceInfo;
    tDeviceID activeDeviceID = DEVICE_ID_NOT_SET;
    GetActiveDevice(OUT activeDeviceID);
    GetDeviceInfo(OUT deviceInfo, IN activeDeviceID);

    if(DTY_USB == deviceInfo.deviceType)
        res = query.Select(LTY_VIDEOPROPERTY_SATURATIONOFFSET, MY_MEDIA, "");
    else if(DTY_DVD_DRIVE == deviceInfo.deviceType)
        res = query.Select(LTY_DVDVIDEOPROPERTY_SATURATIONOFFSET, MY_MEDIA, "");
    else
        res = query.Select(LTY_MTPVIDEOPROPERTY_SATURATIONOFFSET, MY_MEDIA, "");

    if (res) return res;

    res = query.Get("i",OUT &videoSaturationOffset);
    query.End();

    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::GetVideoProperties(OUT me::vprops_t &mEVideoProperties)
{
    ENTRY
    tResult res;
    Query query;

    tDeviceInfo deviceInfo;
    tDeviceID activeDeviceID = DEVICE_ID_NOT_SET;
    GetActiveDevice(OUT activeDeviceID);
    GetDeviceInfo(OUT deviceInfo, IN activeDeviceID);

    if(DTY_DVD_DRIVE == deviceInfo.deviceType)
    {
        res = query.Select(LTY_DVDVIDEOPROPERTY, MY_MEDIA, "");
        if (res) return res;
    }
    else if(DTY_USB == deviceInfo.deviceType)
    {
        res = query.Select(LTY_VIDEOPROPERTY, MY_MEDIA, "");
        if (res) return res;
    }
    else
    {
        res = query.Select(LTY_MTPVIDEOPROPERTY, MY_MEDIA, "");
        if (res) return res;
    }
    tU32        brightness;
    tS32              hue;
    tU32        saturation;
    tU32          contrast;
    tS32 brightnessoffset;
    tS32 saturationoffset;
    tU32         hueoffset;

    res = query.Get("iiiiiii",OUT &brightness, &hue, &saturation, &contrast, &brightnessoffset, &saturationoffset, &hueoffset);
    query.End();

    mEVideoProperties.brightness = brightness;
    mEVideoProperties.hue = hue;
    mEVideoProperties.saturation = saturation;
    mEVideoProperties.contrast = contrast;
    mEVideoProperties.brightnessoffset = brightnessoffset;
    mEVideoProperties.saturationoffset = saturationoffset;
    mEVideoProperties.hueoffset = hueoffset;

    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::UpdateVideoProperty (IN const tVideoProperty videoProperty,IN const tPropertyValue propertyValue)
{
    ENTRY
    Query query;
    tResult res;
    tDeviceInfo deviceInfo;
    tDeviceID activeDeviceID = DEVICE_ID_NOT_SET;
    GetActiveDevice(OUT activeDeviceID);
    GetDeviceInfo(OUT deviceInfo, IN activeDeviceID);

    if(DTY_DVD_DRIVE == deviceInfo.deviceType)
    {
        res = query.Update(LTY_DVDVIDEOPROPERTY_UPDATE);
    }
    else if(DTY_USB == deviceInfo.deviceType)
    {
        res = query.Update(LTY_VIDEOPROPERTY_UPDATE);
    }
    else
    {
        res = query.Update(LTY_MTPVIDEOPROPERTY_UPDATE);
    }
    if (res) return res;


    switch(videoProperty)
    {
        case VP_BRIGHTNESS:
        {
            /* set the video brightness */
            res = query.Change("ii------------",IN 0, IN propertyValue);
            break;
        }
        case VP_HUE:
        {
            /* set the video hue */
             res = query.Change("--ii----------",IN 0, IN propertyValue);
            break;
        }
        case VP_SATURATION:
        {
            /* set the video saturation */
            res = query.Change("----ii--------",IN 0, IN propertyValue);
            break;
        }
        case VP_CONTRAST:
        {
            /* set the video contrast */
            res = query.Change("------ii------",IN 0, IN propertyValue);
            break;
        }
        case VP_BRIGHTNESS_OFFSET:
        {
            /* set the video brightness offset */
            res = query.Change("--------ii----",IN 0, IN propertyValue);
            break;
        }
        case VP_HUE_OFFSET:
        {
            /* set the video hue offset */
            res = query.Change("----------ii--",IN 0, IN propertyValue);
            break;
        }
        case VP_SATURATION_OFFSET:
        {
            /* set the video saturation offset */
            res = query.Change("------------ii",IN 0, IN propertyValue);
            break;
        }
        default:
        {
            ETG_TRACE_ERR(("Invalid video property type: %u",videoProperty));
            break;
        }
    }

    if (res) return res;

    return MP_NO_ERROR;
}
//>--Roadmap 16003 : 'CD Ripping With Gracenote'
tResult DBManager::UpdateRippedFileCategory(tDeviceID deviceID, tRippedFilePath filepath, tCategoryType CategoryType)
{
    ENTRY
    Query query;
    tResult res;

    res = query.Update(LTY_MEDIAOBJECTS_UPDATE_RIPPED_FILE);
    if (res) return res;

    res = query.Change("iti",IN CategoryType, IN filepath, IN deviceID);

    return MP_NO_ERROR;
}

tResult DBManager::UpdateAlbumArtStringForTocHash(tDeviceID deviceID, tAlbumArt albumArtString, const tUUID UUID)
{
    ENTRY
    VARTRACE(albumArtString);
    VARTRACE(UUID);
    Query query;
    tResult res;

    res = query.Update(LTY_MEDIAOBJECTS_UPDATE_ALBUMARTSTRING);
    if (res) return res;

    res = query.Change("tti",IN albumArtString, IN UUID, IN deviceID);
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::GetAlbumArtStringForTocHash(const tUUID UUID, std::string &AlbumArtString)
{
    ENTRY
    tResult res = MP_NO_ERROR;
    Query query;
    char albumArt[1024] ={0};
    res = query.Select(LTY_MEDIAOBJECTS_ALBUMART_FROM_UUID, IN MUSICBOX_DEVICE_ID, "t", IN UUID);
    VARTRACE(res);
    if(!res)
    {
        res = query.Get("t",
                OUT albumArt);
        query.End();
        AlbumArtString.clear();
        AlbumArtString.assign(albumArt);
        if(!res)
        {
            ETG_TRACE_USR4(("AlbumArtString: %s" ,AlbumArtString.c_str()));
        }
        else
        {
            ETG_TRACE_ERR(("GetAlbumArtStringForTocHash: error: %d/%s", res, strerror(res)));
        }
    }
    return res;
}
tResult DBManager::UpdateMusicBoxDevice(tDeviceID deviceID, tNumberOfFiles numberOfAudioFiles, tIndexingState indexingState, tUInt RippingPercentage, tUInt TotalMemory, tUInt FreeMemory)
{
    ENTRY
    (void)indexingState;
    (void)RippingPercentage;

    Query query;
    tResult res;

    res = query.Update(LTY_MEDIAOBJECTS_UPDATE_MUSICBOX_DEVICE);
    if (res) return res;

    /* set the given devices indexing state */
    res = query.Change("iiii",IN numberOfAudioFiles, IN TotalMemory, IN FreeMemory, IN deviceID);
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::DeleteMediaObject(IN const tDeviceID deviceID,IN const std::string mediObjectUrl)
{
    ENTRY
    Query query;
    tResult res;
    Locker l_FlushLocker(&mFlushLock);
    /* query to delete an entry from the mediaobject table */
    res = query.Delete(LTY_MEDIAOBJECTS_DELETE);
    if (res) return res;

    /* delete the internal playlist entry in the mediaobjects table with URL <playlistPath>*/
    res = query.Remove("it", IN deviceID, IN mediObjectUrl.c_str());
    if (res) return res;

    return MP_NO_ERROR;

}

tResult DBManager::GetURLOfMediaObject(IN OUT tURL &url, IN const tObjectID objectID, IN const tDeviceID deviceID) const
{
    ENTRY;
    tResult res;
    Query query;

    res = query.Select(LTY_MEDIAOBJECTS_URL, deviceID, "ii", IN 0, IN objectID);

    if (res) return res;

    res = query.Get("T", OUT url, IN sizeof(url));
    query.End();
    if (res) return res;
    else
    {
        ETG_TRACE_USR4(("URL : %s", url));
    }

    return MP_NO_ERROR;
}


tResult DBManager::UpdateMediaObject(const IN tEditMetaDataByUrl &editMetaDataByUrl)
{
    ENTRY;
    tResult res;
    Query query;
    Locker l_FlushLocker(&mFlushLock);

    /* update the MediaObjects table */
    res = query.Update(LTY_MEDIAOBJECTS_UPDATE);
    if (res) return res;

    /* find match for mediaObjectID and update notPlayable flag */
    res = query.Change("--ititititit------------------------iiiiit",
            IN 0, IN editMetaDataByUrl.TitleField,
            IN 0, IN editMetaDataByUrl.GenreField,
            IN 0, IN editMetaDataByUrl.ArtistField,
            IN 0, IN editMetaDataByUrl.ComposerField,
            IN 0, IN editMetaDataByUrl.AlbumField,
            IN 0, IN editMetaDataByUrl.YearField,
            IN 0, IN editMetaDataByUrl.deviceID,//for WHERE clause
            IN 0, IN editMetaDataByUrl.MediaObjectUrl);//for WHERE clause
    if (res) return res;
    return MP_NO_ERROR;
}

tResult DBManager::UpdateFormerConnectionState(tDeviceType deviceType,tConnectionState formerConnectionState)
{
    ENTRY
    tResult res = MP_NO_ERROR;
    tDeviceID deviceID = DEVICE_ID_NOT_SET;
    Query query;

    // find the recently ejected Disc
    res = query.Select(LTY_LATEST_DISC_EJECTED, MY_MEDIA, "ii",IN 0,IN deviceType);

    if(MP_NO_ERROR == res)
    {
        res = query.Get(IN "i", OUT &deviceID);

        if (res == MP_NO_ERROR)
        {
            res = query.Update(LTY_DEVICES_UPDATE);

            if(MP_NO_ERROR == res)
            {
                VARTRACE(deviceID);
                res = query.Change("----------------------ii------------------------------------------ii", IN 0, IN formerConnectionState, IN 0, IN deviceID);
            }
        }
    }
    return res;
}

tResult DBManager::MarkDeviceCommunicationError(const tDeviceID deviceID,const tConnectionCount connectionCount,const tDeviceType deviceType)
{
    ENTRY
    Query query;
    tResult res = MP_ERR_DB_UNEXPECTED;

    if(IsMassStorageDevice(IN deviceType))
    {
        res = query.Update(LTY_DEVICES_UPDATE_COMMUNICATION_ERROR);
        if(MP_NO_ERROR == res)
        {
            res = query.Change("ii",IN deviceID,IN connectionCount);
        }
    }
    return res;
}

tResult DBManager::DisconnectDeviceOnCommunicationError(const tDeviceID deviceID /*,const tConnectionCount connectionCount */)
{
    ENTRY
    tResult res = MP_ERR_DB_UNEXPECTED;

    Query query;
    res = query.Update(LTY_DEVICES_UPDATE_DEVICE_DISCONNECTED_IN_COMMUNICATION_ERROR);

    if(MP_NO_ERROR == res)
    {
        res = query.Change("i",IN deviceID);
    }

    return res;
}

//>--Roadmap CMG3G-10221 : 'Scene Recorder '
tResult DBManager::UpdateMediaObject(IN const tDeviceID deviceID, IN const tFilePermission filePermission, const char *path)
{
    ENTRY;
    tResult res = MP_NO_ERROR;
    Query query;
    tFileMode fileMode;
    struct stat buf;

    Locker l_FlushLocker(&mFlushLock);
    if(path && !access(path, F_OK))
    {
        res = stat(path, &buf);
    }
    if(!res)
    {
        fileMode = buf.st_mode;
        ETG_TRACE_USR4(("fileMode = %d", fileMode));

    /* update the MediaObjects table */
    res = query.Update(LTY_MEDIAOBJECTS_UPDATE);
    if (res) return res;

    /* find match for tFilePermission and update file mode flag */
    res = query.Change("------------------------------ii------iiitii",
            IN 0, fileMode,
            IN 0, IN deviceID,   //for WHERE clause
            IN 0, IN filePermission.url,   //for WHERE clause
            IN 0, IN filePermission.objectID);     //for WHERE clause
    }
    return res;
}
tResult DBManager::GetFolderSizeInfo(IN const tDeviceID deviceID, IN const tPath Path,
        IN const tCategoryType type, OUT tFolderInfo& info)
{
    ENTRY;
    VARTRACE(deviceID)
    VARTRACE(Path)
    VARTRACE(type)
    tResult res = MP_NO_ERROR;
    tResult res1 = MP_NO_ERROR;
    Query query;
    tU64 TotalSize = 0;
    tU32 TotalNumber = 0;
    tU32 fileMode = 0;
    tU64 size = 0;
    tU64 ProtectedSize = 0;
    tU32 Protectedcount = 0;

    /* prepare the sql statement for the size and coult for total list */
    if(CTY_NONE==type) {
        res = query.Select(LTY_TRACK_COUNT_SIZE, IN deviceID, "t", IN Path);
    }
    else {
        res = query.Select(LTY_TRACK_COUNT_SIZE, IN deviceID, "ti", IN Path, IN type);
    }
    if (res) return res;

    /* get the count */
    res = query.Get("ii", OUT &TotalNumber, OUT &TotalSize);
    if (res) return res;
    query.End();
    ETG_TRACE_USR4(("GetFolderSizeInfo TotalNumber %d TotalSize %ld",TotalNumber, TotalSize));
    info.totalNumber = TotalNumber;
    info.totalSize = TotalSize/1024;

    /*Protected file size and number
     prepare the sql statement for the size and count for total list */
    if(CTY_NONE==type) {
        res = query.Select(LTY_TRACK_COUNT_SIZE_FILEMODE, IN deviceID, "t", IN Path);
    }
    else{
        res = query.Select(LTY_TRACK_COUNT_SIZE_FILEMODE, IN deviceID, "ti", IN Path, IN type);
    }
    if (res) return res;
    while(1)
    {
        /* get the count */
        res1 = query.Get("ii", OUT &fileMode, OUT &size);
        if(res1) break;

        if((fileMode&S_IRUSR) && (!(fileMode&S_IWUSR)))
        {
            ProtectedSize= ProtectedSize + (size/1024);
            Protectedcount++;
            ETG_TRACE_USR4(("fileMode = %d , Size = %d count = %d", fileMode,size,Protectedcount));
        }
    }
    query.End();
    info.protectedNumber = Protectedcount;
    info.protectedSize = ProtectedSize;
    ETG_TRACE_USR4(("GetFolderSizeInfo Protectedcount %d ProtectedSize %ld",Protectedcount, ProtectedSize));
    return res;
}
tResult DBManager::GetDirInfoFromPath(IN const tDeviceID deviceID, IN const string dirPath, OUT vector<tDirInfo> &dirInfo)
{
    ENTRY
    VARTRACE(deviceID)
    VARTRACE(dirPath.c_str())
    tResult res = MP_NO_ERROR;
    tResult res1;
    tDirInfo lDirInfo;
    Query query;
    tURL url;

    res = query.Select(LTY_DIRECTORY_INFO_FROM_FOLDERPATH, IN deviceID, "t", IN dirPath.c_str());
    if (res) return res;
    while(1)
    {
        /* get the count */
        res1 = query.Get("iit", OUT &lDirInfo.Id,OUT &lDirInfo.Category,OUT url);
        if(res1) break;
        lDirInfo.URL = url;
        dirInfo.push_back(lDirInfo);
    }
    return res;
}
tResult DBManager::GetFileMode(IN const tDeviceID deviceID, IN const tObjectID obj, OUT tFileMode &FileMode, OUT tUserID &uId, OUT tGroupID &gId)
{
    ENTRY
    VARTRACE(deviceID)
    VARTRACE(obj)
    tResult res = MP_NO_ERROR;
    Query query;

    res = query.Select(LTY_FILE_PERMISSION, IN deviceID, "i", IN obj);
    if (res) return res;

    res = query.Get("iii", OUT &FileMode, OUT &uId,OUT &gId);
    VARTRACE(FileMode)
    VARTRACE(uId)
    VARTRACE(gId)
    return res;
}
tResult DBManager::GetFileLengthDateSize(IN const tDeviceID deviceID, IN const tObjectID obj, OUT tPlaytime &TotalPlaytime, OUT tDateTime &DateTime, OUT tFileSize &size)
{
    ENTRY
    VARTRACE(deviceID)
    VARTRACE(obj)
    tResult res = MP_NO_ERROR;
    Query query;

    res = query.Select(LTY_DATETIME_SIZE_LENGTH, IN deviceID, "i", IN obj);
    if (res) return res;

    res = query.Get("ilt", OUT &TotalPlaytime, OUT &size, OUT DateTime,IN sizeof(DateTime));
    VARTRACE(TotalPlaytime)
    VARTRACE(DateTime)
    VARTRACE(size)
    return res;
}
//<--Roadmap CMG3G-10221 : 'Scene Recorder '

tResult DBManager::UpdateAppleDeviceMACAddressAndUSBSerialNumber(IN const tDeviceID deviceID, IN const tMACAddress appleDeviceMACAddress, IN const tUSBSerial appleDeviceUSBSerialNumber)
{
    ENTRY
    tResult res;
    Query query;
    res = query.Update(LTY_DEVICES_UPDATE);
    if (res) return res;
    res = query.Change("--------------------------------------------------------itit------ii",
            IN 0, IN appleDeviceMACAddress,
            IN 0, IN appleDeviceUSBSerialNumber,
            IN 0, IN deviceID);
    if (res) return res;
    return MP_NO_ERROR;
}

tResult DBManager::SetMusicBoxAutoRipping(const tBoolean autoRipping)
{
    ENTRY
    tResult res;
    Query query;

    res = query.Update(IN LTY_MUSICBOX_PROP_UPDATE_AUTO_RIPPING);
    if (res) return res;

    res = query.Change("ii",
            IN 0,IN autoRipping);
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::SetMusicBoxEncodingQuality(const tU32 encodingQuality)
{
    ENTRY
    tResult res;
    Query query;

    res = query.Update(IN LTY_MUSICBOX_PROP_UPDATE_ENC_QUALITY);
    if (res) return res;

    res = query.Change("ii",
            IN 0,IN encodingQuality);
    if (res) return res;

    return MP_NO_ERROR;
}
tResult DBManager::GetMusicBoxAutoRipping(tBoolean &autoRipping)
{
    ENTRY
    tResult res;
    Query query;

    res = query.Select(IN LTY_MUSICBOX_PROP_GET_AUTO_RIPPING, MY_MEDIA, "");
    if (res) return res;

    res = query.Get("i",
            OUT &autoRipping);
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::GetMusicBoxEncodingQuality(tU32 &encodingQuality)
{
    ENTRY
    tResult res;
    Query query;

    res = query.Select(IN LTY_MUSICBOX_PROP_GET_ENC_QUALITY, MY_MEDIA, "");
    if (res) return res;

    res = query.Get("i",
            OUT &encodingQuality);
    if (res) return res;

    return MP_NO_ERROR;
}
// Fix for Jira PSARCC30-4136: Below two functions added for Media content check.
tResult DBManager::GetFolderMediaContent(Query &queryMain, tMediaContent &mediaContent,const tURL &url,IN const tDeviceID deviceID)
{
    ENTRY;
    tResult res;
    tMediaContent localMediaContent = MTY_UNKNOWN;

    mediaContent = MTY_UNKNOWN;
    res = queryMain.Select(LTY_MEDIAOBJECTS_GET_MEDIACONTENT, deviceID, "it", IN 0, IN url);
    if(res)
    {
        ETG_TRACE_ERR(("GetFolderMediaContent:Select(LTY_MEDIAOBJECTS_GET_MEDIACONTENT)  Error=%d/%s", res, errorString(res)));
        return res;
    }
    else
    {
        res = queryMain.Get("i", OUT &localMediaContent);
        if(res)
        {
            ETG_TRACE_ERR(("GetFolderMediaContent:Get(LTY_MEDIAOBJECTS_GET_MEDIACONTENT)  Error=%d/%s", res, errorString(res)));
            return res;
        }
        mediaContent = localMediaContent;
    }

    return MP_NO_ERROR;
}

tResult DBManager::UpdateFolderMediaContent(Query &queryMain, const tMediaObject mediaObject)
{
    ENTRY;

    tMediaObject dirMediaObject = mediaObject;
    tResult res = MP_ERR_DB_END_OF_LIST;
    tMediaContent mediaContent = MTY_UNKNOWN;

    GetFolderMediaContent(INOUT queryMain, OUT mediaContent, IN dirMediaObject.fileName, IN dirMediaObject.deviceID);

    if(!(mediaContent & dirMediaObject.mediaType))
    {
        res = queryMain.Update(LTY_MEDIAOBJECTS_UPDATE_MEDIACONTENT);
        if(res)
        {
            ETG_TRACE_ERR(("StoreDirectoriesInternal:Prepare(LTY_MEDIAOBJECTS_UPDATE_MEDIACONTENT)  Error=%d/%s", res, errorString(res)));
            return res;
        }
        else
        {
            res = queryMain.Change("iit",dirMediaObject.mediaType,IN dirMediaObject.deviceID, IN dirMediaObject.fileName);
            if(res)
            {
                ETG_TRACE_ERR(("StoreDirectoriesInternal:update(LTY_MEDIAOBJECTS_UPDATE_MEDIACONTENT)  Error=%d/%s", res, errorString(res)));
                return res;
            }
        }
    }

    return MP_NO_ERROR;
}
//<--Roadmap 17001 : 'Personalization'
tResult DBManager::StorePersonalizedFavorite(tFavoriteID &retFavID, const tObjectID objectID, const tCategoryType objectType, const tLabelText labelText, const tDescriptorText descriptorText, const tUserID userID )
{
    ENTRY
    tResult res;
    Query query;

    tBoolean active = false;
    tBoolean available = true;
    tUserID tempuserID;
    tObjectID tempobjectID;
    res = query.Select(LTY_USER_FAVORITES, MY_MEDIA , "");
    if (res) return res;

    while(1) // iterate till last entry matching the given objectID
    {
       res = query.Get("-i-----i",OUT &tempobjectID, OUT &tempuserID);
       ETG_TRACE_USR2(("StorePersonalizedFavorite: tempobjectID=%d", tempobjectID));
       ETG_TRACE_USR2(("StorePersonalizedFavorite: tempuserID=%d", tempuserID));
       if((objectID == tempobjectID) && (userID == tempuserID))
       {
           ETG_TRACE_USR2(("StorePersonalizedFavorite: tempobjectID=%d already present", tempobjectID));
           break;
       }
       else if(res == MP_ERR_DB_END_OF_LIST)
       {
           /* add the favorite entry into favorite's table */
           res = query.Insert(LTY_USER_FAVORITES_INSERT);
           if (res) return res;

           res = query.Put("iittiii",
                   IN objectID,
                   IN objectType,
                   IN labelText,
                   IN descriptorText,
                   IN active,
                   IN available,
                   IN userID);
           ETG_TRACE_USR3(("StorePersonalizedFavorite  Error=%d/%s", res, errorString(res)));
           if (res) return res;

           ETG_TRACE_USR2(("StorePersonalizedFavorite: tempobjectID=%d inserted", tempobjectID));
           break;
       }
       else if(res) // if some other error than MP_NO_ERROR and MP_ERR_DB_END_OF_LIST
       {
           ETG_TRACE_ERR(("StorePersonalizedFavorite  Error=%d/%s", res, errorString(res)));
           return res;
       }
     }


    /*Get the favoriteID generated for the last entry */
    tFavoriteID favoriteID;
    res = query.Select(LTY_USER_FAVORITES, MY_MEDIA , "iiii", IN 0, IN objectID, IN 0, IN userID);
    if (res) return res;

    while(1) // iterate till last entry matching the given objectID
    {
        res = query.Get("i",OUT &favoriteID);
        if(res == MP_ERR_DB_END_OF_LIST)
        {
            retFavID = favoriteID;
            break;
        }
        if ( res ) return res;
    }

    query.End();

    return MP_NO_ERROR;
}

tResult DBManager::DeletePersonalizedFavorite(IN const tObjectID objectID, IN const tUserID userID)
{
    ENTRY
    tResult res;
    Query query;

    ETG_TRACE_USR3(("DeletePersonalizedFavorite  objectID : %d , userID : %d",objectID,userID));

    res = query.Delete(LTY_USER_FAVORITES_DELETE);
    if (res) return res;

    res = query.Remove("iiii", IN 0, IN objectID, IN 0, userID);
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::DeleteAllUserFavorites(const tUserID userID)
{
    ENTRY
    tResult res;
    Query query;

    ETG_TRACE_USR3(("DBManager::DeleteAllUserFavorites userID : %d",userID));
    res = query.Delete(LTY_USER_FAVORITES_DELETE);
    if (res) return res;

    res = query.Remove("--ii",IN 0, userID);
    if (res) return res;


    return MP_NO_ERROR;
}

tResult DBManager::ResetPersonalization()
{
    ENTRY
    tResult res;
    Query query;

    ETG_TRACE_USR3(("ResetPersonalization"));

    res = query.Delete(LTY_USER_FAVORITES_DELETE);
    if (res) return res;

    res = query.Remove("");
    if (res) return res;

    //Delete users as well
/*  //Not needed as of now
    res = query.Delete(LTY_USERS_DELETE);
    if (res) return res;

    res = query.Remove("");
    if (res) return res;
*/
    return MP_NO_ERROR;
}

tIsFavorite DBManager::IsUserFavoriteObject(const tObjectID objectID)
{
    ETG_TRACE_USR3(("DBManager::IsUserFavoriteObject"));
    tUserID userID;
    tResult res;
    tFavoriteID FavID;
    res = GetCurrentUser(userID);
    ETG_TRACE_USR3(("DBManager::IsUserFavoriteObject userID : %d, objectID : %d",userID,objectID));

    Query query;

    /*Get the favoriteID generated for the last entry */
    tFavoriteID favoriteID;
    res = query.Select(LTY_USER_FAVORITES, MY_MEDIA , "iiii", IN 0, IN objectID, IN 0, IN userID);
    if (res) return res;


    res = query.Get("i",OUT &favoriteID);
    if(res == MP_ERR_DB_END_OF_LIST)
    {
        ETG_TRACE_USR3(("DBManager::IsUserFavoriteObject objectID = %d is NOT a favorite",objectID));
        return false;
    }
    if ( res ) return res;

    query.End();

    ETG_TRACE_USR3(("DBManager::IsUserFavoriteObject objectID = %d is a favorite",objectID));
    return true;
}


///For User Table handling \\\\

tResult DBManager::SetCurrentUser(tUserID userID)
{
    ENTRY
    tResult res;
    Query query;
    tUserID tempuserID;
    tActive active;

    /* set all users to be not active */
    res = query.Update(LTY_USERS_UPDATE);
    if (res) return res;

    res = query.Change("ii", IN 0, IN false);
    if (res) return res;

    res = query.Select(LTY_USERS, MY_MEDIA,"");
    if (res) return res;

    while(1) // iterate till last entry matching the given objectID
    {
        res = query.Get("i",OUT &tempuserID);
        if(userID == tempuserID)
        {
            /* set all users to be not active */
            res = query.Update(LTY_USERS_UPDATE);
            if (res) return res;

            /* set the given user as active */
            res = query.Change("iiii", IN 0, IN true, IN 0, IN userID);
            ETG_TRACE_USR2(("SetCurrentUser: userID=%d is now Updated and set as current user", userID));
            return MP_NO_ERROR;
         }

        if(res == MP_ERR_DB_END_OF_LIST)
        {
            /* add the favorite entry into favorite's table */
            res = query.Insert(LTY_USERS_INSERT);
            if (res) return res;

            res = query.Put("ii",
                    IN userID,
                    IN true);
            ETG_TRACE_USR2(("SetCurrentUser: userID=%d is now Inserted and set as current user", userID));
            return MP_NO_ERROR;
        }
        else if(res) // if some other error than MP_NO_ERROR and MP_ERR_DB_END_OF_LIST
        {
            ETG_TRACE_ERR(("SetCurrentUser  Error=%d/%s", res, errorString(res)));
            return res;
        }


    }

    return MP_NO_ERROR;
}

tResult DBManager::GetCurrentUser(tUserID &userID)
{
    ENTRY
    Query query;
    tResult res;
    tActive active=true;
    tUserID tempuserID;
    res = query.Select(LTY_USERS, MY_MEDIA,"ii",IN 0, IN active);
    ETG_TRACE_USR2(("GetCurrentUser  Error=%d/%s", res, errorString(res)));
    if (res) return res;

    res = query.Get("i",OUT &tempuserID);
    if ( res == MP_NO_ERROR )
    {
        userID = tempuserID;
        ETG_TRACE_USR2(("DBManager::GetCurrentUser userID : %d fetched",userID));
    }
    else
    {
        ETG_TRACE_USR2(("GetCurrentUser  Error=%d/%s", res, errorString(res)));
        return res;
    }
    query.End();

    return MP_NO_ERROR;

}

//<--Roadmap 17001 : 'Personalization'


tResult DBManager::GetTrackNumbersByUUID(vector<tU16>& trackNumbers, const tDeviceID deviceID, const tUUID UUID)
{
    ENTRY
    Query query;
    tResult res;
    VARTRACE(deviceID);
    VARTRACE(UUID);

    res = query.Select(LTY_MEDIAOBJECTS_BY_UUID, IN deviceID, "t", IN UUID);
    if (res) return res;

    tU16 tracknum = 0;
    while(1)

    {
    res = query.Get("------i-----------------------",
             OUT &tracknum);
    if (res)
        {
        break;
        }
    trackNumbers.push_back (tracknum);
    VARTRACE(tracknum);

    }
    query.End();

    return MP_NO_ERROR;
}

tResult DBManager::AddYomiMetadata(tObjectID MediaObjectTagYomiData,tYomiMetadata &yomiMetadata)
{
    ENTRY
    tResult res= MP_NO_ERROR;
    Query query;

    res = query.Insert(LTY_YOMIMETADATA_INSERT);
    if (!res)
    {
        res = query.Put(IN "ittt",
                IN MediaObjectTagYomiData,
                IN yomiMetadata.YomiTitle,
                IN yomiMetadata.YomiArtist,
                IN yomiMetadata.YomiAlbum);

        if (!res)
        {
            ETG_TRACE_USR4(("MediaObjectTag: %d" ,MediaObjectTagYomiData));
            ETG_TRACE_USR4(("YomiTitle: %s" ,yomiMetadata.YomiTitle));
            ETG_TRACE_USR4(("YomiArtist: %s" ,yomiMetadata.YomiArtist));
            ETG_TRACE_USR4(("YomiAlbum: %s" ,yomiMetadata.YomiAlbum));
        }
        query.End();
    }
    return res;
}

tResult DBManager::GetYomiMetadata(tObjectID MediaObjectTagYomiData, tYomiMetadata &yomiMetadata)
{
    ENTRY
    tResult res = MP_NO_ERROR;
    Query query;

    res = query.Select(LTY_YOMIMETADATA, MUSICBOX_DEVICE_ID, "i", IN MediaObjectTagYomiData);
    if(!res)
    {
        res = query.Get("ttt",
                OUT yomiMetadata.YomiTitle,
                OUT yomiMetadata.YomiArtist,
                OUT yomiMetadata.YomiAlbum);

        query.End();
        if(!res)
        {
            ETG_TRACE_USR4(("YomiTitle: %s" ,yomiMetadata.YomiTitle));
            ETG_TRACE_USR4(("YomiArtist: %s" ,yomiMetadata.YomiArtist));
            ETG_TRACE_USR4(("YomiAlbum: %s" ,yomiMetadata.YomiAlbum));
        }
    }
    return res;
}

tResult DBManager::RemoveYomiMetadata(tObjectID MediaObjectTagYomiData)
{
    ENTRY
    tResult res = MP_NO_ERROR;
    Query query;

    res = query.Delete(LTY_YOMIMETADATA_DELETE);
    if (!res)
    {
        res = query.Remove("ii", IN 0, IN MediaObjectTagYomiData);
        if(!res)
        {
            ETG_TRACE_USR4(("Values removed from the table YomiMetadata for MediaObjectTag:%d",MediaObjectTagYomiData));
        }
    }
    return res;
}

tResult DBManager::UpdateYomiMetadata(tObjectID MediaObjectTagYomiData, tYomiMetadata &yomiMetadata)
{
    ENTRY
    tResult res= MP_NO_ERROR;
    Query query;
    tDeviceID deviceID = MUSICBOX_DEVICE_ID;

    /* update the Yomimetadata table with edited yomi metadata */
    res = query.Update(LTY_YOMIMETADATA_UPDATE);

    if (res)
    {
        ETG_TRACE_USR4(("UpdateYomiMetadata: error: %d/%s", res, strerror(res)));
        return res;
    }

    /*Update the edited field alone*/

    if(strlen_r(yomiMetadata.YomiTitle) > 0)
    {
        res = query.Change("it----iii",
                IN 0, IN yomiMetadata.YomiTitle,
                IN 0, IN deviceID,            //for WHERE clause
                IN MediaObjectTagYomiData);
    }

    if(strlen_r(yomiMetadata.YomiArtist) > 0)
    {
        res = query.Change("--it--iii",
                IN 0, IN yomiMetadata.YomiArtist,
                IN 0, IN deviceID,            //for WHERE clause
                IN MediaObjectTagYomiData);
    }

    if(strlen_r(yomiMetadata.YomiAlbum) > 0)
    {
        res = query.Change("----itiii",
                IN 0, IN yomiMetadata.YomiAlbum,
                IN 0, IN deviceID,            //for WHERE clause
                IN MediaObjectTagYomiData);
    }


    if (res)
    {
        ETG_TRACE_USR4(("UpdateYomiMetadata: error: %d", res));
        return res;
    }

    return res;
}

tResult DBManager::AddMusicBoxToC(char* ToCString, tUUID uuid)
{
    ENTRY
    tResult res= MP_NO_ERROR;
    Query query;

    res = query.Insert(LTY_MUSICBOXTOC_INSERT);
    if (!res)
    {
        res = query.Put(IN "tt",
                IN ToCString,
                IN uuid);

        if (!res)
        {
            ETG_TRACE_USR4(("TocString: %s" ,ToCString));
            ETG_TRACE_USR4(("UUID: %s" ,uuid));
        }
        query.End();
    }
    return res;
}

tResult DBManager::GetMusicBoxToC(tUUID uuid,std::string &ToCString)
{
    ENTRY
    tResult res = MP_NO_ERROR;
    Query query;
    char tocString[1024] ={0};
    VARTRACE(uuid);

    res = query.Select(LTY_MUSICBOXTOC, IN MUSICBOX_DEVICE_ID, "t", IN uuid);
    VARTRACE(res);
    if(!res)
    {
        res = query.Get("t",
                OUT tocString);

        query.End();
        ToCString.clear();
        ToCString.assign(tocString);
        if(!res)
        {
            ETG_TRACE_USR4(("ToCString RAW: %s" ,tocString));
            ETG_TRACE_USR4(("ToCString: %s" ,ToCString.c_str()));
        }
        else
        {
            ETG_TRACE_ERR(("GetMusicBoxToC: error: %d/%s", res, strerror(res)));
        }
    }
    return res;
}

tResult DBManager::RemoveMusicBoxToC(tMediaObject &mediaObject)
{
    ENTRY
    tResult res = MP_NO_ERROR;
    Query query;

    res = query.Delete(LTY_MUSICBOXTOC_DELETE);
    if (!res)
    {
        res = query.Remove("ii", IN 0, IN mediaObject.UUID);
        if(!res)
        {
            ETG_TRACE_USR4(("Values removed from the table MusicBoxToC for UUID:%s",mediaObject.UUID));
        }
    }
    return res;
}

tResult DBManager::GetUUIDfromAlbumName(const tMetadata albumName,tUUID uuid )
{
    ENTRY
    tResult res = MP_NO_ERROR;
    Query query;

    res = query.Select(LTY_SONGS_ALBUMNAME_BY_UUID, IN MUSICBOX_DEVICE_ID, "t", IN albumName);
    VARTRACE(res);
    VARTRACE(albumName);
    if(!res)
    {
        res = query.Get("t",
                OUT uuid);
        query.End();

        if(!res)
        {
            ETG_TRACE_USR4(("UUID: %s" ,uuid));
        }
        else
        {
            ETG_TRACE_ERR(("GetUUIDfromAlbumName: error: %d/%s", res, strerror(res)));
        }
    }
    return res;
}

tResult DBManager::UpdateGNmetadatainMediaObject(const tRippedTrackInfo &rippedTrackData, IN const tDeviceID deviceID, IN tUUID UUID, IN tTrackNumber trackNumber)
{
    ENTRY;
    tResult res = MP_NO_ERROR;
    Query query;

    /* update the MediaObjects table */
    res = query.Update(LTY_MEDIAOBJECTS_UPDATE_GNDATA);
    VARTRACE(res);
    if (res) return res;

    res = query.Change("itititititiiiiitii",
            IN 0, IN rippedTrackData.title,
            IN 0, IN rippedTrackData.MetadataField1,
            IN 0, IN rippedTrackData.MetadataField2,
            IN 0, IN rippedTrackData.MetadataField3,
            IN 0, IN rippedTrackData.MetadataField4,
            IN 0, IN rippedTrackData.CompilationFlag,
            IN 0, IN deviceID,            //for WHERE clause
            IN 0, IN UUID,
            IN 0, IN trackNumber);
    VARTRACE(res);
    if (res) return res;
    ETG_TRACE_USR4(("Title: %s" ,rippedTrackData.title));

    return res;
}

tResult DBManager::GetFolderHierarchyForDeviceID(OUT vector<tObjectID>& folderHierarchy, IN const tDeviceID deviceID)
{
    ENTRY;

    tResult res = MP_NO_ERROR;
    Query query;
    res = query.Select(LTY_FOLDERHIERARCHY_CHILDID_BY_DEVICEID, IN deviceID, "");
    if (!res)
    {
        vector<tObjectID> tempFolderHierarchy;
        /* get all folders in the order it was indexed */
        while(1)
        {
            tObjectID childID;
            res = query.Get("i",
                    OUT &childID);
            if (res) break;

            tempFolderHierarchy.push_back(childID);
        }
        folderHierarchy.swap(tempFolderHierarchy);
        if( MP_ERR_DB_END_OF_LIST == res) // When fetching is successful for Get, MP_ERR_DB_END_OF_LIST is not an error.
        {
            res = MP_NO_ERROR;
        }
    }
    query.End();

    return res;
}

tResult DBManager::GetChildIDForFolderPath(OUT tObjectID& childID, IN tDeviceID deviceID, IN tPath folderPath)
{
    ENTRY;

    VARTRACE(deviceID);
    VARTRACE(folderPath);

    tResult res = MP_NO_ERROR;

    // If folder path is root child id for the same is not available in db. Hence use 0 as childid.
    if (0 == strcmp(folderPath, "/"))
    {
        childID = 0;
    }
    else
    {
        // If there is a trailing '/' in the folder path, remove it for getting folder.
        if(FastUTF8::EndsWithNC((const FastUTF8::tString)folderPath, (const FastUTF8::tString)"/")) //lint !e1773
        {
            unsigned char *folder;
            FastUTF8::Split(OUT folder, INOUT (unsigned char *)folderPath); //lint !e1773
        }
        VARTRACE(folderPath);

        Query query;
        res = query.Select(LTY_FOLDERHIERARCHY_CHILDID_BY_FOLDERPATH, IN deviceID, "t", IN folderPath);
        VARTRACE(res);
        if (!res)
        {
            res = query.Get("i",
                    OUT &childID);
        }
        query.End();
    }

    VARTRACE(childID);
    VARTRACE(res);

    return res;
}

tResult DBManager::GetFolderPathForChildID(OUT tPath& folderPath, IN tObjectID childID, IN tDeviceID deviceID)
{
    ENTRY;

    VARTRACE(deviceID);
    VARTRACE(childID);

    tResult res = MP_NO_ERROR;

    if(0 == childID)
    {
        strncpy_r(OUT folderPath, IN "/", IN sizeof(folderPath));
    }
    else
    {
        Query query;
        res = query.Select(LTY_FOLDERHIERARCHY_FOLDERPATH_BY_CHILDID, IN deviceID, "i", IN childID);
        if (!res)
        {
            res = query.Get("t", OUT folderPath);
        }
        query.End();
    }

    VARTRACE(folderPath);
    VARTRACE(res);

    return res;
}
tResult DBManager::CheckifPlaylistExists(OUT int &PlaylistNameCount,IN const tDeviceID deviceID, IN const tPlaylistName playlistName)
{
    ENTRY;
    tResult res = MP_NO_ERROR;
    Query query;
    res = query.Select(LTY_PLAYLISTNAME_COUNT, IN deviceID, "t", IN playlistName);
    if (res) return res;
    /* get the count */
    res = query.Get("i", OUT &PlaylistNameCount);
    query.End();
    if (res) return res;

    //if count is not 0, playlist exists already
    VARTRACE(PlaylistNameCount);
    return res;
}
// Find yomiAlbum name for the given album ID
tResult DBManager::GetYomiAlbumNameForAlbumId(IN tObjectID albumID, tMetadata &yomiAlbum)
{
    ENTRY;
    tResult res = MP_NO_ERROR;
    tObjectID mediaObjectID = OBJECT_ID_NONE;
    Query query;
    res = query.Select(LTY_MEDIAOBJECT_BY_ALBUMID, IN MUSICBOX_DEVICE_ID, "i", IN albumID);
    if (res) return res;

    res = query.Get("i", OUT &mediaObjectID);
    query.End();
    if (res) return res;

    //found a mediaObjectID for the given album ID
    VARTRACE(mediaObjectID);
    if(mediaObjectID )
    {
        tYomiMetadata yomiMetadata;
        res = GetYomiMetadata(mediaObjectID, yomiMetadata);
        if(res == MP_NO_ERROR)
        {
            strncpy_r(yomiAlbum,yomiMetadata.YomiAlbum ,sizeof(yomiMetadata.YomiAlbum));
            VARTRACE(yomiAlbum);
        }
    }
    return res;
}

tResult DBManager::GetQuickFingerprint(tFingerprint &quickfingerprint, const tDeviceID deviceID)
{
    ENTRY
    Query query;
    tResult res;

    res = query.Select(LTY_DEVICES, IN deviceID, "");
    if (res) return res;

    res = query.Get("---------------------------------------T", OUT quickfingerprint, IN sizeof(tFingerprint));
    query.End();

    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::SetQuickFingerprint(const tDeviceID deviceID, const tFingerprint quickFingerprint)
{
    ENTRY
    Query query;
    tResult res;

    res = query.Update(LTY_DEVICES_UPDATE);
    if (res) return res;

    /* set the given devices quickfingerprint(also known as short checksum).For this DeviceUUID field is made use.
     * quickfingerprint applicable for Mass storage devices. For Mass Storage Devices, the field "DeviceUUID" is not in use.
     * Hence field "DeviceUUID" is made use to store quickfingerprint
     */
    res = query.Change("------------------------------------------------------it----------ii", IN 0, IN quickFingerprint, IN 0, IN deviceID);
    if (res) return res;

    return MP_NO_ERROR;
}

tResult DBManager::AddRouteGuidanceDisplayComponentInfo(const tDiPORGDisplayComponentInfo RGDisplayComponentInfo)
{
    ENTRY
    tResult res = MP_NO_ERROR;
    Query query;

    res = query.Insert(LTY_ROUTEGUIDANCEDISPLAYCOMPONENTS_INSERT);
    if (res) return res;

    if(MP_NO_ERROR == res)
    {

        res = query.Put(IN "itiiiii",
                IN RGDisplayComponentInfo.Identifier,
                IN RGDisplayComponentInfo.Name,
                IN RGDisplayComponentInfo.CurrentRoadNameLength,
                IN RGDisplayComponentInfo.DestinationNameLength,
                IN RGDisplayComponentInfo.AfterManeuverRoadNameLength,
                IN RGDisplayComponentInfo.ManeuverDescriptionLength,
                IN RGDisplayComponentInfo.GuidanceManeuverCapacity);
    }
    return res;
}

tResult DBManager::DeleteRouteGuidanceDisplayComponentInfo(const tU16 displayIdentifier)
{
    tResult res = MP_NO_ERROR;
    Query query;
    /* query to delete an entry from the ROUTEGUIDANCEDISPLAYCOMPONENTS table */
    res = query.Delete(LTY_ROUTEGUIDANCEDISPLAYCOMPONENTS_DELETE);

    if(MP_NO_ERROR == res)
    {
        res = query.Remove("ii",0,IN displayIdentifier);
    }

    return res;
}

tResult DBManager::UpdateRouteGuidanceDisplayComponentInfo(const tDiPORGDisplayComponentInfo RGDisplayComponentInfo)
{
    ENTRY;
    tResult res = MP_NO_ERROR;
    Query query;

    VARTRACE(RGDisplayComponentInfo.Identifier);

    /* first check if Identifier is already in the database */
    res = query.Select(LTY_ROUTEGUIDANCEDISPLAYCOMPONENTS, MY_MEDIA, "ii", IN 0,IN RGDisplayComponentInfo.Identifier);
    if (res) return res;

    tU16 identifier;

    res = query.Get("-i", OUT &identifier);
    query.End();

    VARTRACE(res);
    VARTRACE(identifier);

    //Add the Identifier to table If it is not present
    if(MP_ERR_DB_END_OF_LIST == res)
    {
        AddRouteGuidanceDisplayComponentInfo(RGDisplayComponentInfo);
    }
    else
    {
        //Update the the row with new data.
        res = query.Update(LTY_ROUTEGUIDANCEDISPLAYCOMPONENTS_UPDATE);
        if(MP_NO_ERROR == res)
        {

            res = query.Change("itiiiiiiiiiiii",
                    IN 0, IN RGDisplayComponentInfo.Name,
                    IN 0, IN RGDisplayComponentInfo.CurrentRoadNameLength,
                    IN 0, IN RGDisplayComponentInfo.DestinationNameLength,
                    IN 0, IN RGDisplayComponentInfo.AfterManeuverRoadNameLength,
                    IN 0, IN RGDisplayComponentInfo.ManeuverDescriptionLength,
                    IN 0, IN RGDisplayComponentInfo.GuidanceManeuverCapacity,
                    IN 0, IN RGDisplayComponentInfo.Identifier);
        }
    }
    return res;
}

tResult DBManager::GetRouteGuidanceDisplayComponentInfos(tDiPORGDisplayComponentList & RGDisplayComponentInfos)
{
    ENTRY
    tResult res = MP_NO_ERROR;
    Query query;
    RGDisplayComponentInfos.clear();

    /* prepare the sql statement for the routeguidanceDisplayComponentsinfo list */
    res = query.Select(LTY_ROUTEGUIDANCEDISPLAYCOMPONENTS, MY_MEDIA, "");
    VARTRACE(res);
    //if (res) return res;


    if (MP_NO_ERROR == res)
    {
        while(1)
        {
            tDiPORGDisplayComponentInfo displayComponentInfo;
            InitDiPOSetRGDisplayInfo(displayComponentInfo);

            res = query.Get("-iTiiiii",
                    OUT &displayComponentInfo.Identifier,
                    OUT displayComponentInfo.Name,IN sizeof(displayComponentInfo.Name),
                    OUT &displayComponentInfo.CurrentRoadNameLength,
                    OUT &displayComponentInfo.DestinationNameLength,
                    OUT &displayComponentInfo.AfterManeuverRoadNameLength,
                    OUT &displayComponentInfo.ManeuverDescriptionLength,
                    OUT &displayComponentInfo.GuidanceManeuverCapacity);
            if (res) break;

            RGDisplayComponentInfos.push_back(displayComponentInfo);
        }
    }

    return res;
}

tResult DBManager::RemoveMultiPartitionDevice(tDeviceSerialNumber &serialNumber)
{
    ENTRY
    tResult res = MP_NO_ERROR;
    tDeviceID deviceID = DEVICE_ID_NOT_SET;
    Query query;
    tDeviceInfo deviceInfo;

    ETG_TRACE_USR3(("serialNumber from DB:%s",serialNumber));
    ETG_TRACE_USR3(("valid serialNumber:%s",(--mValidPartitionInfo.end())->second.c_str()));

    res = query.Select(IN LTY_DEVICEID_BY_SERIALNUMBER,IN MY_MEDIA, IN "t", IN serialNumber);
    if(res)
    {
        ETG_TRACE_ERR(("Query.Select failed with Error=%d/%s", res, errorString(res)));
        return res;
    }

    res = query.Get("i", OUT &deviceID);
    query.End();

    if (res == MP_NO_ERROR)
    {
        ETG_TRACE_USR3(("deviceID from DB: %d",deviceID));
        res = GetDeviceInfo(OUT deviceInfo, IN deviceID);
        if (res) return res;

        if(deviceInfo.connectionState!=CS_DISCONNECTED)
        {
            ETG_TRACE_USR3(("connectionState is not CS_DISCONNECTED"));
            deviceInfo.connectionState=CS_DISCONNECTED;
            const vector<tDeviceInfo> deviceInfos{deviceInfo};
            res=DeviceChanged(deviceInfos);
            if (res) return res;
        }

        res=RemoveDevice(deviceID);
        if(res)
        {
            ETG_TRACE_ERR(("RemoveDevice failed with Error=%d/%s", res, errorString(res)));
            return res;
        }
    }

    return res;
}
