#ifndef _QUERY_H_
#define _QUERY_H_

/* includes */
#include <map>
#include "Database.h"
#include "Lock.h"

#define MAX_QUERY_LENGTH 2000

/* the list of all queries */
typedef struct {
    const char *query;
} tSQLStatement;

typedef struct {
    unsigned char *dataPtr;  //pointer to start of data
    size_t length;  //length of data in bytes
} tBlob;

/* transaction lock global to all data base connections */
static Lock gTransactionLock;

class Query
{
public:
	Query();
	~Query();

	tResult Select(const tListType listType, const tDeviceID deviceID, const char *format, ...);
	tResult Select(const tListType listType, const tDeviceID deviceID, const tIndex startIndex, const tIndex noOfRows, const char *format, ...);
	tResult Select(const tListType listType, const tDeviceID deviceID, const tIndex startIndex, const tIndex noOfRows, const char *format, va_list vl);
	tResult Explain(char *planResult, size_t size);
	tResult SetRow(tRowNumber row);
	tResult ResetRow();
	tResult GetRow(tRowNumber &row) const;
	tResult SetLimit(const tIndex limit);
	tResult SetLimitAndRow(const tIndex limit, const tRowNumber row);
	tResult Get(const char *format, ...);
	tResult Get(const int doStep, const char *format, ...);
	tResult Save();
	tResult Restore();
	tResult End();

	tResult PositionSelect(const Query *parent, const tDeviceID deviceID, const char *format, ...);
	tResult PositionSelect(const tListType listType, const tDeviceID deviceID, const char *format, ...);
	tResult PositionGet(tPosition &position, const tObjectID objectID);

	tResult Insert(const tListType listType);
	tResult Put(const char *format, ...);

	tResult Update(const tListType listType);
	tResult Change(const char *format, ...);

	tResult Delete(const tListType listType);
	tResult Remove(const char *format, ...);

	tResult BeginTransaction(const tListType listType);
	tResult EndTransaction();

	static tResult Finalize(sqlite3 *db);

	tResult DoLock();
	tResult UnLock();

	static int GetQueryCacheCount();
	static int QueryLengthForListType(tListType listType);

protected:
    tResult GetInternal(const int DoStep, const char *format, va_list vl);
	tResult Execute();
	tResult Peek(const char *format, ...);

    Database *mDB;
	sqlite3_stmt *mStatement;
	tListType mListType;
	class Lock mLock; // to lock against multi threaded

	/* for select only */
	char *mSelectString;
	int mRow;
	int mPrevRow;
	int mStartIndex;
	int mLimit;
	int mEOF;
	int mSavedRow;
	int mSavedLimit;
	int mBindingForOffset;
	int mBindingForLimit;
	static void SqlTrace(void *context, const char *string);
	static void SqlProfile(void *context, const char *string, sqlite3_uint64 nanosecs);
	tResult Prepare(sqlite3_stmt **statement, sqlite3 *db, const char *query);
	tResult Finalize(sqlite3_stmt *statement) const;
	void CheckFatalError(sqlite3 *db, int sqlite3ResultCode);

	/* for others */
	int mLastSqlError;
};

#endif // _QUERY_H_
