/**
 * @addtogroup DBManager
 * @author Dinesh
 *
 * handles Database trigger registration and corresponding SM message callback
 * @{
 */


#ifndef DBTRIGGERMANAGER_H_
#define DBTRIGGERMANAGER_H_

#include "TypeDefinitions.h"
#include "Lock.h"
#include <sqlite3.h>

typedef enum
{
    TRIGGER_ON_INSERT = 0UL,
    TRIGGER_ON_DELETE,
    TRIGGER_ON_UPDATE
} tTriggerMode;

typedef struct
{
    tTriggerID triggerID;        /*unique ID to identify each trigger registration*/
    tTriggerMode type;
    char smName[256];       /*Destination state machine name */
    char msgName[1024];     /*sm message to be sent to dest State machine*/
    char triggerName[1024];
} tTriggerContext;

class DBTriggerManager
{
public:
    DBTriggerManager();
    virtual ~DBTriggerManager(){};

    /*
     *  Get common instance.
     *  note : seprate instance of DBTrigger can be created for multiple DB instances
     *          then common instance need not be used
     */
    static DBTriggerManager& GetInstance()
    {
        static DBTriggerManager triggerManager;
        return triggerManager;
    }
    /*
     *  Setup a Database connection for DBTrigger Manager instance
     */
    tResult AttachDB(sqlite3 *dbHandle);
    tResult DetachDB(sqlite3 *dbHandle);

    /*
     * API to Register a trigger
     */
    tTriggerID RegisterOnInsert(const char* tableName, const char* callbackSMName , const char* callbackMsgName , const char* clause = "", const char* cbArg1 = "COUNT()", const char* cbArg2 = "NEW.rowid");
    tTriggerID RegisterOnDelete(const char* tableName, const char* callbackSMName , const char* callbackMsgName , const char* clause = "", const char* cbArg1 = "COUNT()", const char* cbArg2 = "OLD.rowid");
    tTriggerID RegisteronUpdate(const char* tableName, const char* callbackSMName , const char* callbackMsgName , const char* columnName  = "", const char* clause = "", const char* cbArg1 = "COUNT()", const char* cbArg2 = "OLD.rowid");

    /*
     * API to Deregister a trigger based on its ID as input
     */
    tBool DeRegisterAllTriggers(void);
    tBool DeRegister(tTriggerID triggerID);

    /*
     * Retrieves all the active Triggers from the currently attached DB
     */
    tResult GetTriggers(vector<string> &triggerNames);

    /*
     * Drops all the active triggers from the currently attached DB
     *
     */
    tResult TriggerCleanupDB();

    /*
     * sqlite call back function to indicate a triggers
     *
     */
    tBool TriggerCallback(int num_values, sqlite3_value **values);


private:
    sqlite3 * mDbHandle;
    Lock mCallbackLock;
    tTriggerID mTriggerID;                         /**< running value to assign unique ID for each trigger registration*/
    vector <tTriggerContext *> mActiveTriggers;    /**< list of registered triggers*/

    tBool Initialize(void);
    tBool ExecuteQuey(const char* statement);
    tTriggerID CreateTrigger(tTriggerMode type, const char* tableName,
                                               const char* callbackSMName,
                                               const char* callbackMsgName,
                                               const char* clause = "",
                                               const char* columnName = "",
                                               const char* cbArg1 = "",
                                               const char* cbArg2 = "");


// wrapper calls to map the old DB Trigger API names used by the DBManager. remove once DBManager adopted to use new APIs
public:
    unsigned long regisCallback_onInsert (const char* smName, const char* msgName, sqlite3 * dbHandle, const char* tbl_name, const char* clause = NULL, const char* cbArg1 = "COUNT()", const char* cbArg2 = "NEW.rowid")
    {
        if(clause != NULL)
            return RegisterOnInsert(tbl_name,smName,msgName,clause,cbArg1, cbArg2);
        else
            return RegisterOnInsert(tbl_name,smName,msgName, "", cbArg1, cbArg2);
    }
    unsigned long regisCallback_onDelete (const char* smName, const char* msgName, sqlite3 * dbHandle, const char* tbl_name,const char* clause = NULL, const char* cbArg1 = "COUNT()", const char* cbArg2 = "OLD.rowid")
    {
        if(clause != NULL)
            return RegisterOnDelete(tbl_name,smName,msgName,clause,cbArg1, cbArg2);
        else
            return RegisterOnDelete(tbl_name,smName,msgName, "", cbArg1, cbArg2);
    }
    unsigned long regisCallback_onUpdate (const char* smName, const char* msgName, sqlite3 * dbHandle, const char* tbl_name, const char* col_name, const char* clause = NULL, const char* cbArg1 = "COUNT()", const char* cbArg2 = "OLD.rowid" )
    {
        if(clause != NULL)
            return RegisteronUpdate(tbl_name,smName,msgName,col_name,clause,cbArg1, cbArg2);
        else
            return RegisteronUpdate(tbl_name,smName,msgName,col_name, "", cbArg1, cbArg2);
    }

};

#endif /* DBTRIGGERMANAGER_H_ */
