/**
 * @author Dinesh D
 *
 * Implementation for Favorites handling
 *
 */

/* ETG definitions */
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_mp.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_FAVORITE_MANAGER
#ifdef TARGET_BUILD
#include "trcGenProj/Header/FavoritesManager.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_FAVORITE_MANAGER
#endif
#endif
#include "FunctionTracer.h"
#include "VarTrace.h"

#include "FavoritesManager.h"
#include "LocalSPM.h"

FavoritesManager::FavoritesManager()
{
    ENTRY_INTERNAL

    /* Early initialization of variables */
    m_deactivaterequired = TRUE;
    m_FavID = 0;
}


tResult FavoritesManager::ActivateFavorite(tFavoriteID &retFavID, const tFavoriteID favID, const tLogicalAVChannel audioChn, // finished: 70%
                const tLogicalAVChannel videoChn)
{
    ENTRY
    (void)audioChn;
    (void)videoChn;

    tResult res;

    tBoolean isPlayerActive;
    tObjectID objectID;
    tCategoryType objectType;
    tDeviceID deviceID;

    /*lookup for object information of the requested favorite*/
    res = LocalSPM::GetDBManager().GetObjectIdOfFavorite(OUT objectID, OUT objectType, IN favID);
    if (res)  return res;
	VARTRACE(objectID);
	VARTRACE(objectType);
	
	/*CTY_PLAYLIST_ITEM is not handled in CreatePlayList so it would create LTY_SONG list
	  objectID which belongs to CTY_PLAYLIST_ITEM will not be available.*/
	if(objectType == CTY_PLAYLIST_ITEM)
	{
		tUUID UUID;
		tObjectID newObjectID;
		LocalSPM::GetDBManager().GetObjectUUID(OUT deviceID, OUT UUID, IN objectID);
		VARTRACE(deviceID);
		VARTRACE(UUID);
		LocalSPM::GetDBManager().GetObjectIDByUUID(OUT newObjectID, IN deviceID, IN UUID);
		VARTRACE(newObjectID);
		objectID=newObjectID;
		objectType=CTY_SONG;
	}
	/*favorite with category type CTY_PODCAST could be podcast episode or podcast itself*/
	/*favorite with category type CTY_VIDEO could be video episode or video itself*/
	else if((objectType == CTY_PODCAST) || (objectType == CTY_VIDEO))
	{
	    // Get the real object type
	    res = LocalSPM::GetDBManager().GetObjectType(OUT objectType, INOUT objectID);
	    if (res) return res;
	}

    /*request for play back of stored favorite via CustomControl using its object information*/
    tListID listID;
    res = LocalSPM::GetCustomControl().PlayObjectID(OUT isPlayerActive, OUT listID, IN objectID, IN objectType, PC_NO_REPEAT);
    if (res)  return res;

    /* Get the number of connected devices*/
    tDeviceCount devCount = 0;
    res = LocalSPM::GetDBManager().GetConnectedDeviceCount(OUT devCount);
    VARTRACE(devCount);

    tDeviceID lastDeviceID = DEVICE_ID_NOT_SET;
    if(devCount>1)  //GetLastActiveDevice() will fail if only one device is connected
    {
        /* Get last active device ID */
        res = LocalSPM::GetDBManager().GetLastActiveDevice(OUT lastDeviceID);
        ETG_TRACE_USR3(("FavoritesManager::FavoriteDeactivation: lastDeviceID:%d",lastDeviceID));
        if (res)  lastDeviceID = DEVICE_ID_NOT_SET;
    }

    /* Get current active device ID */
    tDeviceID currentDeviceID = DEVICE_ID_NOT_SET;
    res = LocalSPM::GetDBManager().GetActiveDevice(OUT currentDeviceID);
    ETG_TRACE_USR3(("FavoritesManager::FavoriteDeactivation: currentDeviceID:%d",currentDeviceID));
    if (res)  return res;

    /* Check if deactivation of favorite is required or not */
    if((lastDeviceID == currentDeviceID) || (MY_MEDIA == currentDeviceID))
    {
        SetDeactivationReqFlag(FALSE);
    }
    else
    {
        if(devCount == 1)
        {
            SetDeactivationReqFlag(FALSE);
        }
        else
        {
            SetDeactivationReqFlag(TRUE);
        }
    }

    /*inform DBManager about Favorite activation*/
    //res = LocalSPM::GetDBManager().DeActivateAllFavorites();        //De activate all before activating new one
    res = LocalSPM::GetDBManager().ActivateFavorite(IN favID);
    if (res)  return res;

    retFavID = favID;

    /*If Media player is not the active audio source , then request for AV activation*/
    if(!isPlayerActive)
    {
        res = LocalSPM::GetOutputWrapper().RequestAVActivation(1,LC_MAIN_AUDIO);
        if (res)
        {
            ETG_TRACE_ERR(("RequestAVActivation call failed while activating a favorite"));
            return res;
        }
    }

    return MP_NO_ERROR;
}

tResult FavoritesManager::GetObjectName(tTitle &name ,tObjectID objectID, tCategoryType &objectType) const // finished: 100%
{
    ENTRY_INTERNAL

    tResult res;

    /*Get the media object from DB*/
    tMediaObject mediaObject;
    res = LocalSPM::GetDBManager().GetMediaObject( OUT mediaObject, IN objectID, IN objectType);
    if (res)
    {
        ETG_TRACE_ERR(("FavoritesManager::Store -> Unable to retrieve the media object for objectID:%d , type:%d",objectID,objectType));
        strcpy(name,"UNKNOWN");
        return MP_NO_ERROR;
    }

    /*retrieve the tag name from appropriate mediaobject field */
    switch(objectType)
    {
        case CTY_SONG:          // audio object
        case CTY_AUDIOBOOK:     // audiobook object
        case CTY_VIDEO:         // video object
        case CTY_PODCAST:       // podcast object
        case CTY_CHAPTER:       // audio book chapter name
        case CTY_EPISODE:       // podcast or video episode name
        case CTY_PLAYLIST_ITEM:
            strcpy(name,mediaObject.title);
            break;

        case CTY_GENRE:         // genre name
        case CTY_AUTHOR:        // audiobook author name
        case CTY_PLAYLIST:      // playlist name
            strcpy(name,mediaObject.MetadataField1);
            break;

        case CTY_NAME:          // video or podcast name
            strcpy(name,mediaObject.MetadataField1);

            /*resolve object type between video or podcast*/
            objectType = mediaObject.catType;
            break;

        case CTY_ARTIST:        // artist name
        case CTY_TITLE:         // audiobook title
            strcpy(name,mediaObject.MetadataField2);
            break;

        case CTY_COMPOSER:      // composer name
            strcpy(name,mediaObject.MetadataField3);
            break;

        case CTY_ALBUM:         // album name
            strcpy(name,mediaObject.MetadataField4);
            break;

        default :
            strcpy(name,"UnKnown");
            ETG_TRACE_ERR(("FavoritesManager::GetObjectName Object with ID:%d and CategoryType:%d not handled",objectID,objectType));
            break;

    }
    return MP_NO_ERROR;
}

tResult FavoritesManager::StoreFavorite(tFavoriteID &retFavID, const tObjectID _objectID) // finished: 100%
{
    ENTRY

    tResult res;

    tLabelText labelText;
    tDescriptorText descripterText;
    tTitle title;
	tMediaObject mediaObject;
	
    tCategoryType objectType;
    tObjectID objectID = _objectID;

    memset(labelText,0,sizeof(tLabelText));
    memset(descripterText,0,sizeof(tDescriptorText));

	if (objectID & IPOD_MARKER_BIT) {
		res = LocalSPM::GetDBManager().GetFavoriteNameIPOD(OUT title , INOUT objectID, OUT objectType);
		if (res) return res;

	} else {

		/*
		 * get object type.
		 *
		 * Attention: objectID may be replaced by a correct one in case a live tag was sent
		 */
		res = LocalSPM::GetDBManager().GetObjectType(OUT objectType, INOUT objectID);
		if (res) return res;

		/*TODO : replace with DB::GetObjectInfo once its available or working for all type*/
		res = GetObjectName(OUT title, IN objectID, IN objectType);
		if (res) return res;
	}
    /* Create LabelText and DescripterText from the objectInfo.objectName*/
    strncpy_r(labelText,title,sizeof(tLabelText));
    strncpy_r(descripterText,title,sizeof(tDescriptorText));
	
	/* Check for objectid in db and set favid to zero Fix for GMMY17-3668;GMMY17-3654;GMMY17-4191  */
	res = LocalSPM::GetDBManager().GetMediaObject( OUT mediaObject, IN objectID, IN objectType);
    if (res)
    {
		ETG_TRACE_ERR(("FavoritesManager::Store -> Unable to retrieve the media object for objectID:%d , type:%d",objectID,objectType));
		retFavID=0;
        return MP_NO_ERROR;		
    }
	
    /*Store the favorite in the Database*/
    res = LocalSPM::GetDBManager().StoreFavorite(OUT retFavID, IN objectID, IN objectType, IN labelText, IN descripterText);
    if (res)  return res;

    return  MP_NO_ERROR;
}

tResult FavoritesManager::DeleteFavorite(tFavoriteID &retFavID, const tFavoriteID favID)  // finished: 100%
{
    ENTRY

    tResult res;

    if(favID == 0x0000)                     // delete all favorites
        return ResetFavorites();

    m_FavID=favID;
    /*Delete the favorite in the Database*/
    res = LocalSPM::GetDBManager().DeleteFavorite(IN favID);
    if (res)  return res;

    retFavID = favID;

    return MP_NO_ERROR;
}

tResult FavoritesManager::ResetFavorites(void) const // finished: 100%
{
    ENTRY

    tResult res;

    /*Delete the favorite in the Database*/
    res = LocalSPM::GetDBManager().ResetFavorites();
    if (res)  return res;

    return MP_NO_ERROR;
}

tResult FavoritesManager::GetAllFavoriteInfo(vector<tFavoriteInfo> &favInfo) const // finished: 100%
{
    ENTRY

    tResult res;

    /*retrieve all the favorite info from the Database*/
    res = LocalSPM::GetDBManager().GetAllFavoriteInfo(favInfo);
    if (res)  return res;

    return MP_NO_ERROR;
}


tResult FavoritesManager::GetFavoriteInfo(tFavoriteInfo &favInfo, const tFavoriteID favID) const // finished: 100%
{
    ENTRY

    tResult res;

    res = LocalSPM::GetDBManager().GetFavoriteInfo(OUT favInfo,IN favID);
    if (res)  return res;

    return MP_NO_ERROR;
}

tResult FavoritesManager::OnDeviceUpdate(const tDeviceID deviceID, const tDeviceConnected connected) const // finished: 100%
{
    ENTRY

    ETG_TRACE_USR3(("FavoritesManager::OnDeviceUpdate: deviceID:%d ,connected:%d",deviceID,connected));

    tResult res = MP_NO_ERROR;

    /*retrieve all the favorite info from the Database to check whether they belong to updated device(connected/removed)*/
    vector<tFavoriteInfo> favInfos;
    res = LocalSPM::GetDBManager().GetAllFavoriteInfo( OUT favInfos);
    if (res)  return res;

    /*filter out list of favorites which needed to be updated*/
    unsigned iter ;
    vector<tFavoriteID> affectedFavorites;
    for ( iter = 0 ; iter < favInfos.size(); iter++)
    {
        /*if favorite availability and updated device availability is different -> update the favorite availability*/
        if(connected != favInfos[iter].available)
        {
            vector<tDeviceID> objectDeviceIDs;
            res = LocalSPM::GetDBManager().GetObjectDeviceID( OUT objectDeviceIDs, IN favInfos[iter].objectID , IN favInfos[iter].objectType);

            if (objectDeviceIDs.size() == 0)/*exception : object not found*/
            {
                ETG_TRACE_ERR(("FavoritesManager::OnDeviceUpdate deviceID:%d -> failed to retrieve mediaobject for favInfos[%d].objectID:%d,error:%d",deviceID,iter,favInfos[iter].objectID,res));
                if (favInfos[iter].available)
                {
                    res = LocalSPM::GetDBManager().UpdateFavoriteStatus( IN favInfos[iter].favoriteId, false, false);
                }
            }
            else if (FALSE != connected) /*a new device gets connected*/
            {
                for (unsigned int counter = 0; counter < objectDeviceIDs.size(); counter++)
                {
                    if (deviceID == objectDeviceIDs[counter])
                    {
                        affectedFavorites.push_back(favInfos[iter].favoriteId);
                        break;
                    }
                }
            }
            else /*a device gets removed*/
            {
                tDeviceInfo deviceInfo;
                bool bUpdate = true;
                for (unsigned int counter = 0; counter < objectDeviceIDs.size(); counter++)
                {
                    /*check if at least one of the device to which this fav belongs to , is still in Connected state*/
                    res = LocalSPM::GetDBManager().GetDeviceInfo(OUT deviceInfo, IN objectDeviceIDs[counter]);
                    if (res)
                    {
                        ETG_TRACE_ERR(("FavoritesManager::OnDeviceUpdate -> failed to retrieve deviceinfo for deviceID:%d,error :%d ",objectDeviceIDs[counter],res));
                        continue;
                    }

                    /* one of the device is in connected state , so keep fav as available as earlier */
                    if (CS_CONNECTED == deviceInfo.connectionState)
                    {
                        /*Do not update*/
                        bUpdate = false;
                        break;
                    }
                }

                /* No device related to this favorite is in CONNECTED state*/
                if (bUpdate)
                    affectedFavorites.push_back(favInfos[iter].favoriteId);
            }
        }
        else if(connected == favInfos[iter].available)
        {
            vector<tDeviceID> objectDeviceIDs;
            res = LocalSPM::GetDBManager().GetObjectDeviceID( OUT objectDeviceIDs, IN favInfos[iter].objectID , IN favInfos[iter].objectType);
            if (objectDeviceIDs.size() == 0)/* Device is connected but the favorite item is no longer present in MediaObjects table */
            {
                 ETG_TRACE_ERR(("FavoritesManager::OnDeviceUpdate deviceID:%d ->DeviceConnected but not in DB favInfos[%d].objectID:%d,error:%d",deviceID,iter,favInfos[iter].objectID,res));
                 if (favInfos[iter].available)
                 {
                     res = LocalSPM::GetDBManager().UpdateFavoriteStatus( IN favInfos[iter].favoriteId, false, false);
                 }
            }
        }
    }

    /*if more favorites belongs to the updated device then avoid individual favorite item update trigger*/
    if (affectedFavorites.size() > (unsigned int)LocalSPM::GetDataProvider().FavItemChangedUpdateLimit())
    {
        LocalSPM::GetCustomControl().SwitchDBTrigger(TT_DB_FAVORITE_CHANGED, TS_OFF);
    }

    /*for each affected favorites, set the availability based on the device availability*/
    for (iter = 0; iter < affectedFavorites.size(); iter++)
    {
        res = LocalSPM::GetDBManager().UpdateFavoriteStatus(IN affectedFavorites[iter], connected, false);
        if (res)
            ETG_TRACE_ERR(("FavoritesManager::OnDeviceUpdate deviceID:%d -> Failed to update the status for favInfos[%d]:%d",deviceID,iter,affectedFavorites[iter]));
    }

    if (affectedFavorites.size() > (unsigned int)LocalSPM::GetDataProvider().FavItemChangedUpdateLimit())
    {
        /*If more favorites belongs to the updated device then trigger collective CONTENT_CHANGED update to clients*/
        res = LocalSPM::GetOutputWrapper().SendFavoritesChanged(LCH_CONTENT_CHANGED, affectedFavorites.size(), 0);
        if (res)
            ETG_TRACE_ERR(("FavoritesManager::OnDeviceUpdate, failed to send favorite update"));

        /*restore the individual favorite item update trigger back*/
        res = LocalSPM::GetCustomControl().SwitchDBTrigger(TT_DB_FAVORITE_CHANGED, TS_ON);
        if (res)
            ETG_TRACE_ERR(("FavoritesManager::OnDeviceUpdate, failed to enable trigger"));
    }

    return res;
}

tResult FavoritesManager::UpdateAllFavoriteStatus(void) const // finished: 100%
{
    ENTRY

    tResult res;

    /*retrieve all the favorite info from the Database*/
    vector<tFavoriteInfo> favInfos;
    res = LocalSPM::GetDBManager().GetAllFavoriteInfo( OUT favInfos);
    if (res)  return res;

    /*check and update the status of each favorite in the database*/
    unsigned int iter;
    for(iter = 0 ; iter < favInfos.size(); iter++)
    {
        tBoolean favChanged;
        res = UpdateFavoriteStatus( OUT favChanged, IN favInfos[iter].favoriteId);
        if (res)  return res;
    }

    return MP_NO_ERROR;
}

tResult FavoritesManager::UpdateFavoriteStatus(tBoolean &favChanged, const tFavoriteID favID) const // finished: 70%
{
    ENTRY

    tResult res;
    favChanged = false;

    tFavoriteInfo favInfo;
    res = LocalSPM::GetDBManager().GetFavoriteInfo( OUT favInfo, IN favID);
    if (res)  return res;

    tDeviceID deviceID;
    res = LocalSPM::GetDBManager().GetObjectDeviceID( OUT deviceID, IN favInfo.objectID, IN favInfo.objectType);
    if (res)  return res;

    /* check if the requested device is connected */
    tDeviceInfo deviceInfo;
    res = LocalSPM::GetDBManager().GetDeviceInfo(OUT deviceInfo, IN deviceID);
    if (res)  return res;

    if(favInfo.available != deviceInfo.connected)
    {
        res = LocalSPM::GetDBManager().UpdateFavoriteStatus( IN favInfo.favoriteId, IN deviceInfo.connected, IN favInfo.active);
        favChanged = true;
        if (res)  return res;
    }

    return MP_NO_ERROR;
}

tResult FavoritesManager::SetDeactivationReqFlag(tBoolean deactFav)
{
    ENTRY
    m_deactivaterequired = deactFav;
    return MP_NO_ERROR;
}
tBoolean FavoritesManager::GetDeactivationReqFlag()
{
    ENTRY
    return m_deactivaterequired;
}

tFavoriteID FavoritesManager::GetCurrentFavID()
{
    ENTRY
    return m_FavID;
}
