/**
 * @defgroup FavoritesManager FavoritesManager
 * @author Thomas Beul
 * 
 * @brief Public interface for favorite handling of media player
 *
 * @{
 */
#ifndef _FAVORITESMANAGER_H_
#define _FAVORITESMANAGER_H_

#include "TypeDefinitions.h"

using namespace std;

/**
 * definition of class FavoritesManager 
 * handles the favorite specific actions
 */
class FavoritesManager
{

#define SUPPORTED_FAV_COUNT 20
public:	

    /*Constructor */
    FavoritesManager();


    /**
	 * This function activates a previous stored mediaplayer favorite.
	 * Therefore a playlist will be created and playback started.
	 * In case the mediaplayer has no active audio route a RequestAvActivation will be generated
	 * The parameter AudioChn and VideoChn are currently ignored and the eMAIN_AUDIO channel is used for RequestAvActivation.
	 *
	 * @param[in] favID identifier of the favorite to play
	 * @param[in] audioChn defines the AudioChannel to be used for RequestAvActivation
	 * @param[in] videoChn defines the VideoChannel to be used for RequestAvActivation
	 * @param[out] retFavID identifier of the activated favorite
	 * @return < 0: error, = 0: OK
	 */
	tResult ActivateFavorite(tFavoriteID &retFavID, const tFavoriteID favID, const tLogicalAVChannel audioChn,
	        const tLogicalAVChannel videoChn) ;
	
	/** 
	 * This function stores the given ObjectID under a free FavoritID 
	 * 
	 * @param[in] objectID identifier of the favorite object to be stored
	 * @param[out] retFavID favorite identifier of the stored object
	 * @return < 0: error, = 0: OK
	 */
	tResult StoreFavorite(tFavoriteID &retFavID, const tObjectID objectID);
	
	/** 
	 * This function deletes the favorite identified by the FavoriteID parameter 
	 * 
	 * @param[in] favID identifies the favorite to be deleted
	 * @param[out] retFavID favorite identifier of the deleted object
	 * @return < 0: error, = 0: OK
	 */
	tResult DeleteFavorite(tFavoriteID &retFavID, const tFavoriteID favID) ;

	/** 
	 * The ResetFavorites function causes the Favorite Provider to remove all favorites from its Stored Favorites
	 * and then rebuild its Stored Favorites with its Default Favorites.   
	 * This function returns NumStoredFavorites which is the number of favorites in the Stored Favorites.
	 * 
	 * @return < 0: error, = 0: OK
	 */
	tResult ResetFavorites(void) const;

	/** 
     * The OnDeviceUpdate function causes the favorite manager to update the status from all its stored Favorites
     * which belongs to updated device (connected or removed)
     *
     * @param[in] deviceID device ID of effected device
     * @return < 0: error, = 0: OK
     */
	tResult OnDeviceUpdate(const tDeviceID deviceID, const tDeviceConnected connected) const ;

	/** 
	 * The UpdateFavoriteState function triggers the status update of all stored favorites
	 * 
	 * @return < 0: error, = 0: OK
	 */
	tResult UpdateAllFavoriteStatus(void) const;

	/** 
	 * The UpdateFavoriteState function checks the availability and playstatus of the given favorite 
	 * 
	 * @param[in] favID ID of favorite to update the status
	 * @param[out] favChanged returns if the status of the favorite has changed
	 * @return < 0: error, = 0: OK
	 */
	tResult UpdateFavoriteStatus(tBoolean &favChanged, const tFavoriteID favID) const;

	/** 
	 * The GetAllFavoriteInfo returns the favorite info of all favorites

	 * @param[out] favInfo vector loaded with all the stored favorites
	 * 	 *
	 * @return < 0: error, = 0: OK
	 */
	tResult GetAllFavoriteInfo(vector<tFavoriteInfo> &favInfo) const;

	/** 
	 * The GetFavoriteInfo returns the favorite info of the given favorite 
	 * 
	 * @param[in] favID ID of favorite to update the status
	 * @param[out] favInfo contains the favorite information
	 * @return < 0: error, = 0: OK
	 */
	tResult GetFavoriteInfo(tFavoriteInfo &favInfo, const tFavoriteID favID) const;

    /**
     * This function sets the value of the m_deactivaterequired flag.
     */
	tResult SetDeactivationReqFlag(tBoolean deactFav);

    /**
     * This function checks if favorite deactivation is needed or not. Compare between activated DTY and active source.
     * m_deactivaterequired=TRUE,if deactivation of favorites is required, else for m_deactivaterequired=FALSE, deactivation of favorites is not required  *
     */
	tBoolean GetDeactivationReqFlag();

	/*Store Current FavID for later usage*/
	tFavoriteID GetCurrentFavID();

private:
	tResult GetObjectName(tTitle &name ,tObjectID objectID, tCategoryType &objectType) const;
	tBoolean m_deactivaterequired;
	tFavoriteID m_FavID;
};

#endif  //_FAVORITESMANAGER_H_

/** @} */
