/**
 * @defgroup CustomControl CustomControl
 * @author Thomas Beul
 *
 * @brief Public interface for custom control class to encapsulate customer specific actions of the Mediaplayer
 * @{
 */

#ifndef _CUSTOMCONTROL_H_
#define _CUSTOMCONTROL_H_

#include <iostream>
#include "ILocalSPM.h"
#include "ThreadFactory.h"
#include "Timer.h"
#include "CustomControlSM.h"
#include "FavoritesManager.h"
#include "PersonalizationProxy.h"
#include <vector>
using namespace std;

/**
 * definition of class CustomControl
 * encapsulate customer specific actions of the Mediaplayer
 */
 class CustomControl : public CustomControlSM , public ILocalSPM, public TFThread
{

public:

//SPM part

    /**
     * Constructor of component.
     * This function is used by LocalSPM to store the CustomControl componentID in mComponentID.
     *
     * @param[in] componentID component ID assigned by SPM
     * @return void
     */
    CustomControl(tComponentID componentID);

    /**
     * Destructor of component.
     *
     * @return void
     */
    virtual ~CustomControl();

    /**
     * This function is used by LocalSPM to create the CustomControl.
     * @attention: running in SPM thread context
     * Create the CustomControl state machine (including create of message queue).
     * Inform LocalSPM that Create is ready -> CreateDone(0)
     *
     * @return void
     */
    void Create();

    /**
     * This function is used by LocalSPM to trigger the CustomControl initialization.
     * @attention: running in SPM thread context
     * Init the CustomControl state machine.
     * Register CustomControlSM with dispatcher.
     * Inform LocalSPM that Init is ready -> InitDone(0)
     *
     * @return < 0: error, = 0: OK
     */
    tResult Init(tInitReason reason);

    /**
     * Initialize own member variables
     * Start timer.
     *
     * @return != 0: error, = 0: OK
     */
    tResult InitSM();

    /**
     * This function is used by LocalSPM to start the CustomControl.
     * @attention: running in SPM thread context
     * From now on all other mediaplayer components are available.
     * Start the CustomControl thread and the state machine.
     *
     * @return < 0: error, = 0: OK
     */
    tResult Run();

    /**
     * This function starts a user function in a thread from the thread factory's thread pool.
     * After leaving while loop used thread is stopped and released for other users
     *
     * @return void
     */
    void Do(int functionID, void *ptr);

    /**
     * This function is used by LocalSPM to stop the CustomControl.
     * @attention: running in SPM thread context
     * Store last mode values.
     * Deregister on DB trigger.
     * Set state machine to final state -> SendMessage(STOP_SM).
     * LocalSPM will be informed after STOP_SM event is processed
     *
     * @return != 0: error, = 0: OK
     */
    tResult Stop();

    /**
     * Inform LocalSPM that Stop is ready -> StopDone(0)
     *
     * @return != 0: error, = 0: OK
     */
    tResult StopEventProcessed();

    /**
     * This function is used by LocalSPM to cleanup the CustomControl.
     * @attention: running in SPM thread context
     * Deregister CustomControlSM with dispatcher.
     * Set state machine to final state -> SendMessage(DONE).
     * LocalSPM will be informed after DONE event is processed.
     *
     * @return != 0: error, = 0: OK
     */
    tResult Done();

    /**
     * Inform LocalSPM that Done is ready -> DoneDone(0)
     *
     * @return != 0: error, = 0: OK
     */
    tResult DoneEventProcessed();

    /**
     * Returns the current state the state machine is in (for debugging of shutdown problems)
     * @param[inout] stateName buffer for storing the current state name
     *
     * @return pointer to stateName
     */
    char *GetSMStateName(OUT tGeneralString stateName, IN size_t size);

    /**
    * Returns answer if the component is a state machine
    *
    * @return true or false
    */
    tBoolean IsComponentSM() {return true;}; //component is a state machine

    /**
     * This function gives the favorite manager instance to access fav APIs
     *
     * @return reference to instance of Favorite Manager component
     */
    FavoritesManager &GetFavoritesManager(void);
    /**
     * This function gives the PersonalizationProxy instance to access PersonalizationProxy APIs
     *
     * @return reference to instance of PersonalizationProxy component
     */
    PersonalizationProxy &GetPersonalizationProxy(void);

    /**
     * This function passes the input parameters to the ListControl CreatePlayList to create a
     * list of given list type and starts playback of this list.
     *
     * @param[in] listType list type to create play list filtered after object ID
     * @param[in] filterTags filter tag object id's to generate the complete list
     *                     which ID is in which tag position depends on the list type.
     *                     max 4 different tags are possible [0..3]
     * @param[in] playContinuation decides what to do on end of list
     * @param[in] deviceID ID of the device
     * @param[out] listInfo reference to the object storing the information of the created list
     * @return < 0: error, = 0: OK
     */
    tResult CreateQuickPlayList(tListInfo &listInfo,
        const tListType listType,
        const tObjectID filterTags[4],
        const tPlayContinuation playContinuation,
        const tDeviceID deviceID) const;

    /**
     * This function passes the input parameters to the ListControl CreatePlayList to create a
     * folder list and starts playback of this list.
     *
     * @param[in] path path of folder to play
     * @param[in] playContinuation decides what to do on end of list
     * @param[in] deviceID ID of the device
     * @param[out] listSize reference to the parameter containing the listSize of the created list
     * @return < 0: error, = 0: OK
     */
    tResult CreateFolderQuickPlayList(tListSize &listSize,
        const tPath path,
        const tPlayContinuation playContinuation,
        const tDeviceID deviceID,
        const tListType listType = LTY_FILELIST) const;

    /**
     * This function passes the input parameters to the list control CreatePlayList to create a list and
     * starts playback of these list. Depending on the return value the FAV manager has to request the audio route.
     *
     * @param[in] objectID object ID to play
     * @param[in] objectType decides the object type
     * @param[in] playContinuation decides what to do on end of list
     * @param[out] isActive is the player manager in active state
     * @return < 0: error, = 0: OK
     */
    tResult PlayObjectID(tBoolean &isActive,tListID &listID ,
        const tObjectID objectID,
        const tCategoryType objectType,
        const tPlayContinuation playContinuation) const;

    /**
     * This function passes the input parameters to the list control CreatePlayList to create a list and
     * starts playback of this list.
     *
     * @param[in] objectID object ID to play
     * @param[in] playContinuation  decides what to do on end of list
     * @param[out] listID reference to the parameter containing the listID of the created list
     * @param[out] listSize reference to the parameter containing the listSize of the created list
     * @return < 0: error, = 0: OK
     */
    tResult PlayMediaPlayerObject(tListID &listID,
        tListSize &listSize,
        const tObjectID objectID,
        const tPlayContinuation playContinuation) const;

    /**
     * Function checks list and media object to use and request this list to play.
     * Roadmap 13017
     * Get repeat mode from DBManager -> GetRepeatMode(&repeatMode, deviceID).
     * Get listInfo from ListControl -> GetListInfo(&listInfo, listID).
     * if( listInfo.listType == LTY_FILELIST or LTY_FILELIST_UNSORTED or LTY_BROWSELIST )
     * - if( repeatMode == RPT_LIST_WITH_SUBLISTS )
     *   - Create a file list including sub folders at ListControl -> CreateFileListWithSubFolders(&newListID, path, deviceID).
     * - if( repeatMode == RPT_ALL )
     *   - Create a file list including sub folders at ListControl -> CreateFileListWithSubFolders(&newListID, "/", deviceID).
     * - Get current media object old list via ListControl -> GetCurrentMediaObject(&mediaObject, listID).
     * - Get position of current media object in the new list via ListControl -> GetPositionInList(&position, newListID, objectID).
     * - Map newListID to listID and position to listIndex
     * Play new list in PlayerManager -> SendNewList(listID, listIndex, offset)
     * Get size of list from ListControl -> GetListInfo(&listInfo, listID).
     *
     * @param[out] listSize number of elements in the list
     * @param[in] listID id of the list which should be used for playback
     * @param[in] listIndex index of object in list (-1 = last active object)
     * @param[in] offset playtime to start
     * @return != 0: error, = 0: OK
     */
    tResult PlayItemFromList(tListSize &listSize, //Roadmap 13017: 80%
        const tListID listID,
        const tIndex listIndex,
        const tPlaytime offset) const;

    /**
     * Function checks list and media object to use and request this list to play.
     * Get position of the media object in the list via ListControl -> GetPositionInList(&position, listID, objectID).
     * Map position to listIndex.
     * Call PlayItemFromList(&listSize, listID, listIndex, offset).
     *
     * @param[out] listSize number of elements in the list
     * @param[in] listID ID of the list which should be used for playback
     * @param[in] objectID object ID to play
     * @return != 0: error, = 0: OK
     */
    tResult PlayItemFromListByTag(tListSize &listSize, const tListID listID, const tObjectID objectID) const;

    /**
     * Function sends ACTIVE_MEDIA_DEVICE_SET event to state machine.
     * Call SendMessage(ACTIVE_MEDIA_DEVICE_SET, deviceID, activeSource).
     *
     * @param[in] deviceID ID of the device to activate
     * @param[in] activeSource defines if the device becomes inactive (FALSE) or active (TRUE)
     * @return != 0: error, = 0: OK
     */
    tResult SendActiveMediaDeviceSet(const tDeviceID deviceID, const tDeviceActiveSource activeSource, const tMediaContext mediaContext = MC_ALL /* default value */);

    /**
     * This function retrieves the active device from data provider and the last played list id of the device from ListControl.
     * If the active device is not equal to the requested device playback of last played list starts.
     *
     * @param[in] deviceID ID of the device to activate
     * @param[in] activeSource defines if the device becomes inactive (FALSE) or active (TRUE)
     * @return < 0: error, = 0: OK
     */
    tResult HandleActiveMediaDeviceSet(const tDeviceID deviceID, const tDeviceActiveSource activeSource, const tMediaContext mediaContext = MC_ALL /* default value */);

    /**
     * This function retrieves the active device from data provider and the last played list id of the device from ListControl.
     * If the active device is not equal to the requested device playback of last played list starts.
     *
     * @param[in] deviceID ID of the device to activate
     * @param[in] activeSource defines if the device becomes inactive (FALSE) or active (TRUE)
     * @return < 0: error, = 0: OK
     */
    tResult ActiveMediaDeviceSet(const tDeviceID deviceID, const tDeviceActiveSource activeSource, const tMediaContext mediaContext = MC_ALL /*default value */);

    /**
     * This function retrieves the active device from data provider and returns the deviceInfo of this device.
     *
     * @param[out] deviceInfo reference to the parameter indicating the deviceInfo struct of the active device.
     * @return < 0: error, = 0: OK
     */
    tResult ActiveMediaDeviceGet(tDeviceInfo &deviceInfo) const;

    /**
     * This function is used by PlayManager to inform about no media object available for playback and gives the reason therefor.
     * Depending on the given information the function decides how to go on.
     * if( noObjectReason == NOR_NO_OBJECT ) //Roadmap 13012
     * - Get list info from ListControl -> GetListInfo(&listInfo, listID).
     * - Get device info from DBManager -> GetDeviceInfo(&deviceInfo, listInfo.deviceID).
     * - if deviceInfo.deviceID == MyMedia
     *   - Create a AllSong list at ListControl -> CreatePlayList(&newListId, ALL_SONG_LIST, ..., MY_MEDIA)
     *   - Play new list in PlayerManager -> NewList(newListID,0,0)
     *
     * @attention Obsolete: Handled by ListControl::WrapProcessing instead
     * else if( noObjectReason == NOR_END_OF_LIST/NOR_START_OF_LIST ) //Roadmap 13017
     * - Get next list from ListControl -> GetNextList(&newListID)
     * - Get list info from ListControl -> GetListInfo(&listInfo, listID).
     * - Get device info from DBManager -> GetDeviceInfo(&deviceInfo, listInfo.deviceID).
     * - if newListID is not valid and deviceInfo.deviceID == MyMedia
     *   - Create a AllSong list at ListControl -> CreatePlayList(&newListId, ALL_SONG_LIST, ..., MY_MEDIA)
     * - if newListID is valid
     *   - Play new list in PlayerManager -> NewList(newListID,0,0)
     *
     * @param[in] listID identifier of the current played list
     * @param[in] noObjectReason identifies the reason why no media object is available for playback
     * @return < 0: error, = 0: OK
     */
    tResult ListIsEmpty(const tListID listID, const tNoObjectReason noObjectReason); //Roadmap 13012: 100%, 13017: 0%

    /**
     * This function starts the startup timer with configurable duration
     *
     * @return < 0: error, = 0: OK
     */
    tResult StartStartupTimer(void);

    /**
     * Returns the answer to condition if the MP is in the startup phase (running startup timer).
     *
     * @return true or false
     */
    tBoolean IsStartupPhase(void) const;

    /**
     * This function sends the Message TIMEOUT to own StateMachine
     *
     * @return < 0: error, = 0: OK
     */
    tResult SendMessageTimeout(void);

    /**
    * This function sends the Message RESET_FORMER_CONNECTION to own StateMachine
    *
    * @return < 0: error, = 0: OK
    */
    tResult SendResetFormerConnection(void);

    /**
    * This function resets former connection state in DB -> DBManager::ResetFormerConnection()
    *
    * @return < 0: error, = 0: OK
    */
    virtual tResult ResetFormerConnection();

    /**
     * This function cares about startup timeout handling
     * Stop the startup timer -> StopStartupTimer()
     * Check device activation -> CheckDeviceActivation()
     * Check list selection -> CheckListSelection()
     * Roadmap 13023
     * Get all devices of last mode table -> DBManager::GetLastModeDevices()
     * Get all connected devices -> DBManager::GetMediaplayerDeviceConnections()
     * Loop over all devices which are not connected any longer
     * - Clear last mode for these devices -> DBManager::ClearLastPlayed(deviceID)
     *
     * @return < 0: error, = 0: OK
     */
    virtual tResult HandleTimeout(void); //Roadmap 13023: 100%

    /**
     * This function cares about a newly inserted favorite in the database
     *
     * @param[in] favCount number of favorites in the DB
     * @param[in] favID id of currently added new favorite
     * @return < 0: error, = 0: OK
     */
    virtual tResult HandleFavoriteInserted(const tFavCount favCount, const tFavoriteID favID);

    /**
     * This function cares about a deleted favorite in the database
     *
     * @param[in] favCount number of favorites in the DB
     * @param[in] favID id of currently added new favorite
     * @return < 0: error, = 0: OK
     */
    virtual tResult HandleFavoriteDeleted(const tFavCount favCount, tFavoriteID favID);

    /**
     * This function cares about an updated favorite in the database
     *
     * @param[in] favCount number of favorites in the DB
     * @param[in] favID id of currently added new favorite
     * @return < 0: error, = 0: OK
     */
    virtual tResult HandleFavoriteUpdated(const tFavCount favCount, const tFavoriteID favID);

    /**
     * This function cares about an updated device in the database
     * Call UpdateMediaPlayerDeviceConnections at OutputWrapper -> UpdateMediaPlayerDeviceConnections().
     * Send DB_DEVICE_CHANGED message to own SM -> SendEvent("DB_DEVICE_CHANGED, deviceCount, deviceID").
     * Inform FavoriteManager about the device update -> OnDeviceChanged(deviceID).
     * if device is also active -> UpdateActiveMediaDevice at OutputWrapper
     *
     * @param[in] deviceCount number of devices in the DB
     * @param[in] deviceID device ID of currently added new device
     * @return < 0: error, = 0: OK
     */
    virtual tResult HandleDeviceUpdated(const tDeviceCount deviceCount, const tDeviceID deviceID);

    /**
     * Handle the event when a device was set to CS_CONNECTED in the database.
     *
     * @param[in] deviceCount number of devices in the DB
     * @param[in] deviceID device ID of affected device
     * @return < 0: error, = 0: OK
     */
    virtual tResult HandleDeviceConnected(const tDeviceCount deviceCount, const tDeviceID deviceID);

    /**
     * Handle the event when a device was set the number of files > 0
     * @param[in] deviceCount number of devices in the DB
     * @param[in] deviceID device ID of affected device
     * @return < 0: error, = 0: OK
     */
    virtual tResult HandleDeviceFirstContentAvailable(const tDeviceCount deviceCount, const tDeviceID deviceID);

    /**
     * AudioManager is informed about new device via OutputWrapper
     * @param[in] deviceCount number of devices in the DB
     * @param[in] deviceID device ID of affected device
     * @return < 0: error, = 0: OK
     */
    virtual tResult HandleDeviceNewDevice(const tDeviceCount deviceCount, const tDeviceID deviceID);

    /**
     * AudioManager is informed about the error state or if the same device is insert via OutputWrapper
     * @param[in] deviceCount number of devices in the DB
     * @param[in] deviceID device ID of affected device
     * @return < 0: error, = 0: OK
     */
    virtual tResult HandleDeviceSameDevice(const tDeviceCount deviceCount, const tDeviceID deviceID);

    /**
     * Handle the event when a device was set to CS_DISCONNECTED in the database.
     * Calls: ListControl::DeviceRemoved(deviceID)
     * Clear last mode for this device -> DBManager::ClearLastMode(deviceID) //Roadmap 13023
     *
     * @param[in] deviceCount number of devices in the DB
     * @param[in] deviceID device ID of affected device
     * @return < 0: error, = 0: OK
     */
    virtual tResult HandleDeviceRemoved(const tDeviceCount deviceCount, const tDeviceID deviceID); //Roadmap 13023: 100%

    /**
     * Handle the event when the active device was changed in the database.
     *
     * Calls OutputWrapper::Update ActiveMediaDevice
     *
     * @param[in] deviceCount number of devices in the DB
     * @param[in] deviceID device ID of activated device
     * @return < 0: error, = 0: OK
     */
    virtual tResult HandleDeviceActivated(const tDeviceCount deviceCount, const tDeviceID deviceID);

    /**
     * Handle the event when the repeat mode of a device was changed in the database.
     * Roadmap 13017
     * Get current played list from PlayerManager -> GetListID(&listID).
     * Get list info from ListControl -> GetListInfo(&listInfo, listID).
     * if( listInfo.listType == LTY_FILELIST or LTY_FILELIST_UNSORTED or LTY_BROWSELIST )
     * - Get repeat mode from DBManager -> GetRepeatMode(&repeatMode, deviceID).
     * - Get current played media object from PlayerManager -> GetMediaObject(&mediaObject).
     * - if( repeatMode == RPT_LIST_WITH_SUBLISTS )
     *   - Create a file list including sub folders at ListControl -> CreateFileListWithSubFolders(&listID, path, deviceID).
     * - if( repeatMode == RPT_ALL )
     *   - Create a file list including sub folders at ListControl -> CreateFileListWithSubFolders(&listID, "/", deviceID).
     * - if( repeatMode == RPT_NONE or RPT_ONE or RPT_LIST )
     *   - Create a normal file list at ListControl -> CreateMediaPlayerFileList(&listID, ..., path, deviceID).
     * - Set current media object in created list in ListControl -> SetCurrentMediaObject(listID, mediaObject.objectID).
     * - Set next list in PlayerManager. It will be used after end of current song -> SendNextList(listID)
     *
     * @param[in] deviceCount number of devices in the DB
     * @param[in] deviceID device ID of activated device
     * @return < 0: error, = 0: OK
     */
    virtual tResult HandleDeviceRepeatMode(const tDeviceCount deviceCount, const tDeviceID deviceID); //Roadmap 13017: 70%

    /**
     * Calculates device activation action from config table and do it.
     *
     * @param[inout] deviceID device ID of affected device
     * @param[in] precondition trigger which initiate the device activation check
     * @return < 0: error, = 0: OK
     */
    tResult CheckDeviceActivation(tDeviceID &deviceID, const tConfigTablePrecondition precondition);

    /**
     * Calculates list selection action from config table and do it.
     *
     * @param[in] deviceID device ID of affected device
     * @return < 0: error, = 0: OK
     */
    tResult CheckListSelection(const tDeviceID deviceID) const;

    /**
     * sets switch for auto register on database trigger while Run
     * *
     * @param[in] autoRegister on or off
     * @return != 0: error, = 0: OK
     */
    tResult AutoRegisterOnDBTrigger(const tTriggerState autoRegister);

    /**
     * Switches the database triggers on/off for custom control
     * Add trigger for repeat mode changed -> TT_DB_DEVICE_REPEAT_MODE //Roadmap 13017
     * Add trigger for number of files changed -> TT_DB_DEVICE_NEW/TT_DB_DEVICE_SAME //Roadmap 13002
     *
     * @param[in] trigger = TT_ALL: switches every trigger, or:
     *                      TT_DB_DEVICE_UPDATED, TT_DB_FAVORITE_CHANGED, ...
     * @param[in] onOff = TS_ON or = TS_OFF
     *
     * @return = MP_NO_ERROR : no error, or:
     *           MP_ERR_DB_REGISTER_FAILED: a register has failed
     *           MP_ERR_DB_UNREGISTER_FAILED: a un-register has failed
     */
    tResult SwitchDBTrigger(const tTriggerType trigger, const tTriggerState onOff); //Roadmap 13017: 100%, 13002: 100%

    /**
     * Sets the new streaming device.
     * Roadmap 13008
     * CreatePlaylist(&listID, LTY_CURRENT_SELECTION, streaming=TRUE, ..., deviceID)
     * Only if current device not given device
     * - DBManager::SetActiveDevice(IN deviceID)
     * PlayerManager::SendNewList(IN listID, 0, 0)
     * (- implicit: SetLastPlayedList(...))
     *
     * @param[in] deviceID new streaming device ID
     * @return != 0: error, = 0: OK
     */
    tResult SetActiveStreamingDevice(const tDeviceID deviceID); //Roadmap 13008: 100%

    /**
     * A helper function for the diagnosis to play a song from a given device type
     *
     * - searches for the first device of given type (must be a device which supports directoy browsing)
     * - generates a file list
     * - searches for the given song in that list (max 100 songs supported)
     * - calls internal PlayItemFromList(with id of the found song)
     * - sets the repeat mode for the device to SONG or ONE
     * - calls 'RequestSource' at the audio manager (TODO: implement it in function)
     *
     * @param[in] deviceType type of device which should play the song (must be connected first)
     * @return != 0: error, = 0: OK
     */
    tResult DiagnosisPlaySong(tDeviceType deviceType, tFilename filename); //Roadmap 13025


    /**
     * Sets the current language
     * If dynamic language switch is enabled , then it recreates the DB to reflect the new list sort
     * order based on the new language.
     *
     * @param[in] language current language
     * @return < 0: error, = 0: OK
     */
    tResult SetLanguage(const tLanguageType language) const;

    /**
     * Recreates the DB
     * Switch off the triggers before recreation and switch it on after recreation
     *
     * @return < 0: error, = 0: OK
     */
    tResult RecreateDatabase(const tLanguageType language) const;


    /**Roadmap 13031-BT_DeviceSourceAllocation
     * Decides to make AutomaticSource Switch to DTY_IPHONE/IPOD or to make DTY_BLUETOOTH unavailable,
     * when an Apple Device connected as iAP(via USB)device while already being connected  to target via bluetooth
     *
     * @param[in] appleDeviceInfo device structure representing the connected apple device
     * @return != 0: error, = 0: OK
     */
    tResult CheckSourceSwitchFromBTStreaming(tDeviceInfo &appleDeviceInfo);

    /**Roadmap 13031-BT_DeviceSourceAllocation
     * Decides to make Apple device again available as DTY_BLUETOOTH to MediaPlayer when its iAP(via USB)
     * connection is removed
     *
     * @param[in] appleDeviceInfo device structure representing the connected apple device
     * @return != 0: error, = 0: OK
     */
    tResult HandleAppleDeviceRemoved(tDeviceInfo &appleDeviceInfo);

    /**
     * Function calls Outputwrapper::OnFocusAppCHanged
     *
     * @return != 0: error, = 0: OK
     */
    tResult OnFocusAppChanged(const tDeviceID deviceID, const tAppName appName);

    /**
     * Internal function which checks all indexed list size against current DB contents
     * and sends than: MediaPlayerListChange(listID, LCH_CONTENT_CHANGED, listSize, NULL) via OutputWrapper for each list
     *
     * @param[in] deviceID ID of the affected device to be checked
     * @return < 0: error, = 0: OK
     */
    tResult CheckLists(const tDeviceID deviceID);

    /**
     * Informs custom control that an indexing was started
     *
     * Starts a timer which checks cyclic if a list size has changed (using CheckLists)
     * and sends MediaPlayerListChange(listID, LCH_CONTENT_CHANGED, listSize, 0, NULL) via OutputWrapper
     * the timeout time will be configured by DataProvide (default=3 seconds)
     *
     * @param[in] deviceID id of the device for special filtered list depending on a device
     * @return < 0: error, = 0: OK
     */
    tResult StartCheckListTimer(const tDeviceID deviceID);

    /**
     * Informs custom control that an indexing was stopped
     *
     * Stops the timer which checks cyclic if a list size has changed
     * an rechecks the current list sizes by calling CheckLists.
     *
     * @return < 0: error, = 0: OK
     */
    tResult StopCheckListTimer(void);

     /**
     * Calls the Personalization Profile fetching call
     *
     * @return < 0: error, = 0: OK
     */
     tResult GetProfileID();

    //>>------ Roadmap 16003

    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R8972_UC3_LLD4.1_SEQ7
     * Method to delete media content
     * @param[in] deleteMediaByTagsParam the info packed in a structure related to the tags and deviceid
     * @return < 0: error, = 0: OK
     */
    tResult DeleteMediaContent(const tDeleteMediaByTagsParam & deleteMediaByTagsParam, const tU8 syncMode = 0);

    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R8972_UC3_LLD4.1_SEQ7
     * Method to delete media content
     * @param[in] deleteMediaObjectIDParam     structure of mediaobjects and deviceID
     * @return < 0: error, = 0: OK
     */
    tResult DeleteMediaContent(IN const tDeleteMediaByObjectIDParam &deleteMediaObjectIDParam);

    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R8972_UC3_LLD4.1_SEQ7
     * Method to delete media content
     * @param[in] deleteMediaObjectIDParam     structure of mediaobject urls and deviceID
     * @return < 0: error, = 0: OK
     */
    tResult DeleteMediaContent(IN const tDeleteMediaByUrlParam &deleteMediaByUrlParam);


    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R8972_UC3_LLD4.3_SEQ6
     * Method to edit the metadata of media object
     * @param[in] tEditMetaDataByID struct with the mediaobjectid and metadata
     * @return < 0: error, = 0: OK
     */
    tResult EditMetaData(IN const  tEditMetaDataByID &editMetaDataByID);

    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R8972_UC3_LLD4.3_SEQ6
     * Method to delete media content
     * @param[in] deleteMediaObjectIDParam     structure of mediaobject urls and deviceID
     * @return < 0: error, = 0: OK
     */
    tResult EditMetaData(IN const tEditMetaDataByUrl &editMetaDataByUrl);


    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R8976_UC3_LLD3.1_SEQ2
     * Method to Add the filenames to internal playlist
     * @param[in] tPlayListAddOperation struct with the filenames  and deviceid
     * @return < 0: error, = 0: OK
     */
    tResult AddFileNamesToPlayList(IN const  tPlayListAddOperation &PlayListAddOperation);


    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R8976_UC3_LLD3.2
     * Method to delete the entries from internal playlist
     * @param[in] tPlayListDeleteOperation struct with the playlist positions  and deviceid
     * @return < 0: error, = 0: OK
     */
    tResult DeleteEntriesFromPlaylist(IN const  tPlayListDeleteOperation &PlayListDeleteOperation);

    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R8976_UC3_LLD3.1_SEQ2
     * Method to Add the filenames to internal playlist
     * @param[in] PlayListAddOperationByTags struct with the tags, list type  and deviceid
     * @return < 0: error, = 0: OK
     */
    tResult AddFileNamesToPlayListByTag(IN tPlayListAddOperationByTags PlayListAddOperationByTags);

//<<------ Roadmap 16003
    //CMG3G-10221   scene recorder
    tResult SetFilePermissions(tSetFilePermission pFilePermissions);
    tBool getSystemCondition();
    void setSystemCondition(tBool condition);
    void setSRMount(IN const tDeviceInfo &deviceInfo, IN tBool condition);

    //<--Roadmap 17001 : 'Personalization'
    /**
     * StorePersonalizedFavorite Stores the User Favorite in DB based on the UserID
    *  @param[in] object ID    *
    *  @param[in] userID
    *  @param[OUT] favID
     * @return < 0: error, = 0: OK
     */
    tResult StorePersonalizedFavorite(tFavoriteID &retFavID, const tObjectID objectID,  IN const tUserID userID );

    /**
     * DeletePersonalizedFavorite Deletes the User Favorite in DB based on the UserID
    *  @param[in] userID
    *  @param[in] objectID
     * @return < 0: error, = 0: OK
     */
    tResult DeletePersonalizedFavorite(const tObjectID objectID, const tUserID userID);

    /**
     * SetCurrentUser Adds the userID in DB
    *  @param[in] userID
     * @return < 0: error, = 0: OK
     */
    tResult SetCurrentUser(IN const tUserID userID );

     /**
     * Informs custom control that Retry for ProfileID
     *
     * Start a timer to get profileID from Personalization
     * when the initial startup update is missed by
     * Mediaplayer
     *
     * @return < 0: error, = 0: OK
     */
     tResult StartPersonalizationTimer();

     /**
     * Informs custom control that Retry for ProfileID
     *
     * Stops the timer which checks cyclic if profileID is received
     *
     * @return < 0: error, = 0: OK
     */
     tResult StopPersonalizationTimer();
    //<--Roadmap 17001 : 'Personalization'

     tResult UpdateFolderBrowseStructureCleared(tDeviceID deviceID);

     tDeviceID GetDeviceIDForFolderStructureCleared();

     void SetDeviceIDForFolderStructureCleared(tDeviceID deviceID);

     tResult GetVolumeFromAudio();

     /**
      * This function handles the folder up down request from the user.
      *
      * Gets the folder path of the nowplaying media item.
      * Finds the folder path to switch to.
      * Creates a MediaPlayerFileList for the new folder path.
      * Sends updated playlist for starting playback to PlayerManager.
      *
      * @param[in] playbackAction Specifies if it is a Folder up/down request
      * @param[in] nextPrevSkipCount Denotes the number of folders to skip
      * @return < 0: error, = 0: OK
      */
     tResult HandleFolderUpDown(tPlaybackAction playbackAction, tNextPrevSkipCount nextPrevSkipCount);

     tResult UpdateReadyToPlay(IN const tDeviceSerialNumber deviceAddress, IN const tPEPlaybackState playbackState);

private:

    static bool TimerCallBack(timer_t timerID , void* instance ,const void *userData);

    /**
     * This function stops the startup timer
     *
     * @return < 0: error, = 0: OK
     */
    tResult StopStartupTimer(void);

    //>>------ Roadmap 16003 : CD Ripping With Gracenote
    Lock m_DeletMediaContentLock;

    Lock m_EditMetadataLock;

    Lock m_PlayListOperationLock;

    Lock m_FilePermissionLock;
    Lock m_SystemConditionLock;
    tBool bNormalCondition;
    tBool bPlayPendingInReindexing;
    /**
     * R8972_UC3_LLD4.1_SEQ7
     * This thread function deletes the media content
     * @param[in] functionID of thread
     * @param[in] prt the info packed in a structure related to the tags or objectids or urls and deviceid
     * @return < 0: error, = 0: OK
     */
    tResult DoDeleteMediaContentThread(IN int functionID, IN const void *param);

    /**
     *
     * R8972_UC3_LLD4.1_SEQ7
       * This thread function deletes the media content
       * @param[in] pdeleteMediaByTagsParam structure of device id and tags
       * @param[out] numberofFilesDeleted the number of files deleted
       * @return < 0: error, = 0: OK
       */
    tFileOperationErrorCodes deleteMediaContentByTags(tDeleteMediaByTagsParam &pdeleteMediaByTagsParam,tU16 &numberofFilesDeleted);
    /**
     * R8972_UC3_LLD4.3_SEQ6
     * This thread function edit the metadata
     * @param[in] functionID of thread
     * @param[in] prt the info packed in a structure related to the tags or objectid or url and deviceid
     * @return < 0: error, = 0: OK
     */
    tResult DoEditMetaDataThread(IN int functionID, IN const void *param);


     /**
      * R8976_UC3_LLD3.1_SEQ2,R8976_UC3_LLD3.1_SEQ3
     * This thread function does the operation on the internal playlist
     * @param[in] functionID of thread
     * @param[in] prt the info packed in a structure related to the urls or positions
     * @return < 0: error, = 0: OK
     */
    tResult DoPlayListOperationThread(IN int functionID, IN const void *param);

    //>--Roadmap CMG3G-10221 : 'Scene Recorder '

    /* Find out specified path is a directory
     * */
    tBoolean isDirectory(IN std::string url, IN const tDeviceInfo & deviceID);
    /**
     * DeleteMediaContentFromDirectory function will remove contents from directory
     * read write .
     * @param[in] deviceInfo for the device
     * @param[in] Dir name
     * @param[OUT] number of file deleted
     * @return < 0: error, = 0: OK
     */
    tResult DeleteMediaContentFromDirectory(IN const tDeviceID deviceID,IN  std::string dirName,OUT tU16 &numofFilesDeleted);
    /**
     * DoFilePermissionThread This thread function does the operation to set file permission
     * @param[in] functionID of thread
     * @param[in] prt the info packed in a structure related to the urls or positions
     * @return < 0: error, = 0: OK
     */
    tResult DoFilePermissionThread(IN const void *param);
    //<--Roadmap CMG3G-10221 : 'Scene Recorder '

public:
    FavoritesManager mFavoriteManager;
    tPEPlaybackState GetPlaybackStateForReadyToPlay();
    tBool GetPlayingStatus();
    tResult GetDeviceAddressForReadyToPlay(tDeviceSerialNumber &deviceAddress);

private:

    Timer mTimer;
    timer_t mTimerID;
    tBoolean mDeviceFound;

    Timer mCheckListTimer;
    timer_t mCheckListTimerID;         // answer timer id
    static bool CheckListTimerCallBack(timer_t timerID , void* instance ,const void *userData);

    Timer mPersonalizationTimer;
    timer_t mPersonalizationTimerID;         // answer timer id
    static bool PersonalizationTimerCallback(timer_t timerID , void* instance ,const void *userData);

    tTriggerState mAutoRegisterOnDBTrigger;
    tTriggerID mTriggerID_DB_DEVICE_CHANGED_CSTATE;
    tTriggerID mTriggerID_DB_DEVICE_CHANGED_ISTATE;
    tTriggerID mTriggerID_DB_DEVICE_CHANGED_ACTIVE;
    tTriggerID mTriggerID_DB_DEVICE_CONNECTED;
    tTriggerID mTriggerID_DB_DEVICE_REMOVED;
    tTriggerID mTriggerID_DB_DEVICE_ACTIVATED;
    tTriggerID mTriggerID_DB_DEVICE_REPEAT_MODE;        //Roadmap 13017
    tTriggerID mTriggerID_DB_DEVICE_NUMBER_OF_AUDIO_FILES_NEWDEVICE;    //Roadmap 13002
    tTriggerID mTriggerID_DB_DEVICE_NUMBER_OF_AUDIO_FILES_SAMEDEVICE;    //Roadmap 13002
    tTriggerID mTriggerID_DB_DEVICE_FIRST_CONTENT_AVAILABLE;
    tTriggerID mTriggerID_DB_FAVORITE_UPDATED_AVAIL;
    tTriggerID mTriggerID_DB_FAVORITE_UPDATED_ACTIVE;
    tTriggerID mTriggerID_DB_FAVORITE_INSERTED;
    tTriggerID mTriggerID_DB_FAVORITE_DELETED;
    tTriggerID mTriggerID_DB_DEVICE_CHANGED_DIPOCAPABLE;
    tTriggerID mTriggerID_DB_DEVICE_CHANGED_DIPOACTIVE;
    tTriggerID mTriggerID_DB_DEVICE_CHANGED_NAME;
    tTriggerID mTriggerID_DB_DEVICE_SAME_DEVICE;
    tTriggerID mTriggerID_DB_DEVICE_NEW_DEVICE;
    tTriggerID mTriggerID_DB_DEVICE_NUMBER_OF_VIDEO_FILES_SAMEDEVICE;
    tTriggerID mTriggerID_DB_DEVICE_NUMBER_OF_VIDEO_FILES_NEWDEVICE;
    tTriggerID mTriggerID_DB_DEVICE_ERROR_DEVICE;
    tTriggerID mTriggerID_DB_DEVICE_ERROR_END_DEVICE;
    tTriggerID mTriggerID_DB_DEVICE_INSERTED;
    tTriggerID mTriggerID_DB_DEVICE_CHANGED_DISCONNECTIONREASON;
    tTriggerID mTriggerID_DB_DEVICE_CHANGED_FIRMWARE_VERSION;
    tTriggerID mTriggerID_DB_DEVICE_CHANGED_REPEAT_SUPPORT;
    tTriggerID mTriggerID_DB_DEVICE_CHANGED_SHUFFLE_SUPPORT;
    tTriggerID mTriggerID_DB_DEVICE_NOWPLAYINGLIST_AVAILABLE;
    tTriggerID mTriggerID_DB_DEVICE_CHANGED_DEVICEUUID;
    tTriggerID mTriggerID_DB_DEVICE_CHANGED_APPLEDEVICEMACADDRESS;
    tTriggerID mTriggerID_DB_DEVICE_CHANGED_DISCTYPE;
    tTriggerID mTriggerID_DB_DEVICE_CHANGED_APPLEDEVICEUSBSERIAL;
    tTriggerID mTriggerID_DB_DEVICE_CHANGED_DSTATE;
    tTriggerID mTriggerID_DB_DEVICE_CHANGED_DEVICESIZE;
    tDeviceID mDeviceIDForFolderStructureCleared;
    tTriggerID mTriggerID_DB_DEVICE_CHANGED_IMAGE_FILECOUNT;
    tTriggerID mTriggerID_DB_DEVICE_NUMBER_OF_PLAYABLE_FILES_NEWDEVICE;
    tTriggerID mTriggerID_DB_DEVICE_NUMBER_OF_PLAYABLE_FILES_SAMEDEVICE;
    tDeviceSerialNumber mDeviceAddressForReadyToPlay;
    tPEPlaybackState mPlaybackStateForReadyToPlay;
};

void ConvertErrorCodeFromReturnVal(tResult ret,tFileOperationErrorCodes& fileOperationErrorCodes);

#endif  //_CUSTOMCONTROL_H_

/** @} */
