#ifndef _UTF8STRING_H
#define _UTF8STRING_H
/**************************************************************************//**
* \file      Utf8String.h
* \ingroup   MediaPlayer
*
*            Interface header of class Utf8String.
*            See .h file for description.
*
* \remark    Copyright: 2012, Robert Bosch Car Multimedia GmbH, Hildesheim
* \remark    Scope:     MediaPlayer
* \remark    Authors:
*            - CM-AI/PJ-CD6 Xu Benny
******************************************************************************/
#include <string>
using namespace std;

#include <stdlib.h>
#include <memory.h>

typedef unsigned long UTF8Char;

class UTF8String
{
public:
	/**
	 * @brief Iterator definition for accessing this UTF8String class
	 */	
	class Iterator
	{
	protected:
		Iterator ();
	public:
		Iterator (unsigned long ulPos);
		virtual ~Iterator ();
	private:
		unsigned long m_ulPos;
	
	public:
		bool fine () const;
		unsigned long memory_addr() const;
		
		UTF8Char getChar() const;
				
		int getCharSize() const;
	
		Iterator& operator ++ ();		
		Iterator& operator -- ();	
		Iterator operator ++ (int nOff);		
		Iterator operator -- (int nOff);
		
		Iterator operator + (int nOff) const;
		Iterator operator - (int nOff) const;
		
		Iterator& operator += (int nOff);		
	
		bool operator == ( const Iterator & ref) const;
		bool operator != ( const Iterator & ref) const;
		bool operator < ( const Iterator & ref) const;
		bool operator > ( const Iterator & ref) const;	
		bool operator <= ( const Iterator & ref) const;
		bool operator >= ( const Iterator & ref) const;
		Iterator& operator= (const Iterator & ref);
	};

public:
	/******************************************************
	 * Constructors
	 ******************************************************/
	UTF8String();
	UTF8String(const UTF8String & u8str);
	UTF8String(const char* pStr);

	/******************************************************
	 * Destructors
	 ******************************************************/
	virtual ~UTF8String();

private:
	int 	m_nLen;
	int 	m_nBufSize;
	char* 	m_pBuffer;

private:
	bool reserve(int nSize);
	void init( const char* pBuffer, int nBufferSize);	
	void setCStr( const char* pCStr);
	void appendCStr( const char* pCStr);	
	void appendU8Char( UTF8Char u8Char);
	int getCharSize(UTF8Char u8Char) const;
	int compareCStrings(const char* pubCStr1, const char* pubCStr2) const;	
	unsigned long 	getCharAddress(int nU8CharId) const;
	int 			getByteIndex(int nU8CharId) const;	
	void erase_subcontent(unsigned long start, unsigned long end);

public:
	
	/**
	 * @brief Returns an iterator to the beginning of the string.
	 * @return return the iterator that point to the beginning of string.
	 */	
	Iterator begin( ) const;


	/**
	 * @brief 	 Returns an iterator to the end of the string. It can be seen to point to the
	             character position behind the last character of the string. Can be used to
	             determine whether an iterator has reached the end of the string.
	 * @return return the iterator that point to the end of string.
	 */		 
	Iterator end( ) const;

public:

	/**
	 * @brief To clear all UTF8 characters in this string, but do not release memory.
	 */			 
	void clear();

	/**
	 * @brief To check whether this UTF8 string is empty
	 * @return return true if this string is empty, otherwise, return false.
	 */		 
	bool empty() const;

	/**
	 * @brief To clear all UTF8 characters in this string, and release allocated memory
	 * @return return true if succeed, otherwise, return false.
	 */		 	 
	bool reset ();
	
	/**
	 * @brief Returns the number of UTF-8 characters in the string.
	 * @return Returns the number of UTF-8 characters in the string.
	 */		 
	int size() const;
	
	
	UTF8Char at (int nPos) const;
	
	void limit (const Iterator& it);
	
	/**************************************************************************//**
	* Erases the substring or characters defined by the given iterators. 
	* itSubBegin points to the first character of the substring, itSubEnd points behind the last
	* character of the substring. It's required that itSubBegin <= itSubEnd.
	* In case itSubBegin == itSubEnd, the string remains unchanged. The iterators
	* must be in the range given by member functions itBegin() and itEnd().
	******************************************************************************/
	bool erase(int nPos);
	bool erase(int nStart, int nSize);
	
	bool erase (const Iterator& it);
	bool erase (const Iterator& itStart, int nSize);
	bool erase (const Iterator& itStart, const Iterator& itEnd);
	
	/**
	 * @brief to find the specific UTF-8 character from beginning of string in sequence.
	 * @param u8Char the UTF8 character to be found
	 * @return if succeed, returns the iterator point to the character. Otherwise return end() of string
	 */	
	Iterator find (UTF8Char u8Char) const;
	
	/**
	 * @brief to find the specific alpha character from beginning of string in sequence.
	 * @param cChar the alpha character to be found
	 * @return if succeed, returns the iterator point to the character. Otherwise return end() of string
	 */		
	Iterator find (char cChar) const;
	
	/**
	 * @brief to find the specific UTF-8 character from specific position in sequence.
	 * @param itStart the start position to searching	 
	 * @param u8Char the UTF8 character to be found
	 * @return if succeed, returns the iterator point to the character. Otherwise return end() of string
	 */		
	Iterator find (const Iterator& itStart, UTF8Char u8Char) const;
	
	/**
	 * @brief to find the specific alpha character from beginning of string in sequence.
	 * @param itStart the start position to searching	 
	 * @param cChar the alpha character to be found
	 * @return if succeed, returns the iterator point to the character. Otherwise return end() of string
	 */		
	Iterator find (const Iterator& itStart, char cChar) const;	
	
	/**
	 * @brief to find the specific UTF-8 character in string from end to begin in reverse sequence.
	 * @param u8Char the UTF8 character to be found
	 * @return if succeed, returns the iterator point to the character. Otherwise return end() of string
	 */		
	Iterator rfind (UTF8Char u8Char) const;
	
	/**
	 * @brief to find the specific alpha character from end to begin in reverse sequence.
	 * @param cChar the alpha character to be found
	 * @return if succeed, returns the iterator point to the character. Otherwise return end() of string
	 */			
	Iterator rfind (char cChar) const;
	/**
	 * @brief to find the specific UTF-8 character from itStart to beginning of string in reverse sequence.
	 * @param itStart the start position to searching
	 * @param u8Char the UTF8 character for searching
	 * @return if succeed, returns the iterator point to the character. Otherwise return end() of string
	 */	
	Iterator rfind (const Iterator& itStart, UTF8Char u8Char) const;

	/**
	 * @brief to find the specific alpha character from itStart to beginning of string in reverse sequence.
	 * @param itStart the start position to search
	 * @param cChar the specific alpha character for searching
	 * @return if succeed, returns the iterator point to the character. Otherwise return end() of string
	 */			
	Iterator rfind (const Iterator& itStart, char cChar) const;	
	
	/**
	 * @brief to make all alpha characters to upper case.
	 */		 
	void upper();
	
	/**
	 * @brief to make all alpha characters to upper case.
	 * @return a pointer to the zero-terminated string held by this string object.
	 */	
	const char* c_str() const;
	
	/**
	 * @brief to get sub string specific beginning place to stopper.
	 * @param itStart the start position to get sub string
	 * @param itStop the end position (not include) for sub string	 
	 * @return sub string object.
	 */		
	UTF8String sub_str(const Iterator& itStart, const Iterator& itStop) const;
	
public:
	/**************************************************************************//**
	 * Overloaded operators
	 ******************************************************************************/	
	 
	 
	UTF8String& operator += (const char* pStr);
	UTF8String& operator += (UTF8Char  u8Char);
	UTF8String& operator += (const UTF8String& uStr);

	UTF8String operator + (const char* pStr) const;
	UTF8String operator + (UTF8Char  u8Char) const;
	UTF8String operator + (const UTF8String& uStr) const;	
		
	UTF8Char operator [](int nPos) const;
		
	bool operator == ( const UTF8String & ref) const;
	bool operator < ( const UTF8String & ref) const;
	bool operator > ( const UTF8String & ref) const;	
	bool operator <= ( const UTF8String & ref) const;
	bool operator >= ( const UTF8String & ref) const;
	bool operator != (const UTF8String& ref) const;
	
	UTF8String& operator= (const UTF8String & ref);	 
	UTF8String& operator= (const char* pRef); 
};

typedef UTF8String::Iterator Utf8StringIter;

#endif
/**End of file*/

