/**************************************************************************//**
* \file      Utf8FileName.h
* \ingroup   MediaPlayer
*
*            Declaration of class UTF8FileName which handles UTF-8 format File/PATH name.
*
* \remark    Copyright: 2012, Robert Bosch Car Multimedia GmbH, Hildesheim
* \remark    Scope:     MediaPlayer
* \remark    Authors:
*            - CM-AI/PJ-CD6 Xu Benny
******************************************************************************/
#ifndef _FILEPATHNAME_H_
#define _FILEPATHNAME_H_
#include <string>
using namespace std;

#include <Utf8String.h>

class UTF8FileName
{
	
public:
	/**
	 * @brief Constructors
	 */
	UTF8FileName();
	
	UTF8FileName(const UTF8FileName& ref);
	
	UTF8FileName( const char* pFilePathName );
	UTF8FileName( const string& sFilePathName );
		
	virtual ~UTF8FileName();
		
public:
	/**
	 * @brief to append extra string (pure file name or file extension) to an existing file name.
	 * @param str Extra string to append
	 * @return returns a reference of current UTF8FileName object
	 */
	UTF8FileName& 	append ( const string & str );
		
	/**
	 * @brief to append extra character to an existing file name.
	 * @param alphaChar
	 * @return returns a reference of current UTF8FileName object after appending
	 */
	UTF8FileName& 	append ( char alphaChar );	
		
	/**
	 * @brief 		to append a parent path to current pure file/path name
	 * @param path 	the parent path name to append
	 * @return  returns a reference to current UTF8FileName object
	 */
	UTF8FileName& 	appendPath ( const string & path );
		
	/**
	 * @brief  used to check that whether current file name ends with specific string
	 * @param str The pure file name or extension name to check with
	 * @return If current file name ends with specific string, return true. Otherwise, return false
	 */
	bool endsWith ( const string & str ) const;
	
	
	/**
	 * @brief To set or reset a UTF8FileName object by using a string or buffer
	 * @param sFileName 
	 * @return Return true if succeed, otherwise, return false.
	 */
	bool setFileName(const string& sFileName);
	bool setFileName(const string& sPathName, const string& sFileName);
	
public:
	/**
	 * @brief to reset a UTF8FileName object and make it to be empty
	 */
	void reset();
	
	/**
	 * @brief to check that whether current UTF8FileName object is empty
	 * @return Return true if it is empty, otherwise, return false
	 */
	bool isEmpty() const;
public:
	/**
	 * @brief to get parent directory name
			  Example:
				"/home/myfile.dat"    --> "/home"
				"/home/mydir/"    	  --> "/home"
				"/home/mydir"         --> "/home"
	 * @return the parent path name.
	 */
	UTF8FileName getParentDir() const; 
	
	/**
	 * @brief to get the pure path or file name
			  Example:
				"/home/myfile.dat"    --> "myfile.dat"
				"/home/mydir/"    	  --> "mydir"
				"/home/mydir"         --> "mydir"
	 * @return  pure path/file name
	 */
	UTF8FileName getPureName() const;
	
	/**
	 * @brief check whether this object is a child of a specific path, or a child of its child
	 * @param parentPath The parent path name to check with
	 * @return Return true if this one is a child of parentPath,or child of  its child, otherwise, return false
	 */
	bool	isChildOf(const UTF8FileName& parentPath) const;

public:
	UTF8FileName& operator= (const UTF8FileName& ref);
	
	bool operator == (const UTF8FileName& ref) const;
	bool operator != (const UTF8FileName& ref) const;
	
public:
	/**
	 * @brief Get c type string of this file name
	 * @return 
	 */
	const char* getCString() const;
	
	/**
	 * @brief get std::string format of this file name
	 * @return 
	 */
	string  getString() const;
	
	/**
	 * @brief get UTF8String format of this file name
	 * @return
	 */
	UTF8String  getUTF8String() const;


private:	
	bool reserveMemory(int nSize);
	void removeEndFlag();
	
	string getPureName_cstr() const;
	
	void validateFileName();
	bool validateString(string & str);
	
	void stripLeadingWhiteSpace();
	void stripTrailingWhiteSpace();
	
private:
	UTF8String m_fileName;
};
#endif
/****EOF****/
