/**
 * @addtogroup Simulation
 *
 * @author Stefan Scherber
 *
 * captures the outgoing property update methods from Generic Media player
 * For the module tests this output wrapper sends the property updates to testcases
 * @{
 */

#ifndef SIMULATIONOUTPUTWRAPPER_H_
#define SIMULATIONOUTPUTWRAPPER_H_

#include <iostream>
using namespace std;

#include "MessageQueue.h"
#include "OutputWrapper.h"

#include <glib.h>
#include "EventDispatcher.h"
#include "EventDefinitions.h"
#include "test_util.h"
#include <pthread.h>
#include <sys/types.h>
#include <sys/msg.h>
#include <sys/ipc.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>

#define MQMSGQUEUE "/tmp/MQMSGQUEUE"
#include "Lock.h"



struct tMessage
{
	int type;
	tGeneralString EventName;
	cTestData testData;
};

/**
 * Pure virtual class. Is used by the media player components to send messages to external components.
 */
class SimulationOutputWrapper : public OutputWrapper
{

public:

	
    SimulationOutputWrapper(tComponentID componentId);
    ~SimulationOutputWrapper();

	pthread_mutex_t mutexvar;
	FILE *fMsgQueuepointer;
	
	void sendMsgQueue(const char *EventString, int eventId, cTestData *obj);

	static void * fEvent(void *oSimulatorPtr);


	//added for Tests for Roadmap 16014 - Note: could not use existing Listener-mechanism
	//tbd.: shift to its own class for better reuse
	
    Lock lockSimQ;
	typedef enum
	{
	    eSend_onCreateMediaPlayerListSearchString_MethodReturnFirst  = 1,
		eSend_onCreateMediaPlayerListSearchString_MethodReturnMiddle = 2,
	    eSend_onCreateMediaPlayerListSearchString_MethodReturnLast   = 3,
	    eSend_onCreateMediaPlayerListSearchString_MethodReturnAbort  = 4	
	}tenFctType;
	
	enum
	{
        OK	   = 0,
		ERROR  = 1 //in iMessageQueueWait this might be also a timout
	};
	tBool testMsqQIsInUse();
	int   testMsgQInit();
    int   testMsqQDelete();
    int   testMsgQGetMsgsInQueue();
	int   testMsgQSend(IN const void *pMsg, IN int iLength); //e.g. SimulationOutwrapper: function sends its parameter to test queue
	void* testMsgQWait(OUT unsigned int &uLength, IN unsigned int timeout_ms); //cpp unit test waits on this
	void  testMsgQDropRecBuf(OUT void *pu8Buffer);
    //<--Roadmap 16014

    /**
     * This function request an audio channel from the AVManager of the system.  Must be overwritten by the derived class.
     *
     * @param[in] sourceNr
     * @param[in] logicalAVChannel
     * @return < 0: error, = 0: OK
     */
    tResult RequestAVActivation(const tSourceNr sourceNr, const tLogicalAVChannel logicalAVChannel, const tSource source=0, const tDeviceID deviceID = DEVICE_ID_NOT_SET);
    tResult UpdateDeviceConnection(const tDeviceID deviceID, const tConnectionReason connectionReason);
    /**
     * This function is called by the mediaplayer core when playtime property changed.
     * Can be overwritten by the derived class.
     * @return < 0: error, = 0: OK
     */
    tResult UpdatePlaytime();

    /**
    * This function is called by the mediaplayer core when playbackstate property changed.
    * Can be overwritten by the derived class.
    * @return < 0: error, = 0: OK
    */
    tResult UpdatePlaybackState();

    /**
     * This function is called by the mediaplayer core when nowplaying property changed.
     * Can be overwritten by the derived class.
     * @return < 0: error, = 0: OK
     */
    tResult UpdateNowPlaying();

    /**
     * This function is called when the active device was changed.
     * Can be overwritten by the derived class.
     * @return < 0: error, = 0: OK
     */
    tResult UpdateActiveMediaDevice(){return 0;};

    /**
     * This function is called when any device changed.
     * Can be overwritten by the derived class.
     * @return < 0: error, = 0: OK
     */
    tResult UpdateMediaPlayerDeviceConnections(){return 0;};


    /**
     * This function is called when favorite table changes
     * Can be overwritten by derived class
     * @param[in] listChangeType type of the list changes
     * @param[in] listSize list size
     * @param[in] favID changed favorite ID
     * @return < 0: error, = 0: OK
     */
    tResult SendFavoritesChanged(tListChangeType listChangeType, tListSize listSize, tFavoriteID favID){return 0;}

    /**
     * This function is called by framework to get the changed values of the favorite table
     * @param[out] listChangeType type of the list changes
     * @param[out] listSize list size
     * @param[out] favID changed favorite ID
     * @return < 0: error, = 0: OK
     */
    tResult GetFavoriteChanged(tListChangeType &listChangeType, tListSize &listSize, tFavoriteID &favID){return 0;}

    /**
     * This function is called when a list changes
     * Can be overwritten by derived class
     * @param[in] listID ID of the changed list
     * @param[in] listSize size of list
     * @return < 0: error, = 0: OK
     */
    tResult SendMediaPlayerListChanged(tListID listID, tListSize listSize){return 0;}

    /**
     * This function is called by framework to get the list changes
     * @param[out] listID ID of the changed list
     * @param[out] listSize size of list
     * @return < 0: error, = 0: OK
     */
    tResult GetMediaPlayerListChanged(tListID &listID, tListSize &listSize){return 0;}


    /**
     * This function is called by MediaPlayer awhen the result of GetMediaObjectAlbumArtInfo is calculated
     * Must be overwritte by derived class
     * @param[in] mimeType string contains the mimeType
     * @param[in] userContext Context to assign the answer to the request
     * @return < 0: error, = 0: OK
     */
    tResult SendGetMediaObjectAlbumArtInfoAnswer(tMimeType mimeType, tSize size, tUserContext userContext);

	
    /**
     * This function is called by MediaPlayer when the result of GetMediaObjectAlbumArtInfo cannot provided
     * Must be overwritte by derived class
     * @param[in] errorCode
     * @param[in] userContext Context to assign the answer to the request
     * @return < 0: error, = 0: OK
     */
    tResult SendGetMediaObjectAlbumArtInfoError(tErrorCode errorCode, tUserContext userContext);

    /**
     * This function is called by MediaPlayer when the result of GetMediaObjectAlbumArt is calculated
     * Must be overwritte by derived class
     * @param[in] imageData image data
	 * @param[in] imageSize size of image in bytes
     * @param[in] userContext Context to assign the answer to the request
     * @return < 0: error, = 0: OK
     */
    tResult SendGetMediaObjectAlbumArtAnswer(tImageData imageData, tImageSize imageSize, tUserContext userContext);

    /**
     * This function is called by MediaPlayer when the result of GetMediaObjectAlbumArt cannot provided
     * Must be overwritte by derived class
     * @param[in] errorCode
     * @param[in] userContext Context to assign the answer to the request
     * @return < 0: error, = 0: OK
     */
    tResult SendGetMediaObjectAlbumArtError(tErrorCode errorCode, tUserContext userContext);


    /* Functions for IPOD authentication */

    /**
    * Pure virtual function. Must be written by derived class
    * @param[in] bInit if TRUE initialize authentication; if true deinitalize authentication
    * @return IPOD_OK or IPOD_ERROR
    */
    tS32 AuthenticationInit(tBool bInit) {return 0;};

    /**
    * Pure virtual function. Must be written by derived class
    * @param[out] certLength  length of certificate
    * @param[out] certData certificate
    * @return IPOD_OK or IPOD_ERROR
    */
    tS32 AccessoryCertificate(tU16 *certLength, tU8 *certData) {return 0;};

    /**
    * Pure virtual function. Must be written by derived class
    * @param[in] certLength  length of certificate
    * @param[in] certData certificate
    * @return IPOD_OK or IPOD_ERROR
    */
    tS32 IPodCertificate(tU16 certLength, tU8 *certData) {return 0;};

    /**
    * Pure virtual function. Must be written by derived class
    * @param[in] sigDataLen  length of signature
    * @param[in] sigData signature
    * @return IPOD_OK or IPOD_ERROR
    */
    tS32 SetSignature(tU16 sigDataLen, tU8 *sigData) {return 0;};

    /**
    * Pure virtual function. Must be written by derived class
    * @param[in] ResponseBuffer  challange data from the IPOD
    * @param[out] SignatureDataLength length of the signature
    * @param[out] SignatureData signature
    * @return IPOD_OK or IPOD_ERROR
    */
    tS32 GenSignature(const tU8  *ResponseBuffer, tU16 *SignatureDataLength, tU8 *SignatureData) {return 0;};

    /**
    * Pure virtual function. Must be written by derived class
    * @param[out] ChallengeDataLength  challange data length
    * @param[out] ChallengeData challange
    * @return IPOD_OK or IPOD_ERROR
    */
    tS32 GetChallenge(tU16 *ChallengeDataLength, tU8 *ChallengeData) {return 0;};

    /**
    * Pure virtual function. Must be written by derived class
    * @param[out] DeviceID device ID
    * @return IPOD_OK or IPOD_ERROR
    */
    tS32 DeviceID (tU32 *DeviceID) {return 0;};

    /**
    * Pure virtual function. Must be written by derived class
    * @param[out] majorVer  Major version of firmware
    * @param[out] minorVer  Minor version of firmware
    * @return IPOD_OK or IPOD_ERROR
    */
    tS32 FirmwareVersion (tU8 *majorVer, tU8 *minorVer) {return 0;};

    /**
    * Pure virtual function. Must be written by derived class
    * @param[out] majorVer  Major version of protocol
    * @param[out] minorVer  Minor version of protocol
    * @return IPOD_OK or IPOD_ERROR
    */
    tS32 AuthenticationProtocolVersion (tU8 *majorVer, tU8 *minorVer) {return 0;};

    /**
    * Pure virtual function. Must be written by derived class
    * @return IPOD_OK: passed or IPOD_ERROR: failed
    */
    tS32 IPodAuthenticationSelftest (tU8 *certificate, tU8 *private_key, tU8 *ram_check, tU8 *checksum) {return 0;};

    /**
    * This function requests for connected devices list with DeviceManager
    * In case of SimulationOutputWrapper , it adds the simulated device into GMP core
    * @return =0 : passed , else : failed
    */
    tResult RequestConnectedDevices();

	int test();

	void settestId(guint32 u32nId);


	//-->Roadmap 16014 'full text search'

	/**
	  * helper function to write parameters to queue
	  */
	void helperPackMethod_Send_onCreateMediaPlayerListSearchString(IN tenFctType eFctType, IN tListID listID, IN tListSize listSize, IN tDeviceID deviceID);

	
    /**
    * virtual function of base classes changed to write params to message queue. Test is listening to that queue to check correct paramenter 
    */
	virtual tResult Send_onCreateMediaPlayerListSearchString_MethodReturnFirst(IN tListID listIDParent,IN tListID listID, IN tListSize listSize, IN tDeviceID deviceID);
    /**
    * virtual function of base classes changed to write params to message queue. Test is listening to that queue to check correct paramenter 
    */
	virtual tResult Send_onCreateMediaPlayerListSearchString_MethodReturnMiddle(IN tListID listIDParent,IN tListID listID, IN tListSize listSize, IN tDeviceID deviceID);
    /**
    * virtual function of base classes changed to write params to message queue. Test is listening to that queue to check correct paramenter 
    */
	virtual tResult Send_onCreateMediaPlayerListSearchString_MethodReturnLast(IN tListID listIDParent, IN tListID listID, IN tListSize listSize, IN tDeviceID deviceID);

	/**
    * virtual function of base classes changed to write params to message queue. Test is listening to that queue to check correct paramenter 
    */
	virtual tResult Send_onCreateMediaPlayerListSearchString_MethodReturnAbort(IN tListID listIDParent, IN tListID listID, IN tListSize listSize, IN tDeviceID deviceID);
	//<--Roadmap 16014 'full text search'

    //-->Roadmap 16003 : CD Ripping With Gracenote
    /**
    * This function is called when the delete media content operation is completed.
    * This function calls method return after fi method 'DeleteMediaContentByMediaObjectUrls' or 'DeleteMediaContentByMediaObjectTags' or  'DeleteMediaContentByFilterTags' is called
    * @param [IN] deleteMediaContentType is type of 'deletemediacontent' interface used
    * @param [IN] fileOperationErrorCodes is type of 'deletemediacontent' error code
    * @param [IN] numofFilesDeleted is number of files got deleted
    * @return < 0: error, = 0: OK
    */
    virtual tResult Send_OnDeleteMediaContentMethodResult(IN tDeleteMediaContentType deleteMediaContentType, IN tFileOperationErrorCodes fileOperationErrorCodes, IN tU16 numofFilesDeleted );


    /**
    * This function is called when the edit media metadata operation is completed.
    * This function calls method return after fi method 'EditMetaDataByMediaObjectUrl' or 'EditMetaDataByMediaObjectTag'  is called
    * @param [IN] editMetadatatType is type of metadata interface used
    * @param [IN] fileOperationErrorCodes is type of editmetadata error code
    * @return < 0: error, = 0: OK
    */
    virtual tResult Send_OnEditMetadataMethodResult(IN tEditMetadatatType editMetadatatType, IN tFileOperationErrorCodes fileOperationErrorCodes  );


    /**
    * This function is called when the add entries into playlist  operation is completed.
    * This function calls method return after fi method 'AddFilenameToInternalPlaylist'  is called
    * @param [IN] playlistOperationErrorCodes is type of playlist operation result code
    * @return < 0: error, = 0: OK
    */
    virtual tResult Send_OnAddFileNamesToPlaylistMethodResult(IN tResult playlistOperationErrorCodes);



    /**
     * This function is called when the delete entries into playlist  operation is completed.
     * This function calls method return after fi method 'DeleteEntriesFromPlaylist'  is called
     * @param [IN] playlistOperationErrorCodes is type of playlist operation result code
     * @return < 0: error, = 0: OK
     */
    virtual tResult Send_OnDeleteEntriesFromPlaylistMethodResult(IN tResult playlistOperationErrorCodes );


    /**
    * The below functions(StartFileOperation,isFileOperationCompleted) are helpers for the unit test cases. This will help us to identify an operation done on a thread function completed or not
    */
    virtual void StartFileOperation();
    virtual bool isFileOperationCompleted(tResult &res);

    //<--Roadmap 16003 : 'CD Ripping With Gracenote'


    virtual tResult UpdateRippingStatus();
    virtual tResult UpdateAutoRipping();

    MessageQueue mMessageQueue;




protected:

		   //added for Tests for Roadmap 160014 - Note: could not use existing Listener-mechanism
           //number  of messages posted in the message queue.
           int m_CountMsges;
		   int m_MsgSendNum,m_MsgReceiveNum;
           MessageQueue *m_pMsgQueue;

           //>--Roadmap 16003 : 'CD Ripping With Gracenote'
           bool isFileListOperationInProgress;
           tResult m_Result;
           //<--Roadmap 16003 : 'CD Ripping With Gracenote'

private:
    guint32 m_ntestId;
    guint32 m_nGetAlbuArtinfoAnswer;
};

#endif

/** @} */
