/**
 * @author Stefan Scherber
 *
 * Implentation of CCAClientHandlerDeviceManager
 *
 */

/******************************************************************************/
/*                                                                            */
/* INCLUDES                                                                   */
/*                                                                            */
/******************************************************************************/

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_mp.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_CLIENTHANDLER_DEVICEMANAGER
#include "trcGenProj/Header/FC_MediaPlayer_clienthandler_DeviceManager.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_CLIENTHANDLER_DEVICEMANAGER
#endif

#include "FC_MediaPlayer_main.h"
#include "FC_MediaPlayer_clienthandler_DeviceManager.h"
#include "MediaPlayerInterface.h"
#include "FunctionTracer.h"


/******************************************************************************/
/*                                                                            */
/* DEFINES                                                                    */
/*                                                                            */
/******************************************************************************/

// Version defines for used service
#define DEVICEMANAGER_FI_MAJOR_VERSION  MPLAY_DEVICEMANAGERFI_C_U16_SERVICE_MAJORVERSION
#define DEVICEMANAGER_FI_MINOR_VERSION  MPLAY_DEVICEMANAGERFI_C_U16_SERVICE_MINORVERSION


/******************************************************************************/
/*                                                                            */
/* CCA MESSAGE MAP                                                            */
/*                                                                            */
/******************************************************************************/

BEGIN_MSG_MAP(fc_mediaplayer_tclClientHandler_DeviceManager, ahl_tclBaseWork)
    ON_MESSAGE_SVCDATA(DEVMGR_NOTIFYCONNECTION           ,AMT_C_U8_CCAMSG_OPCODE_STATUS       ,onDeviceManagerNotifyConnection)
    ON_MESSAGE_SVCDATA(DEVMGR_GETALLCONNECTEDDEVICES     ,AMT_C_U8_CCAMSG_OPCODE_METHODRESULT ,onRequestConnectedDevicesMethodResult)
    ON_MESSAGE_SVCDATA(DEVMGR_NOTIFYSLOTSTATEOPTICALDISC ,AMT_C_U8_CCAMSG_OPCODE_STATUS       ,onDeviceManagerNotifySlotStateOpticalDisc)
    ON_MESSAGE_SVCDATA(DEVMGR_EJECTOPTICALDISC           ,AMT_C_U8_CCAMSG_OPCODE_METHODRESULT ,onOpticalDiscEjectMethodResult)
    ON_MESSAGE_SVCDATA(DEVMGR_NOTIFYOPTICALDEVICECDINFO  ,AMT_C_U8_CCAMSG_OPCODE_STATUS       ,onDeviceManagerNotifyCDInfo)

END_MSG_MAP()

/******************************************************************************/
/*                                                                            */
/* METHODS                                                                    */
/*                                                                            */
/******************************************************************************/

/*******************************************************************************
*
* FUNCTION: fc_mediaplayer_tclClientHandler_DeviceManager::
*             fc_mediaplayer_tclClientHandler_DeviceManager(fc_mediaplayer_tclApp* poMainAppl)
*
* DESCRIPTION: Constructor.
*
*              Create an object of the base class
*              ahl_tclBaseOneThreadClientHandler with a pointer to this
*              application, the to be used service identifier and the service
*              version as parameters.
*
* PARAMETER: [IN] poMainAppl = Pointer to the object of this application.
*
* RETURNVALUE: None.
*
*******************************************************************************/
fc_mediaplayer_tclClientHandler_DeviceManager::
  fc_mediaplayer_tclClientHandler_DeviceManager(fc_mediaplayer_tclApp* poMainAppl) //finished 100%
    : ahl_tclBaseOneThreadClientHandler(
      /* Application Pointer          */ poMainAppl,
      /* ID of used Service           */ CCA_C_U16_SRV_DEVICEMANAGER,
      /* MajorVersion of used Service */ DEVICEMANAGER_FI_MAJOR_VERSION,
      /* MinorVersion of used Service */ DEVICEMANAGER_FI_MINOR_VERSION )
{
    ENTRY
    _tclRegisterPair.bAddPair(AMT_C_U32_STATE_OFF, AMT_C_U32_STATE_NORMAL);
    _tclUnregisterPair.bAddPair(AMT_C_U32_STATE_NORMAL, AMT_C_U32_STATE_OFF);
    // in GEN3 dependecy to IPOD-Auth is not available. Property can be registered during startup
    #if defined(TARGET_BUILD_GEN3) || defined(LSIM_GEN2)
    vAddAutoRegisterForProperty(DEVMGR_NOTIFYCONNECTION);
    vAddAutoRegisterForProperty(DEVMGR_NOTIFYSLOTSTATEOPTICALDISC);
    vAddAutoRegisterForProperty(DEVMGR_NOTIFYOPTICALDEVICECDINFO);
    #endif

    ETG_TRACE_COMP(("fc_mediaplayer_tclClientHandler_DeviceManager: MPLAY_DEVICEMANAGERFI_C_U16_SERVICE_MAJORVERSION: %d",(tU16)MPLAY_DEVICEMANAGERFI_C_U16_SERVICE_MAJORVERSION));
    ETG_TRACE_COMP(("fc_mediaplayer_tclClientHandler_DeviceManager: MPLAY_DEVICEMANAGERFI_C_U16_SERVICE_MAJORVERSION: %d",(tU16)MPLAY_DEVICEMANAGERFI_C_U16_SERVICE_MINORVERSION));

#ifdef VARIANT_S_FTR_ENABLE_DEVMGR_IMPROVEDIF_NO_REDUNDAND_MSGS
    if((tU16)MPLAY_DEVICEMANAGERFI_C_U16_SERVICE_MAJORVERSION >=u16MajorVersionOptimisedInterface)
    {
        ETG_TRACE_COMP(("fc_mediaplayer_tclClientHandler_DeviceManager: Version of optimised interface VD_DVM is used"));
    }
#endif

}

/*******************************************************************************
*
* FUNCTION: fc_mediaplayer_tclClientHandler_DeviceManager::
*             ~fc_mediaplayer_tclClientHandler_DeviceManager(tVoid)
*
* DESCRIPTION: Destructor.
*
* PARAMETER: None.
*
* RETURNVALUE: None.
*
*******************************************************************************/
fc_mediaplayer_tclClientHandler_DeviceManager::
  ~fc_mediaplayer_tclClientHandler_DeviceManager(tVoid) //finished 100%
{
    ENTRY
}



tVoid fc_mediaplayer_tclClientHandler_DeviceManager::onDeviceManagerNotifyConnection(amt_tclServiceData* poMessage) // finished: 100%
{
    ENTRY
    tResult result = MP_NO_ERROR;
    unsigned int index = 0;
    FIDEVMGR_NOTIFYCONNECTIONSTATUS oFiDataObjectStatus;
    fi_tclVisitorMessage oVisitorMsg(poMessage);

    if (OSAL_ERROR != oVisitorMsg.s32GetData(oFiDataObjectStatus, DEVICEMANAGER_FI_MAJOR_VERSION))
    {
        vector<tDeviceProperty> deviceProperties;
        //bpstl::vector<FIDEVMGR_DEVICEPROPERTY>::iterator it;
        std::vector<FIDEVMGR_DEVICEPROPERTY>::iterator it;
        for(it = oFiDataObjectStatus.Devicetype.begin(); it < oFiDataObjectStatus.Devicetype.end(); it++)
        {
            tDeviceProperty devicePropertyItem;
            strncpy_r(devicePropertyItem.serialNumber,        it->DeviceSerialNo.szValue,  sizeof devicePropertyItem.serialNumber);         //tDeviceSerialNumber
            strncpy_r(devicePropertyItem.USBPortNumber,       it->USBPortNo.szValue,       sizeof devicePropertyItem.USBPortNumber);        //tUSBPortNumber
            strncpy_r(devicePropertyItem.mountPoint,          it->MountPoint.szValue,      sizeof devicePropertyItem.mountPoint);           //tMountPoint
            ConvertDeviceUnsupportedReasonTypeToInternal(OUT devicePropertyItem.deviceUnsupportedReason, IN it->UnsupportedReason); //tDeviceUnsupportedReason
            ConvertDeviceTypeToInternal(OUT devicePropertyItem.deviceType, IN it->DeviceType, IN devicePropertyItem.deviceUnsupportedReason, IN devicePropertyItem.mountPoint); //tDeviceType
            ConvertDeviceTypeToConnectionType(OUT devicePropertyItem.connectionType, IN devicePropertyItem.deviceType);
            strncpy_r(devicePropertyItem.deviceVersion,       it->DeviceVersion.szValue,   sizeof devicePropertyItem.deviceVersion);        //tDeviceVersion
            strncpy_r(devicePropertyItem.deviceSyspath,       it->DevicePath.szValue,      sizeof devicePropertyItem.deviceSyspath);        //tPath
            strncpy_r(devicePropertyItem.deviceName,          it->DeviceName.szValue,      sizeof devicePropertyItem.deviceName);           //tDeviceName
            strncpy_r(devicePropertyItem.accessoryName,       it->AccessoryName.szValue,   sizeof devicePropertyItem.accessoryName);        //tAccessoryName
            strncpy_r(devicePropertyItem.accessoryName2,      it->AccessoryName2.szValue,  sizeof devicePropertyItem.accessoryName2);       //tAccessoryName2
            ConvertDeviceConnectionStatusToInternal(OUT devicePropertyItem.connectStatus, IN it->DeviceConnectStatus);                      //tDMConnectStatus
#ifdef VARIANT_S_FTR_ENABLE_DEVMGR_GEN3_CCA
            devicePropertyItem.sizeInByte                   = it->TotalSize;                                                                //tTotalSizeByte
            devicePropertyItem.vendorID                     = it->VendorID;                                                                 //tVendorID
            strncpy_r(devicePropertyItem.manufacturer,        it->Manufacturer.szValue,    sizeof devicePropertyItem.manufacturer);         //tManufacturer
            devicePropertyItem.productID                    = it->ProductID;                                                                //tProductID

            devicePropertyItem.totalSizekB                  = it->TotalSize_kB;                                                             //tTotalSize_kB
            devicePropertyItem.usedSizekB                   = it->UsedSize_kB;                                                              //tUsedSize_kB
            devicePropertyItem.freeSizekB                   = it->FreeSize_kB;                                                              //tFreeSize_kB
            devicePropertyItem.partitionNumber              = it->ParitionNum;                                                              //tPartitionNum

            ConvertFileSystemTypeToInternal(OUT devicePropertyItem.fileSystemType, IN it->FileSystemType);                                  //tFileSystemType

            strncpy_r(devicePropertyItem.deviceSpeed,         (const char*)"-"/*it->DeviceSpeedUsb.szValue*/,   sizeof devicePropertyItem.deviceSpeed);         //tDeviceSpeed
            strncpy_r(devicePropertyItem.protocolUsb,         (const char*)"-"/*it->ProtocolUsb.szValue*/,      sizeof devicePropertyItem.protocolUsb);         //tProtocolUsb
            strncpy_r(devicePropertyItem.baseClassCodeUsb,    (const char*)"-"/*it->BaseClassCodeUsb.szValue*/, sizeof devicePropertyItem.baseClassCodeUsb);    //tBaseClassCodeUsb
            strncpy_r(devicePropertyItem.subBaseClassCodeUsb, (const char*)"-"/*it->SubClassCodeUsb.szValue*/,  sizeof devicePropertyItem.subBaseClassCodeUsb); //tSubClassCodeUsb
#endif

            //deviceProperties contain last message at index 0 and all connected device at index=1 ... last index
            //if we hand over the whole list to the database it is able to see which device has been newly added and which
            //of the connected devices have a new status
            //from index 1 to last element connected devices are listed a removed would appear at index 0 only
            //that is why we send index 0 info if removed has been received otherwhise from index 1

            if(isDeviceEmpty(devicePropertyItem))
            {
                //at startup Mediaplayer registers at VD_DeviceManager and thus may receive an empty device. This is not thought to be added to the database
                ETG_TRACE_USR1(("fc_mediaplayer_tclClientHandler_DeviceManager::onDeviceManagerNotifyConnection: received empty device"));
            }
            //first element contains last message if connecton status is removed - then DBMAnager has to be also informed about this item
            else
            {
                //vector [0...last] contains devices and their state. There might be several changes at once notified
#ifdef VARIANT_S_FTR_ENABLE_DEVMGR_IMPROVEDIF_NO_REDUNDAND_MSGS //role out of changed interface for dedicated projects only
                if((tU16)MPLAY_DEVICEMANAGERFI_C_U16_SERVICE_MAJORVERSION >=u16MajorVersionOptimisedInterface)
                {
                    TraceDevicePropertyItem(eOnDeviceManagerNotifyConnection, devicePropertyItem, index);
                    deviceProperties.push_back(devicePropertyItem);
                    index++;
                }
                //old interface: vector[0] contains latest change i.e. if connection is notified in vector[0] same proporty occurs also somewhere between vector[1..last] i.e. double occurence of connected devices
                //vector[1...last] contains connected devices i.e. no devices with status REMOVED
                else
#endif
                {
                    if((it == oFiDataObjectStatus.Devicetype.begin())   )
                    {
                        TraceDevicePropertyItem(eOnDeviceManagerNotifyConnection, devicePropertyItem, index);

                        ETG_TRACE_USR1(("fc_mediaplayer_tclClientHandler_DeviceManager::onDeviceManagerNotifyConnection: care for 1st element"));
                        //first element contains last message if connecton status is removed - then DBMAnager has to be also informed about this item
                        if(devicePropertyItem.connectStatus == DM_CS_REMOVED_BY_USR )
                        {
                            deviceProperties.push_back(devicePropertyItem);
                        }
                    }
                    //index ==1 ...end contains info about currently connected devices - think about replacing the first element totally
                    else
                    {
                        deviceProperties.push_back(devicePropertyItem);
                    }
                }
            }
        }

        vector<tDeviceInfo> deviceInfos;
        ConvertDevicePropertiesToDeviceInfos(OUT deviceInfos, IN deviceProperties);
        tUndervoltage undervoltage = oFiDataObjectStatus.Undervoltage;
        ETG_TRACE_USR4(("onDeviceManagerNotifyConnection: =================="));
        ETG_TRACE_USR4(("onDeviceManagerNotifyConnection: undervoltage:%d", undervoltage));
        ETG_TRACE_USR4(("onDeviceManagerNotifyConnection: =================="));

        result = MediaPlayerInterface::GetInstance().DeviceChanged(IN undervoltage, IN deviceInfos);
    }
    else
    {
        NORMAL_M_ASSERT_ALWAYS();
    }
    if (MP_NO_ERROR != result)
    {
        ETG_TRACE_ERR(("onDeviceManagerNotifyConnection: Device status cannot be set: %d", result));
    }
    oFiDataObjectStatus.vDestroy();
}


tBool  fc_mediaplayer_tclClientHandler_DeviceManager::isDeviceEmpty(const tDeviceProperty deviceProperty)
{
    ENTRY

    tU8  emptyCount = 0;
    tBool bResult = false;

    if(deviceProperty.serialNumber[0] =='\0')
    {
        emptyCount++;
        ETG_TRACE_USR4(("isDeviceEmpty: empty serialNumber"));
    }
    if(   (emptyCount > 0)
       && (deviceProperty.USBPortNumber[0] =='\0'))
    {
        emptyCount++;
        ETG_TRACE_USR1(("isDeviceEmpty: empty USBPortNumber"));
    }
    if(   (emptyCount > 0)
       && (deviceProperty.deviceVersion[0] =='\0'))
    {
        emptyCount++;
        ETG_TRACE_USR1(("isDeviceEmpty: empty deviceVersion"));
    }
    if(   (emptyCount > 0)
       && (deviceProperty.deviceName[0] =='\0'))
    {
        emptyCount++;
        ETG_TRACE_USR1(("isDeviceEmpty: empty deviceName"));
    }
    if(   (emptyCount > 0)
       && (deviceProperty.accessoryName[0] =='\0'))
    {
        emptyCount++;
        ETG_TRACE_USR1(("isDeviceEmpty: empty accessoryName"));
    }
    if(   (emptyCount > 0)
       && (deviceProperty.mountPoint[0] =='\0'))
    {
        emptyCount++;
        ETG_TRACE_USR1(("isDeviceEmpty: empty mountPoint"));
    }
    if(   (emptyCount > 0)
       && (deviceProperty.deviceType ==DTY_UNKNOWN))
    {
        emptyCount++;
        ETG_TRACE_USR1(("isDeviceEmpty: deviceType ==  DTY_UNKNOWN"));
    }
    if(   (emptyCount > 0)
       && (deviceProperty.connectStatus ==DM_CS_UNDEFINED))
    {
        emptyCount++;
        ETG_TRACE_USR1(("isDeviceEmpty: connectStatus == DM_CS_UNDEFINED"));
    }

    //check if all criteria match
    //Note: partial filled devices should not happen at all this would be an error with route cause in VD_DeviceManager
    if(emptyCount >=8)
    {
        bResult = TRUE;
    }
    ETG_TRACE_USR1(("isDeviceEmpty:0x%x",bResult));
    return bResult;
}

void fc_mediaplayer_tclClientHandler_DeviceManager::ConvertDeviceTypeToInternal(tDeviceType &deviceTypeInternal, const FIDEVMGR_E8_DEVICE deviceType, tDeviceUnsupportedReason deviceUnsupportedReason, tMountPoint deviceMountpointInternal) //finished: 100%
{
    ENTRY_INTERNAL
    switch (deviceType.enType)
    {
    case FIDEVMGR_E8_DEVICE::FI_EN_DTY_USB:
        deviceTypeInternal = DTY_USB;
        break;
    case FIDEVMGR_E8_DEVICE::FI_EN_DTY_SDCARD:
        deviceTypeInternal = DTY_SD;
        break;
    case FIDEVMGR_E8_DEVICE::FI_EN_DTY_IPOD:
        deviceTypeInternal = DTY_IPOD;
        break;
    case FIDEVMGR_E8_DEVICE::FI_EN_DTY_IPHONE:
        deviceTypeInternal = DTY_IPHONE;
        break;
    case FIDEVMGR_E8_DEVICE::FI_EN_DTY_MTP:
        deviceTypeInternal = DTY_MTP;
        break;
    case FIDEVMGR_E8_DEVICE::FI_EN_DTY_MSZUNE:
        deviceTypeInternal = DTY_MSZUNE;
        break;
    case FIDEVMGR_E8_DEVICE::FI_EN_DTY_CDROM:
        deviceTypeInternal = DTY_CDROM;
        break;
    case FIDEVMGR_E8_DEVICE::FI_EN_DTY_CDDA://Roadmap 15002 CDaudio Detection
        deviceTypeInternal = DTY_CDDA;
        break;
    case FIDEVMGR_E8_DEVICE::FI_EN_DTY_DVD_DRIVE:
        deviceTypeInternal = DTY_DVD_DRIVE;
        break;
    case FIDEVMGR_E8_DEVICE::FI_EN_DTY_DIGITALAUX:
    case FIDEVMGR_E8_DEVICE::FI_EN_DTY_NOT_SUPPORTED:
        if((OK != deviceUnsupportedReason) || (!(strncmp(deviceMountpointInternal,"MISCELLANEOUS",sizeof(tMountPoint)))))
        {
            deviceTypeInternal = DTY_NOT_USED_BY_GMP;
        }
        else
        {
            deviceTypeInternal = DTY_UNSUPPORTED;
        }
        break;

    case FIDEVMGR_E8_DEVICE::FI_EN_DTY_BTA:
    case FIDEVMGR_E8_DEVICE::FI_EN_DTY_HUB:
        deviceTypeInternal = DTY_NOT_USED_BY_GMP;
        break;
/*
    case FIDEVMGR_E8_DEVICE::FI_EN_DTY_UNKNOWN:
        deviceTypeInternal = DTY_UNKNOWN;
        break;
*/
    default:
        deviceTypeInternal = DTY_UNKNOWN;
        break;
    }
}


void fc_mediaplayer_tclClientHandler_DeviceManager::ConvertDeviceTypeToConnectionType (tConnectionType &connectionType, const tDeviceType deviceTypeInternal)
{
    ENTRY_INTERNAL
    switch (deviceTypeInternal)
    {
    case DTY_UNKNOWN:
        connectionType = DCT_UNKNOWN;
        break;
    case DTY_USB:
    case DTY_SD:
    case DTY_IPOD:
    case DTY_IPHONE:
    case DTY_MTP:
    case DTY_MSZUNE:
    case DTY_DVD_DRIVE:
        connectionType = DCT_USB;
        break;
    case DTY_CDROM:
    case DTY_CDDA:
        connectionType = DCT_CDROM;
        break;
    default:
        connectionType = DCT_UNKNOWN;
        break;
    }
}

void fc_mediaplayer_tclClientHandler_DeviceManager::ConvertFileSystemTypeToInternal(tFileSystemType &fileSystemTypeInternal, const FIDEVMGR_E8_FILESYSTEM fileSystemType)
{
    ENTRY_INTERNAL
    switch (fileSystemType.enType)
    {
    case FIDEVMGR_E8_FILESYSTEM::FI_EN_FSTY_UNKNOWN:
        fileSystemTypeInternal = FSTY_UNKNOWN;
        break;
    case FIDEVMGR_E8_FILESYSTEM::FI_EN_FSTY_FAT:
        fileSystemTypeInternal = FSTY_FAT;
        break;
    case FIDEVMGR_E8_FILESYSTEM::FI_EN_FSTY_NTFS:
        fileSystemTypeInternal = FSTY_NTFS;
        break;
    case FIDEVMGR_E8_FILESYSTEM::FI_EN_FSTY_EXT4:
        fileSystemTypeInternal = FSTY_EXT4;
        break;
    case FIDEVMGR_E8_FILESYSTEM::FI_EN_FSTY_HFS:
        fileSystemTypeInternal = FSTY_HFS;
        break;
    case FIDEVMGR_E8_FILESYSTEM::FI_EN_FSTY_EXFAT:
        fileSystemTypeInternal = FSTY_EXFAT;
        break;
    case FIDEVMGR_E8_FILESYSTEM::FI_EN_FSTY_ISO9660:
        fileSystemTypeInternal = FSTY_ISO9660;
        break;
    case FIDEVMGR_E8_FILESYSTEM::FI_EN_FSTY_UDF:
        fileSystemTypeInternal = FSTY_UDF;
        break;
    default:
        fileSystemTypeInternal = FSTY_UNKNOWN;
        break;
    }
}


void fc_mediaplayer_tclClientHandler_DeviceManager::ConvertDeviceConnectionStatusToInternal(tDMConnectStatus &deviceConnectionStatusInternal, const FIDEVMGR_E8_DEVICECONNECTSTATUS DeviceConnectStatus) //finished: 100%
{
    ENTRY_INTERNAL
    switch (DeviceConnectStatus.enType)
    {
    case FIDEVMGR_E8_DEVICECONNECTSTATUS::FI_EN_USB_DEV_UNDEFINED:
        deviceConnectionStatusInternal = DM_CS_UNDEFINED;
        break;
    case FIDEVMGR_E8_DEVICECONNECTSTATUS::FI_EN_USB_DEV_WARNING:
        deviceConnectionStatusInternal = DM_CS_WARNING;  //normally not configured to be send by VD_DVM
        break;
    case FIDEVMGR_E8_DEVICECONNECTSTATUS::FI_EN_USB_DEV_CONNECTED:
        deviceConnectionStatusInternal = DM_CS_CONNECTED;
        break;
    case FIDEVMGR_E8_DEVICECONNECTSTATUS::FI_EN_USB_DEV_REMOVED_BY_USR:
        deviceConnectionStatusInternal = DM_CS_REMOVED_BY_USR;
        break;
    case FIDEVMGR_E8_DEVICECONNECTSTATUS::FI_EN_USB_DEV_UNAVAIL_BAT_LOWVOLT:
        deviceConnectionStatusInternal = DM_CS_UNAVAIL_BAT_LOWVOLT; //undervoltage i.e. USB connectors are switched off due to voltage to low
        break;
    case FIDEVMGR_E8_DEVICECONNECTSTATUS::FI_EN_USB_DEV_UNAVAIL_HW_MALFUNC:
        deviceConnectionStatusInternal = DM_CS_UNAVAIL_HW_MALFUNC; //send if over current
        break;
    case FIDEVMGR_E8_DEVICECONNECTSTATUS::FI_EN_USB_DEV_UNAVAIL_OVERTEMP: //Roadmap 13035 Overtemperature
        deviceConnectionStatusInternal = DM_CS_UNAVAIL_CDROM_OVERTEMP; //internally handled as removed
        break;
    case FIDEVMGR_E8_DEVICECONNECTSTATUS::FI_EN_USB_DEV_UNAVAIL_HW_MALFUNC_PERMANENT:
        deviceConnectionStatusInternal = DM_CS_UNAVAIL_HW_MALFUNC_PERMANENT; //send if over current
        break;
    default:
        deviceConnectionStatusInternal = DM_CS_UNDEFINED;
        break;
    }
}

void fc_mediaplayer_tclClientHandler_DeviceManager::ConvertDMConnectStatusToConnectionState(tConnectionState &connectionStateInternal, tDisconnectReason &disconnectReasonInternal, const tDMConnectStatus connectStatus) //finished: 100%
{
    ENTRY_INTERNAL
    switch (connectStatus)
    {
    case DM_CS_CONNECTED:
        connectionStateInternal = CS_ATTACHED; // the indexer will set this to connected
        disconnectReasonInternal = DR_REMOVED;
        break;
    case DM_CS_UNAVAIL_BAT_LOWVOLT:
        connectionStateInternal = CS_UNDERVOLTAGE; //is set by undervoltage flag. if returned to normal property of VD_DVM still is 'UNAVAIL_BAT_LOWVOLT' until platform notifies device 'CONNETED' again
        disconnectReasonInternal = DR_HW_MALFUNC;
        break;
    case DM_CS_UNAVAIL_HW_MALFUNC:
        connectionStateInternal = CS_DISCONNECTED;
        disconnectReasonInternal = DR_HW_MALFUNC;
        break;
    case DM_CS_UNAVAIL_CDROM_OVERTEMP: //Roadmap 13035 Overtemperature
        connectionStateInternal = CS_OVERTEMP;
        disconnectReasonInternal = DR_TEMPERATURE;
        break;
    case DM_CS_UNDEFINED:
        connectionStateInternal = CS_UNDEFINED; //Multi Partition Support
        disconnectReasonInternal = DR_INTERNALDISCONNECT;
        break;
    case DM_CS_UNAVAIL_HW_MALFUNC_PERMANENT:
        connectionStateInternal = CS_DISCONNECTED;
        disconnectReasonInternal = DR_HW_MALFUNC_PERMANENT;
        break;
    case DM_CS_WARNING:
    case DM_CS_REMOVED_BY_USR:
    default:
        connectionStateInternal = CS_DISCONNECTED;
        disconnectReasonInternal = DR_REMOVED;
        break;
    }
}

void fc_mediaplayer_tclClientHandler_DeviceManager::ConvertDeviceUnsupportedReasonTypeToInternal(tDeviceUnsupportedReason &deviceUnsupportedReasonInternal, const FISHARED_E8_UNSUPPORTEDREASON deviceUnsupportedReason)
{
    ENTRY_INTERNAL
    switch (deviceUnsupportedReason.enType)
    {
    case FISHARED_E8_UNSUPPORTEDREASON::FI_EN_OK:
        deviceUnsupportedReasonInternal = OK;
        break;
    case FISHARED_E8_UNSUPPORTEDREASON::FI_EN_NOTMOUNTEDDEVICE:
        deviceUnsupportedReasonInternal = NotMountedDevice;
        break;
    case FISHARED_E8_UNSUPPORTEDREASON::FI_EN_DISABLEDINCONFIGURATION:
        deviceUnsupportedReasonInternal = DisabledInConfiguration;
        break;
    case FISHARED_E8_UNSUPPORTEDREASON::FI_EN_NOTSUPPORTEDBYDVM:
        deviceUnsupportedReasonInternal = NotSupportedByDVM;
        break;
    case FISHARED_E8_UNSUPPORTEDREASON::FI_EN_HUBBUTNOTCUSTOMERHUB:
        deviceUnsupportedReasonInternal = HubButNotCustomerHub;
        break;
    default:
        deviceUnsupportedReasonInternal = NotSupportedByDVM;
        break;
    }
}

void fc_mediaplayer_tclClientHandler_DeviceManager::ConvertDevicePropertiesToDeviceInfos(vector<tDeviceInfo> &deviceInfos, const vector<tDeviceProperty> deviceProperties) //finished: 100%
{
    ENTRY_INTERNAL
    tDeviceInfo deviceInfoItem;

    deviceInfos.clear();

    for(tUInt i=0; i<deviceProperties.size(); i++) {
        InitDeviceInfo(INOUT deviceInfoItem);

        strncpy_r(deviceInfoItem.UUID, deviceProperties[i].serialNumber, sizeof(deviceInfoItem.UUID));
        strncpy_r(deviceInfoItem.serialNumber, deviceProperties[i].serialNumber, sizeof(deviceInfoItem.serialNumber));
        strncpy_r(deviceInfoItem.deviceVersion, deviceProperties[i].deviceVersion, sizeof(deviceInfoItem.deviceVersion));
        strncpy_r(deviceInfoItem.deviceName, deviceProperties[i].deviceName, sizeof(deviceInfoItem.deviceName));
        strncpy_r(deviceInfoItem.portNumber, deviceProperties[i].USBPortNumber, sizeof(deviceInfoItem.portNumber));
        strncpy_r(deviceInfoItem.mountPoint, deviceProperties[i].mountPoint, sizeof(deviceInfoItem.mountPoint));
        strncpy_r(deviceInfoItem.accessoryName, deviceProperties[i].accessoryName, sizeof(deviceInfoItem.accessoryName));
        strncpy_r(deviceInfoItem.accessoryName2,deviceProperties[i].accessoryName2,sizeof(deviceInfoItem.accessoryName2));
        strncpy_r(deviceInfoItem.deviceSyspath, deviceProperties[i].deviceSyspath, sizeof(deviceInfoItem.deviceSyspath));

        deviceInfoItem.deviceType      = deviceProperties[i].deviceType;
        deviceInfoItem.fileSystemType  = deviceProperties[i].fileSystemType;
        deviceInfoItem.partitionNumber = deviceProperties[i].partitionNumber;
        deviceInfoItem.totalSize       = (tMemorySize)deviceProperties[i].totalSizekB;
        deviceInfoItem.freeSize        = (tMemorySize)deviceProperties[i].freeSizekB;
        deviceInfoItem.productID       = deviceProperties[i].productID;
        deviceInfoItem.connectionType  = deviceProperties[i].connectionType;

        ConvertDMConnectStatusToConnectionState(OUT deviceInfoItem.connectionState, OUT deviceInfoItem.disconnectReason, IN deviceProperties[i].connectStatus);

        deviceInfos.push_back(deviceInfoItem);
    }
}

void fc_mediaplayer_tclClientHandler_DeviceManager::vRegisterProperties() //finished: 100%
{
    ENTRY
    // in GEN2 dependency to IPOD-Auth is required. Property shall be registered when IPodAuth is available
    #ifndef TARGET_BUILD_GEN3
    vAddAutoRegisterForProperty(DEVMGR_NOTIFYCONNECTION);
    #endif
}

void fc_mediaplayer_tclClientHandler_DeviceManager::vDeregisterProperties()  //finished: 100%
{
    ENTRY
    // in GEN2 dependency to IPOD-Auth is required. Property shall be deregistered when IPodAuth is available
    #ifndef TARGET_BUILD_GEN3
    vRemoveAutoRegisterForProperty(DEVMGR_NOTIFYCONNECTION);
    #endif
}

void fc_mediaplayer_tclClientHandler_DeviceManager::vRequestConnectedDevices()
{
    ENTRY

        // Create object of midw_devicemanagerfi_tclMsgDevManagerGetAllConnectedDevicesMethodStart
        mplay_devicemanagerfi_tclMsgDevManagerGetAllConnectedDevicesMethodStart oFiDataObject;

        // Create FI VisitorMessage. (The FI data object will be streamed (each
        // parameter is copied individually) to shared memory.)
        fi_tclVisitorMessage oMsg(oFiDataObject,DEVICEMANAGER_FI_MAJOR_VERSION);

        // Always destroy the (message related) FI data object (before leaving
        // its creation scope)
        oFiDataObject.vDestroy();

        // Here the message related header information is added and by this the
        // creation of the message is completed.
        vInitServiceData (oMsg,                                                   // ServiceDataMsg
                          0,                                                          // CmdCounter
                          MPLAY_DEVICEMANAGERFI_C_U16_DEVMANAGERGETALLCONNECTEDDEVICES,// Function ID
                          AMT_C_U8_CCAMSG_OPCODE_METHODSTART);                        // Opcode

        // Post the message
        ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMsg, TRUE);

        if (AIL_EN_N_NO_ERROR != enResult)
        {
            // Posting of the message has failed,
            // but it is not necessary to call 'oMsg2.bDelete()' here as the
            // 'TRUE' in "enPostMessage(&oMsg, TRUE)" will take care that the
            // message is deleted in a case of an error.
            ETG_TRACE_ERR(( "RequestAllConnectedDevices: enPostMessage() failed for 'ElapsedTime \
                    - GET' message, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
        }

}

tVoid fc_mediaplayer_tclClientHandler_DeviceManager::onRequestConnectedDevicesMethodResult(amt_tclServiceData* poMessage) // finished: 100%
{
    ENTRY

    fi_tclVisitorMessage oVisitorMsg(poMessage);
    mplay_devicemanagerfi_tclMsgDevManagerGetAllConnectedDevicesMethodResult oFiDataObject;
    vector<tDeviceProperty> deviceProperties;

    // Retrieve the information of all connected device from message
    if (OSAL_ERROR != oVisitorMsg.s32GetData (oFiDataObject,DEVICEMANAGER_FI_MAJOR_VERSION))
    {
        deviceProperties.clear();
        // Retrieve the information of all connected devices from object
        for ( unsigned int l_uiCount=0; l_uiCount < oFiDataObject.DevInfoArr.DevList.size();l_uiCount++)
        {
            tDeviceProperty devicePropertyItem;
            strncpy_r(devicePropertyItem.serialNumber,        oFiDataObject.DevInfoArr.DevList[l_uiCount].DeviceSerialNo.szValue,  sizeof devicePropertyItem.serialNumber);         //tDeviceSerialNumber
            strncpy_r(devicePropertyItem.USBPortNumber,       oFiDataObject.DevInfoArr.DevList[l_uiCount].USBPortNo.szValue,       sizeof devicePropertyItem.USBPortNumber);        //tUSBPortNumber
            strncpy_r(devicePropertyItem.mountPoint,          oFiDataObject.DevInfoArr.DevList[l_uiCount].MountPoint.szValue,      sizeof devicePropertyItem.mountPoint);           //tMountPoint
            ConvertDeviceUnsupportedReasonTypeToInternal(OUT devicePropertyItem.deviceUnsupportedReason, IN oFiDataObject.DevInfoArr.DevList[l_uiCount].UnsupportedReason);          //tDeviceUnsupportedReason
            ConvertDeviceTypeToInternal(OUT devicePropertyItem.deviceType, IN oFiDataObject.DevInfoArr.DevList[l_uiCount].DeviceType, IN devicePropertyItem.deviceUnsupportedReason, IN devicePropertyItem.mountPoint);//tDeviceType
            ConvertDeviceTypeToConnectionType(OUT devicePropertyItem.connectionType, IN devicePropertyItem.deviceType);
            strncpy_r(devicePropertyItem.deviceVersion,       oFiDataObject.DevInfoArr.DevList[l_uiCount].DeviceVersion.szValue,   sizeof devicePropertyItem.deviceVersion);        //tDeviceVersion
            strncpy_r(devicePropertyItem.deviceSyspath,       oFiDataObject.DevInfoArr.DevList[l_uiCount].DevicePath.szValue,      sizeof devicePropertyItem.deviceSyspath);        //tPath
            strncpy_r(devicePropertyItem.deviceName,          oFiDataObject.DevInfoArr.DevList[l_uiCount].DeviceName.szValue,      sizeof devicePropertyItem.deviceName);           //tDeviceName
            strncpy_r(devicePropertyItem.accessoryName,       oFiDataObject.DevInfoArr.DevList[l_uiCount].AccessoryName.szValue,   sizeof devicePropertyItem.accessoryName);        //tAccessoryName
            strncpy_r(devicePropertyItem.accessoryName2,      oFiDataObject.DevInfoArr.DevList[l_uiCount].AccessoryName2.szValue,  sizeof devicePropertyItem.accessoryName2);        //tAccessoryName
            ConvertDeviceConnectionStatusToInternal(OUT devicePropertyItem.connectStatus, IN oFiDataObject.DevInfoArr.DevList[l_uiCount].DeviceConnectStatus);                      //tDMConnectStatus
#ifdef VARIANT_S_FTR_ENABLE_DEVMGR_GEN3_CCA
            devicePropertyItem.sizeInByte                   = oFiDataObject.DevInfoArr.DevList[l_uiCount].TotalSize;                                                                //tTotalSizeByte
            devicePropertyItem.vendorID                     = oFiDataObject.DevInfoArr.DevList[l_uiCount].VendorID;                                                                 //tVendorID
            strncpy_r(devicePropertyItem.manufacturer,        oFiDataObject.DevInfoArr.DevList[l_uiCount].Manufacturer.szValue,    sizeof devicePropertyItem.manufacturer);         //tManufacturer
            devicePropertyItem.productID                    = oFiDataObject.DevInfoArr.DevList[l_uiCount].ProductID;                                                                //tProductID

            devicePropertyItem.totalSizekB                  = oFiDataObject.DevInfoArr.DevList[l_uiCount].TotalSize_kB;                                                             //tTotalSize_kB
            devicePropertyItem.usedSizekB                   = oFiDataObject.DevInfoArr.DevList[l_uiCount].UsedSize_kB;                                                              //tUsedSize_kB
            devicePropertyItem.freeSizekB                   = oFiDataObject.DevInfoArr.DevList[l_uiCount].FreeSize_kB;                                                              //tFreeSize_kB
            devicePropertyItem.partitionNumber              = oFiDataObject.DevInfoArr.DevList[l_uiCount].ParitionNum;                                                              //tPartitionNum

            ConvertFileSystemTypeToInternal(OUT devicePropertyItem.fileSystemType, IN oFiDataObject.DevInfoArr.DevList[l_uiCount].FileSystemType);                                 //tFileSystemType

            strncpy_r(devicePropertyItem.deviceSpeed,         (const char*)"-"/*oFiDataObject.DevInfoArr.DevList[l_uiCount].DeviceSpeedUsb.szValue*/,   sizeof devicePropertyItem.deviceSpeed);         //tDeviceSpeed
            strncpy_r(devicePropertyItem.protocolUsb,         (const char*)"-"/*oFiDataObject.DevInfoArr.DevList[l_uiCount].ProtocolUsb.szValue*/,      sizeof devicePropertyItem.protocolUsb);         //tProtocolUsb
            strncpy_r(devicePropertyItem.baseClassCodeUsb,    (const char*)"-"/*oFiDataObject.DevInfoArr.DevList[l_uiCount].BaseClassCodeUsb.szValue*/, sizeof devicePropertyItem.baseClassCodeUsb);    //tBaseClassCodeUsb
            strncpy_r(devicePropertyItem.subBaseClassCodeUsb, (const char*)"-"/*oFiDataObject.DevInfoArr.DevList[l_uiCount].SubClassCodeUsb.szValue*/,  sizeof devicePropertyItem.subBaseClassCodeUsb); //tSubClassCodeUsb
#endif

            TraceDevicePropertyItem(eOnRequestConnectedDevicesMethodResult, devicePropertyItem, l_uiCount);

            //Push the object into the vector
            deviceProperties.push_back(devicePropertyItem);

        }

        if (!deviceProperties.empty())
        {
            ETG_TRACE_USR3(("onRequestConnectedDevicesMethodResult : Calling DeviceChanged"));

            vector<tDeviceInfo> deviceInfos;
            ConvertDevicePropertiesToDeviceInfos(OUT deviceInfos, IN deviceProperties);
            if(MP_NO_ERROR != MediaPlayerInterface::GetInstance().DeviceChanged(IN false, IN deviceInfos))
            {
                // CID 10332 (#1 of 1): Unchecked return value (CHECKED_RETURN)
                ETG_TRACE_ERR(("onRequestConnectedDevicesMethodResult: Device status cannot be set"));
            }
        }
        else
        {
            ETG_TRACE_USR3(("deviceProperties is Empty"));
        }

    }
    else
    {
        ETG_TRACE_ERR(("onRequestConnectedDevicesMethodResult : Received message is invalid."));
    }

    // Always destroy the (message related) FI data object (before leaving its creation scope)
    oFiDataObject.vDestroy();
}


tVoid fc_mediaplayer_tclClientHandler_DeviceManager::TraceDevicePropertyItem(tenFctType eFcType, const tDeviceProperty &devicePropertyItem, unsigned int index) const
{
    ENTRY
    ETG_TRACE_COMP(("onRequestConnectedDevicesMethodResult: ================INDEX:%d ==============================================",index));
    if(eOnDeviceManagerNotifyConnection == eFcType)
    {
        ETG_TRACE_COMP(("onDeviceManagerNotifyConnection: devicePropertyItem[%d].serialNumber:                       %s",index, devicePropertyItem.serialNumber));
        ETG_TRACE_COMP(("onDeviceManagerNotifyConnection: devicePropertyItem[%d].USBPortNumber:                      %s",index,devicePropertyItem.USBPortNumber));
        ETG_TRACE_COMP(("onDeviceManagerNotifyConnection: devicePropertyItem[%d].deviceType:                         %d",index,ETG_ENUM(DP_DEVTYPE,devicePropertyItem.deviceType)));
        ETG_TRACE_COMP(("onDeviceManagerNotifyConnection: devicePropertyItem[%d].connectionType:                     %d",index,ETG_ENUM(DP_CONNECTION_TYPE,devicePropertyItem.connectionType)));
        ETG_TRACE_COMP(("onDeviceManagerNotifyConnection: devicePropertyItem[%d].deviceVersion:                      %s",index,devicePropertyItem.deviceVersion));
        ETG_TRACE_COMP(("onDeviceManagerNotifyConnection: devicePropertyItem[%d].deviceName:                         %s",index,devicePropertyItem.deviceName));
        ETG_TRACE_COMP(("onDeviceManagerNotifyConnection: devicePropertyItem[%d].deviceSysPath:                         %s",index,devicePropertyItem.deviceSyspath));
        ETG_TRACE_COMP(("onDeviceManagerNotifyConnection: devicePropertyItem[%d].accessoryName:                      %s",index,devicePropertyItem.accessoryName));
        ETG_TRACE_COMP(("onDeviceManagerNotifyConnection: devicePropertyItem[%d].accessoryName2(Original Serial ID):      %s",index,devicePropertyItem.accessoryName2));
        ETG_TRACE_COMP(("onDeviceManagerNotifyConnection: devicePropertyItem[%d].mountPoint:                          %s",index,devicePropertyItem.mountPoint));
        ETG_TRACE_COMP(("onDeviceManagerNotifyConnection: devicePropertyItem[%d].connectStatus:      %d",index,ETG_ENUM(DP_CONNECT_STATUS,devicePropertyItem.connectStatus)));
        ETG_TRACE_COMP(("onDeviceManagerNotifyConnection: devicePropertyItem[%d].sizeInByte:         %u",index,devicePropertyItem.sizeInByte));
        ETG_TRACE_COMP(("onDeviceManagerNotifyConnection: devicePropertyItem[%d].tVendorID:          %u",index,devicePropertyItem.vendorID));
        ETG_TRACE_COMP(("onDeviceManagerNotifyConnection: devicePropertyItem[%d].manufacturer:       %s",index,devicePropertyItem.manufacturer));
        ETG_TRACE_COMP(("onDeviceManagerNotifyConnection: devicePropertyItem[%d].productID:          %u",index,devicePropertyItem.productID));
        ETG_TRACE_COMP(("onDeviceManagerNotifyConnection: devicePropertyItem[%d].totalSizekB:        %u",index,(unsigned int)devicePropertyItem.totalSizekB));
        ETG_TRACE_COMP(("onDeviceManagerNotifyConnection: devicePropertyItem[%d].usedSizekB:         %u",index,(unsigned int)devicePropertyItem.usedSizekB));
        ETG_TRACE_COMP(("onDeviceManagerNotifyConnection: devicePropertyItem[%d].freeSizekB:         %u",index,(unsigned int)devicePropertyItem.freeSizekB));
        ETG_TRACE_COMP(("onDeviceManagerNotifyConnection: devicePropertyItem[%d].partitionNumber:    %u",index,devicePropertyItem.partitionNumber));
        ETG_TRACE_COMP(("onDeviceManagerNotifyConnection: devicePropertyItem[%d].fileSystemType:     %d",index,ETG_ENUM(DP_FILESYSTEM_TYPE,devicePropertyItem.fileSystemType)));
        ETG_TRACE_COMP(("onDeviceManagerNotifyConnection: devicePropertyItem[%d].deviceSpeed:        %s",index,devicePropertyItem.deviceSpeed));
        ETG_TRACE_COMP(("onDeviceManagerNotifyConnection: devicePropertyItem[%d].protocolUsb:        %s",index,devicePropertyItem.protocolUsb));
        ETG_TRACE_COMP(("onDeviceManagerNotifyConnection: devicePropertyItem[%d].baseClassCodeUsb:   %s",index,devicePropertyItem.baseClassCodeUsb));
        ETG_TRACE_COMP(("onDeviceManagerNotifyConnection: devicePropertyItem[%d].subBaseClassCodeUsb:%s",index,devicePropertyItem.subBaseClassCodeUsb));
        ETG_TRACE_COMP(("onDeviceManagerNotifyConnection: devicePropertyItem[%d].deviceUnsupportedReason:%d",index,devicePropertyItem.deviceUnsupportedReason));
    }
    else if(eOnRequestConnectedDevicesMethodResult == eFcType)
    {
        ETG_TRACE_COMP(("onRequestConnectedDevicesMethodResult: devicePropertyItem[%d].serialNumber:       %s",index,devicePropertyItem.serialNumber));
        ETG_TRACE_COMP(("onRequestConnectedDevicesMethodResult: devicePropertyItem[%d].USBPortNumber:      %s",index,devicePropertyItem.USBPortNumber));
        ETG_TRACE_COMP(("onRequestConnectedDevicesMethodResult: devicePropertyItem[%d].deviceType:         %d",index,ETG_CENUM(tDeviceType,devicePropertyItem.deviceType)));
        ETG_TRACE_COMP(("onRequestConnectedDevicesMethodResult: devicePropertyItem[%d].connectionType:     %d",index,ETG_CENUM(tConnectionType,devicePropertyItem.connectionType)));
        ETG_TRACE_COMP(("onRequestConnectedDevicesMethodResult: devicePropertyItem[%d].deviceVersion:      %s",index,devicePropertyItem.deviceVersion));
        ETG_TRACE_COMP(("onRequestConnectedDevicesMethodResult: devicePropertyItem[%d].deviceName:         %s",index,devicePropertyItem.deviceName));
        ETG_TRACE_COMP(("onRequestConnectedDevicesMethodResult: devicePropertyItem[%d].deviceSysPath:      %s",index,devicePropertyItem.deviceSyspath));
        ETG_TRACE_COMP(("onRequestConnectedDevicesMethodResult: devicePropertyItem[%d].accessoryName:      %s",index,devicePropertyItem.accessoryName));
        ETG_TRACE_COMP(("onRequestConnectedDevicesMethodResult: devicePropertyItem[%d].mountPoint:         %s",index,devicePropertyItem.mountPoint));
        ETG_TRACE_COMP(("onRequestConnectedDevicesMethodResult: devicePropertyItem[%d].connectStatus:      %d",index,ETG_CENUM(tDMConnectStatus,devicePropertyItem.connectStatus)));
        ETG_TRACE_COMP(("onRequestConnectedDevicesMethodResult: devicePropertyItem[%d].sizeInByte:         %u",index,devicePropertyItem.sizeInByte));
        ETG_TRACE_COMP(("onRequestConnectedDevicesMethodResult: devicePropertyItem[%d].tVendorID:          %u",index,devicePropertyItem.vendorID));
        ETG_TRACE_COMP(("onRequestConnectedDevicesMethodResult: devicePropertyItem[%d].manufacturer:       %s",index,devicePropertyItem.manufacturer));
        ETG_TRACE_COMP(("onRequestConnectedDevicesMethodResult: devicePropertyItem[%d].productID:          %u",index,devicePropertyItem.productID));
        ETG_TRACE_COMP(("onRequestConnectedDevicesMethodResult: devicePropertyItem[%d].totalSizekB:        %u",index,(unsigned int)devicePropertyItem.totalSizekB));
        ETG_TRACE_COMP(("onRequestConnectedDevicesMethodResult: devicePropertyItem[%d].usedSizekB:         %u",index,(unsigned int)devicePropertyItem.usedSizekB));
        ETG_TRACE_COMP(("onRequestConnectedDevicesMethodResult: devicePropertyItem[%d].freeSizekB:         %u",index,(unsigned int)devicePropertyItem.freeSizekB));
        ETG_TRACE_COMP(("onRequestConnectedDevicesMethodResult: devicePropertyItem[%d].partitionNumber:    %u",index,devicePropertyItem.partitionNumber));
        ETG_TRACE_COMP(("onRequestConnectedDevicesMethodResult: devicePropertyItem[%d].fileSystemType:     %d",index,ETG_CENUM(tFileSystemType,devicePropertyItem.fileSystemType)));
        ETG_TRACE_COMP(("onRequestConnectedDevicesMethodResult: devicePropertyItem[%d].deviceSpeed:        %s",index,devicePropertyItem.deviceSpeed));
        ETG_TRACE_COMP(("onRequestConnectedDevicesMethodResult: devicePropertyItem[%d].protocolUsb:        %s",index,devicePropertyItem.protocolUsb));
        ETG_TRACE_COMP(("onRequestConnectedDevicesMethodResult: devicePropertyItem[%d].baseClassCodeUsb:   %s",index,devicePropertyItem.baseClassCodeUsb));
        ETG_TRACE_COMP(("onRequestConnectedDevicesMethodResult: devicePropertyItem[%d].subBaseClassCodeUsb:%s",index,devicePropertyItem.subBaseClassCodeUsb));
        ETG_TRACE_COMP(("onRequestConnectedDevicesMethodResult: devicePropertyItem[%d].deviceUnsupportedReason:%d",index,devicePropertyItem.deviceUnsupportedReason));
    }


}



void fc_mediaplayer_tclClientHandler_DeviceManager::ConvertOpticalDiscStatusToInternal(OUT tDMOpticalDiscSlotState &opticalDiscStatusInternal, IN const mplay_fi_tcl_e8_SlotStateOpticalDiscType enSlotStateOpticalDisc)
{
    ENTRY_INTERNAL
    switch (enSlotStateOpticalDisc.enType)
    {
    case mplay_fi_tcl_e8_SlotStateOpticalDiscType::FI_EN_INITIALISING:
        opticalDiscStatusInternal = DM_OPTDISC_INITIALISING;
        break;
    case mplay_fi_tcl_e8_SlotStateOpticalDiscType::FI_EN_INITIALISED:
        opticalDiscStatusInternal = DM_OPTDISC_INITIALISED;
        break;
    case mplay_fi_tcl_e8_SlotStateOpticalDiscType::FI_EN_LASTMODE_EMTPY:
        opticalDiscStatusInternal = DM_OPTDISC_LASTMODE_EMTPY;
        break;
    case mplay_fi_tcl_e8_SlotStateOpticalDiscType::FI_EN_LASTMODE_INSERTED_CDDA:
        opticalDiscStatusInternal = DM_OPTDISC_LASTMODE_INSERTED_CDDA;
        break;
    case mplay_fi_tcl_e8_SlotStateOpticalDiscType::FI_EN_LASTMODE_INSERTED_CDROM:
        opticalDiscStatusInternal = DM_OPTDISC_LASTMODE_INSERTED_CDROM;
        break;
    case mplay_fi_tcl_e8_SlotStateOpticalDiscType::FI_EN_LASTMODE_INSERTED_CDERROR:
        opticalDiscStatusInternal = DM_OPTDISC_LASTMODE_INSERTED_CDERROR;
        break;
    case mplay_fi_tcl_e8_SlotStateOpticalDiscType::FI_EN_INSERTING:
        opticalDiscStatusInternal = DM_OPTDISC_INSERTING;
        break;
    case mplay_fi_tcl_e8_SlotStateOpticalDiscType::FI_EN_INSERTED_CDAUDIO:
        opticalDiscStatusInternal = DM_OPTDISC_INSERTED_CDAUDIO;
        break;
    case mplay_fi_tcl_e8_SlotStateOpticalDiscType::FI_EN_INSERTED_CDROM:
        opticalDiscStatusInternal = DM_OPTDISC_INSERTED_CDROM;
        break;
    case mplay_fi_tcl_e8_SlotStateOpticalDiscType::FI_EN_INSERTED_CDERROR:
        opticalDiscStatusInternal = DM_OPTDISC_INSERTED_CDERROR;
        break;
    case mplay_fi_tcl_e8_SlotStateOpticalDiscType::FI_EN_INSERTED_AUTOMATIC_CDAUDIO:
        opticalDiscStatusInternal = DM_OPTDISC_INSERTED_AUTOMATIC_CDAUDIO;
        break;
    case mplay_fi_tcl_e8_SlotStateOpticalDiscType::FI_EN_INSERTED_AUTOMATIC_CDROM:
        opticalDiscStatusInternal = DM_OPTDISC_INSERTED_AUTOMATIC_CDROM;
        break;
    case mplay_fi_tcl_e8_SlotStateOpticalDiscType::FI_EN_INSERTED_AUTOMATIC_CDERROR:
        opticalDiscStatusInternal = DM_OPTDISC_INSERTED_AUTOMATIC_CDERROR;
        break;
    case mplay_fi_tcl_e8_SlotStateOpticalDiscType::FI_EN_EJECTING:
        opticalDiscStatusInternal = DM_OPTDISC_EJECTING;
        break;
    case mplay_fi_tcl_e8_SlotStateOpticalDiscType::FI_EN_EJECTED_READY_TO_REMOVE:
        opticalDiscStatusInternal = DM_OPTDISC_EJECTED_READY_TO_REMOVE;
        break;
    case mplay_fi_tcl_e8_SlotStateOpticalDiscType::FI_EN_EJECTED_EMPTY:
        opticalDiscStatusInternal = DM_OPTDISC_EJECTED_EMPTY;
        break;
    case mplay_fi_tcl_e8_SlotStateOpticalDiscType::FI_EN_UNDEFINED:
    default:
        opticalDiscStatusInternal = DM_OPTDISC_UNDEFINED_STATE;
        break;


    }
}

//send to VD_DeviceManager:
tVoid fc_mediaplayer_tclClientHandler_DeviceManager::SendEjectOpticalDisc(IN const tDMOpticalDiscEjectCmd enOpticalDiscEjectCmd)//Roadmap 15002_CDAudio_DetectionAndDiagnosis
{
    ENTRY

    tBool bSend = TRUE;
    ETG_TRACE_USR4(("SendEjectOpticalDisc: enOpticalDiscEjectCmd: %d",ETG_CENUM(tDMOpticalDiscEjectCmd,enOpticalDiscEjectCmd)));

    FIDEVMGR_EJECTOPTICALDISC_METHODSTART oFiDataObject;

    switch(enOpticalDiscEjectCmd)
    {
        case DM_OPTDISC_CMD_EJECT:
            oFiDataObject.eCmdParam.enType = mplay_fi_tcl_e8_EjectParmType::FI_EN_E8CMD_EJECT;
            break;
        case DM_OPTDISC_CMD_INSERT:
            oFiDataObject.eCmdParam.enType = mplay_fi_tcl_e8_EjectParmType::FI_EN_E8CMD_INSERT;
            break;
        default:
            bSend = FALSE;
            ETG_TRACE_FATAL(("vHandle_OpticalDiscEject: Unexpected value enOpticalDiscEjectCmd: %d", (int)enOpticalDiscEjectCmd));
            break;
    }



    if(bSend)
    {

        // Create FI VisitorMessage. (The FI data object will be streamed (each
        // parameter is copied individually) to shared memory.)
        fi_tclVisitorMessage oMsg(oFiDataObject,DEVICEMANAGER_FI_MAJOR_VERSION);

        // Always destroy the (message related) FI data object (before leaving
        // its creation scope)
        oFiDataObject.vDestroy();

        // Here the message related header information is added and by this the
        // creation of the message is completed.
        vInitServiceData (oMsg,                                                   // ServiceDataMsg
                          0,                                                          // CmdCounter
                          DEVMGR_EJECTOPTICALDISC,// Function ID
                          AMT_C_U8_CCAMSG_OPCODE_METHODSTART);                        // Opcode

        // Post the message
        ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMsg, TRUE); //---->send to VD_DVM

        if (AIL_EN_N_NO_ERROR != enResult)
        {
            // Posting of the message has failed,
            // but it is not necessary to call 'oMsg2.bDelete()' here as the
            // 'TRUE' in "enPostMessage(&oMsg, TRUE)" will take care that the
            // message is deleted in a case of an error.
            ETG_TRACE_ERR(( "vHandle_OpticalDiscEject: enPostMessage() failed for 'CDEject \
                    - GET' message, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
        }
    }
    ETG_TRACE_USR3(( "End ::vHandleOpticalDiscEject" ));

}

//reveived from VD_DeviceManager:
tVoid fc_mediaplayer_tclClientHandler_DeviceManager::onOpticalDiscEjectMethodResult( amt_tclServiceData* poMessage )//Roadmap 15002_CDAudio_DetectionAndDiagnosis
{
    ENTRY
    ETG_TRACE_USR3(( "Begin ::onOpticalDiscEjectMethodResult" ));
    tResult result = MP_NO_ERROR;
    // Create a FI visitor message for the received CCA message
    fi_tclVisitorMessage oResultMsg( poMessage );
    // Create the (message related) FI data object
    FIDEVMGR_EJECTOPTICALDISC_METHODRESULT oResultData;
    oResultMsg.s32GetData(oResultData);

    // if control setting accepted, composing positive response

    switch(oResultData.ServiceStatus.enType)
    {
        case mplay_fi_tcl_e8_EjectOpticalDisc_MethodResult::FI_EN_ACCEPTED:
            ETG_TRACE_USR4 (("onOpticalDiscEjectMethodResult :: ACCEPTED"));
            result = MediaPlayerInterface::GetInstance().SendOpticalDiscEjectMethodResult(ACCEPTED);//---->send via Interface to outputwrapper to HMI
            break;
        case mplay_fi_tcl_e8_EjectOpticalDisc_MethodResult::FI_EN_NOTACCEPTED:
            ETG_TRACE_USR4 (("onOpticalDiscEjectMethodResult :: NOTACCEPTED"));
            result = MediaPlayerInterface::GetInstance().SendOpticalDiscEjectMethodResult(NOTACCEPTED);//---->send via Interface to outputwrapper to HMI
            break;
        case mplay_fi_tcl_e8_EjectOpticalDisc_MethodResult::FI_EN_DEVICENOTREADY:
            ETG_TRACE_USR4 (("onOpticalDiscEjectMethodResult :: DEVICENOTREADY"));
            result = MediaPlayerInterface::GetInstance().SendOpticalDiscEjectMethodResult(DEVICENOTREADY);//---->send via Interface to outputwrapper to HMI
            break;
        default:
            ETG_TRACE_FATAL(("onOpticalDiscEjectMethodResult: oResultData.enType: %d",(int)oResultData.ServiceStatus.enType));
            ETG_TRACE_FATAL(("onOpticalDiscEjectMethodResult: mplay_fi_tcl_e8_EjectOpticalDisc_MethodResult::FI_EN_ACCEPTED      : %d",(int)mplay_fi_tcl_e8_EjectOpticalDisc_MethodResult::FI_EN_ACCEPTED));
            ETG_TRACE_FATAL(("onOpticalDiscEjectMethodResult: mplay_fi_tcl_e8_EjectOpticalDisc_MethodResult::FI_EN_NOTACCEPTED   : %d",(int)mplay_fi_tcl_e8_EjectOpticalDisc_MethodResult::FI_EN_NOTACCEPTED));
            ETG_TRACE_FATAL(("onOpticalDiscEjectMethodResult: mplay_fi_tcl_e8_EjectOpticalDisc_MethodResult::FI_EN_DEVICENOTREADY: %d",(int)mplay_fi_tcl_e8_EjectOpticalDisc_MethodResult::FI_EN_DEVICENOTREADY));
            break;
    }
    oResultData.vDestroy();

    ETG_TRACE_USR3(( "End ::vHandle_OpticalDiscEject_MethodResult result: %d",result));
}

//reveived from VD_DeviceManager
tVoid fc_mediaplayer_tclClientHandler_DeviceManager::onDeviceManagerNotifySlotStateOpticalDisc(amt_tclServiceData* poMessage) //Roadmap 15002_CDAudio_DetectionAndDiagnosis
{
    ENTRY
    tResult result = MP_NO_ERROR;
    tDMOpticalDiscSlotState eOpticalDiscStatusGMP = DM_OPTDISC_UNDEFINED_STATE;

    mplay_devicemanagerfi_tclMsgDevManagerNotifySlotStateOpticalDiscStatus oFiDataObjectStatus;
    fi_tclVisitorMessage oVisitorMsg(poMessage);
    if (OSAL_ERROR != oVisitorMsg.s32GetData(oFiDataObjectStatus, DEVICEMANAGER_FI_MAJOR_VERSION))
    {
        mplay_fi_tcl_e8_SlotStateOpticalDiscType oFiOpticalDiscStatus;
        oFiOpticalDiscStatus = oFiDataObjectStatus.OpticalDiscSlotState.front();
        ConvertOpticalDiscStatusToInternal(OUT eOpticalDiscStatusGMP,IN oFiOpticalDiscStatus);

        ETG_TRACE_USR4(("onDeviceManagerNotifySlotStateopticalDisc: ======================================================================"));
        ETG_TRACE_USR4(("onDeviceManagerNotifySlotStateopticalDisc: oFiOpticalDiscStatus.enType :%d, eOpticalDiscStatusInternal: %d",  oFiOpticalDiscStatus.enType, ETG_ENUM(tDMOpticalDiscSlotState,eOpticalDiscStatusGMP)));
        ETG_TRACE_USR4(("onDeviceManagerNotifySlotStateopticalDisc: ======================================================================"));

        MediaPlayerInterface::GetInstance().OpticalDiscStatusChanged(eOpticalDiscStatusGMP);//---->send via Interface to outputwrapper to HMI
    }
    else
    {
        NORMAL_M_ASSERT_ALWAYS();
    }
    if (MP_NO_ERROR != result)
    {
        ETG_TRACE_ERR(("onDeviceManagerNotifyConnection: Device status cannot be set: %d", result));
    }
    oFiDataObjectStatus.vDestroy();
}

tVoid fc_mediaplayer_tclClientHandler_DeviceManager::onDeviceManagerNotifyCDInfo(amt_tclServiceData* poMessage) // finished: 100%
{
    ENTRY
    tResult result = MP_NO_ERROR;

    FIDEVMGR_NOTIFYCDINFOSTATUS oFiDataObjectStatus;
    fi_tclVisitorMessage oVisitorMsg(poMessage);
    if (OSAL_ERROR != oVisitorMsg.s32GetData(oFiDataObjectStatus, DEVICEMANAGER_FI_MAJOR_VERSION))
    {

        trOpticalDiskCDInfo OpticalDiskCDInfo;
        strncpy_r(OpticalDiskCDInfo.OpticalDriveVersion, oFiDataObjectStatus.CDInfo.DriveVersion.szValue, sizeof OpticalDiskCDInfo.OpticalDriveVersion);

        mplay_fi_tcl_e8DeviceState oFiOpticalDeviceState;
        oFiOpticalDeviceState = oFiDataObjectStatus.CDInfo.e8DeviceState;

        tenOpticalDeviceState DeviceState;
        ConvertOpticalDeviceStateToInternal(OUT DeviceState,IN oFiOpticalDeviceState);
        OpticalDiskCDInfo.OpticalDeviceState = DeviceState;
        ETG_TRACE_USR4(("onDeviceManagerNotifyCDInfo: ======================================================================"));
        ETG_TRACE_USR4(("onDeviceManagerNotifyCDInfo: OpticalDriveVersion:%s", OpticalDiskCDInfo.OpticalDriveVersion));
        ETG_TRACE_USR4(("onDeviceManagerNotifyCDInfo: OpticalDeviceState: %d", OpticalDiskCDInfo.OpticalDeviceState));
        ETG_TRACE_USR4(("onDeviceManagerNotifyCDInfo: ======================================================================"));

        MediaPlayerInterface::GetInstance().DeviceManagerNotifyCDInfo(OpticalDiskCDInfo);//---->send via Interface to outputwrapper to HMI
    }
    else
    {
        NORMAL_M_ASSERT_ALWAYS();
    }
    if (MP_NO_ERROR != result)
    {
        ETG_TRACE_ERR(("onDeviceManagerNotifyConnection: Device status cannot be set: %d", result));
    }
    oFiDataObjectStatus.vDestroy();
}

void fc_mediaplayer_tclClientHandler_DeviceManager::ConvertOpticalDeviceStateToInternal(OUT tenOpticalDeviceState &DeviceState, IN const mplay_fi_tcl_e8DeviceState oFiOpticalDeviceState)
{
    ENTRY_INTERNAL
    switch (oFiOpticalDeviceState.enType)
    {
    case mplay_fi_tcl_e8DeviceState::FI_EN_MMGR_DEVICE_NOT_READY:
        DeviceState = DEVICE_NOT_READY;
        break;
    case mplay_fi_tcl_e8DeviceState::FI_EN_MMGR_DEVICE_READY:
        DeviceState = DEVICE_READY;
        break;
    default:
        DeviceState = DEVICE_NOT_READY;
        break;
    }
}

void fc_mediaplayer_tclClientHandler_DeviceManager::vUpdateDVDDriveInfo()
{
    ENTRY

        // Create object of mplay_devicemanagerfi_tclMsgDevManagerDVDDriveInfoSet
        mplay_devicemanagerfi_tclMsgDevManagerDVDDriveInfoSet oFiDataObject;
        tResult result = -1;
        tU8 discType = 1;
        tU8 driveMode = 1;
        result = MediaPlayerInterface::GetInstance().GetDVDDriveInfo(discType,driveMode);
        if (0 == result)
        {
            oFiDataObject.e8DVDDiscType.enType = (mplay_fi_tcl_e8_MDVDDiscType::tenType)discType ;
            oFiDataObject.e8DVDDriveMode.enType = (mplay_fi_tcl_e8_MDVDDriveMode::tenType)driveMode ;
        }
        ETG_TRACE_USR4(("vUpdateDVDDriveInfo: ==========================DVDDriveInfo================================"));
        ETG_TRACE_USR4(("vUpdateDVDDriveInfo: oFiDataObject.e8DVDDiscType.enType:%d", oFiDataObject.e8DVDDiscType.enType));
        ETG_TRACE_USR4(("vUpdateDVDDriveInfo: oFiDataObject.e8DVDDriveMode.enType: %d", oFiDataObject.e8DVDDriveMode.enType));
        ETG_TRACE_USR4(("vUpdateDVDDriveInfo: ======================================================================"));

        // Create FI VisitorMessage. (The FI data object will be streamed (each
        // parameter is copied individually) to shared memory.)
        fi_tclVisitorMessage oMsg(oFiDataObject,DEVICEMANAGER_FI_MAJOR_VERSION);

        // Always destroy the (message related) FI data object (before leaving
        // its creation scope)
        oFiDataObject.vDestroy();

        // Here the message related header information is added and by this the
        // creation of the message is completed.
        vInitServiceData (oMsg,                                                   // ServiceDataMsg
                          0,                                                          // CmdCounter
                          MPLAY_DEVICEMANAGERFI_C_U16_DEVMANAGERDVDDRIVEINFO,// Function ID
                          AMT_C_U8_CCAMSG_OPCODE_SET);                        // Opcode

        // Post the message
        ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMsg, TRUE);

        if (AIL_EN_N_NO_ERROR != enResult)
        {
            // Posting of the message has failed,
            // but it is not necessary to call 'oMsg2.bDelete()' here as the
            // 'TRUE' in "enPostMessage(&oMsg, TRUE)" will take care that the
            // message is deleted in a case of an error.
            ETG_TRACE_ERR(( "vUpdateDVDDriveInfo: enPostMessage() failed for 'DVDDRIVEINFO \
                    - SET' message, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
        }

}

void fc_mediaplayer_tclClientHandler_DeviceManager::RequestDeviceManagerNotifyConnection()
{
    ENTRY

    mplay_devicemanagerfi_tclMsgDevManagerNotifyConnectionGet oFiDataObject;

    // Create FI VisitorMessage. (The FI data object will be streamed (each
    // parameter is copied individually) to shared memory.)
    fi_tclVisitorMessage oMsg(oFiDataObject,DEVICEMANAGER_FI_MAJOR_VERSION);

    // Always destroy the (message related) FI data object (before leaving
    // its creation scope)
    oFiDataObject.vDestroy();

    // Here the message related header information is added and by this the
    // creation of the message is completed.
    vInitServiceData (oMsg,                                                   // ServiceDataMsg
                      0,                                                          // CmdCounter
                      MPLAY_DEVICEMANAGERFI_C_U16_DEVMANAGERNOTIFYCONNECTION,// Function ID
                      AMT_C_U8_CCAMSG_OPCODE_GET);                        // Opcode

    // Post the message
    ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMsg, TRUE);

    if (AIL_EN_N_NO_ERROR != enResult)
    {
        // Posting of the message has failed,
        // but it is not necessary to call 'oMsg2.bDelete()' here as the
        // 'TRUE' in "enPostMessage(&oMsg, TRUE)" will take care that the
        // message is deleted in a case of an error.
        ETG_TRACE_ERR(( "RequestDeviceManagerNotifyConnection: enPostMessage() failed for 'ElapsedTime \
                - GET' message, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
    }
}

