/************************************************************************
 * \file authentication_ipc.c
 *
 * \version $Id: authentication_ipc.c, 
 *
 * \release $Name: 
 *
 * \brief This is the implementation of the authentication IPC component for ipod on Linux
 *
 * \component ipod control
 *
 * \author M.Shibata
 *
 * \copyright (c) 2003 - 2011 ADIT Corporation
 *
 ***********************************************************************/
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_mp.h"

#include "TraceDefinitions.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_IPOD_CONTROL
#ifdef TARGET_BUILD
#include "trcGenProj/Header/iap_authentication.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_IPOD_CONTROL
#endif
#endif

#include "Utils.h" 
#include <adit_typedef.h>
#include <errno.h>
#include <string.h>
#include <endian.h>
#include <byteswap.h>
#include <asm/types.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <linux/un.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <time.h>

//#include <ipodauth.h>

#include <authentication.h>
//#include <authentication_lx_spi.h>
//#include "authentication_configuration.h"
#include "auth_msc.h"


/* Functions to convert big endian values from CP to host endianness */
#if __BYTE_ORDER == __LITTLE_ENDIAN

#define AUTH_TO_HOST_16(x) bswap_16(x)
#define AUTH_TO_HOST_32(x) bswap_32(x)

//#define IPODAUTH_RESET_HOLD_TIME 1  /* Time to hold reset line [ms] */
//#define IPODAUTH_RESET_WAIT_TIME 100 /* Time to wait after reset [ms] */

#elif __BYTE_ORDER == __BIG_ENDIAN

#define AUTH_TO_HOST_16(x) (x)
#define AUTH_TO_HOST_32(x) (x)

#else

#error - only big and little endian systems are supported.

#endif


//LOCAL S32 g_auth_data_fd    = -1;
//LOCAL S32 g_auth_gpio_reset_fd   = -1;
//LOCAL S32 g_auth_gpio_ready_fd   = -1;

/* Forward declaration of internal functions */

LOCAL S32 AUTH_transmit_cp_pages(S32 addr,
                                 U8 *buf,
                                 U32 length,
                                 U8 page_size,
                                 BOOL do_read);
LOCAL void AUTH_OSSleep(U32 sleep_ms);

#if 0
//libipod-auth-com.so functions, just return 0
extern "C" S32 iPodAuthComInit(void)
{
	ETG_TRACE_USR3(("iPodAuthComInit"));
	return IPOD_AUTH_OK;
}

extern "C" S32 iPodAuthComDeinit(void)
{
	ETG_TRACE_USR3(("iPodAuthComDeinit"));
	return IPOD_AUTH_OK;
}
#endif

//libipod-auth.so functions
S32 AuthenticationSetConfig(VP /*ConfigValues*/)
{
	ETG_TRACE_USR3(("AuthenticationSetConfig"));
    S32 rc = IPOD_AUTH_OK;
    return rc;
}

S32 AuthenticationFreeConfig(void)
{
	ETG_TRACE_USR3(("AuthenticationFreeConfig"));
    S32 rc = IPOD_AUTH_OK;

  

    return rc;
}

S32 AuthenticationInit(void)
{
	ETG_TRACE_USR3(("AuthenticationInit"));
    S32      rc = IPOD_AUTH_OK;
	 rc=AUTH_init_cp();
    return rc;
}

S32 AuthenticationDeinit(void)
{
	ETG_TRACE_USR3(("AuthenticationDeinit"));
    S32 rc = IPOD_AUTH_OK;
	rc=AUTH_deinit_cp();
    return rc;
}

void AuthenticationGetCertificate(U16 *cert_data_len, U8 *cert_data)
{
	ETG_TRACE_USR3(("AuthenticationGetCertificate"));
    S32 rc = IPOD_AUTH_OK;
    U16 tmp_len = 0x00;

    if((cert_data_len != NULL) && (cert_data != NULL))
    {
        rc = AuthenticationInit();
        if (rc == IPOD_AUTH_OK)
        {
            rc = AUTH_reset_cp();
            if (rc == IPOD_AUTH_OK)
            {
                rc = AUTH_read_cp(AUTH_ACC_CERT_DAT_LEN_ADDR,
                                  &tmp_len,
                                  AUTH_ACC_CERT_DAT_LEN_SIZE);
            }

            if (rc == IPOD_AUTH_OK)
            {
                /* PRQA: Lint Message 160: This error is occurred by Linux header. It can not fix in here */
                *cert_data_len = AUTH_TO_HOST_16(tmp_len); /*lint !e160 */

                if (*cert_data_len <= IPOD_AUTH_CP_MAX_CERTLENGTH)
                {
                    rc = AUTH_transmit_cp_pages(AUTH_ACC_CERT_DAT_ADDR,
                                                cert_data,
                                                (U32)*cert_data_len,
                                                AUTH_ACC_CERT_DAT_PAG_SIZE,
                                                TRUE);
                }
                else
                {
                    *cert_data_len = 0;
                }
            }
        }

        AuthenticationDeinit();
    }
    else
    {
        rc = -EINVAL;
    }

    /* TODO from Jakob: error handling? zero at least the out parameter */
	ETG_TRACE_USR3(("AuthenticationGetCertificate"));
}

/* tested by calling the API iPodAuthenticateiPod */
S32 AuthenticationSetCertificate(U16 cert_data_len, U8 *cert_data)
{
	ETG_TRACE_USR3(("AuthenticationSetCertificate"));
    S32 rc;
    /* PRQA: Lint Message 160: This error is occurred by Linux header. It can not fix in here */
    U16 tmp_len = AUTH_TO_HOST_16(cert_data_len); /*lint !e160 */

    if(cert_data != NULL)
    {
        rc = AUTH_reset_cp();
        rc = AUTH_write_cp(AUTH_IPOD_CERT_DAT_LEN_ADDR,
                           &tmp_len,
                           AUTH_IPOD_CERT_DAT_LEN_SIZE);

        if (rc == IPOD_AUTH_OK)
        {
            if (cert_data_len <= AUTH_MAX_IPOD_CERT_LEN)
            {
                rc = AUTH_transmit_cp_pages(AUTH_IPOD_CERT_DAT_ADDR,
                                            cert_data,
                                            (U32)cert_data_len,
                                            AUTH_IPOD_CERT_DAT_PAGE_SIZE,
                                            FALSE);
            }
        }
    }
    else
    {
        rc = -EINVAL;
    }

    return rc;
}

// TODO: test
S32 AuthenticationGetSignatureData(const U8  *response_buffer,
                                   U16 response_length,
                                   U16 *sig_data_len,
                                   U8 *sig_data)
{
	ETG_TRACE_USR3(("AuthenticationGetSignatureData"));
    S32 rc = IPOD_AUTH_OK;
    U8  ctl_byte    = 0x01;
    U8  proc_result = 0x00;
    U16 tmp_sig_data_len;
    U8 i = 0;

    if ((response_buffer == NULL) ||
        (sig_data_len == NULL) ||
        (sig_data == NULL))
    {
        return -EINVAL;
    }
    
    response_length = strlen_r(reinterpret_cast<const char*>(response_buffer));

    rc = AuthenticationInit();
    if (rc == IPOD_AUTH_OK)
    {
        rc = AUTH_reset_cp();
    }
    ETG_TRACE_USR3(("%d", rc));
    rc = AUTH_write_cp(AUTH_CHALL_DAT_ADDR,
                       (VP)response_buffer, //lint !e1773
					   response_length);
    ETG_TRACE_USR3(("%d", rc));
    if (rc == IPOD_AUTH_OK)
    {
        rc = AUTH_write_cp(AUTH_AUTH_CTL_STAT_ADDR,
                           &ctl_byte,
                           AUTH_AUTH_CTL_STAT_SIZE);
    }
    ETG_TRACE_USR3(("%d", rc));
    if (rc == IPOD_AUTH_OK)
    {
        for(i = 0; i < IPOD_I2C_RETRY_COUNT; i++)
        {
        	ETG_TRACE_USR3(("%d", i));
        	AUTH_OSSleep(500); //WORKAROUND!
            rc = AUTH_read_cp(AUTH_AUTH_CTL_STAT_ADDR,
                              &proc_result,
                              sizeof(proc_result));
            if(rc == IPOD_AUTH_OK)
            {
            	ETG_TRACE_USR3(("%d", proc_result));
                if(proc_result == 0x10)
                {
                    break;
                }
            }
            //AUTH_OSSleep(30);
        }
    }
    
    if (rc == IPOD_AUTH_OK)
    {
        rc = AUTH_read_cp(AUTH_SIG_DATA_LEN_ADDR,
                          &tmp_sig_data_len,
                          AUTH_SIG_DATA_LEN_SIZE);
        if (rc == IPOD_AUTH_OK)
        {
            /* PRQA: Lint Message 160: This error is occurred by Linux header. It can not fix in here */
            *sig_data_len = AUTH_TO_HOST_16(tmp_sig_data_len); /*lint !e160 */
            ETG_TRACE_USR3(("%d", *sig_data_len));
        }
        else
        {
            *sig_data_len = 0;
        }
    }

    ETG_TRACE_USR3(("%d", rc));
    ETG_TRACE_USR3(("%d", *sig_data_len));
    if ((rc == IPOD_AUTH_OK) && (IPOD_AUTH_CP_SIGN_DATA_SIZE >= *sig_data_len))
    {
        rc = AUTH_read_cp(AUTH_SIG_DATA_ADDR,
                          sig_data,
                          *sig_data_len);
    }
    else
    {
        rc = IPOD_AUTH_ERROR;
    }

    AuthenticationDeinit();

    ETG_TRACE_USR3(("%d", rc));
	ETG_TRACE_USR3(("AuthenticationGetSignatureData"));
    return rc;
}

/* tested by calling the API iPodAuthenticateiPod */
S32 AuthenticationGetSignature(U16 sig_data_len, U8 *sig_data)
{
	ETG_TRACE_USR3(("AuthenticationGetSignature"));
    S32 rc = IPOD_AUTH_OK;
    U8 ctl_byte = 0x03;
    U8 proc_result = 0x00;
    /* length must be converted for the Apple CP register */
    U16 tmp_sig_data_len = 0;

    if(sig_data != NULL)
    {
        /* PRQA: Lint Message 160: This error is occurred by Linux header. It can not fix in here */
        tmp_sig_data_len = AUTH_TO_HOST_16(sig_data_len); /*lint !e160 */

        rc = AUTH_write_cp(AUTH_SIG_DATA_LEN_ADDR,
                           &tmp_sig_data_len,
                           AUTH_SIG_DATA_LEN_SIZE);
        if (rc == IPOD_AUTH_OK)
        {
            /* do not use the converted length as number of bytes to write */
            rc = AUTH_write_cp(AUTH_SIG_DATA_ADDR,
                               sig_data,
                               sig_data_len);
        }

        if (rc == IPOD_AUTH_OK)
        {
            rc = AUTH_write_cp(AUTH_AUTH_CTL_STAT_ADDR,
                               &ctl_byte,
                               AUTH_AUTH_CTL_STAT_SIZE);
            if (rc == IPOD_AUTH_OK)
            {
                rc = AUTH_read_cp(AUTH_AUTH_CTL_STAT_ADDR,
                                  &proc_result,
                                  AUTH_AUTH_CTL_STAT_SIZE);
                if (rc == IPOD_AUTH_OK)
                {
                    if (proc_result != 0x30)
                    {
                        rc = -EINVAL;
                    }
                }
            }
        }
    }
    else
    {
        rc = -EINVAL;
    }

    return rc;
}

/* tested by calling the API iPodAuthenticateiPod */
S32 AuthenticationGetChallengeData(U16 *challenge_data_len,
                                   U8 *challenge_data)
{
    S32 rc;
    U16 tmp_challenge_data_len;
    U8 ctl_cmd    = 0x02;
    U8 cmd_status = 0x00;

    if((challenge_data != NULL) && (challenge_data_len != NULL))
    {
        rc = AUTH_write_cp(AUTH_AUTH_CTL_STAT_ADDR,
                           &ctl_cmd,
                           AUTH_AUTH_CTL_STAT_SIZE);

        if (rc == IPOD_AUTH_OK)
        {
            rc = AUTH_read_cp(AUTH_AUTH_CTL_STAT_ADDR,
                              &cmd_status,
                              AUTH_AUTH_CTL_STAT_SIZE);

            if (rc == IPOD_AUTH_OK)
            {
                if ((cmd_status & 0x20) != 0x20)
                {
                    rc = -EINVAL;
                }
            }
        }

        if (rc == IPOD_AUTH_OK)
        {
            rc = AUTH_read_cp(AUTH_CHALL_DAT_LEN_ADDR,
                              &tmp_challenge_data_len,
                              AUTH_CHALL_DAT_LEN_SIZE);

            if (rc == IPOD_AUTH_OK)
            {
                /* PRQA: Lint Message 160: This error is occurred by Linux header. It can not fix in here */
                *challenge_data_len = AUTH_TO_HOST_16(tmp_challenge_data_len); /*lint !e160 */
            }
            else
            {
                *challenge_data_len = 0;
            }
        }

        if ((rc == IPOD_AUTH_OK) && (IPOD_AUTH_CP_SIGN_DATA_SIZE >= *challenge_data_len))
        {
            rc = AUTH_read_cp(AUTH_CHALL_DAT_ADDR,
                              challenge_data,
                              *challenge_data_len);
        }
        else
        {
            rc = IPOD_AUTH_ERROR;
        }
    }
    else
    {
        rc = -EINVAL;
    }

    return rc;
}

/* Authentication CP test functions */
S32 AuthenticationGetDeviceID(U32 *auth_dev_id)
{
	ETG_TRACE_USR3(("AuthenticationGetDeviceID"));
    S32 rc = IPOD_AUTH_OK;
    U32 tmp_id = 0;

    if (auth_dev_id != NULL)
    {
        rc = AuthenticationInit();
        if (rc == 0)
        {

            rc = AUTH_reset_cp();
            if (rc == 0)
            {
                rc = AUTH_read_cp(AUTH_DEV_ID_ADDR, &tmp_id, AUTH_DEV_ID_SIZE);
            }
            
            if (rc == 0)
            {
                /* PRQA: Lint Message 160: This error is occurred by Linux header. It can not fix in here */
                *auth_dev_id = AUTH_TO_HOST_32(tmp_id); /*lint !e160 */
				ETG_TRACE_USR3(("%d", tmp_id));
				ETG_TRACE_USR3(("%d", *auth_dev_id));
            }
        }

        AuthenticationDeinit();
    }
    else
    {
        rc = -EINVAL;
    }

    ETG_TRACE_USR3(("AuthenticationGetDeviceID"));
    return rc;
}

S32 AuthenticationGetFirmwareVersion(U8 *majorVer, U8 *minorVer)
{
	ETG_TRACE_USR3(("AuthenticationGetFirmwareVersion"));
    S32 rc = IPOD_AUTH_OK;

    if((majorVer != NULL) && (minorVer != NULL))
    {
        rc = AuthenticationInit();
        if (rc == IPOD_AUTH_OK)
        {
            rc = AUTH_reset_cp();
        }

        if (rc == IPOD_AUTH_OK)
        {
            rc = AUTH_read_cp(AUTH_FW_VER_ADDR, majorVer, AUTH_FW_VER_SIZE);
            *minorVer = 0;
        }
        AuthenticationDeinit();
    }
    else
    {
        rc = -EINVAL;
    }
    
    return rc;
}

S32 AuthenticationGetProtocolVersion(U8 *major_ver, U8 *minor_ver)
{
	ETG_TRACE_USR3(("AuthenticationGetProtocolVersion"));
    S32 rc = IPOD_AUTH_OK;

    if ((major_ver != NULL) && (minor_ver != NULL))
    {
        rc = AuthenticationInit();
        if (rc == IPOD_AUTH_OK)
        {
            rc = AUTH_reset_cp();
            if (rc == IPOD_AUTH_OK)
            {
                rc = AUTH_read_cp(AUTH_PROT_MAJOR_VERS_ADDR,
                                  major_ver,\
                                  AUTH_PROT_MAJOR_VERS_SIZE);
                if (rc == IPOD_AUTH_OK)
                {
                    rc = AUTH_read_cp(AUTH_PROT_MINOR_VERS_ADDR,
                                      minor_ver,
                                      AUTH_PROT_MINOR_VERS_SIZE);
                }
            }
        }
        AuthenticationDeinit();
    }
    else
    {
        rc = -EINVAL;
    }

    return rc;
}

// TODO: test
S32 AuthenticationSelftest(U8 *certificate,
                           U8 *private_key,
                           U8 *ram_check,
                           U8 *checksum)
{
    S32 rc = IPOD_AUTH_OK;
    U8 ctl_cmd = 0x01;
    U8 cmd_status;

    if ((certificate != NULL) && (private_key != NULL) &&
    (ram_check != NULL) && (checksum != NULL))
    {
        rc = AuthenticationInit();

        if (rc == IPOD_AUTH_OK)
        {
            rc = AUTH_reset_cp();
        }

        if (rc == IPOD_AUTH_OK)
        {
            rc = AUTH_write_cp(AUTH_SELF_TEST_CTL_STAT_ADDR,
                       &ctl_cmd,
                       AUTH_SELF_TEST_CTL_STAT_SIZE);

            if (rc == IPOD_AUTH_OK)
            {
                rc = AUTH_read_cp(AUTH_SELF_TEST_CTL_STAT_ADDR,
                      &cmd_status,
                      AUTH_SELF_TEST_CTL_STAT_SIZE);
                if (rc == IPOD_AUTH_OK)
                {
                    *certificate = cmd_status & 0x80;
                    *private_key = cmd_status & 0x40;
                    *ram_check = 0;
                    *checksum = 0;
                }
            }
        }
        
        AuthenticationDeinit();
    }
    else
    {
        rc = -EINVAL;
    }

    return rc;
}

LOCAL S32 AUTH_transmit_cp_pages(S32 addr,
                                 U8 *buf,
                                 U32 length,
                                 U8 page_size,
                                 BOOL do_read)
{
    S32 rc = IPOD_AUTH_OK;
    U8 i;
    U8 num_segments;
    U8 remaining_bytes;
    U8 transfer_len;
    U8 *buf_addr;

    if((buf == NULL) || (page_size == 0))
    {
        return IPOD_AUTH_ERROR;
    }
    
    num_segments = length / page_size;
    remaining_bytes = length % page_size;
    for (i = 0; (i < num_segments + 1) && (rc == IPOD_AUTH_OK); i++)
    {
        buf_addr = &buf[i * page_size];

        if (i < num_segments) /* get full pages */
        {
            transfer_len = page_size;
        }
        else /* get remaining bytes */
        {
            transfer_len = remaining_bytes;
        }

        if (transfer_len != 0) /* only if remaining bytes != 0 */
        {
            if (do_read == TRUE)
            {
                rc = AUTH_read_cp(addr + i,
                                  buf_addr,
                                  (U32)transfer_len);
            }
            else
            {
                rc = AUTH_write_cp(addr + i,
                                   buf_addr,
                                   (U32)transfer_len);
            }
        }
    }

    return rc;
}


LOCAL void AUTH_OSSleep(U32 sleep_ms)
{
    S32 s32ReturnValue = IPOD_AUTH_ERROR;
    struct timespec req;
    struct timespec remain;

    /* Initialize the structure */
    memset(&req, 0, sizeof(req));
    memset(&remain, 0, sizeof(remain));


    req.tv_sec = sleep_ms / IPOD_AUTH_MSEC;
    req.tv_nsec = (sleep_ms % IPOD_AUTH_MSEC) * IPOD_AUTH_NSEC;

    while(1)
    {
        s32ReturnValue = nanosleep(&req, &remain);

        if (s32ReturnValue == 0)
        {
            break;
        }
        else
        {
            if (errno == EINTR)
            {
                req.tv_sec = remain.tv_sec ;
                req.tv_nsec = remain.tv_nsec;
            }
            else
            {
                break;
            }
        }
    }// end while

}
