/*!
********************************************************************************
* \file              ipodauth_tclImpTraceStreamable.cpp
********************************************************************************
*  - PROJECT:        GM Gen2
*  - SW-COMPONENT:   iPod Authentication
*  - DESCRIPTION:    Implementation of the Trace Streamer 
*                    Trace commands are setup using this class.
*  - COPYRIGHT:      &copy; 2010 Robert Bosch GmbH, Hildesheim
********************************************************************************
* \date 17.03.2011 \version 1.2 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* - Added new trace test functionality to read out registry entries.
*
* \date 13.01.2011 \version 1.1 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* - Migrated codebase to use Infotainment Helper Library (IHL).
*
* \date 01.10.2010 \version 1.0 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* \bug No known bugs
*******************************************************************************/

/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/

#define FI_S_IMPORT_INTERFACE_FI_MESSAGE
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_IPODAUTHFI_TYPES
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_IPODAUTHFI_FUNCTIONIDS
#include <midw_fi_if.h>

#include "ipodauth_tclImpTraceStreamable.h"
#include "ipodauth_tclRegistry.h"
#include "ipodauth_Trace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_IPODAUTH_CMD_IF
#include "trcGenProj/Header/ipodauth_tclImpTraceStreamable.cpp.trc.h"
#endif

/******************************************************************************
| defines and macros (scope: module-local)
|----------------------------------------------------------------------------*/
/// Internal function id to read registry entry values.
#define IPODAUTH_C_U16_READREGISTRYENTRIES      (IHL_C_U16_TRACE_STREAM_FID - 1)

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/

/*! 
 * \typedef ihl_tclTraceCmdFunctor<ipodauth_tclImpTraceStreamable> ipodauth_tCommand
 * \brief Trace Command object definition.
 */
typedef ihl_tclTraceCmdFunctor<ipodauth_tclImpTraceStreamable> ipodauth_tCommand;

/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| function prototype (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| function implementation (scope: external-interfaces)
|----------------------------------------------------------------------------*/

/******************************************************************************
** FUNCTION:  ipodauth_tclImpTraceStreamable::ipodauth_tclImpTraceStreamable(ahl_.
******************************************************************************/

/*explicit*/
ipodauth_tclImpTraceStreamable::ipodauth_tclImpTraceStreamable
(
   ahl_tclBaseOneThreadApp *const cpoApp
):ihl_tclTraceStreamable(cpoApp)
{}

/******************************************************************************
** FUNCTION:  virtual ipodauth_tclImpTraceStreamable::~ipodauth_tclImpTraceStrea..
******************************************************************************/

/*virtual*/
ipodauth_tclImpTraceStreamable::~ipodauth_tclImpTraceStreamable(){}

/******************************************************************************
** FUNCTION:  virtual tVoid ipodauth_tclImpTraceStreamable::vSetupCmds()
******************************************************************************/

/*virtual*/
tVoid ipodauth_tclImpTraceStreamable::vSetupCmds()
{
   // Add the command
   vAddCmd(MIDW_IPODAUTHFI_C_U16_FIRMWAREVERSION
      , OSAL_NEW ipodauth_tCommand(this, &ipodauth_tclImpTraceStreamable::bGetFirmwareVer));
   
   vAddCmd(MIDW_IPODAUTHFI_C_U16_AUTHENTICATIONPROTOCOLVERSION
      , OSAL_NEW ipodauth_tCommand(this, &ipodauth_tclImpTraceStreamable::bGetAuthProtocolVer));
   
   vAddCmd(MIDW_IPODAUTHFI_C_U16_DEVICEID
      , OSAL_NEW ipodauth_tCommand(this, &ipodauth_tclImpTraceStreamable::bGetDeviceId));

   vAddCmd(MIDW_IPODAUTHFI_C_U16_ACCESSORYCERTIFICATE
      , OSAL_NEW ipodauth_tCommand(this, &ipodauth_tclImpTraceStreamable::bGetAccessoryCertificate));

   vAddCmd(MIDW_IPODAUTHFI_C_U16_IPODAUTHENTICATIONSELFTEST
      , OSAL_NEW ipodauth_tCommand(this, &ipodauth_tclImpTraceStreamable::biPodAuthSelftest));

   vAddCmd(IPODAUTH_C_U16_READREGISTRYENTRIES
      , OSAL_NEW ipodauth_tCommand(this, &ipodauth_tclImpTraceStreamable::bReadRegistry));

}  // tVoid ipodauth_tclImpTraceStreamable::vSetupCmds()

/******************************************************************************
** FUNCTION:  tBool ipodauth_tclImpTraceStreamable::bGetFirmwareVer(tU8 const..
******************************************************************************/

tBool ipodauth_tclImpTraceStreamable::bGetFirmwareVer(tU8 const* const cpu8Buffer) const
{
   // Discard the Buffer as there is no information that is required.
   // Avoid Lint - Parameter needs to be neglected without LINT objection.
   // This parameter may be required for future.
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(cpu8Buffer);

   midw_ipodauthfi_tclMsgFirmwareVersionGet oFirmwareVer;
   
   return (bSendMsg(oFirmwareVer));
}  // tBool ipodauth_tclImpTraceStreamable::bGetFirmwareVer(tU8 const* con..


/******************************************************************************
** FUNCTION:  tBool ipodauth_tclImpTraceStreamable::bGetAuthProtocolVer(tU8 ..
******************************************************************************/

tBool ipodauth_tclImpTraceStreamable::bGetAuthProtocolVer(tU8 const* const cpu8Buffer) const
{
   // Discard the Buffer as there is no information that is required.
   // Avoid Lint - Parameter needs to be neglected without LINT objection.
   // This parameter may be required for future.
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(cpu8Buffer);

   midw_ipodauthfi_tclMsgAuthenticationProtocolVersionGet oAuthProtocolVer;
   
   return (bSendMsg(oAuthProtocolVer));
}  // tBool ipodauth_tclImpTraceStreamable::bGetAuthProtocolVer(tU8 const* con..

/******************************************************************************
** FUNCTION:  tBool ipodauth_tclImpTraceStreamable::bGetDeviceId(tU8 const..
******************************************************************************/

tBool ipodauth_tclImpTraceStreamable::bGetDeviceId(tU8 const* const cpu8Buffer) const
{
   // Discard the Buffer as there is no information that is required.
   // Avoid Lint - Parameter needs to be neglected without LINT objection.
   // This parameter may be required for future.
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(cpu8Buffer);

   midw_ipodauthfi_tclMsgDeviceIDGet oDeviceId;
   
   return (bSendMsg(oDeviceId));
}  // tBool ipodauth_tclImpTraceStreamable::bGetDeviceId(tU8 const* con..

/******************************************************************************
** FUNCTION:  tBool ipodauth_tclImpTraceStreamable::bGetAccessoryCertificate..
******************************************************************************/

tBool ipodauth_tclImpTraceStreamable::bGetAccessoryCertificate(tU8 const* const cpu8Buffer) const
{
   // Discard the Buffer as there is no information that is required.
   // Avoid Lint - Parameter needs to be neglected without LINT objection.
   // This parameter may be required for future.
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(cpu8Buffer);

   midw_ipodauthfi_tclMsgAccessoryCertificateGet oAccCertificate;
   
   return (bSendMsg(oAccCertificate));
}  // tBool ipodauth_tclImpTraceStreamable::bGetAccessoryCertificate(tU8 cons..

/******************************************************************************
** FUNCTION:  tBool ipodauth_tclImpTraceStreamable::biPodAuthSelftest(tU..
******************************************************************************/

tBool ipodauth_tclImpTraceStreamable::biPodAuthSelftest(tU8 const* const cpu8Buffer) const
{
   // Discard the Buffer as there is no information that is required.
   // Avoid Lint - Parameter needs to be neglected without LINT objection.
   // This parameter may be required for future.
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(cpu8Buffer);

   midw_ipodauthfi_tclMsgiPodAuthenticationSelftestMethodStart oiPodAuthSelfTest;
   
   return (bSendMsg(oiPodAuthSelfTest));
}  // tBool ipodauth_tclImpTraceStreamable::biPodAuthSelftest(tU8 cons..

/******************************************************************************
** FUNCTION:  tBool ipodauth_tclImpTraceStreamable::bReadRegistry(tU..
******************************************************************************/

tBool ipodauth_tclImpTraceStreamable::bReadRegistry(tU8 const* const cpu8Buffer) const
{
   ipodauth_tclRegistry oiPodReg(IPODAUTH_CP_REG_PATH, IPODAUTH_EN_RDWR);
   
   tU32 u32MajorVer  =  0;
   tU32 u32MinorVer  =  0;

   if (FALSE == oiPodReg.bQueryU32(IPODAUTH_CP_FIRMWARE_VER, &u32MajorVer))
   {
      ETG_TRACE_ERR(("Failed to get the registry entry: %s", IPODAUTH_CP_FIRMWARE_VER));
   }
   else if (FALSE == oiPodReg.bQueryU32(IPODAUTH_CP_FIRMWARE_MIN_VER, &u32MinorVer))
   {
      ETG_TRACE_ERR(("Failed to get the registry entry: %s", IPODAUTH_CP_FIRMWARE_MIN_VER));
   }
   else
   {
      ETG_TRACE_USR4(("Read registry - Firmware version:%02d.%02d", u32MajorVer, u32MinorVer));
   }   

   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(cpu8Buffer);
   return TRUE;
}  // tBool ipodauth_tclImpTraceStreamable::bReadRegistry(tU8 cons..

////////////////////////////////////////////////////////////////////////////////
// <EOF>
