/*!
********************************************************************************
* \file              ipodauth_tclImpTraceStreamable.h
********************************************************************************
*  - PROJECT:        GM Gen2
*  - SW-COMPONENT:   iPod Authentication
*  - DESCRIPTION:    Implementation of the Trace Streamer 
*                    Trace commands are setup using this class.
*  - COPYRIGHT:      &copy; 2010 Robert Bosch GmbH, Hildesheim
********************************************************************************
* \date 17.03.2011 \version 1.2 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* - Added new trace test functionality to read out registry entries.
*
* \date 13.01.2011 \version 1.1 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* - Migrated codebase to use Infotainment Helper Library (IHL).
*
* \date 01.10.2010 \version 1.0 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* \bug No known bugs
*******************************************************************************/

#ifndef IPODAUTH_TCLIMPTRACESTREAMABLE_H_
#define IPODAUTH_TCLIMPTRACESTREAMABLE_H_

/******************************************************************************
| includes of component-internal interfaces, if necessary
| (scope: component-local)
|----------------------------------------------------------------------------*/

#define IHL_S_IMPORT_INTERFACE_LOOPBACK
#include <ihl_if.h>
using namespace ihl::loopback;

/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| typedefs (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| forward declarations (scope: global)
|----------------------------------------------------------------------------*/

/*! 
 * \class ipodauth_tclImpTraceStreamable
 * \brief Implementation of the Trace Streamer 
 * Trace commands are setup using this class.
 */

class ipodauth_tclImpTraceStreamable : public ihl_tclTraceStreamable
{
public:
   
   /***************************************************************************
   *********************************PUBLIC*************************************
   ***************************************************************************/
   
   /***************************************************************************
   ** FUNCTION:  ipodauth_tclImpTraceStreamable::ipodauth_tclImpTraceStreamab...
   ***************************************************************************/
   /*!
    * \brief   Parameterized Constructor, based on Dependency Injection 
    *          Principle (DIP)
    * \param   [cpoApp]:        (->I) Pointer to the main application
    * \retval  NONE
    * \date 01.10.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   explicit ipodauth_tclImpTraceStreamable(ahl_tclBaseOneThreadApp* const cpoApp);

   /***************************************************************************
   ** FUNCTION:  virtual ipodauth_tclImpTraceStreamable::~ipodauth_tclImpTrac...
   ***************************************************************************/
   /*!
    * \brief   Destructor
    * \param   NONE
    * \retval  NONE
    * \date 01.10.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual ~ipodauth_tclImpTraceStreamable();

   /***************************************************************************
   ** FUNCTION:  virtual tVoid ipodauth_tclImpTraceStreamable::vSetupCmds()
   ***************************************************************************/
   /*!
    * \brief   The function sets up the commands supported via trace.
    *          This is used for mapping trace commands provided via the Trace 
    *          input channel interface.
    * \param   NONE
    * \retval  NONE
    * \date 01.10.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tVoid vSetupCmds();

   /***************************************************************************
   ****************************END OF PUBLIC***********************************
   ***************************************************************************/

protected:

   /***************************************************************************
   *******************************PROTECTED************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  tBool ipodauth_tclImpTraceStreamable::bGetFirmwareVer(tU8 ..
   ***************************************************************************/
   /*!
    * \brief   Get Apple Coprocessor Firmware Version - Trace command executor
    *          The function converts the trace commands to Loopback message, 
    *          which are posted back to the component to be processed using the 
    *          normal CCA interface.
    * \param   [cpu8Buffer]:  (->I) Input trace channel buffer.
    * \retval  [tBool]: TRUE if Authenticate iPod Firmware Version command 
    *          was successfully executed, FALSE otherwise.
    * \date 01.10.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tBool bGetFirmwareVer(tU8 const* const cpu8Buffer) const;

   /***************************************************************************
   ** FUNCTION:  tBool ipodauth_tclImpTraceStreamable::bGetAuthProtocolVer(tU..
   ***************************************************************************/
   /*!
    * \brief   Get Apple Coprocessor Authentication iPod Protocol Version 
    *          - Trace command executor
    *          The function converts the trace commands to Loopback message, 
    *          which are posted back to the component to be processed using the 
    *          normal CCA interface.
    * \param   [cpu8Buffer]:  (->I) Input trace channel buffer.
    * \retval  [tBool]: TRUE if Authenticate iPod Protocol Version command was 
    *          successfully executed, FALSE otherwise.
    * \date 01.10.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tBool bGetAuthProtocolVer(tU8 const* const cpu8Buffer) const;

   /***************************************************************************
   ** FUNCTION:  tBool ipodauth_tclImpTraceStreamable::bGetDeviceId(tU8 cons..
   ***************************************************************************/
   /*!
    * \brief   Get Apple Coprocessor Device Id - Trace command executor
    *          The function converts the trace commands to Loopback message, 
    *          which are posted back to the component to be processed using the 
    *          normal CCA interface.
    * \param   [cpu8Buffer]:  (->I) Input trace channel buffer.
    * \retval  [tBool]: TRUE if Apple Co processor Get Device Id command was 
    *          successfully executed, FALSE otherwise.
    * \date 01.10.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tBool bGetDeviceId(tU8 const* const cpu8Buffer) const;

   /***************************************************************************
   ** FUNCTION:  tBool ipodauth_tclImpTraceStreamable::bGetAccessoryCertifica..
   ***************************************************************************/
   /*!
    * \brief   Get Apple Coprocessor Acc Certificate - Trace command executor
    *          The function converts the trace commands to Loopback message, 
    *          which are posted back to the component to be processed using the 
    *          normal CCA interface.
    * \param   [cpu8Buffer]:  (->I) Input trace channel buffer.
    * \retval  [tBool]: TRUE if Accessory certificate command was successfully 
    *          executed, FALSE otherwise.
    * \date 01.10.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tBool bGetAccessoryCertificate(tU8 const* const cpu8Buffer) const;

   /***************************************************************************
   ** FUNCTION:  tBool ipodauth_tclImpTraceStreamable::biPodAuthSelftest(tU8..
   ***************************************************************************/
   /*!
    * \brief   Trigger iPod Authentication Self test - Trace command executor
    *          The function converts the trace commands to Loopback message, 
    *          which are posted back to the component to be processed using the 
    *          normal CCA interface.
    * \param   [cpu8Buffer]:  (->I) Input trace channel buffer.
    * \retval  [tBool]: TRUE if iPod Authentication Self test command was 
    *          successfully executed, FALSE otherwise.
    * \date 01.10.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tBool biPodAuthSelftest(tU8 const* const cpu8Buffer) const;

   /***************************************************************************
   ** FUNCTION:  tBool ipodauth_tclImpTraceStreamable::bReadRegistry(tU8..
   ***************************************************************************/
   /*!
    * \brief   Trigger reading registry entry.
    * \param   [cpu8Buffer]:  (->I) Input trace channel buffer.
    * \retval  [tBool]: TRUE if iPod Authentication Self test command was 
    *          successfully executed, FALSE otherwise.
    * \date 01.10.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tBool bReadRegistry(tU8 const* const cpu8Buffer) const;

   /***************************************************************************
   ** FUNCTION:  ipodauth_tclImpTraceStreamable::ipodauth_tclImpTraceStreamable()
   ***************************************************************************/
   /*!
    * \brief   Default Constructor
    * \param   NONE
    * \retval  NONE
    * \note    Default constructor is declared protected to disable it. So 
    *          that any attempt to create without parameter will be caught by 
    *          the compiler.
    * \date 01.10.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   ipodauth_tclImpTraceStreamable();      // No definition exists.

   /***************************************************************************
   ** FUNCTION:  ipodauth_tclImpTraceStreamable::ipodauth_tclImpTraceStream..
   ***************************************************************************/
   /*!
    * \brief   Copy Constructor
    * \param   [rfcoTraceStrm]: (I) Const reference to object to be copied
    * \retval  NONE
    * \note    Default copy constructor is declared protected to disable it. So 
    *          that any attempt to copy will be caught by the compiler.
    * \date 01.10.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   ipodauth_tclImpTraceStreamable(const ipodauth_tclImpTraceStreamable& rfcoTraceStrm);

   /***************************************************************************
   ** FUNCTION:  ipodauth_tclImpTraceStreamable& ipodauth_tclImpTraceStream....
   ***************************************************************************/
   /*!
    * \brief   Assignment Operator
    * \param   [rfcoTraceStrm]: (I) Const reference to object to be copied
    * \retval  [ipodauth_tclImpTraceStreamable&]: Reference to this pointer.
    * \note    Assignment operator is declared protected to disable it. So 
    *          that any attempt for assignment will be caught by the compiler.
    * \date 01.10.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   ipodauth_tclImpTraceStreamable& operator=(const ipodauth_tclImpTraceStreamable& rfcoTraceStrm);

   /***************************************************************************
   ****************************END OF PROTECTED********************************
   ***************************************************************************/

private:

   /***************************************************************************
   *********************************PRIVATE************************************
   ***************************************************************************/

   /***************************************************************************
   ****************************END OF PRIVATE**********************************
   ***************************************************************************/
}; // class ipodauth_tclImpTraceStreamable : public ihl_tclTraceStreamable

#endif   // #ifndef IPODAUTH_TCLIMPTRACESTREAMABLE_H_

////////////////////////////////////////////////////////////////////////////////
// <EOF>
