/**
  * @swcomponent  Life Cycle Management
  * @{
  * @file         spm_Config_generic.h
  * @brief        configuration options for LCM features
  * @copyright    (C) 2012 - 2017 Robert Bosch GmbH.
  *               The reproduction, distribution and utilization of this file as well as the
  *               communication of its contents to others without express authorization is prohibited.
  *               Offenders will be held liable for the payment of damages.
  *               All rights reserved in the event of the grant of a patent, utility model or design.
  * @}
  */

#pragma once

/*! ******************************************************
   used to check if this file is included prior to any other LCM header file
  */
#define _SPM_CONFIG_H

#include "spm_GlobDefs.h"

/*! ******************************************************
 show additional debug info:
 checks in factory if double interfaces are used
 *********************************************************/
#define SPM_ENABLE_ADDITIONAL_DEBUG_INFO

/*! ******************************************************
 Debouncing of single SubStates enabled
**********************************************************/
#define SPM_FEATURE_ENABLE_SUBSTATE_DEBOUNCE

/*! ******************************************************
 import configuration w/o absolute paths -> paths must be added
**********************************************************/
#define SPM_FEATURE_ENABLE_CONFIG_WO_ABS_PATHES

/*! ******************************************************
 define voltage level which blocks statemaschine
 check also SPM_U32_SHUTDOWN_TYPES_EN_LOW_VOLT for exceptions to this rule.
**********************************************************/
#define SPM_U32_LOW_VOLTAGE_BLOCKING_TRIGGER SPM_U32_CRITICAL_LOW_VOLTAGE

/*! ******************************************************
 Use permanent High Voltage Delay (from dev_volt) as high vlotage trigger
******************************************************/
// #define SPM_FEATURE_ENABLE_HIGH_VOLT_PERMANENT_TRIGGER

/*! ******************************************************
 Use permanent CR High Voltage Delay (from dev_volt) as high vlotage trigger
******************************************************/
// #define SPM_FEATURE_ENABLE_CRITICAL_HIGH_VOLT_PERMANENT_TRIGGER

/*! ******************************************************
 Processes to be killed in case of DOWNLOAD mode\n
 e.g.\n
 #define SPM_DOWNLOAD_PROCS_2_KILL    "PROCNAVI", "PROCDAPI", "PROCSDS", "PROCSDSTS", "CCAMEDIAPLAYER"
**********************************************************/
#define SPM_DOWNLOAD_PROCS_2_KILL    "PROCNAVI", "PROCDAPI", "PROCSDS", "PROCSDSTS", "CCAMEDIAPLAYER"

/*! ******************************************************
 services to be stopped in case of DOWNLOAD mode\n
 e.g.\n
 #define SPM_DOWNLOAD_SERVICES_2_STOP    "ExampleApp.service", "rbcm-apphmi_test_01.service"
**********************************************************/
#define SPM_DOWNLOAD_SERVICES_2_STOP    "ExampleApp.service", "rbcm-apphmi_test_01.service"

/*! ******************************************************
 services to be started in case of DOWNLOAD mode\n
 e.g.\n
 #define SPM_DOWNLOAD_SERVICES_2_START    "ExampleApp.service", "rbcm-apphmi_test_01.service"
  */
#define SPM_DOWNLOAD_SERVICES_2_START    "LcmTestApp.service"

/*! ******************************************************
 Define the project download state (default is SPM_SYSTEM_DOWNLOAD)\n
 Needed for CCA method SYSMODE
**********************************************************/
//   #define SPM_PRJ_SYSTEM_STATE_DOWNLOAD    SPM_SYSTEM_<XXX>

/*! ******************************************************
 applications that do not change their applicationstate when DIAG_FAST_SHUTDOWN triggers fast
 shutdown to PREPARE_SHUTDOWN
**********************************************************/
/*
//#define SPM_DIAG_FAST_SHUTDOWN_APPS_IGNORE CCA_C_U16_APP_NAVI,\
//                                           CCA_C_U16_APP_TIMA,\
//                                           CCA_C_U16_APP_DATA_SERVER,\
//                                           CCA_C_U16_APP_FC_TEA,\
//                                           CCA_C_U16_APP_NAVSTORAGE,\
//                                           CCA_C_U16_APP_NAVI_SRV,\
//                                           CCA_C_U16_APP_NAVCOPY,\
//                                           CCA_C_U16_APP_HMI_12

*/
/*! ******************************************************
 cyclic check time for application state transitions
**********************************************************/
#define SPM_APPREQMON_CHECK_INTERVAL     2000

/*! ******************************************************
 default timeout for application state transition supervision
**********************************************************/
#ifdef DEBUG
   #define SPM_APPREQMON_DEFAULT_INTERVAL   20000 // in ms
#else
   #define SPM_APPREQMON_DEFAULT_INTERVAL   13000 // in ms
#endif

/*! ******************************************************
 warning timeout for application state transition supervision
**********************************************************/
#define SPM_APPREQMON_WARNING_INTERVAL    4999    // in ms

/*! ******************************************************
 startup timeout for application state transition supervision
**********************************************************/
#define SPM_APPREQMON_STARTUP_INTERVAL   90000    // in ms

// ******************************************************
// Wakeup configurations
// configuration of certain wake-up reasons to be considered for the initial wake-up of the system.
// ******************************************************

/*!
   defines the default wakeup config and this configuaration is set during startup.
   wake-up reasons will be enabled in default config are : ON_TIPPER, CAN, IGN_PIN and TEL_MUTE.
  */
#define SPM_U32_WAKEUPCONFIG_DEFAULT      ( DEV_WUP_C_U32_WAKEUP_REASONS_MASK_ON_TIPPER | DEV_WUP_C_U32_WAKEUP_REASONS_MASK_CAN | DEV_WUP_C_U32_WAKEUP_REASONS_MASK_IGN_PIN | DEV_WUP_C_U32_WAKEUP_REASONS_MASK_TEL_MUTE )

/*!
   defines the wakeup config for phone.mute.
  */
#define SPM_U32_WAKEUPCONFIG_PHONE_MUTE    ( DEV_WUP_C_U32_WAKEUP_REASONS_MASK_TEL_MUTE )

/*!
   defines the wakeup config for network activity.
  */
#define SPM_U32_WAKEUPCONFIG_NETWORK_ACTIVITY    ( DEV_WUP_C_U32_WAKEUP_REASONS_MASK_CAN )

/*!
   defines the wakeup config for TML testing purpose ( valid wake-up reasons which have mapping substates triggers).
  */
#define SPM_U32_WAKEUPCONFIG_TML_TEST_1    ( DEV_WUP_C_U32_WAKEUP_REASONS_MASK_IGN_PIN | DEV_WUP_C_U32_WAKEUP_REASONS_MASK_ON_TIPPER | DEV_WUP_C_U32_WAKEUP_REASONS_MASK_EXTERNAL_PIN_WAKEUP )

/*!
   defines the wakeup config for TML testing purpose (valid wake-up reasons which does not have mapping substates triggers).
   an ErrMem Entry is added when any of these wake-up reasons are configured.
  */
#define SPM_U32_WAKEUPCONFIG_TML_TEST_2    ( DEV_WUP_C_U32_WAKEUP_REASONS_MASK_MOST | DEV_WUP_C_U32_WAKEUP_REASONS_MASK_DEBUG_WAKEUP | DEV_WUP_C_U32_WAKEUP_REASONS_MASK_RTC_WAKEUP | DEV_WUP_C_U32_WAKEUP_REASONS_MASK_ODOMETER_WAKEUP )

/*! ******************************************************
 ENABLE SPM Wakeup On/Off State - CONTACT
**********************************************************/
// #define SPM_WUP_ONOFF_STATE_S_CONTACT_ENABLED

/*! ******************************************************
 ENABLE SPM Wakeup On/Off State - IGNITION
**********************************************************/
#define SPM_WUP_ONOFF_STATE_IGNITION_ENABLED

/*! ******************************************************
 ENABLE SPM Wakeup On/Off State - NETWORK CAN
**********************************************************/
// #define SPM_WUP_ONOFF_STATE_CAN_ENABLED

/*! ******************************************************
 ENABLE SPM Wakeup On/Off State - CD CLAMP
**********************************************************/
// #define SPM_WUP_ONOFF_STATE_CD_CLAMP_ENABLED

/*! ******************************************************
 ENABLE SPM Wakeup On/Off State - ILLUMINATION
**********************************************************/
// #define SPM_WUP_ONOFF_STATE_ILLUMINATION_ENABLED

// CMF1 configuration with full wakeup support
// {(SPM_U8_HW_WUP_EN_TIPPER | SPM_U8_HW_WUP_EN_EJECT | SPM_U8_HW_WUP_EN_ACC | SPM_U8_HW_WUP_EN_INSERT | SPM_U8_HW_WUP_EN_ILLUM ),                    (0), (0xf0), (0x00)} 

/*! ******************************************************
 additional substates to check if point of no return is reached -> for this states current wakeup config is not checked
 ACTIVE: Changes from Off->On will be lead to a restart
 INACTIVE: Changes from On->Off will be lead to a restart
**********************************************************/
//#define SPM_SUBSTATE_WAKEUP_RESTART_ACTIVE        SPM_U32_IGNITION
#define SPM_SUBSTATE_WAKEUP_RESTART_INACTIVE      SPM_U32_OVERTEMP, SPM_U32_FAST_SHUTDOWN


/*! ******************************************************
 go directly to shutdown when reset in shutdown phase is detected
**********************************************************/
// #define SPM_ENABLE_OFF_AFTER_RESET_IN_SHUTDOWN

// ******************************************************
// Startup configuration
// ******************************************************

/*!
   defines the array with the possible wakeup configurations
   if you want to define different wakeup configurations for different numbers of CPU cores
   put them one after the other
   1. single core
   2. dual core
   3. quad core
  */
#define SPM_STARTUP_CONFIGURATION \
   { \
      /* single core configurations */ \
      "/dev/root/opt/bosch/base/registry/StartConfigDefault.reg", \
      "/dev/root/opt/bosch/base/registry/StartConfigXm.reg", \
      "/dev/root/opt/bosch/base/registry/StartConfigMedia.reg", \
      "/dev/root/opt/bosch/base/registry/StartConfigTml.reg", \
      /* dual core configurations */ \
      "/dev/root/opt/bosch/base/registry/StartConfigDefaultDualCore.reg", \
      "/dev/root/opt/bosch/base/registry/StartConfigXmDualCore.reg", \
      "/dev/root/opt/bosch/base/registry/StartConfigMediaDualCore.reg", \
      "/dev/root/opt/bosch/base/registry/StartConfigTmlDualCore.reg", \
      /* quad core configurations */ \
      "/dev/root/opt/bosch/base/registry/StartConfigDefaultQuadCore.reg", \
      "/dev/root/opt/bosch/base/registry/StartConfigXmQuadCore.reg", \
      "/dev/root/opt/bosch/base/registry/StartConfigMediaQuadCore.reg", \
      "/dev/root/opt/bosch/base/registry/StartConfigTmlQuadCore.reg" \
   }

/*!
   defines the number of configurations per core
  */
#define SPM_STARTUP_CONFIGURATION_MAX_NB_LAST_MODE  4

/*!
   defines the overall number of configurations
  */
#define SPM_STARTUP_CONFIGURATION_MAX_NB  (SPM_STARTUP_CONFIGURATION_MAX_NB_LAST_MODE * 3)

/*!
   defines a name for the index in SPM_STARTUP_CONFIGURATION
  */
#define SPM_STARTUP_CONFIGURATION_DEFAULT 0

/*!
   defines a name for the index in SPM_STARTUP_CONFIGURATION
  */
#define SPM_STARTUP_CONFIGURATION_XM       1

/*!
   defines a name for a "special" start-up registry that potentially (bIsSpecialStartupRegActive) overwrites all previous decisions
  */
//#define SPM_STARTUP_CONFIGURATION_PROJECT_REG                  "/dev/root/opt/bosch/base/registry/StartConfigDiagFactoryMode.reg"

/*!
   defines a name for a registry that is used when the target is not yet configured (bIsTargetUnconfigured)
  */
#define SPM_STARTUP_CONFIGURATION_PROJECT_UNCONFIGURED_REG     "/dev/root/opt/bosch/base/registry/StartConfigMinTarget.reg"

/*!
   defines a name for the index in SPM_STARTUP_CONFIGURATION
  */
#define SPM_STARTUP_CONFIGURATION_MEDIA       2

//#define SPM_STARTUP_CONFIGURATION_BLUETOOTH  3
//#define SPM_STARTUP_CONFIGURATION_MEDIA_CD   4
//#define SPM_STARTUP_CONFIGURATION_SPI        5

/*! ******************************************************
 which configuration is used as MIN target (see cyclic reset handling concept)
**********************************************************/
#define SPM_STARTUP_CONFIGURATION_MIN_TARGET      SPM_STARTUP_CONFIGURATION_XM

/*! ******************************************************
 which configuration is used in case of SoftwareUpdate (see LCM application mode concept)
**********************************************************/
#define SPM_STARTUP_CONFIGURATION_SWU        "/dev/root/opt/bosch/base/registry/StartConfigDefault_SWU.reg"

/*! ******************************************************
 which configuration is used for Operation Mode implementation
 if the corresponding makro is not configured this 
 operational mode is not enabled
**********************************************************/
#define SPM_STARTUP_CONFIGURATION_OPERATION
#define SPM_STARTUP_CONFIGURATION_DIAGNOSIS_0   "/dev/root/opt/bosch/base/registry/StartConfigDiagnosis0.reg"
#define SPM_STARTUP_CONFIGURATION_DIAGNOSIS_1   "/dev/root/opt/bosch/base/registry/StartConfigDiagnosis1.reg"

/*! ******************************************************
 set numbers of parallel start-up threads
**********************************************************/
#define SPM_STARTUP_CONFIGURATION_NUM_OF_SPMS     1

/*! ******************************************************
 name of the first start-up class
**********************************************************/
#define SPM_STARTUP_CONFIGURATION_CONFIG_PATH1    "LINUX"
// #define SPM_STARTUP_CONFIGURATION_CONFIG_PATH2    "NONE"
// #define SPM_STARTUP_CONFIGURATION_CONFIG_PATH3    "NONE"

/*! ******************************************************
 Start processor watchdog supervision.
**********************************************************/
#define SPM_FEATURE_ENABLE_PROCESSOR_WDT

/*! ******************************************************
 how many seconds before the WDR reset should the warning message be printed
**********************************************************/
#define SPM_FEATURE_WD_PREWARN_TIME_SEC          2

/*! ******************************************************
 enable sending of CVM broadcast events
**********************************************************/
//   #define SPM_FEATURE_ENABLE_OSAL_CVM_BROADCAST

/*! ******************************************************
 Network failed time -> send if ignition TO is detected.
**********************************************************/
// #define SPM_FEATURE_ENABLE_IGNITION_TIMEOUT

/*! ******************************************************
 Set to automatic mode after PowerOn (switch on via ignition)
**********************************************************/
#define SPM_FEATURE_ENABLE_IGN_AUTOMATIC_ON

/*! ******************************************************
 All state changes will be counted for the time of\n
 SPM_U32_DEBOUNCE_STATE_CHANGE_TIME. If the count is higher then\n
 SPM_U32_DEBOUNCE_STATE_CHANGE_COUNTER. All state changes\n
 will be blocked for SPM_U32_DEBOUNCE_TIME to reduce system\n
 processor load.
**********************************************************/
#define SPM_FEATURE_ENABLE_DEBOUNCE_STATE_CHANGE

/*! ******************************************************
 time in ms the state must be stable if "fast change" was detected
**********************************************************/
#define SPM_U32_DEBOUNCE_TIME                   2000
/*! ******************************************************
 time in ms the debounce is checked (for SPM_U32_DEBOUNCE_STATE_CHANGE_COUNTER times)
**********************************************************/
#define SPM_U32_DEBOUNCE_STATE_CHANGE_TIME      8000
/*! ******************************************************
 maximum number of times the state may change (within SPM_U32_DEBOUNCE_STATE_CHANGE_TIME)\n
 before a "fast change" is detected.
**********************************************************/
#define SPM_U32_DEBOUNCE_STATE_CHANGE_COUNTER   6

/*! ******************************************************
 Disable watchdog by SPM is enabled.
**********************************************************/
#define SPM_FEATURE_RESET_AND_WATCHDOG

// ******************************************************
// configuration of substate events
// ******************************************************
/*! ******************************************************
 which substate triggers are stored at shutdown
**********************************************************/
#define SPM_U32_STORE_TRIGGER_TYPES \
   SPM_U32_DOORLOCK \
   , SPM_U32_DOOROPEN \
   , SPM_U32_AUTOMATIC \
   , SPM_U32_IGNITION_LOGIC

/*! ******************************************************
 which substate triggers are restored at start-up
**********************************************************/
#define SPM_U32_RESTORE_TRIGGER_TYPES \
   SPM_U32_DOORLOCK \
   , SPM_U32_DOOROPEN \
   , SPM_U32_AUTOMATIC \
   , SPM_U32_IGNITION_LOGIC

/*! ******************************************************
 which substate triggers are valid even in case of LowVoltage
 check also description for macro SPM_U32_LOW_VOLTAGE_BLOCKING_TRIGGER.
 These are the excptions for substate blocking when SPM_U32_LOW_VOLTAGE_BLOCKING_TRIGGER
 is active.
 SPM_U32_SYNC_SHUTDOWN_ACK will be appended directly in source code.
**********************************************************/
#define SPM_U32_SHUTDOWN_TYPES_EN_LOW_VOLT \
   SPM_U32_TRANSPORTMODE \
   , SPM_U32_IGNITION \
   , SPM_U32_S_CONTACT \
   , SPM_U32_FAST_SHUTDOWN \
   , SPM_U32_IGNITION_LOGIC \
   , SPM_U32_IGNITION_TIMEOUT \
   , SPM_U32_EMERGENCY_OFF \
   , SPM_U32_NETWORK_FAILURE \
   , SPM_U32_STATE_TO \
   , SPM_U32_NETWORK_ACTIVITY

/*! ******************************************************
 which substate triggers are (or combined) used to set FAST_SHUTDOWN to active
**********************************************************/
#define SPM_U32_FASTSHUTDOWN_TYPES \
   SPM_U32_EMERGENCY_OFF \
   , SPM_U32_ENDURING_LOW_VOLTAGE \
   , SPM_U32_ENDURING_HIGH_VOLTAGE \
   , SPM_U32_ENDURING_CR_VOLTAGE \
   , SPM_U32_DIAG_FAST_SHUTDOWN \
   , SPM_U32_OVERTEMP

// ******************************************************
// configuration of generic recovery client
// ******************************************************

/*!
   defines the number of times the recoveryclient must collect debug data before the system is reset
  */
#define SPM_MAX_NUMBER_OF_RECOVERCLIENT_ACTIONS          3

// ******************************************************
// configuration cyclic reset handling
// ******************************************************

/*!
   defines the state of the pin when WD_OFF is active -> supervision is disabled
  */
#define SPM_ACTIVE_STATE_WD_OFF                          1

/*! ******************************************************
 If LCM callstack generation is triggered in FATAL cases, wait this ammount of time before
 reset is perfomend, to give callstackgeneration and write to errmem a chance
**********************************************************/
#define  SPM_ERRMEM_WRITE_DELAY_MS 1000

// SystemStateMachineType

#define SPM_CLASSICAL_SPM_STATEMACHINE            (tU8)0
#define SPM_SMC_STATEMACHINE                      (tU8)1

// ******************************************************
// configuration of failing shutdown request
// ******************************************************

/*!
   defines the number of times OSAL_C_S32_IOCTRL_WUP_SHUTDOWN is send to dev_wup before a
   reset is requested by dev_wup.
  */
#define SPM_SHUTDOWN_REQ_MAX_TRY_RESET                          3

/*!
   defines the number of times OSAL_C_S32_IOCTRL_WUP_SHUTDOWN is send before a FATAL_ASSERT is done.
   This is done if also the reset request could not be done by dev_wup.
  */
#define SPM_SHUTDOWN_REQ_MAX_TRY_ASSERT                          5

// ******************************************************
// configuration of OSAL queue fill status
// ******************************************************

/*!
   defines the percentage of fill level that will create a warning message in ErrMem
  */
#define SPM_WARNING_QUEUE_FILL_LEVEL_PERCENTAGE_WARN                  70

/*!
   defines the percentage of fill level that will create a first callstack dump
  */
#define SPM_ERROR_QUEUE_FILL_LEVEL_PERCENTAGE_HIGH                    80

/*!
   defines the percentage of fill level that will create a second callstack dump
  */
#define SPM_ERROR_QUEUE_FILL_LEVEL_PERCENTAGE_CRTICICAL_HIGH          90

// ******************************************************
// configuration of systemd interface
// ******************************************************

/*!
   activates the subsciption to ALL systemd properties
   ATTENTION: This is causing roughly 10% system load
  */
// #define SPM_FEATURE_ENABLE_SYSTEMD_SUBSCRIBE

/*! ******************************************************
 trigger sd_notify by project if needed
 project has to take care that the trigger has to be send in any case when startup is finished

   SPM_GET_IF_REFERENCE_NEW_VAR( poclSvMngr, ISpmSupervisionManager );
   poclSvMngr->vStartUpFinished();

 ******************************************************/
//#define SPM_FEATURE_SD_NOTIFY_BY_PROJECT_CONFIG

// ******************************************************
// configuration for RAM Usage
// ******************************************************
/*!
   defines period in seconds for RAM loading supervision checking. This bases on supervision interval time.
  */
#define SPM_RAM_USAGE_CHECK_PERIOD_SECONDS                            (( 10 * SPM_HW_WATCHDOG_DEFAULT_INTERVAL /*ms*/) / 1000) /*ms*/

// ******************************************************
// configuration of SPM_CORE_FI interface
// ******************************************************

/*! ******************************************************
   Define the allowed AppIDs to trigger StartProcess of Spm_Core_Fi interfaces\n
   ATTENTION: default is CCA_C_U16_APP_SPM_TEST_ADAPTOR used for testing\n
              and will be replaced by certain AppIDs in production
   Needed for CCA method StartProcess
**********************************************************/
#define SPM_CONFIGURED_APP_2_STARTPROCESS \
{ \
    CCA_C_U16_APP_SPM_TEST_ADAPTOR \
}

/*! ******************************************************
   Define the allowed AppIDs to trigger StopProcess of Spm_Core_Fi interfaces\n
   ATTENTION: default is CCA_C_U16_APP_SPM_TEST_ADAPTOR used for testing\n
              and will be replaced by certain AppIDs in production
   Needed for CCA method StopProcess
**********************************************************/
#define SPM_CONFIGURED_APP_2_STOPPROCESS \
{ \
    CCA_C_U16_APP_SPM_TEST_ADAPTOR \
}
/*! ******************************************************
   Define the allowed AppIDs to trigger Restart of Spm_Core_Fi interfaces\n
   ATTENTION: default is CCA_C_U16_APP_SPM_TEST_ADAPTOR used for testing\n
              and will be replaced by certain AppIDs in production
   Needed for CCA method Restart
   Leave it empty if no App configured.
**********************************************************/
#define SPM_CONFIGURED_APP_2_RESTART \
{ \
    CCA_C_U16_APP_SPM_TEST_ADAPTOR \
}

