/*!
  * \file spm_gpio.cpp
  *  \brief
  *    Reads and checks GPIO signals.
  *
  *  \note
  *  \b PROJECT: NextGen \n
   \b SW-COMPONENT: FC SPM \n
   \b COPYRIGHT:    (c) 2011 Robert Bosch GmbH, Hildesheim \n
  *  \see
  *  \version
  * Date      | Author            | Modification
  *
  ******
  */

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#include "spm_Config.h"
#include "spm_GlobDefs.h"
#include "spm_trace.h"
#include "spm_SubStateHandlerConfig.h"
#include "spm_SubStateHandler.h"
#include "spm_gpio.h"
#include "spm_IFactory.h"
#include "spm_IOsalProxy.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM_PRJ
#include "trcGenProj/Header/spm_gpio.cpp.trc.h"
#endif

spm_tclHwSig*spm_tclHwSig::_pMyStaticRef = NULL;

/*!
  * \fn
  *  \brief
  *    Constructor.
  *    Define list of GPIO pins need to be handled hardware signal, assigned substate and the parameter for debouncing
  *    Create message queue to allow context changes
  *
  *  \param
  *    factory: reference to factory class - caller of this function
  *  \version
  *    1.0   - Initial
  ******
  */
spm_tclHwSig::spm_tclHwSig( const ISpmFactory& factory ) : ISpmHwSigs( factory )
   , _hGpioQueueHandle ( OSAL_C_INVALID_HANDLE )
   , _bTestMode ( FALSE )
   , _bPollingActive ( FALSE )
   , _poclOsalProxy ( NULL )
   , _u32ThreadWaitingDelay ( SPM_HWGPIO_DELAY_IDLE )
   , _poclSubStateHandler ( NULL ){

   _pMyStaticRef = this;

}

/*!
  * \fn
  *  \brief
  *    Destructor.
  *
  *  \param
  *
  *  \version
  *    1.0   - Initial
  ******
  */
spm_tclHwSig::~spm_tclHwSig( ){
   _poclOsalProxy = NULL;
   _poclSubStateHandler = NULL;

   if ( _hGpioQueueHandle != OSAL_C_INVALID_HANDLE ){
      OSAL_s32MessageQueueClose( _hGpioQueueHandle );
      OSAL_s32MessageQueueDelete( "SPM_HW_GPIO" );
      _hGpioQueueHandle = OSAL_C_INVALID_HANDLE;
   }

}

/*!
  * \fn
  *  \brief
  *    Get references to all required LCM classes.
  *
  *  \param
  *
  *  \version
  *    1.0   - Initial
  ******
  */
tVoid spm_tclHwSig::vGetReferences( ){
   SPM_GET_IF_REFERENCE_USE_VAR( _poclOsalProxy, ISpmOsalProxy );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclSubStateHandler, ISpmSubStateClient );
}

/*!
  * \fn
  *  \brief
  *    Read initial GPIO pin state and update substate
  *    Register callback function to handle state changing
  *
  *  \param
  *
  *  \version
  *    1.0   - Initial
  ******
  */
tVoid spm_tclHwSig::vStartCommunication(){
	/*!
	* \fn
	*  \brief
	*   This method configures the gpio pin as either input/output
	*
	*  \version
	*    1.0   - Initial
	******
	*/
	tBool              bReturn = FALSE;

	OSAL_trGPIOData    tGpioData;
	OSAL_tIODescriptor tDevGpioHandle;

	// Update the dev id
	tGpioData.tId = OSAL_EN_U140_SW_ENABLE;

	// Open gpio device in read/write mode
	tDevGpioHandle = OSAL_IOOpen(OSAL_C_STRING_DEVICE_GPIO, OSAL_EN_READWRITE);

	if (OSAL_ERROR != tDevGpioHandle){


		tS32 s32GpioIoctl = OSAL_C_32_IOCTRL_GPIO_SET_ACTIVE_STATE;
		// set state
		if (OSAL_ERROR == OSAL_s32IOControl(tDevGpioHandle,
			s32GpioIoctl,
			(intptr_t)(&tGpioData))){

			ETG_TRACE_ERRMEM(("SPM: !!!!!! Error detected setting state of GPIO !!!!!!"));
		}

		if (OSAL_s32IOClose(tDevGpioHandle) != OSAL_OK){
			ETG_TRACE_ERRMEM(("SPM: !!!!!! Error detected closing DevGpio!!!!!!"));
		}
	}

}

/*!
  * \fn
  *  \brief
  *    Method for tracing all GPIO signal
  *
  *  \param
  *
  *  \version
  *    1.0   - Initial
  ******
  */
tVoid spm_tclHwSig::vTraceSignals( ){
   return;
}

/*!
  * \fn
  *  \brief
  *    Method for checking GPIO signal
  *
  *  \param
  *
  *  \version
  *    1.0   - Initial
  ******
  */
tVoid spm_tclHwSig::vCheckSignals( ){
   return;
}

/*!
  * \fn
  *  \brief
  *    Method invoked just before we start the main thread of HwSig
  *
  *  \param
  *
  *  \version
  *    1.0   - Initial
  ******
  */
tVoid spm_tclHwSig::vOnStart( ){
   return;
}

/*!
  * \fn
  *  \brief
  *    Method invoked just before we stop the main thread of HwSig
  *
  *  \param
  *
  *  \version
  *    1.0   - Initial
  ******
  */
tVoid spm_tclHwSig::vOnTerminate( ){
   return;
}

/*!
  * \fn
  *  \brief
  *    Read GPIO pin state from hardware
  *
  *  \param
  *
  *  \version
  *    1.0   - Initial
  ******
  */
tVoid spm_tclHwSig::vReadGpio( tU32 u32GpioId ){
   SPM_NULL_POINTER_CHECK( _poclOsalProxy );
}



/*!
  * \fn
  *  \brief
  *    Counting state changing number of a GPIO pin, blocking it if the pin is changing too fast
  *
  *  \param u32GpioId - ID of the GPIO pin in the GpioCfg Array
  *
  *  \version
  *    1.0   - Initial
  ******
  */
tVoid spm_tclHwSig::vCheckCount( tU32 u32GpioId ){

}

/*!
  * \fn
  *  \brief
  *    Check if the substate status needs to be updated
  *    called cyclically from main function
  *
  *  \param u32GpioId - ID of the GPIO pin in the GpioCfg Array
  *         bForce - always send update
  *
  *  \return SPM_HWGPIO_IDLE | SPM_HWGPIO_SEND | SPM_HWGPIO_WAITING
  *
  *  \version
  *    1.0   - Initial
  ******
  */
tU8 spm_tclHwSig::u8UpdateGpio( tU32  u32GpioId,
                                tBool bForce ){
   // function returns
   tU8 u8Ret = SPM_HWGPIO_IDLE;

   return( u8Ret );
}

/*!
  * \fn
  *  \brief
  *    main function of the thread
  *
  *  \param
  *
  *  \version
  *    1.0   - Initial
  ******
  */
tVoid spm_tclHwSig::main( ){
 
}

/*!
  * \fn
  *  \brief
  *    Entry function to handle GPIO state changing
  *
  *  \param pvArg - ID of the GPIO pin in the GpioCfg Array
  *
  *  \version
  *    1.0   - Initial
  ******
  */
tVoid spm_tclHwSig::vCallbackHdl( tVoid *pvArg ){

}

/*!
  * \fn
  *  \brief
  *    Post a message to change context
  *
  *  \param u32Msg - u32GpioId
  *
  *  \return TRUE - message send
  *          FALSE - message could not be send
  *
  *  \version
  *    1.0   - Initial
  ******
  */
tBool spm_tclHwSig::bPostMessage( tU32 u32Msg ){
   tBool bSuccess = FALSE;
   tS32  s32Ret   = OSAL_s32MessageQueuePost( _hGpioQueueHandle,
                                              (tCU8*)&u32Msg,
                                              sizeof( tU32 ),
                                              OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST );

   if ( s32Ret == OSAL_OK ){
      bSuccess = TRUE;
   }
   return( bSuccess );
}

/*!
  * \fn
  *  \brief
  *    Set test mode for this module, hw signal evaluation will be ignored
  *
  *  \param bTstMode - TRUE: activate test mode, FALSE: deactivate test mode
  *
  *  \version
  *    1.0   - Initial
  ******
  */
tVoid spm_tclHwSig::vSetTestMode( tBool bTstMode ){

}
