/**
  * @swcomponent  Life Cycle Management
  * @{
  * @file         spm_SystemStateMachineServiceHandlerConfig.cpp
  * @brief        Project specific implementation of the INC client handler for system state machine component
  * @copyright    (C) 2016 Robert Bosch Engineering and Business Solutions Vietnam Company.
  *               The reproduction, distribution and utilization of this file as well as the
  *               communication of its contents to others without express authorization is prohibited.
  *               Offenders will be held liable for the payment of damages.
  *               All rights reserved in the event of the grant of a patent, utility model or design.
  * @}
  */

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#include "spm_Config.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_spm_if.h"

#include "spm_SystemStateMachineServiceHandlerConfig.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM
#include "trcGenProj/Header/spm_SystemStateMachineServiceHandlerConfig.cpp.trc.h"
#endif

#include "spm_trace.h"

spm_tclSystemStateMachineServiceHandlerConfig::spm_tclSystemStateMachineServiceHandlerConfig( const ISpmFactory& factory )
   : spm_tclSystemStateMachineServiceHandler( factory )
{

}



spm_tclSystemStateMachineServiceHandlerConfig::~spm_tclSystemStateMachineServiceHandlerConfig( ){

}



tVoid spm_tclSystemStateMachineServiceHandlerConfig::vGetReferences( ){

   spm_tclSystemStateMachineServiceHandler::vGetReferences( );
}

tVoid spm_tclSystemStateMachineServiceHandlerConfig::vStartCommunication( ){

   //todo: IVI implementation not used for inf4cv -srs6cob
   //dp_tclSpmDpRnAiviCurrentPowerPlatform oCurPowerPlatform;
   //oCurPowerPlatform >> _u8PowerPlatform;
   //_u8PowerPlatform = SPM_U8_CONFIG_INF4CV;

   spm_tclSystemStateMachineServiceHandler::vStartCommunication( );
}

tVoid spm_tclSystemStateMachineServiceHandlerConfig::vOnComponentStatusRespond( eSystemSm_ApplicationStatus eServerComponentStatus ){
/*!
  * \fn
  *  \brief
  *      This method handles the server application status responded from VCC in R_COMPONENT_STATUS message.
  *  \param[in]
  *      eServerComponentStatus: The server component status received from Vehicle Communication Controller.
  *  \details
  *      The responded status will be stored locally in the base class and match with client component status.
  ******
  */
   ETG_TRACE_USR1( ( "spm_tclSystemStateMachineServiceHandlerConfig::vOnComponentStatusRespond: Server confirmed the component status is %x",
                     ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_APPLSTAT, eServerComponentStatus )) );
   spm_tclSystemStateMachineServiceHandler::vOnComponentStatusRespond( eServerComponentStatus );

   //vOnComponentStatusRespond: user specific code start
   if( bIsCommunicationActivated() ) {
      //When the communication is activated, client can send the very first messages. See the message sequence for more detail
      //Example: - Set state machine type
      //         - Set the current client app state
      //         - Update to the server the thermal status of modules attached to iMX side.
      bSetStateMachineType( SYSTEMSM_SMTYPE_DEFAULT );
   }
   //vOnComponentStatusRespond: user specific code end
} //vOnComponentStatusRespond

tVoid spm_tclSystemStateMachineServiceHandlerConfig::vOnMessageReject( tU8 u8RejectedMsgID, eSystemSm_RejReason eRejectReason ){
/*!
  * \fn
  *  \brief
  *      This method handles the message was rejected by VCC.
  *  \param[in]
  *      u8RejectedMsgID: Identification of message was rejected.
  *      eRejectReason  : Reason code of rejecting (refer to PDU_IncCommunication for more detail of supported code)
  *  \details
  *      Project specific re-action can be defined here.
  ******
  */
   ETG_TRACE_FATAL( ( "spm_tclSystemStateMachineServiceHandlerConfig::vOnMessageReject: RejectReason = %u and RejectedMsgID = 0x%x", \
                      ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_REJECTREASON, eRejectReason ), u8RejectedMsgID ) );
} //vOnMessageReject

tVoid spm_tclSystemStateMachineServiceHandlerConfig::vOnSetStateMachineTypeRespond( eSystemSm_StateMachineType eStateMachineType ){
/*!
  * \fn
  *  \brief
  *      This method handles the confirmation in the 'R_SET_STATEMACHINE_TYPE' received from Vehicle Communication Controller.
  *  \param[in]
  *      eStateMachineType: state machine type received from VCC.
  *  \detail
  *      Project specific re-action can be defined here.
  ******
  */
   ETG_TRACE_USR1( ( "spm_tclSystemStateMachineServiceHandlerConfig::vOnSetStateMachineTypeRespond: The state machine type as confirmed from the server VCC is %x", \
                      ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_STATEMACHINETYPE, eStateMachineType ) ) );
} //vOnSetStateMachineTypeRespond

tVoid spm_tclSystemStateMachineServiceHandlerConfig::vOnSetStateMachineTriggerRespond( tU8 u8NumOfTriggers,
                                                                                  tU8 *pu8SMTriggerList1, tU8 u8NumOfSMTriggerList1,
                                                                                  tU8 *pu8SMTriggerList2, tU8 u8NumOfSMTriggerList2,
                                                                                  tU8 *pu8SMTriggerList3, tU8 u8NumOfSMTriggerList3 ){
/*!
  * \fn
  *  \brief
  *      This method handles the confirmation received from the server VCC on the request to change the state of a specific trigger.
  *      The response from server includes complete list of state machine triggers and their state.
  *  \param[in]
  *      u8NumOfTriggers      : Total number of state machine triggers.
  *      pu8SMTriggerListN    : List of state machine trigger.
  *      u8NumOfSMTriggerListN: Number of element in the list.
  *  \detail
  *      Project specific re-action can be defined here.
  ******
  */
   ETG_TRACE_USR1( ( "spm_tclSystemStateMachineServiceHandlerConfig::vOnSetStateMachineTriggerRespond: Received the NumOfTriggers = %u", u8NumOfTriggers ) );
} //vOnSetStateMachineTriggerRespond

tVoid spm_tclSystemStateMachineServiceHandlerConfig::vOnSCCSystemStateUpdate( eSystemSm_SccSystemState eSccSystemState ){
/*!
  * \fn
  *  \brief
  *      This method handles event received from the server VCC when there's an update of system state.
  *      This event can be:
  *         - A confirmation from server when receive the update state request from client. (R_SET_SCC_SYSTEMSTATE)
  *         - A spontaneous update from the server. (R_SCC_SYSTEMSTATE)
  *         - A server system state change happen due to the client app system state change. (R_SEND_APP_SYSTEMSTATE)
  *  \param[in]
  *      eSccSystemState: The VCC system state
  *  \detail
  *      Project specific re-action can be defined here.
  ******
  */
   ETG_TRACE_USR1( ( "spm_tclSystemStateMachineServiceHandlerConfig::vOnSCCSystemStateUpdate: The SCC system state machine state is updated to %x", \
                      ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_SCCSYSTEMSTATE, eSccSystemState ) ) );
} //vOnSCCSystemStateUpdate


tVoid spm_tclSystemStateMachineServiceHandlerConfig::vOnSetAppSystemStateRequest( eSystemSm_AppSystemState eAppSystemState ){
/*!
  * \fn
  *  \brief
  *      This method handles request from VCC to transit the client system state to a specific state.
  *  \param[in]
  *      eAppSystemState: the requested system state
  *  \detail
  *      Project specific re-action can be defined here.
  ******
  */
   ETG_TRACE_USR1( ( "spm_tclSystemStateMachineServiceHandlerConfig::vOnSetAppSystemStateRequest: Received the request from server to move the client app system state to %x", \
                      ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_APPSYSTEMSTATE, eAppSystemState ) ) );

} //vOnSetAppSystemStateRequest

