/**
  * @swcomponent  Life Cycle Management
  * @{
  * @file         spm_SystemStateManagerBase.h
  * @brief        This is header file for System State Manager base class methods.
  * @copyright    (C) 2015 - 2016 Robert Bosch GmbH.
  *               The reproduction, distribution and utilization of this file as well as the
  *               communication of its contents to others without express authorization is prohibited.
  *               Offenders will be held liable for the payment of damages.
  *               All rights reserved in the event of the grant of a patent, utility model or design.
  * @}
  */

#pragma once

#ifndef _SPM_CONFIG_H
   #error spm_Config.h must be included!
#endif


#include <list>

#define SPM_FI_S_IMPORT_INTERFACE_SPM_COREFI_TYPES
#include "spm_fi_if.h"


#include "spm_ISystemStateManager.h"

#define SPM_U32_TRANSITION_ID_INVALID          (tU32)0xFFFFFFFF

class ISpmSupervisionManager;
class ISpmGlobalApplicationManager;
class ISpmCcaServiceServer;
class ISpmSubStateClient;
class ISpmSystemPowerManager;
class ISpmOsLinux;
class ISpmOsalProxy;
class ISpmWakeupHandler;

/*!
  *  \class spm_tclSystemStateManagerBase
  *  \brief This class creates state machine. It also handles the internal state transitions.
  *
  *         Any Substate change always gets notified first to SubStateHandler and SubStateHandler
  *         then decides, whether or not to inform SystemStateManager of the new SubState Trigger.
  *         Based on the value of certain SubStates, a new SystemState is calculated.
  *         This class is extended to meet the project specific needs.
  *
  *         All the registered connected applications are intimated of state change when the system
  *         state machine is updated to the next system state, through GlobalApplicationManager.
  *         The process of changing the software block states to the updated state is carried out by
  *         GlobalApplicationManager in conjunction with LocalApplicationManager and ApplicationDatabase
  *         and finally different processes are notified.
  */
class spm_tclSystemStateManagerBase : public ISpmSystemStateManager
{

protected:
TTriggerMsg _tSubStateTrigger;
TTriggerMsg _tCurSubStateTrigger;

virtual tU32  u32CalcNewSystemState( tU32 u32NewSystemState ) = 0;

tVoid  u32PostCalcNewSystemState( tU32 u32OldSystemState,
                                  tU32 u32NewSystemState,
                                  tU32 u32TransitionId );

typedef struct TSysStateHistoryEntry
{
   tU32 u32TimeStampStart;
   tU32 u32TimeStampReached;
   TTriggerMsg tTrigger;
   tU32 u32TransitionId;
} TSysStateHistoryEntry;
typedef std::list < TSysStateHistoryEntry > TSysStateHistoryList;
TSysStateHistoryList _oHistoryList;

public:
virtual tVoid vNewSubState( TTriggerMsg *tNewTrigger );

spm_tclSystemStateManagerBase( const ISpmFactory& factory );
virtual ~spm_tclSystemStateManagerBase( );

// trace handling
virtual tVoid vHandleTraceMessage( const tUChar *puchData );

virtual tVoid  vCheckStartOrderConfiguration( );

protected:
static tVoid vStateTimerCallback( tVoid *pArg );

tVoid vSystemStateTimeOut( );

#ifdef SPM_FEATURE_ENABLE_DEBOUNCE_STATE_CHANGE
   tVoid vDebounceStateChangeTo( );

   static tVoid vDebounceStateChangeTimerCallback( tVoid *pArg );

   OSAL_tTimerHandle                  _hDeboounceTimer;
#endif

OSAL_tSemHandle                       _hStateChangeAccess;                     // semaphore for exclusive

tU32                                  _u32SystemState;
tU32                                  _u32IntermediateSystemState;
tU32                                  _u32NewSystemState;
tU32                                  _u32InitialSystemState;
tU32                                  _u32OrgResetSystemState;
tBool                                 _bPointOfNoReturnReached;
OSAL_tTimerHandle                     _hStateTimeoutTimer;
tU32                                  _u32CurrentInternSubState;
tU32                                  _u32StateChangeCounter;

ISpmGlobalApplicationManager         *_poclGlobalApplicationManager;
ISpmSupervisionManager               *_poclSupervisionManager;
ISpmCcaServiceServer                 *_poclCcaServiceHandler;
ISpmSubStateClient                   *_poclSubStateHandler;
ISpmSystemPowerManager               *_poclSystemPowerManager;
ISpmOsLinux                          *_poSpmOsLinux;
ISpmOsalProxy                        *_poclOsalProxy;
ISpmWorkerServer                     *_poclWorkerServer;
ISpmWakeupHandler                    *_poclWakeupHandler;

protected:
static spm_tclSystemStateManagerBase *_poMyBaseRef;
virtual tBool  bActStateReached( tU32 u32SystemState ) const = 0;

virtual tVoid vStateNoTransition( )                          = 0;

virtual tVoid vCheckNewStateTransition( )                    = 0;

virtual tVoid vInitStateMachine( )                           = 0;

virtual tVoid vRestoreLastState( tU32 u32NewSystemState )    = 0;

public:
// pass through system node
virtual tVoid vNewSystemStateReached( );

virtual tU32  u32GetNewSystemState( ){ return( _u32NewSystemState ); }
virtual tU32  u32GetSystemState( ){ return( _u32SystemState ); }
virtual tU32  u32GetSystemStateBeforeReset( ){ return( _u32OrgResetSystemState ); }

tVoid  vStartStateTimeoutTimer( tU32 u32Time ) const;

virtual tVoid vSetStateTime( tU32 u32State,
                             tU32 u32Time )         = 0;

virtual tU32 u32GetStateTime( tU32 u32State ) const = 0;

virtual tBool bPointOfNoReturnReached( ){ return( _bPointOfNoReturnReached ); }

virtual tU32   u32GetRemainingStateTimeoutTime( );

protected:
virtual tVoid vGetReferences( );

virtual tVoid vStartCommunication( );

virtual const tChar*getName( ) const { return( "spm_tclSystemStateManagerBase" ); }

// Implementation of interface ISpmWorkerClient
virtual tVoid vHandleMessage( tU32 u32Message,
                              tU32 u32Parm = 0 );

virtual tBool bHandleSynchrounousCall( tU32   /*u32Message*/,
                                       tVoid */*args*/ ){ return( FALSE ); }  //lint !e715 Symbol 'xxx' not referenced --> CURRENTLY not used
};

